package org.maskat.core;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;

import javax.servlet.http.HttpServletRequest;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.sun.org.apache.xerces.internal.parsers.DOMParser;

/**
 * maskatリクエストビルダ
 * <pre>
 * リクエストXMLを解析し、MaskatRequestオブジェクトを作成します。
 * </pre>
 */
final class MaskatRequestBuilder {

	/**
	 * コンストラクタ
	 * <pre>
	 * ユーティリティクラスのため、newできないようprivate宣言している。
	 * </pre>
	 */
	private MaskatRequestBuilder() {
	}
	
	/**
	 * ビルド
	 * @param request リクエスト
	 * @return 組みあがったMaskatRequest
	 * @throws IOException
	 * @throws SAXException
	 */
	public static MaskatRequest build(HttpServletRequest request) throws IOException, SAXException {
		String xml = loadRequestString(request);
		Node root = getRootNode(xml);
		return createMaskatRequest(root);
	}
	
	/**
	 * リクエストXML取得
	 * @param request リクエスト
	 * @return XML文字列
	 * @throws IOException
	 */
	private static String loadRequestString(HttpServletRequest request) throws IOException {
		StringBuffer xml = new StringBuffer();
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(request.getReader());
			String line = null;
			while ((line = reader.readLine()) != null) {
				xml.append(line);
			}
		} finally {
			if (null != reader) {
				reader.close();
			}
		}
		return xml.toString();
	}
	
	/**
	 * DOMからルートノードを取得
	 * @param xml 解析XML文字列
	 * @return ルートノード
	 * @throws SAXException
	 * @throws IOException
	 */
	private static Node getRootNode(String xml) throws SAXException, IOException {
		DOMParser parser = new DOMParser();
		parser.parse(new InputSource(new StringReader(xml)));
		return parser.getDocument().getFirstChild();
	}
	
	/**
	 * MaskatRequest組み上げ
	 * @param node 組み上げ対象ノード
	 * @return 組みあがったMaskatRequest
	 */
	private static MaskatRequest createMaskatRequest(Node node) {
		
		MaskatRequest request = new MaskatRequest(node.getNodeName(), node.getTextContent());
		NodeList children = node.getChildNodes();
		
		for (int i = 0; i < children.getLength(); i++) {
			request.add(createMaskatRequest(children.item(i)));
		}
		return request;
	}
}
