package org.maskat.core;

import org.maskat.core.node.CompositeNode;
import org.maskat.core.node.LeafNode;
import org.maskat.core.node.Node;
import org.maskat.core.node.ParentNode;
import org.maskat.core.node.TableNode;


/**
 * maskatレスポンスビルダ
 * <pre>
 * レスポンスXMLを組み上げる。
 * 
 * 階層がないノードの追加は
 * <code>MaskatResponseBuilder#add(String, String)</code>
 * を用い、
 * <code>builder.add("tag", "10");</code>
 * とする。
 * 
 * 1階層の場合、
 * <code>MaskatResponseBuilder#add(String, Node)</code>
 * を用い、
 * <code>builder.add("parent", MaskatResponseBuilder.composite().add("child1", "10").add("child2", "20"));</code>
 * とする。
 * 
 * テーブル(2階層)の作成は
 * <code>MaskatResponseBuilder#add(Node)</code>
 * を用い、
 * <code>TableNode table = MaskatResponseBuilder.table("table");
 * table.newRow("row1").add("col1", 10).add("col2", "20");
 * table.newRow("row2").add("col1", 30).add("col2", "40");
 * builder.add(table);</code>
 * とする。
 * </pre>
 */
public class MaskatResponseBuilder {

	private static final String HEADDER = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
	private CompositeNode nodes = null;
	private final ParentNode rootNode;

	MaskatResponseBuilder(String rootNodeName) {
		nodes = new CompositeNode();
		this.rootNode = new ParentNode(rootNodeName, nodes);
	}
	
	/**
	 * ノード追加
	 * <pre>
	 * LeafNodeを追加します。
	 * 階層がないタグの追加ができます。
	 * </pre>
	 * @param nodeName ノード名
	 * @param value ノード値
	 */
	public void add(String nodeName, String value) {
		nodes.add(new LeafNode(nodeName, value));
	}

	/**
	 * ノード追加
	 * <pre>
	 * nodeをparentNameタグでまとめ、追加します。
	 * </pre>
	 * @param parentName 親タグ名
	 * @param node 子タグ
	 */
	public void add(String parentName, Node node) {
		nodes.add(new ParentNode(parentName, node));
	}

	/**
	 * ノード追加
	 * <pre>
	 * 単純にタグを追加します。
	 * TableNodeなどそれだけで完結しているものを追加する場合に使用してください。
	 * </pre>
	 * @param node 追加ノード
	 */
	public void add(Node node) {
		nodes.add(node);
	}
	
	/**
	 * レスポンスXML取得
	 * @return レスポンスXML
	 */
	public String getResponce() {
		return HEADDER + rootNode.getXML();
	}
	
	/**
	 * 複数ノードを追加できるCompositeを作成します。
	 * @return CompositeNode
	 */
	public static CompositeNode composite() {
		return new CompositeNode();
	}
	
	/**
	 * テーブルノードを作成します。
	 * @param tableName テーブル名
	 * @return TableNode
	 */
	public static TableNode table(String tableName) {
		return new TableNode(tableName);
	}
	
}
