/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * ロギング API を切り替えるためのファクトリを定義するクラスです。
 *
 * ログファクトリには指定された名前をもとに、適切なロガーオブジェクトを
 * 生成して返却する責任があります。
 */
maskat.lang.Class.declare("maskat.log.LogFactory", {

	_static: {
		/** ロガーオブジェクトのキャッシュ */
		cache: {},

		/**
		 * ロギング API を提供するロガーオブジェクトを取得します。
		 *
		 * @param name ロガーオブジェクトの名前
		 *
		 * @return ロガーオブジェクト
		 */
		getLog: function(name) {
			/* ロガーがキャッシュされている場合はそれを返却する */
			if (this.cache[name]) {
				return this.cache[name];
			}

			/* ログファクトリ経由でロガーを生成 */
			var log = this.getFactory().createLog(name);
			if (!log) {
				log = new maskat.log.NullLog();
			}

			this.cache[name] = log;
			return log;
		},

		/**
		 * ロガーの生成に用いるログファクトリを返却します。
		 *
		 * @return ログファクトリ
		 */
		getFactory: function() {
			/* ログファクトリが初期化済みの場合はそれを返却する */
			if (this.factory) {
				return this.factory;
			}
		
			var className;
			var type;
			try {
				/* プロパティで指定されたログファクトリを生成 */
				className = maskat.app.getProperty("maskat.log.factory");
				type = maskat.util.Converter.convert("function", className);
				this.factory = new type();
			} catch (e) {
				/* ファクトリが生成できない場合は NullLog を使用 */
				this.factory = new maskat.log.NullLogFactory();
				throw new maskat.lang.Error("LOG_FACTORY_INIT_FAILED",
					{ className: className }, e);
			}
			return this.factory;
		}
	},

	/**
	 * ロガーオブジェクトを生成します。
	 *
	 * このメソッドは抽象メソッドです。サブクラスはこのメソッドを実装し、
	 * ロガーを返却する必要があります。
	 *
	 * @param name ロガーオブジェクトの名前
	 */
	createLog: function(name) {
		return null;
	}

});