/*
 * Copyright (c) 2006-2008 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.maskat.dao;

import java.io.Serializable;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import org.springframework.orm.jpa.support.JpaDaoSupport;

/**
 * Spring の JPA 連携機能を利用した {@link GenericDAO} の実装です。
 * 
 * @param <T>
 *            エンティティのクラス
 * @param <ID>
 *            主キーのクラス
 */
public abstract class GenericDAOJpaImpl<T, ID extends Serializable> extends
		JpaDaoSupport implements GenericDAO<T, ID> {

	/** 永続化対象となるエンティティクラス */
	private Class<T> persistentClass;

	public GenericDAOJpaImpl(Class<T> type) {
		this.persistentClass = type;
	}

	/**
	 * アノテーション情報を利用して、JPA エンティティマネージャへの
	 * 依存性を DI コンテナ側から自動的に注入するための設定メソッド
	 * です。
	 * 
	 * @param entityManager JPA エンティティマネージャ
	 */
	@PersistenceContext
	public void setEntityManager2(EntityManager entityManager) {
		super.setEntityManager(entityManager);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.dao.GenericDAO#create(java.lang.Object)
	 */
	public T create(T newInstance) {
		getJpaTemplate().persist(newInstance);
		getJpaTemplate().refresh(newInstance);
		return newInstance;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.dao.GenericDAO#find(java.io.Serializable)
	 */
	public T find(ID id) {
		return (T) getJpaTemplate().find(persistentClass, id);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.dao.GenericDAO#findAll()
	 */
	@SuppressWarnings("unchecked")
	public List<T> findAll() {
		return getJpaTemplate().find(
				"SELECT x FROM " + persistentClass.getSimpleName() + " x");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.dao.GenericDAO#update(java.lang.Object)
	 */
	public void update(T transientObject) {
		getJpaTemplate().merge(transientObject);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.dao.GenericDAO#delete(java.lang.Object)
	 */
	public void delete(T persistentObject) {
		getJpaTemplate().remove(persistentObject);
	}

}