/*
 * Copyright (c) 2006-2008 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.maskat.sample.fruitshop.dao;

import java.text.Format;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import jp.sf.maskat.dao.GenericDAOJpaImpl;
import jp.sf.maskat.sample.fruitshop.entity.FruitShopOrder;

import org.apache.commons.lang.time.DateUtils;
import org.springframework.stereotype.Repository;

/**
 * JPA を使用した {@link FruitShopOrderDAO} の実装です。
 */
@Repository
public class FruitShopOrderDAOJpaImpl extends
		GenericDAOJpaImpl<FruitShopOrder, String> implements FruitShopOrderDAO {

	/** 受注番号のフォーマット */
	public static final Format orderIdFormat = new MessageFormat(
			"{0,date,yyyyMMdd}-{1,number,000}");

	/** 受注日時のフォーマット */
	public static final Format orderDateFormat = new SimpleDateFormat(
			"yyyyMMdd");

	/** 受注番号の連番部分のシーケンス */
	private int orderIdSeq;

	/** 受注日時 */
	private Date today;

	public FruitShopOrderDAOJpaImpl() {
		super(FruitShopOrder.class);
	}

	/**
	 * 新しい受注番号を生成して返します。
	 * 
	 * @return 新しい受注番号
	 */
	private synchronized String generateOrderId() {
		Date date = new Date();

		/* 日付が変わった場合はシーケンスをリセット */
		if (today == null || !DateUtils.isSameDay(today, date)) {
			today = date;
			orderIdSeq = getJpaTemplate().find(
				"SELECT x FROM FruitShopOrder x WHERE x.orderId LIKE ?1",
				orderDateFormat.format(date) + "%").size();
		}

		return orderIdFormat.format(new Object[] { date, ++orderIdSeq });
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.dao.GenericDAOJpaImpl#create(java.lang.Object)
	 */
	@Override
	public FruitShopOrder create(FruitShopOrder newInstance) {
		/* 受注番号・受注日時を設定 */
		String orderId = generateOrderId();
		newInstance.setOrderId(orderId);
		newInstance.setOrderDate(today);

		/* 受注オブジェクトを永続化する */
		return super.create(newInstance);
	}

}
