/*
 * Copyright (c) 2006-2008 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.maskat.sample.fruitshop.service;

import java.util.List;

import javax.annotation.Resource;

import jp.sf.maskat.dao.CodeLabelDAO;
import jp.sf.maskat.sample.fruitshop.dao.FruitShopOrderDAO;
import jp.sf.maskat.sample.fruitshop.dao.FruitShopOrderDetailDAO;
import jp.sf.maskat.sample.fruitshop.dao.FruitShopProductDAO;
import jp.sf.maskat.sample.fruitshop.dto.ListOrderRequest;
import jp.sf.maskat.sample.fruitshop.dto.ListOrderResponse;
import jp.sf.maskat.sample.fruitshop.dto.RegisterOrderRequest;
import jp.sf.maskat.sample.fruitshop.dto.RegisterOrderResponse;
import jp.sf.maskat.sample.fruitshop.dto.SearchProductRequest;
import jp.sf.maskat.sample.fruitshop.dto.SearchProductResponse;
import jp.sf.maskat.sample.fruitshop.entity.FruitShopOrder;
import jp.sf.maskat.sample.fruitshop.entity.FruitShopOrderDetail;
import jp.sf.maskat.sample.fruitshop.entity.FruitShopProduct;

import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * POJO を用いた {@link FruitShopOrderService} の実装です。
 */
@Service
@Transactional
public class FruitShopOrderServiceImpl implements FruitShopOrderService {

	/** コード／ラベル対応表の参照に使用する DAO */
	private CodeLabelDAO codeLabelDAO;

	/** 商品情報を管理する DAO */
	private FruitShopProductDAO productDAO;

	/** 受注情報を管理する DAO */
	private FruitShopOrderDAO orderDAO;

	/** 受注明細情報を管理する DAO */
	private FruitShopOrderDetailDAO detailDAO;

	/**
	 * コード／ラベル対応表の参照に使用する DAO を設定します。
	 * 
	 * @param codeLabelDAO
	 *            コード／ラベル対応表の参照に使用する DAO
	 */
	@Resource
	public void setCodeLabelDAO(CodeLabelDAO codeLabelDAO) {
		this.codeLabelDAO = codeLabelDAO;
	}

	/**
	 * 商品情報を管理する DAO を設定します。
	 * 
	 * @param productDAO
	 *            商品情報を管理する DAO
	 */
	@Resource
	public void setProductDAO(FruitShopProductDAO productDAO) {
		this.productDAO = productDAO;
	}

	/**
	 * 受注情報を管理する DAO を設定します。
	 * 
	 * @param orderDAO
	 *            受注情報を管理する DAO
	 */
	@Resource
	public void setOrderDAO(FruitShopOrderDAO orderDAO) {
		this.orderDAO = orderDAO;
	}

	/**
	 * 受注明細情報を管理する DAO を設定します。
	 * 
	 * @param detailDAO
	 *            受注明細情報を管理する DAO
	 */
	@Resource
	public void setDetailDAO(FruitShopOrderDetailDAO detailDAO) {
		this.detailDAO = detailDAO;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.sample.fruitshop.service.FruitShopOrderService#getCustomerCodes(java.lang.String)
	 */
	public List<Object[]> getCustomerCodes(String keyword) {
		return codeLabelDAO.find("fruitshop_customers", "customer_id", "name",
				keyword);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.sample.fruitshop.service.FruitShopOrderService#getProductCodes(java.lang.String)
	 */
	public List<Object[]> getProductCodes(String keyword) {
		return codeLabelDAO.find("fruitshop_products", "product_id", "name",
				keyword);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.sample.fruitshop.service.FruitShopOrderService#searchProduct(jp.sf.maskat.sample.fruitshop.dto.SearchProductRequest)
	 */
	public SearchProductResponse searchProduct(SearchProductRequest request) {
		FruitShopProduct product = productDAO.find(request.getProductId());

		SearchProductResponse result = new SearchProductResponse();
		BeanUtils.copyProperties(product, result);

		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.sample.fruitshop.service.FruitShopOrderService#listOrders(jp.sf.maskat.sample.fruitshop.dto.ListOrderRequest)
	 */
	public ListOrderResponse listOrders(ListOrderRequest request) {
		ListOrderResponse response = new ListOrderResponse();
		response.setOrders(orderDAO.findAll());
		return response;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.sample.fruitshop.service.FruitShopOrderService#registerOrder(jp.sf.maskat.sample.fruitshop.dto.RegisterOrderRequest)
	 */
	public RegisterOrderResponse registerOrder(RegisterOrderRequest request) {
		/* 受注情報を永続化する */
		FruitShopOrder order = new FruitShopOrder();
		order.setCustomerId(request.getCustomerId());
		order = orderDAO.create(order);

		/* 受注明細情報を永続化し、受注情報と関連付ける */
		int total = 0;
		for (FruitShopOrderDetail detail : request.getDetails()) {
			detail.setOrder(order);
			detail.setOrderId(order.getOrderId());
			detail = detailDAO.create(detail);

			FruitShopProduct product = productDAO.find(detail.getProductId());
			total += product.getUnitPrice() * detail.getQuantity();
		}

		/* 受注情報の合計金額を更新する */
		order.setTotal(total);
		orderDAO.update(order);
		return new RegisterOrderResponse();
	}

}
