/*
 * Copyright (c) 2006-2008 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.maskat.sample.gourmet.dto;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import jp.sf.maskat.sample.gourmet.entity.GourmetRestaurant;

import org.springframework.beans.BeanUtils;

@XmlRootElement(name = "response")
@XmlAccessorType(XmlAccessType.FIELD)
public class SearchRestaurantResponse {

	@XmlElementWrapper(name = "restaurants")
	@XmlElement(name = "restaurant")
	private List<Restaurant> restaurants;

	public SearchRestaurantResponse(List<GourmetRestaurant> restaurants) {
		super();
		setRestaurants(restaurants);
	}

	public SearchRestaurantResponse() {
		this(null);
	}

	public List<Restaurant> getRestaurants() {
		return restaurants;
	}

	public void setRestaurants(List<GourmetRestaurant> restaurants) {
		this.restaurants = new ArrayList<Restaurant>();

		for (GourmetRestaurant restaurant : restaurants) {
			Restaurant dto = new Restaurant();
			BeanUtils.copyProperties(restaurant, dto);
			this.restaurants.add(dto);
		}
	}

	@XmlAccessorType(XmlAccessType.FIELD)
	public static class Restaurant {

		private String restaurantId;

		private String name;

		private String address;

		@XmlJavaTypeAdapter(StyleAdapter.class)
		private Integer style;

		private int budget;

		@XmlJavaTypeAdapter(CouponAdapter.class)
		private Boolean coupon;

		private double latitude;

		private double longitude;

		private String description;

		public String getRestaurantId() {
			return restaurantId;
		}

		public void setRestaurantId(String restaurantId) {
			this.restaurantId = restaurantId;
		}

		public String getName() {
			return name;
		}

		public void setName(String name) {
			this.name = name;
		}

		public String getAddress() {
			return address;
		}

		public void setAddress(String address) {
			this.address = address;
		}

		public int getStyle() {
			return style;
		}

		public void setStyle(int style) {
			this.style = style;
		}

		public int getBudget() {
			return budget;
		}

		public void setBudget(int budget) {
			this.budget = budget;
		}

		public boolean isCoupon() {
			return coupon;
		}

		public void setCoupon(boolean coupon) {
			this.coupon = coupon;
		}

		public double getLatitude() {
			return latitude;
		}

		public void setLatitude(double latitude) {
			this.latitude = latitude;
		}

		public double getLongitude() {
			return longitude;
		}

		public void setLongitude(double longitude) {
			this.longitude = longitude;
		}

		public String getDescription() {
			return description;
		}

		public void setDescription(String description) {
			this.description = description;
		}
	}

	public static class StyleAdapter extends XmlAdapter<String, Integer> {

		public static final String STYLE_NAME_WESTERN = "洋食";

		public static final String STYLE_NAME_JAPANESE = "和食";

		public static final String STYLE_NAME_CHINESE = "中華";

		@Override
		public String marshal(Integer v) throws Exception {
			switch (v) {
			case GourmetRestaurant.STYLE_WESTERN:
				return STYLE_NAME_WESTERN;
			case GourmetRestaurant.STYLE_JAPANESE:
				return STYLE_NAME_JAPANESE;
			case GourmetRestaurant.STYLE_CHINESE:
				return STYLE_NAME_CHINESE;
			}
			return null;
		}

		@Override
		public Integer unmarshal(String v) throws Exception {
			if (STYLE_NAME_WESTERN.equals(v)) {
				return GourmetRestaurant.STYLE_WESTERN;
			} else if (STYLE_NAME_JAPANESE.equals(v)) {
				return GourmetRestaurant.STYLE_JAPANESE;
			} else if (STYLE_NAME_CHINESE.equals(v)) {
				return GourmetRestaurant.STYLE_CHINESE;
			}
			return null;
		}
	}

	public static class CouponAdapter extends XmlAdapter<String, Boolean> {

		public static final String COUPON_AVAILABLE = "あり";

		public static final String COUPON_NOT_AVAILABLE = "なし";

		@Override
		public String marshal(Boolean v) throws Exception {
			return v ? COUPON_AVAILABLE : COUPON_NOT_AVAILABLE;
		}

		@Override
		public Boolean unmarshal(String v) throws Exception {
			return COUPON_AVAILABLE.equals(v);
		}

	}
}
