/*
 * Copyright (c) 2006-2009 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.MaskatElementVisitor;
import jp.sf.maskat.core.event.Event;
import jp.sf.maskat.core.event.EventDef;
import jp.sf.maskat.core.event.Source;
import jp.sf.maskat.core.event.Target;

import org.apache.commons.beanutils.BasicDynaBean;
import org.apache.commons.beanutils.DynaClass;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.PredicateUtils;


public class DynaComponent extends BasicDynaBean implements Component {

	private static final long serialVersionUID = 925589796525491481L;

	private PropertyChangeSupport support;

	private MaskatElement parent;
	
	public DynaComponent() {
		super(null);
		this.support = new PropertyChangeSupport(this);
	}
	
	public DynaComponent(DynaClass dynaClass) {
		super(dynaClass);
		this.support = new PropertyChangeSupport(this);
		setDefaultVlaue();
	}
	
	private void setDefaultVlaue() {
		DynaComponentClass dynaClass = (DynaComponentClass) getDynaClass();
		if (dynaClass.isWidthResizable()) {
			this.set("width", new Integer(0));
		}
		if (dynaClass.isHeightResizable()) {
			this.set("height", new Integer(0));
		}
		Map defaultMap = this.getDefaultValues();
		for (Iterator ite = defaultMap.keySet().iterator(); ite.hasNext();) {
			String name = (String) ite.next();
			this.set(name, defaultMap.get(name));
		}
		DynaProperty[] properties = (DynaProperty[])
			this.dynaClass.getDynaProperties();
		for (int i = 0; i < properties.length; i++) {
			if (Boolean.class.isAssignableFrom(properties[i].getType())) {
				if (!(this.get(properties[i].getName()) instanceof Boolean)) {
					this.set(properties[i].getName(), Boolean.FALSE);
				}
			}
		}
	}
	
	public void setDynaClass(DynaClass dynaClass) {
		this.dynaClass = dynaClass;
		setDefaultVlaue();
	}
    
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.apache.commons.beanutils.BasicDynaBean#set(java.lang.String,
	 *      int, java.lang.Object)
	 */
	public void set(String name, int index, Object value) {
		Object oldValue = get(name, index);
		super.set(name, index, value);
		support.firePropertyChange(name, oldValue, value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.apache.commons.beanutils.BasicDynaBean#set(java.lang.String,
	 *      java.lang.Object)
	 */
	public void set(String name, Object value) {
		Object oldValue = get(name);
		super.set(name, value);
		if ("name".equals(name)) {
			updateComponentIdInEvent((String) oldValue, (String) value);
		}
		support.firePropertyChange(name, oldValue, value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.apache.commons.beanutils.BasicDynaBean#set(java.lang.String,
	 *      java.lang.String, java.lang.Object)
	 */
	public void set(String name, String key, Object value) {
		Object oldValue = get(key, name);
		super.set(name, key, value);
		support.firePropertyChange(name, oldValue, value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.apache.commons.beanutils.BasicDynaBean#remove(java.lang.String,
	 *      java.lang.String)
	 */
	public void remove(String name, String key) {
		Object oldValue = get(key, name);
		super.remove(name, key);
		support.firePropertyChange(name, oldValue, null);
	}

	public MaskatElement getParent() {
		return parent;
	}

	public void setParent(MaskatElement parent) {
		this.parent = parent;
	}

	protected List getRawChildren() {
		List result = (List) get("children");
		if (result == null) {
			result = new ArrayList();
			set("children", result);
		}
		return result;
	}

	public List getChildren() {
		return getRawChildren();
	}

	public List getTypedChildren(Class clazz) {
		List result = new ArrayList();
		CollectionUtils.select(getRawChildren(), PredicateUtils
				.instanceofPredicate(clazz), result);
		return result;
	}

	public List getTypedChildren(String name) {
		List result = new ArrayList();
		List children = getRawChildren();
		for (int i = 0; i < children.size(); i++) {
			Object obj = children.get(i);
			if (obj instanceof DynaComponent) {
				DynaComponent dynaComponent = (DynaComponent) obj;
				if (name.equals(dynaComponent.getDynaClass().getName())) {
					result.add(dynaComponent);
				}
			}
		}
		return result;
	}
	
	public void addChild(Object child) {
		getRawChildren().add(child);
		if (child instanceof MaskatElement) {
			((MaskatElement) child).setParent(this);
		}
		support.firePropertyChange("addChild", "", null);
	}

	public void addChildToIdx(Object child, int idx) {
		getRawChildren().add(idx, child);
		if (child instanceof MaskatElement) {
			((MaskatElement) child).setParent(this);
		}
		support.firePropertyChange("addChild", "", null);
	}

	public int getChildIdx(Object child) {
		return getRawChildren().indexOf(child);
	}

	public Object getChildByTypeIdx(Class clazz, int idx) {
		List result = new ArrayList();
		CollectionUtils.select(getRawChildren(), PredicateUtils
				.instanceofPredicate(clazz), result);
		if (result.size() <= idx) {
			return null;
		}
		return result.get(idx);
	}

	public void removeAllChildren() {
		getRawChildren().clear();
		support.firePropertyChange("removeChild", "", null);
	}

	public void removeChild(Object obj) {
		getRawChildren().remove(obj);
		support.firePropertyChange("removeChild", "", null);
	}

	public void removeAllByType(Class clazz) {
		List elements = getRawChildren();
		for (int i = 0; i < elements.size(); i++) {
			Object element = elements.get(i);
			if (clazz.isInstance(element)) {
				elements.remove(element);
			}
		}
		support.firePropertyChange("removeChild", "", null);
	}
	
	public void accept(MaskatElementVisitor visitor) {
		List elements = getChildren();
		for (int i = 0; i < elements.size(); i++) {
			Object element = elements.get(i);
			visitor.visit(element);
			if (element instanceof MaskatElement) {
				((MaskatElement) element).accept(visitor);
			}
		}
	}

	public Layout getLayout() {
		return getParent() == null ? null : ((LayoutElement) getParent()).getLayout();
	}

	public String getName() {
		return values.containsKey("name") ?
				(String) get("name") : "";
	}

	public void setName(String name) {
		set("name", name);
	}

	public int getTop() {
		return values.containsKey("top") ?
			((Integer) get("top")).intValue() : 0;
	}
	
	public void setTop(int top) {
		set("top", new Integer(top));
	}

	public int getLeft() {
		return values.containsKey("left") ?
			((Integer) get("left")).intValue() : 0;
	}
	
	public void setLeft(int left) {
		set("left", new Integer(left));
	}

	public int getHeight() {
		return values.containsKey("height") ?
			((Integer) get("height")).intValue() : 0;
	}

	public void setHeight(int height) {
		set("height", new Integer(height));
	}

	public int getWidth() {
		return values.containsKey("width") ?
			((Integer) get("width")).intValue() : 0;
	}

	public void setWidth(int width) {
		set("width", new Integer(width));
	}

	public void setConstraint(int left, int top, int width, int height) {
		if (getDynaClass().getDynaProperty("left") != null) {
			super.set("left", new Integer(left));
		}
		if (getDynaClass().getDynaProperty("top") != null) {
			super.set("top", new Integer(top));
		}
		if (getDynaClass().getDynaProperty("width") != null) {
			super.set("width", new Integer(width));
		}
		if (getDynaClass().getDynaProperty("height") != null) {
			super.set("height", new Integer(height));
		}
		support.firePropertyChange("constraint", null, null);
	}

	public int getTabIndex() {
		return ((Integer) get("tabIndex")).intValue();
	}

	public void setTabIndex(int tabIndex) {
		set("tabIndex", new Integer(tabIndex));
	}

	public void addPropertyChangeListener(PropertyChangeListener listener) {
		support.addPropertyChangeListener(listener);
	}

	public void removePropertyChangeListener(PropertyChangeListener listener) {
		support.removePropertyChangeListener(listener);
	}

	public Component getContainer() {
		return (Component) getParent();
	}

	public String[] getEventTypes() {
		return ((DynaComponentClass) getDynaClass()).getEventTypes();
	}

	public Map getDefaultValues() {
		return ((DynaComponentClass) getDynaClass()).getDefaultValues();
	}
	
	public boolean hasEvent(String eventType) {
		String[] eventTypes = getEventTypes();
		if (eventTypes != null) {
			for (int i = 0; i < eventTypes.length; i++) {
				if (eventTypes[i].equals(eventType)) {
					return true;
				}
			}
		}
		return false;
	}

	public Event getEvent(String eventType) {
		if (hasEvent(eventType)) {
			EventDef eventDef = getLayout().getLayoutDef().getEventDef();
			if (eventDef != null) {
				jp.sf.maskat.core.event.Component component = eventDef.findComponent(getName());
				if (component != null) {
					return component.findEvent(eventType);
				}
			}
		}
		return null;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.sf.maskat.core.layout.Component#setEvent(Evnet)
	 */
	public void setEvent(Event event) {
		if (hasEvent(event.getId())) {
			EventDef eventDef = getLayout().getLayoutDef().getEventDef();
			jp.sf.maskat.core.event.Component component = eventDef.findComponent(getName());
			if (component == null) {
				component = new jp.sf.maskat.core.event.Component();
				component.setId(getName());
				eventDef.addChild(component);
			}
			component.setEvent(event);
			support.firePropertyChange("#event", null, event);
		}
	}
	
	public Object clone() throws CloneNotSupportedException {
		DynaComponent dynaBean = (DynaComponent) super.clone();
		dynaBean.values = (HashMap) dynaBean.values.clone();
		dynaBean.set("children", new ArrayList());
		dynaBean.setParent(null);

		List children = this.getChildren();
		for (int i = 0; i < children.size(); i++) {
			Object child = children.get(i);
			if (child instanceof MaskatElement) {
				dynaBean.addChild(((MaskatElement) child).clone());
			} else if (child instanceof DynaComponent) {
				dynaBean.addChild(((DynaComponent) child).clone());
			} else {
				dynaBean.addChild(child);
			}
		}
		dynaBean.support = new PropertyChangeSupport(dynaBean);
		return dynaBean;
	}
	
	public List getAllDescendants(MaskatElement parent, Class descendant) {
		ArrayList result = new ArrayList();
		List children = parent.getChildren();
		for (int i = 0; i < children.size(); i++) {
			Object child = children.get(i);
			if (descendant.equals(child.getClass())) {
				result.add(child);
			}
			if (child instanceof MaskatElement) {
				result.addAll(getAllDescendants((MaskatElement) child, descendant));
			}
		}
		return result;
	}
	
	public static boolean isAttributeProperty(String propName) {
		return !("children".equals(propName) || "parent".equals(propName));
	}
	
	private void updateComponentIdInEvent(String oldId, String newId) {
		if (oldId == null || oldId.equals(newId)) {
			return;
		}
		if (getParent() == null || getLayout() == null 
				|| getLayout().getLayoutDef() == null
				|| getLayout().getLayoutDef().getEventDef() == null) {
			return;
		}
		
		EventDef eventDef = getLayout().getLayoutDef().getEventDef();
		jp.sf.maskat.core.event.Component component = eventDef.findComponent(oldId);
		if (component != null) {
			component.setId(newId);
		}
		
		//TODO should update all Source.obj , Source.idxRef, Target.in, Target.out who equals to oldId ?
		List sources = getAllDescendants(eventDef, Source.class);
		for (int i=0; i<sources.size(); i++) {
			Source source = (Source)sources.get(i);
			if (oldId.equals(source.getObj())) {
				source.setObj(newId);
			}
			if (oldId.equals(source.getIdxRef())) {
				source.setIdxRef(newId);
			}
		}
		List targets = getAllDescendants(eventDef, Target.class);
		for (int i=0; i<targets.size(); i++) {
			Target target = (Target)targets.get(i);
			if (oldId.equals(target.getIn())) {
				target.setIn(newId);
			}
			if (oldId.equals(target.getOut())) {
				target.setOut(newId);
			}
		}
	}

	public String getContext() {
		return values.containsKey("context") ?
				(String) get("context") : null;
	}

	public void setContext(String context) {
		if (values.containsKey("context")) {
			set("context", context);
		}
	}
}
