/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.event;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * イベント定義のルート要素クラスです
 */
public class EventDef extends AbstractEventElement {

	/**
	 * ルート要素名
	 */
	private String id;

	/**
	 * リモートURL
	 */
	private RemoteUrl remoteUrl;

	/**
	 * ルート要素が持っているイベント一覧
	 */
	private Map events = new HashMap();

	/**
	 * ルート要素が持っているイベント参照一覧
	 */
	private Map eventRefs = new HashMap();

	/**
	 * デフォルトコンストラクタです
	 */
	public EventDef() {
		super();
	}
	
	/**
	 * ルート要素名を取得します
	 * 
	 * @return ルート要素名
	 */
	public String getId() {
		return id;
	}

	/**
	 * ルート要素名を設定します
	 * 
	 * @param id ルート要素名
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * リモートURLを取得します
	 * 
	 * @return リモートURL
	 */
	public String getRemoteUrl() {
		return (remoteUrl == null) ? null : remoteUrl.getUrl();
	}

	/**
	 * リモートURLを設定します
	 * 
	 * @param remoteUrl リモートURL
	 */
	public void setRemoteUrl(String remoteUrl) {
		if (this.remoteUrl == null) {
			addChild(new RemoteUrl());
		}
		this.remoteUrl.setUrl(remoteUrl);
	}

	/**
	 * イベント定義XMLに参照される部品名の一覧を取得します
	 *  
	 * @return 部品名一覧
	 */
	public Set getAllObjNames() {
		Set result = new HashSet();

		for (Iterator it = events.values().iterator(); it != null
				&& it.hasNext();) {
			Event event = (Event) it.next();
			event.getRelatedObjNames(result);
		}
		for (Iterator it = eventRefs.values().iterator(); it != null
				&& it.hasNext();) {
			EventRef event = (EventRef) it.next();
			event.getRelatedObjNames(result);
		}
		List elements = getTypedChildren(Component.class);
		for (int i = 0; i < elements.size(); i++) {
			Event[] events = ((Component) elements.get(i)).getAllEvents();
			for (int j = 0; j < events.length; j++) {
				events[j].getRelatedObjNames(result);
			}
		}
		return result;
	}

	/**
	 * 子要素を追加します
	 * 
	 * @param child 追加したい子要素
	 */
	public void addChild(Object child) {
		super.addChild(child);
		if (child instanceof Component) {
			addComponent((Component) child);
		}
		if (child instanceof Event) {
			addEvent((Event) child);
		}
		if (child instanceof RemoteUrl) {
			remoteUrl = (RemoteUrl) child;
		}
	}

	/**
	 * Component子要素の追加を行います
	 * 
	 * @param comp 追加したいComponent子要素
	 */
	private void addComponent(Component comp) {
		if (comp == null || comp.getId() == null) {
			throw new IllegalArgumentException();
		}
	}

	/**
	 * 指定された名前のComponentを取得します
	 * 
	 * @param id コンポーネント名
	 * @return コンポーネント、存在しない場合にはnull
	 */
	public Component findComponent(String id) {
		EventElement[] elements = getAllComponents();
		for (int i = 0; i < elements.length; i++) {
			if (id.equals(((Component) elements[i]).getId())) {
				return (Component) elements[i];
			}
		}
		return null;
	}

	/**
	 * 子要素として持っている全てのComponetを配列で取得します
	 * 
	 * @return Componentの配列
	 */
	public Component[] getAllComponents() {
		return (Component[]) getTypedChildren(Component.class)
				.toArray(new Component[0]);
	}

	/**
	 * 子要素としてComponentを持っているか判定します
	 * 
	 * @return 子要素としてComponentを持っている場合
	 *          true、もっていない場合 false
	 */
	public boolean hasComponents() {
		return (this.getTypedChildren(Component.class) == null) ? false : true;
	}

	/**
	 * イベントを追加します
	 * 
	 * @param event イベント
	 */
	private void addEvent(Event event) {
		if (event instanceof EventRef) {
			EventRef er = (EventRef) event;
			synchronized (EventDef.class) {
				if (eventRefs == null) {
					eventRefs = new HashMap();
				}
				eventRefs.put(er.getRefid(), er);
			}
			return;
		}
		events.put(event.getId(), event);

	}

	/**
	 * 指定された名前のイベントを取得します
	 * 
	 * @param id イベント名
	 * @return イベント、指定されたidに該当するイベントが
	 *          存在しない場合にはnull
	 */
	public Event findEvent(String id) {
		return (Event) events.get(id);
	}

	/**
	 * 指定された名前のイベントを削除します
	 * 
	 * @param id イベント名
	 */
	public void removeEvent(String id) {
		events.remove(id);
	}

	/**
	 * 子要素として持っている全てのイベントを配列で取得します
	 * 
	 * @return イベントの配列
	 */
	public Event[] getAllEvents() {
		return (Event[]) events.values().toArray(new Event[0]);
	}

	/**
	 * 指定された名前のイベント参照を取得します
	 * 
	 * @param id イベントさん証明
	 * @return イベント参照、指定されたidに該当するイベント参照
	 *          が存在しない場合にはnull
	 */
	public Event findEventRef(String id) {
		return (Event) eventRefs.get(id);
	}

	/**
	 * 子要素として持っている全てのイベント参照を配列で取得します
	 * 
	 * @return イベント参照の配列
	 */
	public EventRef[] getAllEventRefs() {
		return (EventRef[]) eventRefs.values().toArray(new EventRef[0]);
	}

	/**
	 * イベント参照からイベントの置き換えを行います
	 */
	public void prepareEvent() {
		Component[] components = getAllComponents();
		for (int i = 0; i < components.length; i++) {
			Event[] events = components[i].getAllEvents();
			for (int j = 0; j < events.length; j++) {
				if (events[j].getRef() != null) {
					Event evtRef = findEventRef(events[j].getRef());
					if (evtRef != null) {
						events[j].removeAllChildren();
						List childElements = evtRef.getChildren();
						for (int k = 0; k < childElements.size(); k++) {
							events[j].addChild(childElements.get(k));
						}
					}

				}
			}
		}
	}
}
