/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.MaskatElement;

import org.apache.commons.beanutils.BasicDynaClass;

/**
 * DynaComponentクラスのjava.lang.Classの機能をシミュレーションしたクラスです
 * 
 *  Classの機能の他に以下の情報を保持します
 *  
 * ・完全修飾ドメイン名
 * ・部品編集条件（width, height, top, left, name)の定義
 * ・イベント条件（定義可能であるイベント一覧）
 * ・子要素条件（子要素として登録可能な部品一覧）
 */
public class DynaComponentClass extends BasicDynaClass {

	/**
	 * シリアルバージョンUID
	 */
	private static final long serialVersionUID = 5110101941810053916L;

	/**
	 * 部品編集条件フラグの初期値
	 */
	public static final int NONE = 0;

	/**
	 * 部品の幅を変更可能とするフラグ
	 */
	public static final int RESIZABLE_WIDTH = 0x01;

	/**
	 * 部品の高さを変更可能とするフラグ
	 */
	public static final int RESIZABLE_HEIGHT = 0x02;

	/**
	 * 部品の幅と高さを変更可能とするフラグ
	 */
	public static final int RESIZABLE_BOTH = RESIZABLE_WIDTH | RESIZABLE_HEIGHT;

	/**
	 * 部品のフォーカス制御を可能とするフラグ
	 */
	public static final int NO_FOCUS = 0x04;

	/**
	 * 部品がコンテナであることを示すフラグ
	 */
	public static final int CONTAINER = 0x08;

	/**
	 * 部品がコンポーネントであることを示すフラグ
	 */
	public static final int COMPONENT = 0x10;

	/**
	 * 部品がアイテムであることを示すフラグ
	 */
	public static final int ITEM = 0x20;

	/**
	 * 部品のY座標(top)を変更可能とするフラグ
	 */
	public static final int MOVABLE_TOP = 0x40;

	/**
	 * 部品のX座標(left)を変更可能とするフラグ
	 */
	public static final int MOVABLE_LEFT = 0x80;

	/**
	 * 部品の座標(top, left)が変更可能とするフラグ
	 */
	public static final int MOVABLE_BOTH = MOVABLE_TOP | MOVABLE_LEFT;

	/**
	 * 部品のプロパティにname属性を定義するフラグ
	 */
	public static final int WIDGET_NAME = 0x100;

	/**
	 * この部品が属する部品ライブラリ
	 */
	private ComponentLibrary library;

	/**
	 * この部品が持つイベントタイプ一覧
	 * この部品に定義可能であるイベント名の一覧です。
	 */
	private String[] eventTypes;

	/**
	 * 部品編集条件フラグ
	 */
	private int flag = NONE;

	/**
	 * この部品が持つプロパティのデフォルト値定義
	 * プロパティ名をキーにデフォルト値が定義されています
	 */
	private Map dafaultValues = null;

	/**
	 * この部品に定義可能である子要素リスト
	 * 名前空間をキーにタグ名が定義されています
	 */
	private Map childWidgets = null;

	/**
	 * この部品の完全修飾ドメイン名
	 */
	private String qualifiedName;

	/**
	 * デフォルトコンストラクタです
	 */
	public DynaComponentClass() {
		super();
	}

	/**
	 * コンストラクタです
	 * 
	 * @param name 部品名
	 * @param dynaBeanClass DynaBeanClassクラス
	 * @param properties プロパティ情報
	 */
	public DynaComponentClass(String name, Class dynaBeanClass,
			DynaProperty[] properties) {

		super(name, dynaBeanClass, properties);
		this.dafaultValues = new HashMap();
		this.childWidgets = new HashMap();
	}

	/**
	 * コンストラクタです
	 * 
	 * @param library 部品ライブラリ
	 * @param name 部品名
	 * @param properties プロパティ情報
	 * @param eventTypes イベント情報
	 * @param values デフォルト値情報
	 * @param childWidgets 子要素情報
	 * @param flag 編集条件
	 */
	public DynaComponentClass(ComponentLibrary library, String name,
			DynaProperty[] properties, String[] eventTypes, Map values,
			Map childWidgets, int flag) {

		super(name, null, null);
		this.library = library;
		this.flag = flag;
		if (library.getPrefix() != null) {
			StringBuffer sb = new StringBuffer(library.getPrefix());
			sb.append(":");
			sb.append(name);
			this.qualifiedName = sb.toString();
		}
		this.dafaultValues = (values == null) ? new HashMap() : values;
		this.childWidgets = (childWidgets == null) ? new HashMap()
				: childWidgets;

		setDynaBeanClass(isContainer() ? DynaContainer.class
				: DynaComponent.class);
		setProperties(properties);
		setEventTypes(eventTypes);
	}

    /**
     * {@inheritDoc}
     */
	protected void setProperties(DynaProperty[] properties) {
		List list = new ArrayList();
		list.add(new DynaProperty("parent", MaskatElement.class));
		list.add(new DynaProperty("children", List.class));

		/*
		 * 部品名が定義可能である場合、name属性を追加します
		 */
		if (isWidgetName()) {
			list.add(new DynaProperty("name", String.class));
		}
		/*
		 * topが変更可能である場合、top属性を追加します
		 */
		if (isTopMovable()) {
			list.add(new DynaProperty("top", Integer.class));
		}
		/*
		 * leftが変更可能である場合、left属性を追加します
		 */
		if (isLeftMovable()) {
			list.add(new DynaProperty("left", Integer.class));
		}
		/*
		 * widthが変更可能である場合、width属性を追加します
		 */
		if (isWidthResizable()) {
			list.add(new DynaProperty("width", Integer.class));
		}
		/*
		 * heightが変更可能である場合、height属性を追加します
		 */
		if (isHeightResizable()) {
			list.add(new DynaProperty("height", Integer.class));
		}
		/*
		 * タブフォーカス移動が可能である場合、tabIndex属性を追加します
		 */
		if (canFocus()) {
			list.add(new DynaProperty("tabIndex", Integer.class));
		}
		if (properties != null) {
			for (int i = 0; i < properties.length; i++) {
				list.add(properties[i]);
			}
		}
		super.setProperties((DynaProperty[]) list.toArray(
				new DynaProperty[list.size()]));
	}

	/**
	 * プロパティを追加します
	 * 
	 * @param properties 追加したいプロパティ情報
	 */
	public void setDynaProperties(DynaProperty[] properties) {
		setProperties(properties);
	}

	/**
	 * この部品で定義することができるイベント名を配列で取得します
	 * 
	 * @return イベント名の配列
	 */
	public String[] getEventTypes() {
		return eventTypes;
	}

	/**
	 * この部品に定義されているデフォルト値をマップで取得します
	 * プロパティ名がキーでデフォルト値が格納されています
	 * 
	 * @return デフォルト値マップ
	 */
	public Map getDefaultValues() {
		return dafaultValues;
	}

	/**
	 * この部品に指定された名前空間とタグ名の部品を子要素として持つ
	 * ことができるか判定します
	 * 
	 * @param namespaceURI 名前空間
	 * @param name タグ名
	 * @return 子要素として持つことが出来る場合には true
	 */
	public boolean isAddChildWidget(String namespaceURI, String name) {
		if (this.childWidgets.size() <= 0) {
			return isContainer();
		}
		Map map = (Map) this.childWidgets.get(namespaceURI);
		if (map == null) {
			return false;
		}
		return map.containsKey(name);
	}

	//TODO プロト実装
	public Map getChildWidgetMap() {
		return this.childWidgets;
	}
	
	/**
	 * この部品で定義することができるイベント名を設定します
	 * 
	 * @param eventTypes イベント名の配列
	 */
	protected void setEventTypes(String[] eventTypes) {
		this.eventTypes = eventTypes;
	}

	/**
	 * この部品が属する部品ライブラリを取得します
	 * 
	 * @return 部品ライブラリ
	 */
	public ComponentLibrary getLibrary() {
		return library;
	}

	/**
	 * 指定された数値と編集条件フラグとの論理積を返します
	 * 
	 * @param check 比較したい数値
	 * @return 編集条件フラグとの論理積
	 */
	private boolean isFlagSet(int check) {
		return (flag & check) > 0;
	}

	/**
	 * この部品の幅が編集可能であるか判定します
	 * 
	 * @return 編集可能である場合 true を返します
	 */
	public boolean isWidthResizable() {
		return isFlagSet(RESIZABLE_WIDTH);
	}

	/**
	 * この部品の高さが編集可能であるか判定します
	 * 
	 * @return 編集可能である場合 true を返します
	 */
	public boolean isHeightResizable() {
		return isFlagSet(RESIZABLE_HEIGHT);
	}

	/**
	 * この部品のtopが編集可能であるか判定します
	 * 
	 * @return 編集可能である場合 true を返します
	 */
	public boolean isTopMovable() {
		return isFlagSet(MOVABLE_TOP);
	}

	/**
	 * この部品のleftが編集可能であるか判定します
	 * 
	 * @return 編集可能である場合 true を返します
	 */
	public boolean isLeftMovable() {
		return isFlagSet(MOVABLE_LEFT);
	}

	/**
	 * この部品がコンテナであるか判定します
	 * 
	 * @return コンテナである場合 true を返します
	 */
	public boolean isContainer() {
		return isFlagSet(CONTAINER);
	}

	/**
	 * この部品がコンポーネントであるか判定します
	 * 
	 * @return コンポーネントである場合 true を返します
	 */
	public boolean isComponent() {
		return isFlagSet(COMPONENT);
	}

	/**
	 * この部品がアイテムであるか判定します
	 * 
	 * @return アイテムである場合 true を返します
	 */
	public boolean isItem() {
		return isFlagSet(ITEM);
	}

	/**
	 * この部品の名前が編集可能であるか判定します
	 * 
	 * @return 編集可能である場合 true を返します
	 */
	public boolean isWidgetName() {
		return isFlagSet(WIDGET_NAME);
	}

	/**
	 * この部品のタブフォーカス処理が可能であるか判定します
	 * 
	 * @return 可能である場合 true を返します
	 */
	public boolean canFocus() {
		return !isFlagSet(NO_FOCUS);
	}

	/**
	 * この部品の完全修飾ドメイン名を取得します
	 * 
	 * @return 完全修飾ドメイン名
	 */
	public String getQualifiedName() {
		return this.qualifiedName;
	}

	/**
	 * この部品の完全修飾ドメイン名を設定します
	 * 
	 * @param name 完全修飾ドメイン名
	 */
	public void setqualifiedName(String name) {
		this.qualifiedName = name;
	}
}
