/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

/**
 * DynaBeanのプロパティを表現するためのメタクラスです
 * org.apache.commons.beanutils.DynaPropertyを拡張し以下の情報を保持しています。
 * ・必須
 * ・#cdata-sectionの識別
 * ・enum型のリスト
 * ・編集の有無
 */
public class DynaProperty extends org.apache.commons.beanutils.DynaProperty {

	/**
	 * シリアルバージョンUID
	 */
	private static final long serialVersionUID = -7320819120407974204L;

	/**
	 * enum型に登録されている項目
	 */
	private String[] items;

	/**
	 * 必須の有無
	 */
	private boolean required;

	/**
	 * cdata-sectionの判定
	 */
	private boolean cdataSection;

	/**
	 * 編集の有無
	 */
	private boolean editable = true;

	/**
	 * 有効無効
	 */
	private boolean enabled = true;

	/**
	 *  非推奨
	 */
	private boolean deprecated = false;
	
	/**
	 * コンストラクタです
	 * 
	 * @param name プロパティ名
	 * @param type データ型
	 * @param enabled 有効性
	 */
	public DynaProperty(String name, Class type, boolean enabled) {
		super(name, type);
		this.enabled = enabled;
	}
	
	/**
	 * コンストラクタです
	 * 
	 * @param name プロパティ名
	 * @param type データ型
	 * @param contentType コンテキスト
	 */
	public DynaProperty(String name, Class type, Class contentType) {
		super(name, type, contentType);
	}

	/**
	 * コンストラクタです
	 * 
	 * @param name プロパティ名
	 * @param type データ型
	 */
	public DynaProperty(String name, Class type) {
		super(name, type);
	}

	/**
	 * コンストラクタです
	 * 
	 * @param name プロパティ名
	 */	
	public DynaProperty(String name) {
		super(name);
	}

	/**
	 * データ型に定義されているenum型の項目を配列で取得します
	 * 
	 * @return enum型項目の配列
	 */
	public String[] getItems() {
		return items;
	}

	/**
	 * enum型の項目を設定します
	 * 
	 * @param items enum型項目の配列
	 */
	public void setItems(String[] items) {
		this.items = items;
	}

	/**
	 * このプロパティが必須なのか判定します
	 * 
	 * @return 必須の場合 true が返されます
	 */
	public boolean isRequired() {
		return required;
	}

	/**
	 * このプロパティの必須フラグを設定します
	 * 
	 * @param required 必須にする場合 true を指定します
	 */
	public void setRequired(boolean required) {
		this.required = required;
	}

	/**
	 * このプロパティのデータ型がcdata-sectionなのか判定します
	 * 
	 * @return cdata-sectionの場合 true が返されます
	 */
	public boolean isCdataSection() {
		return cdataSection;
	}

	/**
	 * このプロパティのcata-sectionフラグを設定します
	 * 
	 * @param cdataSection cdata-sectionにする場合 true を指定します
	 */
	public void setCdataSection(boolean cdataSection) {
		this.cdataSection = cdataSection;
	}

	/**
	 * このプロパティが変更可能であるか判定します
	 * 
	 * @return 変更可能である場合 true が返されます
	 */
	public boolean isEditable() {
		return editable;
	}

	/**
	 * このプロパティの編集可能、編集不可能を設定します
	 * 
	 * @param editable 編集可能にする場合 true を指定します
	 */
	public void setEditable(boolean editable) {
		this.editable = editable;
	}

	/**
	 * このプロパティが有効であるか判定します
	 * 
	 * @return 有効である場合 true が返されます
	 */
	public boolean isEnabled() {
		return enabled;
	}

	/**
	 * このプロパティの有効性を設定します
	 * 
	 * @param editable 有効にする場合 true を指定します
	 */
	public void setEnabled(boolean enabled) {
		this.enabled = enabled;
	}

	/**
	 * このプロパティが非推奨であるか判定します
	 * 
	 * @return 非推奨である場合 true が返されます
	 */
	public boolean isDeprecated() {
		return deprecated;
	}

	/**
	 * このプロパティの非推奨属性を設定します
	 * 
	 * @param deprecatedを 有効にする場合 true を指定します
	 */
	public void setDeprecated(boolean deprecated) {
		this.deprecated = deprecated;
	}
}
