/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.wizards.project;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import jp.sf.maskat.ui.ISharedImages;
import jp.sf.maskat.ui.MaskatNature;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.actions.CopyFilesAndFoldersOperation;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.ContainerGenerator;
import org.eclipse.ui.internal.wizards.newresource.ResourceMessages;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;
import org.eclipse.ui.wizards.newresource.BasicNewResourceWizard;


public class NewMaskatProjectWizard extends Wizard implements INewWizard,
		IExecutableExtension {

	/** The wizard ID */
	public static final String WIZARD_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".newMaskatProjectWizard"; //$NON-NLS-1$

	private IWorkbench workbench;

	private NewMaskatProjectCreationPage mainPage;

	/** cache of newly-created project */
	private IProject newProject;

	/** The config element which declares this wizard. */
	private IConfigurationElement configElement;
	
	/**
	 * Creates a wizard for creating a new project resource in the workspace.
	 */
	public NewMaskatProjectWizard() {
		IDialogSettings workbenchSettings = MaskatUIPlugin.getDefault()
				.getDialogSettings();
		IDialogSettings section = workbenchSettings.getSection("NewMaskatProjectWizard"); //$NON-NLS-1$
		if (section == null) {
			section = workbenchSettings.addNewSection("NewMaskatProjectWizard"); //$NON-NLS-1$
		}
		setDialogSettings(section);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbenchWizard#init(org.eclipse.ui.IWorkbench,
	 *      org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection currentSelection) {
		this.workbench = workbench;
		setNeedsProgressMonitor(true);
		setWindowTitle(Messages.getString("wizard.project.windowTitle")); //$NON-NLS-1$
		setDefaultPageImageDescriptor(MaskatUIPlugin
				.getImageDescriptor(ISharedImages.IMG_WIZBAN_NEWPROJECT));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.runtime.IExecutableExtension#setInitializationData(org.eclipse.core.runtime.IConfigurationElement,
	 *      java.lang.String, java.lang.Object)
	 */
	public void setInitializationData(IConfigurationElement config, String propertyName,
			Object data) {
		configElement = config;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.Wizard#addPages()
	 */
	public void addPages() {
		super.addPages();

		mainPage = new NewMaskatProjectCreationPage(
				new Path(MaskatNature.FRAMEWORK_PATH_DEFAULT), true);
		
		addPage(mainPage);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	public boolean performFinish() {
		if (createNewProject() == null) {
			return false;
		}

		BasicNewProjectResourceWizard.updatePerspective(configElement);
		BasicNewResourceWizard.selectAndReveal(newProject, workbench
				.getActiveWorkbenchWindow());
		return true;
	}

	/**
	 * Creates a new project resource with the selected name.
	 * <p>
	 * In normal usage, this method is invoked after the user has pressed Finish
	 * on the wizard; the enablement of the Finish button implies that all
	 * controls on the pages currently contain valid values.
	 * </p>
	 * <p>
	 * Note that this wizard caches the new project once it has been
	 * successfully created; subsequent invocations of this method will answer
	 * the same project resource without attempting to create it again.
	 * </p>
	 * 
	 * @return the created project resource, or <code>null</code> if the
	 *         project was not created
	 */
	private IProject createNewProject() {
		if (newProject != null) {
			return newProject;
		}

		// get a project handle
		final IProject newProjectHandle = mainPage.getProjectHandle();

		// get a project descriptor
		URI location = null;
		if (!mainPage.useDefaults()) {
			location = mainPage.getLocationURI();
		}

		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		final IProjectDescription description = workspace
				.newProjectDescription(newProjectHandle.getName());
		description.setLocationURI(location);
		
		final IPath fwPath = mainPage.getMaskatFrameworkPath();
		final boolean copy = mainPage.shouldCopyMaskatFramework();
		final PluginResourceData[] plugins = mainPage.getSelectedPlugins();
		
		// create the new project operation
		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor) throws CoreException {
				createProject(description, newProjectHandle, monitor, fwPath, copy, plugins);
			}
		};

		// run the new project creation operation
		try {
			getContainer().run(true, true, op);
		} catch (InterruptedException e) {
			return null;
		} catch (InvocationTargetException e) {
			// ie.- one of the steps resulted in a core exception
			Throwable t = e.getTargetException();
			if (t instanceof CoreException) {
				if (((CoreException) t).getStatus().getCode() == IResourceStatus.CASE_VARIANT_EXISTS) {
					MessageDialog.openError(getShell(),
							ResourceMessages.NewProject_errorMessage, NLS.bind(
									ResourceMessages.NewProject_caseVariantExistsError,
									newProjectHandle.getName()));
				} else {
					ErrorDialog.openError(getShell(),
							ResourceMessages.NewProject_errorMessage, null, // no
							// special
							// message
							((CoreException) t).getStatus());
				}
			} else {
				// CoreExceptions are handled above, but unexpected runtime
				// exceptions and errors may still occur.
				MaskatUIPlugin.log(new Status(IStatus.ERROR, MaskatUIPlugin.PLUGIN_ID, 0,
						t.toString(), t));
				MessageDialog.openError(getShell(),
						ResourceMessages.NewProject_errorMessage, NLS
								.bind(ResourceMessages.NewProject_internalError, t
										.getMessage()));
			}
			return null;
		}

		newProject = newProjectHandle;

		return newProject;
	}

	/**
	 * Creates a project resource given the project handle and description.
	 * 
	 * @param description
	 *            the project description to create a project resource for
	 * @param projectHandle
	 *            the project handle to create a project resource for
	 * @param monitor
	 *            the progress monitor to show visual progress with
	 * 
	 * @exception CoreException
	 *                if the operation fails
	 * @exception OperationCanceledException
	 *                if the operation is canceled
	 */
	private void createProject(IProjectDescription description, IProject projectHandle,
			IProgressMonitor monitor, IPath frameWorkPath, boolean shouldCopy,
			PluginResourceData[] plugins)
				throws CoreException, OperationCanceledException {
		try {
			monitor.beginTask("", 4000);//$NON-NLS-1$

			projectHandle.create(description, new SubProgressMonitor(monitor, 1000));
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}

			projectHandle.open(IResource.BACKGROUND_REFRESH, new SubProgressMonitor(
					monitor, 1000));
			
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}

			/* Add Maskat nature to the created project */
			try {
				MaskatNature.addNature(projectHandle, new SubProgressMonitor(monitor, 1000));
				MaskatNature nature = MaskatNature.getNature(projectHandle);
				nature.setMaskatFrameworkPath(frameWorkPath);
				nature.getPreferenceStore().save();
			} catch (IOException e) {
				throw new CoreException(new Status(IStatus.ERROR,
						MaskatUIPlugin.PLUGIN_ID, 0, e.getMessage(), e)); 
			}
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}

			/* Create folder for Maskat framework resources */
			IPath destinationPath = projectHandle.getFullPath().append(frameWorkPath);
			ContainerGenerator generator = new ContainerGenerator(destinationPath);
			IContainer destination = generator.generateContainer(new SubProgressMonitor(
					monitor, 1000));
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}
			if (shouldCopy) {
				Map properties = new LinkedHashMap();
				StringBuffer installPlugins = new StringBuffer();
				for (int i = 0; i < plugins.length; i++) {
					File[] contents = new File(plugins[i].getUrl().getFile()).listFiles();
					URI[] uris = new URI[contents.length];
					for (int j = 0; j < contents.length; j++) {
						uris[j] = contents[j].toURI();
					}
					CopyFilesAndFoldersOperation op = new CopyFilesAndFoldersOperation(
							getShell());
					op.copyFilesInCurrentThread(uris, destination, new SubProgressMonitor(
							monitor, 1000 / plugins.length));
					
					if (plugins[i].getInitializer() != null) {
						plugins[i].getInitializer().initialize(properties, true);
					}
					if (i != 0) {
						installPlugins.append(",");
					}
					installPlugins.append(plugins[i].getUniqueId());
				}
				try {
					MaskatNature nature = MaskatNature.getNature(projectHandle);
					nature.setInstallWidgetPlugins(installPlugins.toString());
					nature.getPreferenceStore().save();
					
				} catch (IOException e) {
					throw new CoreException(new Status(IStatus.ERROR,
							MaskatUIPlugin.PLUGIN_ID, 0, e.getMessage(), e)); 
				}				
				if (!properties.isEmpty()) {
					InputStream in = null;
					try {
						IFile file = destination.getFile(
								new Path("properties.json")); //$NON-NLS-1$

						in = new ByteArrayInputStream(
								encodeJson(properties, 0).getBytes("UTF-8")); //$NON-NLS-1$

						if (file.exists()) {
							file.delete(true, monitor);
						}
						file.create(in, true, monitor);
						
					} catch (UnsupportedEncodingException e) {
						throwCoreException(e.getMessage());
					} finally {
						if (in != null) {
							try {
								in.close();
							} catch (IOException e) { /* suppress */ }
						}
					}
				}
			}
		} finally {
			monitor.done();
		}
	}

	/**
	 * Returns the newly created project.
	 * 
	 * @return the created project, or <code>null</code> if project not
	 *         created
	 */
	public IProject getNewProject() {
		return newProject;
	}

	private String encodeJson(Map map, int indent) {
		String separator = System.getProperty("line.separator"); //$NON-NLS-1$
		StringBuffer sb = new StringBuffer("{"); //$NON-NLS-1$
		sb.append(separator);
		String indentString = repeat("\t", indent + 1); //$NON-NLS-1$
		for (Iterator ite = map.entrySet().iterator(); ite.hasNext();) {
			Map.Entry entry = (Map.Entry) ite.next();
			sb.append(indentString);
			sb.append("\"").append(entry.getKey()).append("\": "); //$NON-NLS-1$
			Object value = entry.getValue();
			if (value instanceof String) {
				sb.append("\"").append(value).append("\""); //$NON-NLS-1$
			} else if (value instanceof Map) {
				sb.append(encodeJson((Map) value, indent + 1));
			} else {
				sb.append(value);			 
			}
			if (ite.hasNext()) {
				sb.append(","); //$NON-NLS-1$
			}
			sb.append(separator);
		}
		return sb.append(repeat("\t", indent)).append("}").toString(); //$NON-NLS-1$
	}
	
	private String repeat(String str, int repeat) {
		StringBuffer sb = new StringBuffer();
		if (str != null && repeat > 0) {
			for (int i = 0; i < repeat; i++) {
				sb.append(str);
			}
		}
		return sb.toString();
	}

	private void throwCoreException(String message) throws CoreException {
		IStatus status = new Status(IStatus.ERROR, MaskatUIPlugin.PLUGIN_ID, IStatus.OK,
				message, null);
		throw new CoreException(status);
	}
}
