/*
 * Copyright (C) 2000-2005 ASANO Masahiro
 */

#include <unistd.h>
#include "crash.h"

#include "flags_fs.h"
#include "flags_mm.h"

PRIVATE addr_t operations();
const commandtable_t command_operations =
	{"operations", operations, "{-a|-b|-d|-e|-f|-i|-n|-s|-q|-v} address", "print address_space, block_device, dentry, export, file, inode, nfs_rpc, super, dquot or vm operations"};

PRIVATE const char *address_space_op[] = {
	"ADDRESS SPACE OPERATIONS",
	"writepage", "readpage", "sync_page",
	"writepages", "set_page_dirty", "readpages",
	"prepare_write", "commit_write", "bmap", "invalidatepage",
	"releasepage", "direct_IO",
	NULL
};

PRIVATE const char *block_device_op[] = {
	"BLOCK DEVICE OPERATIONS",
	"open", "release", "ioctl",
#ifdef _block_device_operations_has_compat_ioctl
	"compat_ioctl",
#endif
	"media_changed", "revalidate_disk",
	"owner",
	NULL
};

PRIVATE const char *dentry_op[] = {
	"DENTRY OPERATIONS",
	"revalidate", "hash", "compare", "delete", "release", "iput",
	NULL
};

PRIVATE const char *file_op[] = {
	"FILE OPERATIONS",
	"owner",
	"llseek", "read", "aio_read", "write", "aio_write",
	"readdir", "poll", "ioctl",
#ifdef _file_operations_has_unlocked_ioctl
	"unlocked_ioctl",
#endif
#ifdef _file_operations_has_compat_ioctl
	"compat_ioctl",
#endif
	"mmap",
	"open", "flush", "release", "fsync", "aio_fsync", "fasync", "lock",
	"readv", "writev",
	"sendfile", "sendpage", "get_unmapped_area",
#ifdef _file_operations_has_fcntl
	"fcntl",
#endif
#ifdef _file_operations_has_check_flags
	"check_flags", "dir_notify",
#endif
#ifdef _file_operations_has_flock
	"flock",
#endif
	NULL
};

PRIVATE const char *inode_op[] = {
	"INODE OPERATIONS",
	"create", "lookup", "link", "unlink", "symlink",
	"mkdir", "rmdir", "mknod", "rename", "readlink", "follow_link",
#ifdef _file_operations_has_fcntl
	"put_link",
#endif
	"truncate", "permission", "setattr", "getattr",
	"setxattr", "getxattr", "listxattr", "removexattr",
	NULL
};

PRIVATE const char *nfs_rpc_op[] = {
	"NFS_RPC OPERATIONS",
	"version",
#ifdef _nfs_rpc_ops_has_dentry_ops
	"dentry_ops",
#endif
#ifdef _nfs_rpc_ops_has_dir_inode_ops
	"dir_inode_ops",
#endif
	"getroot", "getattr", "setattr", "lookup", "access",
	"readlink", "read", "write", "commit", "create", "remove",
	"unlink_setup", "unlink_done", "rename", "link", "symlink",
	"mkdir", "rmdir", "readdir", "mknod", "statfs", "fsinfo", "pathconf",
	"decode_dirent", "read_setup", "write_setup", "commit_setup",
	"file_open",
#ifdef _nfs_rpc_ops_has_file_release
	"file_release",
#endif
#ifdef _nfs_rpc_ops_has_lock
	"lock",
#endif
	NULL
};

PRIVATE const char *super_op[] = {
	"SUPER OPERATIONS",
	"alloc_inode", "destroy_inode",
	"read_inode", "dirty_inode", "write_inode", "put_inode", "drop_inode",
	"delete_inode", "put_super", "write_super", "sync_fs",
	"write_super_lockfs", "unlockfs", "statfs", "remount_fs",
       	"clear_inode", "umount_begin", "show_options",
#ifdef _super_operations_has_quota_read
	"quota_read",
#endif
#ifdef _super_operations_has_quota_write
	"quota_write",
#endif
	NULL
};

PRIVATE const char *dquot_op[] = {
	"DQUOT OPERATIONS",
	"initialize", "drop", "alloc_space", "alloc_inode",
	"free_space", "free_inode", "transfer", "sync_dquot",
	NULL
};

PRIVATE const char *export_op[] = {
	"EXPORT OPERATIONS",
	"decode_fh", "encode_fh", "get_name", "get_parent", "get_dentry",
	"find_exported_dentry",
	NULL
};

PRIVATE const char *vm_op[] = {
	"VM OPERATIONS STRUCT",
	"open", "close", "nopage", "populate",
#ifdef _vm_operations_struct_has_set_policy
	"set_policy", "get_policy",
#endif
	NULL
};

const char head_operations[] = "%24s:   ADDRESS  SYMBOL\n";

void
print_operations(addr, op)
	addr_t addr;
	const char **op;
{
	addr_t buf[32];
	const char *p;
	int i;

	memread(addr, sizeof(buf), buf, "operations");
	mprintf(head_operations, op[0]);
	for (i = 1; op[i]; i++) {
		p = getsymstr_func(buf[i - 1]);
		mprintf("%24s:  " FPTR "  %s\n", op[i], buf[i - 1], p? p: "-");
	}
}

PRIVATE addr_t
operations()
{
	int i, c;
	const char **op = NULL;
	addr_t addr;

	while ((c = getopt(argcnt, args, "abdefinqsv")) != EOF) {
		switch (c) {
		case 'a':
			op = address_space_op;
			break;
		case 'b':
			op = block_device_op;
			break;
		case 'e':
			op = export_op;
			break;
		case 'f':
			op = file_op;
			break;
		case 'i':
			op = inode_op;
			break;
		case 'n':
			op = nfs_rpc_op;
			break;
		case 's':
			op = super_op;
			break;
		case 'q':
			op = dquot_op;
			break;
		case 'd':
			op = dentry_op;
			break;
		case 'v':
			op = vm_op;
			break;
		default:
			THROW(usage);
		}
	}
	if (op == NULL || optind == argcnt) {
		THROW(usage);
	}

	for (i = optind; i < argcnt; i++) {
		addr = getvalue(args[i]);
		print_operations(addr, op);
	}
	return 0;
}
