/*
 * Copyright (C) 2000-2003 ASANO Masahiro
 */

#include <unistd.h>
#include <string.h>
#include "crash.h"

PRIVATE addr_t pte(), pgd();
struct commandtable command_pte =
	{"pte", pte, "pte-addr", "format page-table-entry"};
struct commandtable command_pgd =
	{"pgd", pgd, "[pgd-addr]", "format page-table-entry"};

PRIVATE addr_t vtop();
struct commandtable command_vtop =
	{"vtop", vtop, "vaddress", "virtual to physical address\nuse physical memory interface such as `mcrash -p /dev/mem'"};

#define PAGE_SIZE	4096ul

typedef unsigned long	pte_t;
#define	PML4E	3
#define	PDPE	2
#define	PDE	1
#define	PTE	0
PRIVATE pte_t pte_buffer[4][PAGE_SIZE / sizeof(pte_t)];
PRIVATE addr_t pte_cache[4] = { -1, -1, -1, -1 };

PRIVATE void
format_pte(pte_t val)
{
	if (val & (1<<0)) {
		mprintf("%08lx %c%d%c%c%c%c%c%c",
			(val >> 12)&((1L<<40)-1),
			val & (1ul<<63)? 'X': '-',
			(int)(val >> 9) & 7,
			val & (1<<7)? 'M': '-',
			val & (1<<5)? 'A': '-',
			val & (1<<4)? 'C': '-',
			val & (1<<3)? 'T': '-',
			val & (1<<2)? 'U': 'S',
			val & (1<<1)? 'W': 'R');
	} else {
		mprintf("-------- --------");
	}
}

PRIVATE void
format_page_table(buf)
	const pte_t *buf;
{
	int i;
	int count = PAGE_SIZE / sizeof(pte_t);

	for (i = 0; ; i++) {
		if ((i & 3) == 0) {
			int lp = 0;
			while (i && memcmp(buf + i, buf + i - 4, sizeof(pte_t) * 4) == 0) {
				if (lp == 0) {
					mprintf("*\n");
					lp = 1;
				}
				i += 4;
				if (i >= count)
					break;
			}
			if (i >= count)
				break;
			mprintf("%02x:  ", i / 4);
		}
		format_pte(buf[i]);
		if ((i & 3) == 3) {
			mprintf("\n");
		} else {
			mprintf("  ");
		}
	}
}

PRIVATE addr_t
pte()
{
	int c;
	int vflag = 0;

	while ((c = getopt(argcnt, args, "v")) != EOF) {
		switch (c) {
		case 'v':
			vflag = 1;
			break;
		default:
			THROW(usage);
		}
	}

	if (vflag) {
		while (args[optind]) {
			format_pte(getvalue(args[optind++]));
			mprintf("\n");
		}
		return 0;
	}

	if (argcnt == optind + 1) {
		char buf[PAGE_SIZE];
		addr_t addr = getvalue(args[optind]);
		if (addr & 15) {
			THROW("address unaligned");
		}
		memread(addr, sizeof(buf), buf, "PTE");
		format_page_table(buf);
	} else {
		THROW(usage);
	}

	return 0;
}

PRIVATE addr_t
pgd()
{
	int c;
	int vflag = 0;

	while ((c = getopt(argcnt, args, "v")) != EOF) {
		switch (c) {
		case 'v':
			vflag = 1;
			break;
		default:
			THROW(usage);
		}
	}

	if (vflag) {
		while (args[optind]) {
			format_pte(getvalue(args[optind++]));
			mprintf("\n");
		}
		return 0;
	}

	if (argcnt == optind) {
		if (pte_cache[PML4E] == (addr_t)-1)
			THROW("PGD not loaded");
		mprintf("PGD " FPTR "\n", pte_cache[PML4E]);
		format_page_table(pte_buffer[PML4E]);
	} else if (argcnt == optind + 1) {
		pte_t pgd_buf[PAGE_SIZE / sizeof(pte_t)];
		addr_t addr = getvalue(args[optind]);
		if (addr & 15) {
			THROW("address unaligned");
		}
		memread(addr, sizeof(pgd_buf), pgd_buf, "PGD");
		format_page_table(pgd_buf);
	} else {
		THROW(usage);
	}

	return 0;
}

void
load_pgd(addr)
	addr_t addr;
{
	ll_memread(_PA(addr), PAGE_SIZE, pte_buffer[PML4E], "pgd");
	pte_cache[PML4E] = addr;
	pte_cache[PDPE] = (addr_t)-1;
	pte_cache[PDE] = (addr_t)-1;
	pte_cache[PTE] = (addr_t)-1;
}

PRIVATE const pte_t *
load_pte(addr, level)
	addr_t addr;
	int level;
{
	if (pte_cache[level] != addr) {
		int i;
		ll_memread(addr, PAGE_SIZE, pte_buffer[level], "pte");
		pte_cache[level] = addr;
		for (i = level - 1; i >= PTE; i--) {
			pte_cache[i] = (addr_t)-1;
		}
	}
	return pte_buffer[level];
}

addr_t
translate_addr(addr, count, errmsg)
	addr_t addr;
	addr_t *count;
	const char *errmsg;
{
	/* Long Mode Page Translation */
	pte_t e;
	addr_t p;
	int sign;

	sign = (addr >> 48) & 63;
	if (sign != 0 && sign != 63) {
		THROWF2("Error: Not a Canonical Address Form for %s address %lx", errmsg, addr);
	}

	e = pte_buffer[PML4E][(addr >> 39) & 511];
	if ((e & (1<<0)) == 0)
		THROWF2("Error: PTE page not presented L4 for %s address %lx",
			errmsg, addr);
	p = e & (~(PAGE_SIZE - 1) & ~(1ul << 63));

	e = (load_pte(p, PDPE))[(addr >> 30) & 511];
	if ((e & (1<<0)) == 0)
		THROWF2("Error: PTE page not presented L3 for %s address %lx",
			errmsg, addr);
	p = e & (~(PAGE_SIZE - 1) & ~(1ul << 63));

	e = (load_pte(p, PDE))[(addr >> 21) & 511];
	if ((e & (1<<0)) == 0)
		THROWF2("Error: PTE page not presented L2 for %s address %lx",
			errmsg, addr);

	if (e & (1<<7)) {
		/* 2-Mbyte Page Translation - Long Mode */
		*count = 2*1024*1024 - (addr & (2*1024*1024-1));
		p = e & (~(2*1024*1024 - 1) & ~(1ul << 63));
		return p | (addr & (2*1024*1024 - 1));
	}

	/* 4-Kbyte Page Translation - Long Mode */
	p = e & (~(PAGE_SIZE - 1) & ~(1ul << 63));
		
	e = (load_pte(p, PTE))[(addr >> 12) & 511];
	if ((e & (1<<0)) == 0)
		THROWF2("Error: PTE page not presented L1 for %s address %lx",
			errmsg, addr);
	p = e & (~(PAGE_SIZE - 1) & ~(1ul << 63));

	*count = PAGE_SIZE - (addr & (PAGE_SIZE - 1));
	return p | (addr & (PAGE_SIZE - 1));
}

PRIVATE addr_t
vtop()
{
	int c;
	addr_t vaddr = 0, paddr = 0, dummy;

	while ((c = getopt(argcnt, args, "")) != EOF) {
		switch (c) {
		default:
			THROW(usage);
		}
	}

	if (argcnt == optind)
		THROW(usage);
	if (pte_cache[PML4E] == (addr_t)-1)
		THROW("PGD not loaded");

	mprintf(SPTR "  " SPTR "\n", "VIRTUAL", "PHYSICAL");
	while (args[optind]) {
		vaddr = getvalue(args[optind++]);
		paddr = translate_addr(vaddr, &dummy, "vtop");
		mprintf(FPTR "  " FPTR "\n", vaddr, paddr);
	}
	return paddr;
}
