#ifndef __MERCURY_REGEX_PARSE_NODE__
#define __MERCURY_REGEX_PARSE_NODE__

#include <cstdarg>
#include <utility>
#include <list>
#include "lexer.h"
#include "../parse_tree.h"


namespace mercury
{
	namespace _regex
	{
		using namespace std::rel_ops;


		template<typename _Input>
		class parse_node_element_set : public parse_tree<_Input>
		{
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_not : public parse_node_element_set<_Input>
		{
		public:
			parse_node_not(parse_node_element_set<_Input> *node)
			{
				m_node = node;
			}

			~parse_node_not(void)
			{
				delete m_node;
				m_node = NULL;
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				nfa<_Input> frag = m_node->build_nfa(state);
				frag.invert_accepts();

				return frag;
			}

		private:
			parse_node_element_set<_Input> *m_node;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_set : public parse_node_element_set<_Input>
		{
		public:
			parse_node_set(parse_node_element_set<_Input> *node, ...)
			{
				va_list ap;
				va_start(ap, node);

				parse_node_element_set_ptr_t arg = node;
				do
				{
					m_nodes.push_back(arg);
					arg = va_arg(ap, parse_node_element_set_ptr_t);
				}
				while(arg != NULL);
				va_end(ap);
			}

			~parse_node_set(void)
			{
				for(parse_node_element_set_ptr_list_const_iterator_t p = m_nodes.begin(); p != m_nodes.end(); p++)
				{
					parse_node_element_set_ptr_t node = *p;
					delete node;
				}
				m_nodes.clear();
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const state_t start     = state++;
				const state_t accept    = state++;
				const state_t no_accept = state++;

				nfa<_Input> frag(start);
				frag.add_accept   (   accept);
				frag.add_no_accept(no_accept);
				frag.connect_default(start, no_accept);

				for(parse_node_element_set_ptr_list_const_iterator_t p = m_nodes.begin(); p != m_nodes.end(); p++)
				{
					state_t substate = start;
					nfa<_Input> frag_x = (*p)->build_nfa(substate);

					frag.merge_transition(frag_x);
				}
				return frag;
			}

		private:
			typedef parse_node_element_set<_Input>                            *parse_node_element_set_ptr_t;
			typedef typename std::list<parse_node_element_set_ptr_t>           parse_node_element_set_ptr_list_t;
			typedef typename parse_node_element_set_ptr_list_t::const_iterator parse_node_element_set_ptr_list_const_iterator_t;
			parse_node_element_set_ptr_list_t m_nodes;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_character : public parse_node_element_set<_Input>
		{
		public:
			parse_node_character(const _Input &ch) : m_character(ch)
			{
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const state_t start     = state++;
				const state_t accept    = state++;
				const state_t no_accept = state++;
				nfa<_Input> frag(start);

				frag.add_accept(accept);
				frag.connect(start, accept, m_character);

				frag.add_no_accept(no_accept);
				frag.connect_default(start, no_accept);

				return frag;
			}

		private:
			_Input m_character;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_class : public parse_node_element_set<_Input>
		{
		public:
			parse_node_class(const CHARCLASS &charclass, const bool invert) : m_charclass(charclass), m_invert(invert)
			{
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const _Input underscore = _Traits::char_underscore();

				const struct
				{
					CHARCLASS     type;
					bool          invert;
					size_t        size;
					const _Input *inputs;
				}
				info[] =
				{
					{CHARCLASS_UPPER , false, _Traits::class_upper_count       (), _Traits::class_upper       ()},
					{CHARCLASS_LOWER , false, _Traits::class_lower_count       (), _Traits::class_lower       ()},
					{CHARCLASS_DIGIT , false, _Traits::class_digit_count       (), _Traits::class_digit       ()},
					{CHARCLASS_XDIGIT, false, _Traits::class_xdigit_count      (), _Traits::class_xdigit      ()},
					{CHARCLASS_PUNCT , false, _Traits::class_punct_count       (), _Traits::class_punct       ()},
					{CHARCLASS_BLANK , false, _Traits::class_blank_count       (), _Traits::class_blank       ()},
					{CHARCLASS_SPACE , false, _Traits::class_space_count       (), _Traits::class_space       ()},
					{CHARCLASS_CNTRL , false, _Traits::class_cntrl_count       (), _Traits::class_cntrl       ()},
					{CHARCLASS_GRAPH ,  true, _Traits::class_graph_invert_count(), _Traits::class_graph_invert()},
					{CHARCLASS_PRINT ,  true, _Traits::class_print_invert_count(), _Traits::class_print_invert()},
					{CHARCLASS_UNDER , false,                                   1, &underscore},
				};

				const state_t start     = state++;
				const state_t accept    = state++;
				const state_t no_accept = state++;

				nfa<_Input> frag(start);
				frag.add_accept   (   accept);
				frag.add_no_accept(no_accept);

				for(size_t i = 0; i < mercury_countof(info); i++)
				{
					if(m_charclass & info[i].type)
					{
						const size_t  size   = info[i].size;
						const _Input *inputs = info[i].inputs;
						if(!(info[i].invert ^ m_invert))
						{
							frag.connect(start, accept, &inputs[0], &inputs[size]);
						}
						else
						{
							frag.connect(start, no_accept, &inputs[0], &inputs[size]);
							frag.connect_default(start, accept);
						}
					}
				}

				return frag;
			}

		private:
			CHARCLASS m_charclass;
			bool      m_invert;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_range : public parse_node_element_set<_Input>
		{
		public:
			parse_node_range(const _Input &char1, const _Input &char2) : m_char1(char1), m_char2(char2)
			{
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const state_t start     = state++;
				const state_t accept    = state++;
				const state_t no_accept = state++;

				nfa<_Input> frag(start);
				frag.add_accept   (   accept);
				frag.add_no_accept(no_accept);

				for(_Input ch = m_char1; ch <= m_char2; ch = _Traits::next(ch))
				{
					frag.connect(start, accept, ch);
				}
				frag.connect_default(start, no_accept);

				return frag;
			}

		private:
			_Input m_char1, m_char2;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_none : public parse_tree<_Input>
		{
		public:
			parse_node_none(void)
			{
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const state_t start = state++;
				nfa<_Input> frag(start);

				frag.add_accept(start);
				return frag;
			}
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_any : public parse_tree<_Input>
		{
		public:
			parse_node_any(void)
			{
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const state_t start  = state++;
				const state_t accept = state++;
				nfa<_Input> frag(start);
				frag.add_accept(accept);

				frag.connect_default(start, accept);

				return frag;
			}
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_conjunction : public parse_tree<_Input>
		{
		public:
			parse_node_conjunction(parse_tree<_Input> *node, ...)
			{
				va_list ap;
				va_start(ap, node);

				parse_node_ptr_t arg = node;
				do
				{
					m_nodes.push_back(arg);
					arg = va_arg(ap, parse_node_ptr_t);
				}
				while(arg != NULL);
				va_end(ap);
			}

			~parse_node_conjunction(void)
			{
				for(parse_node_ptr_list_const_iterator_t p = m_nodes.begin(); p != m_nodes.end(); p++)
				{
					parse_node_ptr_t node = *p;
					delete node;
				}
				m_nodes.clear();
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const state_t start  = state++;
				const state_t accept = state++;
				nfa<_Input> frag(start);
				frag.add_accept(accept);
				frag.connect_epsilon(start, accept);

				for(parse_node_ptr_list_const_iterator_t p = m_nodes.begin(); p != m_nodes.end(); p++)
				{
					nfa<_Input> frag_x = (*p)->build_nfa(state);

					const state_set_t accepts = frag.get_accepts();
					for(state_set_t::const_iterator p2 = accepts.begin(); p2 != accepts.end(); p2++)
					{
						frag.connect_epsilon(*p2, frag_x.get_start());
					}

					frag.merge_transition(frag_x);

					frag.set_accepts(frag_x.get_accepts());
				}
				return frag;
			}

		private:
			typedef parse_tree<_Input>                            *parse_node_ptr_t;
			typedef std::list<parse_node_ptr_t>                    parse_node_ptr_list_t;
			typedef typename parse_node_ptr_list_t::const_iterator parse_node_ptr_list_const_iterator_t;
			parse_node_ptr_list_t m_nodes;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_disjunction : public parse_tree<_Input>
		{
		public:
			parse_node_disjunction(parse_tree<_Input> *node, ...)
			{
				va_list ap;
				va_start(ap, node);

				parse_node_ptr_t arg = node;
				do
				{
					m_nodes.push_back(arg);
					arg = va_arg(ap, parse_node_ptr_t);
				}
				while(arg != NULL);
				va_end(ap);
			}

			~parse_node_disjunction(void)
			{
				for(parse_node_ptr_list_const_iterator_t p = m_nodes.begin(); p != m_nodes.end(); p++)
				{
					parse_node_ptr_t node = *p;
					delete node;
				}
				m_nodes.clear();
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				const state_t start = state++;
				nfa<_Input> frag(start);

				for(parse_node_ptr_list_const_iterator_t p = m_nodes.begin(); p != m_nodes.end(); p++)
				{
					nfa<_Input> frag_x = (*p)->build_nfa(state);

					frag.merge_transition(frag_x);
					frag.merge_accepts   (frag_x);

					frag.connect_epsilon(start, frag_x.get_start());
				}
				return frag;
			}

		private:
			typedef parse_tree<_Input>                            *parse_node_ptr_t;
			typedef std::list<parse_node_ptr_t>                    parse_node_ptr_list_t;
			typedef typename parse_node_ptr_list_t::const_iterator parse_node_ptr_list_const_iterator_t;
			parse_node_ptr_list_t m_nodes;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_repeat0 : public parse_tree<_Input>
		{
		public:
			parse_node_repeat0(parse_tree<_Input> *node)
			{
				m_node = node;
			}

			~parse_node_repeat0(void)
			{
				delete m_node;
				m_node = NULL;
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				nfa<_Input> frag = m_node->build_nfa(state);

				const state_t     start   = frag.get_start();
				const state_set_t accepts = frag.get_accepts();
				for(state_set_t::const_iterator p = accepts.begin(); p != accepts.end(); p++)
				{
					frag.connect_epsilon(*p, start);
				}

				frag.add_accept(start);
				return frag;
			}

		private:
			parse_tree<_Input> *m_node;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_repeat1 : public parse_tree<_Input>
		{
		public:
			parse_node_repeat1(parse_tree<_Input> *node)
			{
				m_node = node;
			}

			~parse_node_repeat1(void)
			{
				delete m_node;
				m_node = NULL;
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				nfa<_Input> frag = m_node->build_nfa(state);

				const state_t     start   = frag.get_start();
				const state_set_t accepts = frag.get_accepts();
				for(state_set_t::const_iterator p = accepts.begin(); p != accepts.end(); p++)
				{
					frag.connect_epsilon(*p, start);
				}

				return frag;
			}

		private:
			parse_tree<_Input> *m_node;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_repeat01 : public parse_tree<_Input>
		{
		public:
			parse_node_repeat01(parse_tree<_Input> *node)
			{
				m_node = node;
			}

			~parse_node_repeat01(void)
			{
				delete m_node;
				m_node = NULL;
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				nfa<_Input> frag = m_node->build_nfa(state);

				frag.add_accept(frag.get_start());
				return frag;
			}

		private:
			parse_tree<_Input> *m_node;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_repeatn : public parse_tree<_Input>
		{
		public:
			parse_node_repeatn(parse_tree<_Input> *node, const size_t number)
			{
				m_node   = node;
				m_number = number;
			}

			~parse_node_repeatn(void)
			{
				delete m_node;
				m_node = NULL;
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				nfa<_Input> frag(state++);

				state_set_t accepts;
				accepts.insert(frag.get_start());

				for(size_t i = 0; i < m_number; i++)
				{
					nfa<_Input> frag_x = m_node->build_nfa(state);
					frag.merge_transition(frag_x);

					const state_t start = frag_x.get_start();
					for(state_set_t::const_iterator p = accepts.begin(); p != accepts.end(); p++)
					{
						frag.connect_epsilon(*p, start);
					}
					accepts = frag_x.get_accepts();
				}

				frag.set_accepts(accepts);
				return frag;
			}

		private:
			parse_tree<_Input> *m_node;
			size_t              m_number;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_repeatn_ : public parse_tree<_Input>
		{
		public:
			parse_node_repeatn_(parse_tree<_Input> *node, const size_t number)
			{
				m_node   = node;
				m_number = number;
			}

			~parse_node_repeatn_(void)
			{
				delete m_node;
				m_node = NULL;
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				nfa<_Input> frag(state++);

				state_t start = STATE_ERROR;
				state_set_t accepts;
				accepts.insert(frag.get_start());

				for(size_t i = 0; i < m_number; i++)
				{
					nfa<_Input> frag_x = m_node->build_nfa(state);
					frag.merge_transition(frag_x);

					start = frag_x.get_start();
					for(state_set_t::const_iterator p = accepts.begin(); p != accepts.end(); p++)
					{
						frag.connect_epsilon(*p, start);
					}
					accepts = frag_x.get_accepts();
				}

				for(state_set_t::const_iterator p = accepts.begin(); p != accepts.end(); p++)
				{
					frag.connect_epsilon(*p, start);
				}

				frag.set_accepts(accepts);
				return frag;
			}

		private:
			parse_tree<_Input> *m_node;
			size_t              m_number;
		};


		template<typename _Input, typename _Traits = regex_traits<_Input> >
		class parse_node_repeatmn : public parse_tree<_Input>
		{
		public:
			parse_node_repeatmn(parse_tree<_Input> *node, const size_t number1, const size_t number2)
			{
				m_node    = node;
				m_number1 = number1;
				m_number2 = number2;
			}

			~parse_node_repeatmn(void)
			{
				delete m_node;
				m_node = NULL;
			}

			nfa<_Input> build_nfa(state_t &state) const
			{
				nfa<_Input> frag(state++);

				state_set_t accepts;
				accepts.insert(frag.get_start());

				for(size_t i = 0; i < m_number1; i++)
				{
					nfa<_Input> frag_x = m_node->build_nfa(state);
					frag.merge_transition(frag_x);

					const state_t start = frag_x.get_start();
					for(state_set_t::const_iterator p = accepts.begin(); p != accepts.end(); p++)
					{
						frag.connect_epsilon(*p, start);
					}
					accepts = frag_x.get_accepts();
				}

				for(size_t i = m_number1; i < m_number2; i++)
				{
					nfa<_Input> frag_x = m_node->build_nfa(state);
					frag.merge_transition(frag_x);

					frag.add_accepts(accepts);

					const state_t start = frag_x.get_start();
					for(state_set_t::const_iterator p = accepts.begin(); p != accepts.end(); p++)
					{
						frag.connect_epsilon(*p, start);
					}
					accepts = frag_x.get_accepts();
				}

				frag.add_accepts(accepts);
				return frag;
			}

		private:
			parse_tree<_Input> *m_node;
			size_t              m_number1;
			size_t              m_number2;
		};
	}
}

#endif
