#ifndef __MERCURY_REGEX_TRAITS__
#define __MERCURY_REGEX_TRAITS__

#include <cstddef>

#ifndef mercury_countof
#	define mercury_countof(array)  (sizeof(array) / sizeof(array[0]))
#endif

namespace mercury
{
	namespace _regex
	{
		typedef enum tagCHARCLASS
		{
			CHARCLASS_NONE   = 0x0000,
			CHARCLASS_UPPER  = 0x0001,
			CHARCLASS_LOWER  = 0x0002,
			CHARCLASS_DIGIT  = 0x0004,
			CHARCLASS_XDIGIT = 0x0008,
			CHARCLASS_PUNCT  = 0x0010,
			CHARCLASS_BLANK  = 0x0020,
			CHARCLASS_SPACE  = 0x0040,
			CHARCLASS_CNTRL  = 0x0080,
			CHARCLASS_GRAPH  = 0x0100,
			CHARCLASS_PRINT  = 0x0200,
			CHARCLASS_UNDER  = 0x0400,
			CHARCLASS_ALPHA  = CHARCLASS_UPPER | CHARCLASS_LOWER,
			CHARCLASS_ALNUM  = CHARCLASS_ALPHA | CHARCLASS_DIGIT,
			CHARCLASS_CSYM   = CHARCLASS_ALNUM | CHARCLASS_UNDER,
		} CHARCLASS;

		template<typename _Char>
		class regex_traits
		{
		public:
			static _Char meta_escape          (void) { return _Char('\\'); }
			static _Char meta_head            (void) { return _Char('^'); }
			static _Char meta_tail            (void) { return _Char('$'); }
			static _Char meta_disjunction     (void) { return _Char('|'); }
			static _Char meta_repeat0         (void) { return _Char('*'); }
			static _Char meta_repeat1         (void) { return _Char('+'); }
			static _Char meta_repeat01        (void) { return _Char('?'); }
			static _Char meta_repeat_begin    (void) { return _Char('{'); }
			static _Char meta_repeat_separator(void) { return _Char(','); }
			static _Char meta_repeat_end      (void) { return _Char('}'); }
			static _Char meta_any             (void) { return _Char('.'); }
			static _Char meta_set_begin       (void) { return _Char('['); }
			static _Char meta_set_not         (void) { return _Char('^'); }
			static _Char meta_set_range       (void) { return _Char('-'); }
			static _Char meta_set_end         (void) { return _Char(']'); }
			static _Char meta_group_begin     (void) { return _Char('('); }
			static _Char meta_group_end       (void) { return _Char(')'); }

			static _Char escape_bel(void) { return _Char('a'); }
			static _Char escape_ff (void) { return _Char('f'); }
			static _Char escape_lf (void) { return _Char('n'); }
			static _Char escape_cr (void) { return _Char('r'); }
			static _Char escape_ht (void) { return _Char('t'); }
			static _Char escape_vt (void) { return _Char('v'); }
			static _Char escape_esc(void) { return _Char('e'); }

			static _Char escape_upper       (void) { return _Char('u'); }
			static _Char escape_upper_invert(void) { return _Char('U'); }
			static _Char escape_lower       (void) { return _Char('l'); }
			static _Char escape_lower_invert(void) { return _Char('L'); }
			static _Char escape_digit       (void) { return _Char('d'); }
			static _Char escape_digit_invert(void) { return _Char('D'); }
			static _Char escape_space       (void) { return _Char('s'); }
			static _Char escape_space_invert(void) { return _Char('S'); }
			static _Char escape_csym        (void) { return _Char('w'); }
			static _Char escape_csym_invert (void) { return _Char('W'); }

		public:
			static _Char cntrl_nul(void) { return _Char(0x00); }
			static _Char cntrl_soh(void) { return _Char(0x01); }
			static _Char cntrl_stx(void) { return _Char(0x02); }
			static _Char cntrl_etx(void) { return _Char(0x03); }
			static _Char cntrl_eot(void) { return _Char(0x04); }
			static _Char cntrl_enq(void) { return _Char(0x05); }
			static _Char cntrl_ack(void) { return _Char(0x06); }
			static _Char cntrl_bel(void) { return _Char(0x07); }
			static _Char cntrl_bs (void) { return _Char(0x08); }
			static _Char cntrl_ht (void) { return _Char(0x09); }
			static _Char cntrl_lf (void) { return _Char(0x0a); }
			static _Char cntrl_vt (void) { return _Char(0x0b); }
			static _Char cntrl_ff (void) { return _Char(0x0c); }
			static _Char cntrl_cr (void) { return _Char(0x0d); }
			static _Char cntrl_so (void) { return _Char(0x0e); }
			static _Char cntrl_si (void) { return _Char(0x0f); }
			static _Char cntrl_dle(void) { return _Char(0x10); }
			static _Char cntrl_dc1(void) { return _Char(0x11); }
			static _Char cntrl_dc2(void) { return _Char(0x12); }
			static _Char cntrl_dc3(void) { return _Char(0x13); }
			static _Char cntrl_dc4(void) { return _Char(0x14); }
			static _Char cntrl_nak(void) { return _Char(0x15); }
			static _Char cntrl_syn(void) { return _Char(0x16); }
			static _Char cntrl_etb(void) { return _Char(0x17); }
			static _Char cntrl_can(void) { return _Char(0x18); }
			static _Char cntrl_em (void) { return _Char(0x19); }
			static _Char cntrl_sub(void) { return _Char(0x1a); }
			static _Char cntrl_esc(void) { return _Char(0x1b); }
			static _Char cntrl_fs (void) { return _Char(0x1c); }
			static _Char cntrl_gs (void) { return _Char(0x1d); }
			static _Char cntrl_rs (void) { return _Char(0x1e); }
			static _Char cntrl_us (void) { return _Char(0x1f); }
			static _Char cntrl_del(void) { return _Char(0x7f); }

			static _Char char_underscore(void) { return _Char('_'); }

		public:
			static const _Char *class_upper             (void) { return                 m_class_upper  ; }
			static size_t       class_upper_count       (void) { return mercury_countof(m_class_upper ); }
			static const _Char *class_lower             (void) { return                 m_class_lower  ; }
			static size_t       class_lower_count       (void) { return mercury_countof(m_class_lower ); }
			static const _Char *class_digit             (void) { return                 m_class_digit  ; }
			static size_t       class_digit_count       (void) { return mercury_countof(m_class_digit ); }
			static const _Char *class_xdigit            (void) { return                 m_class_xdigit ; }
			static size_t       class_xdigit_count      (void) { return mercury_countof(m_class_xdigit); }
			static const _Char *class_punct             (void) { return                 m_class_punct  ; }
			static size_t       class_punct_count       (void) { return mercury_countof(m_class_punct) ; }
			static const _Char *class_blank             (void) { return                 m_class_blank  ; }
			static size_t       class_blank_count       (void) { return mercury_countof(m_class_blank) ; }
			static const _Char *class_space             (void) { return                 m_class_space  ; }
			static size_t       class_space_count       (void) { return mercury_countof(m_class_space) ; }
			static const _Char *class_cntrl             (void) { return                 m_class_cntrl  ; }
			static size_t       class_cntrl_count       (void) { return mercury_countof(m_class_cntrl) ; }
			static const _Char *class_graph_invert      (void) { return                 m_class_graph_invert ; }
			static size_t       class_graph_invert_count(void) { return mercury_countof(m_class_graph_invert); }
			static const _Char *class_print_invert      (void) { return                 m_class_print_invert ; }
			static size_t       class_print_invert_count(void) { return mercury_countof(m_class_print_invert); }

		public:
			static int number(const _Char &ch)
			{
				const _Char number_array[] =
				{
					_Char('0'), _Char('1'), _Char('2'), _Char('3'), _Char('4'),
					_Char('5'), _Char('6'), _Char('7'), _Char('8'), _Char('9'),
				};
				for(size_t i = 0; i < mercury_countof(number_array); i++)
				{
					if(ch == number_array[i])
					{
						return static_cast<int>(i);
					}
				}
				return -1;
			}

			template<typename _InputIterator, typename _OutputIterator>
			static int number(const _InputIterator &begin, const _InputIterator &end, _OutputIterator &out)
			{
				int value = 0;

				_InputIterator p;
				for(p = begin; p != end; p++)
				{
					const int now = number(*p);
					if(now == -1) { break; }
					value = value * 10 + now;
				}
				if(p == begin) { return -1; }

				out = p;
				return value;
			}

		public:
			static _Char next(const _Char &ch) { return ch + 1; }

			static size_t length(const _Char *s)
			{
				size_t i = 0;
				while(*s++ != cntrl_nul()) { i++; }
				return i;
			}

			template<typename _InputIterator, typename _OutputIterator>
			static CHARCLASS get_charclass(const _InputIterator &begin, const _InputIterator &end, _OutputIterator &out)
			{
				typedef struct tagCLASSINFO
				{
					CHARCLASS type;
					_Char     name[16];
				} CLASSINFO;
				const CLASSINFO info[] =
				{
					{CHARCLASS_UPPER , {_Char('['), _Char(':'), _Char('u'), _Char('p'), _Char('p'), _Char('e'), _Char('r'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_LOWER , {_Char('['), _Char(':'), _Char('l'), _Char('o'), _Char('w'), _Char('e'), _Char('r'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_ALPHA , {_Char('['), _Char(':'), _Char('a'), _Char('l'), _Char('p'), _Char('h'), _Char('a'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_DIGIT , {_Char('['), _Char(':'), _Char('d'), _Char('i'), _Char('g'), _Char('i'), _Char('t'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_ALNUM , {_Char('['), _Char(':'), _Char('a'), _Char('l'), _Char('n'), _Char('u'), _Char('m'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_XDIGIT, {_Char('['), _Char(':'), _Char('x'), _Char('d'), _Char('i'), _Char('g'), _Char('i'), _Char('t'), _Char(':'), _Char(']'), cntrl_nul()}},
					{CHARCLASS_PUNCT , {_Char('['), _Char(':'), _Char('p'), _Char('u'), _Char('n'), _Char('c'), _Char('t'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_BLANK , {_Char('['), _Char(':'), _Char('b'), _Char('l'), _Char('a'), _Char('n'), _Char('k'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_SPACE , {_Char('['), _Char(':'), _Char('s'), _Char('p'), _Char('a'), _Char('c'), _Char('e'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_CNTRL , {_Char('['), _Char(':'), _Char('c'), _Char('n'), _Char('t'), _Char('r'), _Char('l'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_GRAPH , {_Char('['), _Char(':'), _Char('g'), _Char('r'), _Char('a'), _Char('p'), _Char('h'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_PRINT , {_Char('['), _Char(':'), _Char('p'), _Char('r'), _Char('i'), _Char('n'), _Char('t'), _Char(':'), _Char(']'),             cntrl_nul()}},
				};

				for(size_t i = 0; i < mercury_countof(info); i++)
				{
					_InputIterator now = begin;

					const _Char *classname = info[i].name;
					while(*classname != cntrl_nul())
					{
						if( now   ==  end        ) { goto next; }
						if(*now++ != *classname++) { goto next; }
					}
					out = now;
					return info[i].type;
next:;
				}
				return CHARCLASS_NONE;
			}

		private:
			static const _Char m_class_upper       [26];
			static const _Char m_class_lower       [26];
			static const _Char m_class_digit       [10];
			static const _Char m_class_xdigit      [22];
			static const _Char m_class_punct       [ 6];
			static const _Char m_class_blank       [ 2];
			static const _Char m_class_space       [ 6];
			static const _Char m_class_cntrl       [33];
			static const _Char m_class_graph_invert[ 6];
			static const _Char m_class_print_invert[ 5];
		};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_upper       [] = {_Char('A'), _Char('B'), _Char('C'), _Char('D'), _Char('E'), _Char('F'), _Char('G'), _Char('H'), _Char('I'), _Char('J'), _Char('K'), _Char('L'), _Char('M'), _Char('N'), _Char('O'), _Char('P'), _Char('Q'), _Char('R'), _Char('S'), _Char('T'), _Char('U'), _Char('V'), _Char('W'), _Char('X'), _Char('Y'), _Char('Z')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_lower       [] = {_Char('a'), _Char('b'), _Char('c'), _Char('d'), _Char('e'), _Char('f'), _Char('g'), _Char('h'), _Char('i'), _Char('j'), _Char('k'), _Char('l'), _Char('m'), _Char('n'), _Char('o'), _Char('p'), _Char('q'), _Char('r'), _Char('s'), _Char('t'), _Char('u'), _Char('v'), _Char('w'), _Char('x'), _Char('y'), _Char('z')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_digit       [] = {_Char('0'), _Char('1'), _Char('2'), _Char('3'), _Char('4'), _Char('5'), _Char('6'), _Char('7'), _Char('8'), _Char('9')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_xdigit      [] = {_Char('0'), _Char('1'), _Char('2'), _Char('3'), _Char('4'), _Char('5'), _Char('6'), _Char('7'), _Char('8'), _Char('9'), _Char('A'), _Char('B'), _Char('C'), _Char('D'), _Char('E'), _Char('F'), _Char('a'), _Char('b'), _Char('c'), _Char('d'), _Char('e'), _Char('f')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_punct       [] = {_Char('.'), _Char(','), _Char('!'), _Char('?'), _Char(':'), _Char(';')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_blank       [] = {_Char(' ') , cntrl_ht ()};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_space       [] = {_Char(' ') , cntrl_ht (), cntrl_lf (), cntrl_vt (), cntrl_ff (), cntrl_cr ()};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_cntrl       [] = {cntrl_nul(), cntrl_soh(), cntrl_stx(), cntrl_etx(), cntrl_eot(), cntrl_enq(), cntrl_ack(), cntrl_bel(), cntrl_bs (), cntrl_ht (), cntrl_lf (), cntrl_vt (), cntrl_ff (), cntrl_cr (), cntrl_so (), cntrl_si (), cntrl_dle(), cntrl_dc1(), cntrl_dc2(), cntrl_dc3(), cntrl_dc4(), cntrl_nak(), cntrl_syn(), cntrl_etb(), cntrl_can(), cntrl_em (), cntrl_sub(), cntrl_esc(), cntrl_fs (), cntrl_gs (), cntrl_rs (), cntrl_us (), cntrl_del()};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_graph_invert[] = {_Char(' ') , cntrl_ht (), cntrl_lf (), cntrl_vt (), cntrl_ff (), cntrl_cr ()};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_print_invert[] = {             cntrl_ht (), cntrl_lf (), cntrl_vt (), cntrl_ff (), cntrl_cr ()};
	}
}

#endif
