/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.log;

import java.text.MessageFormat;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;


/**
 * 短い形式でログをフォーマットするフォーマッタです。
 * @author Shinji Kashihara
 */
public class ShortFormatter extends Formatter {

    /** システム改行文字 */
    private static final String LINE_SEPARATOR =
        System.getProperty("line.separator");

    /**
     * フォーマットします。 
     * @param record ログレコード
     * @return ログ
     * @see java.util.logging.Formatter#format(java.util.logging.LogRecord)
     */
    public String format(LogRecord record) {
        String[] args = {
            getLevelName(record),
            getSourceName(record),
            formatMessage(record),
            LINE_SEPARATOR
        };
        return MessageFormat.format("{0} {1}> {2}{3}", args);
    }

    /**
     * レベル名を取得します。
     * Level#getLocalizedName の先頭 2 文字を使用します。
     * <pre>
     *   Name     LocalizedName
     *   -----------------------------------
     *   SEVERE:  致命的          -> 致命  
     *   WARNING: 警告            -> 警告
     *   INFO:    情報            -> 情報
     *   CONFIG:  設定            -> 設定
     *   FINE:    詳細レベル（低）-> 詳細
     *   FINER:   詳細レベル（中）-> 詳細
     *   FINEST:  詳細レベル（高）-> 詳細
     * </pre> 
     * @param record ログレコード
     * @return レベル名
     */        
    private String getLevelName(LogRecord record) {
        String lName = record.getLevel().getLocalizedName();
        lName = lName.replaceFirst("(..).*", "$1");
        return lName;
    }

    /**
     * ソース名を取得します。パッケージ名は省略します。
     * @param record ログレコード
     * @return ソース名
     */        
    private String getSourceName(LogRecord record) {
        String cName = record.getSourceClassName();
        if (cName != null) cName = cName.replaceFirst(".+\\.", "");
        String[] args = { cName, record.getSourceMethodName() }; 
        return MessageFormat.format("{0}#{1}", args);
    }
}
