/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.core;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * ブロックコメントをあらわすクラスです。
 * @author Shinji Kashihara
 */
public class Comment {

    /** 小さいリストの初期容量 */
    private static final int SMALL_LIST_CAPACITY = 1;

    /** 大きいリストの初期容量 */
    private static final int LARGE_LIST_CAPACITY = 9;

    /** デフォルトの横幅 */
    private static final int DEFAULT_WIDTH = 80;


    /** 元の Java ソースコメント */
    private String sourceComment;

    /** インポートアイテムのリスト */
    private Pattern[] importPatterns;
    

    /** 説明 */
    private String description;

    /** deprecated タグコメントのリスト */
    private String deprecate;

    /** see タグコメントのリスト */
    private List sees;

    /** since タグコメントのリスト */
    private List sinces;

    /** param タグコメントのリスト */
    private List params;

    /** return タグコメントのリスト */
    private List returns;

    /** throws タグコメントのリスト */
    private List throwss;


    /** author タグコメントのリスト */
    private List authors;

    /** version タグコメントのリスト */
    private List versions;

    /** serial タグコメントのリスト */
    private List serials;

    /** serialField タグコメントのリスト */
    private List serialFields;

    /** serialData タグコメントのリスト */
    private List serialDatas;

    /** spec タグコメントのリスト（JSR No.記述．非標準タグ） */
    private List specs;

    
    /**
     * コンストラクタです．
     */
    public Comment() {
    }

    
    /**
     * 説明をセットします．
     * @param description 説明
     */    
    public void setDescription(String comment) {

        // HTML の p タグの前後に改行を付加
        StringBuffer sb = new StringBuffer();
        Matcher mat = Pattern.compile("(?s)(.)(<p>)(.)").matcher(comment);
        while (mat.find()) {
            String pre = mat.group(1).equals("\n") ? "$1" : "$1\n";
            String suf = mat.group(3).equals("\n") ? "$3" : "\n$3";
            mat.appendReplacement(sb, pre + "$2" + suf);
        }
        mat.appendTail(sb);

        this.description = sb.toString();
    }
    
    /**
     * deprecated タグコメントを追加します．
     * @param comment コメント
     */    
    public void addDeprecated(String comment) {
        deprecate = comment;
    }
    
    /**
     * see タグコメントを追加します．
     * @param comment コメント
     */    
    public void addSee(String comment) {
        if (sees == null) sees = new ArrayList(LARGE_LIST_CAPACITY);
        sees.add(comment);
    }
    
    /**
     * since タグコメントを追加します．
     * @param comment コメント
     */    
    public void addSince(String comment) {
        if (sinces == null) sinces = new ArrayList(SMALL_LIST_CAPACITY);
        sinces.add(comment);
    }
    
    /**
     * param タグコメントを追加します．
     * @param comment コメント
     */    
    public void addParam(String comment) {
        if (params == null) params = new ArrayList(LARGE_LIST_CAPACITY);
        params.add(comment);
    }
    
    /**
     * return タグコメントを追加します．
     * @param comment コメント
     */    
    public void addReturn(String comment) {
        if (returns == null) returns = new ArrayList(SMALL_LIST_CAPACITY);
        returns.add(comment);
    }
    
    /**
     * throws タグコメントを追加します．
     * @param comment コメント
     */    
    public void addThrows(String comment) {
        if (throwss == null) throwss = new ArrayList(LARGE_LIST_CAPACITY);
        throwss.add(comment);
    }


    /**
     * 元の Java ソースコメントをこのコメントにマージします．
     * @param sourceComment 元の Java ソースコメント
     */
    public void mergeSourceComment(String sourceComment) {

        this.sourceComment = sourceComment;

        Pattern pat = Pattern.compile("[ \t]+@author[ \t]*(.*)\n");
        Matcher mat = pat.matcher(sourceComment);
        while (mat.find()) {
            addAuthor(mat.group(1));
        }

        pat = Pattern.compile("[ \t]+@version[ \t]*(.*)\n");
        mat = pat.matcher(sourceComment);
        while (mat.find()) {
            addVersion(mat.group(1));
        }

        pat = Pattern.compile("[ \t]+@serial[ \t]*(.*)\n");
        mat = pat.matcher(sourceComment);
        while (mat.find()) {
            addSerial(mat.group(1));
        }

        pat = Pattern.compile("[ \t]+@serialField[ \t]*(.*)\n");
        mat = pat.matcher(sourceComment);
        while (mat.find()) {
            addSerialField(mat.group(1));
        }

        pat = Pattern.compile("[ \t]+@serialData[ \t]*(.*)\n");
        mat = pat.matcher(sourceComment);
        while (mat.find()) {
            addSerialData(mat.group(1));
        }

        pat = Pattern.compile("[ \t]+@spec[ \t]*(.*)\n");
        mat = pat.matcher(sourceComment);
        while (mat.find()) {
            addSpec(mat.group(1));
        }
    }
    
    /**
     * author タグコメントを追加します．
     * @param comment コメント
     */    
    private void addAuthor(String comment) {
        if (authors == null) authors = new ArrayList(LARGE_LIST_CAPACITY);
        authors.add(comment);
    }
    
    /**
     * version タグコメントを追加します．
     * @param comment コメント
     */    
    private void addVersion(String comment) {
        if (versions == null) versions = new ArrayList(SMALL_LIST_CAPACITY);
        versions.add(comment);
    }
    
    /**
     * serial タグコメントを追加します．
     * @param comment コメント
     */    
    private void addSerial(String comment) {
        if (serials == null) serials = new ArrayList(SMALL_LIST_CAPACITY);
        serials.add(comment);
    }
    
    /**
     * serialField タグコメントを追加します．
     * @param comment コメント
     */    
    private void addSerialField(String comment) {
        if (serialFields == null) serialFields = new ArrayList(SMALL_LIST_CAPACITY);
        serialFields.add(comment);
    }
    
    /**
     * serialData タグコメントを追加します．
     * @param comment コメント
     */    
    private void addSerialData(String comment) {
        if (serialDatas == null) serialDatas = new ArrayList(SMALL_LIST_CAPACITY);
        serialDatas.add(comment);
    }
    
    /**
     * spec タグコメントを追加します．
     * @param comment コメント
     */    
    private void addSpec(String comment) {
        if (specs == null) specs = new ArrayList(SMALL_LIST_CAPACITY);
        specs.add(comment);
    }


    /**
     * インポートパスのリストをセットします．
     * このパスは link タグや see タグのパスに含まれるパッケージ名の省略に使用されます．
     * @param importPaths インポートパスのリスト
     */
    public void setImportPaths(List importPaths) {

        if (importPaths == null) return;
        int size = importPaths.size();
        importPatterns = new Pattern[size];

        for (int i = 0; i < size; i++) {
            String path = (String) importPaths.get(i);
            
            if (path.endsWith(".*")) {
                String packageName = path.replaceFirst("\\*$", "");
                String regex = "(^|\\W)" + packageName + "(\\w+(\\W|$))";
                importPatterns[i] = Pattern.compile(regex);
            } else {
                String packageName = path.replaceFirst("^(.+?)\\w+$", "$1");
                String className = path.replaceFirst(packageName + "(\\w+)$", "$1");
                String regex = "(^|\\W)" + packageName + "(" + className + "(\\W|$))";
                importPatterns[i] = Pattern.compile(regex);
            }
        }
    }
    
    /**
     * コメントの文字列表現を取得します．
     * @see java.lang.Object#toString()
     */
    public String toString() {

        // 元ソースのコメントから行数とインデント取得
        int decoSize = 2;
        if (sourceComment == null) {
            throw new IllegalStateException(
            "Source comment is null. require #setSourceComment.");
        }
        int originDecoHeight = StringUtils.heightOf(sourceComment);
        if (originDecoHeight <= 0) {
            throw new IllegalStateException(
            "Illegal comment height " + originDecoHeight + "\n" + sourceComment);
        }
        int originHeight = originDecoHeight;
        if (originDecoHeight > decoSize) {
            originHeight = originDecoHeight - decoSize;
        }
        String indent = sourceComment.replaceFirst("(?s)^([ \t]*?)/\\*\\*.*", "$1");

        // 元ソースの飾り付けを含むコメント行数が 2 行以下の場合
        if (originDecoHeight <= 2) {
            StringBuffer sb = new StringBuffer();
            sb.append(indent);
            sb.append("/** ");
            if (description != null && description.length() > 0) {
                String str =  description.replaceAll("\n", "");
                sb.append(str);
            } else if (sinces != null && sinces.size() > 0) {
                sb.append("@since ");
                String str =  (String) sinces.get(0);
                str = str.replaceAll("\n", "");
                sb.append(str);
            }
            if (originDecoHeight == 2) {
                sb.append("\n");
                sb.append(indent);
            }
            sb.append(" */\n");
            return sb.toString();
        }

        // 複数行コメントの作成    
        int width = DEFAULT_WIDTH - indent.length() - OutputComment.LINE_PREFIX.length();
        OutputComment o = new OutputComment(originHeight, width);
        String decoComment = o.toString();
        if (decoComment.length() > 0) {
            if (o.resultHeight() != o.originHeight) {
                decoComment = resizeComment(o, decoComment);
            }
            decoComment = decoComment.replaceAll("(?m)^(.+)", indent + "$1");
        }

        return decoComment;
    }

    /**
     * コメント構築結果を保持するクラスです． 
     */
    private class OutputComment {

        final static String LINE_PREFIX = " * ";
        final int originHeight;
        final int initWidth;
        int width;
        String comment;
        boolean enabledFirstLine;
        
        OutputComment(int originHeight, int width) {
            this.originHeight = originHeight;
            this.initWidth = width;
            this.width = width;
            rebuild();
        }
        
        void rebuild() {
            this.comment = buildComment(width, originHeight);
        }
        
        int resultHeight() {
            if (enabledFirstLine) {
                return StringUtils.heightOf(comment) - 1;
            } else {
                return StringUtils.heightOf(comment);
            }
        }

        void resetWidth() {
            width = initWidth;
        }
        
        public String toString() {
            String str = comment;
            if (str.length() > 0) {
                StringBuffer sb = new StringBuffer();
                str = str.replaceAll("(?m)^", LINE_PREFIX);
                if (enabledFirstLine) {
                    sb.append( "/**" );
                    str = str.replaceFirst("^ \\*", "");
                } else {
                    sb.append( "/**\n" );
                }
                sb.append( str );
                sb.append( " */\n" );
                str = sb.toString();
            }
            return str;
        }
    }
    
    /**
     * コメントサイズを調整します．
     * @param o コメント構築結果
     * @param decoComment 飾り付け済みコメント
     * @return サイズ調整済みのコメント
     */
    private String resizeComment(OutputComment o, String decoComment) {

        // 作成したコメントが元ソースコメント行数より多い場合はコメントを小さくする
        if (o.resultHeight() > o.originHeight) {
            shrinkComment(o);
        
            // 小さくならない場合は説明の 1 つ目の「。」より後を削除して再構築．
            // shrinkComment メソッドでオリジナルより小さく出来ないのは
            // レアケースであり JDK1.4 全ソースでも 2 個所のみ．
            if (o.resultHeight() > o.originHeight && description != null) {
                int pos = description.indexOf('。');
                if (pos != -1) {
                    description = description.substring(0, pos + 1);
                    o.resetWidth();
                    o.rebuild();

                    if (o.resultHeight() > o.originHeight) {
                        shrinkComment(o);
                    }
                }
            }
            decoComment = o.toString();
        }
        
        // 作成したコメントが元ソースコメント行数より少ない場合はコメントを大きくする
        if (o.resultHeight() < o.originHeight) {

            // 先頭に改行追加
            int sub = o.originHeight - o.resultHeight();
            StringBuffer sb = new StringBuffer();
            for (int i = 0; i < sub; i++) {
                sb.append("\n");
            }
            decoComment = sb + decoComment;
        }
        
        return decoComment;
    }

    /**
     * コメントを小さくします．
     * @param o コメント構築結果
     */
    private void shrinkComment(OutputComment o) {

        // 説明が元 Java ソースコメントに無い場合は省略
        String emptyDescRegex = "(?s)\\s*/\\*\\*\\s*\\*\\s*@.*";
        if (sourceComment.matches(emptyDescRegex)) {
            description = null;
            o.rebuild();
            if (o.resultHeight() <= o.originHeight) return;
        }

        // タグが元 Java ソースコメントに無い場合は省略
        boolean b1 = shrinkTagList("@see", sees);
        boolean b2 = shrinkTagList("@(throws|exception)", throwss);
        boolean b3 = shrinkTagList("@param", params);
        boolean b4 = shrinkTagList("@return", returns);
        if (b1 || b2 || b3 || b4) {
            o.rebuild();
            if (o.resultHeight() <= o.originHeight) return;
        }

        // 複数行に跨る HTML タグを 1 行に
        StringBuffer sb = new StringBuffer();
        Pattern continueTagPat = Pattern.compile("(<.+?>)\n(<.+?>)");
        Matcher continueTagMat = continueTagPat.matcher(o.comment);
        int height = o.resultHeight();
        while (height > o.originHeight && continueTagMat.find()) {
            continueTagMat.appendReplacement(sb, "$1$2");
            height--;
        }
        continueTagMat.appendTail(sb);
        o.comment = sb.toString();
        if (height <= o.originHeight) return;

        // 空行を削除
        Pattern emptyLinePat = Pattern.compile("(?m)^\\s*?\n");
        Matcher emptyLineMat = emptyLinePat.matcher(o.comment);
        sb = new StringBuffer();
        while (height > o.originHeight && emptyLineMat.find()) {
            emptyLineMat.appendReplacement(sb, "");
            height--;
        }
        emptyLineMat.appendTail(sb);
        o.comment = sb.toString();
        if (height <= o.originHeight) return;

        // HTML の p タグの後の改行を削除
        shrinkHTMLPTag(o);
        if (o.resultHeight() <= o.originHeight) return;

        // HTML の pre タグを前後の行に含める
        shrinkHTMLPreTag(o);
        if (o.resultHeight() <= o.originHeight) return;
        
        // 元 Java ソースコメントの 1 行目から説明がある場合
        String firstLineRegex = "(?s)\\s*/\\*\\*\\s*\n.*";
        if (!sourceComment.matches(firstLineRegex)) {
            o.enabledFirstLine = true;
            if (o.resultHeight() <= o.originHeight) return;
        }

        // 横幅を伸ばす
        final int maxWidth = 160;
        while (o.resultHeight() > o.originHeight && o.width < maxWidth) {

            o.width += 4;
            o.rebuild();
            if (o.resultHeight() <= o.originHeight) return;

            o.comment = continueTagPat.matcher(o.comment).replaceAll("$1$2");
            if (o.resultHeight() <= o.originHeight) return;

            o.comment = emptyLinePat.matcher(o.comment).replaceAll("");
            if (o.resultHeight() <= o.originHeight) return;

            shrinkHTMLPTag(o);
            if (o.resultHeight() <= o.originHeight) return;

            shrinkHTMLPreTag(o);
        }
    }

    /**
     * 指定したタグが Java ソースコメントに無い場合はタグリストをクリアします．
     * @param tagRegex タグ正規表現
     * @param tagList タグリスト
     * @return タグリストをクリアした場合は true
     */
    private boolean shrinkTagList(String tagRegex, List tagList) {
        if (tagList != null) {
            int tagCount = sourceComment.split("\\s" + tagRegex + "\\s", -1).length - 1;
            if (tagCount == 0) {
                tagList.clear();
                return true;
            }
        }
        return false;
    }
    
    /**
     * HTML の p タグの後ろの改行を削除してコメントを小さくします．
     * @param o コメント構築結果
     */    
    private void shrinkHTMLPTag(OutputComment o) {

        StringBuffer sb = new StringBuffer();
        Pattern pTagPat = Pattern.compile("(\n\\s*<p>\\s*?)\n(.)");
        Matcher pTagMat = pTagPat.matcher(o.comment);
        int height = o.resultHeight();
        while (height > o.originHeight && pTagMat.find()) {
            String nextChar = pTagMat.group(2);
            if (!nextChar.equals("@")) {
                pTagMat.appendReplacement(sb, "$1$2");
                height--;
            }
        }
        pTagMat.appendTail(sb);
        o.comment = sb.toString();
    }
    
    /**
     * HTML の pre タグを前後の改行を削除してコメントを小さくします．
     * @param o コメント構築結果
     */    
    private void shrinkHTMLPreTag(OutputComment o) {

        StringBuffer sb = new StringBuffer();
        Pattern endPreTagPat = Pattern.compile("(\n\\s*</pre>\\s*?)\n(.)");
        Matcher endPreTagMat = endPreTagPat.matcher(o.comment);
        int height = o.resultHeight();
        while (height > o.originHeight && endPreTagMat.find()) {
            String nextChar = endPreTagMat.group(2);
            if (!nextChar.equals("@")) {
                endPreTagMat.appendReplacement(sb, "$1$2");
                height--;
            }
        }
        endPreTagMat.appendTail(sb);
        o.comment = sb.toString();
        if (height <= o.originHeight) return;

        sb = new StringBuffer();
        Pattern beginPreTagPat = Pattern.compile("\n\\s*(<pre>)");
        Matcher beginPreTagMat = beginPreTagPat.matcher(o.comment);
        height = o.resultHeight();
        while (height > o.originHeight && beginPreTagMat.find()) {
            beginPreTagMat.appendReplacement(sb, "$1");
            height--;
        }
        beginPreTagMat.appendTail(sb);
        o.comment = sb.toString();
    }
    
    /**
     * 指定した横幅でコメントを組み立てます．
     * @param width 横幅
     * @param originHeight 行数
     * @return 組み立てたコメント
     */
    private String buildComment(int width, int originHeight) {

        StringBuffer sb = new StringBuffer();

        // 説明の組み立て
        if (description != null && description.length() > 0) {
            String desc = omitPackageName(description);
            desc = desc.replaceAll("(?m)^ (</pre>)", "$1");
            if (originHeight == 1) {
                sb.append( desc.replaceAll("\n", "") );
                sb.append( "\n" );
                return sb.toString();
            }
            sb.append( adjust(desc, width) );
            sb.append( "\n" );
        }

        // deprecated タグの組み立て
        if (deprecate != null && deprecate.length() > 0) {
            String depre = omitPackageName(deprecate);
            depre = "@deprecated " + depre;
            sb.append( adjust(depre, width) );
            sb.append( "\n" );
        }

        appendTo("@author  ",    authors,      sb, width);
        appendTo("@version ",    versions,     sb, width);

        // param タグの組み立て
        if (params != null && params.size() > 0) {
            
            // name の文字数をカウント
            int paramsSize = params.size();
            int[] nameLens = new int[paramsSize];
            int nameLenMax = 3;
            final int nameLenLimit = 12;

            for (int i = 0; i < paramsSize; i++) {
				String comment = (String) params.get(i);
                String name = comment.replaceFirst("(?s)(\\w+)\\s.*", "$1");
                nameLens[i] = name.length();
                if (nameLens[i] > nameLenMax && nameLens[i] < nameLenLimit) {
                    nameLenMax = nameLens[i];
                }
			}

            // 2 行目以降のインデントを作成
            String tag = "@param   ";
            int indentCnt = tag.length() + nameLenMax + 1;
            StringBuffer indent = new StringBuffer(indentCnt);
            for ( ;indentCnt>0; indentCnt--) indent.append(' ');

            // バッファに出力
            for (int i = 0; i < paramsSize; i++) {

                int spaceCnt = nameLenMax - nameLens[i];
                StringBuffer space = new StringBuffer();
                for ( ;spaceCnt>0; spaceCnt--) space.append(' ');
                
                String comment = (String) params.get(i);
                comment = comment.replaceFirst("(?s)(\\w+)\\s+(.*)", "$1" + space + " $2");
                comment = omitPackageName(comment);
                comment = adjust(comment, width - tag.length());
                StringTokenizer st = new StringTokenizer(comment, "\n");
            
                // 1 行目
                sb.append(tag);
                if (st.hasMoreTokens()) {
                    sb.append(st.nextToken());
                }
                sb.append("\n");

                // 2 行目以降
                while (st.hasMoreTokens()) {
                    sb.append(indent);
                    sb.append(st.nextToken());
                    sb.append("\n");
                }
            }
        }

        appendTo("@return  ",    returns,      sb, width);
        appendTo("@throws  ",    throwss,      sb, width);
        appendTo("@serialField", serialFields, sb, width);
        appendTo("@serialData",  serialDatas,  sb, width);
        appendTo("@see     ",    sees,         sb, width);
        appendTo("@since   ",    sinces,       sb, width);
        appendTo("@serial  ",    serials,      sb, width);
        appendTo("@spec    ",    specs,        sb, width);
        
        String str = sb.toString();
        str = str.replaceFirst("\n\n$", "\n");
        
        return str;
    }
    
    /**
     * タグコメントのリストを文字列バッファに追加します． 
     * 複数行になる場合はインデントします．<p>
     * @param tag タグ文字列
     * @param tagList タグコメントのリスト
     * @param sb 文字列バッファ
     * @param width 横幅
     */    
    private void appendTo(String tag, List tagList, StringBuffer sb, int width) {

        if (tagList == null) return;

        for (Iterator it = tagList.iterator(); it.hasNext();) {

            String comment = (String) it.next();
            comment = omitPackageName(comment);
            comment = adjust(comment, width - tag.length());
            StringTokenizer st = new StringTokenizer(comment, "\n");
            
            sb.append(tag);
            if (st.hasMoreTokens()) {
                sb.append(st.nextToken());
            }
            sb.append("\n");

            while (st.hasMoreTokens()) {
                sb.append(tag.replaceAll(".", " "));
                sb.append(st.nextToken());
                sb.append("\n");
            }
        }
    }
    
    /**
     * see タグや link タグの埋め込みクラスパスをインポートクラスパスのリストを
     * 元にパッケージ部分を省略します．
     * @param str 変換前の文字列
     * @return 変換後の文字列
     */    
    private String omitPackageName(String str) {

        if (importPatterns == null) return str; 
        int size = importPatterns.length;

        for (int i = 0; i < size; i++) {
            Matcher m = importPatterns[i].matcher(str);
            str = m.replaceAll("$1$2");
        } 
        return str;
    }
    
    /**
     * 改行を含む文字列の横幅を調整します．
     * @param value 調整する文字列
     * @param width 折り返し幅（バイト）
     * @return 横幅を調整した文字列
     */    
    private String adjust(String value, int width) {

        if (value.getBytes().length < width) {
            return value;
        }

        StringTokenizer st1 = new StringTokenizer(value, "\n");
        StringBuffer resultBuf = new StringBuffer();
        boolean preTagArea = false;
        final int longWordWidth = width - 20;

        // 行単位で解析．ただし pre タグ内はスルー．
        // 今のところ行頭と行末の pre タグのみ対応．
        while (st1.hasMoreTokens()) {

            String lineValue1 = st1.nextToken();
            if (lineValue1.matches("^\\s*</pre>.*")) preTagArea = false;
            if (lineValue1.matches("^\\s*<pre>.*" )) preTagArea = true;
            if (preTagArea) {
                resultBuf.append(lineValue1);
                resultBuf.append("\n");
                if (lineValue1.matches(".*</pre>\\s*$")) preTagArea = false;
                continue;
            }
            if (lineValue1.matches(".*<pre>\\s*$" )) preTagArea = true;

            // 長い英数字文字列は先頭に改行を付加
            String multiLineValue = lineValue1.replaceAll(
                "\\s?(\\p{Graph}{" + longWordWidth + ",})", "\n$1");

            StringTokenizer st2 = new StringTokenizer(multiLineValue, "\n");
            while (st2.hasMoreTokens()) {
                String lineValue2 = st2.nextToken();
                adjustLine(lineValue2, resultBuf, width);
            }

        }
        return resultBuf.toString();
    }
    
    /**
     * 指定した単一行の文字列の横幅を調整します．
     * 入力行の値により最大折り返し幅を超える場合があります．
     * @param inValue 入力行
     * @param resultBuf 横幅を調整した結果文字列を追加するバッファ
     * @param width 折り返し幅（バイト）
     */    
    private void adjustLine(String inValue, StringBuffer resultBuf, int width) {
        
        final int minWidth = width - 10;
        final int maxWidth = width + 10;
        final int ADJUST_SKIP_WIDTH = width + 4;
        final int lastPos = inValue.length() - 1;
        final String PUNCTS = "。、」";
        final String PARTICLES = "はがのをにへとらてる";

        StringBuffer buf = new StringBuffer();
        int bufLen = 0;
        for (int pos = 0; pos < lastPos; pos++) {

            if (bufLen == 0) {
                String after = inValue.substring(pos, lastPos);
                int afterLen = after.getBytes().length;
                if (afterLen <= ADJUST_SKIP_WIDTH) {
                    buf.append(after);
                    break;
                }
            }

            char c = inValue.charAt(pos);
            int cLen = String.valueOf(c).getBytes().length;
            bufLen += cLen;
            boolean isChangeLine = false;

            if (bufLen > minWidth) {
                // 最小折り返し幅を超えている場合は句読点などで改行を挿入
                
                if (c == ' ') {

                    isChangeLine = true;
                    buf.append('\n');

                } else if (PUNCTS.indexOf(c) != -1) {

                    isChangeLine = true;
                    buf.append(c);
                    buf.append('\n');

                } else if (PARTICLES.indexOf(c) != -1) {
                    
                    char next = inValue.charAt(pos + 1);
                    if (PUNCTS.indexOf(next) == -1 && next != ' ') {
                        
                        isChangeLine = true;
                        buf.append(c);
                        buf.append('\n');
                    }

                } else if (bufLen > width) {
                    // 通常折り返し幅を超えている場合は改行を挿入
                    // ただし現在の文字が半角英数字の場合を除く
                        
                    if (c == '<' || cLen > 1) {
                        
                        isChangeLine = true;
                        buf.append('\n');
                        buf.append(c);
                        
                    } else if (bufLen > maxWidth) {
                        // 最大折り返し幅を超えている場合は
                        // 全角文字まで戻って改行を挿入
                        
                        for (int bPos = buf.length() - 1; bPos > 0; bPos--) {
                            char bc = buf.charAt(bPos);
                            
                            if (bc == ' ') {
                                buf.replace(bPos, bPos+1, "\n");
                                bufLen = buf.substring(bPos+1).getBytes().length;
                                break;
                                
                            } else {
                                
                                int bcLen = String.valueOf(bc).getBytes().length;
                                if (bcLen > 1) {
                                    buf.insert(bPos+1, '\n');
                                    bufLen = buf.substring(bPos+2).getBytes().length;
                                    break;
                                }
                            }
                        }
                    }
                }
            }
                
            if (isChangeLine) {
                resultBuf.append(buf);
                buf = new StringBuffer();
                bufLen = 0;
            } else {
                buf.append(c);
            }
        }
        buf.append(inValue.charAt(lastPos));

        resultBuf.append(buf);
        resultBuf.append('\n');
    }
}
