/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.core;

import java.io.File;
import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * マージを行うクラスです。
 * @author Shinji Kashihara
 */
public class Merger {

    /** クラス名 */
    private String className;

    /** API ドキュメントディレクトリ */
    private File docDirectory;

    /** API ドキュメントエンコーディング */
    private String docEncoding = System.getProperty("file.encoding");

    /**
     * コンストラクタです。
     * @param docDirectory
     */
    public Merger(File docDirectory) {
        this.docDirectory = docDirectory;
    }

    /**
     * API ドキュメントエンコーディングを設定します。
     * 設定されなかった場合はデフォルトエンコーディングを使用します。
     * @param docEncoding API ドキュメントエンコーディング
     */
    public void setDocEncoding(String docEncoding) {
        this.docEncoding = docEncoding;
    }
    
    /**
     * Java ソースと Javadoc コメントをマージします。
     * 対応する API ドキュメントが見つからない場合はそのまま Java ソースを返します。
     * @param javaSource Java ソース文字列
     * @return マージ後の Java ソース文字列
     * @throws IOException 入出力例外が発生した場合
     */
    public String merge(String javaSource) throws IOException {

        String source = StringUtils.optimizeLineSeparator(javaSource);
        className = scanClassName(source);
        if (className == null) return source;

        APIDocument apiDoc = new APIDocument(docDirectory, className, docEncoding);
        if (apiDoc.isEmpty()) return source;

        JavaBuffer javaBuf = new JavaBuffer(className, source);
        while (javaBuf.nextComment()) {
            Signature sig = javaBuf.getSignature();
            Comment com = apiDoc.getComment(sig);
            javaBuf.setLocalizedComment(com);
        }

        String result = javaBuf.finishToString();
        return result;
    }

    /**
     * Java ソース文字列をスキャンしクラス名を取得します。
     * package 宣言が無い場合は常に null を返します。
     * @param source Java ソース文字列
     * @return クラス名。取得できない場合は null。
     */
    private String scanClassName(String source) {
        
        Pattern pkgPat = Pattern.compile("(?m)^\\s*package\\s+([\\w\\.]+)");
        Matcher pkgMat = pkgPat.matcher(source);

        if (pkgMat.find()) {
            String packageName = pkgMat.group(1);

            String clsReg = "(?m)^(|[\\w\\s]*?\\s+|.*?\\*/\\s+)(class|interface)\\s+(\\w+)";
            Pattern clsPat = Pattern.compile(clsReg);
            Matcher clsMat = clsPat.matcher(source);
            if (clsMat.find()) {
                String name = packageName + "." + clsMat.group(3);
                return name;
            }
        }
        return null;
    }
    
    /**
     * 直前のマージした Java ソースのクラス名を取得します。
     * @return クラス名
     */
    public String getMergedClassName() {
        return className;
    }
}
