/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.resource;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.mergedoc.pleiades.generator.nls.HelpExtractor;
import jp.sourceforge.mergedoc.pleiades.generator.nls.HtmlFragmentList;
import jp.sourceforge.mergedoc.pleiades.resource.HelpHtmlParser.HtmlFragment;
import junit.framework.TestCase;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

/**
 * HTML パーサー・テスト・クラスです。
 * <p>
 * @author cypher256
 */
public class HelpHtmlParserTest extends TestCase implements FileNames {

	/** 日本語 HTML 文字セット */
	private static final String JA_HTML_CHARSET = "UTF-8";

	/** 辞書 */
	private static final TranslationDictionary dic = new TranslationDictionary();

	/** ルート・ディレクトリー */
	private static File ROOT_DIR;

	/** 入力ストリーム・リスト */
	private List<InputStream> isList = new LinkedList<InputStream>();

	/**
	 * コンストラクタ
	 * @throws IOException
	 */
	public HelpHtmlParserTest() throws IOException {

		String path = HelpHtmlParserTest.class.getName().replace(".", "/") + ".class";
		ClassLoader loader = Thread.currentThread().getContextClassLoader();
		File pathDir = new File(loader.getResource(path).getPath()).getParentFile();
		ROOT_DIR = new File(pathDir.getCanonicalPath().replace("bintest", "srctest"));
	}

	/**
	 * テスト前処理
	 */
	@Override
	protected void setUp() throws Exception {
	}

	/**
	 * テスト後処理
	 */
	@Override
	protected void tearDown() throws Exception {
		for (InputStream is : isList) {
			IOUtils.closeQuietly(is);
		}
	}

	/**
	 * ファイル取得
	 * @param name
	 * @return
	 */
	private File getFile(String name) {
		return new File(ROOT_DIR, name);
	}

	/**
	 * 入力ストリーム取得
	 * @param name
	 * @return 入力ストリーム
	 * @throws FileNotFoundException
	 */
	private InputStream getInputStream(String name) throws FileNotFoundException {
		File file = getFile(name);
		if (!file.exists()) {
			fail("リソースが見つかりません。" + file.getPath());
		}
		return new FileInputStream(file);
	}

	/**
	 * 日本語 HTML 書込み
	 * @param parser
	 * @param targetDir
	 * @return
	 * @throws Exception
	 */
	private String writeJaActualHtml(HelpHtmlParser parser, String targetDir) throws Exception {

		String jaHtml = IOUtils.toString(parser.getInputStream(), JA_HTML_CHARSET);
		File dir = getFile(targetDir + "/en.html").getParentFile();
		OutputStream os = new FileOutputStream(new File(dir, "ja_actual.html"));
		IOUtils.write(jaHtml, os, JA_HTML_CHARSET);
		os.close();
		return jaHtml;
	}

	/**
	 * テスト実行
	 * @param targetDir
	 * @param isLogOut ログ出力する場合は true
	 * @throws Exception
	 */
	private PropertySet doTestLoad(String targetDir, boolean isLogOut)
			throws FileNotFoundException {

		// 英語リソース抽出
		String enPath = targetDir + "/en.html";
		HtmlFragmentList enList = new HtmlFragmentList(enPath);
		for (HtmlFragment f : new HelpHtmlParser(getInputStream(enPath))) {
			enList.add(f);
		}
		// 日本語リソース抽出
		String jaPath = targetDir + "/ja.html";
		HtmlFragmentList jaList = new HtmlFragmentList(jaPath);
		for (HtmlFragment f : new HelpHtmlParser(getInputStream(jaPath))) {
			jaList.add(f);
		}
		String msg = new HelpExtractor().optimizeListSize(enList, jaList);
		if (isLogOut) {
			System.out.println(msg);
		}

		// プロパティー作成
		PropertySet actualProp = new PropertySet();
		for (int i = 0; i < Math.max(enList.size(), jaList.size()); i++) {

			HtmlFragment en = i < enList.size() ? enList.get(i) : null;
			HtmlFragment ja = i < jaList.size() ? jaList.get(i) : null;
			String enText = en == null ? "" : en.getText();
			String jaText = ja == null ? "" : ja.getText();

			// 確認用のログ出力
			if (isLogOut) {
				System.out.println("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
				System.out.println(en);
				System.out.println("──────────────");
				System.out.println(ja);
				System.out.println("──────────────");
				String jaDic = dic.lookup(enText).getValue();
				System.out.println("id" + i + ":辞書" + (jaDic.equals(jaText) ? "○" : "●") + "| " + jaDic);
			}

			actualProp.put(enText, jaText);
		}
		assertEquals(enList.size(), jaList.size());

		return actualProp;
	}

	/**
	 * テスト実行
	 * @param targetDir
	 * @param isLogOut ログ出力する場合は true
	 * @throws Exception
	 */
	private void doTest(String targetDir, boolean isLogOut) throws Exception {

		PropertySet actualProp = doTestLoad(targetDir, isLogOut);
		File transPropFile = getFile(targetDir + "/trans.properties");
		actualProp.store(transPropFile, "JUnit テスト結果プロパティー");

		// HTML の検証
		HelpHtmlParser parser = new HelpHtmlParser(getInputStream(targetDir + "/en.html"));
		for (HtmlFragment f : parser) {
			String en = f.getText();
			String ja = dic.lookupHelp(en).getValue();
			f.setText(ja);
		}
		String jaHtml = writeJaActualHtml(parser, targetDir);
		String expected = FileUtils.readFileToString(getFile(targetDir + "/ja_expected.html"), JA_HTML_CHARSET);
		assertEquals(expected, jaHtml);
	}

	/**
	 * テスト
	 * @throws Exception
	 */
	public void test311a() throws Exception {

		doTest("3.1.1a/org.eclipse.jdt.doc.isv_3.1.1/jdt_api_options", false);
		doTest("3.1.1a/org.eclipse.platform.doc.user_3.1.1/whatsNew/platform_whatsnew", false);
	}

	/**
	 * テスト
	 * @throws Exception
	 */
	public void test321() throws Exception {

		doTest("3.2.1/org.eclipse.help.base/help_home", false);
		doTest("3.2.1/org.eclipse.platform.doc.user/gettingStarted/qs-02a", false);

		doTestLoad("3.2.1/org.eclipse.platform.doc.user/_countTest/ref-19cvs", false);
		doTestLoad("3.2.1/org.eclipse.platform.doc.user/_countTest/ref-19cvs-password", false);
		doTestLoad("3.2.1/org.eclipse.platform.doc.user/_countTest/cprbview", false);
		doTestLoad("3.2.1/org.eclipse.platform.doc.user/_countTest/tasks-cvs-passwords", false);
		doTestLoad("3.2.1/org.eclipse.platform.doc.user/_countTest/colordialog", false);
	}
}
