#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <dirent.h>
#include "common.h"

eCopyOverride gCopyOverride = kNone;

/// last char of dpath is /

bool file_copy(char* spath, char* dpath)
{
    /// key input? ///
    fd_set mask;

    FD_ZERO(&mask);
    FD_SET(0, &mask);

    struct timeval tv;
    tv.tv_sec = 0;
    tv.tv_usec = 0;

    select(1, &mask, NULL, NULL, &tv);
    
    if(FD_ISSET(0, &mask)) {
        int key = getch();

        if(key == 3 || key == 7) {    // CTRL-G and CTRL-C
            gCopyOverride = kCancel;
        }
    }

    /// cancel ///
    if(gCopyOverride == kCancel) {
        return false;
    }

    /// illegal argument check ///
    char spath2[PATH_MAX];

    strcpy(spath2, spath);
    strcat(spath2, "/");
    
    if(strstr(dpath, spath2) == dpath) {
        return false;
    }
    
    /// dir check ///
    struct stat dstat;
    if(lstat(dpath, &dstat) < 0) {
        err_msg("file_copy: destination err(%s)", dpath);
        return false;
    }
    if(!S_ISDIR(dstat.st_mode)) {
        err_msg("file_copy: destination err(%s)", dpath);
        return false;
    }
    
    /// source stat ///
    struct stat stat_;

    if(lstat(spath, &stat_) < 0) {
        err_msg("file_copy: stat err(%s)", spath);
        return false;
    }

    /// get file name ///
    char sfname[1024];
    int i;
    const int len = strlen(spath);
    for(i = len-1;
        i > 0;
        i--)
    {
        if(spath[i] == '/') break;
    }
    memcpy(sfname, spath + i + 1, len - i);

    /// get destination file name ///
    char dfile[PATH_MAX];
    strcpy(dfile, dpath);
    strcat(dfile, sfname);

    /// illegal argument check ///
    if(strcmp(spath, dfile) == 0) {
        return false;
    }
        
    /// dir ///
    if(S_ISDIR(stat_.st_mode)) {
        /// msg ///
        char buf[1024];
        sprintf(buf, "entering %s", spath);
        msg(buf);

        /// override ///
        bool no_mkdir = false;
        if(access(dfile, F_OK) == 0) {
            struct stat dfstat;
            
            if(lstat(dfile, &dfstat) < 0) {
                err_msg("file_copy: stat err(%s)", dfile);
                return false;
            }
    
            /// ok ? ///
            if(S_ISDIR(dfstat.st_mode)) {
                no_mkdir = true;
            }
            else {
                if(gCopyOverride == kNone) {
                    const char* str[] = {
                        "No", "Yes", "No(all)", "Yes(all)", "Cancel"
                    };
           
                    char buf[1024];
                    sprintf(buf, "%s override?", dfile);
                    int ret = select_str(buf, (char**)str, 5, 4);
                    switch(ret) {
                    case 0:
                        return true;
                    case 1:
                        file_remove(dfile);
                        break;
                    case 2:
                        gCopyOverride = kNoAll;
                        return true;
                    case 3:
                        file_remove(dfile);
                        gCopyOverride = kYesAll;
                        break;
                    case 4:
                        gCopyOverride = kCancel;
                        return false;
                    }
                }
                else if(gCopyOverride == kYesAll) {
                    file_remove(dfile);
                }
                else if(gCopyOverride == kNoAll) {
                    return true;
                }
            }
        }
        
        DIR* dir = opendir(spath);
        if(dir == NULL) {
            err_msg("file_copy: opendir err(%s)", spath);
            return false;
        }

        if(!no_mkdir) {
            if(mkdir(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP
                              |S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH)) <0)
            {
                closedir(dir);
                err_msg("file_copy: mkdir err(%s)", dfile);
                return false;
            }
        }
    
        struct dirent* entry;
        while(entry = readdir(dir)) {
            if(strcmp(entry->d_name, ".") != 0
                && strcmp(entry->d_name, "..") != 0)
            {
                char spath2[PATH_MAX];

                strcpy(spath2, spath);
                strcat(spath2, "/");
                strcat(spath2, entry->d_name);

                char dfile2[PATH_MAX];

                strcpy(dfile2, dfile);
                strcat(dfile2, "/");

                if(!file_copy(spath2, dfile2)) {
                    gCopyOverride = kCancel;
                    closedir(dir);
                    return false;
                }
            }
        }
    
        closedir(dir);    
    }
    /// file ///
    else {
        /// override ? ///
        if(access(dfile, F_OK) == 0) {
            if(gCopyOverride == kNone) {
                const char* str[] = {
                    "No", "Yes", "No(all)", "Yes(all)", "Cancel"
                };
       
                char buf[1024];
                sprintf(buf, "%s override?", dfile);
                int ret = select_str(buf, (char**)str, 5, 4);
                switch(ret) {
                case 0:
                    return true;
                case 1:
                    file_remove(dfile);
                    break;
                case 2:
                    gCopyOverride = kNoAll;
                    return true;
                case 3:
                    file_remove(dfile);
                    gCopyOverride = kYesAll;
                    break;
                case 4:
                    gCopyOverride = kCancel;
                    return false;
                }
            }
            else if(gCopyOverride == kYesAll) {
                file_remove(dfile);
            }
            else if(gCopyOverride == kNoAll) {
                return true;
            }
        }

        /// sym link ///
        if(S_ISLNK(stat_.st_mode)) {
            char link[PATH_MAX];
            int n = readlink(spath, link, PATH_MAX);
            if(n < 0) {
                err_msg("file_copy: readlink err(%s)", spath);
                return false;
            }
            link[n] = 0;

            if(symlink(link, dfile) < 0) {
                err_msg("file_copy: symlink err(%s)", dfile);
                return false;
            }
        }
        /// file ///
        else if(S_ISREG(stat_.st_mode)) {
            int fd = open(spath, O_RDONLY);
            if(fd < 0) {
                err_msg("file_copy: open(read) err(%s)", spath);
                return false;
            }
            int fd2 = open(dfile, O_WRONLY|O_TRUNC|O_CREAT
                  , stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP
                                      |S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH)
                );
            if(fd2 < 0) {
                err_msg("file_copy: open(write) err(%s)", dfile);
                close(fd);
                return false;
            }

            char buf[BUFSIZ];
            while(1) {
                int n = read(fd, buf, BUFSIZ-1);
                if(n < 0) {
                    err_msg("file_copy: read err(%s)", spath);
                    close(fd);
                    close(fd2);
                    return false;
                }
                
                if(n == 0) {
                    break;
                }

                if(write(fd2, buf, n) < 0) {
                    err_msg("file_copy: write err(%s)", dfile);
                    close(fd);
                    close(fd2);
                    return false;
                }
            }
        
            if(close(fd) < 0) {
                err_msg("file_copy: close err(%s)", spath);
                return false;
            }
            if(close(fd2) < 0) {
                err_msg("file_copy: close err fd2(%s)", dfile);
                return false;
            }
        }
        /// non suport file ///
        else {
            err_msg("file_copy: sorry, non support file type(%s)", spath);
            return false;
        }
    }

    return true;
}

void file_move(char* spath, char* dpath)
{
    if(file_copy(spath, dpath)) file_remove(spath);
}

/// don't put '/' at last
void file_remove(char* path)
{
    /// key input ? ///
    fd_set mask;

    FD_ZERO(&mask);
    FD_SET(0, &mask);

    struct timeval tv;
    tv.tv_sec = 0;
    tv.tv_usec = 0;

    select(1, &mask, NULL, NULL, &tv);
    
    if(FD_ISSET(0, &mask)) {
        int key = getch();

        if(key == 3 || key == 7) {    // CTRL-C and CTRL-G
            gCopyOverride = kCancel;
        }
    }

    /// cancel ///
    if(gCopyOverride == kCancel) {
        return;
    }
    
    struct stat stat_;

    if(lstat(path, &stat_) < 0) {
        err_msg("file_remove: stat err(%s)", path);
        return;
    }

    /// file ///
    if(!S_ISDIR(stat_.st_mode)) {
        if(unlink(path) < 0) {
            err_msg("file_remove: unlink err(%s)", path);
            return;
        }
    }
    /// directory ///
    else {
        char buf[1024];
        sprintf(buf, "entering %s", path);
        msg(buf);
        
        DIR* dir = opendir(path);
        if(dir == NULL) {
            err_msg("file_remove: opendir err(%s)", path);
            return;
        }

        struct dirent* entry;
        while(entry = readdir(dir)) {
            if(strcmp(entry->d_name, ".") != 0
                && strcmp(entry->d_name, "..") != 0)
            {
                char path2[PATH_MAX];

                strcpy(path2, path);
                strcat(path2, "/");
                strcat(path2, entry->d_name);

                file_remove(path2);
            }
        }

        closedir(dir);

        if(rmdir(path) < 0) {
            err_msg("file_remove: rmdir err(%s)", path);
            return;
        }
    }
}
