#include "common.h"

#include "config.h"

#if defined(HAVE_CURSES_H)
#include <curses.h>
#elif defined(HAVE_NCURSES_H)
#include <ncurses.h>
#elif defined(HAVE_NCURSES_NCURSES_H)
#include <ncurses/ncurses.h>
#endif

#include <unistd.h>
#include <sys/stat.h>
#include <limits.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/select.h>
#include <termios.h>
#include <stdarg.h>
#include <time.h>
#include <ctype.h>
#include <locale.h>
#include <pwd.h>
#include <dirent.h>
#include <stdlib.h>
#include <fcntl.h>
#include <termios.h>

#ifdef __LINUX__

#include <limits.h>
/*
#define __USE_POSIX
#define __USE_XOPEN_EXTENDED
#define __USE_POSIX199309
#define __USE_UNIX98
*/
#include <signal.h>
/*
#undef __USE_POSIX
#undef __USE_XOPEN_EXTENDED
#undef __USE_POSIX199309
#undef __USE_UNIX98
*/
#include <sys/wait.h>

#else

#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>

#endif
/*
#if !defined(__DARWIN__)
extern int kill (__pid_t __pid, int __sig) __THROW;
extern int setenv(const char *name, const char *value, int overwrite);
extern void srandom (unsigned int __seed) __THROW;
#endif
*/

int gMainLoop = -1;
void (*gView)() = NULL;          // 登録描写関数

char gTempDir[PATH_MAX];         // 一時ディレクトリ
char gHomeDir[PATH_MAX];         // ホームディレクトリ

///////////////////////////////////////////////////
// キーボード入力処理
///////////////////////////////////////////////////
static void input(int meta, int key)
{
    if(key == 26) {         // CTRL-Z
        mendwin();
        mreset_tty();
        kill(getpid(), SIGSTOP);
        minitscr();
    }
    else if(gActiveMenu) {
        /// メニュー ///
        menu_input(gActiveMenu, meta, key);
    }
    /// インクリメンタルサーチ ///
    else if(gISearch) {
        isearch_input(meta, key);
    }
    else {
        /// ファイラー ///
        filer_input(meta, key);
    }
}

///////////////////////////////////////////////////
// 描写
///////////////////////////////////////////////////
static void job_view()
{
    const int maxy = mgetmaxy();
    const int maxx = mgetmaxx();

    const int size = xyzsh_job_num();
    int i;
    for(i=0; i<size; i++) {
        char* title = xyzsh_job_title(i);

        const int x = (maxx/size)*i;
        char buf[BUFSIZ];
        snprintf(buf, BUFSIZ, "[%d]%s", i+1, title);
        char buf2[BUFSIZ];
        
        str_cut(gKanjiCode, buf, maxx/size, buf2, BUFSIZ);

        mmvprintw(maxy-3, x, "%s", buf2);
    }
}

static void cmdline_view_filer()
{
    const int maxx = mgetmaxx();        // 画面サイズ
    const int maxy = mgetmaxy();
    
    stack_start_stack();

    sObject* fun = FUN_NEW_STACK(NULL);
    sObject* stackframe = UOBJECT_NEW_GC(8, gxyzshObject, "_stackframe", FALSE);
    vector_add(gStackFrames, stackframe);
    //uobject_init(stackframe);
    SFUN(fun).mLocalObjects = stackframe;

    sObject* nextout = FD_NEW_STACK(kFDKindBuf, 0);

    int rcode;
    if(!run(gMFiler4Prompt, gStdin, nextout, &rcode, gCurrentObject, fun)) {
        if(rcode == RCODE_BREAK) {
            fprintf(stderr, "invalid break. Not in a loop\n");
        }
        else if(rcode == RCODE_RETURN) {
            fprintf(stderr, "invalid return. Not in a function\n");
        }
        else if(rcode == RCODE_EXIT) {
            fprintf(stderr, "invalid exit. In the prompt\n");
        }
        else {
            fprintf(stderr, "run time error\n");
            fprintf(stderr, "%s", string_c_str(gErrMsg));
        }
    }

    mmvprintw(maxy-2, 0, SFD(nextout).fdbuf.mBuf);

    (void)vector_pop_back(gStackFrames);
    stack_end_stack();

    /// カーソル下のファイルの情報を描写 ///          
    char buf[1024];
    sFile* file = filer_cursor_file(adir());

    char permission[12];
    strncpy(permission, "----------", 12);

    if(S_ISDIR(file->mLStat.st_mode)) permission[0] = 'd';  // ファイルの種別
    if(S_ISCHR(file->mLStat.st_mode)) permission[0] = 'c';
    if(S_ISBLK(file->mLStat.st_mode)) permission[0] = 'b';
    if(S_ISFIFO(file->mLStat.st_mode)) permission[0] = 'p';
    if(S_ISLNK(file->mLStat.st_mode)) permission[0] = 'l';
    if(S_ISSOCK(file->mLStat.st_mode)) permission[0] = 's';

    mode_t smode = file->mLStat.st_mode;        // パーミッション
    if(smode & S_IRUSR) permission[1] = 'r';
    if(smode & S_IWUSR) permission[2] = 'w';
    if(smode & S_IXUSR) permission[3] = 'x';
    if(smode & S_IRGRP) permission[4] = 'r';
    if(smode & S_IWGRP) permission[5] = 'w';
    if(smode & S_IXGRP) permission[6] = 'x';
    if(smode & S_IROTH) permission[7] = 'r';
    if(smode & S_IWOTH) permission[8] = 'w';
    if(smode & S_IXOTH) permission[9] = 'x';
    if(smode & S_ISUID) permission[3] = 's';
    if(smode & S_ISGID) permission[6] = 's';
#if defined(S_ISTXT)
    if(smode & S_ISTXT) permission[9] = 't';
#endif
#if defined(S_ISVTX)
    if(smode & S_ISVTX) permission[9] = 't';
#endif
    
    permission[10] = 0;

    time_t t = file->mLStat.st_mtime;                   // 時間
    struct tm* tm_ = (struct tm*)localtime(&t);
    
    char buf2[1024];

    char owner[256];
    char* tmp = string_c_str(file->mUser);
    if(tmp) {
        str_cut2(gKanjiCode, tmp, 8, owner, 256);
    }
    else {
        snprintf(owner, 256, "%d", file->mLStat.st_uid);
    }

    char group[256];
    tmp = string_c_str(file->mGroup);
    if(tmp) {
        str_cut2(gKanjiCode, tmp, 7, group, 256);
    }
    else {
        snprintf(group, 256, "%d", file->mLStat.st_gid);
    }

    char size_buf[256];
    int width = filer_make_size_str(size_buf, file->mLStat.st_size);

    if(S_ISDIR(file->mStat.st_mode)) {
        strncpy(size_buf, "<DIR>", 256);
    }

    char format[128];
    strncpy(format, "%", 128);
    snprintf(format + 1, 126, "%d", width);
    strncat(format, "s", 128);

    char size_buf2[256];
    snprintf(size_buf2, 256, format, size_buf);
    if(!S_ISLNK(file->mLStat.st_mode)) {
        int year = tm_->tm_year-100;
        if(year < 0) year+=100;
        while(year > 100) year-=100;
        
        snprintf(buf2, 1024, 
            "%s %3d %-8s %-7s%s %02d-%02d-%02d %02d:%02d %s"
            //, "%s %3d %-8s %-7s%10lld %02d-%02d-%02d %02d:%02d %s"
            , permission, file->mLStat.st_nlink
            , owner, group
                                
            , size_buf2
            
            , year, tm_->tm_mon+1, tm_->tm_mday
            , tm_->tm_hour, tm_->tm_min
            , string_c_str(file->mNameView));
            
        char buf3[1024];
        str_cut2(gKanjiCode, buf2, maxx-1, buf3, 1024);
                            
        mmvprintw(maxy-1, 0, "%s", buf3);
    }
    else {
        snprintf(buf, 1024,
           "%s %3d %s%s%s %02d-%02d-%02d %02d:%02d %s -> %s"
           //, "%s %3d %s%s%10lld %02d-%02d-%02d %02d:%02d %s -> %s"
           , permission, file->mLStat.st_nlink
           , owner, group
                                
           , size_buf2

           , tm_->tm_year-100, tm_->tm_mon+1, tm_->tm_mday
           , tm_->tm_hour, tm_->tm_min
           , string_c_str(file->mNameView)
           , string_c_str(file->mLinkTo));
           
        char buf2[1024];
        str_cut2(gKanjiCode, buf, maxx-1, buf2, 1024);
                   
        mmvprintw(maxy-1, 0, "%s", buf2);
    }

    mmove(maxy-1, maxx-2);
}

void view()
{
    if(gISearch) {
        filer_view(0);
        filer_view(1);
        job_view();

        isearch_view();
    }
    else if(gActiveMenu) {
        filer_view(0);
        filer_view(1);
        job_view();

        menu_view(gActiveMenu);
        cmdline_view_filer();
    }
    else {
        filer_view(0);
        filer_view(1);
        job_view();
        cmdline_view_filer();
    }

    if(gView) gView();
}

///////////////////////////////////////////////////
// ランタイムスクリプト実行
///////////////////////////////////////////////////
static BOOL read_rc_file(int argc, char** argv)
{
    char rc_fname[PATH_MAX];

    snprintf(rc_fname, PATH_MAX, "%s/mfiler4.xyzsh", SYSCONFDIR);
    if(access(rc_fname, R_OK) == 0) {
        if(!xyzsh_load_file(rc_fname, argv, argc, gMFiler4)) {
            return FALSE;
        }
    }
    else {
        fprintf(stderr, "can't find %s/mfiler4.xyzsh file\n", SYSCONFDIR);
        return FALSE;
    }

    snprintf(rc_fname, PATH_MAX, "%s/mfiler4.xyzsh", gHomeDir);

    if(access(rc_fname, R_OK) == 0) {
        if(!xyzsh_load_file(rc_fname, argv, argc, gMFiler4)) {
            return FALSE;
        }
    }

    return TRUE;
}

///////////////////////////////////////////////////
// exit時に実行される関数
///////////////////////////////////////////////////
static void atexit_fun()
{
}

///////////////////////////////////////////////////
// シグナル処理
///////////////////////////////////////////////////
static void sig_winch(int s)
{
    if(gMainLoop == -1) {
        mclear_immediately();       // 画面の再描写
        view();
        mrefresh();
    }
}

static void sig_cont(int s)
{
    if(gMainLoop == -1) {
        mclear_immediately();       // 画面の再描写
        view();
        mrefresh();
    }
}


void set_signal_mfiler()
{
    xyzsh_set_signal();

    struct sigaction sa4;

    memset(&sa4, 0, sizeof(sa4));
    sa4.sa_handler = SIG_IGN;
    if(sigaction(SIGCONT, &sa4, NULL) < 0) {
        perror("sigaction4");
        exit(1);
    }

    struct sigaction sa5;

    memset(&sa5, 0, sizeof(sa5));
    sa5.sa_handler = sig_winch;
    if(sigaction(SIGWINCH, &sa5, NULL) < 0) {
        perror("sigaction5");
        exit(1);
    }
}

///////////////////////////////////////////////////
// メイン関数
///////////////////////////////////////////////////
static void usage()
{
    printf("usage mfiler4 [-c command] [--version ] [ filer initial directory or script file]\n\n");

    printf("-c : eval a command on mfiler4\n");
    printf("--version : display mfiler4 version\n");

    exit(0);
}

static void version()
{
    printf("mfiler4 version %s with shell scripting system xyzsh version %s.", getenv("VERSION"), getenv("XYZSH_VERSION"));
    puts("This program is a 2pain file manager with a embedded shell scripting system \"xyzsh\". mfiler4 is developped by ab25cq.");
    puts("compiled with");

#if defined(HAVE_MIGEMO_H)
    puts("+migemo");
#else
    puts("-migemo");
#endif
    puts("+oniguruma");
    puts("+xyzsh");
    puts("+math");
    puts("+curses");
    
    exit(0);
}

static void set_mfenv()
{
    setenv("#", "-1", 1);
    setenv("VIEW_OPTION", "2pain", 1);
    setenv("SYSCONFDIR", SYSCONFDIR, 1);
    setenv("MF4DOCDIR", DOCDIR, 1);
    setenv("PROMPT", "can't read run time script '.mfiler4'. press q key to quit", 1);
    setenv("VIEW_FILE_SIZE", "Normal", 1);

    setenv("MF4HOME", gHomeDir, 1);
    setenv("MF4TEMP", gTempDir, 1);
    setenv("DATADIR", DATAROOTDIR, 1);
    setenv("DOTDIR_MASK", "0", 1);

    char buf[128];

    snprintf(buf, 128, "%d", KEY_UP);
    setenv("key_up", buf, 1);
    snprintf(buf, 128, "%d", KEY_RIGHT);
    setenv("key_right", buf, 1);
    snprintf(buf, 128, "%d", KEY_DOWN);
    setenv("key_down", buf, 1);
    snprintf(buf, 128, "%d", KEY_LEFT);
    setenv("key_left", buf, 1);
    snprintf(buf, 128, "%d", KEY_IC);
    setenv("key_insert", buf, 1);
    snprintf(buf, 128, "%d", KEY_DC);
    setenv("key_delete", buf, 1);
    snprintf(buf, 128, "%d", KEY_HOME);
    setenv("key_home", buf, 1);
    snprintf(buf, 128, "%d", KEY_END);
    setenv("key_end", buf, 1);
    snprintf(buf, 128, "%d", KEY_PPAGE);
    setenv("key_pageup", buf, 1);
    snprintf(buf, 128, "%d", KEY_NPAGE);
    setenv("key_pagedown", buf, 1);
    snprintf(buf, 128, "%d", KEY_F(13));
    setenv("key_meta_left", buf, 1);
    snprintf(buf,128, "%d", KEY_F(14));
    setenv("key_meta_right", buf, 1);
    snprintf(buf,128, "%d", KEY_F(15));
    setenv("key_meta_up", buf, 1);
    snprintf(buf,128, "%d", KEY_F(16));
    setenv("key_meta_down", buf, 1);
    snprintf(buf,128, "%d", KEY_F(17));
    setenv("key_ctrl_left", buf, 1);
    snprintf(buf,128, "%d", KEY_F(18));
    setenv("key_ctrl_right", buf, 1);
    snprintf(buf,128, "%d", KEY_F(19));
    setenv("key_ctrl_up", buf, 1);
    snprintf(buf,128, "%d", KEY_F(20));
    setenv("key_ctrl_down", buf, 1);
    snprintf(buf,128, "%d", KEY_F(21));
    setenv("key_ctrl_delete", buf, 1);
    snprintf(buf,128, "%d", 10);
    setenv("key_enter", buf, 1);
    snprintf(buf,128, "%d", KEY_BACKSPACE);
    setenv("key_backspace", buf, 1);
    snprintf(buf,128, "%d", KEY_F(1));
    setenv("key_f1", buf, 1);
    snprintf(buf,128, "%d", KEY_F(2));
    setenv("key_f2", buf, 1);
    snprintf(buf,128, "%d", KEY_F(3));
    setenv("key_f3", buf, 1);
    snprintf(buf,128, "%d", KEY_F(4));
    setenv("key_f4", buf, 1);
    snprintf(buf,128, "%d", KEY_F(5));
    setenv("key_f5", buf, 1);
    snprintf(buf,128, "%d", KEY_F(6));
    setenv("key_f6", buf, 1);
    snprintf(buf,128, "%d", KEY_F(7));
    setenv("key_f7", buf, 1);
    snprintf(buf,128, "%d", KEY_F(8));
    setenv("key_f8", buf, 1);
    snprintf(buf,128, "%d", KEY_F(9));
    setenv("key_f9", buf, 1);
    snprintf(buf,128, "%d", KEY_F(10));
    setenv("key_f10", buf, 1);
    snprintf(buf,128, "%d", KEY_F(11));
    setenv("key_f11", buf, 1);
    snprintf(buf,128, "%d", KEY_F(12));
    setenv("key_f12", buf, 1);
    snprintf(buf,128, "%d", 'a');
    setenv("key_a", buf, 1);
    snprintf(buf,128, "%d", 'b');
    setenv("key_b", buf, 1);
    snprintf(buf,128, "%d", 'c');
    setenv("key_c", buf, 1);
    snprintf(buf,128, "%d", 'd');
    setenv("key_d", buf, 1);
    snprintf(buf,128, "%d", 'e');
    setenv("key_e", buf, 1);
    snprintf(buf,128, "%d", 'f');
    setenv("key_f", buf, 1);
    snprintf(buf,128, "%d", 'g');
    setenv("key_g", buf, 1);
    snprintf(buf,128, "%d", 'h');
    setenv("key_h", buf, 1);
    snprintf(buf,128, "%d", 'i');
    setenv("key_i", buf, 1);
    snprintf(buf,128, "%d", 'j');
    setenv("key_j", buf, 1);
    snprintf(buf,128, "%d", 'k');
    setenv("key_k", buf, 1);
    snprintf(buf,128, "%d", 'l');
    setenv("key_l", buf, 1);
    snprintf(buf,128, "%d", 'm');
    setenv("key_m", buf, 1);
    snprintf(buf,128, "%d", 'n');
    setenv("key_n", buf, 1);
    snprintf(buf,128, "%d", 'o');
    setenv("key_o", buf, 1);
    snprintf(buf,128, "%d", 'p');
    setenv("key_p", buf, 1);
    snprintf(buf,128, "%d", 'q');
    setenv("key_q", buf, 1);
    snprintf(buf,128, "%d", 'r');
    setenv("key_r", buf, 1);
    snprintf(buf,128, "%d", 's');
    setenv("key_s", buf, 1);
    snprintf(buf,128, "%d", 't');
    setenv("key_t", buf, 1);
    snprintf(buf,128, "%d", 'u');
    setenv("key_u", buf, 1);
    snprintf(buf,128, "%d", 'v');
    setenv("key_v", buf, 1);
    snprintf(buf,128, "%d", 'w');
    setenv("key_w", buf, 1);
    snprintf(buf,128, "%d", 'x');
    setenv("key_x", buf, 1);
    snprintf(buf,128, "%d", 'y');
    setenv("key_y", buf, 1);
    snprintf(buf,128, "%d", 'z');
    setenv("key_z", buf, 1);
    snprintf(buf,128, "%d", 'A');
    setenv("key_A", buf, 1);
    snprintf(buf,128, "%d", 'B');
    setenv("key_B", buf, 1);
    snprintf(buf,128, "%d", 'C');
    setenv("key_C", buf, 1);
    snprintf(buf,128, "%d", 'D');
    setenv("key_D", buf, 1);
    snprintf(buf,128, "%d", 'E');
    setenv("key_E", buf, 1);
    snprintf(buf,128, "%d", 'F');
    setenv("key_F", buf, 1);
    snprintf(buf,128, "%d", 'G');
    setenv("key_G", buf, 1);
    snprintf(buf,128, "%d", 'H');
    setenv("key_H", buf, 1);
    snprintf(buf,128, "%d", 'I');
    setenv("key_I", buf, 1);
    snprintf(buf,128, "%d", 'J');
    setenv("key_J", buf, 1);
    snprintf(buf,128, "%d", 'K');
    setenv("key_K", buf, 1);
    snprintf(buf,128, "%d", 'L');
    setenv("key_L", buf, 1);
    snprintf(buf,128, "%d", 'M');
    setenv("key_M", buf, 1);
    snprintf(buf,128, "%d", 'N');
    setenv("key_N", buf, 1);
    snprintf(buf,128, "%d", 'O');
    setenv("key_O", buf, 1);
    snprintf(buf,128, "%d", 'P');
    setenv("key_P", buf, 1);
    snprintf(buf,128, "%d", 'Q');
    setenv("key_Q", buf, 1);
    snprintf(buf,128, "%d", 'R');
    setenv("key_R", buf, 1);
    snprintf(buf,128, "%d", 'S');
    setenv("key_S", buf, 1);
    snprintf(buf,128, "%d", 'T');
    setenv("key_T", buf, 1);
    snprintf(buf,128, "%d", 'U');
    setenv("key_U", buf, 1);
    snprintf(buf,128, "%d", 'V');
    setenv("key_V", buf, 1);
    snprintf(buf,128, "%d", 'W');
    setenv("key_W", buf, 1);
    snprintf(buf,128, "%d", 'X');
    setenv("key_X", buf, 1);
    snprintf(buf,128, "%d", 'Y');
    setenv("key_Y", buf, 1);
    snprintf(buf,128, "%d", 'Z');
    setenv("key_Z", buf, 1);
    snprintf(buf,128, "%d", 32);
    setenv("key_space", buf, 1);
    snprintf(buf,128, "%d", 0);
    setenv("key_ctrl_space", buf, 1);
    snprintf(buf,128, "%d", 1);
    setenv("key_ctrl_a", buf, 1);
    snprintf(buf,128, "%d", 2);
    setenv("key_ctrl_b", buf, 1);
    snprintf(buf,128, "%d", 3);
    setenv("key_ctrl_c", buf, 1);
    snprintf(buf,128, "%d", 4);
    setenv("key_ctrl_d", buf, 1);
    snprintf(buf,128, "%d", 5);
    setenv("key_ctrl_e", buf, 1);
    snprintf(buf,128, "%d", 6);
    setenv("key_ctrl_f", buf, 1);
    snprintf(buf,128, "%d", 7);
    setenv("key_ctrl_g", buf, 1);
    snprintf(buf,128, "%d", 8);
    setenv("key_ctrl_h", buf, 1);
    snprintf(buf,128, "%d", 9);
    setenv("key_ctrl_i", buf, 1);
    snprintf(buf,128, "%d", 10);
    setenv("key_ctrl_j", buf, 1);
    snprintf(buf,128, "%d", 11);
    setenv("key_ctrl_k", buf, 1);
    snprintf(buf,128, "%d", 12);
    setenv("key_ctrl_l", buf, 1);
    snprintf(buf,128, "%d", 13);
    setenv("key_ctrl_m", buf, 1);
    snprintf(buf,128, "%d", 14);
    setenv("key_ctrl_n", buf, 1);
    snprintf(buf,128, "%d", 15);
    setenv("key_ctrl_o", buf, 1);
    snprintf(buf,128, "%d", 16);
    setenv("key_ctrl_p", buf, 1);
    snprintf(buf,128, "%d", 17);
    setenv("key_ctrl_q", buf, 1);
    snprintf(buf,128, "%d", 18);
    setenv("key_ctrl_r", buf, 1);
    snprintf(buf,128, "%d", 19);
    setenv("key_ctrl_s", buf, 1);
    snprintf(buf,128, "%d", 20);
    setenv("key_ctrl_t", buf, 1);
    snprintf(buf,128, "%d", 21);
    setenv("key_ctrl_u", buf, 1);
    snprintf(buf,128, "%d", 22);
    setenv("key_ctrl_v", buf, 1);
    snprintf(buf,128, "%d", 23);
    setenv("key_ctrl_w", buf, 1);
    snprintf(buf,128, "%d", 24);
    setenv("key_ctrl_x", buf, 1);
    snprintf(buf,128, "%d", 25);
    setenv("key_ctrl_y", buf, 1);
    snprintf(buf,128, "%d", 26);
    setenv("key_ctrl_z", buf, 1);
    snprintf(buf,128, "%d", 27);
    setenv("key_escape", buf, 1);
    snprintf(buf,128, "%d", 9);
    setenv("key_tab", buf, 1);
    snprintf(buf,128, "%d", '0');
    setenv("key_0", buf, 1);
    snprintf(buf,128, "%d", '1');
    setenv("key_1", buf, 1);
    snprintf(buf,128, "%d", '2');
    setenv("key_2", buf, 1);
    snprintf(buf,128, "%d", '3');
    setenv("key_3", buf, 1);
    snprintf(buf,128, "%d", '4');
    setenv("key_4", buf, 1);
    snprintf(buf,128, "%d", '5');
    setenv("key_5", buf, 1);
    snprintf(buf,128, "%d", '6');
    setenv("key_6", buf, 1);
    snprintf(buf,128, "%d", '7');
    setenv("key_7", buf, 1);
    snprintf(buf,128, "%d", '8');
    setenv("key_8", buf, 1);
    snprintf(buf,128, "%d", '9');
    setenv("key_9", buf, 1);
    snprintf(buf,128, "%d", '!');
    setenv("key_exclam", buf, 1);
    snprintf(buf,128, "%d", '"');
    setenv("key_dquote", buf, 1);
    snprintf(buf,128, "%d", '#');
    setenv("key_sharp", buf, 1);
    snprintf(buf,128, "%d", '$');
    setenv("key_dollar", buf, 1);
    snprintf(buf,128, "%d", '%');
    setenv("key_percent", buf, 1);
    snprintf(buf,128, "%d", '&');
    setenv("key_and", buf, 1);
    snprintf(buf,128, "%d", '\'');
    setenv("key_squote", buf, 1);
    snprintf(buf,128, "%d", '(');
    setenv("key_lparen", buf, 1);
    snprintf(buf,128, "%d", ')');
    setenv("key_rparen", buf, 1);
    snprintf(buf,128, "%d", '~');
    setenv("key_tilda", buf, 1);
    snprintf(buf,128, "%d", '=');
    setenv("key_equal", buf, 1);
    snprintf(buf,128, "%d", '-');
    setenv("key_minus", buf, 1);
    snprintf(buf,128, "%d", '^');
    setenv("key_cup", buf, 1);
    snprintf(buf,128, "%d", '|');
    setenv("key_vbar", buf, 1);
    snprintf(buf,128, "%d", '\\');
    setenv("key_backslash", buf, 1);
    snprintf(buf,128, "%d", '@');
    setenv("key_atmark", buf, 1);
    snprintf(buf,128, "%d", '`');
    setenv("key_bapostrophe", buf, 1);
    snprintf(buf,128, "%d", '{');
    setenv("key_lcurly", buf, 1);
    snprintf(buf,128, "%d", '[');
    setenv("key_lbrack", buf, 1);
    snprintf(buf,128, "%d", '+');
    setenv("key_plus", buf, 1);
    snprintf(buf,128, "%d", ';');
    setenv("key_semicolon", buf, 1);
    snprintf(buf,128, "%d", '*');
    setenv("key_star", buf, 1);
    snprintf(buf,128, "%d", ':');
    setenv("key_colon", buf, 1);
    snprintf(buf,128, "%d", '}');
    setenv("key_rcurly", buf, 1);
    snprintf(buf,128, "%d", ']');
    setenv("key_rbrack", buf, 1);
    snprintf(buf,128, "%d", '<');
    setenv("key_lss", buf, 1);
    snprintf(buf,128, "%d", ',');
    setenv("key_comma", buf, 1);
    snprintf(buf,128, "%d", '>');
    setenv("key_gtr", buf, 1);
    snprintf(buf,128, "%d", '.');
    setenv("key_dot", buf, 1);
    snprintf(buf,128, "%d", '/');
    setenv("key_slash", buf, 1);
    snprintf(buf,128, "%d", '?');
    setenv("key_qmark", buf, 1);
    snprintf(buf,128, "%d", '_');
    setenv("key_underbar", buf, 1);
    snprintf(buf,128, "%d", 0);
    setenv("nometa", buf, 1);
    snprintf(buf,128, "%d", 1);
    setenv("meta", buf, 1);
    snprintf(buf,128, "%d", kCAReverse);
    setenv("ma_reverse", buf, 1);
    snprintf(buf,128, "%d", kCABold);
    setenv("ma_bold", buf, 1);
    snprintf(buf,128, "%d", kCAUnderline);
    setenv("ma_underline", buf, 1);
    snprintf(buf,128, "%d", kCAWhite);
    setenv("ma_white", buf, 1);
    snprintf(buf,128, "%d", kCABlue);
    setenv("ma_blue", buf, 1);
    snprintf(buf,128, "%d", kCACyan);
    setenv("ma_cyan", buf, 1);
    snprintf(buf,128, "%d", kCAGreen);
    setenv("ma_green", buf, 1);
    snprintf(buf,128, "%d", kCAYellow);
    setenv("ma_yellow", buf, 1);
    snprintf(buf,128, "%d", kCAMagenta);
    setenv("ma_magenta", buf, 1);
    snprintf(buf,128, "%d", kCARed);
    setenv("ma_red", buf, 1);
}

static void mygetcwd(char* result, int result_size)
{
    int l;
    
    l = 50;
    while(!getcwd(result, l)) {
         l += 50;
         if(l+1 >= result_size) {
            break;
         }
    }
}

void extname(char* result, int result_size, char* name)
{
    char* p;
    
    for(p = name + strlen(name); p != name; p--) {
        if(*p == '.' && p == name + strlen(name)) {
            strncpy(result, "", result_size);
            return;
        }
        else if(*p == '/') {
            strncpy(result, "", result_size);
            return;
        }
        else if(*p == '.') {
            strncpy(result, p+1, result_size);
            return;
        }
    }

    strncpy(result, "", result_size);
}

int correct_path(char* current_path, char* path, char* path2, int path2_size)
{
    char tmp[PATH_MAX];
    char tmp2[PATH_MAX];
    char tmp3[PATH_MAX];
    char tmp4[PATH_MAX];

    /// 先頭にカレントパスを加える ///
    {
        if(path[0] == '/') {      /// 絶対パスなら必要ない
            strncpy(tmp, path, PATH_MAX);
        }
        else {
            if(current_path == NULL) {
                char cwd[PATH_MAX];
                mygetcwd(cwd, PATH_MAX);
                
                strncpy(tmp, cwd, PATH_MAX);
                strncat(tmp, "/", PATH_MAX);
                strncat(tmp, path, PATH_MAX);
            }
            else {
                strncpy(tmp, current_path, PATH_MAX);
                if(current_path[strlen(current_path)-1] != '/') {
                    strncat(tmp, "/", PATH_MAX);
                }
                strncat(tmp, path, PATH_MAX);
            }
        }

    }

    strncpy(tmp2, tmp, PATH_MAX);

    /// .を削除する ///
    {
        char* p;
        char* p2;
        int i;

        p = tmp3;
        p2 = tmp2;
        while(*p2) {
            if(*p2 == '/' && *(p2+1) == '.' && *(p2+2) != '.' && ((*p2+3)=='/' || *(p2+3)==0)) {
                p2+=2;
            }
            else {
                *p++ = *p2++;
            }
        }
        *p = 0;

        if(*tmp3 == 0) {
            *tmp3 = '/';
            *(tmp3+1) = 0;
        }

    }

    /// ..を削除する ///
    {
        char* p;
        char* p2;

        p = tmp4;
        p2 = tmp3;

        while(*p2) {
            if(*p2 == '/' && *(p2+1) == '.' && *(p2+2) == '.' 
                && *(p2+3) == '/')
            {
                p2 += 3;

                do {
                    p--;

                    if(p < tmp4) {
                        strncpy(path2, "/", path2_size);
                        return FALSE;
                    }
                } while(*p != '/');

                *p = 0;
            }
            else if(*p2 == '/' && *(p2+1) == '.' && *(p2+2) == '.' 
                && *(p2+3) == 0) 
            {
                do {
                    p--;

                    if(p < tmp4) {
                        strncpy(path2, "/", path2_size);
                        return FALSE;
                    }
                } while(*p != '/');

                *p = 0;
                break;
            }
            else {
                *p++ = *p2++;
            }
        }
        *p = 0;
    }

    if(*tmp4 == 0) {
        strncpy(path2, "/", path2_size);
    }
    else {
        strncpy(path2, tmp4, path2_size);
    }

    return TRUE;
}

int main(int argc, char* argv[])
{
    /// メモリリークの監視を開始 ///
    CHECKML_BEGIN(FALSE);

    /// ファイラでもスクリプトでも共通の環境変数を初期化 ///
    setenv("VERSION", "1.0.2", 1);
    setenv("MFILER4_DATAROOTDIR", DATAROOTDIR, 1);

    /// mfiler4のホームディレクトリを保存しておく ///
    char* home = getenv("HOME");
    if(home == NULL) {
        fprintf(stderr, "$HOME is NULL.exit");
        exit(1);
    }
    snprintf(gHomeDir, PATH_MAX, "%s/.mfiler4", home);

    if(access(gHomeDir, F_OK) != 0) {
        char buf[PATH_MAX];
        snprintf(buf, PATH_MAX, "mkdir -p %s", gHomeDir);
        if(system(buf) < 0) {
            fprintf(stderr, "can't make directory (%s)", gHomeDir);
            exit(1);
        }
    }

    if(chmod(gHomeDir, S_IRUSR|S_IWUSR|S_IXUSR) < 0) {
        fprintf(stderr, "can't chmod %s", gHomeDir);
        exit(1);
    }

    /// mfiler4の一時ディレクトリを保存しておく ///
    snprintf(gTempDir, PATH_MAX, "%s/temp", gHomeDir);
    if(access(gTempDir, F_OK) != 0) {
        if(mkdir(gTempDir, 0700) < 0) {
            fprintf(stderr, "can't mkdir %s", gTempDir);
            exit(1);
        }
    }

    if(chmod(gTempDir, S_IRUSR|S_IWUSR|S_IXUSR) < 0) {
        fprintf(stderr, "can't chmod %s", gTempDir);
        exit(1);
    }

    /// 乱数初期化 ///
    srandom(1000);

    /// 端末の設定を保存 ///
    //msave_ttysettings();

    /// 端末を初期化 ///
    mreset_tty();

    /// オプション処理 ///
    char send_cmd[BUFSIZ];          // コマンドがセットされていたら
    char file_name[PATH_MAX];       // ファイル名が引数の場合
    file_name[0] = 0;

    int i;
    int pid;
    for(i=1 ; i<argc; i++){
        // ファイル名
        if(argv[i][0] != '-') {         // オプションじゃなければ
            xstrncpy(file_name, argv[i], PATH_MAX);     // ファイル名とみなす
        }
        // 引数
        else {
            if(strcmp(argv[i], "--version") == 0) version();

            switch (argv[i][1]){
                case 'c':
                    if(i+1 < argc) {
                        snprintf(send_cmd, BUFSIZ, "c%s", argv[i+1]);
                        i++;
                    }
                    else {
                        usage();
                    }
                    break;

                default:
                    usage();
            }
        }
    }

    /// 標準入力と出力が制御端末かどうか確認 ///
    if(!isatty(0) || !isatty(1)) {
        fprintf(stderr, "standard input is not a tty\n");
        return 1;
    }

    /// 環境変数設定 ///
    set_mfenv();

    /// モジュールの初期化 ///
    xyzsh_init(kATConsoleApp, FALSE);
    menu_init();
    filer_init();
    commands_init();

    /// シグナルを設定 ///
    set_signal_mfiler();

    /// localeの設定 ///
    if(gTerminalKanjiCode == kTKUtf8) setlocale(LC_CTYPE, "ja_JP.UTF-8");

    /// atexit登録 ///
    atexit(atexit_fun);

    /// ディレクトリを作成 ///
    if(vector_count(gDirs) == 0) {
        if(strcmp(file_name, "") == 0) {
            char cwd[PATH_MAX];
            mygetcwd(cwd, PATH_MAX);
            xstrncat(cwd, "/", PATH_MAX);

            if(!filer_new_dir(cwd, FALSE, ".+")) {
                fprintf(stderr, "invalid current directory\n");
                exit(1);
            }
            if(!filer_new_dir(cwd, FALSE, ".+")) {
                fprintf(stderr, "invalid current directory\n");
                exit(1);
            }
        }
        else {
            if(!filer_new_dir(file_name, FALSE, ".+")) {
                fprintf(stderr, "invalid directory(%s)\n", file_name);
                exit(1);
            }
            if(!filer_new_dir(file_name, FALSE, ".+")) {
                fprintf(stderr, "invalid directory(%s)\n", file_name);
                exit(1);
            }
        }
    }

    (void)filer_activate(0);

    /// モジュールの初期化(ランタイムスクリプト以降の方が良い奴 ///
    isearch_init();

    /// 初期キーバインドと初期フック ///
    if(!filer_add_keycommand2(0, 'q', "quit", "system", FALSE)) {
        /// ジョブの消去 ///
        xyzsh_kill_all_jobs();

        /// モジュール解放 ///
        isearch_final();
        menu_final();
        filer_final();
        xyzsh_final();
        commands_final();

        mreset_tty();
        //mrestore_ttysettings();

        /// メモリリーク検出 ///
        CHECKML_END();

        /// 改行 ///
        puts("");

        mreset_tty();
        exit(1);
    }
    if(!filer_add_keycommand2(0, 3, "quit", "system", FALSE)) {
        /// ジョブの消去 ///
        xyzsh_kill_all_jobs();

        /// モジュール解放 ///
        isearch_final();
        menu_final();
        filer_final();
        xyzsh_final();
        commands_final();

        mreset_tty();
        //mrestore_ttysettings();

        /// メモリリーク検出 ///
        CHECKML_END();

        /// 改行 ///
        puts("");

        mreset_tty();
        exit(1);
    }

    /// ランタイムスクリプト実行 ///
    gMainLoop = -1;

    if(!read_rc_file(argc, argv)) {
        /// ジョブの消去 ///
        xyzsh_kill_all_jobs();

        /// モジュール解放 ///
        isearch_final();
        menu_final();
        filer_final();
        xyzsh_final();
        commands_final();

        mreset_tty();
        //mrestore_ttysettings();

        /// メモリリーク検出 ///
        CHECKML_END();

        /// 改行 ///
        puts("");

        mreset_tty();
        exit(1);
    }
    
    /// カーシスの起動 ///
    minitscr();

    /// メインループ ///
    fd_set mask;
    fd_set read_ok;

    FD_ZERO(&mask);
    FD_SET(0, &mask);

    while(gMainLoop == -1) {
        /// 描写 ///
        mclear();
        view();
        mrefresh();

        const int maxy = mgetmaxy();
        mmove_immediately(maxy-2, 0);

        /// selectで入力待ち ///
        read_ok = mask;
        if(!mkbuf_exist()) {     // キーボードのバッファがないならselect
            if(xyzsh_job_num() > 0) {
                struct timeval tv;
                tv.tv_sec = 5;
                tv.tv_usec = 0;

                select(5, &read_ok, NULL, NULL, &tv);
            }
            else {
                select(5, &read_ok, NULL, NULL, NULL);
            }
        }

        /// キー処理 ///
        if(FD_ISSET(0, &read_ok) || mkbuf_exist()) {    // キー入力があったか、キーボードのバッファがあるなら
            int meta;
            int key = mgetch_nonblock(&meta);

            /// ESCキー ///
            if(key == -1) {
            }
            else if(meta) {
                input(1, key);
            }
            
            /// meta key ///
            else if(key >= kKeyMetaFirst 
                    && key <= kKeyMetaFirst + 127) 
            {
                input(1, key - kKeyMetaFirst );
            }

            /// normal key ///
            else {
               input(0, key);
            }
        }

        /// ジョブ処理 ///
        xyzsh_wait_background_job();
    }

    /// カーシス終わり ///
    mendwin();

    /// ジョブの消去 ///
    xyzsh_kill_all_jobs();

    /// モジュール解放 ///
    isearch_final();
    menu_final();
    filer_final();
    xyzsh_final();
    commands_final();

    mreset_tty();
    //mrestore_ttysettings();

    /// メモリリーク検出 ///
    CHECKML_END();

    /// 改行 ///
    puts("");

    mreset_tty();

    return gMainLoop;
}

