/* ---------------------------------------------------------------------------------------------- */
/* MHash-384 - Simple fast portable secure hashing library                                        */
/* Copyright(c) 2016-2020 LoRd_MuldeR <mulder2@gmx.de>                                            */
/*                                                                                                */
/* Permission is hereby granted, free of charge, to any person obtaining a copy of this software  */
/* and associated documentation files (the "Software"), to deal in the Software without           */
/* restriction, including without limitation the rights to use, copy, modify, merge, publish,     */
/* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the  */
/* Software is furnished to do so, subject to the following conditions:                           */
/*                                                                                                */
/* The above copyright notice and this permission notice shall be included in all copies or       */
/* substantial portions of the Software.                                                          */
/*                                                                                                */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING  */
/* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND     */
/* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,   */
/* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, */
/* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.        */
/* ---------------------------------------------------------------------------------------------- */

#include "mhash384.h"
#include <memory.h>

/*
 * Version info
 */
#define MHASH384_VERSION_MAJOR 2
#define MHASH384_VERSION_MINOR 0
#define MHASH384_VERSION_PATCH 0

/*
 * C/C++ compiler support
 */
#if defined(_MSC_VER)
#	define ALWAYS_INLINE __forceinline
#elif defined(__GNUC__)
#	define ALWAYS_INLINE __attribute__((always_inline)) inline
#elif defined(__cplusplus)
#	define ALWAYS_INLINE inline
#else
#	define ALWAYS_INLINE
#endif

/*
 * Types
 */
typedef uint64_t ui64_t;
typedef uint32_t ui32_t;
typedef uint16_t ui16_t;
typedef uint8_t  byte_t;

/* ======================================================================== */
/* CONSTANT TABLES                                                          */
/* ======================================================================== */

/*
 * Const
 */
static const ui64_t KMUL = 0x9DDFEA08EB382D69;

/* Table INI
 * 48 byte initialization vector containing the first 384 bits of the number PI
 */
static const ui64_t MHASH384_INI[MHASH384_WORDS] =
{
	0x243F6A8885A308D3, 0x13198A2E03707344, 0xA4093822299F31D0,
	0x082EFA98EC4E6C89, 0x452821E638D01377, 0xBE5466CF34E90C6C
};

/* Table XOR
 * 257x48 byte matrix containing pre-computed 384-bit words with HamD(a,b) >= 182 for each possible pair (a,b) with a != b
 */
static const ui64_t MHASH384_XOR[257U][MHASH384_WORDS] =
{
	{ 0x01DCDF00414B3037, 0xB1B3AF661B8E96F8, 0x944D2873DB393121, 0x73DA9A36662AE755, 0x1F4F318C4ECB56B1, 0xF09743D99C2AA5BC }, /*00*/
	{ 0xA81FBBC6CBBFC954, 0x39DE43648959EDDB, 0x1A641A0BDA01822F, 0xB52E607266932658, 0x2C5B1731AC802084, 0xC2EF10671FC79DD4 }, /*01*/
	{ 0xCF2A8D8E08810046, 0x8B7E9B2089E268F6, 0x930461652C5D2DEC, 0xF096E42223BFC8B9, 0x8DD338964CFE0970, 0x269C342F7CEC60BD }, /*02*/
	{ 0xB970A920D2ED0552, 0x010F894A254DA7E1, 0x8D5F205F9D1A40D8, 0x50C33DCCC3FD5F58, 0xB49F31BDE7D82C57, 0x7CDE04F62A959033 }, /*03*/
	{ 0x49FAB2822B9C84AC, 0x34B8648CD68CBEF1, 0xE5121147BB4126DE, 0xC0B31F54B2FFE00F, 0x2F193DA38E8CC632, 0x058C984B429B8AFC }, /*04*/
	{ 0xE07F9DA44D8C9942, 0xBE2AF9B39ACA65F2, 0x5D3D8FB1466DC295, 0xC0051E3EC3F962C7, 0xF8FDC3CCD4CE2BB6, 0x9024C0EFC6199937 }, /*05*/
	{ 0xA3111377EF01F5EF, 0x31F59B366C02A3FF, 0x61B82949760D16DC, 0xF6B958AF92BD3BDF, 0x7297AAEFEC69C0B2, 0xFE8A50AD9E8684CD }, /*06*/
	{ 0x3D3DD0C829EA9AA0, 0x3E77597EEC96C7A3, 0x8FD038231E7F1684, 0x64617B131FB7EDE0, 0x85C99CE4C5405874, 0xA58735D41F19C5E3 }, /*07*/
	{ 0x8028628ACAF91B9A, 0x194A640538C97064, 0x04A15E018A4F1680, 0xF4BE1B04C2360955, 0xDFB24D4CEF581A20, 0x3C59D0A0FD15879D }, /*08*/
	{ 0x88F8E2ECE78AF1BC, 0xB46B6E22297364C2, 0x9339F17F926B99F3, 0x31293489B3B8F07C, 0x3909CE1649C9BCF1, 0x0C46103BFA31BCE9 }, /*09*/
	{ 0x4D1A0B8CC7EFE2A9, 0xAFD7878CAD55E871, 0xC89CFBC858CF4153, 0xC4739486C74F75D6, 0x0BF7192C130AC9F2, 0x0084F2BC5E81BD9A }, /*0A*/
	{ 0x4AFBE975014FCACF, 0x41DEAF8CFACC41E4, 0x3C1EC23B53ED16E9, 0x78B06EB30F1C3248, 0xDD11165D04285C72, 0x6546D9B5609491E4 }, /*0B*/
	{ 0xFFD1E2E04DC8D260, 0x07B186948A74ECCE, 0xB5120E45121AC531, 0xBD0BC86330810C85, 0xDE93AFDDDB487730, 0x78DEB0DE6EB99196 }, /*0C*/
	{ 0x79BAA2AC3FDEBA55, 0xB1B7E3E1C92A567E, 0xA53F69AB4A5B0794, 0xF0DA7499954F6DDD, 0x58117C89E1132248, 0xD95DF5F794C51A6E }, /*0D*/
	{ 0x89BC8A0C65091C33, 0xB04EEAEE063135C2, 0xF53267E04AB6E689, 0xB4C579B7207E8BF2, 0x3FD31E51343CD2DF, 0x119E523E2F8197FE }, /*0E*/
	{ 0x2E10CB1C6060F32D, 0xBC92E732A94E6B63, 0xF3220D831FD04267, 0x502C5F7414BCE87F, 0x89E0651DE91D2457, 0x759E56B04482D915 }, /*0F*/
	{ 0x43AEE32C6A84E803, 0x0C5007202C0BD7E3, 0xB4F464474205D32A, 0x7D17FC95DE386C06, 0xE8DFBF64567AA545, 0x55BD889D5853046F }, /*10*/
	{ 0x687ABE14EAB8DA27, 0x397B3AB50D72C344, 0x505EAA6D1FDE618D, 0x61BE79865DA13F69, 0x17BBAB29B5E90E2D, 0x010C921972FA8B2F }, /*11*/
	{ 0x8B7223A4F56FF453, 0x291B7B5CB98B6FE1, 0xFD421625786FBF7D, 0xE33D1020D0E8CDC8, 0xCA530C708B739E87, 0x878AF1F304B8A12F }, /*12*/
	{ 0x7ECE1F24E56DD711, 0x2E0869241B2FA6DF, 0x84B834DCC459B2FD, 0xE022EDA6319E7D3B, 0x59839D8CA03C9928, 0x644790F491BBC774 }, /*13*/
	{ 0xCBDC6F49E6B0DD0D, 0x44BA2F8D00346732, 0x86BCC821586AE61C, 0xC7B7491285CEE55B, 0xED3912FFD97F3851, 0xF4AF0186BEBEBCBF }, /*14*/
	{ 0xCA8A48E54ECCE516, 0xBEDF1864B7F8F506, 0xD56A1F60A4B36AA4, 0x49B25AB5FE0DD9D9, 0x21377DBB5E49FCE1, 0x708F64F5D59D99E7 }, /*15*/
	{ 0xE9F873A569AFE02C, 0xDA66BC05CA997390, 0x8C88174756D35385, 0xEAAAF16CF4FDA730, 0xB39F7A55653A3512, 0xF10AB727BC23E852 }, /*16*/
	{ 0x93E96FF8C5BBE2AF, 0xA47785420253E97C, 0x704B25D1F77D074C, 0xC0B2093F1470559C, 0x8D5BFDD4E364AACF, 0x550518673F5B1BF7 }, /*17*/
	{ 0x8DAC832E5BE81ACB, 0x588BFB202B7583D8, 0xA34D8A70DFD1D7E4, 0xD03B0627B687033E, 0xE3D0BE7EDBDC75CF, 0x818EE8C5B09F8BEA }, /*18*/
	{ 0x8E3B02E1489D7D31, 0x5336752B64E3B532, 0xE4D4CC795C580A65, 0x6DAB08F313ED767E, 0x8E567E88FDBA36BF, 0x259490F1D8E933D0 }, /*19*/
	{ 0xEAB437D0D62CAA62, 0xC090AD28B982B003, 0xE255D768D25704EC, 0xA048511AF6256A43, 0xE79F078F4D498B5F, 0xC41735FB75B357FE }, /*1A*/
	{ 0x70DA9FC3504FF29D, 0xB9AB1F388673FF25, 0x36922F4CD17357BA, 0xF09C3AB292E7E04A, 0x90CE0BC3D9BA13EC, 0x647C4CA63C918DE3 }, /*1B*/
	{ 0xD834A38DD1ECD688, 0xA97A0020DE46AB6A, 0x9CDEC7F6E62EA71F, 0x288A5A6FD74DC47E, 0xD44A2E2765CE50F1, 0xBB9B50B5DB9E4F3C }, /*1C*/
	{ 0xC66DA90E41DDF2E6, 0x5A3FE86F160C5C09, 0x6F6AF0405108CCBE, 0xF938382EB627FC7F, 0x163DD634617F006C, 0x5184B1FEDC908497 }, /*1D*/
	{ 0xC95719ED07FCB21C, 0x5112DF043F6EE7EB, 0x50F9FD60012334CE, 0x589FA85104D96579, 0xB7129E44D71905A7, 0x3314766E0733528D }, /*1E*/
	{ 0xDC6C8014C5457CB8, 0xD635FDCD286A69B6, 0xD66F232CE27D01BF, 0x56AF4AC0F682EC0F, 0x57DF1D55B64328F5, 0x651ED4C52A87CACA }, /*1F*/
	{ 0x26D9D1CC79EEC502, 0x69BF340A34B1EBFE, 0xFAA5AAAC8E397174, 0xD0A8F9BD426BCF6F, 0x5B131F464D6D2452, 0x122DD15660D0D6DA }, /*20*/
	{ 0x6E389AEC5F51A22F, 0x7EF68F1C54C127FB, 0x986D4D46E0485C30, 0xF0A47B39E7CF8A31, 0x1D398DFDB7F2A78F, 0x2FC651D1FBB10D2E }, /*21*/
	{ 0xA44E4E8D1B49DCB0, 0x07A4822049C2F343, 0xC40AC04A8D6505BA, 0xD9B91D3F0729B16C, 0xAAF39951B50F9015, 0x966EF5D3AD3F9076 }, /*22*/
	{ 0xEA78CBAC0EB6D009, 0xA0FEA6725A23DEAB, 0xCE729C7444CB94D9, 0x40A994626627AA0D, 0x37F738CDE3D018D5, 0x4C29491C01CDB3C5 }, /*23*/
	{ 0x7C9792AEA745C87A, 0xD1FF5620C5BD8FD4, 0x9ECA84E3004B56B9, 0x5AFD3923C228B1D6, 0xE5DBF79EB3FD283B, 0x441712E354084B9F }, /*24*/
	{ 0xE35D288BD8E249BC, 0x91776C1453A366E5, 0xF5D1E1684E95EFEC, 0x9108E117D7DDF606, 0x81B30F9DA2CE7C8C, 0x6497DBD786818C0D }, /*25*/
	{ 0xC2F891FF45044BE0, 0x75A1A76D2B87E2EB, 0x85CE65798AF3C2BF, 0x213F532B4EFD09DC, 0x0DAA1DF5A53A6C88, 0x3028606A50D826B2 }, /*26*/
	{ 0x609A62616379F33A, 0xA339A3BC53E4516D, 0xD7AD92616A5ADBEC, 0xD043726D86E924AA, 0x8555B564F4C29865, 0x56AA12AB31C1D6B0 }, /*27*/
	{ 0xCED9ED85C1C17BFF, 0xEB522325ACBAFFC2, 0x04D3D8F4B2D15394, 0xD271504C04756EEA, 0x2DDBA4A91AF827F1, 0x1F67D5C28F8002E4 }, /*28*/
	{ 0x8B9C08AD432DC08F, 0x5A7543E29796BBC7, 0x34A6DB5B3C1967DE, 0x016E3BC2A2804EE4, 0x5B9BCACCE5172F75, 0x7549598B80ADBDBA }, /*29*/
	{ 0x6F3FB117C5CDD155, 0x16C3B0A59CD6EEC5, 0xD9A1A411DE538769, 0x938C54979F4AC37C, 0x3737BCC1D55284DB, 0x6CAD9F8AF9156BB7 }, /*2A*/
	{ 0xEBBF284F9C75EBDF, 0xB383EBB406753DE8, 0xAA86127AEE7C403C, 0x10BFDD10523DE027, 0x138BF6C4EB4A8A13, 0xB1EFF67DDB78B067 }, /*2B*/
	{ 0xF6D1138D7AA3DA5E, 0xBAA8098D8FB66371, 0xDAE76D1B8B6CAAF2, 0x400F1034368D1EDC, 0x7C937F5172E8D277, 0x7D05BBF83CADE6EF }, /*2C*/
	{ 0x0E9C2EA6CF34B081, 0x9036B30D58F60BA0, 0xDB3A2C5848F08BCA, 0xC87AD1B94250D564, 0x7C892E09EEF96166, 0x26DB85CF571085F3 }, /*2D*/
	{ 0x251EE3F58718C12A, 0xF9438D81178A2AE4, 0xF0929A889039A8A8, 0xF06B65225EBDCCFD, 0x2E4D14EDF7BF73C6, 0xA9369895BC1DFACF }, /*2E*/
	{ 0xCAE302B41D6979CB, 0xBBFA5A58B51EE623, 0x5113B99DC81AB52F, 0x6093795BEC17A056, 0x8F71FB4D2E5E355E, 0x762F92EDBA34A2F2 }, /*2F*/
	{ 0xD130015265A4D9FF, 0x09BEA253D71F26C8, 0x81B6EAEDC46521E6, 0xFAE268165682B8A9, 0xA89C3EC4774AB623, 0x0D2E45E055219DB2 }, /*30*/
	{ 0x2B560284C3A692AB, 0x37008AD0B379A7B8, 0xAF11CD2C30F90BFC, 0x7FE87A250F2762ED, 0xC3FBD711647242C3, 0x74ED8264F6B322BD }, /*31*/
	{ 0x28195CC8A7AD3943, 0x53CBE808464C4FC4, 0xD58E3D7A765F8726, 0xD83052F60185AA4F, 0xEFCB0D85223BB4E7, 0x5A31305E787FAC28 }, /*32*/
	{ 0x725D0EE230F19543, 0x9091D2C6BDDF34E0, 0xE3BE49C6C2754601, 0x61BE300BA4AD566B, 0x02D79D7551FA7CC1, 0x6543910F5F1CDA58 }, /*33*/
	{ 0x4099ADC44481B43F, 0xFE1361922FD9EB81, 0xA989C09E441FCEAC, 0x449B3A13D3CB9019, 0x45A9BE396F201134, 0xDC1AD05A046633FE }, /*34*/
	{ 0x1A563A6D522F3E69, 0xBE589E079F475A9E, 0x75A2A9638E4C0038, 0xDA3B6202577A0362, 0x211D3F1E0D727AF6, 0x5E1FFC529AD99233 }, /*35*/
	{ 0x47B61E86C6D6D01B, 0x437D6F83ADADC318, 0xD5A361028DED738C, 0xA00D4C630425164B, 0x1A69AFA5AF4C9DD2, 0xF99E1C67F951B582 }, /*36*/
	{ 0xA66A7740B6BDEA79, 0xFEF7FF1496AF80A3, 0x05AFD43EEACD898C, 0xB00C78ED31AD7134, 0x0ED31A1AD7846673, 0x74B96844161499BE }, /*37*/
	{ 0x46FA8D6CCBF6D12E, 0x31C2FC147F303956, 0x707F4401DE5F067F, 0x3AE5FEC7E33594E9, 0x28E39F8A63531714, 0xB7B329EA1E9FCAB2 }, /*38*/
	{ 0xEFD8F755825C7804, 0x1F5A93870BD30CD1, 0xEFBF894671FF8716, 0x28ED617FF22BDA58, 0x411289CCAE5CB62E, 0x95DD42F41801F2F9 }, /*39*/
	{ 0xA8525B8645FC59E1, 0x75E62DC00A5F7F0C, 0x09C56785210416AC, 0x50EF76E9B30D7626, 0x2B3B2CDC19F5D665, 0xA41297CD11D8F4FF }, /*3A*/
	{ 0xEAC99A649EEE5039, 0xA593C92F143C0065, 0xB314735203071206, 0xEA2761A0C764A4EC, 0x02AA7FD46CAC25B3, 0xC68CC182A96D03BF }, /*3B*/
	{ 0xB2873F024EC83CA8, 0x97470AB8FD8853EB, 0x18FE15C159B305BD, 0xB0AB08F687EAEAFD, 0x510A3FDE73602E43, 0x03E1B84DCCF0FCF0 }, /*3C*/
	{ 0xD85BBBDC8033C0D8, 0x9223D9C39CA9F34F, 0x7D3BCB6D5B63C3FD, 0x1C30F974DA0C0FB5, 0x8B24BC9EBEFB5143, 0xC58954925B7B84FC }, /*3D*/
	{ 0x6ABD7C2E0844D7A7, 0xCCF2EA456CDF530D, 0xE8938CF52B3921B8, 0xBA023CA2F281657C, 0xEC635DA675D1EDAE, 0xB4AA52F22EE1BE6C }, /*3E*/
	{ 0x981C3AC677CB5904, 0x6A92B54C84877B49, 0x745BA6BB40C55815, 0xB7AF550D22A371ED, 0xD5E8BD87C65F5374, 0x67874A37F0F538F5 }, /*3F*/
	{ 0xC23BBA2A9DECC021, 0x4E610E930B0E3450, 0x1A681AA91477577E, 0x38A3209714EDC376, 0x0FD15563EEEB4AB6, 0x7D57668A01D42178 }, /*40*/
	{ 0x6AF88CE145A098B5, 0x1AEB858CD88B8B46, 0xE8B733AFB8E2D6E8, 0x313FAA8C10A7EBFA, 0x127D375E77557CEA, 0x96BDA2F70B2F2155 }, /*41*/
	{ 0xEC8903978FAFB636, 0xC7213C425C079763, 0x760384036AB6D17C, 0xE0C63A26385F1F49, 0x299877D6811A6DF5, 0x876F90FC5304B88D }, /*42*/
	{ 0xA6FABBC2D6E0BA16, 0x9B70C9640080E6BC, 0x29B2D5265598B27B, 0x4A9657C726E4397E, 0xA801CCC6766678D5, 0x800EF7CC72619998 }, /*43*/
	{ 0x235931A8CF5490BF, 0xE798F98E0E8F879F, 0xC6EEE29C38F30CA7, 0x929A79F2D53E0024, 0x88F2E12749587A45, 0x0B85B28F38891965 }, /*44*/
	{ 0x165E0303E4A4D827, 0x67994F42D1E8436A, 0xE6CC8BCF6E130D1B, 0x50101711709DDEFC, 0x373BDEC40CD05328, 0x40B274A4AA5109F6 }, /*45*/
	{ 0xA9F88BA008FDF8C8, 0xECC897E3476EE05A, 0xBCE290AB69D57A74, 0xFA44DB1811E3115D, 0x6267AEFD64480C88, 0x2697D04A2D3AECEB }, /*46*/
	{ 0xC0782AF2ABCD3313, 0x02BA1290F2F96273, 0x63C82F1A56ADC2B9, 0x10F8E8C03EFE51C4, 0xE3EB348625CCAFFD, 0x93D607969CB8E7AE }, /*47*/
	{ 0xCC6E179443E58FBC, 0xD21C93C655A7B8EE, 0x2B9834A31F2B8BA4, 0xC83B69516025ECEE, 0x9176EB7B427AAE94, 0x8CB65B9E30B7A76E }, /*48*/
	{ 0xC1A33A0AD6EDD989, 0x18B3C5D95813B5F7, 0xB024BD263B359A8B, 0xC8C17C2216A99B50, 0x71F9A11D58237729, 0x3AA67C7618284290 }, /*49*/
	{ 0x99B7465E09201C7B, 0x9AF89FA01CA4FA81, 0xFC2EC63E761AD123, 0xE2A9A39585B17D14, 0x08394DE529F94E81, 0x479448E69794FAA4 }, /*4A*/
	{ 0x23CA3D1C4CBDCABB, 0xE3265436CE1A37E4, 0x1BBF10F69E8A4CC9, 0x05A66708048F5C4D, 0xE259DCDD9C5BFEFE, 0x439E65FAFD936EFD }, /*4B*/
	{ 0xA24D73B6978F719C, 0x3F53F343CCB0BB8E, 0xBE3C72769EE07C6A, 0xFACB9E539CF558DD, 0x67B91D4E30DE986A, 0x1DB913D11698913A }, /*4C*/
	{ 0x98BD4E140DC3C3C6, 0x142B1592BF3263E8, 0xCDBEAC59ED095B0E, 0x900763F0F625896A, 0xE213550F30324E39, 0x8A13A4417A803195 }, /*4D*/
	{ 0x2ACD98ED8C626073, 0x1CAAA6B4C4CF3238, 0x04DCB41EB677EB5D, 0xF88B5844A8105B68, 0x981D9E951A061A4D, 0xBC9471894C878EDB }, /*4E*/
	{ 0x4959FEAD5D6C2DBD, 0x6ABD59E28C503049, 0x06D2C5494CAF8B34, 0x70E4541304A4293C, 0x520F3416CAF2F503, 0xB23D09D92613DB85 }, /*4F*/
	{ 0x26B5A815C32D1791, 0x2C99E7555BB033C6, 0x09CE9D6A0002514F, 0xD485282B2B8D7997, 0x9C5B792F4A4A14C6, 0x851D9D02DC0BB4E7 }, /*50*/
	{ 0x62FEB6CACFB060EC, 0x9D977D69D5C661EA, 0xBF08EFD806D81556, 0x25F1EEA460EA5718, 0xA25346B51F5A9665, 0xD92F9ADC358CA274 }, /*51*/
	{ 0x27E63DFC63E8FFA6, 0xCDB9CCE2CE99FDA3, 0x979D5B754974830D, 0x3298C8407D6693BE, 0x629D5FADA39B42B7, 0x2654D31271CD84E1 }, /*52*/
	{ 0xAB1FA4DAF66E583C, 0xEEB6B7A236D24766, 0xA90738CDFDF5C6B3, 0x28CBA9E5648E2D4C, 0xFDE5BF6C0CFE0DA3, 0x9D00B863D7D78485 }, /*53*/
	{ 0x75FBBF094EEA16AA, 0x48931F027CD729F2, 0x5D360679009B2E7F, 0xDDFCD148BD3DE21A, 0x4DBFF544B094D0E1, 0x9C0E5C6294352C22 }, /*54*/
	{ 0x283A27FF968853D2, 0xB0960C6CEA0D03F2, 0x172BBA07A473DB38, 0x688C87D296E6F4BB, 0x5CB7E9BC5D68CF0F, 0x57A5D71B0E47BFB4 }, /*55*/
	{ 0xDE0108AAC1E4FF2F, 0xD346CFABEAC62B99, 0xB72E203F98B5F608, 0x81853D8CA54B29BE, 0xA6AED7C89FAA1680, 0xD2093B155C39D7ED }, /*56*/
	{ 0x0BAEAC99D4974B84, 0xC7F258A699C9B4DA, 0x6F622C5E4ACCF5C1, 0x58AB397D9781BEAA, 0xBF811F67E101FFE3, 0xAFBCC2881C3C0EF3 }, /*57*/
	{ 0x26B211FB518D6C3E, 0x64BADAD51A10784A, 0xE6BE4E06A587186C, 0xD471F5C61343CD5C, 0x8389BB0DD6AAED5D, 0xC88112678914A17D }, /*58*/
	{ 0x2B2D0BC3BB88D27D, 0xC5A7D1FAFF517AD2, 0x96F39056A09F82AD, 0xFB38A61A6CED4D4E, 0x9D308E4EA6F9B264, 0x9097CE294AECC6B3 }, /*59*/
	{ 0x8FCA2B950690B1A2, 0x293EFCBF03D422DF, 0x8C9125B3E76353AB, 0x3D402092A1A70173, 0x9BAB974CAB9BF676, 0x5EA8FCC55D8C586E }, /*5A*/
	{ 0x408C92E8C2E1EC8C, 0x4AF4C914B71B4350, 0x5186AEE0CDFB1069, 0x2385EAFAB9657C67, 0xF708E4D3C898CA80, 0x1EC8B9F89884907E }, /*5B*/
	{ 0x46E8958B6A2C1878, 0x2172FD410F78A647, 0x9D8E9DD83A299004, 0x390913C3265AD025, 0xD231F1E23077CBF1, 0xE7EE3E574E80D7F3 }, /*5C*/
	{ 0x5A8567A3D85E40B2, 0x16ECF161133FCF73, 0x52DA5C6FBA3C0DD7, 0x56E57983DEB34BFB, 0x83254FDCB768D153, 0x9A14F95F35C6B82D }, /*5D*/
	{ 0x498A29C6E19D4AE6, 0x2EF4AAF46027BA11, 0xBDBA7DAA84F39505, 0x940B2A04F6DC944D, 0x4E7ED35610FC0D53, 0xBADD94C2907E59E1 }, /*5E*/
	{ 0x14DF0FC43F475F80, 0x17E2AA8D264BF82F, 0x92625BDFE58B934D, 0x8384F415A4ACEA81, 0x8E9C5EAEC5D8642B, 0x4D8EF55F1C826687 }, /*5F*/
	{ 0x4A2335C4F77128D9, 0x544E1476D29ABA94, 0x654EC86321785044, 0xB04AD9B02F80445A, 0xB0E01B6480C8D020, 0x596E325E88A3CBBF }, /*60*/
	{ 0x896955157448D062, 0x0DB08C4C0F236D68, 0x3BA8FC5B3CD1C4A2, 0x04F57C53E144535B, 0xB7D04DCC7BE46840, 0x4BBE993192334646 }, /*61*/
	{ 0x1D7837E6AB02CE27, 0x3EA35BAED4493EA4, 0xD1CAFDB5DF94FABE, 0x98B580BB62170C4F, 0xC3C57A6CA9421C43, 0x68D65FC2C1201634 }, /*62*/
	{ 0xFAEABABC48717536, 0x454251E8F62F7315, 0xB318E8A7FDCDC523, 0x7C2E832013C91344, 0x4D9E5DAFD1699052, 0x12262E8C870537A7 }, /*63*/
	{ 0x8A3E5D0BEF8402A2, 0xA33BC5FAFA019909, 0x63CBE8ACD00762F5, 0xEA26A3F181984178, 0x6EEB78D1BB4AF6BB, 0x7ECF9671300E845F }, /*64*/
	{ 0x0811B67CCCF5D0FC, 0x9F8CAB3F3496BD6B, 0x57CB7D24F1355C2D, 0x58218594165BDE80, 0xFAF3368A653A78F8, 0xC04CD80176267762 }, /*65*/
	{ 0xE6417CE75AAA23B0, 0x34A7BFE3CBA61761, 0x8C13E396F8C9B6ED, 0x5C9066464B09ED63, 0x76CB6A642C5CE283, 0x498E082A3EB449C6 }, /*66*/
	{ 0x6F2ADEA6357B5AA0, 0x54DA382B15557B69, 0x302BD81946237AAE, 0x8F0CBB82111EFEDC, 0x45DD2DADCE20F2D3, 0x8A77A5E9E8A2D1D8 }, /*67*/
	{ 0xE1EC332735862A28, 0x92B68B1A7E9C7C44, 0xF45618DC99E963E3, 0x7CAC984502DD1A73, 0xC8650598CD70840D, 0x9A5DA584A26D4EFD }, /*68*/
	{ 0x16B19B010740C15C, 0xB4544AC01016439A, 0x221F749C9E2F99A5, 0xA63E8A279A65570F, 0xC7231669ADD072AD, 0xC5BC35BA740BC801 }, /*69*/
	{ 0x6C44E75A4F378694, 0xD27ACE108A577647, 0x17C487FAFA7E15D6, 0x6A3654D5C8E29EDF, 0x0CE35EEDCC611FFA, 0xD88A8C03C0095093 }, /*6A*/
	{ 0xCF106948BC4B1F2C, 0x91C0DC9990B99712, 0x193B21E3E109AB32, 0x3340DE0608DD1666, 0x8A5BB677BF602828, 0x402C410B1197B771 }, /*6B*/
	{ 0xEB080FF49CA5543E, 0xB4B9429542D6CA27, 0x5999D45DC1533205, 0xF7EA9E398A1BEF3E, 0xBE8817775476DEC6, 0x17064D7790C84100 }, /*6C*/
	{ 0xF3328E9150A7F8D6, 0x52E3E61B04ACFDF8, 0x51D82010F3CEB015, 0x59D673336676D5D8, 0x4CB3BCEF1D91C342, 0x0C589AB58033BE49 }, /*6D*/
	{ 0x54B8E70EDCE03855, 0x7BB590E99687FD57, 0x6CFF08688D2B1FDD, 0xFD0F6D068BFE994F, 0xEB9BCE302489AE44, 0x66B21F200661E3E4 }, /*6E*/
	{ 0x2F5E0060189669AD, 0x473AF1D03C00CAE4, 0x0278299268D1F3B4, 0x888714BC3A7EC9D2, 0x9FF9C7F071EBD2D9, 0x875A5DC25DFFDB10 }, /*6F*/
	{ 0xE2A97A3E468399D8, 0x3BF7EACA32C80DA1, 0x13DCAC8EB6C2231D, 0x227EC90E1102EE97, 0xB2344832F0381434, 0x8613888303B190EB }, /*70*/
	{ 0x3A3D3B6CE026BFFE, 0x18D4953B9A68ED59, 0x24BB7B574AB777A0, 0xE0CB7DD64983DCB1, 0xCF768C439869AC97, 0x8062BC7A900E6033 }, /*71*/
	{ 0x39D4C3B78A7A33C7, 0x43D72EF22AB0B4EB, 0x54AE8184DDA50394, 0x0C2A7DA083C38536, 0x9DBC6F921D4AD822, 0x2CBB61FE182EAA42 }, /*72*/
	{ 0xD8CE9A806C0BD24D, 0xF69D65A65845727C, 0xC3FF81CC76F2B048, 0x76B1FDC3CA67CE58, 0xCED0970AFBCBE78A, 0x57502941B726F5F3 }, /*73*/
	{ 0xE006AEC17FCEFCF9, 0x05CAA1629E003591, 0xB7050CC99F585312, 0x669260401E159490, 0x8442D25AA757CC5A, 0x228655CD4038770C }, /*74*/
	{ 0x93EE8D67D3F1F3A1, 0xBEA46D48DBF8D7F4, 0x3C91F02B8646453C, 0x6C3D7C1F04188A58, 0xEFA97287F89CEF84, 0xCB40364E108BFF4B }, /*75*/
	{ 0xC6DCE3730D4FF825, 0x02AF54F87D972790, 0x7D69D20F6F4F788F, 0x90C255C64C166E8F, 0xA3529FBF4BF9C9A2, 0x3ECEC41136694F6B }, /*76*/
	{ 0x3DE10A5EC6CA7B3F, 0x7E196081D085ACAA, 0xDF5F0DE3705D60F7, 0x393E7C83DCC57075, 0xA5F33BC2DCB98F97, 0x0AEB7F050D1204C0 }, /*77*/
	{ 0x6F3B3B3D11A8BC05, 0xB52269AB2B95B8DC, 0x12EDE24EB1385F13, 0x202BBA6B5836B5E1, 0xEE3636C5925ACC49, 0x42224CF6EEB509BF }, /*78*/
	{ 0x5F0CC3BBC4BE9A92, 0x584313FCCC54DD2E, 0xC11FE90F00394036, 0x3371667C72FC9723, 0x9611990B62AC8D9F, 0x4CFCB9EB3C317FAD }, /*79*/
	{ 0xCA8E520A894A3FBA, 0xBD9ED1B80098CC40, 0xBDF24507DFF3757C, 0x47AEC572E68D35EC, 0xF3D4523D27B373E4, 0x1AB11E16973A05AB }, /*7A*/
	{ 0xFFC293A6C26B817D, 0x2C9E9D134959D828, 0x7FA5216408199BBF, 0xA6F002DE0DCCD861, 0xBE8F9DC57F2CF35D, 0x1352E2DF86A47647 }, /*7B*/
	{ 0x84B55BE101708E74, 0x3ADEC53721209F3E, 0xB18F9A1E68DFADBD, 0x09A050819774CF2D, 0xE4AB295D380A8762, 0xA3605B0C689C239F }, /*7C*/
	{ 0xDDC7031FBFDFFE8F, 0x0B175DE65B832F0A, 0x31162ABC65719685, 0x51215E534BBC36B1, 0x9F2F7D3B5D01AE44, 0xCF43A2426E83B61B }, /*7D*/
	{ 0x7E32DB672B16F04A, 0xCE6F45DE0E6AB788, 0x25718548B8E70B41, 0xD7368BCF39A0FAC4, 0x956863EC49880C47, 0x720E335796341674 }, /*7E*/
	{ 0x06707A8E33D9D6C6, 0xB684BFE26CD576C6, 0x44F47E5ECD5FC46C, 0xAF1B23A856D844B7, 0x98A627916AC5657E, 0x040C3964A1127E19 }, /*7F*/
	{ 0xA5DAEC3134C0A39B, 0x0CA04160BD5ADB1F, 0xB50EC5A9F29E1ACB, 0xBE2FA1126AF7BFAF, 0xBEFC0AC4C9C5A4B3, 0x994739C71FB1EB29 }, /*80*/
	{ 0x6FEC2D343E83A763, 0x5BDBA5715757F50C, 0xD6F6282EE46A11B3, 0xA8B501F5922A5524, 0xA782A21006B605CA, 0xA10BD2E896975C81 }, /*81*/
	{ 0xB8AAE0532226D0ED, 0x891831C0470E84B7, 0x74C824D648E8FF28, 0xB5E4E02EAD3906EB, 0x5ABB086ADA60A713, 0xA80C57666A9E29F1 }, /*82*/
	{ 0x529E3E52B1E7230A, 0x0C148861C9F08E26, 0x0CFC8A131BAD803D, 0x8C09F324902FAA9F, 0x0231EE4987999848, 0x3B0688492E2B5457 }, /*83*/
	{ 0xEFA6EAC5036814CD, 0x02773C1F8DAA5DF5, 0x0E4EEDBD0702DE31, 0xBA7FD757D0D740EF, 0xA8805F0C74005F8B, 0x1448467BFF3E1EF8 }, /*84*/
	{ 0x2A07B766016AC70D, 0x64215C35364219E9, 0xCD6F7EFE35FCF6F1, 0xF05CC06084C29267, 0xAB3BF2F32579A444, 0xAC75F42D9A25B9C9 }, /*85*/
	{ 0xEF3A14B5EDDB8464, 0x2314E0802D2DD0E9, 0x14DEAEA9F928762A, 0x5763EBB480E15A02, 0x25F7CA14E8CDF5E6, 0x8E594510DC61E6BC }, /*86*/
	{ 0xE62C38DCFD21000B, 0x7BB32AE917EE3DA7, 0xE49F15E24CC9B656, 0x56E28259DCA361D8, 0xB43B8008A9285F48, 0x0DC6B4AF7E4AE61B }, /*87*/
	{ 0x703C64241142DCAE, 0x732D33342C45063A, 0x37877EA1624567CB, 0x2871D534614DD114, 0xE748092A1D94F5D1, 0x4524056F0C6D1CB7 }, /*88*/
	{ 0xE325B1823A595DF9, 0x742D0DD5C96F397C, 0x44361C9540A9F451, 0x02382F9BF6331FB9, 0x8ECBAFBBE91A0467, 0x528EBF3811F904A8 }, /*89*/
	{ 0xFD2BC6534631FB0D, 0x27A5F036FEEB9A6C, 0xD0F876D7911D0775, 0x12EFB3A29C6E0B72, 0xDC4BCA3D5E871DA1, 0x028FB6E6E608F46F }, /*8A*/
	{ 0xEF17ECC8930A7B4A, 0x9D97B34672FB273D, 0xC6AE835F35A25D8F, 0x6C27469530C21F5B, 0x2FBC16A26150E795, 0x02AD93AAE0B5C71A }, /*8B*/
	{ 0x6D24BE43CF07DD56, 0x63681D62A38D2A2F, 0x9872C9B411724AA0, 0xB882B4857C19690A, 0x87B1BA8D2804C6F4, 0xD7B199CC36F40B49 }, /*8C*/
	{ 0xEEFB8D8573FD9E0F, 0x933403199B91560A, 0xFF0DB41665D5248C, 0x322EE105EA984196, 0xDB8CE0F83890D89B, 0x3A32F8983C901F80 }, /*8D*/
	{ 0x082CDAF93F215BAC, 0x67C118A1B9274FAC, 0xAF74501CFB93198A, 0x53525CABA0E812D3, 0xC9AF3A005EFE8A6E, 0xF242DCB60DA7B2FE }, /*8E*/
	{ 0xD3887FBFBB7314DF, 0xDDDCCCF0F720C342, 0xB2C4331C33C8C415, 0x1666010767F4785B, 0x8455B7C1FD5DE487, 0xA821C5EA181875F2 }, /*8F*/
	{ 0x7E289831418562F0, 0x2AD12E3042B185C3, 0x7C20D0D735A6AE96, 0xA68BEF98E22CBD41, 0xA1411D22F8D93243, 0xD813FB404F3D2F38 }, /*90*/
	{ 0xE13FC0A76F664294, 0x7E21C9D9F7FDDDCB, 0x161E68B366D6B1F8, 0x55BF957EB5743874, 0xB23213EF8364D766, 0x529BB98AF96643D4 }, /*91*/
	{ 0x036D7ADDAADB5C33, 0x0525835F802D032E, 0x7DF7D0D8D7A2BEF2, 0x84927644B27696B7, 0x215E21E4D1F9B5B9, 0x77743669C40EB7FD }, /*92*/
	{ 0xA9B3534BE8897784, 0x5BFD4283541A5090, 0x97AFFCCD121C9778, 0xC146C4C9637989C7, 0x0820E72FCBDA59C7, 0x5526E2F4A0AE4F4F }, /*93*/
	{ 0xA4739E20FD72BDC2, 0x6D6EE5A5C1A54CA6, 0x70A97A6FCB884E5C, 0x2B6108339E979C48, 0x93A63730D6BB23A7, 0x5B1DCEAB00045EE5 }, /*94*/
	{ 0x427C14E4F88C8BDB, 0x1D8630868E039BC2, 0x33DB40A251502D1B, 0xE043C9CCB45D2B3D, 0x292B67B6EE077B2D, 0x1C3A2FBDE24C742A }, /*95*/
	{ 0x3DED69F37016D86A, 0x9A947B13AC66D7C3, 0x822D8645DF4CB39C, 0x2BA20F98F19E10DA, 0x6703138D422AC4C4, 0x8D34D6138FA04A1D }, /*96*/
	{ 0x28E59C8B257D112C, 0x8747068CC5499FCF, 0xD6C16EB780F9191A, 0xB416151633F7AF08, 0xA230E00D6BA1A1C3, 0xFD066FB9965B83D2 }, /*97*/
	{ 0x70F4BC1B7F8FFC37, 0x38DC0331E56B0FDC, 0xA9AB7290AD2B0BBD, 0xB307973C3D0783C6, 0xBDC455F6CDCA111F, 0x23F0E08317B8F0DC }, /*98*/
	{ 0x0AEEC24E9285C50F, 0x3BCDA47833B61ACE, 0x839986F959EE0723, 0xC959034A8D7F5EB9, 0xD4AD7E05B05C4FB5, 0x6C37A3D39F7A0EC4 }, /*99*/
	{ 0x0227B7230FBF2D07, 0x28D7D2AD632BED47, 0x07BD8F8B5012EFD0, 0x48A0D43AE0403442, 0x9B8939207F1449A1, 0x351EAD01B9FDF219 }, /*9A*/
	{ 0xA7119D2E311CEF25, 0x1E532CD0C4ED0479, 0x2272F878D8D30A0B, 0x769C412CED9C4C42, 0x262FFBFA65CBDDF5, 0xDB73D86721EA368E }, /*9B*/
	{ 0x4BDBE90B3FBADCB2, 0x1324EC3A8D6FEA57, 0x6D9EFBE530850D00, 0x401A88AFF8A4C8F4, 0x655CB76B8A2E271C, 0x35505B6DBDE16F43 }, /*9C*/
	{ 0x6E15E57E23F57037, 0x4962737362C1FA26, 0xC962372D1829B80B, 0xA1FE6832EA4D6211, 0x6726E307F96E7763, 0x04C761081677505B }, /*9D*/
	{ 0x42E2FF3A8A6FC164, 0xFB85B2BC9D28B268, 0xC559CFF024533A28, 0x2EC83F3911DAB3CE, 0xAE0FC74A9D736A27, 0xDB9CDD048BAB4CCF }, /*9E*/
	{ 0xD79C52221D20E765, 0x499EDD73903CE704, 0x9B016D987DF48349, 0xFCFAB44AD12FC5C1, 0x811293F3B800FDF9, 0x511DC619CA53CEBE }, /*9F*/
	{ 0xA059EE78B826EDDF, 0x4673AF294D17C85A, 0x5E527D4E4DF282B5, 0xDB5B9A2693F95CE3, 0x6551D304FB54F296, 0xAB3EB70D65912FCC }, /*A0*/
	{ 0x7D0C4F67B6C78135, 0x390CAEA7DE304D37, 0x49E19FABC8D494FE, 0x1A9E1B6437A04516, 0x886CC4BDAB6AF35A, 0x0529217344F502FE }, /*A1*/
	{ 0x3CEDF34141B52CEE, 0x8133BA924753573F, 0xCB32BE22BC66025A, 0x0C480183DE403CB3, 0xBF5B84B427DFCF31, 0x7251428DB0232156 }, /*A2*/
	{ 0x86FCE831C58E25CB, 0x5CC43FFE45CBFC75, 0x33877CC042F199BE, 0x1212FA7F0CC22E1C, 0x448EAB4B7D1F9823, 0xA7B1363A9FA7599E }, /*A3*/
	{ 0x2D8C2FEDA0E5106D, 0x192E366838BBEB3F, 0x36226AA60ACEA0AF, 0xE7E1285DC1F3926A, 0x900371FA1883D9EC, 0xBAC33B1AF360EB66 }, /*A4*/
	{ 0xD4A2A11612BDE0E3, 0x82AB0DA614CB4CB8, 0x189A4D50AC01F4C6, 0xE36A5DA1D9F6A647, 0xE43120D6B16B11B6, 0x7D395F4236E75378 }, /*A5*/
	{ 0xC0C155CD47F3877F, 0x4B03BFE5C334CA71, 0x77710F1F4B844FF7, 0x3443BBAB720E8DC5, 0xF03F8868C5863406, 0x0FD60511C872EB50 }, /*A6*/
	{ 0x8C253DAAB5286306, 0x9AA438F54A6196AC, 0x181D08C723A22C5E, 0x633C49C88E3910A1, 0xC9F54A67992675B0, 0x1FDD98ACBD38D976 }, /*A7*/
	{ 0xA10893DA7575A9F7, 0x8F5F4A025AB2A018, 0xD80538F0336BFFC0, 0x0F9751D33889626F, 0x30383EB925BF911A, 0xE6149F68CE19CC60 }, /*A8*/
	{ 0xB9081DBAC6BE0598, 0x785DD9BC69C71492, 0x8B035A0CA56E172B, 0x8946783500724888, 0xAF1E57C958650569, 0xE1DE4E944FF22261 }, /*A9*/
	{ 0xEA5EDC4D2718C0D2, 0xCB1C5D4DA15A8AE4, 0xC6272382F8163015, 0x94A934E5057B54CE, 0x658E481A3D68D10D, 0xE8F24929E50A46A0 }, /*AA*/
	{ 0x7DF146281AF482CD, 0x014B68E726407B06, 0x6CE564938C70DDBC, 0x36DAD2DE72A5DAA2, 0x6D573BF69C0B2980, 0x684DAB14B4AA0329 }, /*AB*/
	{ 0x9C69DC064E738B5F, 0x83CC16BD5A1C36F5, 0xA99B365E6E141B12, 0x2748FA5AD0FACCE8, 0x26D073A047D99C49, 0xB005B182505B0C0C }, /*AC*/
	{ 0x15B6A2A20ED0FD1C, 0x9333AF729BD65A25, 0x22CC333293BD2C1B, 0xD724D949B15E8BE1, 0x69D0DB0512B97117, 0x85ACA8980DD7653C }, /*AD*/
	{ 0x230EC629D77BB3F2, 0x43115B991D297CB2, 0xA2F955792C53C76F, 0x48A76728EBE25BA7, 0x7CE662A405384400, 0xDDC06B7E6BF49D66 }, /*AE*/
	{ 0x20DDB9BD7644410B, 0x056391B1FA2E8C06, 0xCA4EDE51CF167C00, 0x46602B550536F870, 0x5040672597C21FF4, 0x0AF8EC6E8AFB844B }, /*AF*/
	{ 0x0023C5749251B883, 0x335A4F86D66B7E00, 0xAE353DED3EFACE8F, 0x3FC80526D67B35DE, 0x0D9078FBDA80BC53, 0x467900DFF3FE4C14 }, /*B0*/
	{ 0x0F9CB2BE6A448113, 0xE38D541B6A9A5829, 0x673953DAF354FC0E, 0x3C818A277F8569E9, 0x8D16EA77DB122A3B, 0xE40A860318B6EA84 }, /*B1*/
	{ 0x78CE11F42D7D5E50, 0x84F76DFF199C998D, 0x999B578E3AE935CB, 0xD9FD092C1BE63212, 0x31F33C63ACD316D8, 0x5AA08030B8D65C0C }, /*B2*/
	{ 0x0098DBE19CA84FE9, 0xE2426617D1142137, 0x63C3C4166A78E21B, 0x74B145353E03B0E4, 0xF43C0824EAE508C4, 0x58C1E6622528602A }, /*B3*/
	{ 0x9E27EBE6D1426A6F, 0x2A6A600A6B5FA342, 0x8FF7E2306BA90370, 0xDF83D91A683EDDDD, 0x29572442F0225388, 0xE9CC0F1B6437320A }, /*B4*/
	{ 0x054DF380E896064E, 0xFAB81A4AA3AD88A4, 0xF87426486CCA156F, 0xBB1B3C8237472960, 0x7EC0B87CF73F960A, 0x5C57D7E6470F7808 }, /*B5*/
	{ 0x5758E103AC614A1A, 0x766AEE86F81358DF, 0x203FBA51DC74396A, 0x78C93DF969C5721F, 0xE69E32E230196597, 0xE287C6CECD8AB95B }, /*B6*/
	{ 0x2A06A7C10C0DCC97, 0x99D5298268A6745F, 0xF2D818BB774858B3, 0xD52A820D4F64D886, 0x2F808EF87A263981, 0xBB91206E6347C676 }, /*B7*/
	{ 0x0847C6D71CE0C746, 0x86FD451B447C1E11, 0xC20623B0E2856FCC, 0x3ADDFA2D0398181E, 0x6736A0A06B336B46, 0xD1C70AEEB2B1257D }, /*B8*/
	{ 0x5633260D141A9776, 0xD530805F596CA3DB, 0x8CE33EF69437CE46, 0xF62D54E97E747088, 0xDF5C9318489B45EA, 0xA4AAD29F0BA850CA }, /*B9*/
	{ 0xBDBD7B16767F6D9F, 0xF7968427F1B7B6DD, 0x58C76599B35276EE, 0x286F4C7F6CADD791, 0x8188C0401742117B, 0xCEC4F1964266D163 }, /*BA*/
	{ 0x97E4E8A6B5135B24, 0x8A8BD785E5297977, 0x4545C1A0975BC5BB, 0x13FAE3BD9F59E37D, 0xAFD5627C0E91DE2B, 0xA223AC778474E1A9 }, /*BB*/
	{ 0xDE1BF1EAF86C6B3B, 0xA246A3ACD50035FE, 0x6F80179DD96A21CD, 0x3F8DB7CB17300D03, 0x497A798B5D94506C, 0xAD52DCC6F61AE841 }, /*BC*/
	{ 0xF4A4E1D08E1F440B, 0x5E27633CD56422E0, 0x1465C14F1DB41420, 0x9A939043988D37C2, 0xCBE65CFA245DB368, 0x6340AEDE28DDA855 }, /*BD*/
	{ 0x1F7AB65A3F892454, 0xD70AB4167EBEB5A1, 0x9B2631E824C2028D, 0xD5D97BDEE31519BC, 0xEA2DC77449E4058C, 0xEB204F2D6D2FBAFF }, /*BE*/
	{ 0x6537E69171A2665D, 0x3FD2F835435A3F23, 0xADD5DD3E622D6C8A, 0xC522CDD5E5E243F8, 0x5AEC27F3DBFDA8A2, 0x477A65ED570E1445 }, /*BF*/
	{ 0x3BA7CB01D32E9D63, 0x9E335734E7B5416B, 0x0ED96A84F94539F6, 0x45CEE2E46DF5A70D, 0xDE142EE1E9AFEC1C, 0x78D6121C4FDC72DD }, /*C0*/
	{ 0x7BB30AF653390B77, 0x2D394F2B7F8F7BB6, 0x0277A3C213AF3489, 0x7DF6E674DD56D084, 0x5643CD3073C42451, 0xFAB15F8BD1A1DC18 }, /*C1*/
	{ 0x42B453ABF5150D8B, 0x913F109C1188E18C, 0xC27BB7631FB43BF9, 0xEBDDE685EF108419, 0x76D67C87C56D33EA, 0x95EC73C0AF40F084 }, /*C2*/
	{ 0xBCE43D59A1F50BFB, 0xBA7027CA04D84600, 0xFB6FDB98A2BE644B, 0xD5DE777E993DED4A, 0xFCA39F1EDF710F3A, 0xA5E5893C858D8841 }, /*C3*/
	{ 0xC68AC776E6AEACFC, 0x538067C7866106EB, 0xD27B4A352F4EFDE3, 0x847DA2B3BF01E378, 0x3C79E3C136926D58, 0xF957BC8726AA1610 }, /*C4*/
	{ 0x95492C4203C7C612, 0x0DD60DB1EE8321FC, 0xE1D9EBA902F62B42, 0xEA2DBF7D0E37A4F2, 0xE11FB9098BF5DA48, 0xDBFE213F818EA338 }, /*C5*/
	{ 0x17CB21316D4756DD, 0xB88952498140146A, 0x648112F580844288, 0x4947ADC3F7D58F35, 0x651CCE28E26A5377, 0x0B3803DAF337F89B }, /*C6*/
	{ 0xBEAB16E2DCE6B6E3, 0x8F39ECC8E39172DF, 0x607CC9553FF29C0E, 0x4BFD15154F4F0BA7, 0xEE6230B6BD408CE4, 0x35B654110D164E99 }, /*C7*/
	{ 0xADDDFF1BD2C11CD4, 0x2A1A262CBA6E1AA0, 0x0BF2291D09475A46, 0x4C93A0ABADF4DE32, 0x73EE8E1327333E63, 0xF3AE2031F5D13B28 }, /*C8*/
	{ 0x246C7CABB2D9A55C, 0x50E9C7282C1EE0F6, 0x2FBDA09565A0D3D7, 0x196552679C04A4EB, 0x137C66DA29A6DD82, 0x08A76B6B4BDA56BF }, /*C9*/
	{ 0x7CA3C59BE3E28610, 0x6ADD75CF1F7AE248, 0x01747450737A6435, 0xA1F2259CB2B4923B, 0xE0C8F55E8ECE7210, 0xD7964398F350B69B }, /*CA*/
	{ 0xE045864ED1825101, 0xAC54969193E1A1C5, 0x23D85A934D0794C7, 0xB4FA88CB734A4213, 0x7C5CBFD6BDA3D5F9, 0x66607FE938748825 }, /*CB*/
	{ 0xBAF36FD2A180D481, 0xEAC440AC1B9598F7, 0x9AA24D80FFB7B06C, 0x79601F517358F163, 0xD1071831418BB63B, 0x819609A6AE7D3A03 }, /*CC*/
	{ 0x3E9152D8CDBAE551, 0x86AD793F203DD016, 0xBE3AEB778AD4A891, 0x2810254DD76B6618, 0x9B5DCDE36636C327, 0x0A8AAD65868BC58C }, /*CD*/
	{ 0x6D0672780D93152A, 0xEEE705247B828091, 0x9EBDB976F137463F, 0xA7DE3E73A2D0C1BF, 0xF871A00BA0046AC7, 0x484C96A803F23486 }, /*CE*/
	{ 0x0FC7BCDABB06BFFB, 0xF75C3FFB3D6309B3, 0xECA305D103109162, 0x373F503B204FFF61, 0xCE332C9F54963FA2, 0x9A4420A52242CDB4 }, /*CF*/
	{ 0xC71D481179D198C1, 0x505A2845CEE92569, 0xF339BFF6DD6755B5, 0x8BEAD52B8DE89245, 0x4B686E65920DCA2B, 0x99593FA43EE68A37 }, /*D0*/
	{ 0xD90A68D717E61501, 0x9BB920AEA19161A6, 0x2F3D6F96D90EB1E4, 0xDF15ECBA10513D7D, 0xE6E5D539B4F01831, 0xC7D17A7528FECE36 }, /*D1*/
	{ 0xA04FF0BEB4EBFBAF, 0xE5E90A5B3DDAA3CA, 0x8453542209F4A145, 0x80A6FFD72BB5A707, 0x14E0C4705A1ABF6A, 0xD699EC1FC18A677D }, /*D2*/
	{ 0x7021A124E3181575, 0xDC7AAE2817AD945F, 0x8BB5521E7F0D565A, 0x6671D3792F0805EE, 0xD3888EA394413A1A, 0xCE4D7E47B55BF9CC }, /*D3*/
	{ 0x22F440263CAADE68, 0xE77BB287772EAC7B, 0x29493775962A40E9, 0x1E06A27FA68CB91B, 0xDDEF02932ABDB9C7, 0x79F03B88DC175233 }, /*D4*/
	{ 0x65F6D517B53E2391, 0x97DB65A2F00B1C39, 0x1D77AE9B85AA4855, 0x19133B9B3E9B0771, 0x6376D9F11A7DB3D4, 0x949AD02F5AE16184 }, /*D5*/
	{ 0xFE4434CDE09D923B, 0x03B0FCFD713B7052, 0x2D713290D4A67238, 0x2B56946FF629EE96, 0x60A15D01B2B3C428, 0x0B1D5EAF793933A0 }, /*D6*/
	{ 0xBC40FCFB0E0D494B, 0xA31C4648C7B3D1DE, 0xF1113C219A07EC8D, 0x2378BEB1A5C2BD1C, 0x190CC3478070A194, 0x63DAB6E1CCF56329 }, /*D7*/
	{ 0x901B6B9E82BABF91, 0x872A234C45D61001, 0x6CA46A95C1CC6D6C, 0x22779315E0F02295, 0x60A59396346BE6AC, 0xFB67A503CB488846 }, /*D8*/
	{ 0x50D440F74C97660B, 0xE71ECABF64EDFE0C, 0x80201B895718CE22, 0xA05D89804D35D306, 0x8F700402A2B0D086, 0x326FCB334CA4DFC0 }, /*D9*/
	{ 0xBCFBD02EA005CDD5, 0xF0225A4675553115, 0x08E18B3692A7AF62, 0x05D34A820C8CED0A, 0x51A8D7CEC33E80EA, 0x0AC007503FAE879C }, /*DA*/
	{ 0xF43EEFB5C83C521A, 0xE5E9B05FC48841AC, 0x79C52C38BF85B5F9, 0x26CD0818AE3BF7A9, 0x4F385C32CA8F5F74, 0xF17B22107B954752 }, /*DB*/
	{ 0x1A48FC969198A4B0, 0xD9A78940BB0C4E1C, 0x42781D9BE60E7691, 0x87D1CAF3680F8A30, 0xD09FF193606AAF29, 0x4518DABC60048793 }, /*DC*/
	{ 0xF05D48134A56A034, 0x89A65EEB91DC69B9, 0x8FC7F43960E63C62, 0xFA1C6B9FF9415E92, 0x7E219D4E56347935, 0x2B6A48D6DE0AEF85 }, /*DD*/
	{ 0x1A7FF9C54B045FFD, 0x44A0A9562E9468B2, 0xF11425A22D1EBF92, 0x208D33120BD28E0E, 0xF2D74197AF80E162, 0xCEEDCA73DFE66C93 }, /*DE*/
	{ 0xD57190439D29C9A4, 0x44C007DC2B5EAF9D, 0xEF6DDF48A780CEDC, 0x61B205E4A96024B1, 0x1885B6CE84C3FE5D, 0xB8B56986B6E2CE21 }, /*DF*/
	{ 0xF36DACFA34237E99, 0xBE45EB5253BCFED0, 0x402C6946B8B21AC0, 0x2460A6FCE7E9CD67, 0xF89A6D5B162629FC, 0xF66CCEA374DB821E }, /*E0*/
	{ 0x16E06074DCC31A1D, 0xF172017AC3FA38C3, 0xBBC1CE4BB784ED60, 0xDA89A8BCE82AE671, 0xA6DACFFB8D26C0BB, 0x185181AE9609F6D6 }, /*E1*/
	{ 0xF110DBDD94D17661, 0xF59FBB4CBA69F393, 0x463B60FB3F3C5E00, 0x1C60B896FE8E78AC, 0x5EB3E26795DE5AB6, 0x997328D4654D6219 }, /*E2*/
	{ 0x21069118ABE24B61, 0x811CB8C48FCEFC6A, 0x483B032CFB56F902, 0xFB32E848198CC057, 0xA620815462A04F70, 0x900038D1894959E2 }, /*E3*/
	{ 0x5AD509789BFFECD0, 0xDDCD5E8325F69CA0, 0x154D8F1ACD9B8C82, 0xAC7DF75E94CE3CAF, 0x6D6554D1B38754BE, 0xB5DB64AF738486E7 }, /*E4*/
	{ 0x35A308A1AC9A43BF, 0x2647805AB3E6E492, 0x4BB74A616F61588F, 0xFA4602EE5BDBF54E, 0x3FDD62470A7174DB, 0x5795433CA808FAAC }, /*E5*/
	{ 0x51A094B8774CA605, 0x5F07974C74EEF225, 0x022AFEF7AD81A953, 0x0967C44BBA336FD6, 0x8AA327918AECBA3D, 0xF70B8436573C3F0A }, /*E6*/
	{ 0xCF374F83420766C3, 0x71F31901A13EF07C, 0x63AD56C7DEF9DC0F, 0x9E5BB5E859F5A231, 0xD0BF453BB9893E4C, 0xA1E14B66C2719760 }, /*E7*/
	{ 0xB41861CC73FD3E48, 0x461D79A138B04BE1, 0x4010D37D37FBA817, 0x7D9622AA693225A4, 0x2204454B8126799A, 0x33A5D487DCCD6EB6 }, /*E8*/
	{ 0xD291D0317A053320, 0xE27678F1E50D1F76, 0x9A3D663A63159FC7, 0xAD7B4D3F67BAB452, 0x269CC05E2B33CE1C, 0x0FB8261CD734BCC3 }, /*E9*/
	{ 0xF3D0546D3D4A25EE, 0xB42874AD28C9B7F2, 0x73EC788B29962D28, 0x4AE73A48132B8553, 0x756C99D7A0910B66, 0xECA7E2C2712D555C }, /*EA*/
	{ 0x559FA5BF24911FDD, 0xA1DDF5DE3770554B, 0xC7C3FD139366B946, 0x6E7ECC0C881D2BA4, 0x14E76D6A27E54B87, 0x7352D5FBC4FAB878 }, /*EB*/
	{ 0xF19A622BED8DAC0A, 0x35548E5D7EFC5A2E, 0xCAC84974B4F057B2, 0xAB317ED03D0335AE, 0x710FC138F2C51738, 0x9C90CC495A403416 }, /*EC*/
	{ 0x9FA7DEB936F10461, 0xA1529B0B58462F9D, 0x9F109111C8B9EC65, 0x23A3EB28444E33EA, 0x554084CA75118937, 0x599D58A7C946EAC2 }, /*ED*/
	{ 0x6EC3AABB7856AC4E, 0x980E6907C1CBCCAF, 0x1F8557ADC700CBF5, 0x7DCB1CE0AF48D9F4, 0x7FB3DADF8199AB8A, 0xE6B36DB8FADBF312 }, /*EE*/
	{ 0xC00F0D3F7A101660, 0x605B94B12DB6C697, 0x79944F7BA2B65F38, 0x40858ADEDD47E2BC, 0x1E044BDB0E9FB02B, 0x86C79D01A3109539 }, /*EF*/
	{ 0x9731893D5B98482A, 0xFB8DE267F9790326, 0x8780F407143A505D, 0xA41CAEFCCCD3A8E3, 0xA042F0B3D7B7A7FE, 0x3E3151FEBB19A1AC }, /*F0*/
	{ 0xE7EDF679003A6950, 0xBAFC97D4A8C6AB12, 0x13C096B49C79559A, 0xC3052501434B5019, 0x1280FB23E7ADFB09, 0x1959905D31BD2FC0 }, /*F1*/
	{ 0x575C0C46FCFCC65B, 0xFE625E873F34B419, 0x1696FDCC7F51B8A3, 0xC79C56F30E5AE7C0, 0x14E3461CD27FAD15, 0x1B7BCCB9CB472859 }, /*F2*/
	{ 0x3806FE58E5CC8F16, 0xF8244ED76734C1BF, 0x4E04940E0F5DDB56, 0x5BD0AFDDC4158B7B, 0xA4C6BA949911C5C9, 0xFF6E2AC155AE9726 }, /*F3*/
	{ 0x49C7C844B8114144, 0xB450E41BCA35CB00, 0x302450EC67BEF97C, 0xA8662049DB1E0D8B, 0xDA69C022528EB8FA, 0x6ABBF16585C1A2F7 }, /*F4*/
	{ 0x37BB420DF67F044E, 0xDCC0E9F3E2EF07B3, 0x4D10088618777841, 0x0492E5379305DAAE, 0x3DA4791C37E4128F, 0x80688445CBA4EA17 }, /*F5*/
	{ 0x51398A7CE4CF8D9D, 0x49A5FCD891A69CA5, 0x3D72A60EC2392DA5, 0x0E8296B879AB5539, 0x6BCB00AF2EDC0BDE, 0xBEB93848E54B3E90 }, /*F6*/
	{ 0x7AD7C52A18922E19, 0x29292C57C4F5B8F5, 0xF0CF1F98A577C10B, 0x072B9F293BB660CD, 0x09B8604F5575B6FB, 0xDECB396A81B9FCDB }, /*F7*/
	{ 0x254AD7ADB4C220DE, 0x6C62E20F95A0070D, 0xADEB89F339309BD8, 0xA2F685CC178B289F, 0x9343905B5DEE95A5, 0xE0C30F34A2977C86 }, /*F8*/
	{ 0x669CD51AF7CFBFAA, 0xE3E0806F6880271D, 0x6934C259E098BF90, 0x5DFEEAF0FBCA7249, 0x89F74B948B4118B6, 0x53640AEAFB6807C3 }, /*F9*/
	{ 0xDD3BACDCC04BE120, 0x6D4949BD64198E51, 0x31FDB39666598A74, 0xBBBC6DE9C0C15A81, 0xF27F201C61C06279, 0x2738AFE3E84E5CDD }, /*FA*/
	{ 0xCDD71FD35A6411DE, 0x3CC012793E87523F, 0xB0CFF8720FCA36F3, 0x93E85FE07300F012, 0xE894A085263F090B, 0x2DF60A01DAFA90EC }, /*FB*/
	{ 0x9DA50DB1EEB4FADD, 0xE524E49C9974799A, 0xDE09FFF26A24CBB9, 0xAF9D71E9F3ACE7CD, 0xEB62B1A62566EC9D, 0x06D02AB1217D3553 }, /*FC*/
	{ 0xDD31E6391AE03522, 0x93ACD1065B35E915, 0xF4EB56CC03E79218, 0x0717815C850C97F1, 0xBF4F6A8AC0540A6F, 0xFCF8AE5DE9507FF0 }, /*FD*/
	{ 0xAB45B413DC50B207, 0x40B417369551D8D5, 0xCA32286A108E7210, 0x03225E54D8D093AF, 0x4B6CA5591EA576E9, 0x4E12AB774DC4E062 }, /*FE*/
	{ 0xD9F4F850DF6CB96C, 0x8ABAD81B1667335D, 0xCB4079CFE79C72E5, 0xE5542F763E316996, 0x303E4B79B9D397C4, 0xE46933038B945111 }, /*FF*/
	{ 0x75B15CC53B0D2502, 0xDA1BCA6BA0524358, 0x9EDA977556C06B7E, 0x6C57727ECF0A1325, 0xDC613D5A78E5C3F8, 0xCE062D94A3B4945A }  /*ZZ*/
};

/* Table ADD
 * 257x48 byte matrix containing pre-computed 384-bit words with HamD(a,b) >= 182 for each possible pair (a,b) with a != b
 */
static const ui64_t MHASH384_ADD[257U][MHASH384_WORDS] =
{
	{ 0x7D8058C289965A68, 0xA40E3AC93F6EEC96, 0x42962AB56A64DA88, 0xFB6391D25934AA91, 0xD9958F7E87AFCDCC, 0x988314D182C9F6F8 }, /*00*/
	{ 0x1C460224D6627D46, 0x0C8338ED6877A302, 0x099686848A2D1E6B, 0x43B36D215D7BFB49, 0xB38786F894DB7C04, 0x6F4638AAE0B68B90 }, /*01*/
	{ 0xD673B0582AC721CE, 0x07D11CCDBA43618B, 0xD68F9B3F987F9C39, 0x1BC69CA5367DC351, 0x19315FB5231457D5, 0xBC49FF8023C4FF51 }, /*02*/
	{ 0x6179643A91A32DE4, 0x18872E1DEF8D3303, 0x7FBDD8315412DA52, 0xF1A8B5CAB53FA8AB, 0x801DE3F48AC23AC6, 0xAF7AE49C8161AA48 }, /*03*/
	{ 0x0C9B6A8C119BB0BB, 0x0DE55AE85D1E20E1, 0x48B7435DC9CB6B41, 0xA3A6E2E44FEF6841, 0xF91669E868B80060, 0x214BCF908258CACB }, /*04*/
	{ 0x4594391DE2947DB2, 0x3E6571CC6E9FC2BC, 0x64067ABBC8223821, 0xBEB4F1834702E047, 0xB4F106D9A146BE05, 0x9B9DE8FE614B6843 }, /*05*/
	{ 0x09375BD6DC4C374E, 0x8CB31691EB60E267, 0xA915B81DF0365544, 0x9F047AB054483B51, 0xD01FC79B688E9C7E, 0xCCB8A7DE2359118B }, /*06*/
	{ 0xF7270D40BCECDDB0, 0xC4F73BC45FBD7E70, 0xD70FE5EF89507E10, 0xE3A64DADD16A7BC9, 0xCE016154B38CF9D6, 0xA606A6C6DFD85353 }, /*07*/
	{ 0xF8B96EE488E00CA4, 0x068955EEAEF58101, 0x4BAB59CF6A61C1CC, 0x3FE0542F5EE9AA03, 0xF6908622388193FC, 0xB78BA6EEF47774C5 }, /*08*/
	{ 0x798F73B22108ADAE, 0x37D72947BB130490, 0xA43D1E4C39ACFC00, 0xCEB11D35414762B1, 0xE88D1CF560020AA4, 0xE20322F9157491C8 }, /*09*/
	{ 0xDA615FCBB078E1BE, 0x28F309E1EC35AC02, 0x64B75ABB1D84D8AE, 0x3B9ADA205AF05E6E, 0x0D6B8EE88FCBC8EF, 0xAF8AAD011769C943 }, /*0A*/
	{ 0x1251C038180201DC, 0x9A6305C757BFFE37, 0xDB962943CD1C1956, 0x8A94C2B95A569A74, 0xFBB4EA7CA016049D, 0xF303048F4477F00A }, /*0B*/
	{ 0x16372888CB6911E6, 0x30C579D5ACBC6F67, 0xC41B20D00BEA9005, 0x881990C9DA4D0240, 0x0599214F8DABB860, 0xE449620AE3A535C8 }, /*0C*/
	{ 0x116913EAF90B1D84, 0x5E4E196134B1E8E1, 0x4B8F43212575ED59, 0x9C7698B6CD7EC34F, 0x94C12A54396BF23E, 0xE9FF19D802D11909 }, /*0D*/
	{ 0x8C6BA0E85A04768A, 0x8D37B3F0B6550290, 0xC7C2087D733E9C5D, 0xA7B932B8E1E9F298, 0x385122C401E9CCC7, 0xB3EB819FF727425A }, /*0E*/
	{ 0x5F75DBFB1AB09D6C, 0xB837083CAC1D4197, 0xD9AF89142B757260, 0xAC1F71E71DE3FE48, 0x71F71A44098C5790, 0x0293E4022D18395E }, /*0F*/
	{ 0xC8FB61B088003D36, 0x3CFE1604B5077F1C, 0x4D2C22DD6706BBC9, 0x39D056A9BB307D41, 0x2D86532AA15F3A94, 0xA4AF1FA82ACC017A }, /*10*/
	{ 0x74697EA12B79DD5C, 0xA6265A2F060D0937, 0xED8EFE88EEA85D70, 0x9FA37CA8284E09D5, 0x1D1D438E3048C097, 0xA519883362A1DAE2 }, /*11*/
	{ 0x68560840BAAAE48A, 0xD0381C27B707F307, 0x3A8F4E24DB22F809, 0xACF5526D582F1A64, 0xC07CE4E335EC904D, 0xE2FBF875C3DC4970 }, /*12*/
	{ 0x52017982B27469EF, 0xC67A9FC8A2F076A4, 0xF9D7147CE22CE931, 0x2DC153AF477EB32F, 0xE1396B65889A0890, 0x874E5D7AAEA92AC4 }, /*13*/
	{ 0xE617BEE1CC4CBD99, 0xACED2BABBDCA6939, 0x638550E76297B871, 0xAAF03AF65E178F21, 0x5B7D12FD81CAB450, 0x6772ACA9BCE1D19D }, /*14*/
	{ 0x5E71B2B341EFEB7A, 0x1EAC0B05DE6FEE0A, 0x5886F65EDB18C054, 0x8B575466C729B380, 0xCE330220886A7A0A, 0x28DFF2FF1786D02F }, /*15*/
	{ 0x802724C7B2706DEF, 0x33FE52147D2AE900, 0xC8EEAA5EA3B7BA1C, 0xBBD47F205D8F423E, 0xB2D0CE5F3A08AF46, 0xAD535615CC83F30A }, /*16*/
	{ 0xC9B37BD962AB7542, 0x173D188F8CAE9D63, 0xF3667A498076FD60, 0x08E774F0F7E192E4, 0xAF4FF77803A16CB8, 0xAFE70CD9CDF327D6 }, /*17*/
	{ 0xD3A19143201E15AA, 0x1025BCAEB5554A50, 0x59B8767FDCAF8440, 0x92453353C9FB884F, 0xF32742ED3401E9D0, 0x8C382E0E82B5C522 }, /*18*/
	{ 0xCF0B5908B90B7C7A, 0xDA71C53CAE386706, 0x4F91D6EDCA05CB68, 0x8687E6B8DF5CC6F3, 0xA3880F9B1BE4E7C2, 0xB03F60E1B5E4910E }, /*19*/
	{ 0xF4C3B389781069C6, 0xF54F3E15FCC2A228, 0xF97B887F280435C0, 0xA583DFEC4E50A8C0, 0x9675E5B417EDC206, 0x618F6826807DF5AF }, /*1A*/
	{ 0x83466C0D2D22615A, 0x8B4520E19E06F207, 0x4D8B659E2502E034, 0x9D653DA04BE0E9A9, 0x075712F7BA04C6A8, 0x83E2676A06D50698 }, /*1B*/
	{ 0xFC6533CCEFB8F372, 0x00D61064FF94F011, 0x09893AF7FA325361, 0x88C2BD50BB518B5F, 0x9868CB65E9B2209C, 0xF327D227B7128E8E }, /*1C*/
	{ 0x7D5F0333C85EBA92, 0xB2A44BC9CE37F22B, 0xD6B71F1A308F64C9, 0xA9A03A8A9FC98E19, 0xB18BEB659B148F96, 0xE7FF1073C89DD552 }, /*1D*/
	{ 0x53C442C7718B75BC, 0xE8B44AC1E2426A18, 0x5A0FBAC96A720DD7, 0x5AE971ED50F5C272, 0xBBDB6BC682C8F3A4, 0xE13CB728C55C6096 }, /*1E*/
	{ 0x5ABD70EB9EC9F62D, 0xB0EF3DA118B44783, 0xDA82EBAAEC869682, 0x8AA41D684388D533, 0x9235637110AD1E04, 0xBB2F8B40082C0B8C }, /*1F*/
	{ 0x96837B867A03E5E8, 0x8C2669AB805671C7, 0xF32732DA537197AE, 0x95908762F114EB00, 0xFCA7F60DBF082AD7, 0xA37EC2E780D25948 }, /*20*/
	{ 0x7F21F2E004AAD61A, 0xD86636C5909C7B39, 0x513E16EEA170FADF, 0xB725BC2B77C7EB60, 0xB17C8D89AC011FEE, 0xCDDF2930B7EFBF0A }, /*21*/
	{ 0xCA1F338EB96F5BC1, 0x062D44E49A20AE12, 0x156DBB2E7B71BE10, 0xA038F0F94BE68990, 0x23E68370AE685CD5, 0xC63A5AB27069E983 }, /*22*/
	{ 0x8FF4583A6399A1A7, 0xA73A26BD26F67E05, 0x5D91489A52F5D590, 0xB3A277B55047D949, 0xAE2289743521BC08, 0x75DBBB30F5C1901A }, /*23*/
	{ 0x47E13042E97AFE13, 0x2A9F408D94BEA397, 0x6FCF5E491356CED8, 0x75D023044F645350, 0xB82D60DDE48F54C0, 0x7E0B676648313427 }, /*24*/
	{ 0x64858BB42AA007E0, 0x804C6B350C627021, 0x5C1FE2FD591EF9CC, 0x4666721802EE9743, 0x112344F4D05B4DE4, 0xAF2AC0716C423D1A }, /*25*/
	{ 0x5A81C32177CE6FD4, 0x00E819A62C377A64, 0xBDE22A6B206AF003, 0xE7F82E145905C381, 0x23178BA6E9AED6FA, 0xB1B96C84A900BA62 }, /*26*/
	{ 0xA7230393D376BE98, 0x47CF7E2020D6F126, 0x57A2F211BC4C101B, 0x8EEFF7F313A55B71, 0xA15852603D83808F, 0xA40F66BE1468719A }, /*27*/
	{ 0x93AC6C6932A1F9E7, 0x910E19918E175673, 0x58AF127D1ADE8A7D, 0x7CBA0FF053C6AD51, 0xF2DF666AA9D6C55E, 0x7C3FBAFF36EDB3A3 }, /*28*/
	{ 0x49B12AC66082C518, 0xBD27D5DCCEC17332, 0xDC951AC23D021869, 0xFB36C7F64C64078B, 0x62590331579B46BD, 0xE45B1A06CE802308 }, /*29*/
	{ 0xB2C3604AAAB01949, 0xB6A32902676CA87A, 0x8BA1A6D8983AEAC9, 0x3EF5B1EC87C5C9C8, 0x632B9B91591B524C, 0xF7AEAFD5E1ED580A }, /*2A*/
	{ 0xD0382A8380B2F1FB, 0x9FA01FB52D14CF12, 0x2224B9299058B751, 0x1F60EB188D54FB18, 0x56E028FE148AD6A0, 0xA76DCC72B754B578 }, /*2B*/
	{ 0x1B13108B02AA6909, 0x9D3D1E296EDECC6D, 0x25AB0A87FCE068C1, 0x45912228513A5748, 0x16A14185AA8327D6, 0xB42392BB4D45EF89 }, /*2C*/
	{ 0xA1D77344920FF03D, 0x0ED43DA5A7BBF945, 0x44CC6251F9E4F646, 0x59B553F306C48B8A, 0x703D2DB69A7DCF92, 0x278DB0A673B1EB1E }, /*2D*/
	{ 0x8E032861E1FA89F0, 0x513B13B12F22AB11, 0x710368583D2CD42A, 0xB284CE6D253EEE93, 0x794D2B43840137F4, 0x6BA5F09B20F8462F }, /*2E*/
	{ 0x15436111CA65657C, 0x71A054BD169B4953, 0x62049C6C7B5471C3, 0xAB2A2B215A674CE8, 0xD1221A351BAA696C, 0x93BE1BFE0A614603 }, /*2F*/
	{ 0x32C733C5A828F00B, 0x110F399801ACEE05, 0x9F14CC0840853875, 0x6E88F1A7627FAE58, 0xB8060330D90DB2FB, 0xA8B9D6A99646BE60 }, /*30*/
	{ 0xCA15A83D330E9FC0, 0x30B934C33F0CB130, 0x17069ED0BFF4C5F5, 0x3F52FBF392287423, 0xAAB3CDD4D8CDA382, 0x231B0AC292491FE0 }, /*31*/
	{ 0xF02AB9566B53A895, 0x133D20A7CF1CEE15, 0x0F4A39A36A1470E5, 0x533E5E3D6DD5736B, 0xF31D7A2D49D168C6, 0x2DDA4BD3C1CC68C8 }, /*32*/
	{ 0x172D6D06FABCB264, 0x6FA29A330E363099, 0x320D036B9C027050, 0xF751B0B15E9D1625, 0x3EE5273819657ADE, 0xC212652AABC6E78A }, /*33*/
	{ 0xFEE1424BFE3029F1, 0x29B4328B932813C6, 0x40DF970FE818B040, 0x92302AAEF34DAFF6, 0xAAE05776B89BAEAB, 0x40C9889C60440D54 }, /*34*/
	{ 0x5E0528DD38A964B2, 0x3F280772BBA6F7F8, 0xE59BC201AA3C18F6, 0x4B3037833BF58159, 0x5386B6855EA3D81E, 0x072B41D496F8D86F }, /*35*/
	{ 0xD10F321C58883051, 0xF5122AB53CBDACA0, 0x70E5F26CEF85C03D, 0xDF85840F21010126, 0xFB52964C8A9E5098, 0xD57BE9A2C1511532 }, /*36*/
	{ 0xD4F34B50F922A024, 0x6BA9FBE07C185482, 0xB90C7E15FEBF7AF7, 0xA806293C6065CF86, 0xA4158143B26BD592, 0x3EF34DF685EEF5CA }, /*37*/
	{ 0x2BD94AF4722563D0, 0x7CFD11ACF73C3B20, 0xC0B1C431101154B1, 0xF785193842C24110, 0x159CDA6C108563D5, 0x4866B3F88380E5E6 }, /*38*/
	{ 0xC5C77CE05C3FA920, 0x006559F76A049F3E, 0xEA62873BEC378339, 0x2167933D052D9E18, 0xB804358CF8C04D3C, 0xE49A22281F41480C }, /*39*/
	{ 0xDE977AA346C13C95, 0x094636D6A712F836, 0xCE45CF3E5D3B7E21, 0x1C83990378ACE264, 0xDE89C2BADCE171C8, 0x22FF4D935CA9850A }, /*3A*/
	{ 0x43413AA1C370E983, 0x8006CCE02D7428C7, 0xE63F2069FE1313D3, 0xF8B67CBF92C9D3E3, 0xCABE8ADD2235777E, 0x048050F68A678178 }, /*3B*/
	{ 0xA1F1021BA5846050, 0xAC7D55A11A3DE4D1, 0xEB46205610DE2660, 0xA78B17253DF78FE7, 0xAC8A42CED4C45B4E, 0xC619F830D2CF4D8C }, /*3C*/
	{ 0xD1D45E4330F5FFD8, 0x12F368B42E99E598, 0x95334607C1DED3C9, 0xD6C77ACA7AF16231, 0xC8A72710A6431032, 0xC5FA41BF6285269C }, /*3D*/
	{ 0x6FE8C2B7B32AFCD1, 0x33266CF1BC266852, 0xB7DEE777F460F16D, 0xAD0697A9D35E2EDE, 0xD4172EBBA3137515, 0x772ABF88FD0054CC }, /*3E*/
	{ 0x55396011142CBCC3, 0x075DE9A98D29F336, 0xAC8B2EEBE8770D49, 0x1B0D817B62517938, 0x4C4BAB763BCB33CE, 0x0B137A31AC17581B }, /*3F*/
	{ 0x94F12CDE96BA6BD2, 0xFC2CB9FFB139E6A6, 0x6D2731DCBFE85D63, 0x71E42CB34ABF4AD6, 0x99BF7270DB61BE4C, 0xF4BA0BA31D5CB61E }, /*40*/
	{ 0x3B65FB61284C73B4, 0x5F89D8808763AFAC, 0x7A0F8167D9C69BB0, 0xBD27B760A461AAAC, 0xB1BF104F7E03B6CC, 0x316A189375080992 }, /*41*/
	{ 0xDE26A14218B11558, 0x0F6E78EBAE16DD20, 0x003DC052B7062183, 0x97AE533E9E5CCAB3, 0xBC2FC2D5ABEE5B09, 0x7C219189F8AC3AE3 }, /*42*/
	{ 0x48347AF0DAF519BB, 0x7520A9A5A488E209, 0xCA7532CF3848F90E, 0x85AEA3F53AFBF19D, 0x8881E60DFC45E690, 0x3A613D6453655BAA }, /*43*/
	{ 0x571F98C7B9E7196C, 0x25432582F55597A5, 0xA0D7245B73425C59, 0x5F162BAC31FF98E0, 0x92D608788155ABA8, 0xB53F6A3399A2CDDF }, /*44*/
	{ 0x526EA8A7A86E5A58, 0x5FFB5C59E44CC4E0, 0x6DCD32D24479FB01, 0x78A22F7B2249CA31, 0x7A436F0410E41C4D, 0x0B8AA84C46C2A54A }, /*45*/
	{ 0xE72BEBF02D8188F3, 0x187E39797296CAFD, 0x7F450B19BB428C69, 0x7B8773B2D8871BBA, 0xA9F38F2B9429101F, 0x0B6A62D2CB268DB5 }, /*46*/
	{ 0xACE3B97A6374FF68, 0x2D71557589348A28, 0x1AF9423B6A208BD5, 0x4D844C4352D67AD6, 0xFB0130E2D32E100A, 0x827C16B0E0E15F58 }, /*47*/
	{ 0x78F0B8653B0D18FD, 0x1D7D0B7A15D5E306, 0xB61B9A89DFA6220F, 0xA5B22E9507D0CE70, 0xEFCE8136B8094C5D, 0x8D56DD3E8320571E }, /*48*/
	{ 0x40B298D243D432E6, 0xD6B750A19212B772, 0x6F9D873AC1A0B5FD, 0xE8AC406409A6872F, 0x9A469696B9C1FA72, 0x93848EA38124B42B }, /*49*/
	{ 0xE4C1448EC8CC58B8, 0x76CB35652609202F, 0x9494B86A062621F7, 0x60156A21E1D1A861, 0xA0EF7E66BCA14C10, 0x686BD6D21D825969 }, /*4A*/
	{ 0x29BECA487442E3F1, 0x705663E33AA12922, 0x3CDD921E4C7679A0, 0x9371E7BB0FF9D9FC, 0x3209288C29283298, 0xBC97E7FF1A8EFDF9 }, /*4B*/
	{ 0x49C0E9D5EF0DC498, 0x20AE20504A3B8535, 0xABB1D036D70CD941, 0x149D24EC9D61E114, 0xBEF1E86CBA595DE8, 0xAF0D28A0AF5DFBB9 }, /*4C*/
	{ 0x9A3771E70EBF65C5, 0x9DD4C2D127A8039C, 0xFF1A7E1393C364EB, 0xEB856EFC41DB2ACD, 0x3DBAB2E4AC7A4C88, 0xBA8C2CE090C611A9 }, /*4D*/
	{ 0xD583F464FC44A171, 0x1DBF5431C1A2CAAF, 0x57B3775B16E8D883, 0xAC61D57DB8CA91C5, 0xDACF0AD601CACD66, 0xCE5B41E29D0DE6D2 }, /*4E*/
	{ 0x994FC88062273D20, 0x198F252AE17F369B, 0x707653C6AEF6EF26, 0xFA8CEE9559C0A3B8, 0x983848F20A8BF25D, 0x868FDC206A0854A0 }, /*4F*/
	{ 0x63531AC9C4359CFE, 0x5C22759DCD60D0B6, 0x954D6B25FDC9C3BB, 0x32993504E3D5530C, 0xB39BC4E581285E4A, 0xA113930CBCC476FB }, /*50*/
	{ 0x4AFFC564E9745EA1, 0xCD50F2932DBEE0F7, 0x1032984598329F15, 0x9941126E11F6D9DB, 0xADFD922183C06E74, 0xA19B6D2544808965 }, /*51*/
	{ 0x9EB31084EF95FEF1, 0x729B4DE7CD5BFA76, 0x9E4F5B940206737E, 0x67058AE2316A80F2, 0x51B21B8530E99B98, 0xAED28138B275A1D3 }, /*52*/
	{ 0x7723C4C7401A2B37, 0x54E058559B3AF793, 0xF0A9EC517E50E5A5, 0x429DA5B648A2BB41, 0x288039FC335DF40D, 0xAFF3C95FF1270B84 }, /*53*/
	{ 0x4D66B0327425E5BD, 0x150B198A06ECB238, 0x2DCDAF4C35CE47C5, 0x42A1AAFA9B86CA17, 0xD5A416CA8BA5B7A4, 0xAC4BE5C6A3CFC804 }, /*54*/
	{ 0x955E6ED869A96EFD, 0xB85B562A0D3C2356, 0xCA5D2256C80980DB, 0xD9D6BF37E9EA2739, 0xFD6589B4A1632DAE, 0xAEE8A86655397AD3 }, /*55*/
	{ 0xDD138357ABC5D9A4, 0x58253FBCE2AF893B, 0xEFB96A2CBB3A4995, 0xAB059E2A6B24EB2B, 0x7FCF6DDC799F2E0A, 0x7490FF607212E7FA }, /*56*/
	{ 0xD47753D03F6A7CB2, 0xEEBD589BB4B81439, 0x9E7F083C1E48D465, 0x99F4AC9B24B4D403, 0xCA1E18B7341A0C6A, 0xD927D48AFA8D48AA }, /*57*/
	{ 0xCAC4E3E4097460F2, 0x59A31F4BFC89D1A1, 0x7E9027F8FAE125F1, 0x643EB68156F413D8, 0x6C64525F84C83A82, 0xAD97A09548999292 }, /*58*/
	{ 0xB98BB614D1B97B3D, 0x686559913959D004, 0x7E1D4EA292F75119, 0x987907A95DCCAAD2, 0x0367434C0605031A, 0x0B4E5DC73D450C6B }, /*59*/
	{ 0x9DC90461E4C01281, 0x5D624A0BBAC1060E, 0x40003A12F9D17B5C, 0xBDC72495EB56F20A, 0x52164EC4FFADA430, 0xAF1660B1EBAD29C6 }, /*5A*/
	{ 0xCF2631F3A00975D1, 0xC457D947C598615A, 0x939363C9FC07C9B8, 0x31712E2149678C18, 0xBE33C87A4CB7DB2B, 0x5351E4BF294CEB48 }, /*5B*/
	{ 0x884B290301CF3A77, 0x687C20EBC40B8084, 0x9DA5834D701450FA, 0x9904B5DF529ED2C5, 0x36B6B441548AD976, 0xEC0FAC65F72202D8 }, /*5C*/
	{ 0x49636EC515319667, 0x7E3723B6C9F2BC0A, 0xDC9830A275CEAD6B, 0xA0C03C2B5545CFC1, 0x79206E9438CEF1EC, 0x323A48D2418BCB7D }, /*5D*/
	{ 0xD5943E9A792516E7, 0x5E64D8C5140F9BE1, 0x1C907C1C7CA4DA05, 0xFF82AE6927DBDFC9, 0x3A78FA37EF8D734E, 0xE6EE0491BE7460F5 }, /*5E*/
	{ 0x58CA1A4E84332E63, 0x3AFD9E300CA530A5, 0xD3C19661390E5D17, 0x0BA539299074A7EA, 0x3CD74C3AED20A418, 0x8B50CA03F2ED504B }, /*5F*/
	{ 0xA003D052EDA9F9FE, 0xB14048FBCBB623A0, 0xC5822CE44E3AEF19, 0x8D165342C6F54D95, 0x2AB46EC27C1DFF9F, 0x8E568CAEDD482CAB }, /*60*/
	{ 0xC7AF417AA59466A6, 0x14F56A3CE906F87D, 0x22DC95D0C20DC86D, 0xB632CFF822E2CB24, 0x1FB562502E297678, 0x173B1C675BC652C6 }, /*61*/
	{ 0x401A4084B0F53485, 0xB9C5000746244D2E, 0x73353CE7D8550065, 0x3D9A4C72536FD28F, 0xBD415DC7743D1A9C, 0xEBEAC1963ABCD6BE }, /*62*/
	{ 0x3ED003D74ADC2CF3, 0x15C520F60A291D95, 0x5A95344407639018, 0x2996C63BE5D4D908, 0x916FC19210E6B286, 0x94128D59A765C785 }, /*63*/
	{ 0x0D33A6423A2CA0D2, 0x748CEF7CF523C571, 0x1159E992CA34169A, 0x89E169B453F6D2A1, 0xF873438EECD366AE, 0xA25BBCC3ABEB2434 }, /*64*/
	{ 0x4261894340814F6B, 0x3C4B4A81711F50BE, 0xD0C860AF93B9420D, 0x20B4FCBB2E7402BC, 0xA613BB36D880E8B5, 0x76BB31E2BA788D99 }, /*65*/
	{ 0x639363FA40E53CD8, 0xE1B175630C11DF2F, 0x2FEBC29D14D2399C, 0xD292B92745502B99, 0x9BB56F770A39E529, 0xBA1A7927B29B8166 }, /*66*/
	{ 0xAC090C8658C0A121, 0x1EA6090D930C19A0, 0x97DBAB25D464956F, 0x6AE63A3269786CCC, 0x89B5430308043F68, 0x171D792CC6BBB11B }, /*67*/
	{ 0x664C7BC982732FDB, 0x5AC75069AB3F2028, 0x82346B945264BD14, 0xEF014E9E4025F1E5, 0x6F0645079DD7665C, 0xBC5B2706AC1FCD0F }, /*68*/
	{ 0xE6F121844D18A3F8, 0xB6E1178B863226CD, 0x9D4DD2A7C6F7E803, 0xFBAFCB690BE7725E, 0x5DD68673CDCD670E, 0xB536445E7BF0BB78 }, /*69*/
	{ 0x3170DB48F8E52419, 0x90EF7AE181633410, 0x66DB3F789D314D42, 0x6DD8FCD41B03D852, 0xA5B1263FD8901502, 0x04B3481AF58203EB }, /*6A*/
	{ 0xD2EC1A5547213C98, 0x14A13BDD017B1EE9, 0xAFADA55AFB32F2E8, 0x89706761C174587F, 0x2D784C000F84C7DA, 0xE8EA7078982FB288 }, /*6B*/
	{ 0x994B00C75D9F1956, 0x03774AA164A047A5, 0x1A9CC89FB9A0BB72, 0xE3F88A73F2A58104, 0xE52B66BE4310D70C, 0x27CE615E8FAE5DE0 }, /*6C*/
	{ 0xC3D4A291F8B3B3CD, 0x2AF2CF29DDFA40B8, 0x2F9AAA1F4B6AB0B2, 0x2FB5843AD45EE053, 0xB14DB852FA816858, 0x879998071BE483F0 }, /*6D*/
	{ 0xC83E206608AD2D54, 0xDEA2BBB3781232B6, 0x3ED671260B448844, 0xB8A73D001273CD55, 0x7A8989CDC3398AF3, 0x248E565FAD436886 }, /*6E*/
	{ 0xFB03EB2FC1E478EB, 0x3DD79817D0C15EB1, 0x8C07461CCE15D1D7, 0x9BE7C49891B7CA3B, 0x25AD2437E9E5D4CD, 0x661E9916095B7D6C }, /*6F*/
	{ 0x57B02982A1A4BD74, 0x9CEB65D956C9A021, 0xC21A8F31EC09B134, 0x11DD7E723DD70182, 0xBA0EA7C02D3FA5DC, 0x32AEF95B5F199554 }, /*70*/
	{ 0xFD6D3A6F7D756654, 0x7F402C41EFD32C53, 0xDBE12C3D8AF63D69, 0x92E4672B03AE3500, 0xBEBF0BE79A87BCC1, 0xE0D0C8451B769362 }, /*71*/
	{ 0xFA407554424447C4, 0x38276EB9DC78FFCA, 0xDC030B40E392A111, 0x8E7329B733BF7986, 0xB8CD20BAFEB081DC, 0x7F2FE440A604C24F }, /*72*/
	{ 0xC4676A5698AC5BE1, 0xD96C0D04CE7E00E3, 0x8B5AF70A44A35D2D, 0x1AF81796FE376360, 0x19222827B6E94686, 0xD0833BBAA728C2E0 }, /*73*/
	{ 0x205DB55620E47932, 0x3DBCF9480473B896, 0x5396A9B3E95BF1F4, 0x84A18E3FC1AB5C51, 0x1B0FAF9FAD895128, 0xE6776AEC62E1BFEF }, /*74*/
	{ 0x597731C0CFD0977D, 0x68E5A99B3307D464, 0xCB1A59CA0874D1E1, 0x7F91E66083BD27B9, 0x1004BCF22A307507, 0xC473FA32C0EBE146 }, /*75*/
	{ 0x8F2D73F152382CA0, 0x36DB3631484FCBA2, 0xAB8E7DAC58534B85, 0xC9DC83F64BDCACF0, 0xE18A86D5A404EC7A, 0xF74C8002CBCCAE7E }, /*76*/
	{ 0xABB3E7C985B14838, 0x1E526991B02E100C, 0xB60333765F302EF1, 0x82027211096A8028, 0xDB5FB63C32DFB78A, 0xEDC90E36AA68D244 }, /*77*/
	{ 0xDBF8824CE804FC56, 0x833E3CCDDF0509A9, 0xC612931D08B2EB27, 0xE8C1E576400E5739, 0x4046E16C0E41B9A9, 0xD52A44E631E44B2F }, /*78*/
	{ 0x7BF702A8863F41AA, 0x441C1315EA08D9B7, 0x175B695FC295287B, 0xFFC4A7021DC57CE7, 0xAE0F2E105C0D35A5, 0x8466C9C22121D0C3 }, /*79*/
	{ 0xA6C82B7A5762339C, 0xC3FD8BDBE6A6D1F3, 0x98E37E76CF051B41, 0xFCA1443254D39FA0, 0xD1820D3C2089FB12, 0xA6297940114D0E2B }, /*7A*/
	{ 0x91A8E88F5B269B84, 0x5CE5BD13B839F915, 0x71D3BADF590793F9, 0x61A5799A1F2F635C, 0xCB6E8EE70E0CFCBE, 0x7A518532DCCC48E1 }, /*7B*/
	{ 0xED6159C438D8ACCE, 0xBD2D47B352FFF864, 0x00E3FE4DEBD8B0F3, 0xC8488D49055FAE39, 0x082BCA7E34657650, 0x2CEF272EFB4F23C1 }, /*7C*/
	{ 0x02A9BAEC8AAA2950, 0x04C3089CCF6A42DA, 0x56BB1005ABAE17B1, 0xDF89B6EB4B37DF67, 0xC645E8D7B0290BAB, 0xBAFE2D335B9B504F }, /*7D*/
	{ 0x9864724EC1AC9C32, 0xD037FE97EE4E6504, 0xB0CE6A578619DCF1, 0x2FA089F59ED451A9, 0x11A7D846BA8F897D, 0xB0DC4BBC68269D1D }, /*7E*/
	{ 0x47B36255297B7BEC, 0x3DB226CD2B457EA1, 0xBBDA45CD1B953901, 0x24E5C510DB98C91A, 0xFA5E4E879EF31B7E, 0xA2212D3B738379D9 }, /*7F*/
	{ 0x3AFB0F15D9C24D59, 0x45E33E5A843C691E, 0x5958CFBFBF6FFD65, 0xABFA135003DC8135, 0x3ABE384DF795707E, 0x8E9BB4F2215782DF }, /*80*/
	{ 0x745DCEE8F8004B47, 0x93BC66870AD2CDA4, 0xF2A073FE17265071, 0x512DEBC3E3F74A5B, 0xFD1C4F58BA4F426A, 0x2202BC54E90C844B }, /*81*/
	{ 0x484CE432EF1C5151, 0x18EAD594A4505900, 0xA48F57DD88981D6E, 0x459E102A4516B809, 0xF97EC9A01788A21B, 0xBCA356C25A537D9E }, /*82*/
	{ 0x3EFD68177C157D7F, 0x425F45151A7C1BA3, 0x162E8946EA69AC73, 0xA0B53DF003311A8B, 0x5C61F6F3D0BFA1F2, 0xED923DA5D3C1ACEB }, /*83*/
	{ 0x7C50E2DEECFE9DC6, 0x17F8099650DA08D7, 0xC87BCAAA940703D4, 0xAB80498FCBEC19ED, 0x78950EDF9FF203D9, 0xADBBC8AF1F48039E }, /*84*/
	{ 0x08496491E4F2B575, 0x066D1ED2A0ACA263, 0xBA61B5BDC394DC8F, 0xE4336312E97F1E50, 0x3479BB15D217CA47, 0x9012EB13F695CDF7 }, /*85*/
	{ 0xDCBC43CB441BFA6B, 0xD9C1108252E9B959, 0x660D0087622D1089, 0x26E35E54C71A3192, 0xAE95DEEC1A505722, 0xB61F5182CF903E42 }, /*86*/
	{ 0xAA03F75764204EAD, 0x769095C51E867443, 0xEDF66399EA6D0B73, 0xC73B6CDE5431CC14, 0x5CCCD864790A3340, 0x165EE303CF64E397 }, /*87*/
	{ 0xE5A3F256C577C042, 0x5B41300E3827A9B4, 0xE80F8A8EFC5CAD04, 0xFF7866B9B15E9429, 0xC1BCEA40F26841E2, 0x5BE4FF4ABC25A3DA }, /*88*/
	{ 0xC932FB419D304C04, 0x627556497691F88A, 0x149D21E8615C5A62, 0x6A5307B2E54FF029, 0x1CBE87CBD3F40139, 0x867C8331111850A0 }, /*89*/
	{ 0x9470924D82A490A0, 0x6F75446E246C7283, 0xBC4911F93A9CF101, 0xD15E5CC08522A4BC, 0x91196DD95C4A30AF, 0xA0BB0D4F88E78FB7 }, /*8A*/
	{ 0xC9DA7CE16D82D128, 0x07CA96B97BFD7D15, 0xD63FA64A4B19C926, 0xD1281837870B10C7, 0xB1E36B87EE2FE287, 0x634F2C0DA84AB3BA }, /*8B*/
	{ 0xD872ACF3F9CBF460, 0x14757FDD55762BBD, 0x2BA56AF060A4B36C, 0xD293899C12817B8E, 0xE4D275E538C060B3, 0xD84D47DF3C4DB097 }, /*8C*/
	{ 0x5D4C3DDD50A2EC7D, 0x44B4A410EFB0AAFB, 0x3E0649FFD5BD29CA, 0xA1DACCE1DDAD8B20, 0x09BE5954F50DCE81, 0x082D39AFC260384C }, /*8D*/
	{ 0x637AA6D9A7A27B94, 0x09BAB935847415EC, 0x71883D0EE0D78920, 0x4095CB4D44B5A0BB, 0xC4E1832FD93E1DE3, 0x2D1E07C9D27DD252 }, /*8E*/
	{ 0x788FFE15E2CB720C, 0xA2F18380E98A68F7, 0x888ED12A39E1B45B, 0x2147EB5AAFD2BBD3, 0x8A2366C7A15A3999, 0x95DB871AA96EE06E }, /*8F*/
	{ 0x5FB195C85F2F2470, 0xB142421DCB5CE256, 0x2D3E6238594FF595, 0x6DE0791FA43742BF, 0x447F564C389D3217, 0x421D5FCB76E747C8 }, /*90*/
	{ 0x5A9BDC549C36B14E, 0xC84CE5E5028EBD48, 0x884D50681D73904D, 0xA7A22C924C94B45E, 0x9582CA4780A39426, 0x2E04FADFDA58CA5E }, /*91*/
	{ 0xF0EB5499E595AE44, 0xAF98790CBE0F0B11, 0x25535BECA50938B7, 0x91975A7F08BDBB48, 0xD6B24859B68E14DC, 0x827E3214B9BAAEAA }, /*92*/
	{ 0xF3CBA8401657547E, 0x2D888C5B6C28CD49, 0x7FD2F22A39AE6C79, 0xD143CCB45B066C33, 0x98A5B0E7BD952A16, 0xD6B1535EFE28311D }, /*93*/
	{ 0x82AF42B659F6429B, 0x2D2C6E768FD2315D, 0x1D3C0B6D4D75C6B1, 0x6ABD0E260A0BBEDD, 0x86FB2558990A819A, 0xBF9D7759CFC849C4 }, /*94*/
	{ 0xF916C0983358AC5C, 0x492790E5C9B7C3DC, 0xD4CAB0EF1DA545D4, 0xB2F9F3BC2535E812, 0xC8535D33B2AFA578, 0xD46CAC719EC39F01 }, /*95*/
	{ 0x9AA4FE6990B7BF05, 0x97D652F0900FF0E8, 0xE5D1D944DE2E692A, 0x0A38AA2FB3A6379B, 0xDB391D65146D58B6, 0x75B6726983066D12 }, /*96*/
	{ 0xF7E784F55B126999, 0xF5B7E58D24A3DA95, 0xB6981236C66D0F52, 0x0D44B10C88E736A1, 0x060A25EDFAB45A76, 0x3CCAE89FF2D88BE7 }, /*97*/
	{ 0xCBC4B3EE83E356FF, 0x3783369DE63EB7D4, 0x6A310DDEAEC3396F, 0x814D399E294EC44C, 0x4B417B42926E855F, 0xA61BFBFF8FCC59F2 }, /*98*/
	{ 0x0E0A7D90E6844DC9, 0xC870E60D5DFB6783, 0x63E90F52BD262DED, 0xD2A20DE1A685D4D8, 0xC307AC6628425EB3, 0xA4B2D411F6207D65 }, /*99*/
	{ 0xA795A18DC574F004, 0xC4209A366244C78A, 0x8EDCE1C5273AF12A, 0xDD84593E08F74270, 0xB6AB7BFB29530513, 0x6BD7849206552A47 }, /*9A*/
	{ 0xE49FA24FE6FD4577, 0xC637DFF8968F5D9E, 0x13677E14C51A130C, 0xD17DAB9D96FA8853, 0x0C0608E1185D42DE, 0x33FB328304CA93CC }, /*9B*/
	{ 0x8DA0961B29675077, 0x7E47E2C260B350B6, 0xEB3FF461D163E926, 0xD6F748675A8CFF3A, 0x1887D161919F65E6, 0x81633132F35CD84F }, /*9C*/
	{ 0xAA9988566F29F8C4, 0x8CC55C13E5CE1E80, 0x603A6F04724D3822, 0x7470FF2B8E81C26C, 0x8831AC0C9BC2FCBE, 0xB065FB970E730227 }, /*9D*/
	{ 0xD27A7737E059C046, 0x27E5B10C9BDCE6A6, 0x735F82565D36C77F, 0x8E6B4A4F545C0B0D, 0x0BD2ACF5F8A1A584, 0xE4B7A499DAFB4A33 }, /*9E*/
	{ 0x2013C2654E67A3A2, 0x3A778A90F6943C18, 0x03FEA3FAA0282189, 0x1808A150645E1DD3, 0xED9DF15005A9D48A, 0xA05A65F7BA562A25 }, /*9F*/
	{ 0x16C78B90F574B969, 0xAAA54454B18375B0, 0xB32F6B432283644D, 0x72E506CB0F3ADDAD, 0x3F25CFACFC30C0ED, 0x594F225D77641B52 }, /*A0*/
	{ 0x3D8A338E34E0A73C, 0xFB2991B5E7A95529, 0x93BC4F7EC0AA813E, 0xFC7273C5CB0DD2D8, 0x6FD8AAFEB3E871DC, 0x15884FB7E22B8D96 }, /*A1*/
	{ 0xEBCD4F36BB1FA30B, 0xDA6512FBC79CFA88, 0x290DB13C9C2F8B8A, 0x4397DE0D71E91181, 0x92C72AC60F0EA0E8, 0x6BD53E93FB21B325 }, /*A2*/
	{ 0x01A7E269EA27FC44, 0xB9EE9A6D27E2CE14, 0x9CE539B0F44DC8EC, 0x93760B66A3912048, 0x8940A4DE651422C7, 0x479325E2376ABC43 }, /*A3*/
	{ 0xB978D1AF3EE870AD, 0x1A102039BCC2BFB4, 0xFE26E467F82FB5A7, 0x7AA2AAE78A371518, 0xA6C049980EFE030C, 0xCE5B165A31B9CF55 }, /*A4*/
	{ 0xE079A70EFA5E7BF3, 0x591215F801CC7032, 0xC6EB485231416A2E, 0x79015550313DF7B4, 0x6DBEE87A70C69171, 0xB3A9223B8FAD9830 }, /*A5*/
	{ 0x1E65C5246BF32C0A, 0x746F71887112AC88, 0x3768EE0D1E4C803A, 0x13285A9217D9472A, 0x50D3653EC5B92FDA, 0xD3D08125121C1B16 }, /*A6*/
	{ 0xB26658B8A7D7C6E0, 0x98BABAB8F6A75826, 0x0C5CDC5B7C571E2F, 0xA0757C35E27C3C15, 0x5A4CA5EC989F5323, 0x12040917D1353181 }, /*A7*/
	{ 0x283C3F5D1D0F20CE, 0x35EFFEF7A76E80FA, 0xC284C26FB60A607B, 0xF45C41973151D4AB, 0xDF0449DFC681E6A4, 0x9F4E923144BE2940 }, /*A8*/
	{ 0xA37EBA8348C56B09, 0x1BA4A5AF2150A1BB, 0x8538E2ECA1865E02, 0xEC70C5F094E6B54F, 0x28532F0E3E2F1CCC, 0x6AA481AB8C522467 }, /*A9*/
	{ 0xCB6BCBCC64F474EC, 0xB06AC92DA299C98E, 0x6703AAD7D1EF4018, 0xF674C9C07943BFE2, 0xF39AF9B962D69A7B, 0x2C563C88D9FA4631 }, /*AA*/
	{ 0x919FEF03EB6ACF89, 0x6C7E22DA8609DBE3, 0x1FA1E2E6B8DFAD9E, 0xAB95CC3584E76CCE, 0x1D04DAA8C2B3FE15, 0x1D43007A1F9F889A }, /*AB*/
	{ 0xE7818E5789081774, 0xC0704429AF659E0E, 0xA373CAECB9D9966D, 0x09B15BFB604EF493, 0x37416D1C8D30D8EB, 0x14E79C0CC95BB0C1 }, /*AC*/
	{ 0x524F475F142A020B, 0x1C6451BA2E4BAF9A, 0x9B539D994A1AAF6E, 0xF4275FE42F97D113, 0xC427C07D61E709C3, 0x24CCDF222FF915B2 }, /*AD*/
	{ 0x14DE747F8D1669B8, 0x4E0E531B85D62D43, 0x78E2A7D71BA2E18D, 0x1D946B89B1433FBF, 0x00E387ED73F89378, 0x52992A8283A832C6 }, /*AE*/
	{ 0xCBDB69FBCB21009D, 0x0EC6FDEC4E8269EA, 0x79D908AA89FED13F, 0x7548E553A1103EF0, 0xBBE4048A115EC944, 0xE1C60EBF7FBDDFD1 }, /*AF*/
	{ 0x24522AA6EC219FB7, 0x78C952E8A5DEDDDA, 0x2822F9EF97929DF0, 0xA70C9D1D8258B682, 0x8B37AC646D341E93, 0xD67C34FF4BBD5E0E }, /*B0*/
	{ 0xE770CD22FABC425A, 0x7E9C34D3ACC5075C, 0x0AFFA78C7BC76623, 0x4EED9851852C03F2, 0x7203EC6EE60426FE, 0x6517CFB85022DC4E }, /*B1*/
	{ 0x1581F89650F997A1, 0xDB9CE979F407D39E, 0x0AD4800E39347F3F, 0xC0C35B8D9FD1392F, 0xE74D894A6167AE43, 0xD462E2F4E3CA9AC1 }, /*B2*/
	{ 0x05EA545C5B41C787, 0xCECD2DC56104C690, 0x7337C0746CC963CB, 0xFDF08B81F136B8B1, 0x7F7A862159C72990, 0x0AD9111B6CB0ED2E }, /*B3*/
	{ 0x6FE578032A920367, 0x49EFAEE55C832164, 0xB3157997365AEF57, 0x98F84917C903C67B, 0x75CF75DA2AA67BF3, 0xD2CD8AB69E83D8A4 }, /*B4*/
	{ 0x65B99EB5C4E335A0, 0x486A759227255280, 0x4F7A3B558F86EAC0, 0x03DF89CC2686DBF0, 0xA95693164EEAA311, 0xD3149307EF5BDB09 }, /*B5*/
	{ 0x653261B529527041, 0x9A559ECA9A789241, 0x5931FF0E09E8677A, 0xF6D53AFDF41458EC, 0x02B94442E96BF81E, 0x5C4A1B461FA95C7D }, /*B6*/
	{ 0xB6FA3DB3F3096A56, 0xAD1E678369F141A0, 0x81708DA9ED5E5188, 0xE2EE29D337A81DED, 0xD7EC74F0AC9B37C0, 0x1B785A6FA3C76037 }, /*B7*/
	{ 0xAE153D618202ED33, 0xE8E91115185E46DD, 0x05D9BA34364372BC, 0x672EE1629BDBE13D, 0xFC2F5EA8624A61CB, 0xD843198BE3925035 }, /*B8*/
	{ 0xBCE8F7C1FC130582, 0xA571AEBDC5C5B472, 0x7F08C70583911E9C, 0xF4BC16EA95A9D958, 0xA9EE3D6BDDEA2C72, 0x1B651B74994DF683 }, /*B9*/
	{ 0x13A7D9DA16EA461D, 0x7FCF04FC4EDC32B1, 0x66607C39138582F8, 0x8831546EAD73A091, 0xD8A4C4F349D1F5FD, 0x4975C9D5701638DB }, /*BA*/
	{ 0xBF0AA6A41F0EC78F, 0xDC2BF64F490226F3, 0xCDA313AAF2D03538, 0x04B8DA4460595A63, 0xA23F1B729BB763BA, 0xF793736A73F77272 }, /*BB*/
	{ 0x3BA334AEBFDF6E3A, 0x212037EAF6A1E5B2, 0x004B3C1920BF41E0, 0xD8A11B5BA862706A, 0x6579E5027315F8AB, 0x3C1F9CBB6D54E13D }, /*BC*/
	{ 0xBE24DE02BDA2A2D7, 0x4D69A648E027E9DE, 0xFEBC2A5373BD90C5, 0x42C7AC0E55A414DB, 0xD65962B721659EDC, 0x53831B2E73FB0D23 }, /*BD*/
	{ 0x3DE68C3D94080E94, 0x1D9E0BC868E087DC, 0xE7AD1EF14270B35C, 0x9A14CF373E0F67E5, 0x00EF98ABE6B7E600, 0x82353F1F18234EAA }, /*BE*/
	{ 0x6034549E932CF51C, 0x3CE2E36E3EB4D407, 0x9FBA80CA22B31E3F, 0x142EAEAC8CAD93F8, 0x5D3D5CA6E5909CF2, 0x1B2C70B482B625B5 }, /*BF*/
	{ 0xBC212277FE451938, 0xE21637EF3A3EF4DA, 0xC39C70D155CE7AD5, 0x5C3EDC78491F3AF8, 0x1A405C9B847D16DF, 0x0F446B056E291875 }, /*C0*/
	{ 0x69D791C758B8E1DC, 0x49D8A84512B8B0CA, 0x2474F7B05E061DBC, 0xA79C97FFB1493F6D, 0x6FB9E77B764A9772, 0x311593614CB6D8FE }, /*C1*/
	{ 0xF67F8D77D5A45E27, 0x17A3287F58D26572, 0x182792E595E52080, 0x7D15B7052C373A90, 0x536CF2976B910733, 0xF0E483B43ACBD504 }, /*C2*/
	{ 0x9F1EE351D08E27EE, 0x76158F0101251C50, 0xC1269FD22FDE67FA, 0x6C1930A26E6F3F5E, 0xA32AA0AAA396B071, 0x497A599416938BD9 }, /*C3*/
	{ 0x569916F08E7D1767, 0xAB20F046F165C725, 0xBD4C1CAFC3DE32DA, 0xCF19F61CCA274CDF, 0x4DEBBDC90446F779, 0x0D07522E07429B0E }, /*C4*/
	{ 0xB0F295FB72F3921F, 0xC692DFA79FAD80EE, 0x69011629E0238DA6, 0x0046AF22910AEE2C, 0x77DEBB94C94A9DBD, 0x4B401110EE8A03A8 }, /*C5*/
	{ 0xFF6D2FCA08D69F8B, 0x97592AF1712A5ACF, 0x9D540AF103BB0CA6, 0xD55933E572328F10, 0xEC7218CA2FB0F7A7, 0x8D154ECE583152AD }, /*C6*/
	{ 0x6CEFB6A3BBB18EFA, 0xE47A0437AD48BB5D, 0xAC9695D38CEB73B4, 0x4E5468B6CEB06906, 0x3A57007B43F1591F, 0x16EDF10ED21C2651 }, /*C7*/
	{ 0x3012A14CAEE92A6B, 0xC7CE2D0A16397B6E, 0xB2CE1D676CDD36B6, 0xF563F9683F8C3E45, 0x3C112343EE739443, 0xD2FD36380C31457B }, /*C8*/
	{ 0x1082D7EDE79C89E8, 0x57D1C088D807C597, 0x9E5CE9CCC772F2AF, 0xD30AD1CAF61BB8D9, 0x1E40D024A6939CA4, 0x4985DBEB358DF104 }, /*C9*/
	{ 0x616EBF3FC09BC5E4, 0xA949C7DA50B367ED, 0x0C289EF8AF79CDB9, 0x5C23303A5F396469, 0x3316727F556A9DB8, 0xB7CCEC943229C3FC }, /*CA*/
	{ 0xB9B4B8C04D465EAE, 0xED44836D54AA7798, 0xA6F7DC5F0FF5918E, 0x700DD1C0F68E4E23, 0x4BF2FAA0516222AC, 0x3C2AE26002348642 }, /*CB*/
	{ 0x8A7ECE878B79532A, 0x23613836116B497D, 0xFE665E9804C10AC7, 0x45E6B949601D50DB, 0x5B2A9E4887F1AE92, 0x447FF94BA4FF2079 }, /*CC*/
	{ 0xE31CED1B1F588060, 0xE60A18CE70D51BE7, 0x62C08A1E01A2E1F3, 0x008EB3D687EAE77C, 0x1D01A7F254134224, 0x7E007BE76E56F3A0 }, /*CD*/
	{ 0xE31995299E1D8C4B, 0xD1FBE650613BCD97, 0x1D3C3F36A806C06D, 0x49F278C3D9F4A9C6, 0x0EAC518E5FEA0B19, 0x7D37A390C9A06EDA }, /*CE*/
	{ 0xE38F2A3ED827813E, 0xC5CD4C09D275547E, 0x1CF0E8313F2ABED6, 0x9ECFE847ECCF6C5E, 0x60E68184C974253D, 0xED6AB7636CDD66F8 }, /*CF*/
	{ 0xB6C9DC4E954DEC83, 0x337A86D43C1E137B, 0x611A5CECFD4FD4F2, 0x4FABE0C6259F490A, 0x274393117710FEF9, 0xF80A64FFDE29B591 }, /*D0*/
	{ 0xCAB995B447838136, 0xF8BAEE20D66998C9, 0x037F8691364D2B07, 0xF711FE824949F810, 0x0D1B4E2773B4E944, 0xFABA24FEC54F5E29 }, /*D1*/
	{ 0x308C0B2974968FCE, 0x305CE82B64FDE7FA, 0xCBB957FA7AC79C07, 0x938BB1699D7C5FB1, 0x4DFC5FD652B4D859, 0xC76D7FC21D9AB354 }, /*D2*/
	{ 0x213F9D2ADEB6B2B9, 0x0A501ADE808BA7E6, 0xE1F1A9F9AD58121D, 0xE1DA4CE9E3D30307, 0x4F8FF0EEBD51C22E, 0x5EAF5D79C0C902F0 }, /*D3*/
	{ 0x66D86B9486FA29FD, 0xAC59FAC357C6881B, 0x6BC55595A36E36A6, 0x162BF06D315103EC, 0x126CF6AA5E1F26F7, 0x9DF080C49CCA232A }, /*D4*/
	{ 0x26A2DBCA3C8D42FE, 0xA09DCB40FB746B29, 0xCB007706B3E9B9B7, 0xABCE248527C87837, 0x825E3DF5D74B619E, 0xF2D7FC9D9CB58C30 }, /*D5*/
	{ 0x5CA546BA21141BE3, 0x4588716BD734AA98, 0x4AEEE69AAF4D57A5, 0xE01ED8CBD87F4734, 0x243AA59D601A4FB6, 0x3EACC74DBBE7E0F5 }, /*D6*/
	{ 0x123DCC7D18D37E12, 0xA9D7BD9B59849E03, 0x8944A203A613C897, 0x17612271EE2EC452, 0xCFC11C233E5F2219, 0xF0A47DE1B0F8FDF0 }, /*D7*/
	{ 0x4D84263BE85828AC, 0xC45768F30F7393CF, 0x0D4CC4C0F7F80A70, 0x017880DD846965FE, 0xC3BC9F2DBBE68DCC, 0xC236C7E247AA3536 }, /*D8*/
	{ 0x8EB4E4B8B7DA54C3, 0xFA1427A6D1389409, 0x744449AB8B19F0B9, 0x6BFBE32B2D0BDB94, 0x91313EB6E06C4FF5, 0x0170A64FE878578D }, /*D9*/
	{ 0x179E05C17E3694F5, 0x31599DAEA9E72F8B, 0x028704B1352027F3, 0xDAB6F4975B6EF09A, 0x5E3E730ABC6E8FE7, 0x41753647414763D1 }, /*DA*/
	{ 0xDDBA4F2D0434B98A, 0x0FBAC522F3DAEAAC, 0xD315AED4BB31715E, 0x5C1F3D78EF9E5425, 0xB8DD542D8BF637EB, 0x5FA8A2EEE095CE1C }, /*DB*/
	{ 0x87519C3B117EDF1A, 0xF7C469924B306E46, 0xD2C859D56EE93FD6, 0x7E43975DB42844FB, 0xB6ECE5FF0483394E, 0x02D62595E91E3398 }, /*DC*/
	{ 0x326812111FD4C7BC, 0x099544A2C2D81A7D, 0xB429BB12ACEECF49, 0xF916080CAAD15FC8, 0xE814BE190473E0C0, 0x8245A66C5FA6F410 }, /*DD*/
	{ 0x7C28F51D33181D06, 0xC2E1BFACD6F225FD, 0x6D7477373FB058CA, 0x273120A4ACBA0FB4, 0xE740E22190BC3F2F, 0xEBF7F50765758830 }, /*DE*/
	{ 0xA9D4BB665C151EB1, 0xC6A1EA32577F1B57, 0xF12A0091BBD1C87A, 0x52EDEB266D0D06A5, 0x6682457922D38306, 0x9E208003AC81FFE1 }, /*DF*/
	{ 0x88456DFF8FCC14EE, 0xB7DC6813F9D16338, 0x774C779B6719CD40, 0x421EAD66567CB6CE, 0xEF04E66F4FB17361, 0x708C07F995B8EE67 }, /*E0*/
	{ 0xFD516C065215660A, 0xBC97FC5E0F7C426D, 0x3D72E2DAB55CE389, 0x48BB97924B7289FD, 0x01DBC3476578FDA5, 0x5464B44925146D56 }, /*E1*/
	{ 0x12B45F28FE734036, 0x2BAB852D174B4C3C, 0xF0D3F5F0A0D12AFA, 0x3E09CE8FC0F88FB1, 0xBA2BE405CF690227, 0x3100CDADAD5E9BAC }, /*E2*/
	{ 0x43C5DB2C16CBF0F3, 0x4158F77B00AF1683, 0x048F4888D0AE4B4A, 0x36AD98F815351E5F, 0xC7D6309CAF1F7A02, 0x4EFC524D45B4F6C4 }, /*E3*/
	{ 0x254F60E0A929D2C3, 0xAEAAE6B4C271D87F, 0xD0DC0C713AA202AA, 0xD75004DAB803027E, 0x822FFE90F2BDF421, 0x9998FF9D9599BE72 }, /*E4*/
	{ 0xF4F26F3D3DAA4018, 0x63A3CC4733CFEB16, 0x596209EB60AE0BDD, 0x43BCB7F9F0D863E7, 0x6505D0951F747F58, 0x9A34D71F7A102D47 }, /*E5*/
	{ 0x6A06660A549ABA78, 0x263EF1081353B190, 0x096735234BA57B0D, 0x1D4E120293CE8CC0, 0x4E4A3CD82A707C44, 0x699027EE00372934 }, /*E6*/
	{ 0x0E0D83FCE670DBDA, 0x9F04D71D37DF79DE, 0x9443D35035446607, 0x101FB87DCE888457, 0x627BF79224057C18, 0xDC4000FD953FCCCB }, /*E7*/
	{ 0xFF6901D6C9FFE0A5, 0x2CDAADDF05DF6DEF, 0x98DAF476D3EBBBDE, 0x92D37135CE42C5B2, 0x45187D72CB51905E, 0x9B34C328AC445E42 }, /*E8*/
	{ 0x593B4F221871A532, 0x5B58EEC73E4A07D9, 0x14A3579B51593BA6, 0x2D7C1D6AB98636FB, 0x9B51F3C6582D4171, 0x482BE23EC7FBBF93 }, /*E9*/
	{ 0xCF88AA1D3E0FC932, 0x2118F037989B0264, 0xB37DAC83C4AD9E9C, 0xACDE94CC47AA2BAA, 0x118839C26BCD83CD, 0x32B5843A3193C9DB }, /*EA*/
	{ 0x273694EBF59C4C26, 0x43E86EDEE50D48C2, 0x67F97093ED643CDB, 0xA4AF5201D787AFCE, 0x88E89F3243AC55FA, 0x48CA4CCA84975E2F }, /*EB*/
	{ 0x07F23D3790D99634, 0x5352170C09973E9E, 0x4FF7EBEFD208181F, 0x5D097ED48E177DB8, 0xC83233D54225EDD5, 0xBDB2981A360C76CD }, /*EC*/
	{ 0xBF7BB735C21F63AD, 0xD42AECB81B9AC971, 0x4184209556619D2B, 0x56FB088DB24AB737, 0xEF459017CC4ED7CD, 0x68894E750102D4F3 }, /*ED*/
	{ 0xEC3009E11646864C, 0xABB76EBE218996B3, 0xA44366E48B433684, 0x0250CD9B5F8F205B, 0x37F4022177866A6A, 0x21DC8EB446BD99DF }, /*EE*/
	{ 0x0EA6152DA50313DB, 0x4B18FCBD7B7F2AA2, 0xA520BD2EA4D34091, 0x9293C20836323441, 0xF72FAA3F44FC8B1F, 0x1AE7A9220C877A87 }, /*EF*/
	{ 0xA2F2E7EC8BA8BE87, 0x99784F38EAC42DF6, 0xD018F5FB1F54A5DE, 0x8EDB8A128F89A8EA, 0xE9F4FD01005F0A25, 0xF6865090F443A384 }, /*F0*/
	{ 0x648DF1028E44B6C8, 0x239AAE12588EA54E, 0xC73245098AEC57B2, 0x39B69938F8D4B60E, 0xE98CCDA118CC7F99, 0x6564349C25C899AD }, /*F1*/
	{ 0xEEBA071910AE8CD7, 0xAE13D4E444718CEB, 0x5A38CCEA27116EC0, 0x6626F950B9449503, 0x4D20BE0C08D92A2D, 0xC099D6E99A849DDB }, /*F2*/
	{ 0xB96DF109572B1DEB, 0xCD4EFA1EEF503627, 0x2092C32B138766AE, 0x5F738409FAFC1CCB, 0x48CEEF9F8712DBD0, 0x4A20E8766AF8C679 }, /*F3*/
	{ 0x13FA0D8677476531, 0xF6560311FA218AFF, 0xDD829DF52780CE19, 0xA82A13DF3D99E4F5, 0x5570CF897B7844C8, 0xB4E0BA76CD6CCCCA }, /*F4*/
	{ 0x5E6358BD596DCA88, 0xC480A84997C11DC4, 0x44BDFDA1C4162793, 0xEA575920359081B6, 0xCF404DBDDE42A8B1, 0x6DB72A06F75FE7FC }, /*F5*/
	{ 0x2FFE4156076B8F69, 0xBE08BD673985DB1B, 0x2EFBB12338712477, 0xC3DAC6CA86CBC223, 0x00F0141DC65C1921, 0x694B556816B81608 }, /*F6*/
	{ 0x590CEF2457F28B3C, 0x09B6C56115EC28B3, 0x3D38681EF0B9A6B0, 0x5D6EE687F269E86E, 0x36E01FE904D7F496, 0x7A384C59557A56E7 }, /*F7*/
	{ 0x1C06EC0F2387A7F4, 0xA1C441581DDF954D, 0x61FA1425D4E1CC47, 0xBA7C9CA7801198BD, 0x309C7389F48046B8, 0x6CC511F3581F3A56 }, /*F8*/
	{ 0x50F49296D440C856, 0x972CDAF27319B835, 0x2D355D862AA6B78B, 0xAF1BC5AFDAB3343A, 0x4B421C12E8AE0F14, 0x37592D957F1AC660 }, /*F9*/
	{ 0xA0A45CA6265E0FD3, 0x4623CF6A78E8CB75, 0xD055864C565E57B7, 0x9B1487DC5803BF9E, 0xD9B85F561B55CB24, 0x4139E1014A64E67F }, /*FA*/
	{ 0x2634AABA637740C3, 0xEB490F3ADFEBFECA, 0xB968932FDDF0AB43, 0x4A481F40FCFDE9B2, 0x4E26D3F72A9CA725, 0x59B90F4C85A3389A }, /*FB*/
	{ 0x2D859DB3E1C4F01F, 0xF78DCB6825F90549, 0x5E639D6F79843CBC, 0x063866DA32B0E564, 0x1DC17D7032F46923, 0x64CF9FD8F825A61C }, /*FC*/
	{ 0xD2DE7DBA02DC8648, 0x70860B4EFE423C78, 0x130846413C803BEE, 0x97EFE51BD7370CD7, 0x677250C63776F48B, 0xBBD1BB68B76A158D }, /*FD*/
	{ 0x3A9EE16977A19B3B, 0xF229BE1E11B2120E, 0xB5B4E6E443C31DB1, 0x3DD07F9C6595FDBE, 0xB4498DAB67547C34, 0xAB43F4D1B9166314 }, /*FE*/
	{ 0x08FC179DE2DDE233, 0xBCA1827680C614BA, 0x2C5229280DAF459C, 0x1B522BC4EC0527A7, 0x84125C25BF23DBCD, 0x7850D044113F83FB }, /*FF*/
	{ 0x1D968E3B41E8C8BD, 0x52F0B3C77BEED767, 0xB916233D479F860A, 0xCEEB3FDA20E4BE56, 0xEB8F74A25AF28C6B, 0xBFC13FA4B59E4E49 }  /*ZZ*/
};

/* Table MIX
 * 257x6 matrix containing pre-computed mixing indices with HamD(a,b) >= 2 for each possible pair (a,b) with a != b
 */
static const byte_t MHASH384_MIX[256U][MHASH384_WORDS] =
{
	{ 0x05, 0x03, 0x04, 0x01, 0x02, 0x00 }, /*00*/
	{ 0x02, 0x05, 0x00, 0x04, 0x03, 0x01 }, /*01*/
	{ 0x04, 0x03, 0x01, 0x02, 0x05, 0x00 }, /*02*/
	{ 0x03, 0x05, 0x04, 0x00, 0x02, 0x01 }, /*03*/
	{ 0x02, 0x03, 0x05, 0x01, 0x00, 0x04 }, /*04*/
	{ 0x04, 0x00, 0x03, 0x05, 0x02, 0x01 }, /*05*/
	{ 0x03, 0x02, 0x05, 0x01, 0x00, 0x04 }, /*06*/
	{ 0x04, 0x05, 0x03, 0x02, 0x01, 0x00 }, /*07*/
	{ 0x03, 0x04, 0x01, 0x05, 0x00, 0x02 }, /*08*/
	{ 0x01, 0x02, 0x04, 0x00, 0x05, 0x03 }, /*09*/
	{ 0x05, 0x04, 0x03, 0x01, 0x00, 0x02 }, /*0A*/
	{ 0x04, 0x02, 0x01, 0x05, 0x03, 0x00 }, /*0B*/
	{ 0x05, 0x03, 0x04, 0x01, 0x00, 0x02 }, /*0C*/
	{ 0x04, 0x00, 0x01, 0x05, 0x02, 0x03 }, /*0D*/
	{ 0x02, 0x05, 0x03, 0x04, 0x00, 0x01 }, /*0E*/
	{ 0x05, 0x02, 0x01, 0x00, 0x03, 0x04 }, /*0F*/
	{ 0x04, 0x05, 0x03, 0x01, 0x00, 0x02 }, /*10*/
	{ 0x01, 0x00, 0x04, 0x02, 0x05, 0x03 }, /*11*/
	{ 0x02, 0x05, 0x01, 0x04, 0x03, 0x00 }, /*12*/
	{ 0x04, 0x02, 0x00, 0x01, 0x05, 0x03 }, /*13*/
	{ 0x01, 0x03, 0x05, 0x02, 0x00, 0x04 }, /*14*/
	{ 0x04, 0x00, 0x01, 0x05, 0x03, 0x02 }, /*15*/
	{ 0x01, 0x05, 0x04, 0x02, 0x00, 0x03 }, /*16*/
	{ 0x03, 0x04, 0x05, 0x00, 0x01, 0x02 }, /*17*/
	{ 0x01, 0x02, 0x00, 0x05, 0x03, 0x04 }, /*18*/
	{ 0x04, 0x05, 0x03, 0x02, 0x00, 0x01 }, /*19*/
	{ 0x05, 0x03, 0x00, 0x04, 0x01, 0x02 }, /*1A*/
	{ 0x02, 0x05, 0x04, 0x01, 0x00, 0x03 }, /*1B*/
	{ 0x05, 0x00, 0x03, 0x04, 0x02, 0x01 }, /*1C*/
	{ 0x04, 0x03, 0x00, 0x01, 0x05, 0x02 }, /*1D*/
	{ 0x01, 0x02, 0x04, 0x05, 0x03, 0x00 }, /*1E*/
	{ 0x04, 0x00, 0x03, 0x02, 0x05, 0x01 }, /*1F*/
	{ 0x02, 0x03, 0x00, 0x05, 0x01, 0x04 }, /*20*/
	{ 0x03, 0x04, 0x01, 0x00, 0x05, 0x02 }, /*21*/
	{ 0x05, 0x00, 0x03, 0x01, 0x02, 0x04 }, /*22*/
	{ 0x04, 0x03, 0x00, 0x02, 0x05, 0x01 }, /*23*/
	{ 0x05, 0x02, 0x03, 0x01, 0x00, 0x04 }, /*24*/
	{ 0x02, 0x04, 0x05, 0x00, 0x01, 0x03 }, /*25*/
	{ 0x04, 0x03, 0x00, 0x05, 0x02, 0x01 }, /*26*/
	{ 0x05, 0x02, 0x01, 0x04, 0x00, 0x03 }, /*27*/
	{ 0x03, 0x05, 0x04, 0x01, 0x02, 0x00 }, /*28*/
	{ 0x02, 0x03, 0x05, 0x00, 0x01, 0x04 }, /*29*/
	{ 0x04, 0x05, 0x00, 0x01, 0x02, 0x03 }, /*2A*/
	{ 0x05, 0x03, 0x01, 0x02, 0x00, 0x04 }, /*2B*/
	{ 0x02, 0x05, 0x00, 0x04, 0x01, 0x03 }, /*2C*/
	{ 0x05, 0x02, 0x04, 0x01, 0x03, 0x00 }, /*2D*/
	{ 0x04, 0x05, 0x03, 0x00, 0x01, 0x02 }, /*2E*/
	{ 0x05, 0x00, 0x01, 0x02, 0x03, 0x04 }, /*2F*/
	{ 0x03, 0x05, 0x04, 0x01, 0x00, 0x02 }, /*30*/
	{ 0x01, 0x00, 0x05, 0x02, 0x03, 0x04 }, /*31*/
	{ 0x05, 0x04, 0x03, 0x00, 0x01, 0x02 }, /*32*/
	{ 0x03, 0x00, 0x04, 0x05, 0x02, 0x01 }, /*33*/
	{ 0x05, 0x02, 0x00, 0x04, 0x01, 0x03 }, /*34*/
	{ 0x01, 0x04, 0x05, 0x00, 0x03, 0x02 }, /*35*/
	{ 0x03, 0x00, 0x01, 0x05, 0x02, 0x04 }, /*36*/
	{ 0x01, 0x05, 0x04, 0x00, 0x03, 0x02 }, /*37*/
	{ 0x03, 0x02, 0x01, 0x05, 0x00, 0x04 }, /*38*/
	{ 0x02, 0x00, 0x05, 0x04, 0x01, 0x03 }, /*39*/
	{ 0x04, 0x02, 0x03, 0x01, 0x05, 0x00 }, /*3A*/
	{ 0x03, 0x05, 0x01, 0x04, 0x00, 0x02 }, /*3B*/
	{ 0x05, 0x00, 0x04, 0x02, 0x01, 0x03 }, /*3C*/
	{ 0x04, 0x03, 0x05, 0x00, 0x02, 0x01 }, /*3D*/
	{ 0x05, 0x02, 0x01, 0x04, 0x03, 0x00 }, /*3E*/
	{ 0x01, 0x05, 0x03, 0x02, 0x00, 0x04 }, /*3F*/
	{ 0x05, 0x00, 0x01, 0x04, 0x03, 0x02 }, /*40*/
	{ 0x02, 0x05, 0x03, 0x01, 0x00, 0x04 }, /*41*/
	{ 0x03, 0x04, 0x05, 0x02, 0x01, 0x00 }, /*42*/
	{ 0x01, 0x05, 0x04, 0x00, 0x02, 0x03 }, /*43*/
	{ 0x02, 0x04, 0x01, 0x05, 0x03, 0x00 }, /*44*/
	{ 0x04, 0x00, 0x05, 0x01, 0x02, 0x03 }, /*45*/
	{ 0x03, 0x05, 0x00, 0x02, 0x01, 0x04 }, /*46*/
	{ 0x05, 0x00, 0x04, 0x01, 0x02, 0x03 }, /*47*/
	{ 0x03, 0x02, 0x00, 0x05, 0x01, 0x04 }, /*48*/
	{ 0x02, 0x03, 0x01, 0x04, 0x05, 0x00 }, /*49*/
	{ 0x01, 0x04, 0x03, 0x05, 0x00, 0x02 }, /*4A*/
	{ 0x03, 0x05, 0x00, 0x01, 0x02, 0x04 }, /*4B*/
	{ 0x02, 0x00, 0x03, 0x04, 0x05, 0x01 }, /*4C*/
	{ 0x05, 0x04, 0x01, 0x02, 0x00, 0x03 }, /*4D*/
	{ 0x03, 0x00, 0x05, 0x01, 0x02, 0x04 }, /*4E*/
	{ 0x01, 0x03, 0x04, 0x02, 0x05, 0x00 }, /*4F*/
	{ 0x03, 0x04, 0x05, 0x00, 0x02, 0x01 }, /*50*/
	{ 0x04, 0x02, 0x00, 0x05, 0x01, 0x03 }, /*51*/
	{ 0x01, 0x05, 0x03, 0x04, 0x00, 0x02 }, /*52*/
	{ 0x03, 0x02, 0x00, 0x01, 0x05, 0x04 }, /*53*/
	{ 0x04, 0x03, 0x05, 0x00, 0x01, 0x02 }, /*54*/
	{ 0x05, 0x02, 0x03, 0x04, 0x00, 0x01 }, /*55*/
	{ 0x03, 0x00, 0x05, 0x02, 0x01, 0x04 }, /*56*/
	{ 0x01, 0x03, 0x04, 0x00, 0x05, 0x02 }, /*57*/
	{ 0x02, 0x04, 0x03, 0x05, 0x01, 0x00 }, /*58*/
	{ 0x05, 0x03, 0x00, 0x04, 0x02, 0x01 }, /*59*/
	{ 0x04, 0x02, 0x03, 0x05, 0x01, 0x00 }, /*5A*/
	{ 0x05, 0x04, 0x00, 0x01, 0x03, 0x02 }, /*5B*/
	{ 0x03, 0x05, 0x01, 0x00, 0x02, 0x04 }, /*5C*/
	{ 0x04, 0x03, 0x05, 0x02, 0x00, 0x01 }, /*5D*/
	{ 0x03, 0x00, 0x01, 0x04, 0x05, 0x02 }, /*5E*/
	{ 0x05, 0x03, 0x04, 0x02, 0x01, 0x00 }, /*5F*/
	{ 0x04, 0x05, 0x01, 0x00, 0x02, 0x03 }, /*60*/
	{ 0x01, 0x03, 0x00, 0x02, 0x05, 0x04 }, /*61*/
	{ 0x02, 0x05, 0x04, 0x01, 0x03, 0x00 }, /*62*/
	{ 0x05, 0x04, 0x03, 0x02, 0x00, 0x01 }, /*63*/
	{ 0x01, 0x00, 0x04, 0x05, 0x03, 0x02 }, /*64*/
	{ 0x02, 0x04, 0x05, 0x01, 0x00, 0x03 }, /*65*/
	{ 0x03, 0x02, 0x01, 0x04, 0x05, 0x00 }, /*66*/
	{ 0x02, 0x05, 0x00, 0x01, 0x03, 0x04 }, /*67*/
	{ 0x03, 0x02, 0x05, 0x04, 0x00, 0x01 }, /*68*/
	{ 0x01, 0x05, 0x03, 0x00, 0x02, 0x04 }, /*69*/
	{ 0x02, 0x00, 0x04, 0x01, 0x05, 0x03 }, /*6A*/
	{ 0x01, 0x02, 0x03, 0x05, 0x00, 0x04 }, /*6B*/
	{ 0x04, 0x05, 0x00, 0x02, 0x01, 0x03 }, /*6C*/
	{ 0x02, 0x04, 0x05, 0x01, 0x03, 0x00 }, /*6D*/
	{ 0x04, 0x03, 0x01, 0x05, 0x00, 0x02 }, /*6E*/
	{ 0x05, 0x02, 0x00, 0x04, 0x03, 0x01 }, /*6F*/
	{ 0x02, 0x03, 0x04, 0x05, 0x01, 0x00 }, /*70*/
	{ 0x04, 0x00, 0x03, 0x01, 0x05, 0x02 }, /*71*/
	{ 0x01, 0x03, 0x05, 0x00, 0x02, 0x04 }, /*72*/
	{ 0x02, 0x00, 0x04, 0x05, 0x03, 0x01 }, /*73*/
	{ 0x04, 0x03, 0x05, 0x02, 0x01, 0x00 }, /*74*/
	{ 0x01, 0x04, 0x03, 0x00, 0x05, 0x02 }, /*75*/
	{ 0x05, 0x00, 0x01, 0x04, 0x02, 0x03 }, /*76*/
	{ 0x02, 0x03, 0x04, 0x05, 0x00, 0x01 }, /*77*/
	{ 0x05, 0x04, 0x03, 0x02, 0x01, 0x00 }, /*78*/
	{ 0x04, 0x03, 0x05, 0x01, 0x00, 0x02 }, /*79*/
	{ 0x05, 0x04, 0x00, 0x02, 0x03, 0x01 }, /*7A*/
	{ 0x01, 0x00, 0x04, 0x05, 0x02, 0x03 }, /*7B*/
	{ 0x02, 0x05, 0x03, 0x00, 0x01, 0x04 }, /*7C*/
	{ 0x01, 0x04, 0x00, 0x02, 0x05, 0x03 }, /*7D*/
	{ 0x05, 0x02, 0x03, 0x04, 0x01, 0x00 }, /*7E*/
	{ 0x04, 0x05, 0x00, 0x01, 0x03, 0x02 }, /*7F*/
	{ 0x01, 0x02, 0x03, 0x04, 0x05, 0x00 }, /*80*/
	{ 0x05, 0x00, 0x04, 0x02, 0x03, 0x01 }, /*81*/
	{ 0x01, 0x02, 0x03, 0x00, 0x05, 0x04 }, /*82*/
	{ 0x04, 0x00, 0x05, 0x01, 0x03, 0x02 }, /*83*/
	{ 0x02, 0x04, 0x03, 0x05, 0x00, 0x01 }, /*84*/
	{ 0x05, 0x02, 0x00, 0x01, 0x03, 0x04 }, /*85*/
	{ 0x02, 0x03, 0x05, 0x04, 0x01, 0x00 }, /*86*/
	{ 0x04, 0x05, 0x00, 0x02, 0x03, 0x01 }, /*87*/
	{ 0x02, 0x04, 0x03, 0x01, 0x05, 0x00 }, /*88*/
	{ 0x03, 0x02, 0x05, 0x00, 0x01, 0x04 }, /*89*/
	{ 0x01, 0x05, 0x00, 0x04, 0x03, 0x02 }, /*8A*/
	{ 0x04, 0x03, 0x01, 0x05, 0x02, 0x00 }, /*8B*/
	{ 0x03, 0x05, 0x04, 0x02, 0x00, 0x01 }, /*8C*/
	{ 0x02, 0x04, 0x00, 0x01, 0x05, 0x03 }, /*8D*/
	{ 0x03, 0x05, 0x04, 0x00, 0x01, 0x02 }, /*8E*/
	{ 0x01, 0x04, 0x05, 0x02, 0x00, 0x03 }, /*8F*/
	{ 0x04, 0x05, 0x01, 0x00, 0x03, 0x02 }, /*90*/
	{ 0x02, 0x00, 0x04, 0x05, 0x01, 0x03 }, /*91*/
	{ 0x04, 0x03, 0x05, 0x01, 0x02, 0x00 }, /*92*/
	{ 0x02, 0x00, 0x01, 0x04, 0x05, 0x03 }, /*93*/
	{ 0x03, 0x04, 0x05, 0x02, 0x00, 0x01 }, /*94*/
	{ 0x02, 0x05, 0x03, 0x04, 0x01, 0x00 }, /*95*/
	{ 0x05, 0x03, 0x04, 0x02, 0x00, 0x01 }, /*96*/
	{ 0x02, 0x05, 0x01, 0x00, 0x03, 0x04 }, /*97*/
	{ 0x04, 0x03, 0x00, 0x05, 0x01, 0x02 }, /*98*/
	{ 0x01, 0x00, 0x05, 0x04, 0x02, 0x03 }, /*99*/
	{ 0x03, 0x04, 0x01, 0x02, 0x05, 0x00 }, /*9A*/
	{ 0x05, 0x00, 0x03, 0x04, 0x01, 0x02 }, /*9B*/
	{ 0x03, 0x05, 0x01, 0x02, 0x00, 0x04 }, /*9C*/
	{ 0x04, 0x02, 0x00, 0x05, 0x03, 0x01 }, /*9D*/
	{ 0x03, 0x00, 0x05, 0x04, 0x01, 0x02 }, /*9E*/
	{ 0x04, 0x05, 0x01, 0x02, 0x03, 0x00 }, /*9F*/
	{ 0x02, 0x00, 0x03, 0x05, 0x01, 0x04 }, /*A0*/
	{ 0x01, 0x03, 0x00, 0x04, 0x05, 0x02 }, /*A1*/
	{ 0x02, 0x00, 0x05, 0x01, 0x03, 0x04 }, /*A2*/
	{ 0x04, 0x02, 0x01, 0x00, 0x05, 0x03 }, /*A3*/
	{ 0x05, 0x03, 0x00, 0x02, 0x01, 0x04 }, /*A4*/
	{ 0x02, 0x04, 0x05, 0x00, 0x03, 0x01 }, /*A5*/
	{ 0x01, 0x00, 0x03, 0x04, 0x05, 0x02 }, /*A6*/
	{ 0x05, 0x03, 0x01, 0x00, 0x02, 0x04 }, /*A7*/
	{ 0x01, 0x05, 0x04, 0x02, 0x03, 0x00 }, /*A8*/
	{ 0x02, 0x03, 0x00, 0x01, 0x05, 0x04 }, /*A9*/
	{ 0x04, 0x02, 0x01, 0x05, 0x00, 0x03 }, /*AA*/
	{ 0x03, 0x00, 0x05, 0x04, 0x02, 0x01 }, /*AB*/
	{ 0x05, 0x04, 0x01, 0x00, 0x03, 0x02 }, /*AC*/
	{ 0x02, 0x03, 0x05, 0x04, 0x00, 0x01 }, /*AD*/
	{ 0x03, 0x02, 0x04, 0x05, 0x01, 0x00 }, /*AE*/
	{ 0x02, 0x00, 0x05, 0x04, 0x03, 0x01 }, /*AF*/
	{ 0x05, 0x02, 0x03, 0x00, 0x01, 0x04 }, /*B0*/
	{ 0x03, 0x04, 0x05, 0x01, 0x00, 0x02 }, /*B1*/
	{ 0x02, 0x00, 0x01, 0x05, 0x03, 0x04 }, /*B2*/
	{ 0x03, 0x05, 0x04, 0x02, 0x01, 0x00 }, /*B3*/
	{ 0x01, 0x00, 0x03, 0x05, 0x02, 0x04 }, /*B4*/
	{ 0x02, 0x05, 0x04, 0x00, 0x01, 0x03 }, /*B5*/
	{ 0x05, 0x04, 0x01, 0x02, 0x03, 0x00 }, /*B6*/
	{ 0x04, 0x02, 0x05, 0x00, 0x01, 0x03 }, /*B7*/
	{ 0x03, 0x05, 0x01, 0x04, 0x02, 0x00 }, /*B8*/
	{ 0x05, 0x00, 0x04, 0x01, 0x03, 0x02 }, /*B9*/
	{ 0x04, 0x02, 0x03, 0x05, 0x00, 0x01 }, /*BA*/
	{ 0x02, 0x03, 0x01, 0x00, 0x05, 0x04 }, /*BB*/
	{ 0x01, 0x05, 0x03, 0x04, 0x02, 0x00 }, /*BC*/
	{ 0x02, 0x04, 0x01, 0x05, 0x00, 0x03 }, /*BD*/
	{ 0x03, 0x02, 0x04, 0x01, 0x05, 0x00 }, /*BE*/
	{ 0x05, 0x04, 0x01, 0x00, 0x02, 0x03 }, /*BF*/
	{ 0x02, 0x00, 0x03, 0x01, 0x05, 0x04 }, /*C0*/
	{ 0x05, 0x04, 0x00, 0x02, 0x01, 0x03 }, /*C1*/
	{ 0x04, 0x02, 0x05, 0x00, 0x03, 0x01 }, /*C2*/
	{ 0x05, 0x03, 0x00, 0x01, 0x02, 0x04 }, /*C3*/
	{ 0x02, 0x05, 0x01, 0x04, 0x00, 0x03 }, /*C4*/
	{ 0x04, 0x00, 0x05, 0x02, 0x03, 0x01 }, /*C5*/
	{ 0x03, 0x04, 0x01, 0x05, 0x02, 0x00 }, /*C6*/
	{ 0x04, 0x00, 0x05, 0x02, 0x01, 0x03 }, /*C7*/
	{ 0x03, 0x02, 0x04, 0x00, 0x05, 0x01 }, /*C8*/
	{ 0x01, 0x04, 0x05, 0x02, 0x03, 0x00 }, /*C9*/
	{ 0x03, 0x00, 0x04, 0x05, 0x01, 0x02 }, /*CA*/
	{ 0x05, 0x03, 0x01, 0x04, 0x02, 0x00 }, /*CB*/
	{ 0x03, 0x04, 0x00, 0x01, 0x05, 0x02 }, /*CC*/
	{ 0x01, 0x03, 0x04, 0x05, 0x02, 0x00 }, /*CD*/
	{ 0x05, 0x00, 0x03, 0x02, 0x01, 0x04 }, /*CE*/
	{ 0x01, 0x02, 0x04, 0x05, 0x00, 0x03 }, /*CF*/
	{ 0x02, 0x03, 0x00, 0x04, 0x05, 0x01 }, /*D0*/
	{ 0x05, 0x02, 0x04, 0x00, 0x01, 0x03 }, /*D1*/
	{ 0x02, 0x03, 0x01, 0x05, 0x00, 0x04 }, /*D2*/
	{ 0x01, 0x05, 0x00, 0x04, 0x02, 0x03 }, /*D3*/
	{ 0x02, 0x04, 0x03, 0x00, 0x05, 0x01 }, /*D4*/
	{ 0x01, 0x03, 0x00, 0x05, 0x02, 0x04 }, /*D5*/
	{ 0x03, 0x00, 0x04, 0x02, 0x05, 0x01 }, /*D6*/
	{ 0x01, 0x03, 0x05, 0x04, 0x00, 0x02 }, /*D7*/
	{ 0x04, 0x05, 0x03, 0x01, 0x02, 0x00 }, /*D8*/
	{ 0x02, 0x03, 0x04, 0x00, 0x05, 0x01 }, /*D9*/
	{ 0x03, 0x02, 0x05, 0x04, 0x01, 0x00 }, /*DA*/
	{ 0x04, 0x03, 0x01, 0x00, 0x05, 0x02 }, /*DB*/
	{ 0x01, 0x02, 0x05, 0x04, 0x03, 0x00 }, /*DC*/
	{ 0x03, 0x04, 0x00, 0x02, 0x05, 0x01 }, /*DD*/
	{ 0x01, 0x03, 0x05, 0x04, 0x02, 0x00 }, /*DE*/
	{ 0x02, 0x04, 0x01, 0x00, 0x05, 0x03 }, /*DF*/
	{ 0x03, 0x05, 0x00, 0x04, 0x01, 0x02 }, /*E0*/
	{ 0x05, 0x03, 0x04, 0x00, 0x02, 0x01 }, /*E1*/
	{ 0x03, 0x04, 0x00, 0x05, 0x01, 0x02 }, /*E2*/
	{ 0x04, 0x05, 0x01, 0x02, 0x00, 0x03 }, /*E3*/
	{ 0x03, 0x04, 0x00, 0x05, 0x02, 0x01 }, /*E4*/
	{ 0x01, 0x00, 0x05, 0x04, 0x03, 0x02 }, /*E5*/
	{ 0x03, 0x02, 0x04, 0x05, 0x00, 0x01 }, /*E6*/
	{ 0x01, 0x04, 0x05, 0x00, 0x02, 0x03 }, /*E7*/
	{ 0x04, 0x00, 0x03, 0x05, 0x01, 0x02 }, /*E8*/
	{ 0x03, 0x02, 0x00, 0x04, 0x05, 0x01 }, /*E9*/
	{ 0x05, 0x03, 0x04, 0x00, 0x01, 0x02 }, /*EA*/
	{ 0x04, 0x02, 0x05, 0x01, 0x00, 0x03 }, /*EB*/
	{ 0x01, 0x05, 0x00, 0x02, 0x03, 0x04 }, /*EC*/
	{ 0x04, 0x02, 0x03, 0x00, 0x05, 0x01 }, /*ED*/
	{ 0x01, 0x04, 0x00, 0x05, 0x02, 0x03 }, /*EE*/
	{ 0x05, 0x03, 0x01, 0x04, 0x00, 0x02 }, /*EF*/
	{ 0x01, 0x04, 0x03, 0x05, 0x02, 0x00 }, /*F0*/
	{ 0x05, 0x02, 0x04, 0x01, 0x00, 0x03 }, /*F1*/
	{ 0x03, 0x00, 0x01, 0x02, 0x05, 0x04 }, /*F2*/
	{ 0x05, 0x02, 0x04, 0x00, 0x03, 0x01 }, /*F3*/
	{ 0x02, 0x04, 0x00, 0x05, 0x01, 0x03 }, /*F4*/
	{ 0x01, 0x02, 0x05, 0x00, 0x03, 0x04 }, /*F5*/
	{ 0x05, 0x04, 0x03, 0x01, 0x02, 0x00 }, /*F6*/
	{ 0x01, 0x02, 0x05, 0x04, 0x00, 0x03 }, /*F7*/
	{ 0x02, 0x04, 0x00, 0x05, 0x03, 0x01 }, /*F8*/
	{ 0x04, 0x00, 0x01, 0x02, 0x05, 0x03 }, /*F9*/
	{ 0x03, 0x04, 0x05, 0x01, 0x02, 0x00 }, /*FA*/
	{ 0x01, 0x00, 0x03, 0x02, 0x05, 0x04 }, /*FB*/
	{ 0x03, 0x05, 0x00, 0x04, 0x02, 0x01 }, /*FC*/
	{ 0x02, 0x03, 0x04, 0x01, 0x05, 0x00 }, /*FD*/
	{ 0x04, 0x05, 0x03, 0x00, 0x02, 0x01 }, /*FE*/
	{ 0x01, 0x04, 0x00, 0x05, 0x03, 0x02 }  /*FF*/
};

/* Table FIN
 * 48 entries vector containing pre-computed indices to be used for extracting the final digest
 */
static const byte_t MHASH384_FIN[MHASH384_SIZE] =
{
	0x13, 0x19, 0x15, 0x1B, 0x00, 0x11, 0x08, 0x1A, 0x2C, 0x1E, 0x12, 0x0D, 0x1C, 0x0E, 0x17, 0x26,
	0x14, 0x2A, 0x1D, 0x03, 0x20, 0x2D, 0x01, 0x1F, 0x05, 0x24, 0x06, 0x10, 0x2E, 0x27, 0x09, 0x07,
	0x16, 0x28, 0x0B, 0x04, 0x22, 0x0F, 0x29, 0x18, 0x21, 0x0A, 0x2F, 0x0C, 0x2B, 0x25, 0x02, 0x23
};

/* ======================================================================== */
/* INTERNAL FUNCTIONS                                                       */
/* ======================================================================== */

/*
 * CityHash 128-Bit to 64-Bit mixing function (Murmur-inspired)
 */
static ALWAYS_INLINE ui64_t mix128to64(const ui64_t u, ui64_t v)
{
	v = (v ^ u) * KMUL;
	v ^= (v >> 47U);
	v = (v ^ u) * KMUL;
	v ^= (v >> 47U);
	return (v * KMUL);
}

/*
 * Extract output byte from the context array
 */
static byte_t ALWAYS_INLINE get_byte(const ui64_t *const hash, const byte_t idx)
{
	return (byte_t)(hash[idx / 8U] >> ((idx % 8U) * 8U));
}

/*
 * Apply next "ADD-then-MIX-then-XOR" iteration on the context array
 */
#define MHASH384_UPDATE() do \
{ \
	ui64_t temp[MHASH384_WORDS]; \
	size_t j; \
	for(j = 0U; j < MHASH384_WORDS; ++j) \
	{ \
		temp[j] = mix128to64(ctx->hash[j] + p_add[j], ctx->hash[p_mix[j]]) ^ p_xor[j]; \
	} \
	memcpy(ctx->hash, temp, MHASH384_SIZE); \
} \
while(0)

/* ======================================================================== */
/* PUBLIC FUNCTIONS                                                         */
/* ======================================================================== */

/*
 * Initialize hash computation
 */
void mhash384_init(mhash384_t *const ctx)
{
	ctx->rnd = 0U;
	memcpy(ctx->hash, MHASH384_INI, MHASH384_SIZE);
}

/*
 * Process next block of input data
 */
void mhash384_update(mhash384_t *const ctx, const byte_t *const data_in, const size_t len)
{
	size_t i;
	for(i = 0U; i < len; ++i)
	{
		const ui64_t *const p_xor = MHASH384_XOR[data_in[i]];
		const ui64_t *const p_add = MHASH384_ADD[data_in[i]];
		const byte_t *const p_mix = MHASH384_MIX[ctx->rnd++];
		MHASH384_UPDATE();
	}
}

/*
 * Compute the final hash value
 */
void mhash384_final(mhash384_t *const ctx, byte_t *const digest_out)
{
	ui16_t prev_value = 256U;
	size_t i;
	for(i = 0U; i < MHASH384_SIZE; ++i)
	{
		const ui64_t *const p_xor = MHASH384_XOR[prev_value];
		const ui64_t *const p_add = MHASH384_ADD[prev_value];
		const byte_t *const p_mix = MHASH384_MIX[ctx->rnd++];
		MHASH384_UPDATE();
		prev_value = digest_out[i] = get_byte(ctx->hash, MHASH384_FIN[i]);
	}
}

/*
 * Get hash value for given input at once
 */
void mhash384_compute(byte_t *const digest_out, const byte_t *const data_in, const size_t len)
{
	mhash384_t ctx; /*transient ctx*/
	mhash384_init  (&ctx);
	mhash384_update(&ctx, data_in, len);
	mhash384_final (&ctx, digest_out);
}

/*
 * Query version information
 */
void mhash384_version(ui16_t *const major, ui16_t *const minor, ui16_t *const patch)
{
	*major = MHASH384_VERSION_MAJOR;
	*minor = MHASH384_VERSION_MINOR;
	*patch = MHASH384_VERSION_PATCH;
}

/* ======================================================================== */
/* SELF-TEST FUNCTIONS                                                      */
/* ======================================================================== */

#ifndef MHASH384_NOSELFTEST

/*
 * Minimum of three 32-Bit values
 */
static ALWAYS_INLINE ui32_t minimum3(const ui32_t x, const ui32_t y, const ui32_t z)
{
	return (x < y) ? ((x < z) ? x : z) : ((y < z) ? y : z);
}

/*
 * Population count of 64-Bit word
 */
static ALWAYS_INLINE ui32_t popcnt64(ui64_t u)
{
	u -= (u >> 1U) & 0x5555555555555555;
	u = (u & 0x3333333333333333) + ((u >> 2U) & 0x3333333333333333);
	u = (u + (u >> 4U)) & 0x0f0f0f0f0f0f0f0f;
	return (ui32_t)((u * 0x0101010101010101) >> 56U);
}

/*
 * Hamming distance of 384-Bit table row
 */
static ALWAYS_INLINE ui32_t hamming_distance(const ui64_t *const row_x, const ui64_t *const row_y)
{
	ui32_t distance = 0U;
	size_t i;
	for(i = 0; i < MHASH384_WORDS; ++i)
	{
		distance += popcnt64(row_x[i] ^ row_y[i]);
	}
	return distance;
}

/*
 * MHash384 self-test
 */
int mhash384_selftest(void)
{
	size_t i, j;
	for(i = 0; i <= 256U; ++i)
	{
		ui32_t min_distance = UINT32_MAX;
		for(j = 0; j <= 256U; ++j)
		{
			if(i != j)
			{
				const ui32_t distance_xor = hamming_distance(MHASH384_XOR[i], MHASH384_XOR[j]);
				const ui32_t distance_add = hamming_distance(MHASH384_ADD[i], MHASH384_ADD[j]);
				min_distance = minimum3(min_distance, distance_xor, distance_add);
			}
		}
		if(min_distance < 182U)
		{
			return 0; //self-test has failed!
		}
	}
	return 1;
}

#endif //MHASH384_NOSELFTEST
