{* ---------------------------------------------------------------------------------------------- *}
{* MHash-384 - Simple fast portable secure hashing library                                        *}
{* Copyright(c) 2016-2020 LoRd_MuldeR <mulder2@gmx.de>                                            *}
{*                                                                                                *}
{* Permission is hereby granted, free of charge, to any person obtaining a copy of this software  *}
{* and associated documentation files (the "Software"), to deal in the Software without           *}
{* restriction, including without limitation the rights to use, copy, modify, merge, publish,     *}
{* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the  *}
{* Software is furnished to do so, subject to the following conditions:                           *}
{*                                                                                                *}
{* The above copyright notice and this permission notice shall be included in all copies or       *}
{* substantial portions of the Software.                                                          *}
{*                                                                                                *}
{* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING  *}
{* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND     *}
{* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,   *}
{* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, *}
{* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.        *}
{* ---------------------------------------------------------------------------------------------- *}

unit MHash384;

{ ================================================================================================ }
{ INTERFACE                                                                                        }
{ ================================================================================================ }

interface

uses
  SysUtils, Math, Classes;

const
  MHASH384_VERSION_MAJOR: Smallint = 2;
  MHASH384_VERSION_MINOR: Smallint = 0;
  MHASH384_VERSION_PATCH: Smallint = 0;

  MHASH384_WORDS = 6;
  MHASH384_SIZE = (MHASH384_WORDS * 8);
  
type
  TMHash384Digest = array[0..MHASH384_SIZE-1] of Byte;
  PMHash384Digest = ^TMHash384Digest;

  TMHash384 = Class
  public
    constructor Create;
    procedure Update(const value: Byte); overload;
    procedure Update(const data: TByteArray); overload;
    procedure Update(const data: TByteArray; const len: Integer); overload;
    procedure Update(const data: TByteArray; const offset: Integer; const len: Integer); overload;
    procedure Update(const data: PChar; const len: LongWord); overload;
    procedure Update(const str: String); overload;
    procedure Update(const stream: TStream); overload;
    function GetDigest: TMHash384Digest;
    procedure Reset;
    class function Compute(const data: TByteArray): TMHash384Digest; overload;
    class function Compute(const data: TByteArray; const len: Integer): TMHash384Digest; overload;
    class function Compute(const data: TByteArray; const offset: Integer; const len: Integer): TMHash384Digest; overload;
    class function Compute(const data: PChar; const len: LongWord): TMHash384Digest; overload;
    class function Compute(const str: String): TMHash384Digest; overload;
    class function Compute(const stream: TStream): TMHash384Digest; overload;
  private
    rnd: Byte;
    hash: array[0..MHASH384_WORDS-1] of UInt64;
    finished: Boolean;
  end;

{ ================================================================================================ }
{ IMPLEMENTATION                                                                                   }
{ ================================================================================================ }

implementation

type
  UI32 = LongWord;
  UI64 = UInt64;
  
  TRowByte = array[0..MHASH384_WORDS-1] of Byte;
  TRowUI64 = array[0..MHASH384_WORDS-1] of UI64;

  PRowByte = ^TRowByte;
  PRowUI64 = ^TRowUI64;

{ --------------------------------------------------------------------------- }
{ CONST TABLES                                                                }
{ --------------------------------------------------------------------------- }

const
  MHASH384_INI: TRowUI64 =
  (
    UI64($243F6A8885A308D3), UI64($13198A2E03707344), UI64($A4093822299F31D0),
    UI64($082EFA98EC4E6C89), UI64($452821E638D01377), UI64($BE5466CF34E90C6C)
  );

  MHASH384_XOR: array[0..256] of TRowUI64 =
  (
    ( UI64($01DCDF00414B3037), UI64($B1B3AF661B8E96F8), UI64($944D2873DB393121), UI64($73DA9A36662AE755), UI64($1F4F318C4ECB56B1), UI64($F09743D99C2AA5BC) ), { 00 }
    ( UI64($A81FBBC6CBBFC954), UI64($39DE43648959EDDB), UI64($1A641A0BDA01822F), UI64($B52E607266932658), UI64($2C5B1731AC802084), UI64($C2EF10671FC79DD4) ), { 01 }
    ( UI64($CF2A8D8E08810046), UI64($8B7E9B2089E268F6), UI64($930461652C5D2DEC), UI64($F096E42223BFC8B9), UI64($8DD338964CFE0970), UI64($269C342F7CEC60BD) ), { 02 }
    ( UI64($B970A920D2ED0552), UI64($010F894A254DA7E1), UI64($8D5F205F9D1A40D8), UI64($50C33DCCC3FD5F58), UI64($B49F31BDE7D82C57), UI64($7CDE04F62A959033) ), { 03 }
    ( UI64($49FAB2822B9C84AC), UI64($34B8648CD68CBEF1), UI64($E5121147BB4126DE), UI64($C0B31F54B2FFE00F), UI64($2F193DA38E8CC632), UI64($058C984B429B8AFC) ), { 04 }
    ( UI64($E07F9DA44D8C9942), UI64($BE2AF9B39ACA65F2), UI64($5D3D8FB1466DC295), UI64($C0051E3EC3F962C7), UI64($F8FDC3CCD4CE2BB6), UI64($9024C0EFC6199937) ), { 05 }
    ( UI64($A3111377EF01F5EF), UI64($31F59B366C02A3FF), UI64($61B82949760D16DC), UI64($F6B958AF92BD3BDF), UI64($7297AAEFEC69C0B2), UI64($FE8A50AD9E8684CD) ), { 06 }
    ( UI64($3D3DD0C829EA9AA0), UI64($3E77597EEC96C7A3), UI64($8FD038231E7F1684), UI64($64617B131FB7EDE0), UI64($85C99CE4C5405874), UI64($A58735D41F19C5E3) ), { 07 }
    ( UI64($8028628ACAF91B9A), UI64($194A640538C97064), UI64($04A15E018A4F1680), UI64($F4BE1B04C2360955), UI64($DFB24D4CEF581A20), UI64($3C59D0A0FD15879D) ), { 08 }
    ( UI64($88F8E2ECE78AF1BC), UI64($B46B6E22297364C2), UI64($9339F17F926B99F3), UI64($31293489B3B8F07C), UI64($3909CE1649C9BCF1), UI64($0C46103BFA31BCE9) ), { 09 }
    ( UI64($4D1A0B8CC7EFE2A9), UI64($AFD7878CAD55E871), UI64($C89CFBC858CF4153), UI64($C4739486C74F75D6), UI64($0BF7192C130AC9F2), UI64($0084F2BC5E81BD9A) ), { 0A }
    ( UI64($4AFBE975014FCACF), UI64($41DEAF8CFACC41E4), UI64($3C1EC23B53ED16E9), UI64($78B06EB30F1C3248), UI64($DD11165D04285C72), UI64($6546D9B5609491E4) ), { 0B }
    ( UI64($FFD1E2E04DC8D260), UI64($07B186948A74ECCE), UI64($B5120E45121AC531), UI64($BD0BC86330810C85), UI64($DE93AFDDDB487730), UI64($78DEB0DE6EB99196) ), { 0C }
    ( UI64($79BAA2AC3FDEBA55), UI64($B1B7E3E1C92A567E), UI64($A53F69AB4A5B0794), UI64($F0DA7499954F6DDD), UI64($58117C89E1132248), UI64($D95DF5F794C51A6E) ), { 0D }
    ( UI64($89BC8A0C65091C33), UI64($B04EEAEE063135C2), UI64($F53267E04AB6E689), UI64($B4C579B7207E8BF2), UI64($3FD31E51343CD2DF), UI64($119E523E2F8197FE) ), { 0E }
    ( UI64($2E10CB1C6060F32D), UI64($BC92E732A94E6B63), UI64($F3220D831FD04267), UI64($502C5F7414BCE87F), UI64($89E0651DE91D2457), UI64($759E56B04482D915) ), { 0F }
    ( UI64($43AEE32C6A84E803), UI64($0C5007202C0BD7E3), UI64($B4F464474205D32A), UI64($7D17FC95DE386C06), UI64($E8DFBF64567AA545), UI64($55BD889D5853046F) ), { 10 }
    ( UI64($687ABE14EAB8DA27), UI64($397B3AB50D72C344), UI64($505EAA6D1FDE618D), UI64($61BE79865DA13F69), UI64($17BBAB29B5E90E2D), UI64($010C921972FA8B2F) ), { 11 }
    ( UI64($8B7223A4F56FF453), UI64($291B7B5CB98B6FE1), UI64($FD421625786FBF7D), UI64($E33D1020D0E8CDC8), UI64($CA530C708B739E87), UI64($878AF1F304B8A12F) ), { 12 }
    ( UI64($7ECE1F24E56DD711), UI64($2E0869241B2FA6DF), UI64($84B834DCC459B2FD), UI64($E022EDA6319E7D3B), UI64($59839D8CA03C9928), UI64($644790F491BBC774) ), { 13 }
    ( UI64($CBDC6F49E6B0DD0D), UI64($44BA2F8D00346732), UI64($86BCC821586AE61C), UI64($C7B7491285CEE55B), UI64($ED3912FFD97F3851), UI64($F4AF0186BEBEBCBF) ), { 14 }
    ( UI64($CA8A48E54ECCE516), UI64($BEDF1864B7F8F506), UI64($D56A1F60A4B36AA4), UI64($49B25AB5FE0DD9D9), UI64($21377DBB5E49FCE1), UI64($708F64F5D59D99E7) ), { 15 }
    ( UI64($E9F873A569AFE02C), UI64($DA66BC05CA997390), UI64($8C88174756D35385), UI64($EAAAF16CF4FDA730), UI64($B39F7A55653A3512), UI64($F10AB727BC23E852) ), { 16 }
    ( UI64($93E96FF8C5BBE2AF), UI64($A47785420253E97C), UI64($704B25D1F77D074C), UI64($C0B2093F1470559C), UI64($8D5BFDD4E364AACF), UI64($550518673F5B1BF7) ), { 17 }
    ( UI64($8DAC832E5BE81ACB), UI64($588BFB202B7583D8), UI64($A34D8A70DFD1D7E4), UI64($D03B0627B687033E), UI64($E3D0BE7EDBDC75CF), UI64($818EE8C5B09F8BEA) ), { 18 }
    ( UI64($8E3B02E1489D7D31), UI64($5336752B64E3B532), UI64($E4D4CC795C580A65), UI64($6DAB08F313ED767E), UI64($8E567E88FDBA36BF), UI64($259490F1D8E933D0) ), { 19 }
    ( UI64($EAB437D0D62CAA62), UI64($C090AD28B982B003), UI64($E255D768D25704EC), UI64($A048511AF6256A43), UI64($E79F078F4D498B5F), UI64($C41735FB75B357FE) ), { 1A }
    ( UI64($70DA9FC3504FF29D), UI64($B9AB1F388673FF25), UI64($36922F4CD17357BA), UI64($F09C3AB292E7E04A), UI64($90CE0BC3D9BA13EC), UI64($647C4CA63C918DE3) ), { 1B }
    ( UI64($D834A38DD1ECD688), UI64($A97A0020DE46AB6A), UI64($9CDEC7F6E62EA71F), UI64($288A5A6FD74DC47E), UI64($D44A2E2765CE50F1), UI64($BB9B50B5DB9E4F3C) ), { 1C }
    ( UI64($C66DA90E41DDF2E6), UI64($5A3FE86F160C5C09), UI64($6F6AF0405108CCBE), UI64($F938382EB627FC7F), UI64($163DD634617F006C), UI64($5184B1FEDC908497) ), { 1D }
    ( UI64($C95719ED07FCB21C), UI64($5112DF043F6EE7EB), UI64($50F9FD60012334CE), UI64($589FA85104D96579), UI64($B7129E44D71905A7), UI64($3314766E0733528D) ), { 1E }
    ( UI64($DC6C8014C5457CB8), UI64($D635FDCD286A69B6), UI64($D66F232CE27D01BF), UI64($56AF4AC0F682EC0F), UI64($57DF1D55B64328F5), UI64($651ED4C52A87CACA) ), { 1F }
    ( UI64($26D9D1CC79EEC502), UI64($69BF340A34B1EBFE), UI64($FAA5AAAC8E397174), UI64($D0A8F9BD426BCF6F), UI64($5B131F464D6D2452), UI64($122DD15660D0D6DA) ), { 20 }
    ( UI64($6E389AEC5F51A22F), UI64($7EF68F1C54C127FB), UI64($986D4D46E0485C30), UI64($F0A47B39E7CF8A31), UI64($1D398DFDB7F2A78F), UI64($2FC651D1FBB10D2E) ), { 21 }
    ( UI64($A44E4E8D1B49DCB0), UI64($07A4822049C2F343), UI64($C40AC04A8D6505BA), UI64($D9B91D3F0729B16C), UI64($AAF39951B50F9015), UI64($966EF5D3AD3F9076) ), { 22 }
    ( UI64($EA78CBAC0EB6D009), UI64($A0FEA6725A23DEAB), UI64($CE729C7444CB94D9), UI64($40A994626627AA0D), UI64($37F738CDE3D018D5), UI64($4C29491C01CDB3C5) ), { 23 }
    ( UI64($7C9792AEA745C87A), UI64($D1FF5620C5BD8FD4), UI64($9ECA84E3004B56B9), UI64($5AFD3923C228B1D6), UI64($E5DBF79EB3FD283B), UI64($441712E354084B9F) ), { 24 }
    ( UI64($E35D288BD8E249BC), UI64($91776C1453A366E5), UI64($F5D1E1684E95EFEC), UI64($9108E117D7DDF606), UI64($81B30F9DA2CE7C8C), UI64($6497DBD786818C0D) ), { 25 }
    ( UI64($C2F891FF45044BE0), UI64($75A1A76D2B87E2EB), UI64($85CE65798AF3C2BF), UI64($213F532B4EFD09DC), UI64($0DAA1DF5A53A6C88), UI64($3028606A50D826B2) ), { 26 }
    ( UI64($609A62616379F33A), UI64($A339A3BC53E4516D), UI64($D7AD92616A5ADBEC), UI64($D043726D86E924AA), UI64($8555B564F4C29865), UI64($56AA12AB31C1D6B0) ), { 27 }
    ( UI64($CED9ED85C1C17BFF), UI64($EB522325ACBAFFC2), UI64($04D3D8F4B2D15394), UI64($D271504C04756EEA), UI64($2DDBA4A91AF827F1), UI64($1F67D5C28F8002E4) ), { 28 }
    ( UI64($8B9C08AD432DC08F), UI64($5A7543E29796BBC7), UI64($34A6DB5B3C1967DE), UI64($016E3BC2A2804EE4), UI64($5B9BCACCE5172F75), UI64($7549598B80ADBDBA) ), { 29 }
    ( UI64($6F3FB117C5CDD155), UI64($16C3B0A59CD6EEC5), UI64($D9A1A411DE538769), UI64($938C54979F4AC37C), UI64($3737BCC1D55284DB), UI64($6CAD9F8AF9156BB7) ), { 2A }
    ( UI64($EBBF284F9C75EBDF), UI64($B383EBB406753DE8), UI64($AA86127AEE7C403C), UI64($10BFDD10523DE027), UI64($138BF6C4EB4A8A13), UI64($B1EFF67DDB78B067) ), { 2B }
    ( UI64($F6D1138D7AA3DA5E), UI64($BAA8098D8FB66371), UI64($DAE76D1B8B6CAAF2), UI64($400F1034368D1EDC), UI64($7C937F5172E8D277), UI64($7D05BBF83CADE6EF) ), { 2C }
    ( UI64($0E9C2EA6CF34B081), UI64($9036B30D58F60BA0), UI64($DB3A2C5848F08BCA), UI64($C87AD1B94250D564), UI64($7C892E09EEF96166), UI64($26DB85CF571085F3) ), { 2D }
    ( UI64($251EE3F58718C12A), UI64($F9438D81178A2AE4), UI64($F0929A889039A8A8), UI64($F06B65225EBDCCFD), UI64($2E4D14EDF7BF73C6), UI64($A9369895BC1DFACF) ), { 2E }
    ( UI64($CAE302B41D6979CB), UI64($BBFA5A58B51EE623), UI64($5113B99DC81AB52F), UI64($6093795BEC17A056), UI64($8F71FB4D2E5E355E), UI64($762F92EDBA34A2F2) ), { 2F }
    ( UI64($D130015265A4D9FF), UI64($09BEA253D71F26C8), UI64($81B6EAEDC46521E6), UI64($FAE268165682B8A9), UI64($A89C3EC4774AB623), UI64($0D2E45E055219DB2) ), { 30 }
    ( UI64($2B560284C3A692AB), UI64($37008AD0B379A7B8), UI64($AF11CD2C30F90BFC), UI64($7FE87A250F2762ED), UI64($C3FBD711647242C3), UI64($74ED8264F6B322BD) ), { 31 }
    ( UI64($28195CC8A7AD3943), UI64($53CBE808464C4FC4), UI64($D58E3D7A765F8726), UI64($D83052F60185AA4F), UI64($EFCB0D85223BB4E7), UI64($5A31305E787FAC28) ), { 32 }
    ( UI64($725D0EE230F19543), UI64($9091D2C6BDDF34E0), UI64($E3BE49C6C2754601), UI64($61BE300BA4AD566B), UI64($02D79D7551FA7CC1), UI64($6543910F5F1CDA58) ), { 33 }
    ( UI64($4099ADC44481B43F), UI64($FE1361922FD9EB81), UI64($A989C09E441FCEAC), UI64($449B3A13D3CB9019), UI64($45A9BE396F201134), UI64($DC1AD05A046633FE) ), { 34 }
    ( UI64($1A563A6D522F3E69), UI64($BE589E079F475A9E), UI64($75A2A9638E4C0038), UI64($DA3B6202577A0362), UI64($211D3F1E0D727AF6), UI64($5E1FFC529AD99233) ), { 35 }
    ( UI64($47B61E86C6D6D01B), UI64($437D6F83ADADC318), UI64($D5A361028DED738C), UI64($A00D4C630425164B), UI64($1A69AFA5AF4C9DD2), UI64($F99E1C67F951B582) ), { 36 }
    ( UI64($A66A7740B6BDEA79), UI64($FEF7FF1496AF80A3), UI64($05AFD43EEACD898C), UI64($B00C78ED31AD7134), UI64($0ED31A1AD7846673), UI64($74B96844161499BE) ), { 37 }
    ( UI64($46FA8D6CCBF6D12E), UI64($31C2FC147F303956), UI64($707F4401DE5F067F), UI64($3AE5FEC7E33594E9), UI64($28E39F8A63531714), UI64($B7B329EA1E9FCAB2) ), { 38 }
    ( UI64($EFD8F755825C7804), UI64($1F5A93870BD30CD1), UI64($EFBF894671FF8716), UI64($28ED617FF22BDA58), UI64($411289CCAE5CB62E), UI64($95DD42F41801F2F9) ), { 39 }
    ( UI64($A8525B8645FC59E1), UI64($75E62DC00A5F7F0C), UI64($09C56785210416AC), UI64($50EF76E9B30D7626), UI64($2B3B2CDC19F5D665), UI64($A41297CD11D8F4FF) ), { 3A }
    ( UI64($EAC99A649EEE5039), UI64($A593C92F143C0065), UI64($B314735203071206), UI64($EA2761A0C764A4EC), UI64($02AA7FD46CAC25B3), UI64($C68CC182A96D03BF) ), { 3B }
    ( UI64($B2873F024EC83CA8), UI64($97470AB8FD8853EB), UI64($18FE15C159B305BD), UI64($B0AB08F687EAEAFD), UI64($510A3FDE73602E43), UI64($03E1B84DCCF0FCF0) ), { 3C }
    ( UI64($D85BBBDC8033C0D8), UI64($9223D9C39CA9F34F), UI64($7D3BCB6D5B63C3FD), UI64($1C30F974DA0C0FB5), UI64($8B24BC9EBEFB5143), UI64($C58954925B7B84FC) ), { 3D }
    ( UI64($6ABD7C2E0844D7A7), UI64($CCF2EA456CDF530D), UI64($E8938CF52B3921B8), UI64($BA023CA2F281657C), UI64($EC635DA675D1EDAE), UI64($B4AA52F22EE1BE6C) ), { 3E }
    ( UI64($981C3AC677CB5904), UI64($6A92B54C84877B49), UI64($745BA6BB40C55815), UI64($B7AF550D22A371ED), UI64($D5E8BD87C65F5374), UI64($67874A37F0F538F5) ), { 3F }
    ( UI64($C23BBA2A9DECC021), UI64($4E610E930B0E3450), UI64($1A681AA91477577E), UI64($38A3209714EDC376), UI64($0FD15563EEEB4AB6), UI64($7D57668A01D42178) ), { 40 }
    ( UI64($6AF88CE145A098B5), UI64($1AEB858CD88B8B46), UI64($E8B733AFB8E2D6E8), UI64($313FAA8C10A7EBFA), UI64($127D375E77557CEA), UI64($96BDA2F70B2F2155) ), { 41 }
    ( UI64($EC8903978FAFB636), UI64($C7213C425C079763), UI64($760384036AB6D17C), UI64($E0C63A26385F1F49), UI64($299877D6811A6DF5), UI64($876F90FC5304B88D) ), { 42 }
    ( UI64($A6FABBC2D6E0BA16), UI64($9B70C9640080E6BC), UI64($29B2D5265598B27B), UI64($4A9657C726E4397E), UI64($A801CCC6766678D5), UI64($800EF7CC72619998) ), { 43 }
    ( UI64($235931A8CF5490BF), UI64($E798F98E0E8F879F), UI64($C6EEE29C38F30CA7), UI64($929A79F2D53E0024), UI64($88F2E12749587A45), UI64($0B85B28F38891965) ), { 44 }
    ( UI64($165E0303E4A4D827), UI64($67994F42D1E8436A), UI64($E6CC8BCF6E130D1B), UI64($50101711709DDEFC), UI64($373BDEC40CD05328), UI64($40B274A4AA5109F6) ), { 45 }
    ( UI64($A9F88BA008FDF8C8), UI64($ECC897E3476EE05A), UI64($BCE290AB69D57A74), UI64($FA44DB1811E3115D), UI64($6267AEFD64480C88), UI64($2697D04A2D3AECEB) ), { 46 }
    ( UI64($C0782AF2ABCD3313), UI64($02BA1290F2F96273), UI64($63C82F1A56ADC2B9), UI64($10F8E8C03EFE51C4), UI64($E3EB348625CCAFFD), UI64($93D607969CB8E7AE) ), { 47 }
    ( UI64($CC6E179443E58FBC), UI64($D21C93C655A7B8EE), UI64($2B9834A31F2B8BA4), UI64($C83B69516025ECEE), UI64($9176EB7B427AAE94), UI64($8CB65B9E30B7A76E) ), { 48 }
    ( UI64($C1A33A0AD6EDD989), UI64($18B3C5D95813B5F7), UI64($B024BD263B359A8B), UI64($C8C17C2216A99B50), UI64($71F9A11D58237729), UI64($3AA67C7618284290) ), { 49 }
    ( UI64($99B7465E09201C7B), UI64($9AF89FA01CA4FA81), UI64($FC2EC63E761AD123), UI64($E2A9A39585B17D14), UI64($08394DE529F94E81), UI64($479448E69794FAA4) ), { 4A }
    ( UI64($23CA3D1C4CBDCABB), UI64($E3265436CE1A37E4), UI64($1BBF10F69E8A4CC9), UI64($05A66708048F5C4D), UI64($E259DCDD9C5BFEFE), UI64($439E65FAFD936EFD) ), { 4B }
    ( UI64($A24D73B6978F719C), UI64($3F53F343CCB0BB8E), UI64($BE3C72769EE07C6A), UI64($FACB9E539CF558DD), UI64($67B91D4E30DE986A), UI64($1DB913D11698913A) ), { 4C }
    ( UI64($98BD4E140DC3C3C6), UI64($142B1592BF3263E8), UI64($CDBEAC59ED095B0E), UI64($900763F0F625896A), UI64($E213550F30324E39), UI64($8A13A4417A803195) ), { 4D }
    ( UI64($2ACD98ED8C626073), UI64($1CAAA6B4C4CF3238), UI64($04DCB41EB677EB5D), UI64($F88B5844A8105B68), UI64($981D9E951A061A4D), UI64($BC9471894C878EDB) ), { 4E }
    ( UI64($4959FEAD5D6C2DBD), UI64($6ABD59E28C503049), UI64($06D2C5494CAF8B34), UI64($70E4541304A4293C), UI64($520F3416CAF2F503), UI64($B23D09D92613DB85) ), { 4F }
    ( UI64($26B5A815C32D1791), UI64($2C99E7555BB033C6), UI64($09CE9D6A0002514F), UI64($D485282B2B8D7997), UI64($9C5B792F4A4A14C6), UI64($851D9D02DC0BB4E7) ), { 50 }
    ( UI64($62FEB6CACFB060EC), UI64($9D977D69D5C661EA), UI64($BF08EFD806D81556), UI64($25F1EEA460EA5718), UI64($A25346B51F5A9665), UI64($D92F9ADC358CA274) ), { 51 }
    ( UI64($27E63DFC63E8FFA6), UI64($CDB9CCE2CE99FDA3), UI64($979D5B754974830D), UI64($3298C8407D6693BE), UI64($629D5FADA39B42B7), UI64($2654D31271CD84E1) ), { 52 }
    ( UI64($AB1FA4DAF66E583C), UI64($EEB6B7A236D24766), UI64($A90738CDFDF5C6B3), UI64($28CBA9E5648E2D4C), UI64($FDE5BF6C0CFE0DA3), UI64($9D00B863D7D78485) ), { 53 }
    ( UI64($75FBBF094EEA16AA), UI64($48931F027CD729F2), UI64($5D360679009B2E7F), UI64($DDFCD148BD3DE21A), UI64($4DBFF544B094D0E1), UI64($9C0E5C6294352C22) ), { 54 }
    ( UI64($283A27FF968853D2), UI64($B0960C6CEA0D03F2), UI64($172BBA07A473DB38), UI64($688C87D296E6F4BB), UI64($5CB7E9BC5D68CF0F), UI64($57A5D71B0E47BFB4) ), { 55 }
    ( UI64($DE0108AAC1E4FF2F), UI64($D346CFABEAC62B99), UI64($B72E203F98B5F608), UI64($81853D8CA54B29BE), UI64($A6AED7C89FAA1680), UI64($D2093B155C39D7ED) ), { 56 }
    ( UI64($0BAEAC99D4974B84), UI64($C7F258A699C9B4DA), UI64($6F622C5E4ACCF5C1), UI64($58AB397D9781BEAA), UI64($BF811F67E101FFE3), UI64($AFBCC2881C3C0EF3) ), { 57 }
    ( UI64($26B211FB518D6C3E), UI64($64BADAD51A10784A), UI64($E6BE4E06A587186C), UI64($D471F5C61343CD5C), UI64($8389BB0DD6AAED5D), UI64($C88112678914A17D) ), { 58 }
    ( UI64($2B2D0BC3BB88D27D), UI64($C5A7D1FAFF517AD2), UI64($96F39056A09F82AD), UI64($FB38A61A6CED4D4E), UI64($9D308E4EA6F9B264), UI64($9097CE294AECC6B3) ), { 59 }
    ( UI64($8FCA2B950690B1A2), UI64($293EFCBF03D422DF), UI64($8C9125B3E76353AB), UI64($3D402092A1A70173), UI64($9BAB974CAB9BF676), UI64($5EA8FCC55D8C586E) ), { 5A }
    ( UI64($408C92E8C2E1EC8C), UI64($4AF4C914B71B4350), UI64($5186AEE0CDFB1069), UI64($2385EAFAB9657C67), UI64($F708E4D3C898CA80), UI64($1EC8B9F89884907E) ), { 5B }
    ( UI64($46E8958B6A2C1878), UI64($2172FD410F78A647), UI64($9D8E9DD83A299004), UI64($390913C3265AD025), UI64($D231F1E23077CBF1), UI64($E7EE3E574E80D7F3) ), { 5C }
    ( UI64($5A8567A3D85E40B2), UI64($16ECF161133FCF73), UI64($52DA5C6FBA3C0DD7), UI64($56E57983DEB34BFB), UI64($83254FDCB768D153), UI64($9A14F95F35C6B82D) ), { 5D }
    ( UI64($498A29C6E19D4AE6), UI64($2EF4AAF46027BA11), UI64($BDBA7DAA84F39505), UI64($940B2A04F6DC944D), UI64($4E7ED35610FC0D53), UI64($BADD94C2907E59E1) ), { 5E }
    ( UI64($14DF0FC43F475F80), UI64($17E2AA8D264BF82F), UI64($92625BDFE58B934D), UI64($8384F415A4ACEA81), UI64($8E9C5EAEC5D8642B), UI64($4D8EF55F1C826687) ), { 5F }
    ( UI64($4A2335C4F77128D9), UI64($544E1476D29ABA94), UI64($654EC86321785044), UI64($B04AD9B02F80445A), UI64($B0E01B6480C8D020), UI64($596E325E88A3CBBF) ), { 60 }
    ( UI64($896955157448D062), UI64($0DB08C4C0F236D68), UI64($3BA8FC5B3CD1C4A2), UI64($04F57C53E144535B), UI64($B7D04DCC7BE46840), UI64($4BBE993192334646) ), { 61 }
    ( UI64($1D7837E6AB02CE27), UI64($3EA35BAED4493EA4), UI64($D1CAFDB5DF94FABE), UI64($98B580BB62170C4F), UI64($C3C57A6CA9421C43), UI64($68D65FC2C1201634) ), { 62 }
    ( UI64($FAEABABC48717536), UI64($454251E8F62F7315), UI64($B318E8A7FDCDC523), UI64($7C2E832013C91344), UI64($4D9E5DAFD1699052), UI64($12262E8C870537A7) ), { 63 }
    ( UI64($8A3E5D0BEF8402A2), UI64($A33BC5FAFA019909), UI64($63CBE8ACD00762F5), UI64($EA26A3F181984178), UI64($6EEB78D1BB4AF6BB), UI64($7ECF9671300E845F) ), { 64 }
    ( UI64($0811B67CCCF5D0FC), UI64($9F8CAB3F3496BD6B), UI64($57CB7D24F1355C2D), UI64($58218594165BDE80), UI64($FAF3368A653A78F8), UI64($C04CD80176267762) ), { 65 }
    ( UI64($E6417CE75AAA23B0), UI64($34A7BFE3CBA61761), UI64($8C13E396F8C9B6ED), UI64($5C9066464B09ED63), UI64($76CB6A642C5CE283), UI64($498E082A3EB449C6) ), { 66 }
    ( UI64($6F2ADEA6357B5AA0), UI64($54DA382B15557B69), UI64($302BD81946237AAE), UI64($8F0CBB82111EFEDC), UI64($45DD2DADCE20F2D3), UI64($8A77A5E9E8A2D1D8) ), { 67 }
    ( UI64($E1EC332735862A28), UI64($92B68B1A7E9C7C44), UI64($F45618DC99E963E3), UI64($7CAC984502DD1A73), UI64($C8650598CD70840D), UI64($9A5DA584A26D4EFD) ), { 68 }
    ( UI64($16B19B010740C15C), UI64($B4544AC01016439A), UI64($221F749C9E2F99A5), UI64($A63E8A279A65570F), UI64($C7231669ADD072AD), UI64($C5BC35BA740BC801) ), { 69 }
    ( UI64($6C44E75A4F378694), UI64($D27ACE108A577647), UI64($17C487FAFA7E15D6), UI64($6A3654D5C8E29EDF), UI64($0CE35EEDCC611FFA), UI64($D88A8C03C0095093) ), { 6A }
    ( UI64($CF106948BC4B1F2C), UI64($91C0DC9990B99712), UI64($193B21E3E109AB32), UI64($3340DE0608DD1666), UI64($8A5BB677BF602828), UI64($402C410B1197B771) ), { 6B }
    ( UI64($EB080FF49CA5543E), UI64($B4B9429542D6CA27), UI64($5999D45DC1533205), UI64($F7EA9E398A1BEF3E), UI64($BE8817775476DEC6), UI64($17064D7790C84100) ), { 6C }
    ( UI64($F3328E9150A7F8D6), UI64($52E3E61B04ACFDF8), UI64($51D82010F3CEB015), UI64($59D673336676D5D8), UI64($4CB3BCEF1D91C342), UI64($0C589AB58033BE49) ), { 6D }
    ( UI64($54B8E70EDCE03855), UI64($7BB590E99687FD57), UI64($6CFF08688D2B1FDD), UI64($FD0F6D068BFE994F), UI64($EB9BCE302489AE44), UI64($66B21F200661E3E4) ), { 6E }
    ( UI64($2F5E0060189669AD), UI64($473AF1D03C00CAE4), UI64($0278299268D1F3B4), UI64($888714BC3A7EC9D2), UI64($9FF9C7F071EBD2D9), UI64($875A5DC25DFFDB10) ), { 6F }
    ( UI64($E2A97A3E468399D8), UI64($3BF7EACA32C80DA1), UI64($13DCAC8EB6C2231D), UI64($227EC90E1102EE97), UI64($B2344832F0381434), UI64($8613888303B190EB) ), { 70 }
    ( UI64($3A3D3B6CE026BFFE), UI64($18D4953B9A68ED59), UI64($24BB7B574AB777A0), UI64($E0CB7DD64983DCB1), UI64($CF768C439869AC97), UI64($8062BC7A900E6033) ), { 71 }
    ( UI64($39D4C3B78A7A33C7), UI64($43D72EF22AB0B4EB), UI64($54AE8184DDA50394), UI64($0C2A7DA083C38536), UI64($9DBC6F921D4AD822), UI64($2CBB61FE182EAA42) ), { 72 }
    ( UI64($D8CE9A806C0BD24D), UI64($F69D65A65845727C), UI64($C3FF81CC76F2B048), UI64($76B1FDC3CA67CE58), UI64($CED0970AFBCBE78A), UI64($57502941B726F5F3) ), { 73 }
    ( UI64($E006AEC17FCEFCF9), UI64($05CAA1629E003591), UI64($B7050CC99F585312), UI64($669260401E159490), UI64($8442D25AA757CC5A), UI64($228655CD4038770C) ), { 74 }
    ( UI64($93EE8D67D3F1F3A1), UI64($BEA46D48DBF8D7F4), UI64($3C91F02B8646453C), UI64($6C3D7C1F04188A58), UI64($EFA97287F89CEF84), UI64($CB40364E108BFF4B) ), { 75 }
    ( UI64($C6DCE3730D4FF825), UI64($02AF54F87D972790), UI64($7D69D20F6F4F788F), UI64($90C255C64C166E8F), UI64($A3529FBF4BF9C9A2), UI64($3ECEC41136694F6B) ), { 76 }
    ( UI64($3DE10A5EC6CA7B3F), UI64($7E196081D085ACAA), UI64($DF5F0DE3705D60F7), UI64($393E7C83DCC57075), UI64($A5F33BC2DCB98F97), UI64($0AEB7F050D1204C0) ), { 77 }
    ( UI64($6F3B3B3D11A8BC05), UI64($B52269AB2B95B8DC), UI64($12EDE24EB1385F13), UI64($202BBA6B5836B5E1), UI64($EE3636C5925ACC49), UI64($42224CF6EEB509BF) ), { 78 }
    ( UI64($5F0CC3BBC4BE9A92), UI64($584313FCCC54DD2E), UI64($C11FE90F00394036), UI64($3371667C72FC9723), UI64($9611990B62AC8D9F), UI64($4CFCB9EB3C317FAD) ), { 79 }
    ( UI64($CA8E520A894A3FBA), UI64($BD9ED1B80098CC40), UI64($BDF24507DFF3757C), UI64($47AEC572E68D35EC), UI64($F3D4523D27B373E4), UI64($1AB11E16973A05AB) ), { 7A }
    ( UI64($FFC293A6C26B817D), UI64($2C9E9D134959D828), UI64($7FA5216408199BBF), UI64($A6F002DE0DCCD861), UI64($BE8F9DC57F2CF35D), UI64($1352E2DF86A47647) ), { 7B }
    ( UI64($84B55BE101708E74), UI64($3ADEC53721209F3E), UI64($B18F9A1E68DFADBD), UI64($09A050819774CF2D), UI64($E4AB295D380A8762), UI64($A3605B0C689C239F) ), { 7C }
    ( UI64($DDC7031FBFDFFE8F), UI64($0B175DE65B832F0A), UI64($31162ABC65719685), UI64($51215E534BBC36B1), UI64($9F2F7D3B5D01AE44), UI64($CF43A2426E83B61B) ), { 7D }
    ( UI64($7E32DB672B16F04A), UI64($CE6F45DE0E6AB788), UI64($25718548B8E70B41), UI64($D7368BCF39A0FAC4), UI64($956863EC49880C47), UI64($720E335796341674) ), { 7E }
    ( UI64($06707A8E33D9D6C6), UI64($B684BFE26CD576C6), UI64($44F47E5ECD5FC46C), UI64($AF1B23A856D844B7), UI64($98A627916AC5657E), UI64($040C3964A1127E19) ), { 7F }
    ( UI64($A5DAEC3134C0A39B), UI64($0CA04160BD5ADB1F), UI64($B50EC5A9F29E1ACB), UI64($BE2FA1126AF7BFAF), UI64($BEFC0AC4C9C5A4B3), UI64($994739C71FB1EB29) ), { 80 }
    ( UI64($6FEC2D343E83A763), UI64($5BDBA5715757F50C), UI64($D6F6282EE46A11B3), UI64($A8B501F5922A5524), UI64($A782A21006B605CA), UI64($A10BD2E896975C81) ), { 81 }
    ( UI64($B8AAE0532226D0ED), UI64($891831C0470E84B7), UI64($74C824D648E8FF28), UI64($B5E4E02EAD3906EB), UI64($5ABB086ADA60A713), UI64($A80C57666A9E29F1) ), { 82 }
    ( UI64($529E3E52B1E7230A), UI64($0C148861C9F08E26), UI64($0CFC8A131BAD803D), UI64($8C09F324902FAA9F), UI64($0231EE4987999848), UI64($3B0688492E2B5457) ), { 83 }
    ( UI64($EFA6EAC5036814CD), UI64($02773C1F8DAA5DF5), UI64($0E4EEDBD0702DE31), UI64($BA7FD757D0D740EF), UI64($A8805F0C74005F8B), UI64($1448467BFF3E1EF8) ), { 84 }
    ( UI64($2A07B766016AC70D), UI64($64215C35364219E9), UI64($CD6F7EFE35FCF6F1), UI64($F05CC06084C29267), UI64($AB3BF2F32579A444), UI64($AC75F42D9A25B9C9) ), { 85 }
    ( UI64($EF3A14B5EDDB8464), UI64($2314E0802D2DD0E9), UI64($14DEAEA9F928762A), UI64($5763EBB480E15A02), UI64($25F7CA14E8CDF5E6), UI64($8E594510DC61E6BC) ), { 86 }
    ( UI64($E62C38DCFD21000B), UI64($7BB32AE917EE3DA7), UI64($E49F15E24CC9B656), UI64($56E28259DCA361D8), UI64($B43B8008A9285F48), UI64($0DC6B4AF7E4AE61B) ), { 87 }
    ( UI64($703C64241142DCAE), UI64($732D33342C45063A), UI64($37877EA1624567CB), UI64($2871D534614DD114), UI64($E748092A1D94F5D1), UI64($4524056F0C6D1CB7) ), { 88 }
    ( UI64($E325B1823A595DF9), UI64($742D0DD5C96F397C), UI64($44361C9540A9F451), UI64($02382F9BF6331FB9), UI64($8ECBAFBBE91A0467), UI64($528EBF3811F904A8) ), { 89 }
    ( UI64($FD2BC6534631FB0D), UI64($27A5F036FEEB9A6C), UI64($D0F876D7911D0775), UI64($12EFB3A29C6E0B72), UI64($DC4BCA3D5E871DA1), UI64($028FB6E6E608F46F) ), { 8A }
    ( UI64($EF17ECC8930A7B4A), UI64($9D97B34672FB273D), UI64($C6AE835F35A25D8F), UI64($6C27469530C21F5B), UI64($2FBC16A26150E795), UI64($02AD93AAE0B5C71A) ), { 8B }
    ( UI64($6D24BE43CF07DD56), UI64($63681D62A38D2A2F), UI64($9872C9B411724AA0), UI64($B882B4857C19690A), UI64($87B1BA8D2804C6F4), UI64($D7B199CC36F40B49) ), { 8C }
    ( UI64($EEFB8D8573FD9E0F), UI64($933403199B91560A), UI64($FF0DB41665D5248C), UI64($322EE105EA984196), UI64($DB8CE0F83890D89B), UI64($3A32F8983C901F80) ), { 8D }
    ( UI64($082CDAF93F215BAC), UI64($67C118A1B9274FAC), UI64($AF74501CFB93198A), UI64($53525CABA0E812D3), UI64($C9AF3A005EFE8A6E), UI64($F242DCB60DA7B2FE) ), { 8E }
    ( UI64($D3887FBFBB7314DF), UI64($DDDCCCF0F720C342), UI64($B2C4331C33C8C415), UI64($1666010767F4785B), UI64($8455B7C1FD5DE487), UI64($A821C5EA181875F2) ), { 8F }
    ( UI64($7E289831418562F0), UI64($2AD12E3042B185C3), UI64($7C20D0D735A6AE96), UI64($A68BEF98E22CBD41), UI64($A1411D22F8D93243), UI64($D813FB404F3D2F38) ), { 90 }
    ( UI64($E13FC0A76F664294), UI64($7E21C9D9F7FDDDCB), UI64($161E68B366D6B1F8), UI64($55BF957EB5743874), UI64($B23213EF8364D766), UI64($529BB98AF96643D4) ), { 91 }
    ( UI64($036D7ADDAADB5C33), UI64($0525835F802D032E), UI64($7DF7D0D8D7A2BEF2), UI64($84927644B27696B7), UI64($215E21E4D1F9B5B9), UI64($77743669C40EB7FD) ), { 92 }
    ( UI64($A9B3534BE8897784), UI64($5BFD4283541A5090), UI64($97AFFCCD121C9778), UI64($C146C4C9637989C7), UI64($0820E72FCBDA59C7), UI64($5526E2F4A0AE4F4F) ), { 93 }
    ( UI64($A4739E20FD72BDC2), UI64($6D6EE5A5C1A54CA6), UI64($70A97A6FCB884E5C), UI64($2B6108339E979C48), UI64($93A63730D6BB23A7), UI64($5B1DCEAB00045EE5) ), { 94 }
    ( UI64($427C14E4F88C8BDB), UI64($1D8630868E039BC2), UI64($33DB40A251502D1B), UI64($E043C9CCB45D2B3D), UI64($292B67B6EE077B2D), UI64($1C3A2FBDE24C742A) ), { 95 }
    ( UI64($3DED69F37016D86A), UI64($9A947B13AC66D7C3), UI64($822D8645DF4CB39C), UI64($2BA20F98F19E10DA), UI64($6703138D422AC4C4), UI64($8D34D6138FA04A1D) ), { 96 }
    ( UI64($28E59C8B257D112C), UI64($8747068CC5499FCF), UI64($D6C16EB780F9191A), UI64($B416151633F7AF08), UI64($A230E00D6BA1A1C3), UI64($FD066FB9965B83D2) ), { 97 }
    ( UI64($70F4BC1B7F8FFC37), UI64($38DC0331E56B0FDC), UI64($A9AB7290AD2B0BBD), UI64($B307973C3D0783C6), UI64($BDC455F6CDCA111F), UI64($23F0E08317B8F0DC) ), { 98 }
    ( UI64($0AEEC24E9285C50F), UI64($3BCDA47833B61ACE), UI64($839986F959EE0723), UI64($C959034A8D7F5EB9), UI64($D4AD7E05B05C4FB5), UI64($6C37A3D39F7A0EC4) ), { 99 }
    ( UI64($0227B7230FBF2D07), UI64($28D7D2AD632BED47), UI64($07BD8F8B5012EFD0), UI64($48A0D43AE0403442), UI64($9B8939207F1449A1), UI64($351EAD01B9FDF219) ), { 9A }
    ( UI64($A7119D2E311CEF25), UI64($1E532CD0C4ED0479), UI64($2272F878D8D30A0B), UI64($769C412CED9C4C42), UI64($262FFBFA65CBDDF5), UI64($DB73D86721EA368E) ), { 9B }
    ( UI64($4BDBE90B3FBADCB2), UI64($1324EC3A8D6FEA57), UI64($6D9EFBE530850D00), UI64($401A88AFF8A4C8F4), UI64($655CB76B8A2E271C), UI64($35505B6DBDE16F43) ), { 9C }
    ( UI64($6E15E57E23F57037), UI64($4962737362C1FA26), UI64($C962372D1829B80B), UI64($A1FE6832EA4D6211), UI64($6726E307F96E7763), UI64($04C761081677505B) ), { 9D }
    ( UI64($42E2FF3A8A6FC164), UI64($FB85B2BC9D28B268), UI64($C559CFF024533A28), UI64($2EC83F3911DAB3CE), UI64($AE0FC74A9D736A27), UI64($DB9CDD048BAB4CCF) ), { 9E }
    ( UI64($D79C52221D20E765), UI64($499EDD73903CE704), UI64($9B016D987DF48349), UI64($FCFAB44AD12FC5C1), UI64($811293F3B800FDF9), UI64($511DC619CA53CEBE) ), { 9F }
    ( UI64($A059EE78B826EDDF), UI64($4673AF294D17C85A), UI64($5E527D4E4DF282B5), UI64($DB5B9A2693F95CE3), UI64($6551D304FB54F296), UI64($AB3EB70D65912FCC) ), { A0 }
    ( UI64($7D0C4F67B6C78135), UI64($390CAEA7DE304D37), UI64($49E19FABC8D494FE), UI64($1A9E1B6437A04516), UI64($886CC4BDAB6AF35A), UI64($0529217344F502FE) ), { A1 }
    ( UI64($3CEDF34141B52CEE), UI64($8133BA924753573F), UI64($CB32BE22BC66025A), UI64($0C480183DE403CB3), UI64($BF5B84B427DFCF31), UI64($7251428DB0232156) ), { A2 }
    ( UI64($86FCE831C58E25CB), UI64($5CC43FFE45CBFC75), UI64($33877CC042F199BE), UI64($1212FA7F0CC22E1C), UI64($448EAB4B7D1F9823), UI64($A7B1363A9FA7599E) ), { A3 }
    ( UI64($2D8C2FEDA0E5106D), UI64($192E366838BBEB3F), UI64($36226AA60ACEA0AF), UI64($E7E1285DC1F3926A), UI64($900371FA1883D9EC), UI64($BAC33B1AF360EB66) ), { A4 }
    ( UI64($D4A2A11612BDE0E3), UI64($82AB0DA614CB4CB8), UI64($189A4D50AC01F4C6), UI64($E36A5DA1D9F6A647), UI64($E43120D6B16B11B6), UI64($7D395F4236E75378) ), { A5 }
    ( UI64($C0C155CD47F3877F), UI64($4B03BFE5C334CA71), UI64($77710F1F4B844FF7), UI64($3443BBAB720E8DC5), UI64($F03F8868C5863406), UI64($0FD60511C872EB50) ), { A6 }
    ( UI64($8C253DAAB5286306), UI64($9AA438F54A6196AC), UI64($181D08C723A22C5E), UI64($633C49C88E3910A1), UI64($C9F54A67992675B0), UI64($1FDD98ACBD38D976) ), { A7 }
    ( UI64($A10893DA7575A9F7), UI64($8F5F4A025AB2A018), UI64($D80538F0336BFFC0), UI64($0F9751D33889626F), UI64($30383EB925BF911A), UI64($E6149F68CE19CC60) ), { A8 }
    ( UI64($B9081DBAC6BE0598), UI64($785DD9BC69C71492), UI64($8B035A0CA56E172B), UI64($8946783500724888), UI64($AF1E57C958650569), UI64($E1DE4E944FF22261) ), { A9 }
    ( UI64($EA5EDC4D2718C0D2), UI64($CB1C5D4DA15A8AE4), UI64($C6272382F8163015), UI64($94A934E5057B54CE), UI64($658E481A3D68D10D), UI64($E8F24929E50A46A0) ), { AA }
    ( UI64($7DF146281AF482CD), UI64($014B68E726407B06), UI64($6CE564938C70DDBC), UI64($36DAD2DE72A5DAA2), UI64($6D573BF69C0B2980), UI64($684DAB14B4AA0329) ), { AB }
    ( UI64($9C69DC064E738B5F), UI64($83CC16BD5A1C36F5), UI64($A99B365E6E141B12), UI64($2748FA5AD0FACCE8), UI64($26D073A047D99C49), UI64($B005B182505B0C0C) ), { AC }
    ( UI64($15B6A2A20ED0FD1C), UI64($9333AF729BD65A25), UI64($22CC333293BD2C1B), UI64($D724D949B15E8BE1), UI64($69D0DB0512B97117), UI64($85ACA8980DD7653C) ), { AD }
    ( UI64($230EC629D77BB3F2), UI64($43115B991D297CB2), UI64($A2F955792C53C76F), UI64($48A76728EBE25BA7), UI64($7CE662A405384400), UI64($DDC06B7E6BF49D66) ), { AE }
    ( UI64($20DDB9BD7644410B), UI64($056391B1FA2E8C06), UI64($CA4EDE51CF167C00), UI64($46602B550536F870), UI64($5040672597C21FF4), UI64($0AF8EC6E8AFB844B) ), { AF }
    ( UI64($0023C5749251B883), UI64($335A4F86D66B7E00), UI64($AE353DED3EFACE8F), UI64($3FC80526D67B35DE), UI64($0D9078FBDA80BC53), UI64($467900DFF3FE4C14) ), { B0 }
    ( UI64($0F9CB2BE6A448113), UI64($E38D541B6A9A5829), UI64($673953DAF354FC0E), UI64($3C818A277F8569E9), UI64($8D16EA77DB122A3B), UI64($E40A860318B6EA84) ), { B1 }
    ( UI64($78CE11F42D7D5E50), UI64($84F76DFF199C998D), UI64($999B578E3AE935CB), UI64($D9FD092C1BE63212), UI64($31F33C63ACD316D8), UI64($5AA08030B8D65C0C) ), { B2 }
    ( UI64($0098DBE19CA84FE9), UI64($E2426617D1142137), UI64($63C3C4166A78E21B), UI64($74B145353E03B0E4), UI64($F43C0824EAE508C4), UI64($58C1E6622528602A) ), { B3 }
    ( UI64($9E27EBE6D1426A6F), UI64($2A6A600A6B5FA342), UI64($8FF7E2306BA90370), UI64($DF83D91A683EDDDD), UI64($29572442F0225388), UI64($E9CC0F1B6437320A) ), { B4 }
    ( UI64($054DF380E896064E), UI64($FAB81A4AA3AD88A4), UI64($F87426486CCA156F), UI64($BB1B3C8237472960), UI64($7EC0B87CF73F960A), UI64($5C57D7E6470F7808) ), { B5 }
    ( UI64($5758E103AC614A1A), UI64($766AEE86F81358DF), UI64($203FBA51DC74396A), UI64($78C93DF969C5721F), UI64($E69E32E230196597), UI64($E287C6CECD8AB95B) ), { B6 }
    ( UI64($2A06A7C10C0DCC97), UI64($99D5298268A6745F), UI64($F2D818BB774858B3), UI64($D52A820D4F64D886), UI64($2F808EF87A263981), UI64($BB91206E6347C676) ), { B7 }
    ( UI64($0847C6D71CE0C746), UI64($86FD451B447C1E11), UI64($C20623B0E2856FCC), UI64($3ADDFA2D0398181E), UI64($6736A0A06B336B46), UI64($D1C70AEEB2B1257D) ), { B8 }
    ( UI64($5633260D141A9776), UI64($D530805F596CA3DB), UI64($8CE33EF69437CE46), UI64($F62D54E97E747088), UI64($DF5C9318489B45EA), UI64($A4AAD29F0BA850CA) ), { B9 }
    ( UI64($BDBD7B16767F6D9F), UI64($F7968427F1B7B6DD), UI64($58C76599B35276EE), UI64($286F4C7F6CADD791), UI64($8188C0401742117B), UI64($CEC4F1964266D163) ), { BA }
    ( UI64($97E4E8A6B5135B24), UI64($8A8BD785E5297977), UI64($4545C1A0975BC5BB), UI64($13FAE3BD9F59E37D), UI64($AFD5627C0E91DE2B), UI64($A223AC778474E1A9) ), { BB }
    ( UI64($DE1BF1EAF86C6B3B), UI64($A246A3ACD50035FE), UI64($6F80179DD96A21CD), UI64($3F8DB7CB17300D03), UI64($497A798B5D94506C), UI64($AD52DCC6F61AE841) ), { BC }
    ( UI64($F4A4E1D08E1F440B), UI64($5E27633CD56422E0), UI64($1465C14F1DB41420), UI64($9A939043988D37C2), UI64($CBE65CFA245DB368), UI64($6340AEDE28DDA855) ), { BD }
    ( UI64($1F7AB65A3F892454), UI64($D70AB4167EBEB5A1), UI64($9B2631E824C2028D), UI64($D5D97BDEE31519BC), UI64($EA2DC77449E4058C), UI64($EB204F2D6D2FBAFF) ), { BE }
    ( UI64($6537E69171A2665D), UI64($3FD2F835435A3F23), UI64($ADD5DD3E622D6C8A), UI64($C522CDD5E5E243F8), UI64($5AEC27F3DBFDA8A2), UI64($477A65ED570E1445) ), { BF }
    ( UI64($3BA7CB01D32E9D63), UI64($9E335734E7B5416B), UI64($0ED96A84F94539F6), UI64($45CEE2E46DF5A70D), UI64($DE142EE1E9AFEC1C), UI64($78D6121C4FDC72DD) ), { C0 }
    ( UI64($7BB30AF653390B77), UI64($2D394F2B7F8F7BB6), UI64($0277A3C213AF3489), UI64($7DF6E674DD56D084), UI64($5643CD3073C42451), UI64($FAB15F8BD1A1DC18) ), { C1 }
    ( UI64($42B453ABF5150D8B), UI64($913F109C1188E18C), UI64($C27BB7631FB43BF9), UI64($EBDDE685EF108419), UI64($76D67C87C56D33EA), UI64($95EC73C0AF40F084) ), { C2 }
    ( UI64($BCE43D59A1F50BFB), UI64($BA7027CA04D84600), UI64($FB6FDB98A2BE644B), UI64($D5DE777E993DED4A), UI64($FCA39F1EDF710F3A), UI64($A5E5893C858D8841) ), { C3 }
    ( UI64($C68AC776E6AEACFC), UI64($538067C7866106EB), UI64($D27B4A352F4EFDE3), UI64($847DA2B3BF01E378), UI64($3C79E3C136926D58), UI64($F957BC8726AA1610) ), { C4 }
    ( UI64($95492C4203C7C612), UI64($0DD60DB1EE8321FC), UI64($E1D9EBA902F62B42), UI64($EA2DBF7D0E37A4F2), UI64($E11FB9098BF5DA48), UI64($DBFE213F818EA338) ), { C5 }
    ( UI64($17CB21316D4756DD), UI64($B88952498140146A), UI64($648112F580844288), UI64($4947ADC3F7D58F35), UI64($651CCE28E26A5377), UI64($0B3803DAF337F89B) ), { C6 }
    ( UI64($BEAB16E2DCE6B6E3), UI64($8F39ECC8E39172DF), UI64($607CC9553FF29C0E), UI64($4BFD15154F4F0BA7), UI64($EE6230B6BD408CE4), UI64($35B654110D164E99) ), { C7 }
    ( UI64($ADDDFF1BD2C11CD4), UI64($2A1A262CBA6E1AA0), UI64($0BF2291D09475A46), UI64($4C93A0ABADF4DE32), UI64($73EE8E1327333E63), UI64($F3AE2031F5D13B28) ), { C8 }
    ( UI64($246C7CABB2D9A55C), UI64($50E9C7282C1EE0F6), UI64($2FBDA09565A0D3D7), UI64($196552679C04A4EB), UI64($137C66DA29A6DD82), UI64($08A76B6B4BDA56BF) ), { C9 }
    ( UI64($7CA3C59BE3E28610), UI64($6ADD75CF1F7AE248), UI64($01747450737A6435), UI64($A1F2259CB2B4923B), UI64($E0C8F55E8ECE7210), UI64($D7964398F350B69B) ), { CA }
    ( UI64($E045864ED1825101), UI64($AC54969193E1A1C5), UI64($23D85A934D0794C7), UI64($B4FA88CB734A4213), UI64($7C5CBFD6BDA3D5F9), UI64($66607FE938748825) ), { CB }
    ( UI64($BAF36FD2A180D481), UI64($EAC440AC1B9598F7), UI64($9AA24D80FFB7B06C), UI64($79601F517358F163), UI64($D1071831418BB63B), UI64($819609A6AE7D3A03) ), { CC }
    ( UI64($3E9152D8CDBAE551), UI64($86AD793F203DD016), UI64($BE3AEB778AD4A891), UI64($2810254DD76B6618), UI64($9B5DCDE36636C327), UI64($0A8AAD65868BC58C) ), { CD }
    ( UI64($6D0672780D93152A), UI64($EEE705247B828091), UI64($9EBDB976F137463F), UI64($A7DE3E73A2D0C1BF), UI64($F871A00BA0046AC7), UI64($484C96A803F23486) ), { CE }
    ( UI64($0FC7BCDABB06BFFB), UI64($F75C3FFB3D6309B3), UI64($ECA305D103109162), UI64($373F503B204FFF61), UI64($CE332C9F54963FA2), UI64($9A4420A52242CDB4) ), { CF }
    ( UI64($C71D481179D198C1), UI64($505A2845CEE92569), UI64($F339BFF6DD6755B5), UI64($8BEAD52B8DE89245), UI64($4B686E65920DCA2B), UI64($99593FA43EE68A37) ), { D0 }
    ( UI64($D90A68D717E61501), UI64($9BB920AEA19161A6), UI64($2F3D6F96D90EB1E4), UI64($DF15ECBA10513D7D), UI64($E6E5D539B4F01831), UI64($C7D17A7528FECE36) ), { D1 }
    ( UI64($A04FF0BEB4EBFBAF), UI64($E5E90A5B3DDAA3CA), UI64($8453542209F4A145), UI64($80A6FFD72BB5A707), UI64($14E0C4705A1ABF6A), UI64($D699EC1FC18A677D) ), { D2 }
    ( UI64($7021A124E3181575), UI64($DC7AAE2817AD945F), UI64($8BB5521E7F0D565A), UI64($6671D3792F0805EE), UI64($D3888EA394413A1A), UI64($CE4D7E47B55BF9CC) ), { D3 }
    ( UI64($22F440263CAADE68), UI64($E77BB287772EAC7B), UI64($29493775962A40E9), UI64($1E06A27FA68CB91B), UI64($DDEF02932ABDB9C7), UI64($79F03B88DC175233) ), { D4 }
    ( UI64($65F6D517B53E2391), UI64($97DB65A2F00B1C39), UI64($1D77AE9B85AA4855), UI64($19133B9B3E9B0771), UI64($6376D9F11A7DB3D4), UI64($949AD02F5AE16184) ), { D5 }
    ( UI64($FE4434CDE09D923B), UI64($03B0FCFD713B7052), UI64($2D713290D4A67238), UI64($2B56946FF629EE96), UI64($60A15D01B2B3C428), UI64($0B1D5EAF793933A0) ), { D6 }
    ( UI64($BC40FCFB0E0D494B), UI64($A31C4648C7B3D1DE), UI64($F1113C219A07EC8D), UI64($2378BEB1A5C2BD1C), UI64($190CC3478070A194), UI64($63DAB6E1CCF56329) ), { D7 }
    ( UI64($901B6B9E82BABF91), UI64($872A234C45D61001), UI64($6CA46A95C1CC6D6C), UI64($22779315E0F02295), UI64($60A59396346BE6AC), UI64($FB67A503CB488846) ), { D8 }
    ( UI64($50D440F74C97660B), UI64($E71ECABF64EDFE0C), UI64($80201B895718CE22), UI64($A05D89804D35D306), UI64($8F700402A2B0D086), UI64($326FCB334CA4DFC0) ), { D9 }
    ( UI64($BCFBD02EA005CDD5), UI64($F0225A4675553115), UI64($08E18B3692A7AF62), UI64($05D34A820C8CED0A), UI64($51A8D7CEC33E80EA), UI64($0AC007503FAE879C) ), { DA }
    ( UI64($F43EEFB5C83C521A), UI64($E5E9B05FC48841AC), UI64($79C52C38BF85B5F9), UI64($26CD0818AE3BF7A9), UI64($4F385C32CA8F5F74), UI64($F17B22107B954752) ), { DB }
    ( UI64($1A48FC969198A4B0), UI64($D9A78940BB0C4E1C), UI64($42781D9BE60E7691), UI64($87D1CAF3680F8A30), UI64($D09FF193606AAF29), UI64($4518DABC60048793) ), { DC }
    ( UI64($F05D48134A56A034), UI64($89A65EEB91DC69B9), UI64($8FC7F43960E63C62), UI64($FA1C6B9FF9415E92), UI64($7E219D4E56347935), UI64($2B6A48D6DE0AEF85) ), { DD }
    ( UI64($1A7FF9C54B045FFD), UI64($44A0A9562E9468B2), UI64($F11425A22D1EBF92), UI64($208D33120BD28E0E), UI64($F2D74197AF80E162), UI64($CEEDCA73DFE66C93) ), { DE }
    ( UI64($D57190439D29C9A4), UI64($44C007DC2B5EAF9D), UI64($EF6DDF48A780CEDC), UI64($61B205E4A96024B1), UI64($1885B6CE84C3FE5D), UI64($B8B56986B6E2CE21) ), { DF }
    ( UI64($F36DACFA34237E99), UI64($BE45EB5253BCFED0), UI64($402C6946B8B21AC0), UI64($2460A6FCE7E9CD67), UI64($F89A6D5B162629FC), UI64($F66CCEA374DB821E) ), { E0 }
    ( UI64($16E06074DCC31A1D), UI64($F172017AC3FA38C3), UI64($BBC1CE4BB784ED60), UI64($DA89A8BCE82AE671), UI64($A6DACFFB8D26C0BB), UI64($185181AE9609F6D6) ), { E1 }
    ( UI64($F110DBDD94D17661), UI64($F59FBB4CBA69F393), UI64($463B60FB3F3C5E00), UI64($1C60B896FE8E78AC), UI64($5EB3E26795DE5AB6), UI64($997328D4654D6219) ), { E2 }
    ( UI64($21069118ABE24B61), UI64($811CB8C48FCEFC6A), UI64($483B032CFB56F902), UI64($FB32E848198CC057), UI64($A620815462A04F70), UI64($900038D1894959E2) ), { E3 }
    ( UI64($5AD509789BFFECD0), UI64($DDCD5E8325F69CA0), UI64($154D8F1ACD9B8C82), UI64($AC7DF75E94CE3CAF), UI64($6D6554D1B38754BE), UI64($B5DB64AF738486E7) ), { E4 }
    ( UI64($35A308A1AC9A43BF), UI64($2647805AB3E6E492), UI64($4BB74A616F61588F), UI64($FA4602EE5BDBF54E), UI64($3FDD62470A7174DB), UI64($5795433CA808FAAC) ), { E5 }
    ( UI64($51A094B8774CA605), UI64($5F07974C74EEF225), UI64($022AFEF7AD81A953), UI64($0967C44BBA336FD6), UI64($8AA327918AECBA3D), UI64($F70B8436573C3F0A) ), { E6 }
    ( UI64($CF374F83420766C3), UI64($71F31901A13EF07C), UI64($63AD56C7DEF9DC0F), UI64($9E5BB5E859F5A231), UI64($D0BF453BB9893E4C), UI64($A1E14B66C2719760) ), { E7 }
    ( UI64($B41861CC73FD3E48), UI64($461D79A138B04BE1), UI64($4010D37D37FBA817), UI64($7D9622AA693225A4), UI64($2204454B8126799A), UI64($33A5D487DCCD6EB6) ), { E8 }
    ( UI64($D291D0317A053320), UI64($E27678F1E50D1F76), UI64($9A3D663A63159FC7), UI64($AD7B4D3F67BAB452), UI64($269CC05E2B33CE1C), UI64($0FB8261CD734BCC3) ), { E9 }
    ( UI64($F3D0546D3D4A25EE), UI64($B42874AD28C9B7F2), UI64($73EC788B29962D28), UI64($4AE73A48132B8553), UI64($756C99D7A0910B66), UI64($ECA7E2C2712D555C) ), { EA }
    ( UI64($559FA5BF24911FDD), UI64($A1DDF5DE3770554B), UI64($C7C3FD139366B946), UI64($6E7ECC0C881D2BA4), UI64($14E76D6A27E54B87), UI64($7352D5FBC4FAB878) ), { EB }
    ( UI64($F19A622BED8DAC0A), UI64($35548E5D7EFC5A2E), UI64($CAC84974B4F057B2), UI64($AB317ED03D0335AE), UI64($710FC138F2C51738), UI64($9C90CC495A403416) ), { EC }
    ( UI64($9FA7DEB936F10461), UI64($A1529B0B58462F9D), UI64($9F109111C8B9EC65), UI64($23A3EB28444E33EA), UI64($554084CA75118937), UI64($599D58A7C946EAC2) ), { ED }
    ( UI64($6EC3AABB7856AC4E), UI64($980E6907C1CBCCAF), UI64($1F8557ADC700CBF5), UI64($7DCB1CE0AF48D9F4), UI64($7FB3DADF8199AB8A), UI64($E6B36DB8FADBF312) ), { EE }
    ( UI64($C00F0D3F7A101660), UI64($605B94B12DB6C697), UI64($79944F7BA2B65F38), UI64($40858ADEDD47E2BC), UI64($1E044BDB0E9FB02B), UI64($86C79D01A3109539) ), { EF }
    ( UI64($9731893D5B98482A), UI64($FB8DE267F9790326), UI64($8780F407143A505D), UI64($A41CAEFCCCD3A8E3), UI64($A042F0B3D7B7A7FE), UI64($3E3151FEBB19A1AC) ), { F0 }
    ( UI64($E7EDF679003A6950), UI64($BAFC97D4A8C6AB12), UI64($13C096B49C79559A), UI64($C3052501434B5019), UI64($1280FB23E7ADFB09), UI64($1959905D31BD2FC0) ), { F1 }
    ( UI64($575C0C46FCFCC65B), UI64($FE625E873F34B419), UI64($1696FDCC7F51B8A3), UI64($C79C56F30E5AE7C0), UI64($14E3461CD27FAD15), UI64($1B7BCCB9CB472859) ), { F2 }
    ( UI64($3806FE58E5CC8F16), UI64($F8244ED76734C1BF), UI64($4E04940E0F5DDB56), UI64($5BD0AFDDC4158B7B), UI64($A4C6BA949911C5C9), UI64($FF6E2AC155AE9726) ), { F3 }
    ( UI64($49C7C844B8114144), UI64($B450E41BCA35CB00), UI64($302450EC67BEF97C), UI64($A8662049DB1E0D8B), UI64($DA69C022528EB8FA), UI64($6ABBF16585C1A2F7) ), { F4 }
    ( UI64($37BB420DF67F044E), UI64($DCC0E9F3E2EF07B3), UI64($4D10088618777841), UI64($0492E5379305DAAE), UI64($3DA4791C37E4128F), UI64($80688445CBA4EA17) ), { F5 }
    ( UI64($51398A7CE4CF8D9D), UI64($49A5FCD891A69CA5), UI64($3D72A60EC2392DA5), UI64($0E8296B879AB5539), UI64($6BCB00AF2EDC0BDE), UI64($BEB93848E54B3E90) ), { F6 }
    ( UI64($7AD7C52A18922E19), UI64($29292C57C4F5B8F5), UI64($F0CF1F98A577C10B), UI64($072B9F293BB660CD), UI64($09B8604F5575B6FB), UI64($DECB396A81B9FCDB) ), { F7 }
    ( UI64($254AD7ADB4C220DE), UI64($6C62E20F95A0070D), UI64($ADEB89F339309BD8), UI64($A2F685CC178B289F), UI64($9343905B5DEE95A5), UI64($E0C30F34A2977C86) ), { F8 }
    ( UI64($669CD51AF7CFBFAA), UI64($E3E0806F6880271D), UI64($6934C259E098BF90), UI64($5DFEEAF0FBCA7249), UI64($89F74B948B4118B6), UI64($53640AEAFB6807C3) ), { F9 }
    ( UI64($DD3BACDCC04BE120), UI64($6D4949BD64198E51), UI64($31FDB39666598A74), UI64($BBBC6DE9C0C15A81), UI64($F27F201C61C06279), UI64($2738AFE3E84E5CDD) ), { FA }
    ( UI64($CDD71FD35A6411DE), UI64($3CC012793E87523F), UI64($B0CFF8720FCA36F3), UI64($93E85FE07300F012), UI64($E894A085263F090B), UI64($2DF60A01DAFA90EC) ), { FB }
    ( UI64($9DA50DB1EEB4FADD), UI64($E524E49C9974799A), UI64($DE09FFF26A24CBB9), UI64($AF9D71E9F3ACE7CD), UI64($EB62B1A62566EC9D), UI64($06D02AB1217D3553) ), { FC }
    ( UI64($DD31E6391AE03522), UI64($93ACD1065B35E915), UI64($F4EB56CC03E79218), UI64($0717815C850C97F1), UI64($BF4F6A8AC0540A6F), UI64($FCF8AE5DE9507FF0) ), { FD }
    ( UI64($AB45B413DC50B207), UI64($40B417369551D8D5), UI64($CA32286A108E7210), UI64($03225E54D8D093AF), UI64($4B6CA5591EA576E9), UI64($4E12AB774DC4E062) ), { FE }
    ( UI64($D9F4F850DF6CB96C), UI64($8ABAD81B1667335D), UI64($CB4079CFE79C72E5), UI64($E5542F763E316996), UI64($303E4B79B9D397C4), UI64($E46933038B945111) ), { FF }
    ( UI64($75B15CC53B0D2502), UI64($DA1BCA6BA0524358), UI64($9EDA977556C06B7E), UI64($6C57727ECF0A1325), UI64($DC613D5A78E5C3F8), UI64($CE062D94A3B4945A) )  { ZZ }
  );

  MHASH384_ADD: array[0..256] of TRowUI64 =
  (
    ( UI64($7D8058C289965A68), UI64($A40E3AC93F6EEC96), UI64($42962AB56A64DA88), UI64($FB6391D25934AA91), UI64($D9958F7E87AFCDCC), UI64($988314D182C9F6F8) ), { 00 }
    ( UI64($1C460224D6627D46), UI64($0C8338ED6877A302), UI64($099686848A2D1E6B), UI64($43B36D215D7BFB49), UI64($B38786F894DB7C04), UI64($6F4638AAE0B68B90) ), { 01 }
    ( UI64($D673B0582AC721CE), UI64($07D11CCDBA43618B), UI64($D68F9B3F987F9C39), UI64($1BC69CA5367DC351), UI64($19315FB5231457D5), UI64($BC49FF8023C4FF51) ), { 02 }
    ( UI64($6179643A91A32DE4), UI64($18872E1DEF8D3303), UI64($7FBDD8315412DA52), UI64($F1A8B5CAB53FA8AB), UI64($801DE3F48AC23AC6), UI64($AF7AE49C8161AA48) ), { 03 }
    ( UI64($0C9B6A8C119BB0BB), UI64($0DE55AE85D1E20E1), UI64($48B7435DC9CB6B41), UI64($A3A6E2E44FEF6841), UI64($F91669E868B80060), UI64($214BCF908258CACB) ), { 04 }
    ( UI64($4594391DE2947DB2), UI64($3E6571CC6E9FC2BC), UI64($64067ABBC8223821), UI64($BEB4F1834702E047), UI64($B4F106D9A146BE05), UI64($9B9DE8FE614B6843) ), { 05 }
    ( UI64($09375BD6DC4C374E), UI64($8CB31691EB60E267), UI64($A915B81DF0365544), UI64($9F047AB054483B51), UI64($D01FC79B688E9C7E), UI64($CCB8A7DE2359118B) ), { 06 }
    ( UI64($F7270D40BCECDDB0), UI64($C4F73BC45FBD7E70), UI64($D70FE5EF89507E10), UI64($E3A64DADD16A7BC9), UI64($CE016154B38CF9D6), UI64($A606A6C6DFD85353) ), { 07 }
    ( UI64($F8B96EE488E00CA4), UI64($068955EEAEF58101), UI64($4BAB59CF6A61C1CC), UI64($3FE0542F5EE9AA03), UI64($F6908622388193FC), UI64($B78BA6EEF47774C5) ), { 08 }
    ( UI64($798F73B22108ADAE), UI64($37D72947BB130490), UI64($A43D1E4C39ACFC00), UI64($CEB11D35414762B1), UI64($E88D1CF560020AA4), UI64($E20322F9157491C8) ), { 09 }
    ( UI64($DA615FCBB078E1BE), UI64($28F309E1EC35AC02), UI64($64B75ABB1D84D8AE), UI64($3B9ADA205AF05E6E), UI64($0D6B8EE88FCBC8EF), UI64($AF8AAD011769C943) ), { 0A }
    ( UI64($1251C038180201DC), UI64($9A6305C757BFFE37), UI64($DB962943CD1C1956), UI64($8A94C2B95A569A74), UI64($FBB4EA7CA016049D), UI64($F303048F4477F00A) ), { 0B }
    ( UI64($16372888CB6911E6), UI64($30C579D5ACBC6F67), UI64($C41B20D00BEA9005), UI64($881990C9DA4D0240), UI64($0599214F8DABB860), UI64($E449620AE3A535C8) ), { 0C }
    ( UI64($116913EAF90B1D84), UI64($5E4E196134B1E8E1), UI64($4B8F43212575ED59), UI64($9C7698B6CD7EC34F), UI64($94C12A54396BF23E), UI64($E9FF19D802D11909) ), { 0D }
    ( UI64($8C6BA0E85A04768A), UI64($8D37B3F0B6550290), UI64($C7C2087D733E9C5D), UI64($A7B932B8E1E9F298), UI64($385122C401E9CCC7), UI64($B3EB819FF727425A) ), { 0E }
    ( UI64($5F75DBFB1AB09D6C), UI64($B837083CAC1D4197), UI64($D9AF89142B757260), UI64($AC1F71E71DE3FE48), UI64($71F71A44098C5790), UI64($0293E4022D18395E) ), { 0F }
    ( UI64($C8FB61B088003D36), UI64($3CFE1604B5077F1C), UI64($4D2C22DD6706BBC9), UI64($39D056A9BB307D41), UI64($2D86532AA15F3A94), UI64($A4AF1FA82ACC017A) ), { 10 }
    ( UI64($74697EA12B79DD5C), UI64($A6265A2F060D0937), UI64($ED8EFE88EEA85D70), UI64($9FA37CA8284E09D5), UI64($1D1D438E3048C097), UI64($A519883362A1DAE2) ), { 11 }
    ( UI64($68560840BAAAE48A), UI64($D0381C27B707F307), UI64($3A8F4E24DB22F809), UI64($ACF5526D582F1A64), UI64($C07CE4E335EC904D), UI64($E2FBF875C3DC4970) ), { 12 }
    ( UI64($52017982B27469EF), UI64($C67A9FC8A2F076A4), UI64($F9D7147CE22CE931), UI64($2DC153AF477EB32F), UI64($E1396B65889A0890), UI64($874E5D7AAEA92AC4) ), { 13 }
    ( UI64($E617BEE1CC4CBD99), UI64($ACED2BABBDCA6939), UI64($638550E76297B871), UI64($AAF03AF65E178F21), UI64($5B7D12FD81CAB450), UI64($6772ACA9BCE1D19D) ), { 14 }
    ( UI64($5E71B2B341EFEB7A), UI64($1EAC0B05DE6FEE0A), UI64($5886F65EDB18C054), UI64($8B575466C729B380), UI64($CE330220886A7A0A), UI64($28DFF2FF1786D02F) ), { 15 }
    ( UI64($802724C7B2706DEF), UI64($33FE52147D2AE900), UI64($C8EEAA5EA3B7BA1C), UI64($BBD47F205D8F423E), UI64($B2D0CE5F3A08AF46), UI64($AD535615CC83F30A) ), { 16 }
    ( UI64($C9B37BD962AB7542), UI64($173D188F8CAE9D63), UI64($F3667A498076FD60), UI64($08E774F0F7E192E4), UI64($AF4FF77803A16CB8), UI64($AFE70CD9CDF327D6) ), { 17 }
    ( UI64($D3A19143201E15AA), UI64($1025BCAEB5554A50), UI64($59B8767FDCAF8440), UI64($92453353C9FB884F), UI64($F32742ED3401E9D0), UI64($8C382E0E82B5C522) ), { 18 }
    ( UI64($CF0B5908B90B7C7A), UI64($DA71C53CAE386706), UI64($4F91D6EDCA05CB68), UI64($8687E6B8DF5CC6F3), UI64($A3880F9B1BE4E7C2), UI64($B03F60E1B5E4910E) ), { 19 }
    ( UI64($F4C3B389781069C6), UI64($F54F3E15FCC2A228), UI64($F97B887F280435C0), UI64($A583DFEC4E50A8C0), UI64($9675E5B417EDC206), UI64($618F6826807DF5AF) ), { 1A }
    ( UI64($83466C0D2D22615A), UI64($8B4520E19E06F207), UI64($4D8B659E2502E034), UI64($9D653DA04BE0E9A9), UI64($075712F7BA04C6A8), UI64($83E2676A06D50698) ), { 1B }
    ( UI64($FC6533CCEFB8F372), UI64($00D61064FF94F011), UI64($09893AF7FA325361), UI64($88C2BD50BB518B5F), UI64($9868CB65E9B2209C), UI64($F327D227B7128E8E) ), { 1C }
    ( UI64($7D5F0333C85EBA92), UI64($B2A44BC9CE37F22B), UI64($D6B71F1A308F64C9), UI64($A9A03A8A9FC98E19), UI64($B18BEB659B148F96), UI64($E7FF1073C89DD552) ), { 1D }
    ( UI64($53C442C7718B75BC), UI64($E8B44AC1E2426A18), UI64($5A0FBAC96A720DD7), UI64($5AE971ED50F5C272), UI64($BBDB6BC682C8F3A4), UI64($E13CB728C55C6096) ), { 1E }
    ( UI64($5ABD70EB9EC9F62D), UI64($B0EF3DA118B44783), UI64($DA82EBAAEC869682), UI64($8AA41D684388D533), UI64($9235637110AD1E04), UI64($BB2F8B40082C0B8C) ), { 1F }
    ( UI64($96837B867A03E5E8), UI64($8C2669AB805671C7), UI64($F32732DA537197AE), UI64($95908762F114EB00), UI64($FCA7F60DBF082AD7), UI64($A37EC2E780D25948) ), { 20 }
    ( UI64($7F21F2E004AAD61A), UI64($D86636C5909C7B39), UI64($513E16EEA170FADF), UI64($B725BC2B77C7EB60), UI64($B17C8D89AC011FEE), UI64($CDDF2930B7EFBF0A) ), { 21 }
    ( UI64($CA1F338EB96F5BC1), UI64($062D44E49A20AE12), UI64($156DBB2E7B71BE10), UI64($A038F0F94BE68990), UI64($23E68370AE685CD5), UI64($C63A5AB27069E983) ), { 22 }
    ( UI64($8FF4583A6399A1A7), UI64($A73A26BD26F67E05), UI64($5D91489A52F5D590), UI64($B3A277B55047D949), UI64($AE2289743521BC08), UI64($75DBBB30F5C1901A) ), { 23 }
    ( UI64($47E13042E97AFE13), UI64($2A9F408D94BEA397), UI64($6FCF5E491356CED8), UI64($75D023044F645350), UI64($B82D60DDE48F54C0), UI64($7E0B676648313427) ), { 24 }
    ( UI64($64858BB42AA007E0), UI64($804C6B350C627021), UI64($5C1FE2FD591EF9CC), UI64($4666721802EE9743), UI64($112344F4D05B4DE4), UI64($AF2AC0716C423D1A) ), { 25 }
    ( UI64($5A81C32177CE6FD4), UI64($00E819A62C377A64), UI64($BDE22A6B206AF003), UI64($E7F82E145905C381), UI64($23178BA6E9AED6FA), UI64($B1B96C84A900BA62) ), { 26 }
    ( UI64($A7230393D376BE98), UI64($47CF7E2020D6F126), UI64($57A2F211BC4C101B), UI64($8EEFF7F313A55B71), UI64($A15852603D83808F), UI64($A40F66BE1468719A) ), { 27 }
    ( UI64($93AC6C6932A1F9E7), UI64($910E19918E175673), UI64($58AF127D1ADE8A7D), UI64($7CBA0FF053C6AD51), UI64($F2DF666AA9D6C55E), UI64($7C3FBAFF36EDB3A3) ), { 28 }
    ( UI64($49B12AC66082C518), UI64($BD27D5DCCEC17332), UI64($DC951AC23D021869), UI64($FB36C7F64C64078B), UI64($62590331579B46BD), UI64($E45B1A06CE802308) ), { 29 }
    ( UI64($B2C3604AAAB01949), UI64($B6A32902676CA87A), UI64($8BA1A6D8983AEAC9), UI64($3EF5B1EC87C5C9C8), UI64($632B9B91591B524C), UI64($F7AEAFD5E1ED580A) ), { 2A }
    ( UI64($D0382A8380B2F1FB), UI64($9FA01FB52D14CF12), UI64($2224B9299058B751), UI64($1F60EB188D54FB18), UI64($56E028FE148AD6A0), UI64($A76DCC72B754B578) ), { 2B }
    ( UI64($1B13108B02AA6909), UI64($9D3D1E296EDECC6D), UI64($25AB0A87FCE068C1), UI64($45912228513A5748), UI64($16A14185AA8327D6), UI64($B42392BB4D45EF89) ), { 2C }
    ( UI64($A1D77344920FF03D), UI64($0ED43DA5A7BBF945), UI64($44CC6251F9E4F646), UI64($59B553F306C48B8A), UI64($703D2DB69A7DCF92), UI64($278DB0A673B1EB1E) ), { 2D }
    ( UI64($8E032861E1FA89F0), UI64($513B13B12F22AB11), UI64($710368583D2CD42A), UI64($B284CE6D253EEE93), UI64($794D2B43840137F4), UI64($6BA5F09B20F8462F) ), { 2E }
    ( UI64($15436111CA65657C), UI64($71A054BD169B4953), UI64($62049C6C7B5471C3), UI64($AB2A2B215A674CE8), UI64($D1221A351BAA696C), UI64($93BE1BFE0A614603) ), { 2F }
    ( UI64($32C733C5A828F00B), UI64($110F399801ACEE05), UI64($9F14CC0840853875), UI64($6E88F1A7627FAE58), UI64($B8060330D90DB2FB), UI64($A8B9D6A99646BE60) ), { 30 }
    ( UI64($CA15A83D330E9FC0), UI64($30B934C33F0CB130), UI64($17069ED0BFF4C5F5), UI64($3F52FBF392287423), UI64($AAB3CDD4D8CDA382), UI64($231B0AC292491FE0) ), { 31 }
    ( UI64($F02AB9566B53A895), UI64($133D20A7CF1CEE15), UI64($0F4A39A36A1470E5), UI64($533E5E3D6DD5736B), UI64($F31D7A2D49D168C6), UI64($2DDA4BD3C1CC68C8) ), { 32 }
    ( UI64($172D6D06FABCB264), UI64($6FA29A330E363099), UI64($320D036B9C027050), UI64($F751B0B15E9D1625), UI64($3EE5273819657ADE), UI64($C212652AABC6E78A) ), { 33 }
    ( UI64($FEE1424BFE3029F1), UI64($29B4328B932813C6), UI64($40DF970FE818B040), UI64($92302AAEF34DAFF6), UI64($AAE05776B89BAEAB), UI64($40C9889C60440D54) ), { 34 }
    ( UI64($5E0528DD38A964B2), UI64($3F280772BBA6F7F8), UI64($E59BC201AA3C18F6), UI64($4B3037833BF58159), UI64($5386B6855EA3D81E), UI64($072B41D496F8D86F) ), { 35 }
    ( UI64($D10F321C58883051), UI64($F5122AB53CBDACA0), UI64($70E5F26CEF85C03D), UI64($DF85840F21010126), UI64($FB52964C8A9E5098), UI64($D57BE9A2C1511532) ), { 36 }
    ( UI64($D4F34B50F922A024), UI64($6BA9FBE07C185482), UI64($B90C7E15FEBF7AF7), UI64($A806293C6065CF86), UI64($A4158143B26BD592), UI64($3EF34DF685EEF5CA) ), { 37 }
    ( UI64($2BD94AF4722563D0), UI64($7CFD11ACF73C3B20), UI64($C0B1C431101154B1), UI64($F785193842C24110), UI64($159CDA6C108563D5), UI64($4866B3F88380E5E6) ), { 38 }
    ( UI64($C5C77CE05C3FA920), UI64($006559F76A049F3E), UI64($EA62873BEC378339), UI64($2167933D052D9E18), UI64($B804358CF8C04D3C), UI64($E49A22281F41480C) ), { 39 }
    ( UI64($DE977AA346C13C95), UI64($094636D6A712F836), UI64($CE45CF3E5D3B7E21), UI64($1C83990378ACE264), UI64($DE89C2BADCE171C8), UI64($22FF4D935CA9850A) ), { 3A }
    ( UI64($43413AA1C370E983), UI64($8006CCE02D7428C7), UI64($E63F2069FE1313D3), UI64($F8B67CBF92C9D3E3), UI64($CABE8ADD2235777E), UI64($048050F68A678178) ), { 3B }
    ( UI64($A1F1021BA5846050), UI64($AC7D55A11A3DE4D1), UI64($EB46205610DE2660), UI64($A78B17253DF78FE7), UI64($AC8A42CED4C45B4E), UI64($C619F830D2CF4D8C) ), { 3C }
    ( UI64($D1D45E4330F5FFD8), UI64($12F368B42E99E598), UI64($95334607C1DED3C9), UI64($D6C77ACA7AF16231), UI64($C8A72710A6431032), UI64($C5FA41BF6285269C) ), { 3D }
    ( UI64($6FE8C2B7B32AFCD1), UI64($33266CF1BC266852), UI64($B7DEE777F460F16D), UI64($AD0697A9D35E2EDE), UI64($D4172EBBA3137515), UI64($772ABF88FD0054CC) ), { 3E }
    ( UI64($55396011142CBCC3), UI64($075DE9A98D29F336), UI64($AC8B2EEBE8770D49), UI64($1B0D817B62517938), UI64($4C4BAB763BCB33CE), UI64($0B137A31AC17581B) ), { 3F }
    ( UI64($94F12CDE96BA6BD2), UI64($FC2CB9FFB139E6A6), UI64($6D2731DCBFE85D63), UI64($71E42CB34ABF4AD6), UI64($99BF7270DB61BE4C), UI64($F4BA0BA31D5CB61E) ), { 40 }
    ( UI64($3B65FB61284C73B4), UI64($5F89D8808763AFAC), UI64($7A0F8167D9C69BB0), UI64($BD27B760A461AAAC), UI64($B1BF104F7E03B6CC), UI64($316A189375080992) ), { 41 }
    ( UI64($DE26A14218B11558), UI64($0F6E78EBAE16DD20), UI64($003DC052B7062183), UI64($97AE533E9E5CCAB3), UI64($BC2FC2D5ABEE5B09), UI64($7C219189F8AC3AE3) ), { 42 }
    ( UI64($48347AF0DAF519BB), UI64($7520A9A5A488E209), UI64($CA7532CF3848F90E), UI64($85AEA3F53AFBF19D), UI64($8881E60DFC45E690), UI64($3A613D6453655BAA) ), { 43 }
    ( UI64($571F98C7B9E7196C), UI64($25432582F55597A5), UI64($A0D7245B73425C59), UI64($5F162BAC31FF98E0), UI64($92D608788155ABA8), UI64($B53F6A3399A2CDDF) ), { 44 }
    ( UI64($526EA8A7A86E5A58), UI64($5FFB5C59E44CC4E0), UI64($6DCD32D24479FB01), UI64($78A22F7B2249CA31), UI64($7A436F0410E41C4D), UI64($0B8AA84C46C2A54A) ), { 45 }
    ( UI64($E72BEBF02D8188F3), UI64($187E39797296CAFD), UI64($7F450B19BB428C69), UI64($7B8773B2D8871BBA), UI64($A9F38F2B9429101F), UI64($0B6A62D2CB268DB5) ), { 46 }
    ( UI64($ACE3B97A6374FF68), UI64($2D71557589348A28), UI64($1AF9423B6A208BD5), UI64($4D844C4352D67AD6), UI64($FB0130E2D32E100A), UI64($827C16B0E0E15F58) ), { 47 }
    ( UI64($78F0B8653B0D18FD), UI64($1D7D0B7A15D5E306), UI64($B61B9A89DFA6220F), UI64($A5B22E9507D0CE70), UI64($EFCE8136B8094C5D), UI64($8D56DD3E8320571E) ), { 48 }
    ( UI64($40B298D243D432E6), UI64($D6B750A19212B772), UI64($6F9D873AC1A0B5FD), UI64($E8AC406409A6872F), UI64($9A469696B9C1FA72), UI64($93848EA38124B42B) ), { 49 }
    ( UI64($E4C1448EC8CC58B8), UI64($76CB35652609202F), UI64($9494B86A062621F7), UI64($60156A21E1D1A861), UI64($A0EF7E66BCA14C10), UI64($686BD6D21D825969) ), { 4A }
    ( UI64($29BECA487442E3F1), UI64($705663E33AA12922), UI64($3CDD921E4C7679A0), UI64($9371E7BB0FF9D9FC), UI64($3209288C29283298), UI64($BC97E7FF1A8EFDF9) ), { 4B }
    ( UI64($49C0E9D5EF0DC498), UI64($20AE20504A3B8535), UI64($ABB1D036D70CD941), UI64($149D24EC9D61E114), UI64($BEF1E86CBA595DE8), UI64($AF0D28A0AF5DFBB9) ), { 4C }
    ( UI64($9A3771E70EBF65C5), UI64($9DD4C2D127A8039C), UI64($FF1A7E1393C364EB), UI64($EB856EFC41DB2ACD), UI64($3DBAB2E4AC7A4C88), UI64($BA8C2CE090C611A9) ), { 4D }
    ( UI64($D583F464FC44A171), UI64($1DBF5431C1A2CAAF), UI64($57B3775B16E8D883), UI64($AC61D57DB8CA91C5), UI64($DACF0AD601CACD66), UI64($CE5B41E29D0DE6D2) ), { 4E }
    ( UI64($994FC88062273D20), UI64($198F252AE17F369B), UI64($707653C6AEF6EF26), UI64($FA8CEE9559C0A3B8), UI64($983848F20A8BF25D), UI64($868FDC206A0854A0) ), { 4F }
    ( UI64($63531AC9C4359CFE), UI64($5C22759DCD60D0B6), UI64($954D6B25FDC9C3BB), UI64($32993504E3D5530C), UI64($B39BC4E581285E4A), UI64($A113930CBCC476FB) ), { 50 }
    ( UI64($4AFFC564E9745EA1), UI64($CD50F2932DBEE0F7), UI64($1032984598329F15), UI64($9941126E11F6D9DB), UI64($ADFD922183C06E74), UI64($A19B6D2544808965) ), { 51 }
    ( UI64($9EB31084EF95FEF1), UI64($729B4DE7CD5BFA76), UI64($9E4F5B940206737E), UI64($67058AE2316A80F2), UI64($51B21B8530E99B98), UI64($AED28138B275A1D3) ), { 52 }
    ( UI64($7723C4C7401A2B37), UI64($54E058559B3AF793), UI64($F0A9EC517E50E5A5), UI64($429DA5B648A2BB41), UI64($288039FC335DF40D), UI64($AFF3C95FF1270B84) ), { 53 }
    ( UI64($4D66B0327425E5BD), UI64($150B198A06ECB238), UI64($2DCDAF4C35CE47C5), UI64($42A1AAFA9B86CA17), UI64($D5A416CA8BA5B7A4), UI64($AC4BE5C6A3CFC804) ), { 54 }
    ( UI64($955E6ED869A96EFD), UI64($B85B562A0D3C2356), UI64($CA5D2256C80980DB), UI64($D9D6BF37E9EA2739), UI64($FD6589B4A1632DAE), UI64($AEE8A86655397AD3) ), { 55 }
    ( UI64($DD138357ABC5D9A4), UI64($58253FBCE2AF893B), UI64($EFB96A2CBB3A4995), UI64($AB059E2A6B24EB2B), UI64($7FCF6DDC799F2E0A), UI64($7490FF607212E7FA) ), { 56 }
    ( UI64($D47753D03F6A7CB2), UI64($EEBD589BB4B81439), UI64($9E7F083C1E48D465), UI64($99F4AC9B24B4D403), UI64($CA1E18B7341A0C6A), UI64($D927D48AFA8D48AA) ), { 57 }
    ( UI64($CAC4E3E4097460F2), UI64($59A31F4BFC89D1A1), UI64($7E9027F8FAE125F1), UI64($643EB68156F413D8), UI64($6C64525F84C83A82), UI64($AD97A09548999292) ), { 58 }
    ( UI64($B98BB614D1B97B3D), UI64($686559913959D004), UI64($7E1D4EA292F75119), UI64($987907A95DCCAAD2), UI64($0367434C0605031A), UI64($0B4E5DC73D450C6B) ), { 59 }
    ( UI64($9DC90461E4C01281), UI64($5D624A0BBAC1060E), UI64($40003A12F9D17B5C), UI64($BDC72495EB56F20A), UI64($52164EC4FFADA430), UI64($AF1660B1EBAD29C6) ), { 5A }
    ( UI64($CF2631F3A00975D1), UI64($C457D947C598615A), UI64($939363C9FC07C9B8), UI64($31712E2149678C18), UI64($BE33C87A4CB7DB2B), UI64($5351E4BF294CEB48) ), { 5B }
    ( UI64($884B290301CF3A77), UI64($687C20EBC40B8084), UI64($9DA5834D701450FA), UI64($9904B5DF529ED2C5), UI64($36B6B441548AD976), UI64($EC0FAC65F72202D8) ), { 5C }
    ( UI64($49636EC515319667), UI64($7E3723B6C9F2BC0A), UI64($DC9830A275CEAD6B), UI64($A0C03C2B5545CFC1), UI64($79206E9438CEF1EC), UI64($323A48D2418BCB7D) ), { 5D }
    ( UI64($D5943E9A792516E7), UI64($5E64D8C5140F9BE1), UI64($1C907C1C7CA4DA05), UI64($FF82AE6927DBDFC9), UI64($3A78FA37EF8D734E), UI64($E6EE0491BE7460F5) ), { 5E }
    ( UI64($58CA1A4E84332E63), UI64($3AFD9E300CA530A5), UI64($D3C19661390E5D17), UI64($0BA539299074A7EA), UI64($3CD74C3AED20A418), UI64($8B50CA03F2ED504B) ), { 5F }
    ( UI64($A003D052EDA9F9FE), UI64($B14048FBCBB623A0), UI64($C5822CE44E3AEF19), UI64($8D165342C6F54D95), UI64($2AB46EC27C1DFF9F), UI64($8E568CAEDD482CAB) ), { 60 }
    ( UI64($C7AF417AA59466A6), UI64($14F56A3CE906F87D), UI64($22DC95D0C20DC86D), UI64($B632CFF822E2CB24), UI64($1FB562502E297678), UI64($173B1C675BC652C6) ), { 61 }
    ( UI64($401A4084B0F53485), UI64($B9C5000746244D2E), UI64($73353CE7D8550065), UI64($3D9A4C72536FD28F), UI64($BD415DC7743D1A9C), UI64($EBEAC1963ABCD6BE) ), { 62 }
    ( UI64($3ED003D74ADC2CF3), UI64($15C520F60A291D95), UI64($5A95344407639018), UI64($2996C63BE5D4D908), UI64($916FC19210E6B286), UI64($94128D59A765C785) ), { 63 }
    ( UI64($0D33A6423A2CA0D2), UI64($748CEF7CF523C571), UI64($1159E992CA34169A), UI64($89E169B453F6D2A1), UI64($F873438EECD366AE), UI64($A25BBCC3ABEB2434) ), { 64 }
    ( UI64($4261894340814F6B), UI64($3C4B4A81711F50BE), UI64($D0C860AF93B9420D), UI64($20B4FCBB2E7402BC), UI64($A613BB36D880E8B5), UI64($76BB31E2BA788D99) ), { 65 }
    ( UI64($639363FA40E53CD8), UI64($E1B175630C11DF2F), UI64($2FEBC29D14D2399C), UI64($D292B92745502B99), UI64($9BB56F770A39E529), UI64($BA1A7927B29B8166) ), { 66 }
    ( UI64($AC090C8658C0A121), UI64($1EA6090D930C19A0), UI64($97DBAB25D464956F), UI64($6AE63A3269786CCC), UI64($89B5430308043F68), UI64($171D792CC6BBB11B) ), { 67 }
    ( UI64($664C7BC982732FDB), UI64($5AC75069AB3F2028), UI64($82346B945264BD14), UI64($EF014E9E4025F1E5), UI64($6F0645079DD7665C), UI64($BC5B2706AC1FCD0F) ), { 68 }
    ( UI64($E6F121844D18A3F8), UI64($B6E1178B863226CD), UI64($9D4DD2A7C6F7E803), UI64($FBAFCB690BE7725E), UI64($5DD68673CDCD670E), UI64($B536445E7BF0BB78) ), { 69 }
    ( UI64($3170DB48F8E52419), UI64($90EF7AE181633410), UI64($66DB3F789D314D42), UI64($6DD8FCD41B03D852), UI64($A5B1263FD8901502), UI64($04B3481AF58203EB) ), { 6A }
    ( UI64($D2EC1A5547213C98), UI64($14A13BDD017B1EE9), UI64($AFADA55AFB32F2E8), UI64($89706761C174587F), UI64($2D784C000F84C7DA), UI64($E8EA7078982FB288) ), { 6B }
    ( UI64($994B00C75D9F1956), UI64($03774AA164A047A5), UI64($1A9CC89FB9A0BB72), UI64($E3F88A73F2A58104), UI64($E52B66BE4310D70C), UI64($27CE615E8FAE5DE0) ), { 6C }
    ( UI64($C3D4A291F8B3B3CD), UI64($2AF2CF29DDFA40B8), UI64($2F9AAA1F4B6AB0B2), UI64($2FB5843AD45EE053), UI64($B14DB852FA816858), UI64($879998071BE483F0) ), { 6D }
    ( UI64($C83E206608AD2D54), UI64($DEA2BBB3781232B6), UI64($3ED671260B448844), UI64($B8A73D001273CD55), UI64($7A8989CDC3398AF3), UI64($248E565FAD436886) ), { 6E }
    ( UI64($FB03EB2FC1E478EB), UI64($3DD79817D0C15EB1), UI64($8C07461CCE15D1D7), UI64($9BE7C49891B7CA3B), UI64($25AD2437E9E5D4CD), UI64($661E9916095B7D6C) ), { 6F }
    ( UI64($57B02982A1A4BD74), UI64($9CEB65D956C9A021), UI64($C21A8F31EC09B134), UI64($11DD7E723DD70182), UI64($BA0EA7C02D3FA5DC), UI64($32AEF95B5F199554) ), { 70 }
    ( UI64($FD6D3A6F7D756654), UI64($7F402C41EFD32C53), UI64($DBE12C3D8AF63D69), UI64($92E4672B03AE3500), UI64($BEBF0BE79A87BCC1), UI64($E0D0C8451B769362) ), { 71 }
    ( UI64($FA407554424447C4), UI64($38276EB9DC78FFCA), UI64($DC030B40E392A111), UI64($8E7329B733BF7986), UI64($B8CD20BAFEB081DC), UI64($7F2FE440A604C24F) ), { 72 }
    ( UI64($C4676A5698AC5BE1), UI64($D96C0D04CE7E00E3), UI64($8B5AF70A44A35D2D), UI64($1AF81796FE376360), UI64($19222827B6E94686), UI64($D0833BBAA728C2E0) ), { 73 }
    ( UI64($205DB55620E47932), UI64($3DBCF9480473B896), UI64($5396A9B3E95BF1F4), UI64($84A18E3FC1AB5C51), UI64($1B0FAF9FAD895128), UI64($E6776AEC62E1BFEF) ), { 74 }
    ( UI64($597731C0CFD0977D), UI64($68E5A99B3307D464), UI64($CB1A59CA0874D1E1), UI64($7F91E66083BD27B9), UI64($1004BCF22A307507), UI64($C473FA32C0EBE146) ), { 75 }
    ( UI64($8F2D73F152382CA0), UI64($36DB3631484FCBA2), UI64($AB8E7DAC58534B85), UI64($C9DC83F64BDCACF0), UI64($E18A86D5A404EC7A), UI64($F74C8002CBCCAE7E) ), { 76 }
    ( UI64($ABB3E7C985B14838), UI64($1E526991B02E100C), UI64($B60333765F302EF1), UI64($82027211096A8028), UI64($DB5FB63C32DFB78A), UI64($EDC90E36AA68D244) ), { 77 }
    ( UI64($DBF8824CE804FC56), UI64($833E3CCDDF0509A9), UI64($C612931D08B2EB27), UI64($E8C1E576400E5739), UI64($4046E16C0E41B9A9), UI64($D52A44E631E44B2F) ), { 78 }
    ( UI64($7BF702A8863F41AA), UI64($441C1315EA08D9B7), UI64($175B695FC295287B), UI64($FFC4A7021DC57CE7), UI64($AE0F2E105C0D35A5), UI64($8466C9C22121D0C3) ), { 79 }
    ( UI64($A6C82B7A5762339C), UI64($C3FD8BDBE6A6D1F3), UI64($98E37E76CF051B41), UI64($FCA1443254D39FA0), UI64($D1820D3C2089FB12), UI64($A6297940114D0E2B) ), { 7A }
    ( UI64($91A8E88F5B269B84), UI64($5CE5BD13B839F915), UI64($71D3BADF590793F9), UI64($61A5799A1F2F635C), UI64($CB6E8EE70E0CFCBE), UI64($7A518532DCCC48E1) ), { 7B }
    ( UI64($ED6159C438D8ACCE), UI64($BD2D47B352FFF864), UI64($00E3FE4DEBD8B0F3), UI64($C8488D49055FAE39), UI64($082BCA7E34657650), UI64($2CEF272EFB4F23C1) ), { 7C }
    ( UI64($02A9BAEC8AAA2950), UI64($04C3089CCF6A42DA), UI64($56BB1005ABAE17B1), UI64($DF89B6EB4B37DF67), UI64($C645E8D7B0290BAB), UI64($BAFE2D335B9B504F) ), { 7D }
    ( UI64($9864724EC1AC9C32), UI64($D037FE97EE4E6504), UI64($B0CE6A578619DCF1), UI64($2FA089F59ED451A9), UI64($11A7D846BA8F897D), UI64($B0DC4BBC68269D1D) ), { 7E }
    ( UI64($47B36255297B7BEC), UI64($3DB226CD2B457EA1), UI64($BBDA45CD1B953901), UI64($24E5C510DB98C91A), UI64($FA5E4E879EF31B7E), UI64($A2212D3B738379D9) ), { 7F }
    ( UI64($3AFB0F15D9C24D59), UI64($45E33E5A843C691E), UI64($5958CFBFBF6FFD65), UI64($ABFA135003DC8135), UI64($3ABE384DF795707E), UI64($8E9BB4F2215782DF) ), { 80 }
    ( UI64($745DCEE8F8004B47), UI64($93BC66870AD2CDA4), UI64($F2A073FE17265071), UI64($512DEBC3E3F74A5B), UI64($FD1C4F58BA4F426A), UI64($2202BC54E90C844B) ), { 81 }
    ( UI64($484CE432EF1C5151), UI64($18EAD594A4505900), UI64($A48F57DD88981D6E), UI64($459E102A4516B809), UI64($F97EC9A01788A21B), UI64($BCA356C25A537D9E) ), { 82 }
    ( UI64($3EFD68177C157D7F), UI64($425F45151A7C1BA3), UI64($162E8946EA69AC73), UI64($A0B53DF003311A8B), UI64($5C61F6F3D0BFA1F2), UI64($ED923DA5D3C1ACEB) ), { 83 }
    ( UI64($7C50E2DEECFE9DC6), UI64($17F8099650DA08D7), UI64($C87BCAAA940703D4), UI64($AB80498FCBEC19ED), UI64($78950EDF9FF203D9), UI64($ADBBC8AF1F48039E) ), { 84 }
    ( UI64($08496491E4F2B575), UI64($066D1ED2A0ACA263), UI64($BA61B5BDC394DC8F), UI64($E4336312E97F1E50), UI64($3479BB15D217CA47), UI64($9012EB13F695CDF7) ), { 85 }
    ( UI64($DCBC43CB441BFA6B), UI64($D9C1108252E9B959), UI64($660D0087622D1089), UI64($26E35E54C71A3192), UI64($AE95DEEC1A505722), UI64($B61F5182CF903E42) ), { 86 }
    ( UI64($AA03F75764204EAD), UI64($769095C51E867443), UI64($EDF66399EA6D0B73), UI64($C73B6CDE5431CC14), UI64($5CCCD864790A3340), UI64($165EE303CF64E397) ), { 87 }
    ( UI64($E5A3F256C577C042), UI64($5B41300E3827A9B4), UI64($E80F8A8EFC5CAD04), UI64($FF7866B9B15E9429), UI64($C1BCEA40F26841E2), UI64($5BE4FF4ABC25A3DA) ), { 88 }
    ( UI64($C932FB419D304C04), UI64($627556497691F88A), UI64($149D21E8615C5A62), UI64($6A5307B2E54FF029), UI64($1CBE87CBD3F40139), UI64($867C8331111850A0) ), { 89 }
    ( UI64($9470924D82A490A0), UI64($6F75446E246C7283), UI64($BC4911F93A9CF101), UI64($D15E5CC08522A4BC), UI64($91196DD95C4A30AF), UI64($A0BB0D4F88E78FB7) ), { 8A }
    ( UI64($C9DA7CE16D82D128), UI64($07CA96B97BFD7D15), UI64($D63FA64A4B19C926), UI64($D1281837870B10C7), UI64($B1E36B87EE2FE287), UI64($634F2C0DA84AB3BA) ), { 8B }
    ( UI64($D872ACF3F9CBF460), UI64($14757FDD55762BBD), UI64($2BA56AF060A4B36C), UI64($D293899C12817B8E), UI64($E4D275E538C060B3), UI64($D84D47DF3C4DB097) ), { 8C }
    ( UI64($5D4C3DDD50A2EC7D), UI64($44B4A410EFB0AAFB), UI64($3E0649FFD5BD29CA), UI64($A1DACCE1DDAD8B20), UI64($09BE5954F50DCE81), UI64($082D39AFC260384C) ), { 8D }
    ( UI64($637AA6D9A7A27B94), UI64($09BAB935847415EC), UI64($71883D0EE0D78920), UI64($4095CB4D44B5A0BB), UI64($C4E1832FD93E1DE3), UI64($2D1E07C9D27DD252) ), { 8E }
    ( UI64($788FFE15E2CB720C), UI64($A2F18380E98A68F7), UI64($888ED12A39E1B45B), UI64($2147EB5AAFD2BBD3), UI64($8A2366C7A15A3999), UI64($95DB871AA96EE06E) ), { 8F }
    ( UI64($5FB195C85F2F2470), UI64($B142421DCB5CE256), UI64($2D3E6238594FF595), UI64($6DE0791FA43742BF), UI64($447F564C389D3217), UI64($421D5FCB76E747C8) ), { 90 }
    ( UI64($5A9BDC549C36B14E), UI64($C84CE5E5028EBD48), UI64($884D50681D73904D), UI64($A7A22C924C94B45E), UI64($9582CA4780A39426), UI64($2E04FADFDA58CA5E) ), { 91 }
    ( UI64($F0EB5499E595AE44), UI64($AF98790CBE0F0B11), UI64($25535BECA50938B7), UI64($91975A7F08BDBB48), UI64($D6B24859B68E14DC), UI64($827E3214B9BAAEAA) ), { 92 }
    ( UI64($F3CBA8401657547E), UI64($2D888C5B6C28CD49), UI64($7FD2F22A39AE6C79), UI64($D143CCB45B066C33), UI64($98A5B0E7BD952A16), UI64($D6B1535EFE28311D) ), { 93 }
    ( UI64($82AF42B659F6429B), UI64($2D2C6E768FD2315D), UI64($1D3C0B6D4D75C6B1), UI64($6ABD0E260A0BBEDD), UI64($86FB2558990A819A), UI64($BF9D7759CFC849C4) ), { 94 }
    ( UI64($F916C0983358AC5C), UI64($492790E5C9B7C3DC), UI64($D4CAB0EF1DA545D4), UI64($B2F9F3BC2535E812), UI64($C8535D33B2AFA578), UI64($D46CAC719EC39F01) ), { 95 }
    ( UI64($9AA4FE6990B7BF05), UI64($97D652F0900FF0E8), UI64($E5D1D944DE2E692A), UI64($0A38AA2FB3A6379B), UI64($DB391D65146D58B6), UI64($75B6726983066D12) ), { 96 }
    ( UI64($F7E784F55B126999), UI64($F5B7E58D24A3DA95), UI64($B6981236C66D0F52), UI64($0D44B10C88E736A1), UI64($060A25EDFAB45A76), UI64($3CCAE89FF2D88BE7) ), { 97 }
    ( UI64($CBC4B3EE83E356FF), UI64($3783369DE63EB7D4), UI64($6A310DDEAEC3396F), UI64($814D399E294EC44C), UI64($4B417B42926E855F), UI64($A61BFBFF8FCC59F2) ), { 98 }
    ( UI64($0E0A7D90E6844DC9), UI64($C870E60D5DFB6783), UI64($63E90F52BD262DED), UI64($D2A20DE1A685D4D8), UI64($C307AC6628425EB3), UI64($A4B2D411F6207D65) ), { 99 }
    ( UI64($A795A18DC574F004), UI64($C4209A366244C78A), UI64($8EDCE1C5273AF12A), UI64($DD84593E08F74270), UI64($B6AB7BFB29530513), UI64($6BD7849206552A47) ), { 9A }
    ( UI64($E49FA24FE6FD4577), UI64($C637DFF8968F5D9E), UI64($13677E14C51A130C), UI64($D17DAB9D96FA8853), UI64($0C0608E1185D42DE), UI64($33FB328304CA93CC) ), { 9B }
    ( UI64($8DA0961B29675077), UI64($7E47E2C260B350B6), UI64($EB3FF461D163E926), UI64($D6F748675A8CFF3A), UI64($1887D161919F65E6), UI64($81633132F35CD84F) ), { 9C }
    ( UI64($AA9988566F29F8C4), UI64($8CC55C13E5CE1E80), UI64($603A6F04724D3822), UI64($7470FF2B8E81C26C), UI64($8831AC0C9BC2FCBE), UI64($B065FB970E730227) ), { 9D }
    ( UI64($D27A7737E059C046), UI64($27E5B10C9BDCE6A6), UI64($735F82565D36C77F), UI64($8E6B4A4F545C0B0D), UI64($0BD2ACF5F8A1A584), UI64($E4B7A499DAFB4A33) ), { 9E }
    ( UI64($2013C2654E67A3A2), UI64($3A778A90F6943C18), UI64($03FEA3FAA0282189), UI64($1808A150645E1DD3), UI64($ED9DF15005A9D48A), UI64($A05A65F7BA562A25) ), { 9F }
    ( UI64($16C78B90F574B969), UI64($AAA54454B18375B0), UI64($B32F6B432283644D), UI64($72E506CB0F3ADDAD), UI64($3F25CFACFC30C0ED), UI64($594F225D77641B52) ), { A0 }
    ( UI64($3D8A338E34E0A73C), UI64($FB2991B5E7A95529), UI64($93BC4F7EC0AA813E), UI64($FC7273C5CB0DD2D8), UI64($6FD8AAFEB3E871DC), UI64($15884FB7E22B8D96) ), { A1 }
    ( UI64($EBCD4F36BB1FA30B), UI64($DA6512FBC79CFA88), UI64($290DB13C9C2F8B8A), UI64($4397DE0D71E91181), UI64($92C72AC60F0EA0E8), UI64($6BD53E93FB21B325) ), { A2 }
    ( UI64($01A7E269EA27FC44), UI64($B9EE9A6D27E2CE14), UI64($9CE539B0F44DC8EC), UI64($93760B66A3912048), UI64($8940A4DE651422C7), UI64($479325E2376ABC43) ), { A3 }
    ( UI64($B978D1AF3EE870AD), UI64($1A102039BCC2BFB4), UI64($FE26E467F82FB5A7), UI64($7AA2AAE78A371518), UI64($A6C049980EFE030C), UI64($CE5B165A31B9CF55) ), { A4 }
    ( UI64($E079A70EFA5E7BF3), UI64($591215F801CC7032), UI64($C6EB485231416A2E), UI64($79015550313DF7B4), UI64($6DBEE87A70C69171), UI64($B3A9223B8FAD9830) ), { A5 }
    ( UI64($1E65C5246BF32C0A), UI64($746F71887112AC88), UI64($3768EE0D1E4C803A), UI64($13285A9217D9472A), UI64($50D3653EC5B92FDA), UI64($D3D08125121C1B16) ), { A6 }
    ( UI64($B26658B8A7D7C6E0), UI64($98BABAB8F6A75826), UI64($0C5CDC5B7C571E2F), UI64($A0757C35E27C3C15), UI64($5A4CA5EC989F5323), UI64($12040917D1353181) ), { A7 }
    ( UI64($283C3F5D1D0F20CE), UI64($35EFFEF7A76E80FA), UI64($C284C26FB60A607B), UI64($F45C41973151D4AB), UI64($DF0449DFC681E6A4), UI64($9F4E923144BE2940) ), { A8 }
    ( UI64($A37EBA8348C56B09), UI64($1BA4A5AF2150A1BB), UI64($8538E2ECA1865E02), UI64($EC70C5F094E6B54F), UI64($28532F0E3E2F1CCC), UI64($6AA481AB8C522467) ), { A9 }
    ( UI64($CB6BCBCC64F474EC), UI64($B06AC92DA299C98E), UI64($6703AAD7D1EF4018), UI64($F674C9C07943BFE2), UI64($F39AF9B962D69A7B), UI64($2C563C88D9FA4631) ), { AA }
    ( UI64($919FEF03EB6ACF89), UI64($6C7E22DA8609DBE3), UI64($1FA1E2E6B8DFAD9E), UI64($AB95CC3584E76CCE), UI64($1D04DAA8C2B3FE15), UI64($1D43007A1F9F889A) ), { AB }
    ( UI64($E7818E5789081774), UI64($C0704429AF659E0E), UI64($A373CAECB9D9966D), UI64($09B15BFB604EF493), UI64($37416D1C8D30D8EB), UI64($14E79C0CC95BB0C1) ), { AC }
    ( UI64($524F475F142A020B), UI64($1C6451BA2E4BAF9A), UI64($9B539D994A1AAF6E), UI64($F4275FE42F97D113), UI64($C427C07D61E709C3), UI64($24CCDF222FF915B2) ), { AD }
    ( UI64($14DE747F8D1669B8), UI64($4E0E531B85D62D43), UI64($78E2A7D71BA2E18D), UI64($1D946B89B1433FBF), UI64($00E387ED73F89378), UI64($52992A8283A832C6) ), { AE }
    ( UI64($CBDB69FBCB21009D), UI64($0EC6FDEC4E8269EA), UI64($79D908AA89FED13F), UI64($7548E553A1103EF0), UI64($BBE4048A115EC944), UI64($E1C60EBF7FBDDFD1) ), { AF }
    ( UI64($24522AA6EC219FB7), UI64($78C952E8A5DEDDDA), UI64($2822F9EF97929DF0), UI64($A70C9D1D8258B682), UI64($8B37AC646D341E93), UI64($D67C34FF4BBD5E0E) ), { B0 }
    ( UI64($E770CD22FABC425A), UI64($7E9C34D3ACC5075C), UI64($0AFFA78C7BC76623), UI64($4EED9851852C03F2), UI64($7203EC6EE60426FE), UI64($6517CFB85022DC4E) ), { B1 }
    ( UI64($1581F89650F997A1), UI64($DB9CE979F407D39E), UI64($0AD4800E39347F3F), UI64($C0C35B8D9FD1392F), UI64($E74D894A6167AE43), UI64($D462E2F4E3CA9AC1) ), { B2 }
    ( UI64($05EA545C5B41C787), UI64($CECD2DC56104C690), UI64($7337C0746CC963CB), UI64($FDF08B81F136B8B1), UI64($7F7A862159C72990), UI64($0AD9111B6CB0ED2E) ), { B3 }
    ( UI64($6FE578032A920367), UI64($49EFAEE55C832164), UI64($B3157997365AEF57), UI64($98F84917C903C67B), UI64($75CF75DA2AA67BF3), UI64($D2CD8AB69E83D8A4) ), { B4 }
    ( UI64($65B99EB5C4E335A0), UI64($486A759227255280), UI64($4F7A3B558F86EAC0), UI64($03DF89CC2686DBF0), UI64($A95693164EEAA311), UI64($D3149307EF5BDB09) ), { B5 }
    ( UI64($653261B529527041), UI64($9A559ECA9A789241), UI64($5931FF0E09E8677A), UI64($F6D53AFDF41458EC), UI64($02B94442E96BF81E), UI64($5C4A1B461FA95C7D) ), { B6 }
    ( UI64($B6FA3DB3F3096A56), UI64($AD1E678369F141A0), UI64($81708DA9ED5E5188), UI64($E2EE29D337A81DED), UI64($D7EC74F0AC9B37C0), UI64($1B785A6FA3C76037) ), { B7 }
    ( UI64($AE153D618202ED33), UI64($E8E91115185E46DD), UI64($05D9BA34364372BC), UI64($672EE1629BDBE13D), UI64($FC2F5EA8624A61CB), UI64($D843198BE3925035) ), { B8 }
    ( UI64($BCE8F7C1FC130582), UI64($A571AEBDC5C5B472), UI64($7F08C70583911E9C), UI64($F4BC16EA95A9D958), UI64($A9EE3D6BDDEA2C72), UI64($1B651B74994DF683) ), { B9 }
    ( UI64($13A7D9DA16EA461D), UI64($7FCF04FC4EDC32B1), UI64($66607C39138582F8), UI64($8831546EAD73A091), UI64($D8A4C4F349D1F5FD), UI64($4975C9D5701638DB) ), { BA }
    ( UI64($BF0AA6A41F0EC78F), UI64($DC2BF64F490226F3), UI64($CDA313AAF2D03538), UI64($04B8DA4460595A63), UI64($A23F1B729BB763BA), UI64($F793736A73F77272) ), { BB }
    ( UI64($3BA334AEBFDF6E3A), UI64($212037EAF6A1E5B2), UI64($004B3C1920BF41E0), UI64($D8A11B5BA862706A), UI64($6579E5027315F8AB), UI64($3C1F9CBB6D54E13D) ), { BC }
    ( UI64($BE24DE02BDA2A2D7), UI64($4D69A648E027E9DE), UI64($FEBC2A5373BD90C5), UI64($42C7AC0E55A414DB), UI64($D65962B721659EDC), UI64($53831B2E73FB0D23) ), { BD }
    ( UI64($3DE68C3D94080E94), UI64($1D9E0BC868E087DC), UI64($E7AD1EF14270B35C), UI64($9A14CF373E0F67E5), UI64($00EF98ABE6B7E600), UI64($82353F1F18234EAA) ), { BE }
    ( UI64($6034549E932CF51C), UI64($3CE2E36E3EB4D407), UI64($9FBA80CA22B31E3F), UI64($142EAEAC8CAD93F8), UI64($5D3D5CA6E5909CF2), UI64($1B2C70B482B625B5) ), { BF }
    ( UI64($BC212277FE451938), UI64($E21637EF3A3EF4DA), UI64($C39C70D155CE7AD5), UI64($5C3EDC78491F3AF8), UI64($1A405C9B847D16DF), UI64($0F446B056E291875) ), { C0 }
    ( UI64($69D791C758B8E1DC), UI64($49D8A84512B8B0CA), UI64($2474F7B05E061DBC), UI64($A79C97FFB1493F6D), UI64($6FB9E77B764A9772), UI64($311593614CB6D8FE) ), { C1 }
    ( UI64($F67F8D77D5A45E27), UI64($17A3287F58D26572), UI64($182792E595E52080), UI64($7D15B7052C373A90), UI64($536CF2976B910733), UI64($F0E483B43ACBD504) ), { C2 }
    ( UI64($9F1EE351D08E27EE), UI64($76158F0101251C50), UI64($C1269FD22FDE67FA), UI64($6C1930A26E6F3F5E), UI64($A32AA0AAA396B071), UI64($497A599416938BD9) ), { C3 }
    ( UI64($569916F08E7D1767), UI64($AB20F046F165C725), UI64($BD4C1CAFC3DE32DA), UI64($CF19F61CCA274CDF), UI64($4DEBBDC90446F779), UI64($0D07522E07429B0E) ), { C4 }
    ( UI64($B0F295FB72F3921F), UI64($C692DFA79FAD80EE), UI64($69011629E0238DA6), UI64($0046AF22910AEE2C), UI64($77DEBB94C94A9DBD), UI64($4B401110EE8A03A8) ), { C5 }
    ( UI64($FF6D2FCA08D69F8B), UI64($97592AF1712A5ACF), UI64($9D540AF103BB0CA6), UI64($D55933E572328F10), UI64($EC7218CA2FB0F7A7), UI64($8D154ECE583152AD) ), { C6 }
    ( UI64($6CEFB6A3BBB18EFA), UI64($E47A0437AD48BB5D), UI64($AC9695D38CEB73B4), UI64($4E5468B6CEB06906), UI64($3A57007B43F1591F), UI64($16EDF10ED21C2651) ), { C7 }
    ( UI64($3012A14CAEE92A6B), UI64($C7CE2D0A16397B6E), UI64($B2CE1D676CDD36B6), UI64($F563F9683F8C3E45), UI64($3C112343EE739443), UI64($D2FD36380C31457B) ), { C8 }
    ( UI64($1082D7EDE79C89E8), UI64($57D1C088D807C597), UI64($9E5CE9CCC772F2AF), UI64($D30AD1CAF61BB8D9), UI64($1E40D024A6939CA4), UI64($4985DBEB358DF104) ), { C9 }
    ( UI64($616EBF3FC09BC5E4), UI64($A949C7DA50B367ED), UI64($0C289EF8AF79CDB9), UI64($5C23303A5F396469), UI64($3316727F556A9DB8), UI64($B7CCEC943229C3FC) ), { CA }
    ( UI64($B9B4B8C04D465EAE), UI64($ED44836D54AA7798), UI64($A6F7DC5F0FF5918E), UI64($700DD1C0F68E4E23), UI64($4BF2FAA0516222AC), UI64($3C2AE26002348642) ), { CB }
    ( UI64($8A7ECE878B79532A), UI64($23613836116B497D), UI64($FE665E9804C10AC7), UI64($45E6B949601D50DB), UI64($5B2A9E4887F1AE92), UI64($447FF94BA4FF2079) ), { CC }
    ( UI64($E31CED1B1F588060), UI64($E60A18CE70D51BE7), UI64($62C08A1E01A2E1F3), UI64($008EB3D687EAE77C), UI64($1D01A7F254134224), UI64($7E007BE76E56F3A0) ), { CD }
    ( UI64($E31995299E1D8C4B), UI64($D1FBE650613BCD97), UI64($1D3C3F36A806C06D), UI64($49F278C3D9F4A9C6), UI64($0EAC518E5FEA0B19), UI64($7D37A390C9A06EDA) ), { CE }
    ( UI64($E38F2A3ED827813E), UI64($C5CD4C09D275547E), UI64($1CF0E8313F2ABED6), UI64($9ECFE847ECCF6C5E), UI64($60E68184C974253D), UI64($ED6AB7636CDD66F8) ), { CF }
    ( UI64($B6C9DC4E954DEC83), UI64($337A86D43C1E137B), UI64($611A5CECFD4FD4F2), UI64($4FABE0C6259F490A), UI64($274393117710FEF9), UI64($F80A64FFDE29B591) ), { D0 }
    ( UI64($CAB995B447838136), UI64($F8BAEE20D66998C9), UI64($037F8691364D2B07), UI64($F711FE824949F810), UI64($0D1B4E2773B4E944), UI64($FABA24FEC54F5E29) ), { D1 }
    ( UI64($308C0B2974968FCE), UI64($305CE82B64FDE7FA), UI64($CBB957FA7AC79C07), UI64($938BB1699D7C5FB1), UI64($4DFC5FD652B4D859), UI64($C76D7FC21D9AB354) ), { D2 }
    ( UI64($213F9D2ADEB6B2B9), UI64($0A501ADE808BA7E6), UI64($E1F1A9F9AD58121D), UI64($E1DA4CE9E3D30307), UI64($4F8FF0EEBD51C22E), UI64($5EAF5D79C0C902F0) ), { D3 }
    ( UI64($66D86B9486FA29FD), UI64($AC59FAC357C6881B), UI64($6BC55595A36E36A6), UI64($162BF06D315103EC), UI64($126CF6AA5E1F26F7), UI64($9DF080C49CCA232A) ), { D4 }
    ( UI64($26A2DBCA3C8D42FE), UI64($A09DCB40FB746B29), UI64($CB007706B3E9B9B7), UI64($ABCE248527C87837), UI64($825E3DF5D74B619E), UI64($F2D7FC9D9CB58C30) ), { D5 }
    ( UI64($5CA546BA21141BE3), UI64($4588716BD734AA98), UI64($4AEEE69AAF4D57A5), UI64($E01ED8CBD87F4734), UI64($243AA59D601A4FB6), UI64($3EACC74DBBE7E0F5) ), { D6 }
    ( UI64($123DCC7D18D37E12), UI64($A9D7BD9B59849E03), UI64($8944A203A613C897), UI64($17612271EE2EC452), UI64($CFC11C233E5F2219), UI64($F0A47DE1B0F8FDF0) ), { D7 }
    ( UI64($4D84263BE85828AC), UI64($C45768F30F7393CF), UI64($0D4CC4C0F7F80A70), UI64($017880DD846965FE), UI64($C3BC9F2DBBE68DCC), UI64($C236C7E247AA3536) ), { D8 }
    ( UI64($8EB4E4B8B7DA54C3), UI64($FA1427A6D1389409), UI64($744449AB8B19F0B9), UI64($6BFBE32B2D0BDB94), UI64($91313EB6E06C4FF5), UI64($0170A64FE878578D) ), { D9 }
    ( UI64($179E05C17E3694F5), UI64($31599DAEA9E72F8B), UI64($028704B1352027F3), UI64($DAB6F4975B6EF09A), UI64($5E3E730ABC6E8FE7), UI64($41753647414763D1) ), { DA }
    ( UI64($DDBA4F2D0434B98A), UI64($0FBAC522F3DAEAAC), UI64($D315AED4BB31715E), UI64($5C1F3D78EF9E5425), UI64($B8DD542D8BF637EB), UI64($5FA8A2EEE095CE1C) ), { DB }
    ( UI64($87519C3B117EDF1A), UI64($F7C469924B306E46), UI64($D2C859D56EE93FD6), UI64($7E43975DB42844FB), UI64($B6ECE5FF0483394E), UI64($02D62595E91E3398) ), { DC }
    ( UI64($326812111FD4C7BC), UI64($099544A2C2D81A7D), UI64($B429BB12ACEECF49), UI64($F916080CAAD15FC8), UI64($E814BE190473E0C0), UI64($8245A66C5FA6F410) ), { DD }
    ( UI64($7C28F51D33181D06), UI64($C2E1BFACD6F225FD), UI64($6D7477373FB058CA), UI64($273120A4ACBA0FB4), UI64($E740E22190BC3F2F), UI64($EBF7F50765758830) ), { DE }
    ( UI64($A9D4BB665C151EB1), UI64($C6A1EA32577F1B57), UI64($F12A0091BBD1C87A), UI64($52EDEB266D0D06A5), UI64($6682457922D38306), UI64($9E208003AC81FFE1) ), { DF }
    ( UI64($88456DFF8FCC14EE), UI64($B7DC6813F9D16338), UI64($774C779B6719CD40), UI64($421EAD66567CB6CE), UI64($EF04E66F4FB17361), UI64($708C07F995B8EE67) ), { E0 }
    ( UI64($FD516C065215660A), UI64($BC97FC5E0F7C426D), UI64($3D72E2DAB55CE389), UI64($48BB97924B7289FD), UI64($01DBC3476578FDA5), UI64($5464B44925146D56) ), { E1 }
    ( UI64($12B45F28FE734036), UI64($2BAB852D174B4C3C), UI64($F0D3F5F0A0D12AFA), UI64($3E09CE8FC0F88FB1), UI64($BA2BE405CF690227), UI64($3100CDADAD5E9BAC) ), { E2 }
    ( UI64($43C5DB2C16CBF0F3), UI64($4158F77B00AF1683), UI64($048F4888D0AE4B4A), UI64($36AD98F815351E5F), UI64($C7D6309CAF1F7A02), UI64($4EFC524D45B4F6C4) ), { E3 }
    ( UI64($254F60E0A929D2C3), UI64($AEAAE6B4C271D87F), UI64($D0DC0C713AA202AA), UI64($D75004DAB803027E), UI64($822FFE90F2BDF421), UI64($9998FF9D9599BE72) ), { E4 }
    ( UI64($F4F26F3D3DAA4018), UI64($63A3CC4733CFEB16), UI64($596209EB60AE0BDD), UI64($43BCB7F9F0D863E7), UI64($6505D0951F747F58), UI64($9A34D71F7A102D47) ), { E5 }
    ( UI64($6A06660A549ABA78), UI64($263EF1081353B190), UI64($096735234BA57B0D), UI64($1D4E120293CE8CC0), UI64($4E4A3CD82A707C44), UI64($699027EE00372934) ), { E6 }
    ( UI64($0E0D83FCE670DBDA), UI64($9F04D71D37DF79DE), UI64($9443D35035446607), UI64($101FB87DCE888457), UI64($627BF79224057C18), UI64($DC4000FD953FCCCB) ), { E7 }
    ( UI64($FF6901D6C9FFE0A5), UI64($2CDAADDF05DF6DEF), UI64($98DAF476D3EBBBDE), UI64($92D37135CE42C5B2), UI64($45187D72CB51905E), UI64($9B34C328AC445E42) ), { E8 }
    ( UI64($593B4F221871A532), UI64($5B58EEC73E4A07D9), UI64($14A3579B51593BA6), UI64($2D7C1D6AB98636FB), UI64($9B51F3C6582D4171), UI64($482BE23EC7FBBF93) ), { E9 }
    ( UI64($CF88AA1D3E0FC932), UI64($2118F037989B0264), UI64($B37DAC83C4AD9E9C), UI64($ACDE94CC47AA2BAA), UI64($118839C26BCD83CD), UI64($32B5843A3193C9DB) ), { EA }
    ( UI64($273694EBF59C4C26), UI64($43E86EDEE50D48C2), UI64($67F97093ED643CDB), UI64($A4AF5201D787AFCE), UI64($88E89F3243AC55FA), UI64($48CA4CCA84975E2F) ), { EB }
    ( UI64($07F23D3790D99634), UI64($5352170C09973E9E), UI64($4FF7EBEFD208181F), UI64($5D097ED48E177DB8), UI64($C83233D54225EDD5), UI64($BDB2981A360C76CD) ), { EC }
    ( UI64($BF7BB735C21F63AD), UI64($D42AECB81B9AC971), UI64($4184209556619D2B), UI64($56FB088DB24AB737), UI64($EF459017CC4ED7CD), UI64($68894E750102D4F3) ), { ED }
    ( UI64($EC3009E11646864C), UI64($ABB76EBE218996B3), UI64($A44366E48B433684), UI64($0250CD9B5F8F205B), UI64($37F4022177866A6A), UI64($21DC8EB446BD99DF) ), { EE }
    ( UI64($0EA6152DA50313DB), UI64($4B18FCBD7B7F2AA2), UI64($A520BD2EA4D34091), UI64($9293C20836323441), UI64($F72FAA3F44FC8B1F), UI64($1AE7A9220C877A87) ), { EF }
    ( UI64($A2F2E7EC8BA8BE87), UI64($99784F38EAC42DF6), UI64($D018F5FB1F54A5DE), UI64($8EDB8A128F89A8EA), UI64($E9F4FD01005F0A25), UI64($F6865090F443A384) ), { F0 }
    ( UI64($648DF1028E44B6C8), UI64($239AAE12588EA54E), UI64($C73245098AEC57B2), UI64($39B69938F8D4B60E), UI64($E98CCDA118CC7F99), UI64($6564349C25C899AD) ), { F1 }
    ( UI64($EEBA071910AE8CD7), UI64($AE13D4E444718CEB), UI64($5A38CCEA27116EC0), UI64($6626F950B9449503), UI64($4D20BE0C08D92A2D), UI64($C099D6E99A849DDB) ), { F2 }
    ( UI64($B96DF109572B1DEB), UI64($CD4EFA1EEF503627), UI64($2092C32B138766AE), UI64($5F738409FAFC1CCB), UI64($48CEEF9F8712DBD0), UI64($4A20E8766AF8C679) ), { F3 }
    ( UI64($13FA0D8677476531), UI64($F6560311FA218AFF), UI64($DD829DF52780CE19), UI64($A82A13DF3D99E4F5), UI64($5570CF897B7844C8), UI64($B4E0BA76CD6CCCCA) ), { F4 }
    ( UI64($5E6358BD596DCA88), UI64($C480A84997C11DC4), UI64($44BDFDA1C4162793), UI64($EA575920359081B6), UI64($CF404DBDDE42A8B1), UI64($6DB72A06F75FE7FC) ), { F5 }
    ( UI64($2FFE4156076B8F69), UI64($BE08BD673985DB1B), UI64($2EFBB12338712477), UI64($C3DAC6CA86CBC223), UI64($00F0141DC65C1921), UI64($694B556816B81608) ), { F6 }
    ( UI64($590CEF2457F28B3C), UI64($09B6C56115EC28B3), UI64($3D38681EF0B9A6B0), UI64($5D6EE687F269E86E), UI64($36E01FE904D7F496), UI64($7A384C59557A56E7) ), { F7 }
    ( UI64($1C06EC0F2387A7F4), UI64($A1C441581DDF954D), UI64($61FA1425D4E1CC47), UI64($BA7C9CA7801198BD), UI64($309C7389F48046B8), UI64($6CC511F3581F3A56) ), { F8 }
    ( UI64($50F49296D440C856), UI64($972CDAF27319B835), UI64($2D355D862AA6B78B), UI64($AF1BC5AFDAB3343A), UI64($4B421C12E8AE0F14), UI64($37592D957F1AC660) ), { F9 }
    ( UI64($A0A45CA6265E0FD3), UI64($4623CF6A78E8CB75), UI64($D055864C565E57B7), UI64($9B1487DC5803BF9E), UI64($D9B85F561B55CB24), UI64($4139E1014A64E67F) ), { FA }
    ( UI64($2634AABA637740C3), UI64($EB490F3ADFEBFECA), UI64($B968932FDDF0AB43), UI64($4A481F40FCFDE9B2), UI64($4E26D3F72A9CA725), UI64($59B90F4C85A3389A) ), { FB }
    ( UI64($2D859DB3E1C4F01F), UI64($F78DCB6825F90549), UI64($5E639D6F79843CBC), UI64($063866DA32B0E564), UI64($1DC17D7032F46923), UI64($64CF9FD8F825A61C) ), { FC }
    ( UI64($D2DE7DBA02DC8648), UI64($70860B4EFE423C78), UI64($130846413C803BEE), UI64($97EFE51BD7370CD7), UI64($677250C63776F48B), UI64($BBD1BB68B76A158D) ), { FD }
    ( UI64($3A9EE16977A19B3B), UI64($F229BE1E11B2120E), UI64($B5B4E6E443C31DB1), UI64($3DD07F9C6595FDBE), UI64($B4498DAB67547C34), UI64($AB43F4D1B9166314) ), { FE }
    ( UI64($08FC179DE2DDE233), UI64($BCA1827680C614BA), UI64($2C5229280DAF459C), UI64($1B522BC4EC0527A7), UI64($84125C25BF23DBCD), UI64($7850D044113F83FB) ), { FF }
    ( UI64($1D968E3B41E8C8BD), UI64($52F0B3C77BEED767), UI64($B916233D479F860A), UI64($CEEB3FDA20E4BE56), UI64($EB8F74A25AF28C6B), UI64($BFC13FA4B59E4E49) )  { ZZ }
  );

  MHASH384_MIX: array[0..255] of TRowByte =
  (
    ( $05, $03, $04, $01, $02, $00 ), { 00 }
    ( $02, $05, $00, $04, $03, $01 ), { 01 }
    ( $04, $03, $01, $02, $05, $00 ), { 02 }
    ( $03, $05, $04, $00, $02, $01 ), { 03 }
    ( $02, $03, $05, $01, $00, $04 ), { 04 }
    ( $04, $00, $03, $05, $02, $01 ), { 05 }
    ( $03, $02, $05, $01, $00, $04 ), { 06 }
    ( $04, $05, $03, $02, $01, $00 ), { 07 }
    ( $03, $04, $01, $05, $00, $02 ), { 08 }
    ( $01, $02, $04, $00, $05, $03 ), { 09 }
    ( $05, $04, $03, $01, $00, $02 ), { 0A }
    ( $04, $02, $01, $05, $03, $00 ), { 0B }
    ( $05, $03, $04, $01, $00, $02 ), { 0C }
    ( $04, $00, $01, $05, $02, $03 ), { 0D }
    ( $02, $05, $03, $04, $00, $01 ), { 0E }
    ( $05, $02, $01, $00, $03, $04 ), { 0F }
    ( $04, $05, $03, $01, $00, $02 ), { 10 }
    ( $01, $00, $04, $02, $05, $03 ), { 11 }
    ( $02, $05, $01, $04, $03, $00 ), { 12 }
    ( $04, $02, $00, $01, $05, $03 ), { 13 }
    ( $01, $03, $05, $02, $00, $04 ), { 14 }
    ( $04, $00, $01, $05, $03, $02 ), { 15 }
    ( $01, $05, $04, $02, $00, $03 ), { 16 }
    ( $03, $04, $05, $00, $01, $02 ), { 17 }
    ( $01, $02, $00, $05, $03, $04 ), { 18 }
    ( $04, $05, $03, $02, $00, $01 ), { 19 }
    ( $05, $03, $00, $04, $01, $02 ), { 1A }
    ( $02, $05, $04, $01, $00, $03 ), { 1B }
    ( $05, $00, $03, $04, $02, $01 ), { 1C }
    ( $04, $03, $00, $01, $05, $02 ), { 1D }
    ( $01, $02, $04, $05, $03, $00 ), { 1E }
    ( $04, $00, $03, $02, $05, $01 ), { 1F }
    ( $02, $03, $00, $05, $01, $04 ), { 20 }
    ( $03, $04, $01, $00, $05, $02 ), { 21 }
    ( $05, $00, $03, $01, $02, $04 ), { 22 }
    ( $04, $03, $00, $02, $05, $01 ), { 23 }
    ( $05, $02, $03, $01, $00, $04 ), { 24 }
    ( $02, $04, $05, $00, $01, $03 ), { 25 }
    ( $04, $03, $00, $05, $02, $01 ), { 26 }
    ( $05, $02, $01, $04, $00, $03 ), { 27 }
    ( $03, $05, $04, $01, $02, $00 ), { 28 }
    ( $02, $03, $05, $00, $01, $04 ), { 29 }
    ( $04, $05, $00, $01, $02, $03 ), { 2A }
    ( $05, $03, $01, $02, $00, $04 ), { 2B }
    ( $02, $05, $00, $04, $01, $03 ), { 2C }
    ( $05, $02, $04, $01, $03, $00 ), { 2D }
    ( $04, $05, $03, $00, $01, $02 ), { 2E }
    ( $05, $00, $01, $02, $03, $04 ), { 2F }
    ( $03, $05, $04, $01, $00, $02 ), { 30 }
    ( $01, $00, $05, $02, $03, $04 ), { 31 }
    ( $05, $04, $03, $00, $01, $02 ), { 32 }
    ( $03, $00, $04, $05, $02, $01 ), { 33 }
    ( $05, $02, $00, $04, $01, $03 ), { 34 }
    ( $01, $04, $05, $00, $03, $02 ), { 35 }
    ( $03, $00, $01, $05, $02, $04 ), { 36 }
    ( $01, $05, $04, $00, $03, $02 ), { 37 }
    ( $03, $02, $01, $05, $00, $04 ), { 38 }
    ( $02, $00, $05, $04, $01, $03 ), { 39 }
    ( $04, $02, $03, $01, $05, $00 ), { 3A }
    ( $03, $05, $01, $04, $00, $02 ), { 3B }
    ( $05, $00, $04, $02, $01, $03 ), { 3C }
    ( $04, $03, $05, $00, $02, $01 ), { 3D }
    ( $05, $02, $01, $04, $03, $00 ), { 3E }
    ( $01, $05, $03, $02, $00, $04 ), { 3F }
    ( $05, $00, $01, $04, $03, $02 ), { 40 }
    ( $02, $05, $03, $01, $00, $04 ), { 41 }
    ( $03, $04, $05, $02, $01, $00 ), { 42 }
    ( $01, $05, $04, $00, $02, $03 ), { 43 }
    ( $02, $04, $01, $05, $03, $00 ), { 44 }
    ( $04, $00, $05, $01, $02, $03 ), { 45 }
    ( $03, $05, $00, $02, $01, $04 ), { 46 }
    ( $05, $00, $04, $01, $02, $03 ), { 47 }
    ( $03, $02, $00, $05, $01, $04 ), { 48 }
    ( $02, $03, $01, $04, $05, $00 ), { 49 }
    ( $01, $04, $03, $05, $00, $02 ), { 4A }
    ( $03, $05, $00, $01, $02, $04 ), { 4B }
    ( $02, $00, $03, $04, $05, $01 ), { 4C }
    ( $05, $04, $01, $02, $00, $03 ), { 4D }
    ( $03, $00, $05, $01, $02, $04 ), { 4E }
    ( $01, $03, $04, $02, $05, $00 ), { 4F }
    ( $03, $04, $05, $00, $02, $01 ), { 50 }
    ( $04, $02, $00, $05, $01, $03 ), { 51 }
    ( $01, $05, $03, $04, $00, $02 ), { 52 }
    ( $03, $02, $00, $01, $05, $04 ), { 53 }
    ( $04, $03, $05, $00, $01, $02 ), { 54 }
    ( $05, $02, $03, $04, $00, $01 ), { 55 }
    ( $03, $00, $05, $02, $01, $04 ), { 56 }
    ( $01, $03, $04, $00, $05, $02 ), { 57 }
    ( $02, $04, $03, $05, $01, $00 ), { 58 }
    ( $05, $03, $00, $04, $02, $01 ), { 59 }
    ( $04, $02, $03, $05, $01, $00 ), { 5A }
    ( $05, $04, $00, $01, $03, $02 ), { 5B }
    ( $03, $05, $01, $00, $02, $04 ), { 5C }
    ( $04, $03, $05, $02, $00, $01 ), { 5D }
    ( $03, $00, $01, $04, $05, $02 ), { 5E }
    ( $05, $03, $04, $02, $01, $00 ), { 5F }
    ( $04, $05, $01, $00, $02, $03 ), { 60 }
    ( $01, $03, $00, $02, $05, $04 ), { 61 }
    ( $02, $05, $04, $01, $03, $00 ), { 62 }
    ( $05, $04, $03, $02, $00, $01 ), { 63 }
    ( $01, $00, $04, $05, $03, $02 ), { 64 }
    ( $02, $04, $05, $01, $00, $03 ), { 65 }
    ( $03, $02, $01, $04, $05, $00 ), { 66 }
    ( $02, $05, $00, $01, $03, $04 ), { 67 }
    ( $03, $02, $05, $04, $00, $01 ), { 68 }
    ( $01, $05, $03, $00, $02, $04 ), { 69 }
    ( $02, $00, $04, $01, $05, $03 ), { 6A }
    ( $01, $02, $03, $05, $00, $04 ), { 6B }
    ( $04, $05, $00, $02, $01, $03 ), { 6C }
    ( $02, $04, $05, $01, $03, $00 ), { 6D }
    ( $04, $03, $01, $05, $00, $02 ), { 6E }
    ( $05, $02, $00, $04, $03, $01 ), { 6F }
    ( $02, $03, $04, $05, $01, $00 ), { 70 }
    ( $04, $00, $03, $01, $05, $02 ), { 71 }
    ( $01, $03, $05, $00, $02, $04 ), { 72 }
    ( $02, $00, $04, $05, $03, $01 ), { 73 }
    ( $04, $03, $05, $02, $01, $00 ), { 74 }
    ( $01, $04, $03, $00, $05, $02 ), { 75 }
    ( $05, $00, $01, $04, $02, $03 ), { 76 }
    ( $02, $03, $04, $05, $00, $01 ), { 77 }
    ( $05, $04, $03, $02, $01, $00 ), { 78 }
    ( $04, $03, $05, $01, $00, $02 ), { 79 }
    ( $05, $04, $00, $02, $03, $01 ), { 7A }
    ( $01, $00, $04, $05, $02, $03 ), { 7B }
    ( $02, $05, $03, $00, $01, $04 ), { 7C }
    ( $01, $04, $00, $02, $05, $03 ), { 7D }
    ( $05, $02, $03, $04, $01, $00 ), { 7E }
    ( $04, $05, $00, $01, $03, $02 ), { 7F }
    ( $01, $02, $03, $04, $05, $00 ), { 80 }
    ( $05, $00, $04, $02, $03, $01 ), { 81 }
    ( $01, $02, $03, $00, $05, $04 ), { 82 }
    ( $04, $00, $05, $01, $03, $02 ), { 83 }
    ( $02, $04, $03, $05, $00, $01 ), { 84 }
    ( $05, $02, $00, $01, $03, $04 ), { 85 }
    ( $02, $03, $05, $04, $01, $00 ), { 86 }
    ( $04, $05, $00, $02, $03, $01 ), { 87 }
    ( $02, $04, $03, $01, $05, $00 ), { 88 }
    ( $03, $02, $05, $00, $01, $04 ), { 89 }
    ( $01, $05, $00, $04, $03, $02 ), { 8A }
    ( $04, $03, $01, $05, $02, $00 ), { 8B }
    ( $03, $05, $04, $02, $00, $01 ), { 8C }
    ( $02, $04, $00, $01, $05, $03 ), { 8D }
    ( $03, $05, $04, $00, $01, $02 ), { 8E }
    ( $01, $04, $05, $02, $00, $03 ), { 8F }
    ( $04, $05, $01, $00, $03, $02 ), { 90 }
    ( $02, $00, $04, $05, $01, $03 ), { 91 }
    ( $04, $03, $05, $01, $02, $00 ), { 92 }
    ( $02, $00, $01, $04, $05, $03 ), { 93 }
    ( $03, $04, $05, $02, $00, $01 ), { 94 }
    ( $02, $05, $03, $04, $01, $00 ), { 95 }
    ( $05, $03, $04, $02, $00, $01 ), { 96 }
    ( $02, $05, $01, $00, $03, $04 ), { 97 }
    ( $04, $03, $00, $05, $01, $02 ), { 98 }
    ( $01, $00, $05, $04, $02, $03 ), { 99 }
    ( $03, $04, $01, $02, $05, $00 ), { 9A }
    ( $05, $00, $03, $04, $01, $02 ), { 9B }
    ( $03, $05, $01, $02, $00, $04 ), { 9C }
    ( $04, $02, $00, $05, $03, $01 ), { 9D }
    ( $03, $00, $05, $04, $01, $02 ), { 9E }
    ( $04, $05, $01, $02, $03, $00 ), { 9F }
    ( $02, $00, $03, $05, $01, $04 ), { A0 }
    ( $01, $03, $00, $04, $05, $02 ), { A1 }
    ( $02, $00, $05, $01, $03, $04 ), { A2 }
    ( $04, $02, $01, $00, $05, $03 ), { A3 }
    ( $05, $03, $00, $02, $01, $04 ), { A4 }
    ( $02, $04, $05, $00, $03, $01 ), { A5 }
    ( $01, $00, $03, $04, $05, $02 ), { A6 }
    ( $05, $03, $01, $00, $02, $04 ), { A7 }
    ( $01, $05, $04, $02, $03, $00 ), { A8 }
    ( $02, $03, $00, $01, $05, $04 ), { A9 }
    ( $04, $02, $01, $05, $00, $03 ), { AA }
    ( $03, $00, $05, $04, $02, $01 ), { AB }
    ( $05, $04, $01, $00, $03, $02 ), { AC }
    ( $02, $03, $05, $04, $00, $01 ), { AD }
    ( $03, $02, $04, $05, $01, $00 ), { AE }
    ( $02, $00, $05, $04, $03, $01 ), { AF }
    ( $05, $02, $03, $00, $01, $04 ), { B0 }
    ( $03, $04, $05, $01, $00, $02 ), { B1 }
    ( $02, $00, $01, $05, $03, $04 ), { B2 }
    ( $03, $05, $04, $02, $01, $00 ), { B3 }
    ( $01, $00, $03, $05, $02, $04 ), { B4 }
    ( $02, $05, $04, $00, $01, $03 ), { B5 }
    ( $05, $04, $01, $02, $03, $00 ), { B6 }
    ( $04, $02, $05, $00, $01, $03 ), { B7 }
    ( $03, $05, $01, $04, $02, $00 ), { B8 }
    ( $05, $00, $04, $01, $03, $02 ), { B9 }
    ( $04, $02, $03, $05, $00, $01 ), { BA }
    ( $02, $03, $01, $00, $05, $04 ), { BB }
    ( $01, $05, $03, $04, $02, $00 ), { BC }
    ( $02, $04, $01, $05, $00, $03 ), { BD }
    ( $03, $02, $04, $01, $05, $00 ), { BE }
    ( $05, $04, $01, $00, $02, $03 ), { BF }
    ( $02, $00, $03, $01, $05, $04 ), { C0 }
    ( $05, $04, $00, $02, $01, $03 ), { C1 }
    ( $04, $02, $05, $00, $03, $01 ), { C2 }
    ( $05, $03, $00, $01, $02, $04 ), { C3 }
    ( $02, $05, $01, $04, $00, $03 ), { C4 }
    ( $04, $00, $05, $02, $03, $01 ), { C5 }
    ( $03, $04, $01, $05, $02, $00 ), { C6 }
    ( $04, $00, $05, $02, $01, $03 ), { C7 }
    ( $03, $02, $04, $00, $05, $01 ), { C8 }
    ( $01, $04, $05, $02, $03, $00 ), { C9 }
    ( $03, $00, $04, $05, $01, $02 ), { CA }
    ( $05, $03, $01, $04, $02, $00 ), { CB }
    ( $03, $04, $00, $01, $05, $02 ), { CC }
    ( $01, $03, $04, $05, $02, $00 ), { CD }
    ( $05, $00, $03, $02, $01, $04 ), { CE }
    ( $01, $02, $04, $05, $00, $03 ), { CF }
    ( $02, $03, $00, $04, $05, $01 ), { D0 }
    ( $05, $02, $04, $00, $01, $03 ), { D1 }
    ( $02, $03, $01, $05, $00, $04 ), { D2 }
    ( $01, $05, $00, $04, $02, $03 ), { D3 }
    ( $02, $04, $03, $00, $05, $01 ), { D4 }
    ( $01, $03, $00, $05, $02, $04 ), { D5 }
    ( $03, $00, $04, $02, $05, $01 ), { D6 }
    ( $01, $03, $05, $04, $00, $02 ), { D7 }
    ( $04, $05, $03, $01, $02, $00 ), { D8 }
    ( $02, $03, $04, $00, $05, $01 ), { D9 }
    ( $03, $02, $05, $04, $01, $00 ), { DA }
    ( $04, $03, $01, $00, $05, $02 ), { DB }
    ( $01, $02, $05, $04, $03, $00 ), { DC }
    ( $03, $04, $00, $02, $05, $01 ), { DD }
    ( $01, $03, $05, $04, $02, $00 ), { DE }
    ( $02, $04, $01, $00, $05, $03 ), { DF }
    ( $03, $05, $00, $04, $01, $02 ), { E0 }
    ( $05, $03, $04, $00, $02, $01 ), { E1 }
    ( $03, $04, $00, $05, $01, $02 ), { E2 }
    ( $04, $05, $01, $02, $00, $03 ), { E3 }
    ( $03, $04, $00, $05, $02, $01 ), { E4 }
    ( $01, $00, $05, $04, $03, $02 ), { E5 }
    ( $03, $02, $04, $05, $00, $01 ), { E6 }
    ( $01, $04, $05, $00, $02, $03 ), { E7 }
    ( $04, $00, $03, $05, $01, $02 ), { E8 }
    ( $03, $02, $00, $04, $05, $01 ), { E9 }
    ( $05, $03, $04, $00, $01, $02 ), { EA }
    ( $04, $02, $05, $01, $00, $03 ), { EB }
    ( $01, $05, $00, $02, $03, $04 ), { EC }
    ( $04, $02, $03, $00, $05, $01 ), { ED }
    ( $01, $04, $00, $05, $02, $03 ), { EE }
    ( $05, $03, $01, $04, $00, $02 ), { EF }
    ( $01, $04, $03, $05, $02, $00 ), { F0 }
    ( $05, $02, $04, $01, $00, $03 ), { F1 }
    ( $03, $00, $01, $02, $05, $04 ), { F2 }
    ( $05, $02, $04, $00, $03, $01 ), { F3 }
    ( $02, $04, $00, $05, $01, $03 ), { F4 }
    ( $01, $02, $05, $00, $03, $04 ), { F5 }
    ( $05, $04, $03, $01, $02, $00 ), { F6 }
    ( $01, $02, $05, $04, $00, $03 ), { F7 }
    ( $02, $04, $00, $05, $03, $01 ), { F8 }
    ( $04, $00, $01, $02, $05, $03 ), { F9 }
    ( $03, $04, $05, $01, $02, $00 ), { FA }
    ( $01, $00, $03, $02, $05, $04 ), { FB }
    ( $03, $05, $00, $04, $02, $01 ), { FC }
    ( $02, $03, $04, $01, $05, $00 ), { FD }
    ( $04, $05, $03, $00, $02, $01 ), { FE }
    ( $01, $04, $00, $05, $03, $02 )  { FF }
  );

  MHASH384_FIN: array[0..MHASH384_SIZE-1] of Byte =
  (
	  $13, $19, $15, $1B, $00, $11, $08, $1A, $2C, $1E, $12, $0D, $1C, $0E, $17, $26,
	  $14, $2A, $1D, $03, $20, $2D, $01, $1F, $05, $24, $06, $10, $2E, $27, $09, $07,
	  $16, $28, $0B, $04, $22, $0F, $29, $18, $21, $0A, $2F, $0C, $2B, $25, $02, $23
  );

{ --------------------------------------------------------------------------- }
{ UTILITY FUNCTIONS                                                           }
{ --------------------------------------------------------------------------- }

function Mix128To64(const u: UI64; v: UI64): UI64;
const
  KMUL: UI64 = UI64($9DDFEA08EB382D69);
begin
	v := (v xor u) * KMUL;
	v := v xor (v shr 47);
	v := (v xor u) * KMUL;
	v := v xor (v shr 47);
	Result := v * KMUL;
end;

procedure Iterate(const hash: PRowUI64; const p_xor: PRowUI64; const p_mix: PRowByte; const p_add: PRowUI64);
var
  i: Integer;
  temp: TRowUI64;
begin
  for i := 0 to MHASH384_WORDS-1 do
  begin
    temp[i] := Mix128To64(hash^[i] + p_add^[i], hash^[p_mix^[i]]) xor p_xor^[i];
  end;
  Move(temp[0], hash[0], SizeOf(UI64) * MHASH384_WORDS);
end;

function GetByte(const hash: PRowUI64; const idx: Byte): Byte;
begin
  Result := hash^[idx div 8] shr ((idx mod 8) * 8);
end;

{ --------------------------------------------------------------------------- }
{ PUBLIC FUNCTIONS                                                            }
{ --------------------------------------------------------------------------- }

constructor TMHash384.Create;
begin
  Reset;
end;

procedure TMHash384.Update(const value: Byte);
begin
  if finished then
  begin
    raise EInvalidOp.Create('Hash computation finished. Must call reset() first!');
  end;
  Iterate(@hash, @MHASH384_XOR[value], @MHASH384_MIX[rnd], @MHASH384_ADD[value]);
  rnd := rnd + 1;
end;

procedure TMHash384.Update(const data: TByteArray);
var
  i: Integer;
begin
  if finished then
  begin
    raise EInvalidOp.Create('Hash computation finished. Must call reset() first!');
  end;
  for i := 0 to High(data) do
  begin
    Iterate(@hash, @MHASH384_XOR[data[i]], @MHASH384_MIX[rnd], @MHASH384_ADD[data[i]]);
    rnd := rnd + 1;
  end;
end;

procedure TMHash384.Update(const data: TByteArray; const len: Integer);
var
  i: Integer;
begin
  if finished then
  begin
    raise EInvalidOp.Create('Hash computation finished. Must call reset() first!');
  end;
  if len < 0 then
  begin
    raise EInvalidArgument.Create('Length must not be negative!');
  end;
  if len > Length(data) then
  begin
    raise EOverflow.Create('Length exceeds the size of TByteArray!');
  end;
  for i := 0 to len-1 do
  begin
    Iterate(@hash, @MHASH384_XOR[data[i]], @MHASH384_MIX[rnd], @MHASH384_ADD[data[i]]);
    rnd := rnd + 1;
  end;
end;

procedure TMHash384.Update(const data: TByteArray; const offset: Integer; const len: Integer);
var
  i, limit: Integer;
begin
  if finished then
  begin
    raise EInvalidOp.Create('Hash computation finished. Must call reset() first!');
  end;
  if (offset < 0) or (len < 0) then
  begin
    raise EInvalidArgument.Create('Length or offset must not be negative!');
  end;
  limit := offset + len;
  if (limit > Length(data)) or (limit < 0) then
  begin
    raise EOverflow.Create('Length plus offset exceeds the size of TByteArray!');
  end;
  for i := offset to limit-1 do
  begin
    Iterate(@hash, @MHASH384_XOR[data[i]], @MHASH384_MIX[rnd], @MHASH384_ADD[data[i]]);
    rnd := rnd + 1;
  end;
end;

procedure TMHash384.Update(const data: PChar; const len: LongWord);
var
  i: LongWord;
begin
  if finished then
  begin
    raise EInvalidOp.Create('Hash computation finished. Must call reset() first!');
  end;
  if len > 0 then
  begin
    for i := 0 to len-1 do
    begin
      Iterate(@hash, @MHASH384_XOR[Byte(data[i])], @MHASH384_MIX[rnd], @MHASH384_ADD[Byte(data[i])]);
      rnd := rnd + 1;
    end;
  end;
end;

procedure TMHash384.Update(const str: String);
begin
  Update(PChar(str), Length(str));
end;

procedure TMHash384.Update(const stream: TStream);
var
  len: Integer;
  buffer: TByteArray;
begin
  repeat
    len := stream.Read(buffer, Length(buffer));
    if len > 0 then
    begin
      Update(buffer, len);
    end;
  until len <= 0;
end;

function TMHash384.GetDigest: TMHash384Digest;
var
  i: Integer;
  val: Word;
begin
  if finished then
  begin
    raise EInvalidOp.Create('Hash computation finished. Must call reset() first!');
  end;
  val := 256;
  for i := 0 to MHASH384_SIZE-1 do
  begin
    Iterate(@hash, @MHASH384_XOR[val], @MHASH384_MIX[rnd], @MHASH384_ADD[val]);
    rnd := rnd + 1;
    Result[i] := GetByte(@hash, MHASH384_FIN[i]);
    val := Result[i];
  end;
  finished := True;
end;

procedure TMHash384.Reset;
begin
  Move(MHASH384_INI[0], hash[0], SizeOf(UI64) * MHASH384_WORDS);
  rnd := 0;
  finished := False;
end;

{ --------------------------------------------------------------------------- }
{ CONVENIENCE FUNCTIONS                                                       }
{ --------------------------------------------------------------------------- }

class function TMHash384.Compute(const data: TByteArray): TMHash384Digest;
var
  instance: TMHash384;
begin
  instance := TMHash384.Create();
  instance.Update(data);
  Result := instance.GetDigest();
  FreeAndNil(instance);
end;

class function TMHash384.Compute(const data: TByteArray; const len: Integer): TMHash384Digest;
var
  instance: TMHash384;
begin
  instance := TMHash384.Create();
  instance.Update(data, len);
  Result := instance.GetDigest();
  instance.Free;
end;

class function TMHash384.Compute(const data: TByteArray; const offset: Integer; const len: Integer): TMHash384Digest;
var
  instance: TMHash384;
begin
  instance := TMHash384.Create();
  instance.Update(data, offset, len);
  Result := instance.GetDigest();
  instance.Free;
end;

class function TMHash384.Compute(const data: PChar; const len: LongWord): TMHash384Digest;
var
  instance: TMHash384;
begin
  instance := TMHash384.Create();
  instance.Update(data, len);
  Result := instance.GetDigest();
  instance.Free;
end;

class function TMHash384.Compute(const str: String): TMHash384Digest;
var
  instance: TMHash384;
begin
  instance := TMHash384.Create();
  instance.Update(str);
  Result := instance.GetDigest();
  instance.Free;
end;

class function TMHash384.Compute(const stream: TStream): TMHash384Digest;
var
  instance: TMHash384;
begin
  instance := TMHash384.Create();
  instance.Update(stream);
  Result := instance.GetDigest();
  instance.Free;
end;

end.
