#!/usr/bin/env python3

"""MHash-384 - Simple fast portable secure hashing library
Copyright(c) 2016-2020 LoRd_MuldeR <mulder2@gmx.de>

Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
and associated documentation files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or
substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import struct #required for unpack()

# ===========================================================================
# MHASH-384 CLASS
# ===========================================================================

class MHash384:

	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# CONSTANTS
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	__hash_words = 6
	__hash_size = __hash_words * 8

	__version_major = 2
	__version_minor = 0
	__version_patch = 0

	__vector_ini = \
		struct.unpack('<6Q', b'\xD3\x08\xA3\x85\x88\x6A\x3F\x24\x44\x73\x70\x03\x2E\x8A\x19\x13\xD0\x31\x9F\x29\x22\x38\x09\xA4\x89\x6C\x4E\xEC\x98\xFA\x2E\x08\x77\x13\xD0\x38\xE6\x21\x28\x45\x6C\x0C\xE9\x34\xCF\x66\x54\xBE')

	__table_xor = (
		struct.unpack('<6Q', b'\x37\x30\x4B\x41\x00\xDF\xDC\x01\xF8\x96\x8E\x1B\x66\xAF\xB3\xB1\x21\x31\x39\xDB\x73\x28\x4D\x94\x55\xE7\x2A\x66\x36\x9A\xDA\x73\xB1\x56\xCB\x4E\x8C\x31\x4F\x1F\xBC\xA5\x2A\x9C\xD9\x43\x97\xF0'), #00
		struct.unpack('<6Q', b'\x54\xC9\xBF\xCB\xC6\xBB\x1F\xA8\xDB\xED\x59\x89\x64\x43\xDE\x39\x2F\x82\x01\xDA\x0B\x1A\x64\x1A\x58\x26\x93\x66\x72\x60\x2E\xB5\x84\x20\x80\xAC\x31\x17\x5B\x2C\xD4\x9D\xC7\x1F\x67\x10\xEF\xC2'), #01
		struct.unpack('<6Q', b'\x46\x00\x81\x08\x8E\x8D\x2A\xCF\xF6\x68\xE2\x89\x20\x9B\x7E\x8B\xEC\x2D\x5D\x2C\x65\x61\x04\x93\xB9\xC8\xBF\x23\x22\xE4\x96\xF0\x70\x09\xFE\x4C\x96\x38\xD3\x8D\xBD\x60\xEC\x7C\x2F\x34\x9C\x26'), #02
		struct.unpack('<6Q', b'\x52\x05\xED\xD2\x20\xA9\x70\xB9\xE1\xA7\x4D\x25\x4A\x89\x0F\x01\xD8\x40\x1A\x9D\x5F\x20\x5F\x8D\x58\x5F\xFD\xC3\xCC\x3D\xC3\x50\x57\x2C\xD8\xE7\xBD\x31\x9F\xB4\x33\x90\x95\x2A\xF6\x04\xDE\x7C'), #03
		struct.unpack('<6Q', b'\xAC\x84\x9C\x2B\x82\xB2\xFA\x49\xF1\xBE\x8C\xD6\x8C\x64\xB8\x34\xDE\x26\x41\xBB\x47\x11\x12\xE5\x0F\xE0\xFF\xB2\x54\x1F\xB3\xC0\x32\xC6\x8C\x8E\xA3\x3D\x19\x2F\xFC\x8A\x9B\x42\x4B\x98\x8C\x05'), #04
		struct.unpack('<6Q', b'\x42\x99\x8C\x4D\xA4\x9D\x7F\xE0\xF2\x65\xCA\x9A\xB3\xF9\x2A\xBE\x95\xC2\x6D\x46\xB1\x8F\x3D\x5D\xC7\x62\xF9\xC3\x3E\x1E\x05\xC0\xB6\x2B\xCE\xD4\xCC\xC3\xFD\xF8\x37\x99\x19\xC6\xEF\xC0\x24\x90'), #05
		struct.unpack('<6Q', b'\xEF\xF5\x01\xEF\x77\x13\x11\xA3\xFF\xA3\x02\x6C\x36\x9B\xF5\x31\xDC\x16\x0D\x76\x49\x29\xB8\x61\xDF\x3B\xBD\x92\xAF\x58\xB9\xF6\xB2\xC0\x69\xEC\xEF\xAA\x97\x72\xCD\x84\x86\x9E\xAD\x50\x8A\xFE'), #06
		struct.unpack('<6Q', b'\xA0\x9A\xEA\x29\xC8\xD0\x3D\x3D\xA3\xC7\x96\xEC\x7E\x59\x77\x3E\x84\x16\x7F\x1E\x23\x38\xD0\x8F\xE0\xED\xB7\x1F\x13\x7B\x61\x64\x74\x58\x40\xC5\xE4\x9C\xC9\x85\xE3\xC5\x19\x1F\xD4\x35\x87\xA5'), #07
		struct.unpack('<6Q', b'\x9A\x1B\xF9\xCA\x8A\x62\x28\x80\x64\x70\xC9\x38\x05\x64\x4A\x19\x80\x16\x4F\x8A\x01\x5E\xA1\x04\x55\x09\x36\xC2\x04\x1B\xBE\xF4\x20\x1A\x58\xEF\x4C\x4D\xB2\xDF\x9D\x87\x15\xFD\xA0\xD0\x59\x3C'), #08
		struct.unpack('<6Q', b'\xBC\xF1\x8A\xE7\xEC\xE2\xF8\x88\xC2\x64\x73\x29\x22\x6E\x6B\xB4\xF3\x99\x6B\x92\x7F\xF1\x39\x93\x7C\xF0\xB8\xB3\x89\x34\x29\x31\xF1\xBC\xC9\x49\x16\xCE\x09\x39\xE9\xBC\x31\xFA\x3B\x10\x46\x0C'), #09
		struct.unpack('<6Q', b'\xA9\xE2\xEF\xC7\x8C\x0B\x1A\x4D\x71\xE8\x55\xAD\x8C\x87\xD7\xAF\x53\x41\xCF\x58\xC8\xFB\x9C\xC8\xD6\x75\x4F\xC7\x86\x94\x73\xC4\xF2\xC9\x0A\x13\x2C\x19\xF7\x0B\x9A\xBD\x81\x5E\xBC\xF2\x84\x00'), #0A
		struct.unpack('<6Q', b'\xCF\xCA\x4F\x01\x75\xE9\xFB\x4A\xE4\x41\xCC\xFA\x8C\xAF\xDE\x41\xE9\x16\xED\x53\x3B\xC2\x1E\x3C\x48\x32\x1C\x0F\xB3\x6E\xB0\x78\x72\x5C\x28\x04\x5D\x16\x11\xDD\xE4\x91\x94\x60\xB5\xD9\x46\x65'), #0B
		struct.unpack('<6Q', b'\x60\xD2\xC8\x4D\xE0\xE2\xD1\xFF\xCE\xEC\x74\x8A\x94\x86\xB1\x07\x31\xC5\x1A\x12\x45\x0E\x12\xB5\x85\x0C\x81\x30\x63\xC8\x0B\xBD\x30\x77\x48\xDB\xDD\xAF\x93\xDE\x96\x91\xB9\x6E\xDE\xB0\xDE\x78'), #0C
		struct.unpack('<6Q', b'\x55\xBA\xDE\x3F\xAC\xA2\xBA\x79\x7E\x56\x2A\xC9\xE1\xE3\xB7\xB1\x94\x07\x5B\x4A\xAB\x69\x3F\xA5\xDD\x6D\x4F\x95\x99\x74\xDA\xF0\x48\x22\x13\xE1\x89\x7C\x11\x58\x6E\x1A\xC5\x94\xF7\xF5\x5D\xD9'), #0D
		struct.unpack('<6Q', b'\x33\x1C\x09\x65\x0C\x8A\xBC\x89\xC2\x35\x31\x06\xEE\xEA\x4E\xB0\x89\xE6\xB6\x4A\xE0\x67\x32\xF5\xF2\x8B\x7E\x20\xB7\x79\xC5\xB4\xDF\xD2\x3C\x34\x51\x1E\xD3\x3F\xFE\x97\x81\x2F\x3E\x52\x9E\x11'), #0E
		struct.unpack('<6Q', b'\x2D\xF3\x60\x60\x1C\xCB\x10\x2E\x63\x6B\x4E\xA9\x32\xE7\x92\xBC\x67\x42\xD0\x1F\x83\x0D\x22\xF3\x7F\xE8\xBC\x14\x74\x5F\x2C\x50\x57\x24\x1D\xE9\x1D\x65\xE0\x89\x15\xD9\x82\x44\xB0\x56\x9E\x75'), #0F
		struct.unpack('<6Q', b'\x03\xE8\x84\x6A\x2C\xE3\xAE\x43\xE3\xD7\x0B\x2C\x20\x07\x50\x0C\x2A\xD3\x05\x42\x47\x64\xF4\xB4\x06\x6C\x38\xDE\x95\xFC\x17\x7D\x45\xA5\x7A\x56\x64\xBF\xDF\xE8\x6F\x04\x53\x58\x9D\x88\xBD\x55'), #10
		struct.unpack('<6Q', b'\x27\xDA\xB8\xEA\x14\xBE\x7A\x68\x44\xC3\x72\x0D\xB5\x3A\x7B\x39\x8D\x61\xDE\x1F\x6D\xAA\x5E\x50\x69\x3F\xA1\x5D\x86\x79\xBE\x61\x2D\x0E\xE9\xB5\x29\xAB\xBB\x17\x2F\x8B\xFA\x72\x19\x92\x0C\x01'), #11
		struct.unpack('<6Q', b'\x53\xF4\x6F\xF5\xA4\x23\x72\x8B\xE1\x6F\x8B\xB9\x5C\x7B\x1B\x29\x7D\xBF\x6F\x78\x25\x16\x42\xFD\xC8\xCD\xE8\xD0\x20\x10\x3D\xE3\x87\x9E\x73\x8B\x70\x0C\x53\xCA\x2F\xA1\xB8\x04\xF3\xF1\x8A\x87'), #12
		struct.unpack('<6Q', b'\x11\xD7\x6D\xE5\x24\x1F\xCE\x7E\xDF\xA6\x2F\x1B\x24\x69\x08\x2E\xFD\xB2\x59\xC4\xDC\x34\xB8\x84\x3B\x7D\x9E\x31\xA6\xED\x22\xE0\x28\x99\x3C\xA0\x8C\x9D\x83\x59\x74\xC7\xBB\x91\xF4\x90\x47\x64'), #13
		struct.unpack('<6Q', b'\x0D\xDD\xB0\xE6\x49\x6F\xDC\xCB\x32\x67\x34\x00\x8D\x2F\xBA\x44\x1C\xE6\x6A\x58\x21\xC8\xBC\x86\x5B\xE5\xCE\x85\x12\x49\xB7\xC7\x51\x38\x7F\xD9\xFF\x12\x39\xED\xBF\xBC\xBE\xBE\x86\x01\xAF\xF4'), #14
		struct.unpack('<6Q', b'\x16\xE5\xCC\x4E\xE5\x48\x8A\xCA\x06\xF5\xF8\xB7\x64\x18\xDF\xBE\xA4\x6A\xB3\xA4\x60\x1F\x6A\xD5\xD9\xD9\x0D\xFE\xB5\x5A\xB2\x49\xE1\xFC\x49\x5E\xBB\x7D\x37\x21\xE7\x99\x9D\xD5\xF5\x64\x8F\x70'), #15
		struct.unpack('<6Q', b'\x2C\xE0\xAF\x69\xA5\x73\xF8\xE9\x90\x73\x99\xCA\x05\xBC\x66\xDA\x85\x53\xD3\x56\x47\x17\x88\x8C\x30\xA7\xFD\xF4\x6C\xF1\xAA\xEA\x12\x35\x3A\x65\x55\x7A\x9F\xB3\x52\xE8\x23\xBC\x27\xB7\x0A\xF1'), #16
		struct.unpack('<6Q', b'\xAF\xE2\xBB\xC5\xF8\x6F\xE9\x93\x7C\xE9\x53\x02\x42\x85\x77\xA4\x4C\x07\x7D\xF7\xD1\x25\x4B\x70\x9C\x55\x70\x14\x3F\x09\xB2\xC0\xCF\xAA\x64\xE3\xD4\xFD\x5B\x8D\xF7\x1B\x5B\x3F\x67\x18\x05\x55'), #17
		struct.unpack('<6Q', b'\xCB\x1A\xE8\x5B\x2E\x83\xAC\x8D\xD8\x83\x75\x2B\x20\xFB\x8B\x58\xE4\xD7\xD1\xDF\x70\x8A\x4D\xA3\x3E\x03\x87\xB6\x27\x06\x3B\xD0\xCF\x75\xDC\xDB\x7E\xBE\xD0\xE3\xEA\x8B\x9F\xB0\xC5\xE8\x8E\x81'), #18
		struct.unpack('<6Q', b'\x31\x7D\x9D\x48\xE1\x02\x3B\x8E\x32\xB5\xE3\x64\x2B\x75\x36\x53\x65\x0A\x58\x5C\x79\xCC\xD4\xE4\x7E\x76\xED\x13\xF3\x08\xAB\x6D\xBF\x36\xBA\xFD\x88\x7E\x56\x8E\xD0\x33\xE9\xD8\xF1\x90\x94\x25'), #19
		struct.unpack('<6Q', b'\x62\xAA\x2C\xD6\xD0\x37\xB4\xEA\x03\xB0\x82\xB9\x28\xAD\x90\xC0\xEC\x04\x57\xD2\x68\xD7\x55\xE2\x43\x6A\x25\xF6\x1A\x51\x48\xA0\x5F\x8B\x49\x4D\x8F\x07\x9F\xE7\xFE\x57\xB3\x75\xFB\x35\x17\xC4'), #1A
		struct.unpack('<6Q', b'\x9D\xF2\x4F\x50\xC3\x9F\xDA\x70\x25\xFF\x73\x86\x38\x1F\xAB\xB9\xBA\x57\x73\xD1\x4C\x2F\x92\x36\x4A\xE0\xE7\x92\xB2\x3A\x9C\xF0\xEC\x13\xBA\xD9\xC3\x0B\xCE\x90\xE3\x8D\x91\x3C\xA6\x4C\x7C\x64'), #1B
		struct.unpack('<6Q', b'\x88\xD6\xEC\xD1\x8D\xA3\x34\xD8\x6A\xAB\x46\xDE\x20\x00\x7A\xA9\x1F\xA7\x2E\xE6\xF6\xC7\xDE\x9C\x7E\xC4\x4D\xD7\x6F\x5A\x8A\x28\xF1\x50\xCE\x65\x27\x2E\x4A\xD4\x3C\x4F\x9E\xDB\xB5\x50\x9B\xBB'), #1C
		struct.unpack('<6Q', b'\xE6\xF2\xDD\x41\x0E\xA9\x6D\xC6\x09\x5C\x0C\x16\x6F\xE8\x3F\x5A\xBE\xCC\x08\x51\x40\xF0\x6A\x6F\x7F\xFC\x27\xB6\x2E\x38\x38\xF9\x6C\x00\x7F\x61\x34\xD6\x3D\x16\x97\x84\x90\xDC\xFE\xB1\x84\x51'), #1D
		struct.unpack('<6Q', b'\x1C\xB2\xFC\x07\xED\x19\x57\xC9\xEB\xE7\x6E\x3F\x04\xDF\x12\x51\xCE\x34\x23\x01\x60\xFD\xF9\x50\x79\x65\xD9\x04\x51\xA8\x9F\x58\xA7\x05\x19\xD7\x44\x9E\x12\xB7\x8D\x52\x33\x07\x6E\x76\x14\x33'), #1E
		struct.unpack('<6Q', b'\xB8\x7C\x45\xC5\x14\x80\x6C\xDC\xB6\x69\x6A\x28\xCD\xFD\x35\xD6\xBF\x01\x7D\xE2\x2C\x23\x6F\xD6\x0F\xEC\x82\xF6\xC0\x4A\xAF\x56\xF5\x28\x43\xB6\x55\x1D\xDF\x57\xCA\xCA\x87\x2A\xC5\xD4\x1E\x65'), #1F
		struct.unpack('<6Q', b'\x02\xC5\xEE\x79\xCC\xD1\xD9\x26\xFE\xEB\xB1\x34\x0A\x34\xBF\x69\x74\x71\x39\x8E\xAC\xAA\xA5\xFA\x6F\xCF\x6B\x42\xBD\xF9\xA8\xD0\x52\x24\x6D\x4D\x46\x1F\x13\x5B\xDA\xD6\xD0\x60\x56\xD1\x2D\x12'), #20
		struct.unpack('<6Q', b'\x2F\xA2\x51\x5F\xEC\x9A\x38\x6E\xFB\x27\xC1\x54\x1C\x8F\xF6\x7E\x30\x5C\x48\xE0\x46\x4D\x6D\x98\x31\x8A\xCF\xE7\x39\x7B\xA4\xF0\x8F\xA7\xF2\xB7\xFD\x8D\x39\x1D\x2E\x0D\xB1\xFB\xD1\x51\xC6\x2F'), #21
		struct.unpack('<6Q', b'\xB0\xDC\x49\x1B\x8D\x4E\x4E\xA4\x43\xF3\xC2\x49\x20\x82\xA4\x07\xBA\x05\x65\x8D\x4A\xC0\x0A\xC4\x6C\xB1\x29\x07\x3F\x1D\xB9\xD9\x15\x90\x0F\xB5\x51\x99\xF3\xAA\x76\x90\x3F\xAD\xD3\xF5\x6E\x96'), #22
		struct.unpack('<6Q', b'\x09\xD0\xB6\x0E\xAC\xCB\x78\xEA\xAB\xDE\x23\x5A\x72\xA6\xFE\xA0\xD9\x94\xCB\x44\x74\x9C\x72\xCE\x0D\xAA\x27\x66\x62\x94\xA9\x40\xD5\x18\xD0\xE3\xCD\x38\xF7\x37\xC5\xB3\xCD\x01\x1C\x49\x29\x4C'), #23
		struct.unpack('<6Q', b'\x7A\xC8\x45\xA7\xAE\x92\x97\x7C\xD4\x8F\xBD\xC5\x20\x56\xFF\xD1\xB9\x56\x4B\x00\xE3\x84\xCA\x9E\xD6\xB1\x28\xC2\x23\x39\xFD\x5A\x3B\x28\xFD\xB3\x9E\xF7\xDB\xE5\x9F\x4B\x08\x54\xE3\x12\x17\x44'), #24
		struct.unpack('<6Q', b'\xBC\x49\xE2\xD8\x8B\x28\x5D\xE3\xE5\x66\xA3\x53\x14\x6C\x77\x91\xEC\xEF\x95\x4E\x68\xE1\xD1\xF5\x06\xF6\xDD\xD7\x17\xE1\x08\x91\x8C\x7C\xCE\xA2\x9D\x0F\xB3\x81\x0D\x8C\x81\x86\xD7\xDB\x97\x64'), #25
		struct.unpack('<6Q', b'\xE0\x4B\x04\x45\xFF\x91\xF8\xC2\xEB\xE2\x87\x2B\x6D\xA7\xA1\x75\xBF\xC2\xF3\x8A\x79\x65\xCE\x85\xDC\x09\xFD\x4E\x2B\x53\x3F\x21\x88\x6C\x3A\xA5\xF5\x1D\xAA\x0D\xB2\x26\xD8\x50\x6A\x60\x28\x30'), #26
		struct.unpack('<6Q', b'\x3A\xF3\x79\x63\x61\x62\x9A\x60\x6D\x51\xE4\x53\xBC\xA3\x39\xA3\xEC\xDB\x5A\x6A\x61\x92\xAD\xD7\xAA\x24\xE9\x86\x6D\x72\x43\xD0\x65\x98\xC2\xF4\x64\xB5\x55\x85\xB0\xD6\xC1\x31\xAB\x12\xAA\x56'), #27
		struct.unpack('<6Q', b'\xFF\x7B\xC1\xC1\x85\xED\xD9\xCE\xC2\xFF\xBA\xAC\x25\x23\x52\xEB\x94\x53\xD1\xB2\xF4\xD8\xD3\x04\xEA\x6E\x75\x04\x4C\x50\x71\xD2\xF1\x27\xF8\x1A\xA9\xA4\xDB\x2D\xE4\x02\x80\x8F\xC2\xD5\x67\x1F'), #28
		struct.unpack('<6Q', b'\x8F\xC0\x2D\x43\xAD\x08\x9C\x8B\xC7\xBB\x96\x97\xE2\x43\x75\x5A\xDE\x67\x19\x3C\x5B\xDB\xA6\x34\xE4\x4E\x80\xA2\xC2\x3B\x6E\x01\x75\x2F\x17\xE5\xCC\xCA\x9B\x5B\xBA\xBD\xAD\x80\x8B\x59\x49\x75'), #29
		struct.unpack('<6Q', b'\x55\xD1\xCD\xC5\x17\xB1\x3F\x6F\xC5\xEE\xD6\x9C\xA5\xB0\xC3\x16\x69\x87\x53\xDE\x11\xA4\xA1\xD9\x7C\xC3\x4A\x9F\x97\x54\x8C\x93\xDB\x84\x52\xD5\xC1\xBC\x37\x37\xB7\x6B\x15\xF9\x8A\x9F\xAD\x6C'), #2A
		struct.unpack('<6Q', b'\xDF\xEB\x75\x9C\x4F\x28\xBF\xEB\xE8\x3D\x75\x06\xB4\xEB\x83\xB3\x3C\x40\x7C\xEE\x7A\x12\x86\xAA\x27\xE0\x3D\x52\x10\xDD\xBF\x10\x13\x8A\x4A\xEB\xC4\xF6\x8B\x13\x67\xB0\x78\xDB\x7D\xF6\xEF\xB1'), #2B
		struct.unpack('<6Q', b'\x5E\xDA\xA3\x7A\x8D\x13\xD1\xF6\x71\x63\xB6\x8F\x8D\x09\xA8\xBA\xF2\xAA\x6C\x8B\x1B\x6D\xE7\xDA\xDC\x1E\x8D\x36\x34\x10\x0F\x40\x77\xD2\xE8\x72\x51\x7F\x93\x7C\xEF\xE6\xAD\x3C\xF8\xBB\x05\x7D'), #2C
		struct.unpack('<6Q', b'\x81\xB0\x34\xCF\xA6\x2E\x9C\x0E\xA0\x0B\xF6\x58\x0D\xB3\x36\x90\xCA\x8B\xF0\x48\x58\x2C\x3A\xDB\x64\xD5\x50\x42\xB9\xD1\x7A\xC8\x66\x61\xF9\xEE\x09\x2E\x89\x7C\xF3\x85\x10\x57\xCF\x85\xDB\x26'), #2D
		struct.unpack('<6Q', b'\x2A\xC1\x18\x87\xF5\xE3\x1E\x25\xE4\x2A\x8A\x17\x81\x8D\x43\xF9\xA8\xA8\x39\x90\x88\x9A\x92\xF0\xFD\xCC\xBD\x5E\x22\x65\x6B\xF0\xC6\x73\xBF\xF7\xED\x14\x4D\x2E\xCF\xFA\x1D\xBC\x95\x98\x36\xA9'), #2E
		struct.unpack('<6Q', b'\xCB\x79\x69\x1D\xB4\x02\xE3\xCA\x23\xE6\x1E\xB5\x58\x5A\xFA\xBB\x2F\xB5\x1A\xC8\x9D\xB9\x13\x51\x56\xA0\x17\xEC\x5B\x79\x93\x60\x5E\x35\x5E\x2E\x4D\xFB\x71\x8F\xF2\xA2\x34\xBA\xED\x92\x2F\x76'), #2F
		struct.unpack('<6Q', b'\xFF\xD9\xA4\x65\x52\x01\x30\xD1\xC8\x26\x1F\xD7\x53\xA2\xBE\x09\xE6\x21\x65\xC4\xED\xEA\xB6\x81\xA9\xB8\x82\x56\x16\x68\xE2\xFA\x23\xB6\x4A\x77\xC4\x3E\x9C\xA8\xB2\x9D\x21\x55\xE0\x45\x2E\x0D'), #30
		struct.unpack('<6Q', b'\xAB\x92\xA6\xC3\x84\x02\x56\x2B\xB8\xA7\x79\xB3\xD0\x8A\x00\x37\xFC\x0B\xF9\x30\x2C\xCD\x11\xAF\xED\x62\x27\x0F\x25\x7A\xE8\x7F\xC3\x42\x72\x64\x11\xD7\xFB\xC3\xBD\x22\xB3\xF6\x64\x82\xED\x74'), #31
		struct.unpack('<6Q', b'\x43\x39\xAD\xA7\xC8\x5C\x19\x28\xC4\x4F\x4C\x46\x08\xE8\xCB\x53\x26\x87\x5F\x76\x7A\x3D\x8E\xD5\x4F\xAA\x85\x01\xF6\x52\x30\xD8\xE7\xB4\x3B\x22\x85\x0D\xCB\xEF\x28\xAC\x7F\x78\x5E\x30\x31\x5A'), #32
		struct.unpack('<6Q', b'\x43\x95\xF1\x30\xE2\x0E\x5D\x72\xE0\x34\xDF\xBD\xC6\xD2\x91\x90\x01\x46\x75\xC2\xC6\x49\xBE\xE3\x6B\x56\xAD\xA4\x0B\x30\xBE\x61\xC1\x7C\xFA\x51\x75\x9D\xD7\x02\x58\xDA\x1C\x5F\x0F\x91\x43\x65'), #33
		struct.unpack('<6Q', b'\x3F\xB4\x81\x44\xC4\xAD\x99\x40\x81\xEB\xD9\x2F\x92\x61\x13\xFE\xAC\xCE\x1F\x44\x9E\xC0\x89\xA9\x19\x90\xCB\xD3\x13\x3A\x9B\x44\x34\x11\x20\x6F\x39\xBE\xA9\x45\xFE\x33\x66\x04\x5A\xD0\x1A\xDC'), #34
		struct.unpack('<6Q', b'\x69\x3E\x2F\x52\x6D\x3A\x56\x1A\x9E\x5A\x47\x9F\x07\x9E\x58\xBE\x38\x00\x4C\x8E\x63\xA9\xA2\x75\x62\x03\x7A\x57\x02\x62\x3B\xDA\xF6\x7A\x72\x0D\x1E\x3F\x1D\x21\x33\x92\xD9\x9A\x52\xFC\x1F\x5E'), #35
		struct.unpack('<6Q', b'\x1B\xD0\xD6\xC6\x86\x1E\xB6\x47\x18\xC3\xAD\xAD\x83\x6F\x7D\x43\x8C\x73\xED\x8D\x02\x61\xA3\xD5\x4B\x16\x25\x04\x63\x4C\x0D\xA0\xD2\x9D\x4C\xAF\xA5\xAF\x69\x1A\x82\xB5\x51\xF9\x67\x1C\x9E\xF9'), #36
		struct.unpack('<6Q', b'\x79\xEA\xBD\xB6\x40\x77\x6A\xA6\xA3\x80\xAF\x96\x14\xFF\xF7\xFE\x8C\x89\xCD\xEA\x3E\xD4\xAF\x05\x34\x71\xAD\x31\xED\x78\x0C\xB0\x73\x66\x84\xD7\x1A\x1A\xD3\x0E\xBE\x99\x14\x16\x44\x68\xB9\x74'), #37
		struct.unpack('<6Q', b'\x2E\xD1\xF6\xCB\x6C\x8D\xFA\x46\x56\x39\x30\x7F\x14\xFC\xC2\x31\x7F\x06\x5F\xDE\x01\x44\x7F\x70\xE9\x94\x35\xE3\xC7\xFE\xE5\x3A\x14\x17\x53\x63\x8A\x9F\xE3\x28\xB2\xCA\x9F\x1E\xEA\x29\xB3\xB7'), #38
		struct.unpack('<6Q', b'\x04\x78\x5C\x82\x55\xF7\xD8\xEF\xD1\x0C\xD3\x0B\x87\x93\x5A\x1F\x16\x87\xFF\x71\x46\x89\xBF\xEF\x58\xDA\x2B\xF2\x7F\x61\xED\x28\x2E\xB6\x5C\xAE\xCC\x89\x12\x41\xF9\xF2\x01\x18\xF4\x42\xDD\x95'), #39
		struct.unpack('<6Q', b'\xE1\x59\xFC\x45\x86\x5B\x52\xA8\x0C\x7F\x5F\x0A\xC0\x2D\xE6\x75\xAC\x16\x04\x21\x85\x67\xC5\x09\x26\x76\x0D\xB3\xE9\x76\xEF\x50\x65\xD6\xF5\x19\xDC\x2C\x3B\x2B\xFF\xF4\xD8\x11\xCD\x97\x12\xA4'), #3A
		struct.unpack('<6Q', b'\x39\x50\xEE\x9E\x64\x9A\xC9\xEA\x65\x00\x3C\x14\x2F\xC9\x93\xA5\x06\x12\x07\x03\x52\x73\x14\xB3\xEC\xA4\x64\xC7\xA0\x61\x27\xEA\xB3\x25\xAC\x6C\xD4\x7F\xAA\x02\xBF\x03\x6D\xA9\x82\xC1\x8C\xC6'), #3B
		struct.unpack('<6Q', b'\xA8\x3C\xC8\x4E\x02\x3F\x87\xB2\xEB\x53\x88\xFD\xB8\x0A\x47\x97\xBD\x05\xB3\x59\xC1\x15\xFE\x18\xFD\xEA\xEA\x87\xF6\x08\xAB\xB0\x43\x2E\x60\x73\xDE\x3F\x0A\x51\xF0\xFC\xF0\xCC\x4D\xB8\xE1\x03'), #3C
		struct.unpack('<6Q', b'\xD8\xC0\x33\x80\xDC\xBB\x5B\xD8\x4F\xF3\xA9\x9C\xC3\xD9\x23\x92\xFD\xC3\x63\x5B\x6D\xCB\x3B\x7D\xB5\x0F\x0C\xDA\x74\xF9\x30\x1C\x43\x51\xFB\xBE\x9E\xBC\x24\x8B\xFC\x84\x7B\x5B\x92\x54\x89\xC5'), #3D
		struct.unpack('<6Q', b'\xA7\xD7\x44\x08\x2E\x7C\xBD\x6A\x0D\x53\xDF\x6C\x45\xEA\xF2\xCC\xB8\x21\x39\x2B\xF5\x8C\x93\xE8\x7C\x65\x81\xF2\xA2\x3C\x02\xBA\xAE\xED\xD1\x75\xA6\x5D\x63\xEC\x6C\xBE\xE1\x2E\xF2\x52\xAA\xB4'), #3E
		struct.unpack('<6Q', b'\x04\x59\xCB\x77\xC6\x3A\x1C\x98\x49\x7B\x87\x84\x4C\xB5\x92\x6A\x15\x58\xC5\x40\xBB\xA6\x5B\x74\xED\x71\xA3\x22\x0D\x55\xAF\xB7\x74\x53\x5F\xC6\x87\xBD\xE8\xD5\xF5\x38\xF5\xF0\x37\x4A\x87\x67'), #3F
		struct.unpack('<6Q', b'\x21\xC0\xEC\x9D\x2A\xBA\x3B\xC2\x50\x34\x0E\x0B\x93\x0E\x61\x4E\x7E\x57\x77\x14\xA9\x1A\x68\x1A\x76\xC3\xED\x14\x97\x20\xA3\x38\xB6\x4A\xEB\xEE\x63\x55\xD1\x0F\x78\x21\xD4\x01\x8A\x66\x57\x7D'), #40
		struct.unpack('<6Q', b'\xB5\x98\xA0\x45\xE1\x8C\xF8\x6A\x46\x8B\x8B\xD8\x8C\x85\xEB\x1A\xE8\xD6\xE2\xB8\xAF\x33\xB7\xE8\xFA\xEB\xA7\x10\x8C\xAA\x3F\x31\xEA\x7C\x55\x77\x5E\x37\x7D\x12\x55\x21\x2F\x0B\xF7\xA2\xBD\x96'), #41
		struct.unpack('<6Q', b'\x36\xB6\xAF\x8F\x97\x03\x89\xEC\x63\x97\x07\x5C\x42\x3C\x21\xC7\x7C\xD1\xB6\x6A\x03\x84\x03\x76\x49\x1F\x5F\x38\x26\x3A\xC6\xE0\xF5\x6D\x1A\x81\xD6\x77\x98\x29\x8D\xB8\x04\x53\xFC\x90\x6F\x87'), #42
		struct.unpack('<6Q', b'\x16\xBA\xE0\xD6\xC2\xBB\xFA\xA6\xBC\xE6\x80\x00\x64\xC9\x70\x9B\x7B\xB2\x98\x55\x26\xD5\xB2\x29\x7E\x39\xE4\x26\xC7\x57\x96\x4A\xD5\x78\x66\x76\xC6\xCC\x01\xA8\x98\x99\x61\x72\xCC\xF7\x0E\x80'), #43
		struct.unpack('<6Q', b'\xBF\x90\x54\xCF\xA8\x31\x59\x23\x9F\x87\x8F\x0E\x8E\xF9\x98\xE7\xA7\x0C\xF3\x38\x9C\xE2\xEE\xC6\x24\x00\x3E\xD5\xF2\x79\x9A\x92\x45\x7A\x58\x49\x27\xE1\xF2\x88\x65\x19\x89\x38\x8F\xB2\x85\x0B'), #44
		struct.unpack('<6Q', b'\x27\xD8\xA4\xE4\x03\x03\x5E\x16\x6A\x43\xE8\xD1\x42\x4F\x99\x67\x1B\x0D\x13\x6E\xCF\x8B\xCC\xE6\xFC\xDE\x9D\x70\x11\x17\x10\x50\x28\x53\xD0\x0C\xC4\xDE\x3B\x37\xF6\x09\x51\xAA\xA4\x74\xB2\x40'), #45
		struct.unpack('<6Q', b'\xC8\xF8\xFD\x08\xA0\x8B\xF8\xA9\x5A\xE0\x6E\x47\xE3\x97\xC8\xEC\x74\x7A\xD5\x69\xAB\x90\xE2\xBC\x5D\x11\xE3\x11\x18\xDB\x44\xFA\x88\x0C\x48\x64\xFD\xAE\x67\x62\xEB\xEC\x3A\x2D\x4A\xD0\x97\x26'), #46
		struct.unpack('<6Q', b'\x13\x33\xCD\xAB\xF2\x2A\x78\xC0\x73\x62\xF9\xF2\x90\x12\xBA\x02\xB9\xC2\xAD\x56\x1A\x2F\xC8\x63\xC4\x51\xFE\x3E\xC0\xE8\xF8\x10\xFD\xAF\xCC\x25\x86\x34\xEB\xE3\xAE\xE7\xB8\x9C\x96\x07\xD6\x93'), #47
		struct.unpack('<6Q', b'\xBC\x8F\xE5\x43\x94\x17\x6E\xCC\xEE\xB8\xA7\x55\xC6\x93\x1C\xD2\xA4\x8B\x2B\x1F\xA3\x34\x98\x2B\xEE\xEC\x25\x60\x51\x69\x3B\xC8\x94\xAE\x7A\x42\x7B\xEB\x76\x91\x6E\xA7\xB7\x30\x9E\x5B\xB6\x8C'), #48
		struct.unpack('<6Q', b'\x89\xD9\xED\xD6\x0A\x3A\xA3\xC1\xF7\xB5\x13\x58\xD9\xC5\xB3\x18\x8B\x9A\x35\x3B\x26\xBD\x24\xB0\x50\x9B\xA9\x16\x22\x7C\xC1\xC8\x29\x77\x23\x58\x1D\xA1\xF9\x71\x90\x42\x28\x18\x76\x7C\xA6\x3A'), #49
		struct.unpack('<6Q', b'\x7B\x1C\x20\x09\x5E\x46\xB7\x99\x81\xFA\xA4\x1C\xA0\x9F\xF8\x9A\x23\xD1\x1A\x76\x3E\xC6\x2E\xFC\x14\x7D\xB1\x85\x95\xA3\xA9\xE2\x81\x4E\xF9\x29\xE5\x4D\x39\x08\xA4\xFA\x94\x97\xE6\x48\x94\x47'), #4A
		struct.unpack('<6Q', b'\xBB\xCA\xBD\x4C\x1C\x3D\xCA\x23\xE4\x37\x1A\xCE\x36\x54\x26\xE3\xC9\x4C\x8A\x9E\xF6\x10\xBF\x1B\x4D\x5C\x8F\x04\x08\x67\xA6\x05\xFE\xFE\x5B\x9C\xDD\xDC\x59\xE2\xFD\x6E\x93\xFD\xFA\x65\x9E\x43'), #4B
		struct.unpack('<6Q', b'\x9C\x71\x8F\x97\xB6\x73\x4D\xA2\x8E\xBB\xB0\xCC\x43\xF3\x53\x3F\x6A\x7C\xE0\x9E\x76\x72\x3C\xBE\xDD\x58\xF5\x9C\x53\x9E\xCB\xFA\x6A\x98\xDE\x30\x4E\x1D\xB9\x67\x3A\x91\x98\x16\xD1\x13\xB9\x1D'), #4C
		struct.unpack('<6Q', b'\xC6\xC3\xC3\x0D\x14\x4E\xBD\x98\xE8\x63\x32\xBF\x92\x15\x2B\x14\x0E\x5B\x09\xED\x59\xAC\xBE\xCD\x6A\x89\x25\xF6\xF0\x63\x07\x90\x39\x4E\x32\x30\x0F\x55\x13\xE2\x95\x31\x80\x7A\x41\xA4\x13\x8A'), #4D
		struct.unpack('<6Q', b'\x73\x60\x62\x8C\xED\x98\xCD\x2A\x38\x32\xCF\xC4\xB4\xA6\xAA\x1C\x5D\xEB\x77\xB6\x1E\xB4\xDC\x04\x68\x5B\x10\xA8\x44\x58\x8B\xF8\x4D\x1A\x06\x1A\x95\x9E\x1D\x98\xDB\x8E\x87\x4C\x89\x71\x94\xBC'), #4E
		struct.unpack('<6Q', b'\xBD\x2D\x6C\x5D\xAD\xFE\x59\x49\x49\x30\x50\x8C\xE2\x59\xBD\x6A\x34\x8B\xAF\x4C\x49\xC5\xD2\x06\x3C\x29\xA4\x04\x13\x54\xE4\x70\x03\xF5\xF2\xCA\x16\x34\x0F\x52\x85\xDB\x13\x26\xD9\x09\x3D\xB2'), #4F
		struct.unpack('<6Q', b'\x91\x17\x2D\xC3\x15\xA8\xB5\x26\xC6\x33\xB0\x5B\x55\xE7\x99\x2C\x4F\x51\x02\x00\x6A\x9D\xCE\x09\x97\x79\x8D\x2B\x2B\x28\x85\xD4\xC6\x14\x4A\x4A\x2F\x79\x5B\x9C\xE7\xB4\x0B\xDC\x02\x9D\x1D\x85'), #50
		struct.unpack('<6Q', b'\xEC\x60\xB0\xCF\xCA\xB6\xFE\x62\xEA\x61\xC6\xD5\x69\x7D\x97\x9D\x56\x15\xD8\x06\xD8\xEF\x08\xBF\x18\x57\xEA\x60\xA4\xEE\xF1\x25\x65\x96\x5A\x1F\xB5\x46\x53\xA2\x74\xA2\x8C\x35\xDC\x9A\x2F\xD9'), #51
		struct.unpack('<6Q', b'\xA6\xFF\xE8\x63\xFC\x3D\xE6\x27\xA3\xFD\x99\xCE\xE2\xCC\xB9\xCD\x0D\x83\x74\x49\x75\x5B\x9D\x97\xBE\x93\x66\x7D\x40\xC8\x98\x32\xB7\x42\x9B\xA3\xAD\x5F\x9D\x62\xE1\x84\xCD\x71\x12\xD3\x54\x26'), #52
		struct.unpack('<6Q', b'\x3C\x58\x6E\xF6\xDA\xA4\x1F\xAB\x66\x47\xD2\x36\xA2\xB7\xB6\xEE\xB3\xC6\xF5\xFD\xCD\x38\x07\xA9\x4C\x2D\x8E\x64\xE5\xA9\xCB\x28\xA3\x0D\xFE\x0C\x6C\xBF\xE5\xFD\x85\x84\xD7\xD7\x63\xB8\x00\x9D'), #53
		struct.unpack('<6Q', b'\xAA\x16\xEA\x4E\x09\xBF\xFB\x75\xF2\x29\xD7\x7C\x02\x1F\x93\x48\x7F\x2E\x9B\x00\x79\x06\x36\x5D\x1A\xE2\x3D\xBD\x48\xD1\xFC\xDD\xE1\xD0\x94\xB0\x44\xF5\xBF\x4D\x22\x2C\x35\x94\x62\x5C\x0E\x9C'), #54
		struct.unpack('<6Q', b'\xD2\x53\x88\x96\xFF\x27\x3A\x28\xF2\x03\x0D\xEA\x6C\x0C\x96\xB0\x38\xDB\x73\xA4\x07\xBA\x2B\x17\xBB\xF4\xE6\x96\xD2\x87\x8C\x68\x0F\xCF\x68\x5D\xBC\xE9\xB7\x5C\xB4\xBF\x47\x0E\x1B\xD7\xA5\x57'), #55
		struct.unpack('<6Q', b'\x2F\xFF\xE4\xC1\xAA\x08\x01\xDE\x99\x2B\xC6\xEA\xAB\xCF\x46\xD3\x08\xF6\xB5\x98\x3F\x20\x2E\xB7\xBE\x29\x4B\xA5\x8C\x3D\x85\x81\x80\x16\xAA\x9F\xC8\xD7\xAE\xA6\xED\xD7\x39\x5C\x15\x3B\x09\xD2'), #56
		struct.unpack('<6Q', b'\x84\x4B\x97\xD4\x99\xAC\xAE\x0B\xDA\xB4\xC9\x99\xA6\x58\xF2\xC7\xC1\xF5\xCC\x4A\x5E\x2C\x62\x6F\xAA\xBE\x81\x97\x7D\x39\xAB\x58\xE3\xFF\x01\xE1\x67\x1F\x81\xBF\xF3\x0E\x3C\x1C\x88\xC2\xBC\xAF'), #57
		struct.unpack('<6Q', b'\x3E\x6C\x8D\x51\xFB\x11\xB2\x26\x4A\x78\x10\x1A\xD5\xDA\xBA\x64\x6C\x18\x87\xA5\x06\x4E\xBE\xE6\x5C\xCD\x43\x13\xC6\xF5\x71\xD4\x5D\xED\xAA\xD6\x0D\xBB\x89\x83\x7D\xA1\x14\x89\x67\x12\x81\xC8'), #58
		struct.unpack('<6Q', b'\x7D\xD2\x88\xBB\xC3\x0B\x2D\x2B\xD2\x7A\x51\xFF\xFA\xD1\xA7\xC5\xAD\x82\x9F\xA0\x56\x90\xF3\x96\x4E\x4D\xED\x6C\x1A\xA6\x38\xFB\x64\xB2\xF9\xA6\x4E\x8E\x30\x9D\xB3\xC6\xEC\x4A\x29\xCE\x97\x90'), #59
		struct.unpack('<6Q', b'\xA2\xB1\x90\x06\x95\x2B\xCA\x8F\xDF\x22\xD4\x03\xBF\xFC\x3E\x29\xAB\x53\x63\xE7\xB3\x25\x91\x8C\x73\x01\xA7\xA1\x92\x20\x40\x3D\x76\xF6\x9B\xAB\x4C\x97\xAB\x9B\x6E\x58\x8C\x5D\xC5\xFC\xA8\x5E'), #5A
		struct.unpack('<6Q', b'\x8C\xEC\xE1\xC2\xE8\x92\x8C\x40\x50\x43\x1B\xB7\x14\xC9\xF4\x4A\x69\x10\xFB\xCD\xE0\xAE\x86\x51\x67\x7C\x65\xB9\xFA\xEA\x85\x23\x80\xCA\x98\xC8\xD3\xE4\x08\xF7\x7E\x90\x84\x98\xF8\xB9\xC8\x1E'), #5B
		struct.unpack('<6Q', b'\x78\x18\x2C\x6A\x8B\x95\xE8\x46\x47\xA6\x78\x0F\x41\xFD\x72\x21\x04\x90\x29\x3A\xD8\x9D\x8E\x9D\x25\xD0\x5A\x26\xC3\x13\x09\x39\xF1\xCB\x77\x30\xE2\xF1\x31\xD2\xF3\xD7\x80\x4E\x57\x3E\xEE\xE7'), #5C
		struct.unpack('<6Q', b'\xB2\x40\x5E\xD8\xA3\x67\x85\x5A\x73\xCF\x3F\x13\x61\xF1\xEC\x16\xD7\x0D\x3C\xBA\x6F\x5C\xDA\x52\xFB\x4B\xB3\xDE\x83\x79\xE5\x56\x53\xD1\x68\xB7\xDC\x4F\x25\x83\x2D\xB8\xC6\x35\x5F\xF9\x14\x9A'), #5D
		struct.unpack('<6Q', b'\xE6\x4A\x9D\xE1\xC6\x29\x8A\x49\x11\xBA\x27\x60\xF4\xAA\xF4\x2E\x05\x95\xF3\x84\xAA\x7D\xBA\xBD\x4D\x94\xDC\xF6\x04\x2A\x0B\x94\x53\x0D\xFC\x10\x56\xD3\x7E\x4E\xE1\x59\x7E\x90\xC2\x94\xDD\xBA'), #5E
		struct.unpack('<6Q', b'\x80\x5F\x47\x3F\xC4\x0F\xDF\x14\x2F\xF8\x4B\x26\x8D\xAA\xE2\x17\x4D\x93\x8B\xE5\xDF\x5B\x62\x92\x81\xEA\xAC\xA4\x15\xF4\x84\x83\x2B\x64\xD8\xC5\xAE\x5E\x9C\x8E\x87\x66\x82\x1C\x5F\xF5\x8E\x4D'), #5F
		struct.unpack('<6Q', b'\xD9\x28\x71\xF7\xC4\x35\x23\x4A\x94\xBA\x9A\xD2\x76\x14\x4E\x54\x44\x50\x78\x21\x63\xC8\x4E\x65\x5A\x44\x80\x2F\xB0\xD9\x4A\xB0\x20\xD0\xC8\x80\x64\x1B\xE0\xB0\xBF\xCB\xA3\x88\x5E\x32\x6E\x59'), #60
		struct.unpack('<6Q', b'\x62\xD0\x48\x74\x15\x55\x69\x89\x68\x6D\x23\x0F\x4C\x8C\xB0\x0D\xA2\xC4\xD1\x3C\x5B\xFC\xA8\x3B\x5B\x53\x44\xE1\x53\x7C\xF5\x04\x40\x68\xE4\x7B\xCC\x4D\xD0\xB7\x46\x46\x33\x92\x31\x99\xBE\x4B'), #61
		struct.unpack('<6Q', b'\x27\xCE\x02\xAB\xE6\x37\x78\x1D\xA4\x3E\x49\xD4\xAE\x5B\xA3\x3E\xBE\xFA\x94\xDF\xB5\xFD\xCA\xD1\x4F\x0C\x17\x62\xBB\x80\xB5\x98\x43\x1C\x42\xA9\x6C\x7A\xC5\xC3\x34\x16\x20\xC1\xC2\x5F\xD6\x68'), #62
		struct.unpack('<6Q', b'\x36\x75\x71\x48\xBC\xBA\xEA\xFA\x15\x73\x2F\xF6\xE8\x51\x42\x45\x23\xC5\xCD\xFD\xA7\xE8\x18\xB3\x44\x13\xC9\x13\x20\x83\x2E\x7C\x52\x90\x69\xD1\xAF\x5D\x9E\x4D\xA7\x37\x05\x87\x8C\x2E\x26\x12'), #63
		struct.unpack('<6Q', b'\xA2\x02\x84\xEF\x0B\x5D\x3E\x8A\x09\x99\x01\xFA\xFA\xC5\x3B\xA3\xF5\x62\x07\xD0\xAC\xE8\xCB\x63\x78\x41\x98\x81\xF1\xA3\x26\xEA\xBB\xF6\x4A\xBB\xD1\x78\xEB\x6E\x5F\x84\x0E\x30\x71\x96\xCF\x7E'), #64
		struct.unpack('<6Q', b'\xFC\xD0\xF5\xCC\x7C\xB6\x11\x08\x6B\xBD\x96\x34\x3F\xAB\x8C\x9F\x2D\x5C\x35\xF1\x24\x7D\xCB\x57\x80\xDE\x5B\x16\x94\x85\x21\x58\xF8\x78\x3A\x65\x8A\x36\xF3\xFA\x62\x77\x26\x76\x01\xD8\x4C\xC0'), #65
		struct.unpack('<6Q', b'\xB0\x23\xAA\x5A\xE7\x7C\x41\xE6\x61\x17\xA6\xCB\xE3\xBF\xA7\x34\xED\xB6\xC9\xF8\x96\xE3\x13\x8C\x63\xED\x09\x4B\x46\x66\x90\x5C\x83\xE2\x5C\x2C\x64\x6A\xCB\x76\xC6\x49\xB4\x3E\x2A\x08\x8E\x49'), #66
		struct.unpack('<6Q', b'\xA0\x5A\x7B\x35\xA6\xDE\x2A\x6F\x69\x7B\x55\x15\x2B\x38\xDA\x54\xAE\x7A\x23\x46\x19\xD8\x2B\x30\xDC\xFE\x1E\x11\x82\xBB\x0C\x8F\xD3\xF2\x20\xCE\xAD\x2D\xDD\x45\xD8\xD1\xA2\xE8\xE9\xA5\x77\x8A'), #67
		struct.unpack('<6Q', b'\x28\x2A\x86\x35\x27\x33\xEC\xE1\x44\x7C\x9C\x7E\x1A\x8B\xB6\x92\xE3\x63\xE9\x99\xDC\x18\x56\xF4\x73\x1A\xDD\x02\x45\x98\xAC\x7C\x0D\x84\x70\xCD\x98\x05\x65\xC8\xFD\x4E\x6D\xA2\x84\xA5\x5D\x9A'), #68
		struct.unpack('<6Q', b'\x5C\xC1\x40\x07\x01\x9B\xB1\x16\x9A\x43\x16\x10\xC0\x4A\x54\xB4\xA5\x99\x2F\x9E\x9C\x74\x1F\x22\x0F\x57\x65\x9A\x27\x8A\x3E\xA6\xAD\x72\xD0\xAD\x69\x16\x23\xC7\x01\xC8\x0B\x74\xBA\x35\xBC\xC5'), #69
		struct.unpack('<6Q', b'\x94\x86\x37\x4F\x5A\xE7\x44\x6C\x47\x76\x57\x8A\x10\xCE\x7A\xD2\xD6\x15\x7E\xFA\xFA\x87\xC4\x17\xDF\x9E\xE2\xC8\xD5\x54\x36\x6A\xFA\x1F\x61\xCC\xED\x5E\xE3\x0C\x93\x50\x09\xC0\x03\x8C\x8A\xD8'), #6A
		struct.unpack('<6Q', b'\x2C\x1F\x4B\xBC\x48\x69\x10\xCF\x12\x97\xB9\x90\x99\xDC\xC0\x91\x32\xAB\x09\xE1\xE3\x21\x3B\x19\x66\x16\xDD\x08\x06\xDE\x40\x33\x28\x28\x60\xBF\x77\xB6\x5B\x8A\x71\xB7\x97\x11\x0B\x41\x2C\x40'), #6B
		struct.unpack('<6Q', b'\x3E\x54\xA5\x9C\xF4\x0F\x08\xEB\x27\xCA\xD6\x42\x95\x42\xB9\xB4\x05\x32\x53\xC1\x5D\xD4\x99\x59\x3E\xEF\x1B\x8A\x39\x9E\xEA\xF7\xC6\xDE\x76\x54\x77\x17\x88\xBE\x00\x41\xC8\x90\x77\x4D\x06\x17'), #6C
		struct.unpack('<6Q', b'\xD6\xF8\xA7\x50\x91\x8E\x32\xF3\xF8\xFD\xAC\x04\x1B\xE6\xE3\x52\x15\xB0\xCE\xF3\x10\x20\xD8\x51\xD8\xD5\x76\x66\x33\x73\xD6\x59\x42\xC3\x91\x1D\xEF\xBC\xB3\x4C\x49\xBE\x33\x80\xB5\x9A\x58\x0C'), #6D
		struct.unpack('<6Q', b'\x55\x38\xE0\xDC\x0E\xE7\xB8\x54\x57\xFD\x87\x96\xE9\x90\xB5\x7B\xDD\x1F\x2B\x8D\x68\x08\xFF\x6C\x4F\x99\xFE\x8B\x06\x6D\x0F\xFD\x44\xAE\x89\x24\x30\xCE\x9B\xEB\xE4\xE3\x61\x06\x20\x1F\xB2\x66'), #6E
		struct.unpack('<6Q', b'\xAD\x69\x96\x18\x60\x00\x5E\x2F\xE4\xCA\x00\x3C\xD0\xF1\x3A\x47\xB4\xF3\xD1\x68\x92\x29\x78\x02\xD2\xC9\x7E\x3A\xBC\x14\x87\x88\xD9\xD2\xEB\x71\xF0\xC7\xF9\x9F\x10\xDB\xFF\x5D\xC2\x5D\x5A\x87'), #6F
		struct.unpack('<6Q', b'\xD8\x99\x83\x46\x3E\x7A\xA9\xE2\xA1\x0D\xC8\x32\xCA\xEA\xF7\x3B\x1D\x23\xC2\xB6\x8E\xAC\xDC\x13\x97\xEE\x02\x11\x0E\xC9\x7E\x22\x34\x14\x38\xF0\x32\x48\x34\xB2\xEB\x90\xB1\x03\x83\x88\x13\x86'), #70
		struct.unpack('<6Q', b'\xFE\xBF\x26\xE0\x6C\x3B\x3D\x3A\x59\xED\x68\x9A\x3B\x95\xD4\x18\xA0\x77\xB7\x4A\x57\x7B\xBB\x24\xB1\xDC\x83\x49\xD6\x7D\xCB\xE0\x97\xAC\x69\x98\x43\x8C\x76\xCF\x33\x60\x0E\x90\x7A\xBC\x62\x80'), #71
		struct.unpack('<6Q', b'\xC7\x33\x7A\x8A\xB7\xC3\xD4\x39\xEB\xB4\xB0\x2A\xF2\x2E\xD7\x43\x94\x03\xA5\xDD\x84\x81\xAE\x54\x36\x85\xC3\x83\xA0\x7D\x2A\x0C\x22\xD8\x4A\x1D\x92\x6F\xBC\x9D\x42\xAA\x2E\x18\xFE\x61\xBB\x2C'), #72
		struct.unpack('<6Q', b'\x4D\xD2\x0B\x6C\x80\x9A\xCE\xD8\x7C\x72\x45\x58\xA6\x65\x9D\xF6\x48\xB0\xF2\x76\xCC\x81\xFF\xC3\x58\xCE\x67\xCA\xC3\xFD\xB1\x76\x8A\xE7\xCB\xFB\x0A\x97\xD0\xCE\xF3\xF5\x26\xB7\x41\x29\x50\x57'), #73
		struct.unpack('<6Q', b'\xF9\xFC\xCE\x7F\xC1\xAE\x06\xE0\x91\x35\x00\x9E\x62\xA1\xCA\x05\x12\x53\x58\x9F\xC9\x0C\x05\xB7\x90\x94\x15\x1E\x40\x60\x92\x66\x5A\xCC\x57\xA7\x5A\xD2\x42\x84\x0C\x77\x38\x40\xCD\x55\x86\x22'), #74
		struct.unpack('<6Q', b'\xA1\xF3\xF1\xD3\x67\x8D\xEE\x93\xF4\xD7\xF8\xDB\x48\x6D\xA4\xBE\x3C\x45\x46\x86\x2B\xF0\x91\x3C\x58\x8A\x18\x04\x1F\x7C\x3D\x6C\x84\xEF\x9C\xF8\x87\x72\xA9\xEF\x4B\xFF\x8B\x10\x4E\x36\x40\xCB'), #75
		struct.unpack('<6Q', b'\x25\xF8\x4F\x0D\x73\xE3\xDC\xC6\x90\x27\x97\x7D\xF8\x54\xAF\x02\x8F\x78\x4F\x6F\x0F\xD2\x69\x7D\x8F\x6E\x16\x4C\xC6\x55\xC2\x90\xA2\xC9\xF9\x4B\xBF\x9F\x52\xA3\x6B\x4F\x69\x36\x11\xC4\xCE\x3E'), #76
		struct.unpack('<6Q', b'\x3F\x7B\xCA\xC6\x5E\x0A\xE1\x3D\xAA\xAC\x85\xD0\x81\x60\x19\x7E\xF7\x60\x5D\x70\xE3\x0D\x5F\xDF\x75\x70\xC5\xDC\x83\x7C\x3E\x39\x97\x8F\xB9\xDC\xC2\x3B\xF3\xA5\xC0\x04\x12\x0D\x05\x7F\xEB\x0A'), #77
		struct.unpack('<6Q', b'\x05\xBC\xA8\x11\x3D\x3B\x3B\x6F\xDC\xB8\x95\x2B\xAB\x69\x22\xB5\x13\x5F\x38\xB1\x4E\xE2\xED\x12\xE1\xB5\x36\x58\x6B\xBA\x2B\x20\x49\xCC\x5A\x92\xC5\x36\x36\xEE\xBF\x09\xB5\xEE\xF6\x4C\x22\x42'), #78
		struct.unpack('<6Q', b'\x92\x9A\xBE\xC4\xBB\xC3\x0C\x5F\x2E\xDD\x54\xCC\xFC\x13\x43\x58\x36\x40\x39\x00\x0F\xE9\x1F\xC1\x23\x97\xFC\x72\x7C\x66\x71\x33\x9F\x8D\xAC\x62\x0B\x99\x11\x96\xAD\x7F\x31\x3C\xEB\xB9\xFC\x4C'), #79
		struct.unpack('<6Q', b'\xBA\x3F\x4A\x89\x0A\x52\x8E\xCA\x40\xCC\x98\x00\xB8\xD1\x9E\xBD\x7C\x75\xF3\xDF\x07\x45\xF2\xBD\xEC\x35\x8D\xE6\x72\xC5\xAE\x47\xE4\x73\xB3\x27\x3D\x52\xD4\xF3\xAB\x05\x3A\x97\x16\x1E\xB1\x1A'), #7A
		struct.unpack('<6Q', b'\x7D\x81\x6B\xC2\xA6\x93\xC2\xFF\x28\xD8\x59\x49\x13\x9D\x9E\x2C\xBF\x9B\x19\x08\x64\x21\xA5\x7F\x61\xD8\xCC\x0D\xDE\x02\xF0\xA6\x5D\xF3\x2C\x7F\xC5\x9D\x8F\xBE\x47\x76\xA4\x86\xDF\xE2\x52\x13'), #7B
		struct.unpack('<6Q', b'\x74\x8E\x70\x01\xE1\x5B\xB5\x84\x3E\x9F\x20\x21\x37\xC5\xDE\x3A\xBD\xAD\xDF\x68\x1E\x9A\x8F\xB1\x2D\xCF\x74\x97\x81\x50\xA0\x09\x62\x87\x0A\x38\x5D\x29\xAB\xE4\x9F\x23\x9C\x68\x0C\x5B\x60\xA3'), #7C
		struct.unpack('<6Q', b'\x8F\xFE\xDF\xBF\x1F\x03\xC7\xDD\x0A\x2F\x83\x5B\xE6\x5D\x17\x0B\x85\x96\x71\x65\xBC\x2A\x16\x31\xB1\x36\xBC\x4B\x53\x5E\x21\x51\x44\xAE\x01\x5D\x3B\x7D\x2F\x9F\x1B\xB6\x83\x6E\x42\xA2\x43\xCF'), #7D
		struct.unpack('<6Q', b'\x4A\xF0\x16\x2B\x67\xDB\x32\x7E\x88\xB7\x6A\x0E\xDE\x45\x6F\xCE\x41\x0B\xE7\xB8\x48\x85\x71\x25\xC4\xFA\xA0\x39\xCF\x8B\x36\xD7\x47\x0C\x88\x49\xEC\x63\x68\x95\x74\x16\x34\x96\x57\x33\x0E\x72'), #7E
		struct.unpack('<6Q', b'\xC6\xD6\xD9\x33\x8E\x7A\x70\x06\xC6\x76\xD5\x6C\xE2\xBF\x84\xB6\x6C\xC4\x5F\xCD\x5E\x7E\xF4\x44\xB7\x44\xD8\x56\xA8\x23\x1B\xAF\x7E\x65\xC5\x6A\x91\x27\xA6\x98\x19\x7E\x12\xA1\x64\x39\x0C\x04'), #7F
		struct.unpack('<6Q', b'\x9B\xA3\xC0\x34\x31\xEC\xDA\xA5\x1F\xDB\x5A\xBD\x60\x41\xA0\x0C\xCB\x1A\x9E\xF2\xA9\xC5\x0E\xB5\xAF\xBF\xF7\x6A\x12\xA1\x2F\xBE\xB3\xA4\xC5\xC9\xC4\x0A\xFC\xBE\x29\xEB\xB1\x1F\xC7\x39\x47\x99'), #80
		struct.unpack('<6Q', b'\x63\xA7\x83\x3E\x34\x2D\xEC\x6F\x0C\xF5\x57\x57\x71\xA5\xDB\x5B\xB3\x11\x6A\xE4\x2E\x28\xF6\xD6\x24\x55\x2A\x92\xF5\x01\xB5\xA8\xCA\x05\xB6\x06\x10\xA2\x82\xA7\x81\x5C\x97\x96\xE8\xD2\x0B\xA1'), #81
		struct.unpack('<6Q', b'\xED\xD0\x26\x22\x53\xE0\xAA\xB8\xB7\x84\x0E\x47\xC0\x31\x18\x89\x28\xFF\xE8\x48\xD6\x24\xC8\x74\xEB\x06\x39\xAD\x2E\xE0\xE4\xB5\x13\xA7\x60\xDA\x6A\x08\xBB\x5A\xF1\x29\x9E\x6A\x66\x57\x0C\xA8'), #82
		struct.unpack('<6Q', b'\x0A\x23\xE7\xB1\x52\x3E\x9E\x52\x26\x8E\xF0\xC9\x61\x88\x14\x0C\x3D\x80\xAD\x1B\x13\x8A\xFC\x0C\x9F\xAA\x2F\x90\x24\xF3\x09\x8C\x48\x98\x99\x87\x49\xEE\x31\x02\x57\x54\x2B\x2E\x49\x88\x06\x3B'), #83
		struct.unpack('<6Q', b'\xCD\x14\x68\x03\xC5\xEA\xA6\xEF\xF5\x5D\xAA\x8D\x1F\x3C\x77\x02\x31\xDE\x02\x07\xBD\xED\x4E\x0E\xEF\x40\xD7\xD0\x57\xD7\x7F\xBA\x8B\x5F\x00\x74\x0C\x5F\x80\xA8\xF8\x1E\x3E\xFF\x7B\x46\x48\x14'), #84
		struct.unpack('<6Q', b'\x0D\xC7\x6A\x01\x66\xB7\x07\x2A\xE9\x19\x42\x36\x35\x5C\x21\x64\xF1\xF6\xFC\x35\xFE\x7E\x6F\xCD\x67\x92\xC2\x84\x60\xC0\x5C\xF0\x44\xA4\x79\x25\xF3\xF2\x3B\xAB\xC9\xB9\x25\x9A\x2D\xF4\x75\xAC'), #85
		struct.unpack('<6Q', b'\x64\x84\xDB\xED\xB5\x14\x3A\xEF\xE9\xD0\x2D\x2D\x80\xE0\x14\x23\x2A\x76\x28\xF9\xA9\xAE\xDE\x14\x02\x5A\xE1\x80\xB4\xEB\x63\x57\xE6\xF5\xCD\xE8\x14\xCA\xF7\x25\xBC\xE6\x61\xDC\x10\x45\x59\x8E'), #86
		struct.unpack('<6Q', b'\x0B\x00\x21\xFD\xDC\x38\x2C\xE6\xA7\x3D\xEE\x17\xE9\x2A\xB3\x7B\x56\xB6\xC9\x4C\xE2\x15\x9F\xE4\xD8\x61\xA3\xDC\x59\x82\xE2\x56\x48\x5F\x28\xA9\x08\x80\x3B\xB4\x1B\xE6\x4A\x7E\xAF\xB4\xC6\x0D'), #87
		struct.unpack('<6Q', b'\xAE\xDC\x42\x11\x24\x64\x3C\x70\x3A\x06\x45\x2C\x34\x33\x2D\x73\xCB\x67\x45\x62\xA1\x7E\x87\x37\x14\xD1\x4D\x61\x34\xD5\x71\x28\xD1\xF5\x94\x1D\x2A\x09\x48\xE7\xB7\x1C\x6D\x0C\x6F\x05\x24\x45'), #88
		struct.unpack('<6Q', b'\xF9\x5D\x59\x3A\x82\xB1\x25\xE3\x7C\x39\x6F\xC9\xD5\x0D\x2D\x74\x51\xF4\xA9\x40\x95\x1C\x36\x44\xB9\x1F\x33\xF6\x9B\x2F\x38\x02\x67\x04\x1A\xE9\xBB\xAF\xCB\x8E\xA8\x04\xF9\x11\x38\xBF\x8E\x52'), #89
		struct.unpack('<6Q', b'\x0D\xFB\x31\x46\x53\xC6\x2B\xFD\x6C\x9A\xEB\xFE\x36\xF0\xA5\x27\x75\x07\x1D\x91\xD7\x76\xF8\xD0\x72\x0B\x6E\x9C\xA2\xB3\xEF\x12\xA1\x1D\x87\x5E\x3D\xCA\x4B\xDC\x6F\xF4\x08\xE6\xE6\xB6\x8F\x02'), #8A
		struct.unpack('<6Q', b'\x4A\x7B\x0A\x93\xC8\xEC\x17\xEF\x3D\x27\xFB\x72\x46\xB3\x97\x9D\x8F\x5D\xA2\x35\x5F\x83\xAE\xC6\x5B\x1F\xC2\x30\x95\x46\x27\x6C\x95\xE7\x50\x61\xA2\x16\xBC\x2F\x1A\xC7\xB5\xE0\xAA\x93\xAD\x02'), #8B
		struct.unpack('<6Q', b'\x56\xDD\x07\xCF\x43\xBE\x24\x6D\x2F\x2A\x8D\xA3\x62\x1D\x68\x63\xA0\x4A\x72\x11\xB4\xC9\x72\x98\x0A\x69\x19\x7C\x85\xB4\x82\xB8\xF4\xC6\x04\x28\x8D\xBA\xB1\x87\x49\x0B\xF4\x36\xCC\x99\xB1\xD7'), #8C
		struct.unpack('<6Q', b'\x0F\x9E\xFD\x73\x85\x8D\xFB\xEE\x0A\x56\x91\x9B\x19\x03\x34\x93\x8C\x24\xD5\x65\x16\xB4\x0D\xFF\x96\x41\x98\xEA\x05\xE1\x2E\x32\x9B\xD8\x90\x38\xF8\xE0\x8C\xDB\x80\x1F\x90\x3C\x98\xF8\x32\x3A'), #8D
		struct.unpack('<6Q', b'\xAC\x5B\x21\x3F\xF9\xDA\x2C\x08\xAC\x4F\x27\xB9\xA1\x18\xC1\x67\x8A\x19\x93\xFB\x1C\x50\x74\xAF\xD3\x12\xE8\xA0\xAB\x5C\x52\x53\x6E\x8A\xFE\x5E\x00\x3A\xAF\xC9\xFE\xB2\xA7\x0D\xB6\xDC\x42\xF2'), #8E
		struct.unpack('<6Q', b'\xDF\x14\x73\xBB\xBF\x7F\x88\xD3\x42\xC3\x20\xF7\xF0\xCC\xDC\xDD\x15\xC4\xC8\x33\x1C\x33\xC4\xB2\x5B\x78\xF4\x67\x07\x01\x66\x16\x87\xE4\x5D\xFD\xC1\xB7\x55\x84\xF2\x75\x18\x18\xEA\xC5\x21\xA8'), #8F
		struct.unpack('<6Q', b'\xF0\x62\x85\x41\x31\x98\x28\x7E\xC3\x85\xB1\x42\x30\x2E\xD1\x2A\x96\xAE\xA6\x35\xD7\xD0\x20\x7C\x41\xBD\x2C\xE2\x98\xEF\x8B\xA6\x43\x32\xD9\xF8\x22\x1D\x41\xA1\x38\x2F\x3D\x4F\x40\xFB\x13\xD8'), #90
		struct.unpack('<6Q', b'\x94\x42\x66\x6F\xA7\xC0\x3F\xE1\xCB\xDD\xFD\xF7\xD9\xC9\x21\x7E\xF8\xB1\xD6\x66\xB3\x68\x1E\x16\x74\x38\x74\xB5\x7E\x95\xBF\x55\x66\xD7\x64\x83\xEF\x13\x32\xB2\xD4\x43\x66\xF9\x8A\xB9\x9B\x52'), #91
		struct.unpack('<6Q', b'\x33\x5C\xDB\xAA\xDD\x7A\x6D\x03\x2E\x03\x2D\x80\x5F\x83\x25\x05\xF2\xBE\xA2\xD7\xD8\xD0\xF7\x7D\xB7\x96\x76\xB2\x44\x76\x92\x84\xB9\xB5\xF9\xD1\xE4\x21\x5E\x21\xFD\xB7\x0E\xC4\x69\x36\x74\x77'), #92
		struct.unpack('<6Q', b'\x84\x77\x89\xE8\x4B\x53\xB3\xA9\x90\x50\x1A\x54\x83\x42\xFD\x5B\x78\x97\x1C\x12\xCD\xFC\xAF\x97\xC7\x89\x79\x63\xC9\xC4\x46\xC1\xC7\x59\xDA\xCB\x2F\xE7\x20\x08\x4F\x4F\xAE\xA0\xF4\xE2\x26\x55'), #93
		struct.unpack('<6Q', b'\xC2\xBD\x72\xFD\x20\x9E\x73\xA4\xA6\x4C\xA5\xC1\xA5\xE5\x6E\x6D\x5C\x4E\x88\xCB\x6F\x7A\xA9\x70\x48\x9C\x97\x9E\x33\x08\x61\x2B\xA7\x23\xBB\xD6\x30\x37\xA6\x93\xE5\x5E\x04\x00\xAB\xCE\x1D\x5B'), #94
		struct.unpack('<6Q', b'\xDB\x8B\x8C\xF8\xE4\x14\x7C\x42\xC2\x9B\x03\x8E\x86\x30\x86\x1D\x1B\x2D\x50\x51\xA2\x40\xDB\x33\x3D\x2B\x5D\xB4\xCC\xC9\x43\xE0\x2D\x7B\x07\xEE\xB6\x67\x2B\x29\x2A\x74\x4C\xE2\xBD\x2F\x3A\x1C'), #95
		struct.unpack('<6Q', b'\x6A\xD8\x16\x70\xF3\x69\xED\x3D\xC3\xD7\x66\xAC\x13\x7B\x94\x9A\x9C\xB3\x4C\xDF\x45\x86\x2D\x82\xDA\x10\x9E\xF1\x98\x0F\xA2\x2B\xC4\xC4\x2A\x42\x8D\x13\x03\x67\x1D\x4A\xA0\x8F\x13\xD6\x34\x8D'), #96
		struct.unpack('<6Q', b'\x2C\x11\x7D\x25\x8B\x9C\xE5\x28\xCF\x9F\x49\xC5\x8C\x06\x47\x87\x1A\x19\xF9\x80\xB7\x6E\xC1\xD6\x08\xAF\xF7\x33\x16\x15\x16\xB4\xC3\xA1\xA1\x6B\x0D\xE0\x30\xA2\xD2\x83\x5B\x96\xB9\x6F\x06\xFD'), #97
		struct.unpack('<6Q', b'\x37\xFC\x8F\x7F\x1B\xBC\xF4\x70\xDC\x0F\x6B\xE5\x31\x03\xDC\x38\xBD\x0B\x2B\xAD\x90\x72\xAB\xA9\xC6\x83\x07\x3D\x3C\x97\x07\xB3\x1F\x11\xCA\xCD\xF6\x55\xC4\xBD\xDC\xF0\xB8\x17\x83\xE0\xF0\x23'), #98
		struct.unpack('<6Q', b'\x0F\xC5\x85\x92\x4E\xC2\xEE\x0A\xCE\x1A\xB6\x33\x78\xA4\xCD\x3B\x23\x07\xEE\x59\xF9\x86\x99\x83\xB9\x5E\x7F\x8D\x4A\x03\x59\xC9\xB5\x4F\x5C\xB0\x05\x7E\xAD\xD4\xC4\x0E\x7A\x9F\xD3\xA3\x37\x6C'), #99
		struct.unpack('<6Q', b'\x07\x2D\xBF\x0F\x23\xB7\x27\x02\x47\xED\x2B\x63\xAD\xD2\xD7\x28\xD0\xEF\x12\x50\x8B\x8F\xBD\x07\x42\x34\x40\xE0\x3A\xD4\xA0\x48\xA1\x49\x14\x7F\x20\x39\x89\x9B\x19\xF2\xFD\xB9\x01\xAD\x1E\x35'), #9A
		struct.unpack('<6Q', b'\x25\xEF\x1C\x31\x2E\x9D\x11\xA7\x79\x04\xED\xC4\xD0\x2C\x53\x1E\x0B\x0A\xD3\xD8\x78\xF8\x72\x22\x42\x4C\x9C\xED\x2C\x41\x9C\x76\xF5\xDD\xCB\x65\xFA\xFB\x2F\x26\x8E\x36\xEA\x21\x67\xD8\x73\xDB'), #9B
		struct.unpack('<6Q', b'\xB2\xDC\xBA\x3F\x0B\xE9\xDB\x4B\x57\xEA\x6F\x8D\x3A\xEC\x24\x13\x00\x0D\x85\x30\xE5\xFB\x9E\x6D\xF4\xC8\xA4\xF8\xAF\x88\x1A\x40\x1C\x27\x2E\x8A\x6B\xB7\x5C\x65\x43\x6F\xE1\xBD\x6D\x5B\x50\x35'), #9C
		struct.unpack('<6Q', b'\x37\x70\xF5\x23\x7E\xE5\x15\x6E\x26\xFA\xC1\x62\x73\x73\x62\x49\x0B\xB8\x29\x18\x2D\x37\x62\xC9\x11\x62\x4D\xEA\x32\x68\xFE\xA1\x63\x77\x6E\xF9\x07\xE3\x26\x67\x5B\x50\x77\x16\x08\x61\xC7\x04'), #9D
		struct.unpack('<6Q', b'\x64\xC1\x6F\x8A\x3A\xFF\xE2\x42\x68\xB2\x28\x9D\xBC\xB2\x85\xFB\x28\x3A\x53\x24\xF0\xCF\x59\xC5\xCE\xB3\xDA\x11\x39\x3F\xC8\x2E\x27\x6A\x73\x9D\x4A\xC7\x0F\xAE\xCF\x4C\xAB\x8B\x04\xDD\x9C\xDB'), #9E
		struct.unpack('<6Q', b'\x65\xE7\x20\x1D\x22\x52\x9C\xD7\x04\xE7\x3C\x90\x73\xDD\x9E\x49\x49\x83\xF4\x7D\x98\x6D\x01\x9B\xC1\xC5\x2F\xD1\x4A\xB4\xFA\xFC\xF9\xFD\x00\xB8\xF3\x93\x12\x81\xBE\xCE\x53\xCA\x19\xC6\x1D\x51'), #9F
		struct.unpack('<6Q', b'\xDF\xED\x26\xB8\x78\xEE\x59\xA0\x5A\xC8\x17\x4D\x29\xAF\x73\x46\xB5\x82\xF2\x4D\x4E\x7D\x52\x5E\xE3\x5C\xF9\x93\x26\x9A\x5B\xDB\x96\xF2\x54\xFB\x04\xD3\x51\x65\xCC\x2F\x91\x65\x0D\xB7\x3E\xAB'), #A0
		struct.unpack('<6Q', b'\x35\x81\xC7\xB6\x67\x4F\x0C\x7D\x37\x4D\x30\xDE\xA7\xAE\x0C\x39\xFE\x94\xD4\xC8\xAB\x9F\xE1\x49\x16\x45\xA0\x37\x64\x1B\x9E\x1A\x5A\xF3\x6A\xAB\xBD\xC4\x6C\x88\xFE\x02\xF5\x44\x73\x21\x29\x05'), #A1
		struct.unpack('<6Q', b'\xEE\x2C\xB5\x41\x41\xF3\xED\x3C\x3F\x57\x53\x47\x92\xBA\x33\x81\x5A\x02\x66\xBC\x22\xBE\x32\xCB\xB3\x3C\x40\xDE\x83\x01\x48\x0C\x31\xCF\xDF\x27\xB4\x84\x5B\xBF\x56\x21\x23\xB0\x8D\x42\x51\x72'), #A2
		struct.unpack('<6Q', b'\xCB\x25\x8E\xC5\x31\xE8\xFC\x86\x75\xFC\xCB\x45\xFE\x3F\xC4\x5C\xBE\x99\xF1\x42\xC0\x7C\x87\x33\x1C\x2E\xC2\x0C\x7F\xFA\x12\x12\x23\x98\x1F\x7D\x4B\xAB\x8E\x44\x9E\x59\xA7\x9F\x3A\x36\xB1\xA7'), #A3
		struct.unpack('<6Q', b'\x6D\x10\xE5\xA0\xED\x2F\x8C\x2D\x3F\xEB\xBB\x38\x68\x36\x2E\x19\xAF\xA0\xCE\x0A\xA6\x6A\x22\x36\x6A\x92\xF3\xC1\x5D\x28\xE1\xE7\xEC\xD9\x83\x18\xFA\x71\x03\x90\x66\xEB\x60\xF3\x1A\x3B\xC3\xBA'), #A4
		struct.unpack('<6Q', b'\xE3\xE0\xBD\x12\x16\xA1\xA2\xD4\xB8\x4C\xCB\x14\xA6\x0D\xAB\x82\xC6\xF4\x01\xAC\x50\x4D\x9A\x18\x47\xA6\xF6\xD9\xA1\x5D\x6A\xE3\xB6\x11\x6B\xB1\xD6\x20\x31\xE4\x78\x53\xE7\x36\x42\x5F\x39\x7D'), #A5
		struct.unpack('<6Q', b'\x7F\x87\xF3\x47\xCD\x55\xC1\xC0\x71\xCA\x34\xC3\xE5\xBF\x03\x4B\xF7\x4F\x84\x4B\x1F\x0F\x71\x77\xC5\x8D\x0E\x72\xAB\xBB\x43\x34\x06\x34\x86\xC5\x68\x88\x3F\xF0\x50\xEB\x72\xC8\x11\x05\xD6\x0F'), #A6
		struct.unpack('<6Q', b'\x06\x63\x28\xB5\xAA\x3D\x25\x8C\xAC\x96\x61\x4A\xF5\x38\xA4\x9A\x5E\x2C\xA2\x23\xC7\x08\x1D\x18\xA1\x10\x39\x8E\xC8\x49\x3C\x63\xB0\x75\x26\x99\x67\x4A\xF5\xC9\x76\xD9\x38\xBD\xAC\x98\xDD\x1F'), #A7
		struct.unpack('<6Q', b'\xF7\xA9\x75\x75\xDA\x93\x08\xA1\x18\xA0\xB2\x5A\x02\x4A\x5F\x8F\xC0\xFF\x6B\x33\xF0\x38\x05\xD8\x6F\x62\x89\x38\xD3\x51\x97\x0F\x1A\x91\xBF\x25\xB9\x3E\x38\x30\x60\xCC\x19\xCE\x68\x9F\x14\xE6'), #A8
		struct.unpack('<6Q', b'\x98\x05\xBE\xC6\xBA\x1D\x08\xB9\x92\x14\xC7\x69\xBC\xD9\x5D\x78\x2B\x17\x6E\xA5\x0C\x5A\x03\x8B\x88\x48\x72\x00\x35\x78\x46\x89\x69\x05\x65\x58\xC9\x57\x1E\xAF\x61\x22\xF2\x4F\x94\x4E\xDE\xE1'), #A9
		struct.unpack('<6Q', b'\xD2\xC0\x18\x27\x4D\xDC\x5E\xEA\xE4\x8A\x5A\xA1\x4D\x5D\x1C\xCB\x15\x30\x16\xF8\x82\x23\x27\xC6\xCE\x54\x7B\x05\xE5\x34\xA9\x94\x0D\xD1\x68\x3D\x1A\x48\x8E\x65\xA0\x46\x0A\xE5\x29\x49\xF2\xE8'), #AA
		struct.unpack('<6Q', b'\xCD\x82\xF4\x1A\x28\x46\xF1\x7D\x06\x7B\x40\x26\xE7\x68\x4B\x01\xBC\xDD\x70\x8C\x93\x64\xE5\x6C\xA2\xDA\xA5\x72\xDE\xD2\xDA\x36\x80\x29\x0B\x9C\xF6\x3B\x57\x6D\x29\x03\xAA\xB4\x14\xAB\x4D\x68'), #AB
		struct.unpack('<6Q', b'\x5F\x8B\x73\x4E\x06\xDC\x69\x9C\xF5\x36\x1C\x5A\xBD\x16\xCC\x83\x12\x1B\x14\x6E\x5E\x36\x9B\xA9\xE8\xCC\xFA\xD0\x5A\xFA\x48\x27\x49\x9C\xD9\x47\xA0\x73\xD0\x26\x0C\x0C\x5B\x50\x82\xB1\x05\xB0'), #AC
		struct.unpack('<6Q', b'\x1C\xFD\xD0\x0E\xA2\xA2\xB6\x15\x25\x5A\xD6\x9B\x72\xAF\x33\x93\x1B\x2C\xBD\x93\x32\x33\xCC\x22\xE1\x8B\x5E\xB1\x49\xD9\x24\xD7\x17\x71\xB9\x12\x05\xDB\xD0\x69\x3C\x65\xD7\x0D\x98\xA8\xAC\x85'), #AD
		struct.unpack('<6Q', b'\xF2\xB3\x7B\xD7\x29\xC6\x0E\x23\xB2\x7C\x29\x1D\x99\x5B\x11\x43\x6F\xC7\x53\x2C\x79\x55\xF9\xA2\xA7\x5B\xE2\xEB\x28\x67\xA7\x48\x00\x44\x38\x05\xA4\x62\xE6\x7C\x66\x9D\xF4\x6B\x7E\x6B\xC0\xDD'), #AE
		struct.unpack('<6Q', b'\x0B\x41\x44\x76\xBD\xB9\xDD\x20\x06\x8C\x2E\xFA\xB1\x91\x63\x05\x00\x7C\x16\xCF\x51\xDE\x4E\xCA\x70\xF8\x36\x05\x55\x2B\x60\x46\xF4\x1F\xC2\x97\x25\x67\x40\x50\x4B\x84\xFB\x8A\x6E\xEC\xF8\x0A'), #AF
		struct.unpack('<6Q', b'\x83\xB8\x51\x92\x74\xC5\x23\x00\x00\x7E\x6B\xD6\x86\x4F\x5A\x33\x8F\xCE\xFA\x3E\xED\x3D\x35\xAE\xDE\x35\x7B\xD6\x26\x05\xC8\x3F\x53\xBC\x80\xDA\xFB\x78\x90\x0D\x14\x4C\xFE\xF3\xDF\x00\x79\x46'), #B0
		struct.unpack('<6Q', b'\x13\x81\x44\x6A\xBE\xB2\x9C\x0F\x29\x58\x9A\x6A\x1B\x54\x8D\xE3\x0E\xFC\x54\xF3\xDA\x53\x39\x67\xE9\x69\x85\x7F\x27\x8A\x81\x3C\x3B\x2A\x12\xDB\x77\xEA\x16\x8D\x84\xEA\xB6\x18\x03\x86\x0A\xE4'), #B1
		struct.unpack('<6Q', b'\x50\x5E\x7D\x2D\xF4\x11\xCE\x78\x8D\x99\x9C\x19\xFF\x6D\xF7\x84\xCB\x35\xE9\x3A\x8E\x57\x9B\x99\x12\x32\xE6\x1B\x2C\x09\xFD\xD9\xD8\x16\xD3\xAC\x63\x3C\xF3\x31\x0C\x5C\xD6\xB8\x30\x80\xA0\x5A'), #B2
		struct.unpack('<6Q', b'\xE9\x4F\xA8\x9C\xE1\xDB\x98\x00\x37\x21\x14\xD1\x17\x66\x42\xE2\x1B\xE2\x78\x6A\x16\xC4\xC3\x63\xE4\xB0\x03\x3E\x35\x45\xB1\x74\xC4\x08\xE5\xEA\x24\x08\x3C\xF4\x2A\x60\x28\x25\x62\xE6\xC1\x58'), #B3
		struct.unpack('<6Q', b'\x6F\x6A\x42\xD1\xE6\xEB\x27\x9E\x42\xA3\x5F\x6B\x0A\x60\x6A\x2A\x70\x03\xA9\x6B\x30\xE2\xF7\x8F\xDD\xDD\x3E\x68\x1A\xD9\x83\xDF\x88\x53\x22\xF0\x42\x24\x57\x29\x0A\x32\x37\x64\x1B\x0F\xCC\xE9'), #B4
		struct.unpack('<6Q', b'\x4E\x06\x96\xE8\x80\xF3\x4D\x05\xA4\x88\xAD\xA3\x4A\x1A\xB8\xFA\x6F\x15\xCA\x6C\x48\x26\x74\xF8\x60\x29\x47\x37\x82\x3C\x1B\xBB\x0A\x96\x3F\xF7\x7C\xB8\xC0\x7E\x08\x78\x0F\x47\xE6\xD7\x57\x5C'), #B5
		struct.unpack('<6Q', b'\x1A\x4A\x61\xAC\x03\xE1\x58\x57\xDF\x58\x13\xF8\x86\xEE\x6A\x76\x6A\x39\x74\xDC\x51\xBA\x3F\x20\x1F\x72\xC5\x69\xF9\x3D\xC9\x78\x97\x65\x19\x30\xE2\x32\x9E\xE6\x5B\xB9\x8A\xCD\xCE\xC6\x87\xE2'), #B6
		struct.unpack('<6Q', b'\x97\xCC\x0D\x0C\xC1\xA7\x06\x2A\x5F\x74\xA6\x68\x82\x29\xD5\x99\xB3\x58\x48\x77\xBB\x18\xD8\xF2\x86\xD8\x64\x4F\x0D\x82\x2A\xD5\x81\x39\x26\x7A\xF8\x8E\x80\x2F\x76\xC6\x47\x63\x6E\x20\x91\xBB'), #B7
		struct.unpack('<6Q', b'\x46\xC7\xE0\x1C\xD7\xC6\x47\x08\x11\x1E\x7C\x44\x1B\x45\xFD\x86\xCC\x6F\x85\xE2\xB0\x23\x06\xC2\x1E\x18\x98\x03\x2D\xFA\xDD\x3A\x46\x6B\x33\x6B\xA0\xA0\x36\x67\x7D\x25\xB1\xB2\xEE\x0A\xC7\xD1'), #B8
		struct.unpack('<6Q', b'\x76\x97\x1A\x14\x0D\x26\x33\x56\xDB\xA3\x6C\x59\x5F\x80\x30\xD5\x46\xCE\x37\x94\xF6\x3E\xE3\x8C\x88\x70\x74\x7E\xE9\x54\x2D\xF6\xEA\x45\x9B\x48\x18\x93\x5C\xDF\xCA\x50\xA8\x0B\x9F\xD2\xAA\xA4'), #B9
		struct.unpack('<6Q', b'\x9F\x6D\x7F\x76\x16\x7B\xBD\xBD\xDD\xB6\xB7\xF1\x27\x84\x96\xF7\xEE\x76\x52\xB3\x99\x65\xC7\x58\x91\xD7\xAD\x6C\x7F\x4C\x6F\x28\x7B\x11\x42\x17\x40\xC0\x88\x81\x63\xD1\x66\x42\x96\xF1\xC4\xCE'), #BA
		struct.unpack('<6Q', b'\x24\x5B\x13\xB5\xA6\xE8\xE4\x97\x77\x79\x29\xE5\x85\xD7\x8B\x8A\xBB\xC5\x5B\x97\xA0\xC1\x45\x45\x7D\xE3\x59\x9F\xBD\xE3\xFA\x13\x2B\xDE\x91\x0E\x7C\x62\xD5\xAF\xA9\xE1\x74\x84\x77\xAC\x23\xA2'), #BB
		struct.unpack('<6Q', b'\x3B\x6B\x6C\xF8\xEA\xF1\x1B\xDE\xFE\x35\x00\xD5\xAC\xA3\x46\xA2\xCD\x21\x6A\xD9\x9D\x17\x80\x6F\x03\x0D\x30\x17\xCB\xB7\x8D\x3F\x6C\x50\x94\x5D\x8B\x79\x7A\x49\x41\xE8\x1A\xF6\xC6\xDC\x52\xAD'), #BC
		struct.unpack('<6Q', b'\x0B\x44\x1F\x8E\xD0\xE1\xA4\xF4\xE0\x22\x64\xD5\x3C\x63\x27\x5E\x20\x14\xB4\x1D\x4F\xC1\x65\x14\xC2\x37\x8D\x98\x43\x90\x93\x9A\x68\xB3\x5D\x24\xFA\x5C\xE6\xCB\x55\xA8\xDD\x28\xDE\xAE\x40\x63'), #BD
		struct.unpack('<6Q', b'\x54\x24\x89\x3F\x5A\xB6\x7A\x1F\xA1\xB5\xBE\x7E\x16\xB4\x0A\xD7\x8D\x02\xC2\x24\xE8\x31\x26\x9B\xBC\x19\x15\xE3\xDE\x7B\xD9\xD5\x8C\x05\xE4\x49\x74\xC7\x2D\xEA\xFF\xBA\x2F\x6D\x2D\x4F\x20\xEB'), #BE
		struct.unpack('<6Q', b'\x5D\x66\xA2\x71\x91\xE6\x37\x65\x23\x3F\x5A\x43\x35\xF8\xD2\x3F\x8A\x6C\x2D\x62\x3E\xDD\xD5\xAD\xF8\x43\xE2\xE5\xD5\xCD\x22\xC5\xA2\xA8\xFD\xDB\xF3\x27\xEC\x5A\x45\x14\x0E\x57\xED\x65\x7A\x47'), #BF
		struct.unpack('<6Q', b'\x63\x9D\x2E\xD3\x01\xCB\xA7\x3B\x6B\x41\xB5\xE7\x34\x57\x33\x9E\xF6\x39\x45\xF9\x84\x6A\xD9\x0E\x0D\xA7\xF5\x6D\xE4\xE2\xCE\x45\x1C\xEC\xAF\xE9\xE1\x2E\x14\xDE\xDD\x72\xDC\x4F\x1C\x12\xD6\x78'), #C0
		struct.unpack('<6Q', b'\x77\x0B\x39\x53\xF6\x0A\xB3\x7B\xB6\x7B\x8F\x7F\x2B\x4F\x39\x2D\x89\x34\xAF\x13\xC2\xA3\x77\x02\x84\xD0\x56\xDD\x74\xE6\xF6\x7D\x51\x24\xC4\x73\x30\xCD\x43\x56\x18\xDC\xA1\xD1\x8B\x5F\xB1\xFA'), #C1
		struct.unpack('<6Q', b'\x8B\x0D\x15\xF5\xAB\x53\xB4\x42\x8C\xE1\x88\x11\x9C\x10\x3F\x91\xF9\x3B\xB4\x1F\x63\xB7\x7B\xC2\x19\x84\x10\xEF\x85\xE6\xDD\xEB\xEA\x33\x6D\xC5\x87\x7C\xD6\x76\x84\xF0\x40\xAF\xC0\x73\xEC\x95'), #C2
		struct.unpack('<6Q', b'\xFB\x0B\xF5\xA1\x59\x3D\xE4\xBC\x00\x46\xD8\x04\xCA\x27\x70\xBA\x4B\x64\xBE\xA2\x98\xDB\x6F\xFB\x4A\xED\x3D\x99\x7E\x77\xDE\xD5\x3A\x0F\x71\xDF\x1E\x9F\xA3\xFC\x41\x88\x8D\x85\x3C\x89\xE5\xA5'), #C3
		struct.unpack('<6Q', b'\xFC\xAC\xAE\xE6\x76\xC7\x8A\xC6\xEB\x06\x61\x86\xC7\x67\x80\x53\xE3\xFD\x4E\x2F\x35\x4A\x7B\xD2\x78\xE3\x01\xBF\xB3\xA2\x7D\x84\x58\x6D\x92\x36\xC1\xE3\x79\x3C\x10\x16\xAA\x26\x87\xBC\x57\xF9'), #C4
		struct.unpack('<6Q', b'\x12\xC6\xC7\x03\x42\x2C\x49\x95\xFC\x21\x83\xEE\xB1\x0D\xD6\x0D\x42\x2B\xF6\x02\xA9\xEB\xD9\xE1\xF2\xA4\x37\x0E\x7D\xBF\x2D\xEA\x48\xDA\xF5\x8B\x09\xB9\x1F\xE1\x38\xA3\x8E\x81\x3F\x21\xFE\xDB'), #C5
		struct.unpack('<6Q', b'\xDD\x56\x47\x6D\x31\x21\xCB\x17\x6A\x14\x40\x81\x49\x52\x89\xB8\x88\x42\x84\x80\xF5\x12\x81\x64\x35\x8F\xD5\xF7\xC3\xAD\x47\x49\x77\x53\x6A\xE2\x28\xCE\x1C\x65\x9B\xF8\x37\xF3\xDA\x03\x38\x0B'), #C6
		struct.unpack('<6Q', b'\xE3\xB6\xE6\xDC\xE2\x16\xAB\xBE\xDF\x72\x91\xE3\xC8\xEC\x39\x8F\x0E\x9C\xF2\x3F\x55\xC9\x7C\x60\xA7\x0B\x4F\x4F\x15\x15\xFD\x4B\xE4\x8C\x40\xBD\xB6\x30\x62\xEE\x99\x4E\x16\x0D\x11\x54\xB6\x35'), #C7
		struct.unpack('<6Q', b'\xD4\x1C\xC1\xD2\x1B\xFF\xDD\xAD\xA0\x1A\x6E\xBA\x2C\x26\x1A\x2A\x46\x5A\x47\x09\x1D\x29\xF2\x0B\x32\xDE\xF4\xAD\xAB\xA0\x93\x4C\x63\x3E\x33\x27\x13\x8E\xEE\x73\x28\x3B\xD1\xF5\x31\x20\xAE\xF3'), #C8
		struct.unpack('<6Q', b'\x5C\xA5\xD9\xB2\xAB\x7C\x6C\x24\xF6\xE0\x1E\x2C\x28\xC7\xE9\x50\xD7\xD3\xA0\x65\x95\xA0\xBD\x2F\xEB\xA4\x04\x9C\x67\x52\x65\x19\x82\xDD\xA6\x29\xDA\x66\x7C\x13\xBF\x56\xDA\x4B\x6B\x6B\xA7\x08'), #C9
		struct.unpack('<6Q', b'\x10\x86\xE2\xE3\x9B\xC5\xA3\x7C\x48\xE2\x7A\x1F\xCF\x75\xDD\x6A\x35\x64\x7A\x73\x50\x74\x74\x01\x3B\x92\xB4\xB2\x9C\x25\xF2\xA1\x10\x72\xCE\x8E\x5E\xF5\xC8\xE0\x9B\xB6\x50\xF3\x98\x43\x96\xD7'), #CA
		struct.unpack('<6Q', b'\x01\x51\x82\xD1\x4E\x86\x45\xE0\xC5\xA1\xE1\x93\x91\x96\x54\xAC\xC7\x94\x07\x4D\x93\x5A\xD8\x23\x13\x42\x4A\x73\xCB\x88\xFA\xB4\xF9\xD5\xA3\xBD\xD6\xBF\x5C\x7C\x25\x88\x74\x38\xE9\x7F\x60\x66'), #CB
		struct.unpack('<6Q', b'\x81\xD4\x80\xA1\xD2\x6F\xF3\xBA\xF7\x98\x95\x1B\xAC\x40\xC4\xEA\x6C\xB0\xB7\xFF\x80\x4D\xA2\x9A\x63\xF1\x58\x73\x51\x1F\x60\x79\x3B\xB6\x8B\x41\x31\x18\x07\xD1\x03\x3A\x7D\xAE\xA6\x09\x96\x81'), #CC
		struct.unpack('<6Q', b'\x51\xE5\xBA\xCD\xD8\x52\x91\x3E\x16\xD0\x3D\x20\x3F\x79\xAD\x86\x91\xA8\xD4\x8A\x77\xEB\x3A\xBE\x18\x66\x6B\xD7\x4D\x25\x10\x28\x27\xC3\x36\x66\xE3\xCD\x5D\x9B\x8C\xC5\x8B\x86\x65\xAD\x8A\x0A'), #CD
		struct.unpack('<6Q', b'\x2A\x15\x93\x0D\x78\x72\x06\x6D\x91\x80\x82\x7B\x24\x05\xE7\xEE\x3F\x46\x37\xF1\x76\xB9\xBD\x9E\xBF\xC1\xD0\xA2\x73\x3E\xDE\xA7\xC7\x6A\x04\xA0\x0B\xA0\x71\xF8\x86\x34\xF2\x03\xA8\x96\x4C\x48'), #CE
		struct.unpack('<6Q', b'\xFB\xBF\x06\xBB\xDA\xBC\xC7\x0F\xB3\x09\x63\x3D\xFB\x3F\x5C\xF7\x62\x91\x10\x03\xD1\x05\xA3\xEC\x61\xFF\x4F\x20\x3B\x50\x3F\x37\xA2\x3F\x96\x54\x9F\x2C\x33\xCE\xB4\xCD\x42\x22\xA5\x20\x44\x9A'), #CF
		struct.unpack('<6Q', b'\xC1\x98\xD1\x79\x11\x48\x1D\xC7\x69\x25\xE9\xCE\x45\x28\x5A\x50\xB5\x55\x67\xDD\xF6\xBF\x39\xF3\x45\x92\xE8\x8D\x2B\xD5\xEA\x8B\x2B\xCA\x0D\x92\x65\x6E\x68\x4B\x37\x8A\xE6\x3E\xA4\x3F\x59\x99'), #D0
		struct.unpack('<6Q', b'\x01\x15\xE6\x17\xD7\x68\x0A\xD9\xA6\x61\x91\xA1\xAE\x20\xB9\x9B\xE4\xB1\x0E\xD9\x96\x6F\x3D\x2F\x7D\x3D\x51\x10\xBA\xEC\x15\xDF\x31\x18\xF0\xB4\x39\xD5\xE5\xE6\x36\xCE\xFE\x28\x75\x7A\xD1\xC7'), #D1
		struct.unpack('<6Q', b'\xAF\xFB\xEB\xB4\xBE\xF0\x4F\xA0\xCA\xA3\xDA\x3D\x5B\x0A\xE9\xE5\x45\xA1\xF4\x09\x22\x54\x53\x84\x07\xA7\xB5\x2B\xD7\xFF\xA6\x80\x6A\xBF\x1A\x5A\x70\xC4\xE0\x14\x7D\x67\x8A\xC1\x1F\xEC\x99\xD6'), #D2
		struct.unpack('<6Q', b'\x75\x15\x18\xE3\x24\xA1\x21\x70\x5F\x94\xAD\x17\x28\xAE\x7A\xDC\x5A\x56\x0D\x7F\x1E\x52\xB5\x8B\xEE\x05\x08\x2F\x79\xD3\x71\x66\x1A\x3A\x41\x94\xA3\x8E\x88\xD3\xCC\xF9\x5B\xB5\x47\x7E\x4D\xCE'), #D3
		struct.unpack('<6Q', b'\x68\xDE\xAA\x3C\x26\x40\xF4\x22\x7B\xAC\x2E\x77\x87\xB2\x7B\xE7\xE9\x40\x2A\x96\x75\x37\x49\x29\x1B\xB9\x8C\xA6\x7F\xA2\x06\x1E\xC7\xB9\xBD\x2A\x93\x02\xEF\xDD\x33\x52\x17\xDC\x88\x3B\xF0\x79'), #D4
		struct.unpack('<6Q', b'\x91\x23\x3E\xB5\x17\xD5\xF6\x65\x39\x1C\x0B\xF0\xA2\x65\xDB\x97\x55\x48\xAA\x85\x9B\xAE\x77\x1D\x71\x07\x9B\x3E\x9B\x3B\x13\x19\xD4\xB3\x7D\x1A\xF1\xD9\x76\x63\x84\x61\xE1\x5A\x2F\xD0\x9A\x94'), #D5
		struct.unpack('<6Q', b'\x3B\x92\x9D\xE0\xCD\x34\x44\xFE\x52\x70\x3B\x71\xFD\xFC\xB0\x03\x38\x72\xA6\xD4\x90\x32\x71\x2D\x96\xEE\x29\xF6\x6F\x94\x56\x2B\x28\xC4\xB3\xB2\x01\x5D\xA1\x60\xA0\x33\x39\x79\xAF\x5E\x1D\x0B'), #D6
		struct.unpack('<6Q', b'\x4B\x49\x0D\x0E\xFB\xFC\x40\xBC\xDE\xD1\xB3\xC7\x48\x46\x1C\xA3\x8D\xEC\x07\x9A\x21\x3C\x11\xF1\x1C\xBD\xC2\xA5\xB1\xBE\x78\x23\x94\xA1\x70\x80\x47\xC3\x0C\x19\x29\x63\xF5\xCC\xE1\xB6\xDA\x63'), #D7
		struct.unpack('<6Q', b'\x91\xBF\xBA\x82\x9E\x6B\x1B\x90\x01\x10\xD6\x45\x4C\x23\x2A\x87\x6C\x6D\xCC\xC1\x95\x6A\xA4\x6C\x95\x22\xF0\xE0\x15\x93\x77\x22\xAC\xE6\x6B\x34\x96\x93\xA5\x60\x46\x88\x48\xCB\x03\xA5\x67\xFB'), #D8
		struct.unpack('<6Q', b'\x0B\x66\x97\x4C\xF7\x40\xD4\x50\x0C\xFE\xED\x64\xBF\xCA\x1E\xE7\x22\xCE\x18\x57\x89\x1B\x20\x80\x06\xD3\x35\x4D\x80\x89\x5D\xA0\x86\xD0\xB0\xA2\x02\x04\x70\x8F\xC0\xDF\xA4\x4C\x33\xCB\x6F\x32'), #D9
		struct.unpack('<6Q', b'\xD5\xCD\x05\xA0\x2E\xD0\xFB\xBC\x15\x31\x55\x75\x46\x5A\x22\xF0\x62\xAF\xA7\x92\x36\x8B\xE1\x08\x0A\xED\x8C\x0C\x82\x4A\xD3\x05\xEA\x80\x3E\xC3\xCE\xD7\xA8\x51\x9C\x87\xAE\x3F\x50\x07\xC0\x0A'), #DA
		struct.unpack('<6Q', b'\x1A\x52\x3C\xC8\xB5\xEF\x3E\xF4\xAC\x41\x88\xC4\x5F\xB0\xE9\xE5\xF9\xB5\x85\xBF\x38\x2C\xC5\x79\xA9\xF7\x3B\xAE\x18\x08\xCD\x26\x74\x5F\x8F\xCA\x32\x5C\x38\x4F\x52\x47\x95\x7B\x10\x22\x7B\xF1'), #DB
		struct.unpack('<6Q', b'\xB0\xA4\x98\x91\x96\xFC\x48\x1A\x1C\x4E\x0C\xBB\x40\x89\xA7\xD9\x91\x76\x0E\xE6\x9B\x1D\x78\x42\x30\x8A\x0F\x68\xF3\xCA\xD1\x87\x29\xAF\x6A\x60\x93\xF1\x9F\xD0\x93\x87\x04\x60\xBC\xDA\x18\x45'), #DC
		struct.unpack('<6Q', b'\x34\xA0\x56\x4A\x13\x48\x5D\xF0\xB9\x69\xDC\x91\xEB\x5E\xA6\x89\x62\x3C\xE6\x60\x39\xF4\xC7\x8F\x92\x5E\x41\xF9\x9F\x6B\x1C\xFA\x35\x79\x34\x56\x4E\x9D\x21\x7E\x85\xEF\x0A\xDE\xD6\x48\x6A\x2B'), #DD
		struct.unpack('<6Q', b'\xFD\x5F\x04\x4B\xC5\xF9\x7F\x1A\xB2\x68\x94\x2E\x56\xA9\xA0\x44\x92\xBF\x1E\x2D\xA2\x25\x14\xF1\x0E\x8E\xD2\x0B\x12\x33\x8D\x20\x62\xE1\x80\xAF\x97\x41\xD7\xF2\x93\x6C\xE6\xDF\x73\xCA\xED\xCE'), #DE
		struct.unpack('<6Q', b'\xA4\xC9\x29\x9D\x43\x90\x71\xD5\x9D\xAF\x5E\x2B\xDC\x07\xC0\x44\xDC\xCE\x80\xA7\x48\xDF\x6D\xEF\xB1\x24\x60\xA9\xE4\x05\xB2\x61\x5D\xFE\xC3\x84\xCE\xB6\x85\x18\x21\xCE\xE2\xB6\x86\x69\xB5\xB8'), #DF
		struct.unpack('<6Q', b'\x99\x7E\x23\x34\xFA\xAC\x6D\xF3\xD0\xFE\xBC\x53\x52\xEB\x45\xBE\xC0\x1A\xB2\xB8\x46\x69\x2C\x40\x67\xCD\xE9\xE7\xFC\xA6\x60\x24\xFC\x29\x26\x16\x5B\x6D\x9A\xF8\x1E\x82\xDB\x74\xA3\xCE\x6C\xF6'), #E0
		struct.unpack('<6Q', b'\x1D\x1A\xC3\xDC\x74\x60\xE0\x16\xC3\x38\xFA\xC3\x7A\x01\x72\xF1\x60\xED\x84\xB7\x4B\xCE\xC1\xBB\x71\xE6\x2A\xE8\xBC\xA8\x89\xDA\xBB\xC0\x26\x8D\xFB\xCF\xDA\xA6\xD6\xF6\x09\x96\xAE\x81\x51\x18'), #E1
		struct.unpack('<6Q', b'\x61\x76\xD1\x94\xDD\xDB\x10\xF1\x93\xF3\x69\xBA\x4C\xBB\x9F\xF5\x00\x5E\x3C\x3F\xFB\x60\x3B\x46\xAC\x78\x8E\xFE\x96\xB8\x60\x1C\xB6\x5A\xDE\x95\x67\xE2\xB3\x5E\x19\x62\x4D\x65\xD4\x28\x73\x99'), #E2
		struct.unpack('<6Q', b'\x61\x4B\xE2\xAB\x18\x91\x06\x21\x6A\xFC\xCE\x8F\xC4\xB8\x1C\x81\x02\xF9\x56\xFB\x2C\x03\x3B\x48\x57\xC0\x8C\x19\x48\xE8\x32\xFB\x70\x4F\xA0\x62\x54\x81\x20\xA6\xE2\x59\x49\x89\xD1\x38\x00\x90'), #E3
		struct.unpack('<6Q', b'\xD0\xEC\xFF\x9B\x78\x09\xD5\x5A\xA0\x9C\xF6\x25\x83\x5E\xCD\xDD\x82\x8C\x9B\xCD\x1A\x8F\x4D\x15\xAF\x3C\xCE\x94\x5E\xF7\x7D\xAC\xBE\x54\x87\xB3\xD1\x54\x65\x6D\xE7\x86\x84\x73\xAF\x64\xDB\xB5'), #E4
		struct.unpack('<6Q', b'\xBF\x43\x9A\xAC\xA1\x08\xA3\x35\x92\xE4\xE6\xB3\x5A\x80\x47\x26\x8F\x58\x61\x6F\x61\x4A\xB7\x4B\x4E\xF5\xDB\x5B\xEE\x02\x46\xFA\xDB\x74\x71\x0A\x47\x62\xDD\x3F\xAC\xFA\x08\xA8\x3C\x43\x95\x57'), #E5
		struct.unpack('<6Q', b'\x05\xA6\x4C\x77\xB8\x94\xA0\x51\x25\xF2\xEE\x74\x4C\x97\x07\x5F\x53\xA9\x81\xAD\xF7\xFE\x2A\x02\xD6\x6F\x33\xBA\x4B\xC4\x67\x09\x3D\xBA\xEC\x8A\x91\x27\xA3\x8A\x0A\x3F\x3C\x57\x36\x84\x0B\xF7'), #E6
		struct.unpack('<6Q', b'\xC3\x66\x07\x42\x83\x4F\x37\xCF\x7C\xF0\x3E\xA1\x01\x19\xF3\x71\x0F\xDC\xF9\xDE\xC7\x56\xAD\x63\x31\xA2\xF5\x59\xE8\xB5\x5B\x9E\x4C\x3E\x89\xB9\x3B\x45\xBF\xD0\x60\x97\x71\xC2\x66\x4B\xE1\xA1'), #E7
		struct.unpack('<6Q', b'\x48\x3E\xFD\x73\xCC\x61\x18\xB4\xE1\x4B\xB0\x38\xA1\x79\x1D\x46\x17\xA8\xFB\x37\x7D\xD3\x10\x40\xA4\x25\x32\x69\xAA\x22\x96\x7D\x9A\x79\x26\x81\x4B\x45\x04\x22\xB6\x6E\xCD\xDC\x87\xD4\xA5\x33'), #E8
		struct.unpack('<6Q', b'\x20\x33\x05\x7A\x31\xD0\x91\xD2\x76\x1F\x0D\xE5\xF1\x78\x76\xE2\xC7\x9F\x15\x63\x3A\x66\x3D\x9A\x52\xB4\xBA\x67\x3F\x4D\x7B\xAD\x1C\xCE\x33\x2B\x5E\xC0\x9C\x26\xC3\xBC\x34\xD7\x1C\x26\xB8\x0F'), #E9
		struct.unpack('<6Q', b'\xEE\x25\x4A\x3D\x6D\x54\xD0\xF3\xF2\xB7\xC9\x28\xAD\x74\x28\xB4\x28\x2D\x96\x29\x8B\x78\xEC\x73\x53\x85\x2B\x13\x48\x3A\xE7\x4A\x66\x0B\x91\xA0\xD7\x99\x6C\x75\x5C\x55\x2D\x71\xC2\xE2\xA7\xEC'), #EA
		struct.unpack('<6Q', b'\xDD\x1F\x91\x24\xBF\xA5\x9F\x55\x4B\x55\x70\x37\xDE\xF5\xDD\xA1\x46\xB9\x66\x93\x13\xFD\xC3\xC7\xA4\x2B\x1D\x88\x0C\xCC\x7E\x6E\x87\x4B\xE5\x27\x6A\x6D\xE7\x14\x78\xB8\xFA\xC4\xFB\xD5\x52\x73'), #EB
		struct.unpack('<6Q', b'\x0A\xAC\x8D\xED\x2B\x62\x9A\xF1\x2E\x5A\xFC\x7E\x5D\x8E\x54\x35\xB2\x57\xF0\xB4\x74\x49\xC8\xCA\xAE\x35\x03\x3D\xD0\x7E\x31\xAB\x38\x17\xC5\xF2\x38\xC1\x0F\x71\x16\x34\x40\x5A\x49\xCC\x90\x9C'), #EC
		struct.unpack('<6Q', b'\x61\x04\xF1\x36\xB9\xDE\xA7\x9F\x9D\x2F\x46\x58\x0B\x9B\x52\xA1\x65\xEC\xB9\xC8\x11\x91\x10\x9F\xEA\x33\x4E\x44\x28\xEB\xA3\x23\x37\x89\x11\x75\xCA\x84\x40\x55\xC2\xEA\x46\xC9\xA7\x58\x9D\x59'), #ED
		struct.unpack('<6Q', b'\x4E\xAC\x56\x78\xBB\xAA\xC3\x6E\xAF\xCC\xCB\xC1\x07\x69\x0E\x98\xF5\xCB\x00\xC7\xAD\x57\x85\x1F\xF4\xD9\x48\xAF\xE0\x1C\xCB\x7D\x8A\xAB\x99\x81\xDF\xDA\xB3\x7F\x12\xF3\xDB\xFA\xB8\x6D\xB3\xE6'), #EE
		struct.unpack('<6Q', b'\x60\x16\x10\x7A\x3F\x0D\x0F\xC0\x97\xC6\xB6\x2D\xB1\x94\x5B\x60\x38\x5F\xB6\xA2\x7B\x4F\x94\x79\xBC\xE2\x47\xDD\xDE\x8A\x85\x40\x2B\xB0\x9F\x0E\xDB\x4B\x04\x1E\x39\x95\x10\xA3\x01\x9D\xC7\x86'), #EF
		struct.unpack('<6Q', b'\x2A\x48\x98\x5B\x3D\x89\x31\x97\x26\x03\x79\xF9\x67\xE2\x8D\xFB\x5D\x50\x3A\x14\x07\xF4\x80\x87\xE3\xA8\xD3\xCC\xFC\xAE\x1C\xA4\xFE\xA7\xB7\xD7\xB3\xF0\x42\xA0\xAC\xA1\x19\xBB\xFE\x51\x31\x3E'), #F0
		struct.unpack('<6Q', b'\x50\x69\x3A\x00\x79\xF6\xED\xE7\x12\xAB\xC6\xA8\xD4\x97\xFC\xBA\x9A\x55\x79\x9C\xB4\x96\xC0\x13\x19\x50\x4B\x43\x01\x25\x05\xC3\x09\xFB\xAD\xE7\x23\xFB\x80\x12\xC0\x2F\xBD\x31\x5D\x90\x59\x19'), #F1
		struct.unpack('<6Q', b'\x5B\xC6\xFC\xFC\x46\x0C\x5C\x57\x19\xB4\x34\x3F\x87\x5E\x62\xFE\xA3\xB8\x51\x7F\xCC\xFD\x96\x16\xC0\xE7\x5A\x0E\xF3\x56\x9C\xC7\x15\xAD\x7F\xD2\x1C\x46\xE3\x14\x59\x28\x47\xCB\xB9\xCC\x7B\x1B'), #F2
		struct.unpack('<6Q', b'\x16\x8F\xCC\xE5\x58\xFE\x06\x38\xBF\xC1\x34\x67\xD7\x4E\x24\xF8\x56\xDB\x5D\x0F\x0E\x94\x04\x4E\x7B\x8B\x15\xC4\xDD\xAF\xD0\x5B\xC9\xC5\x11\x99\x94\xBA\xC6\xA4\x26\x97\xAE\x55\xC1\x2A\x6E\xFF'), #F3
		struct.unpack('<6Q', b'\x44\x41\x11\xB8\x44\xC8\xC7\x49\x00\xCB\x35\xCA\x1B\xE4\x50\xB4\x7C\xF9\xBE\x67\xEC\x50\x24\x30\x8B\x0D\x1E\xDB\x49\x20\x66\xA8\xFA\xB8\x8E\x52\x22\xC0\x69\xDA\xF7\xA2\xC1\x85\x65\xF1\xBB\x6A'), #F4
		struct.unpack('<6Q', b'\x4E\x04\x7F\xF6\x0D\x42\xBB\x37\xB3\x07\xEF\xE2\xF3\xE9\xC0\xDC\x41\x78\x77\x18\x86\x08\x10\x4D\xAE\xDA\x05\x93\x37\xE5\x92\x04\x8F\x12\xE4\x37\x1C\x79\xA4\x3D\x17\xEA\xA4\xCB\x45\x84\x68\x80'), #F5
		struct.unpack('<6Q', b'\x9D\x8D\xCF\xE4\x7C\x8A\x39\x51\xA5\x9C\xA6\x91\xD8\xFC\xA5\x49\xA5\x2D\x39\xC2\x0E\xA6\x72\x3D\x39\x55\xAB\x79\xB8\x96\x82\x0E\xDE\x0B\xDC\x2E\xAF\x00\xCB\x6B\x90\x3E\x4B\xE5\x48\x38\xB9\xBE'), #F6
		struct.unpack('<6Q', b'\x19\x2E\x92\x18\x2A\xC5\xD7\x7A\xF5\xB8\xF5\xC4\x57\x2C\x29\x29\x0B\xC1\x77\xA5\x98\x1F\xCF\xF0\xCD\x60\xB6\x3B\x29\x9F\x2B\x07\xFB\xB6\x75\x55\x4F\x60\xB8\x09\xDB\xFC\xB9\x81\x6A\x39\xCB\xDE'), #F7
		struct.unpack('<6Q', b'\xDE\x20\xC2\xB4\xAD\xD7\x4A\x25\x0D\x07\xA0\x95\x0F\xE2\x62\x6C\xD8\x9B\x30\x39\xF3\x89\xEB\xAD\x9F\x28\x8B\x17\xCC\x85\xF6\xA2\xA5\x95\xEE\x5D\x5B\x90\x43\x93\x86\x7C\x97\xA2\x34\x0F\xC3\xE0'), #F8
		struct.unpack('<6Q', b'\xAA\xBF\xCF\xF7\x1A\xD5\x9C\x66\x1D\x27\x80\x68\x6F\x80\xE0\xE3\x90\xBF\x98\xE0\x59\xC2\x34\x69\x49\x72\xCA\xFB\xF0\xEA\xFE\x5D\xB6\x18\x41\x8B\x94\x4B\xF7\x89\xC3\x07\x68\xFB\xEA\x0A\x64\x53'), #F9
		struct.unpack('<6Q', b'\x20\xE1\x4B\xC0\xDC\xAC\x3B\xDD\x51\x8E\x19\x64\xBD\x49\x49\x6D\x74\x8A\x59\x66\x96\xB3\xFD\x31\x81\x5A\xC1\xC0\xE9\x6D\xBC\xBB\x79\x62\xC0\x61\x1C\x20\x7F\xF2\xDD\x5C\x4E\xE8\xE3\xAF\x38\x27'), #FA
		struct.unpack('<6Q', b'\xDE\x11\x64\x5A\xD3\x1F\xD7\xCD\x3F\x52\x87\x3E\x79\x12\xC0\x3C\xF3\x36\xCA\x0F\x72\xF8\xCF\xB0\x12\xF0\x00\x73\xE0\x5F\xE8\x93\x0B\x09\x3F\x26\x85\xA0\x94\xE8\xEC\x90\xFA\xDA\x01\x0A\xF6\x2D'), #FB
		struct.unpack('<6Q', b'\xDD\xFA\xB4\xEE\xB1\x0D\xA5\x9D\x9A\x79\x74\x99\x9C\xE4\x24\xE5\xB9\xCB\x24\x6A\xF2\xFF\x09\xDE\xCD\xE7\xAC\xF3\xE9\x71\x9D\xAF\x9D\xEC\x66\x25\xA6\xB1\x62\xEB\x53\x35\x7D\x21\xB1\x2A\xD0\x06'), #FC
		struct.unpack('<6Q', b'\x22\x35\xE0\x1A\x39\xE6\x31\xDD\x15\xE9\x35\x5B\x06\xD1\xAC\x93\x18\x92\xE7\x03\xCC\x56\xEB\xF4\xF1\x97\x0C\x85\x5C\x81\x17\x07\x6F\x0A\x54\xC0\x8A\x6A\x4F\xBF\xF0\x7F\x50\xE9\x5D\xAE\xF8\xFC'), #FD
		struct.unpack('<6Q', b'\x07\xB2\x50\xDC\x13\xB4\x45\xAB\xD5\xD8\x51\x95\x36\x17\xB4\x40\x10\x72\x8E\x10\x6A\x28\x32\xCA\xAF\x93\xD0\xD8\x54\x5E\x22\x03\xE9\x76\xA5\x1E\x59\xA5\x6C\x4B\x62\xE0\xC4\x4D\x77\xAB\x12\x4E'), #FE
		struct.unpack('<6Q', b'\x6C\xB9\x6C\xDF\x50\xF8\xF4\xD9\x5D\x33\x67\x16\x1B\xD8\xBA\x8A\xE5\x72\x9C\xE7\xCF\x79\x40\xCB\x96\x69\x31\x3E\x76\x2F\x54\xE5\xC4\x97\xD3\xB9\x79\x4B\x3E\x30\x11\x51\x94\x8B\x03\x33\x69\xE4'), #FF
		struct.unpack('<6Q', b'\x02\x25\x0D\x3B\xC5\x5C\xB1\x75\x58\x43\x52\xA0\x6B\xCA\x1B\xDA\x7E\x6B\xC0\x56\x75\x97\xDA\x9E\x25\x13\x0A\xCF\x7E\x72\x57\x6C\xF8\xC3\xE5\x78\x5A\x3D\x61\xDC\x5A\x94\xB4\xA3\x94\x2D\x06\xCE'), #ZZ
	)

	__table_add = (
		struct.unpack('<6Q', b'\x68\x5A\x96\x89\xC2\x58\x80\x7D\x96\xEC\x6E\x3F\xC9\x3A\x0E\xA4\x88\xDA\x64\x6A\xB5\x2A\x96\x42\x91\xAA\x34\x59\xD2\x91\x63\xFB\xCC\xCD\xAF\x87\x7E\x8F\x95\xD9\xF8\xF6\xC9\x82\xD1\x14\x83\x98'), #00
		struct.unpack('<6Q', b'\x46\x7D\x62\xD6\x24\x02\x46\x1C\x02\xA3\x77\x68\xED\x38\x83\x0C\x6B\x1E\x2D\x8A\x84\x86\x96\x09\x49\xFB\x7B\x5D\x21\x6D\xB3\x43\x04\x7C\xDB\x94\xF8\x86\x87\xB3\x90\x8B\xB6\xE0\xAA\x38\x46\x6F'), #01
		struct.unpack('<6Q', b'\xCE\x21\xC7\x2A\x58\xB0\x73\xD6\x8B\x61\x43\xBA\xCD\x1C\xD1\x07\x39\x9C\x7F\x98\x3F\x9B\x8F\xD6\x51\xC3\x7D\x36\xA5\x9C\xC6\x1B\xD5\x57\x14\x23\xB5\x5F\x31\x19\x51\xFF\xC4\x23\x80\xFF\x49\xBC'), #02
		struct.unpack('<6Q', b'\xE4\x2D\xA3\x91\x3A\x64\x79\x61\x03\x33\x8D\xEF\x1D\x2E\x87\x18\x52\xDA\x12\x54\x31\xD8\xBD\x7F\xAB\xA8\x3F\xB5\xCA\xB5\xA8\xF1\xC6\x3A\xC2\x8A\xF4\xE3\x1D\x80\x48\xAA\x61\x81\x9C\xE4\x7A\xAF'), #03
		struct.unpack('<6Q', b'\xBB\xB0\x9B\x11\x8C\x6A\x9B\x0C\xE1\x20\x1E\x5D\xE8\x5A\xE5\x0D\x41\x6B\xCB\xC9\x5D\x43\xB7\x48\x41\x68\xEF\x4F\xE4\xE2\xA6\xA3\x60\x00\xB8\x68\xE8\x69\x16\xF9\xCB\xCA\x58\x82\x90\xCF\x4B\x21'), #04
		struct.unpack('<6Q', b'\xB2\x7D\x94\xE2\x1D\x39\x94\x45\xBC\xC2\x9F\x6E\xCC\x71\x65\x3E\x21\x38\x22\xC8\xBB\x7A\x06\x64\x47\xE0\x02\x47\x83\xF1\xB4\xBE\x05\xBE\x46\xA1\xD9\x06\xF1\xB4\x43\x68\x4B\x61\xFE\xE8\x9D\x9B'), #05
		struct.unpack('<6Q', b'\x4E\x37\x4C\xDC\xD6\x5B\x37\x09\x67\xE2\x60\xEB\x91\x16\xB3\x8C\x44\x55\x36\xF0\x1D\xB8\x15\xA9\x51\x3B\x48\x54\xB0\x7A\x04\x9F\x7E\x9C\x8E\x68\x9B\xC7\x1F\xD0\x8B\x11\x59\x23\xDE\xA7\xB8\xCC'), #06
		struct.unpack('<6Q', b'\xB0\xDD\xEC\xBC\x40\x0D\x27\xF7\x70\x7E\xBD\x5F\xC4\x3B\xF7\xC4\x10\x7E\x50\x89\xEF\xE5\x0F\xD7\xC9\x7B\x6A\xD1\xAD\x4D\xA6\xE3\xD6\xF9\x8C\xB3\x54\x61\x01\xCE\x53\x53\xD8\xDF\xC6\xA6\x06\xA6'), #07
		struct.unpack('<6Q', b'\xA4\x0C\xE0\x88\xE4\x6E\xB9\xF8\x01\x81\xF5\xAE\xEE\x55\x89\x06\xCC\xC1\x61\x6A\xCF\x59\xAB\x4B\x03\xAA\xE9\x5E\x2F\x54\xE0\x3F\xFC\x93\x81\x38\x22\x86\x90\xF6\xC5\x74\x77\xF4\xEE\xA6\x8B\xB7'), #08
		struct.unpack('<6Q', b'\xAE\xAD\x08\x21\xB2\x73\x8F\x79\x90\x04\x13\xBB\x47\x29\xD7\x37\x00\xFC\xAC\x39\x4C\x1E\x3D\xA4\xB1\x62\x47\x41\x35\x1D\xB1\xCE\xA4\x0A\x02\x60\xF5\x1C\x8D\xE8\xC8\x91\x74\x15\xF9\x22\x03\xE2'), #09
		struct.unpack('<6Q', b'\xBE\xE1\x78\xB0\xCB\x5F\x61\xDA\x02\xAC\x35\xEC\xE1\x09\xF3\x28\xAE\xD8\x84\x1D\xBB\x5A\xB7\x64\x6E\x5E\xF0\x5A\x20\xDA\x9A\x3B\xEF\xC8\xCB\x8F\xE8\x8E\x6B\x0D\x43\xC9\x69\x17\x01\xAD\x8A\xAF'), #0A
		struct.unpack('<6Q', b'\xDC\x01\x02\x18\x38\xC0\x51\x12\x37\xFE\xBF\x57\xC7\x05\x63\x9A\x56\x19\x1C\xCD\x43\x29\x96\xDB\x74\x9A\x56\x5A\xB9\xC2\x94\x8A\x9D\x04\x16\xA0\x7C\xEA\xB4\xFB\x0A\xF0\x77\x44\x8F\x04\x03\xF3'), #0B
		struct.unpack('<6Q', b'\xE6\x11\x69\xCB\x88\x28\x37\x16\x67\x6F\xBC\xAC\xD5\x79\xC5\x30\x05\x90\xEA\x0B\xD0\x20\x1B\xC4\x40\x02\x4D\xDA\xC9\x90\x19\x88\x60\xB8\xAB\x8D\x4F\x21\x99\x05\xC8\x35\xA5\xE3\x0A\x62\x49\xE4'), #0C
		struct.unpack('<6Q', b'\x84\x1D\x0B\xF9\xEA\x13\x69\x11\xE1\xE8\xB1\x34\x61\x19\x4E\x5E\x59\xED\x75\x25\x21\x43\x8F\x4B\x4F\xC3\x7E\xCD\xB6\x98\x76\x9C\x3E\xF2\x6B\x39\x54\x2A\xC1\x94\x09\x19\xD1\x02\xD8\x19\xFF\xE9'), #0D
		struct.unpack('<6Q', b'\x8A\x76\x04\x5A\xE8\xA0\x6B\x8C\x90\x02\x55\xB6\xF0\xB3\x37\x8D\x5D\x9C\x3E\x73\x7D\x08\xC2\xC7\x98\xF2\xE9\xE1\xB8\x32\xB9\xA7\xC7\xCC\xE9\x01\xC4\x22\x51\x38\x5A\x42\x27\xF7\x9F\x81\xEB\xB3'), #0E
		struct.unpack('<6Q', b'\x6C\x9D\xB0\x1A\xFB\xDB\x75\x5F\x97\x41\x1D\xAC\x3C\x08\x37\xB8\x60\x72\x75\x2B\x14\x89\xAF\xD9\x48\xFE\xE3\x1D\xE7\x71\x1F\xAC\x90\x57\x8C\x09\x44\x1A\xF7\x71\x5E\x39\x18\x2D\x02\xE4\x93\x02'), #0F
		struct.unpack('<6Q', b'\x36\x3D\x00\x88\xB0\x61\xFB\xC8\x1C\x7F\x07\xB5\x04\x16\xFE\x3C\xC9\xBB\x06\x67\xDD\x22\x2C\x4D\x41\x7D\x30\xBB\xA9\x56\xD0\x39\x94\x3A\x5F\xA1\x2A\x53\x86\x2D\x7A\x01\xCC\x2A\xA8\x1F\xAF\xA4'), #10
		struct.unpack('<6Q', b'\x5C\xDD\x79\x2B\xA1\x7E\x69\x74\x37\x09\x0D\x06\x2F\x5A\x26\xA6\x70\x5D\xA8\xEE\x88\xFE\x8E\xED\xD5\x09\x4E\x28\xA8\x7C\xA3\x9F\x97\xC0\x48\x30\x8E\x43\x1D\x1D\xE2\xDA\xA1\x62\x33\x88\x19\xA5'), #11
		struct.unpack('<6Q', b'\x8A\xE4\xAA\xBA\x40\x08\x56\x68\x07\xF3\x07\xB7\x27\x1C\x38\xD0\x09\xF8\x22\xDB\x24\x4E\x8F\x3A\x64\x1A\x2F\x58\x6D\x52\xF5\xAC\x4D\x90\xEC\x35\xE3\xE4\x7C\xC0\x70\x49\xDC\xC3\x75\xF8\xFB\xE2'), #12
		struct.unpack('<6Q', b'\xEF\x69\x74\xB2\x82\x79\x01\x52\xA4\x76\xF0\xA2\xC8\x9F\x7A\xC6\x31\xE9\x2C\xE2\x7C\x14\xD7\xF9\x2F\xB3\x7E\x47\xAF\x53\xC1\x2D\x90\x08\x9A\x88\x65\x6B\x39\xE1\xC4\x2A\xA9\xAE\x7A\x5D\x4E\x87'), #13
		struct.unpack('<6Q', b'\x99\xBD\x4C\xCC\xE1\xBE\x17\xE6\x39\x69\xCA\xBD\xAB\x2B\xED\xAC\x71\xB8\x97\x62\xE7\x50\x85\x63\x21\x8F\x17\x5E\xF6\x3A\xF0\xAA\x50\xB4\xCA\x81\xFD\x12\x7D\x5B\x9D\xD1\xE1\xBC\xA9\xAC\x72\x67'), #14
		struct.unpack('<6Q', b'\x7A\xEB\xEF\x41\xB3\xB2\x71\x5E\x0A\xEE\x6F\xDE\x05\x0B\xAC\x1E\x54\xC0\x18\xDB\x5E\xF6\x86\x58\x80\xB3\x29\xC7\x66\x54\x57\x8B\x0A\x7A\x6A\x88\x20\x02\x33\xCE\x2F\xD0\x86\x17\xFF\xF2\xDF\x28'), #15
		struct.unpack('<6Q', b'\xEF\x6D\x70\xB2\xC7\x24\x27\x80\x00\xE9\x2A\x7D\x14\x52\xFE\x33\x1C\xBA\xB7\xA3\x5E\xAA\xEE\xC8\x3E\x42\x8F\x5D\x20\x7F\xD4\xBB\x46\xAF\x08\x3A\x5F\xCE\xD0\xB2\x0A\xF3\x83\xCC\x15\x56\x53\xAD'), #16
		struct.unpack('<6Q', b'\x42\x75\xAB\x62\xD9\x7B\xB3\xC9\x63\x9D\xAE\x8C\x8F\x18\x3D\x17\x60\xFD\x76\x80\x49\x7A\x66\xF3\xE4\x92\xE1\xF7\xF0\x74\xE7\x08\xB8\x6C\xA1\x03\x78\xF7\x4F\xAF\xD6\x27\xF3\xCD\xD9\x0C\xE7\xAF'), #17
		struct.unpack('<6Q', b'\xAA\x15\x1E\x20\x43\x91\xA1\xD3\x50\x4A\x55\xB5\xAE\xBC\x25\x10\x40\x84\xAF\xDC\x7F\x76\xB8\x59\x4F\x88\xFB\xC9\x53\x33\x45\x92\xD0\xE9\x01\x34\xED\x42\x27\xF3\x22\xC5\xB5\x82\x0E\x2E\x38\x8C'), #18
		struct.unpack('<6Q', b'\x7A\x7C\x0B\xB9\x08\x59\x0B\xCF\x06\x67\x38\xAE\x3C\xC5\x71\xDA\x68\xCB\x05\xCA\xED\xD6\x91\x4F\xF3\xC6\x5C\xDF\xB8\xE6\x87\x86\xC2\xE7\xE4\x1B\x9B\x0F\x88\xA3\x0E\x91\xE4\xB5\xE1\x60\x3F\xB0'), #19
		struct.unpack('<6Q', b'\xC6\x69\x10\x78\x89\xB3\xC3\xF4\x28\xA2\xC2\xFC\x15\x3E\x4F\xF5\xC0\x35\x04\x28\x7F\x88\x7B\xF9\xC0\xA8\x50\x4E\xEC\xDF\x83\xA5\x06\xC2\xED\x17\xB4\xE5\x75\x96\xAF\xF5\x7D\x80\x26\x68\x8F\x61'), #1A
		struct.unpack('<6Q', b'\x5A\x61\x22\x2D\x0D\x6C\x46\x83\x07\xF2\x06\x9E\xE1\x20\x45\x8B\x34\xE0\x02\x25\x9E\x65\x8B\x4D\xA9\xE9\xE0\x4B\xA0\x3D\x65\x9D\xA8\xC6\x04\xBA\xF7\x12\x57\x07\x98\x06\xD5\x06\x6A\x67\xE2\x83'), #1B
		struct.unpack('<6Q', b'\x72\xF3\xB8\xEF\xCC\x33\x65\xFC\x11\xF0\x94\xFF\x64\x10\xD6\x00\x61\x53\x32\xFA\xF7\x3A\x89\x09\x5F\x8B\x51\xBB\x50\xBD\xC2\x88\x9C\x20\xB2\xE9\x65\xCB\x68\x98\x8E\x8E\x12\xB7\x27\xD2\x27\xF3'), #1C
		struct.unpack('<6Q', b'\x92\xBA\x5E\xC8\x33\x03\x5F\x7D\x2B\xF2\x37\xCE\xC9\x4B\xA4\xB2\xC9\x64\x8F\x30\x1A\x1F\xB7\xD6\x19\x8E\xC9\x9F\x8A\x3A\xA0\xA9\x96\x8F\x14\x9B\x65\xEB\x8B\xB1\x52\xD5\x9D\xC8\x73\x10\xFF\xE7'), #1D
		struct.unpack('<6Q', b'\xBC\x75\x8B\x71\xC7\x42\xC4\x53\x18\x6A\x42\xE2\xC1\x4A\xB4\xE8\xD7\x0D\x72\x6A\xC9\xBA\x0F\x5A\x72\xC2\xF5\x50\xED\x71\xE9\x5A\xA4\xF3\xC8\x82\xC6\x6B\xDB\xBB\x96\x60\x5C\xC5\x28\xB7\x3C\xE1'), #1E
		struct.unpack('<6Q', b'\x2D\xF6\xC9\x9E\xEB\x70\xBD\x5A\x83\x47\xB4\x18\xA1\x3D\xEF\xB0\x82\x96\x86\xEC\xAA\xEB\x82\xDA\x33\xD5\x88\x43\x68\x1D\xA4\x8A\x04\x1E\xAD\x10\x71\x63\x35\x92\x8C\x0B\x2C\x08\x40\x8B\x2F\xBB'), #1F
		struct.unpack('<6Q', b'\xE8\xE5\x03\x7A\x86\x7B\x83\x96\xC7\x71\x56\x80\xAB\x69\x26\x8C\xAE\x97\x71\x53\xDA\x32\x27\xF3\x00\xEB\x14\xF1\x62\x87\x90\x95\xD7\x2A\x08\xBF\x0D\xF6\xA7\xFC\x48\x59\xD2\x80\xE7\xC2\x7E\xA3'), #20
		struct.unpack('<6Q', b'\x1A\xD6\xAA\x04\xE0\xF2\x21\x7F\x39\x7B\x9C\x90\xC5\x36\x66\xD8\xDF\xFA\x70\xA1\xEE\x16\x3E\x51\x60\xEB\xC7\x77\x2B\xBC\x25\xB7\xEE\x1F\x01\xAC\x89\x8D\x7C\xB1\x0A\xBF\xEF\xB7\x30\x29\xDF\xCD'), #21
		struct.unpack('<6Q', b'\xC1\x5B\x6F\xB9\x8E\x33\x1F\xCA\x12\xAE\x20\x9A\xE4\x44\x2D\x06\x10\xBE\x71\x7B\x2E\xBB\x6D\x15\x90\x89\xE6\x4B\xF9\xF0\x38\xA0\xD5\x5C\x68\xAE\x70\x83\xE6\x23\x83\xE9\x69\x70\xB2\x5A\x3A\xC6'), #22
		struct.unpack('<6Q', b'\xA7\xA1\x99\x63\x3A\x58\xF4\x8F\x05\x7E\xF6\x26\xBD\x26\x3A\xA7\x90\xD5\xF5\x52\x9A\x48\x91\x5D\x49\xD9\x47\x50\xB5\x77\xA2\xB3\x08\xBC\x21\x35\x74\x89\x22\xAE\x1A\x90\xC1\xF5\x30\xBB\xDB\x75'), #23
		struct.unpack('<6Q', b'\x13\xFE\x7A\xE9\x42\x30\xE1\x47\x97\xA3\xBE\x94\x8D\x40\x9F\x2A\xD8\xCE\x56\x13\x49\x5E\xCF\x6F\x50\x53\x64\x4F\x04\x23\xD0\x75\xC0\x54\x8F\xE4\xDD\x60\x2D\xB8\x27\x34\x31\x48\x66\x67\x0B\x7E'), #24
		struct.unpack('<6Q', b'\xE0\x07\xA0\x2A\xB4\x8B\x85\x64\x21\x70\x62\x0C\x35\x6B\x4C\x80\xCC\xF9\x1E\x59\xFD\xE2\x1F\x5C\x43\x97\xEE\x02\x18\x72\x66\x46\xE4\x4D\x5B\xD0\xF4\x44\x23\x11\x1A\x3D\x42\x6C\x71\xC0\x2A\xAF'), #25
		struct.unpack('<6Q', b'\xD4\x6F\xCE\x77\x21\xC3\x81\x5A\x64\x7A\x37\x2C\xA6\x19\xE8\x00\x03\xF0\x6A\x20\x6B\x2A\xE2\xBD\x81\xC3\x05\x59\x14\x2E\xF8\xE7\xFA\xD6\xAE\xE9\xA6\x8B\x17\x23\x62\xBA\x00\xA9\x84\x6C\xB9\xB1'), #26
		struct.unpack('<6Q', b'\x98\xBE\x76\xD3\x93\x03\x23\xA7\x26\xF1\xD6\x20\x20\x7E\xCF\x47\x1B\x10\x4C\xBC\x11\xF2\xA2\x57\x71\x5B\xA5\x13\xF3\xF7\xEF\x8E\x8F\x80\x83\x3D\x60\x52\x58\xA1\x9A\x71\x68\x14\xBE\x66\x0F\xA4'), #27
		struct.unpack('<6Q', b'\xE7\xF9\xA1\x32\x69\x6C\xAC\x93\x73\x56\x17\x8E\x91\x19\x0E\x91\x7D\x8A\xDE\x1A\x7D\x12\xAF\x58\x51\xAD\xC6\x53\xF0\x0F\xBA\x7C\x5E\xC5\xD6\xA9\x6A\x66\xDF\xF2\xA3\xB3\xED\x36\xFF\xBA\x3F\x7C'), #28
		struct.unpack('<6Q', b'\x18\xC5\x82\x60\xC6\x2A\xB1\x49\x32\x73\xC1\xCE\xDC\xD5\x27\xBD\x69\x18\x02\x3D\xC2\x1A\x95\xDC\x8B\x07\x64\x4C\xF6\xC7\x36\xFB\xBD\x46\x9B\x57\x31\x03\x59\x62\x08\x23\x80\xCE\x06\x1A\x5B\xE4'), #29
		struct.unpack('<6Q', b'\x49\x19\xB0\xAA\x4A\x60\xC3\xB2\x7A\xA8\x6C\x67\x02\x29\xA3\xB6\xC9\xEA\x3A\x98\xD8\xA6\xA1\x8B\xC8\xC9\xC5\x87\xEC\xB1\xF5\x3E\x4C\x52\x1B\x59\x91\x9B\x2B\x63\x0A\x58\xED\xE1\xD5\xAF\xAE\xF7'), #2A
		struct.unpack('<6Q', b'\xFB\xF1\xB2\x80\x83\x2A\x38\xD0\x12\xCF\x14\x2D\xB5\x1F\xA0\x9F\x51\xB7\x58\x90\x29\xB9\x24\x22\x18\xFB\x54\x8D\x18\xEB\x60\x1F\xA0\xD6\x8A\x14\xFE\x28\xE0\x56\x78\xB5\x54\xB7\x72\xCC\x6D\xA7'), #2B
		struct.unpack('<6Q', b'\x09\x69\xAA\x02\x8B\x10\x13\x1B\x6D\xCC\xDE\x6E\x29\x1E\x3D\x9D\xC1\x68\xE0\xFC\x87\x0A\xAB\x25\x48\x57\x3A\x51\x28\x22\x91\x45\xD6\x27\x83\xAA\x85\x41\xA1\x16\x89\xEF\x45\x4D\xBB\x92\x23\xB4'), #2C
		struct.unpack('<6Q', b'\x3D\xF0\x0F\x92\x44\x73\xD7\xA1\x45\xF9\xBB\xA7\xA5\x3D\xD4\x0E\x46\xF6\xE4\xF9\x51\x62\xCC\x44\x8A\x8B\xC4\x06\xF3\x53\xB5\x59\x92\xCF\x7D\x9A\xB6\x2D\x3D\x70\x1E\xEB\xB1\x73\xA6\xB0\x8D\x27'), #2D
		struct.unpack('<6Q', b'\xF0\x89\xFA\xE1\x61\x28\x03\x8E\x11\xAB\x22\x2F\xB1\x13\x3B\x51\x2A\xD4\x2C\x3D\x58\x68\x03\x71\x93\xEE\x3E\x25\x6D\xCE\x84\xB2\xF4\x37\x01\x84\x43\x2B\x4D\x79\x2F\x46\xF8\x20\x9B\xF0\xA5\x6B'), #2E
		struct.unpack('<6Q', b'\x7C\x65\x65\xCA\x11\x61\x43\x15\x53\x49\x9B\x16\xBD\x54\xA0\x71\xC3\x71\x54\x7B\x6C\x9C\x04\x62\xE8\x4C\x67\x5A\x21\x2B\x2A\xAB\x6C\x69\xAA\x1B\x35\x1A\x22\xD1\x03\x46\x61\x0A\xFE\x1B\xBE\x93'), #2F
		struct.unpack('<6Q', b'\x0B\xF0\x28\xA8\xC5\x33\xC7\x32\x05\xEE\xAC\x01\x98\x39\x0F\x11\x75\x38\x85\x40\x08\xCC\x14\x9F\x58\xAE\x7F\x62\xA7\xF1\x88\x6E\xFB\xB2\x0D\xD9\x30\x03\x06\xB8\x60\xBE\x46\x96\xA9\xD6\xB9\xA8'), #30
		struct.unpack('<6Q', b'\xC0\x9F\x0E\x33\x3D\xA8\x15\xCA\x30\xB1\x0C\x3F\xC3\x34\xB9\x30\xF5\xC5\xF4\xBF\xD0\x9E\x06\x17\x23\x74\x28\x92\xF3\xFB\x52\x3F\x82\xA3\xCD\xD8\xD4\xCD\xB3\xAA\xE0\x1F\x49\x92\xC2\x0A\x1B\x23'), #31
		struct.unpack('<6Q', b'\x95\xA8\x53\x6B\x56\xB9\x2A\xF0\x15\xEE\x1C\xCF\xA7\x20\x3D\x13\xE5\x70\x14\x6A\xA3\x39\x4A\x0F\x6B\x73\xD5\x6D\x3D\x5E\x3E\x53\xC6\x68\xD1\x49\x2D\x7A\x1D\xF3\xC8\x68\xCC\xC1\xD3\x4B\xDA\x2D'), #32
		struct.unpack('<6Q', b'\x64\xB2\xBC\xFA\x06\x6D\x2D\x17\x99\x30\x36\x0E\x33\x9A\xA2\x6F\x50\x70\x02\x9C\x6B\x03\x0D\x32\x25\x16\x9D\x5E\xB1\xB0\x51\xF7\xDE\x7A\x65\x19\x38\x27\xE5\x3E\x8A\xE7\xC6\xAB\x2A\x65\x12\xC2'), #33
		struct.unpack('<6Q', b'\xF1\x29\x30\xFE\x4B\x42\xE1\xFE\xC6\x13\x28\x93\x8B\x32\xB4\x29\x40\xB0\x18\xE8\x0F\x97\xDF\x40\xF6\xAF\x4D\xF3\xAE\x2A\x30\x92\xAB\xAE\x9B\xB8\x76\x57\xE0\xAA\x54\x0D\x44\x60\x9C\x88\xC9\x40'), #34
		struct.unpack('<6Q', b'\xB2\x64\xA9\x38\xDD\x28\x05\x5E\xF8\xF7\xA6\xBB\x72\x07\x28\x3F\xF6\x18\x3C\xAA\x01\xC2\x9B\xE5\x59\x81\xF5\x3B\x83\x37\x30\x4B\x1E\xD8\xA3\x5E\x85\xB6\x86\x53\x6F\xD8\xF8\x96\xD4\x41\x2B\x07'), #35
		struct.unpack('<6Q', b'\x51\x30\x88\x58\x1C\x32\x0F\xD1\xA0\xAC\xBD\x3C\xB5\x2A\x12\xF5\x3D\xC0\x85\xEF\x6C\xF2\xE5\x70\x26\x01\x01\x21\x0F\x84\x85\xDF\x98\x50\x9E\x8A\x4C\x96\x52\xFB\x32\x15\x51\xC1\xA2\xE9\x7B\xD5'), #36
		struct.unpack('<6Q', b'\x24\xA0\x22\xF9\x50\x4B\xF3\xD4\x82\x54\x18\x7C\xE0\xFB\xA9\x6B\xF7\x7A\xBF\xFE\x15\x7E\x0C\xB9\x86\xCF\x65\x60\x3C\x29\x06\xA8\x92\xD5\x6B\xB2\x43\x81\x15\xA4\xCA\xF5\xEE\x85\xF6\x4D\xF3\x3E'), #37
		struct.unpack('<6Q', b'\xD0\x63\x25\x72\xF4\x4A\xD9\x2B\x20\x3B\x3C\xF7\xAC\x11\xFD\x7C\xB1\x54\x11\x10\x31\xC4\xB1\xC0\x10\x41\xC2\x42\x38\x19\x85\xF7\xD5\x63\x85\x10\x6C\xDA\x9C\x15\xE6\xE5\x80\x83\xF8\xB3\x66\x48'), #38
		struct.unpack('<6Q', b'\x20\xA9\x3F\x5C\xE0\x7C\xC7\xC5\x3E\x9F\x04\x6A\xF7\x59\x65\x00\x39\x83\x37\xEC\x3B\x87\x62\xEA\x18\x9E\x2D\x05\x3D\x93\x67\x21\x3C\x4D\xC0\xF8\x8C\x35\x04\xB8\x0C\x48\x41\x1F\x28\x22\x9A\xE4'), #39
		struct.unpack('<6Q', b'\x95\x3C\xC1\x46\xA3\x7A\x97\xDE\x36\xF8\x12\xA7\xD6\x36\x46\x09\x21\x7E\x3B\x5D\x3E\xCF\x45\xCE\x64\xE2\xAC\x78\x03\x99\x83\x1C\xC8\x71\xE1\xDC\xBA\xC2\x89\xDE\x0A\x85\xA9\x5C\x93\x4D\xFF\x22'), #3A
		struct.unpack('<6Q', b'\x83\xE9\x70\xC3\xA1\x3A\x41\x43\xC7\x28\x74\x2D\xE0\xCC\x06\x80\xD3\x13\x13\xFE\x69\x20\x3F\xE6\xE3\xD3\xC9\x92\xBF\x7C\xB6\xF8\x7E\x77\x35\x22\xDD\x8A\xBE\xCA\x78\x81\x67\x8A\xF6\x50\x80\x04'), #3B
		struct.unpack('<6Q', b'\x50\x60\x84\xA5\x1B\x02\xF1\xA1\xD1\xE4\x3D\x1A\xA1\x55\x7D\xAC\x60\x26\xDE\x10\x56\x20\x46\xEB\xE7\x8F\xF7\x3D\x25\x17\x8B\xA7\x4E\x5B\xC4\xD4\xCE\x42\x8A\xAC\x8C\x4D\xCF\xD2\x30\xF8\x19\xC6'), #3C
		struct.unpack('<6Q', b'\xD8\xFF\xF5\x30\x43\x5E\xD4\xD1\x98\xE5\x99\x2E\xB4\x68\xF3\x12\xC9\xD3\xDE\xC1\x07\x46\x33\x95\x31\x62\xF1\x7A\xCA\x7A\xC7\xD6\x32\x10\x43\xA6\x10\x27\xA7\xC8\x9C\x26\x85\x62\xBF\x41\xFA\xC5'), #3D
		struct.unpack('<6Q', b'\xD1\xFC\x2A\xB3\xB7\xC2\xE8\x6F\x52\x68\x26\xBC\xF1\x6C\x26\x33\x6D\xF1\x60\xF4\x77\xE7\xDE\xB7\xDE\x2E\x5E\xD3\xA9\x97\x06\xAD\x15\x75\x13\xA3\xBB\x2E\x17\xD4\xCC\x54\x00\xFD\x88\xBF\x2A\x77'), #3E
		struct.unpack('<6Q', b'\xC3\xBC\x2C\x14\x11\x60\x39\x55\x36\xF3\x29\x8D\xA9\xE9\x5D\x07\x49\x0D\x77\xE8\xEB\x2E\x8B\xAC\x38\x79\x51\x62\x7B\x81\x0D\x1B\xCE\x33\xCB\x3B\x76\xAB\x4B\x4C\x1B\x58\x17\xAC\x31\x7A\x13\x0B'), #3F
		struct.unpack('<6Q', b'\xD2\x6B\xBA\x96\xDE\x2C\xF1\x94\xA6\xE6\x39\xB1\xFF\xB9\x2C\xFC\x63\x5D\xE8\xBF\xDC\x31\x27\x6D\xD6\x4A\xBF\x4A\xB3\x2C\xE4\x71\x4C\xBE\x61\xDB\x70\x72\xBF\x99\x1E\xB6\x5C\x1D\xA3\x0B\xBA\xF4'), #40
		struct.unpack('<6Q', b'\xB4\x73\x4C\x28\x61\xFB\x65\x3B\xAC\xAF\x63\x87\x80\xD8\x89\x5F\xB0\x9B\xC6\xD9\x67\x81\x0F\x7A\xAC\xAA\x61\xA4\x60\xB7\x27\xBD\xCC\xB6\x03\x7E\x4F\x10\xBF\xB1\x92\x09\x08\x75\x93\x18\x6A\x31'), #41
		struct.unpack('<6Q', b'\x58\x15\xB1\x18\x42\xA1\x26\xDE\x20\xDD\x16\xAE\xEB\x78\x6E\x0F\x83\x21\x06\xB7\x52\xC0\x3D\x00\xB3\xCA\x5C\x9E\x3E\x53\xAE\x97\x09\x5B\xEE\xAB\xD5\xC2\x2F\xBC\xE3\x3A\xAC\xF8\x89\x91\x21\x7C'), #42
		struct.unpack('<6Q', b'\xBB\x19\xF5\xDA\xF0\x7A\x34\x48\x09\xE2\x88\xA4\xA5\xA9\x20\x75\x0E\xF9\x48\x38\xCF\x32\x75\xCA\x9D\xF1\xFB\x3A\xF5\xA3\xAE\x85\x90\xE6\x45\xFC\x0D\xE6\x81\x88\xAA\x5B\x65\x53\x64\x3D\x61\x3A'), #43
		struct.unpack('<6Q', b'\x6C\x19\xE7\xB9\xC7\x98\x1F\x57\xA5\x97\x55\xF5\x82\x25\x43\x25\x59\x5C\x42\x73\x5B\x24\xD7\xA0\xE0\x98\xFF\x31\xAC\x2B\x16\x5F\xA8\xAB\x55\x81\x78\x08\xD6\x92\xDF\xCD\xA2\x99\x33\x6A\x3F\xB5'), #44
		struct.unpack('<6Q', b'\x58\x5A\x6E\xA8\xA7\xA8\x6E\x52\xE0\xC4\x4C\xE4\x59\x5C\xFB\x5F\x01\xFB\x79\x44\xD2\x32\xCD\x6D\x31\xCA\x49\x22\x7B\x2F\xA2\x78\x4D\x1C\xE4\x10\x04\x6F\x43\x7A\x4A\xA5\xC2\x46\x4C\xA8\x8A\x0B'), #45
		struct.unpack('<6Q', b'\xF3\x88\x81\x2D\xF0\xEB\x2B\xE7\xFD\xCA\x96\x72\x79\x39\x7E\x18\x69\x8C\x42\xBB\x19\x0B\x45\x7F\xBA\x1B\x87\xD8\xB2\x73\x87\x7B\x1F\x10\x29\x94\x2B\x8F\xF3\xA9\xB5\x8D\x26\xCB\xD2\x62\x6A\x0B'), #46
		struct.unpack('<6Q', b'\x68\xFF\x74\x63\x7A\xB9\xE3\xAC\x28\x8A\x34\x89\x75\x55\x71\x2D\xD5\x8B\x20\x6A\x3B\x42\xF9\x1A\xD6\x7A\xD6\x52\x43\x4C\x84\x4D\x0A\x10\x2E\xD3\xE2\x30\x01\xFB\x58\x5F\xE1\xE0\xB0\x16\x7C\x82'), #47
		struct.unpack('<6Q', b'\xFD\x18\x0D\x3B\x65\xB8\xF0\x78\x06\xE3\xD5\x15\x7A\x0B\x7D\x1D\x0F\x22\xA6\xDF\x89\x9A\x1B\xB6\x70\xCE\xD0\x07\x95\x2E\xB2\xA5\x5D\x4C\x09\xB8\x36\x81\xCE\xEF\x1E\x57\x20\x83\x3E\xDD\x56\x8D'), #48
		struct.unpack('<6Q', b'\xE6\x32\xD4\x43\xD2\x98\xB2\x40\x72\xB7\x12\x92\xA1\x50\xB7\xD6\xFD\xB5\xA0\xC1\x3A\x87\x9D\x6F\x2F\x87\xA6\x09\x64\x40\xAC\xE8\x72\xFA\xC1\xB9\x96\x96\x46\x9A\x2B\xB4\x24\x81\xA3\x8E\x84\x93'), #49
		struct.unpack('<6Q', b'\xB8\x58\xCC\xC8\x8E\x44\xC1\xE4\x2F\x20\x09\x26\x65\x35\xCB\x76\xF7\x21\x26\x06\x6A\xB8\x94\x94\x61\xA8\xD1\xE1\x21\x6A\x15\x60\x10\x4C\xA1\xBC\x66\x7E\xEF\xA0\x69\x59\x82\x1D\xD2\xD6\x6B\x68'), #4A
		struct.unpack('<6Q', b'\xF1\xE3\x42\x74\x48\xCA\xBE\x29\x22\x29\xA1\x3A\xE3\x63\x56\x70\xA0\x79\x76\x4C\x1E\x92\xDD\x3C\xFC\xD9\xF9\x0F\xBB\xE7\x71\x93\x98\x32\x28\x29\x8C\x28\x09\x32\xF9\xFD\x8E\x1A\xFF\xE7\x97\xBC'), #4B
		struct.unpack('<6Q', b'\x98\xC4\x0D\xEF\xD5\xE9\xC0\x49\x35\x85\x3B\x4A\x50\x20\xAE\x20\x41\xD9\x0C\xD7\x36\xD0\xB1\xAB\x14\xE1\x61\x9D\xEC\x24\x9D\x14\xE8\x5D\x59\xBA\x6C\xE8\xF1\xBE\xB9\xFB\x5D\xAF\xA0\x28\x0D\xAF'), #4C
		struct.unpack('<6Q', b'\xC5\x65\xBF\x0E\xE7\x71\x37\x9A\x9C\x03\xA8\x27\xD1\xC2\xD4\x9D\xEB\x64\xC3\x93\x13\x7E\x1A\xFF\xCD\x2A\xDB\x41\xFC\x6E\x85\xEB\x88\x4C\x7A\xAC\xE4\xB2\xBA\x3D\xA9\x11\xC6\x90\xE0\x2C\x8C\xBA'), #4D
		struct.unpack('<6Q', b'\x71\xA1\x44\xFC\x64\xF4\x83\xD5\xAF\xCA\xA2\xC1\x31\x54\xBF\x1D\x83\xD8\xE8\x16\x5B\x77\xB3\x57\xC5\x91\xCA\xB8\x7D\xD5\x61\xAC\x66\xCD\xCA\x01\xD6\x0A\xCF\xDA\xD2\xE6\x0D\x9D\xE2\x41\x5B\xCE'), #4E
		struct.unpack('<6Q', b'\x20\x3D\x27\x62\x80\xC8\x4F\x99\x9B\x36\x7F\xE1\x2A\x25\x8F\x19\x26\xEF\xF6\xAE\xC6\x53\x76\x70\xB8\xA3\xC0\x59\x95\xEE\x8C\xFA\x5D\xF2\x8B\x0A\xF2\x48\x38\x98\xA0\x54\x08\x6A\x20\xDC\x8F\x86'), #4F
		struct.unpack('<6Q', b'\xFE\x9C\x35\xC4\xC9\x1A\x53\x63\xB6\xD0\x60\xCD\x9D\x75\x22\x5C\xBB\xC3\xC9\xFD\x25\x6B\x4D\x95\x0C\x53\xD5\xE3\x04\x35\x99\x32\x4A\x5E\x28\x81\xE5\xC4\x9B\xB3\xFB\x76\xC4\xBC\x0C\x93\x13\xA1'), #50
		struct.unpack('<6Q', b'\xA1\x5E\x74\xE9\x64\xC5\xFF\x4A\xF7\xE0\xBE\x2D\x93\xF2\x50\xCD\x15\x9F\x32\x98\x45\x98\x32\x10\xDB\xD9\xF6\x11\x6E\x12\x41\x99\x74\x6E\xC0\x83\x21\x92\xFD\xAD\x65\x89\x80\x44\x25\x6D\x9B\xA1'), #51
		struct.unpack('<6Q', b'\xF1\xFE\x95\xEF\x84\x10\xB3\x9E\x76\xFA\x5B\xCD\xE7\x4D\x9B\x72\x7E\x73\x06\x02\x94\x5B\x4F\x9E\xF2\x80\x6A\x31\xE2\x8A\x05\x67\x98\x9B\xE9\x30\x85\x1B\xB2\x51\xD3\xA1\x75\xB2\x38\x81\xD2\xAE'), #52
		struct.unpack('<6Q', b'\x37\x2B\x1A\x40\xC7\xC4\x23\x77\x93\xF7\x3A\x9B\x55\x58\xE0\x54\xA5\xE5\x50\x7E\x51\xEC\xA9\xF0\x41\xBB\xA2\x48\xB6\xA5\x9D\x42\x0D\xF4\x5D\x33\xFC\x39\x80\x28\x84\x0B\x27\xF1\x5F\xC9\xF3\xAF'), #53
		struct.unpack('<6Q', b'\xBD\xE5\x25\x74\x32\xB0\x66\x4D\x38\xB2\xEC\x06\x8A\x19\x0B\x15\xC5\x47\xCE\x35\x4C\xAF\xCD\x2D\x17\xCA\x86\x9B\xFA\xAA\xA1\x42\xA4\xB7\xA5\x8B\xCA\x16\xA4\xD5\x04\xC8\xCF\xA3\xC6\xE5\x4B\xAC'), #54
		struct.unpack('<6Q', b'\xFD\x6E\xA9\x69\xD8\x6E\x5E\x95\x56\x23\x3C\x0D\x2A\x56\x5B\xB8\xDB\x80\x09\xC8\x56\x22\x5D\xCA\x39\x27\xEA\xE9\x37\xBF\xD6\xD9\xAE\x2D\x63\xA1\xB4\x89\x65\xFD\xD3\x7A\x39\x55\x66\xA8\xE8\xAE'), #55
		struct.unpack('<6Q', b'\xA4\xD9\xC5\xAB\x57\x83\x13\xDD\x3B\x89\xAF\xE2\xBC\x3F\x25\x58\x95\x49\x3A\xBB\x2C\x6A\xB9\xEF\x2B\xEB\x24\x6B\x2A\x9E\x05\xAB\x0A\x2E\x9F\x79\xDC\x6D\xCF\x7F\xFA\xE7\x12\x72\x60\xFF\x90\x74'), #56
		struct.unpack('<6Q', b'\xB2\x7C\x6A\x3F\xD0\x53\x77\xD4\x39\x14\xB8\xB4\x9B\x58\xBD\xEE\x65\xD4\x48\x1E\x3C\x08\x7F\x9E\x03\xD4\xB4\x24\x9B\xAC\xF4\x99\x6A\x0C\x1A\x34\xB7\x18\x1E\xCA\xAA\x48\x8D\xFA\x8A\xD4\x27\xD9'), #57
		struct.unpack('<6Q', b'\xF2\x60\x74\x09\xE4\xE3\xC4\xCA\xA1\xD1\x89\xFC\x4B\x1F\xA3\x59\xF1\x25\xE1\xFA\xF8\x27\x90\x7E\xD8\x13\xF4\x56\x81\xB6\x3E\x64\x82\x3A\xC8\x84\x5F\x52\x64\x6C\x92\x92\x99\x48\x95\xA0\x97\xAD'), #58
		struct.unpack('<6Q', b'\x3D\x7B\xB9\xD1\x14\xB6\x8B\xB9\x04\xD0\x59\x39\x91\x59\x65\x68\x19\x51\xF7\x92\xA2\x4E\x1D\x7E\xD2\xAA\xCC\x5D\xA9\x07\x79\x98\x1A\x03\x05\x06\x4C\x43\x67\x03\x6B\x0C\x45\x3D\xC7\x5D\x4E\x0B'), #59
		struct.unpack('<6Q', b'\x81\x12\xC0\xE4\x61\x04\xC9\x9D\x0E\x06\xC1\xBA\x0B\x4A\x62\x5D\x5C\x7B\xD1\xF9\x12\x3A\x00\x40\x0A\xF2\x56\xEB\x95\x24\xC7\xBD\x30\xA4\xAD\xFF\xC4\x4E\x16\x52\xC6\x29\xAD\xEB\xB1\x60\x16\xAF'), #5A
		struct.unpack('<6Q', b'\xD1\x75\x09\xA0\xF3\x31\x26\xCF\x5A\x61\x98\xC5\x47\xD9\x57\xC4\xB8\xC9\x07\xFC\xC9\x63\x93\x93\x18\x8C\x67\x49\x21\x2E\x71\x31\x2B\xDB\xB7\x4C\x7A\xC8\x33\xBE\x48\xEB\x4C\x29\xBF\xE4\x51\x53'), #5B
		struct.unpack('<6Q', b'\x77\x3A\xCF\x01\x03\x29\x4B\x88\x84\x80\x0B\xC4\xEB\x20\x7C\x68\xFA\x50\x14\x70\x4D\x83\xA5\x9D\xC5\xD2\x9E\x52\xDF\xB5\x04\x99\x76\xD9\x8A\x54\x41\xB4\xB6\x36\xD8\x02\x22\xF7\x65\xAC\x0F\xEC'), #5C
		struct.unpack('<6Q', b'\x67\x96\x31\x15\xC5\x6E\x63\x49\x0A\xBC\xF2\xC9\xB6\x23\x37\x7E\x6B\xAD\xCE\x75\xA2\x30\x98\xDC\xC1\xCF\x45\x55\x2B\x3C\xC0\xA0\xEC\xF1\xCE\x38\x94\x6E\x20\x79\x7D\xCB\x8B\x41\xD2\x48\x3A\x32'), #5D
		struct.unpack('<6Q', b'\xE7\x16\x25\x79\x9A\x3E\x94\xD5\xE1\x9B\x0F\x14\xC5\xD8\x64\x5E\x05\xDA\xA4\x7C\x1C\x7C\x90\x1C\xC9\xDF\xDB\x27\x69\xAE\x82\xFF\x4E\x73\x8D\xEF\x37\xFA\x78\x3A\xF5\x60\x74\xBE\x91\x04\xEE\xE6'), #5E
		struct.unpack('<6Q', b'\x63\x2E\x33\x84\x4E\x1A\xCA\x58\xA5\x30\xA5\x0C\x30\x9E\xFD\x3A\x17\x5D\x0E\x39\x61\x96\xC1\xD3\xEA\xA7\x74\x90\x29\x39\xA5\x0B\x18\xA4\x20\xED\x3A\x4C\xD7\x3C\x4B\x50\xED\xF2\x03\xCA\x50\x8B'), #5F
		struct.unpack('<6Q', b'\xFE\xF9\xA9\xED\x52\xD0\x03\xA0\xA0\x23\xB6\xCB\xFB\x48\x40\xB1\x19\xEF\x3A\x4E\xE4\x2C\x82\xC5\x95\x4D\xF5\xC6\x42\x53\x16\x8D\x9F\xFF\x1D\x7C\xC2\x6E\xB4\x2A\xAB\x2C\x48\xDD\xAE\x8C\x56\x8E'), #60
		struct.unpack('<6Q', b'\xA6\x66\x94\xA5\x7A\x41\xAF\xC7\x7D\xF8\x06\xE9\x3C\x6A\xF5\x14\x6D\xC8\x0D\xC2\xD0\x95\xDC\x22\x24\xCB\xE2\x22\xF8\xCF\x32\xB6\x78\x76\x29\x2E\x50\x62\xB5\x1F\xC6\x52\xC6\x5B\x67\x1C\x3B\x17'), #61
		struct.unpack('<6Q', b'\x85\x34\xF5\xB0\x84\x40\x1A\x40\x2E\x4D\x24\x46\x07\x00\xC5\xB9\x65\x00\x55\xD8\xE7\x3C\x35\x73\x8F\xD2\x6F\x53\x72\x4C\x9A\x3D\x9C\x1A\x3D\x74\xC7\x5D\x41\xBD\xBE\xD6\xBC\x3A\x96\xC1\xEA\xEB'), #62
		struct.unpack('<6Q', b'\xF3\x2C\xDC\x4A\xD7\x03\xD0\x3E\x95\x1D\x29\x0A\xF6\x20\xC5\x15\x18\x90\x63\x07\x44\x34\x95\x5A\x08\xD9\xD4\xE5\x3B\xC6\x96\x29\x86\xB2\xE6\x10\x92\xC1\x6F\x91\x85\xC7\x65\xA7\x59\x8D\x12\x94'), #63
		struct.unpack('<6Q', b'\xD2\xA0\x2C\x3A\x42\xA6\x33\x0D\x71\xC5\x23\xF5\x7C\xEF\x8C\x74\x9A\x16\x34\xCA\x92\xE9\x59\x11\xA1\xD2\xF6\x53\xB4\x69\xE1\x89\xAE\x66\xD3\xEC\x8E\x43\x73\xF8\x34\x24\xEB\xAB\xC3\xBC\x5B\xA2'), #64
		struct.unpack('<6Q', b'\x6B\x4F\x81\x40\x43\x89\x61\x42\xBE\x50\x1F\x71\x81\x4A\x4B\x3C\x0D\x42\xB9\x93\xAF\x60\xC8\xD0\xBC\x02\x74\x2E\xBB\xFC\xB4\x20\xB5\xE8\x80\xD8\x36\xBB\x13\xA6\x99\x8D\x78\xBA\xE2\x31\xBB\x76'), #65
		struct.unpack('<6Q', b'\xD8\x3C\xE5\x40\xFA\x63\x93\x63\x2F\xDF\x11\x0C\x63\x75\xB1\xE1\x9C\x39\xD2\x14\x9D\xC2\xEB\x2F\x99\x2B\x50\x45\x27\xB9\x92\xD2\x29\xE5\x39\x0A\x77\x6F\xB5\x9B\x66\x81\x9B\xB2\x27\x79\x1A\xBA'), #66
		struct.unpack('<6Q', b'\x21\xA1\xC0\x58\x86\x0C\x09\xAC\xA0\x19\x0C\x93\x0D\x09\xA6\x1E\x6F\x95\x64\xD4\x25\xAB\xDB\x97\xCC\x6C\x78\x69\x32\x3A\xE6\x6A\x68\x3F\x04\x08\x03\x43\xB5\x89\x1B\xB1\xBB\xC6\x2C\x79\x1D\x17'), #67
		struct.unpack('<6Q', b'\xDB\x2F\x73\x82\xC9\x7B\x4C\x66\x28\x20\x3F\xAB\x69\x50\xC7\x5A\x14\xBD\x64\x52\x94\x6B\x34\x82\xE5\xF1\x25\x40\x9E\x4E\x01\xEF\x5C\x66\xD7\x9D\x07\x45\x06\x6F\x0F\xCD\x1F\xAC\x06\x27\x5B\xBC'), #68
		struct.unpack('<6Q', b'\xF8\xA3\x18\x4D\x84\x21\xF1\xE6\xCD\x26\x32\x86\x8B\x17\xE1\xB6\x03\xE8\xF7\xC6\xA7\xD2\x4D\x9D\x5E\x72\xE7\x0B\x69\xCB\xAF\xFB\x0E\x67\xCD\xCD\x73\x86\xD6\x5D\x78\xBB\xF0\x7B\x5E\x44\x36\xB5'), #69
		struct.unpack('<6Q', b'\x19\x24\xE5\xF8\x48\xDB\x70\x31\x10\x34\x63\x81\xE1\x7A\xEF\x90\x42\x4D\x31\x9D\x78\x3F\xDB\x66\x52\xD8\x03\x1B\xD4\xFC\xD8\x6D\x02\x15\x90\xD8\x3F\x26\xB1\xA5\xEB\x03\x82\xF5\x1A\x48\xB3\x04'), #6A
		struct.unpack('<6Q', b'\x98\x3C\x21\x47\x55\x1A\xEC\xD2\xE9\x1E\x7B\x01\xDD\x3B\xA1\x14\xE8\xF2\x32\xFB\x5A\xA5\xAD\xAF\x7F\x58\x74\xC1\x61\x67\x70\x89\xDA\xC7\x84\x0F\x00\x4C\x78\x2D\x88\xB2\x2F\x98\x78\x70\xEA\xE8'), #6B
		struct.unpack('<6Q', b'\x56\x19\x9F\x5D\xC7\x00\x4B\x99\xA5\x47\xA0\x64\xA1\x4A\x77\x03\x72\xBB\xA0\xB9\x9F\xC8\x9C\x1A\x04\x81\xA5\xF2\x73\x8A\xF8\xE3\x0C\xD7\x10\x43\xBE\x66\x2B\xE5\xE0\x5D\xAE\x8F\x5E\x61\xCE\x27'), #6C
		struct.unpack('<6Q', b'\xCD\xB3\xB3\xF8\x91\xA2\xD4\xC3\xB8\x40\xFA\xDD\x29\xCF\xF2\x2A\xB2\xB0\x6A\x4B\x1F\xAA\x9A\x2F\x53\xE0\x5E\xD4\x3A\x84\xB5\x2F\x58\x68\x81\xFA\x52\xB8\x4D\xB1\xF0\x83\xE4\x1B\x07\x98\x99\x87'), #6D
		struct.unpack('<6Q', b'\x54\x2D\xAD\x08\x66\x20\x3E\xC8\xB6\x32\x12\x78\xB3\xBB\xA2\xDE\x44\x88\x44\x0B\x26\x71\xD6\x3E\x55\xCD\x73\x12\x00\x3D\xA7\xB8\xF3\x8A\x39\xC3\xCD\x89\x89\x7A\x86\x68\x43\xAD\x5F\x56\x8E\x24'), #6E
		struct.unpack('<6Q', b'\xEB\x78\xE4\xC1\x2F\xEB\x03\xFB\xB1\x5E\xC1\xD0\x17\x98\xD7\x3D\xD7\xD1\x15\xCE\x1C\x46\x07\x8C\x3B\xCA\xB7\x91\x98\xC4\xE7\x9B\xCD\xD4\xE5\xE9\x37\x24\xAD\x25\x6C\x7D\x5B\x09\x16\x99\x1E\x66'), #6F
		struct.unpack('<6Q', b'\x74\xBD\xA4\xA1\x82\x29\xB0\x57\x21\xA0\xC9\x56\xD9\x65\xEB\x9C\x34\xB1\x09\xEC\x31\x8F\x1A\xC2\x82\x01\xD7\x3D\x72\x7E\xDD\x11\xDC\xA5\x3F\x2D\xC0\xA7\x0E\xBA\x54\x95\x19\x5F\x5B\xF9\xAE\x32'), #70
		struct.unpack('<6Q', b'\x54\x66\x75\x7D\x6F\x3A\x6D\xFD\x53\x2C\xD3\xEF\x41\x2C\x40\x7F\x69\x3D\xF6\x8A\x3D\x2C\xE1\xDB\x00\x35\xAE\x03\x2B\x67\xE4\x92\xC1\xBC\x87\x9A\xE7\x0B\xBF\xBE\x62\x93\x76\x1B\x45\xC8\xD0\xE0'), #71
		struct.unpack('<6Q', b'\xC4\x47\x44\x42\x54\x75\x40\xFA\xCA\xFF\x78\xDC\xB9\x6E\x27\x38\x11\xA1\x92\xE3\x40\x0B\x03\xDC\x86\x79\xBF\x33\xB7\x29\x73\x8E\xDC\x81\xB0\xFE\xBA\x20\xCD\xB8\x4F\xC2\x04\xA6\x40\xE4\x2F\x7F'), #72
		struct.unpack('<6Q', b'\xE1\x5B\xAC\x98\x56\x6A\x67\xC4\xE3\x00\x7E\xCE\x04\x0D\x6C\xD9\x2D\x5D\xA3\x44\x0A\xF7\x5A\x8B\x60\x63\x37\xFE\x96\x17\xF8\x1A\x86\x46\xE9\xB6\x27\x28\x22\x19\xE0\xC2\x28\xA7\xBA\x3B\x83\xD0'), #73
		struct.unpack('<6Q', b'\x32\x79\xE4\x20\x56\xB5\x5D\x20\x96\xB8\x73\x04\x48\xF9\xBC\x3D\xF4\xF1\x5B\xE9\xB3\xA9\x96\x53\x51\x5C\xAB\xC1\x3F\x8E\xA1\x84\x28\x51\x89\xAD\x9F\xAF\x0F\x1B\xEF\xBF\xE1\x62\xEC\x6A\x77\xE6'), #74
		struct.unpack('<6Q', b'\x7D\x97\xD0\xCF\xC0\x31\x77\x59\x64\xD4\x07\x33\x9B\xA9\xE5\x68\xE1\xD1\x74\x08\xCA\x59\x1A\xCB\xB9\x27\xBD\x83\x60\xE6\x91\x7F\x07\x75\x30\x2A\xF2\xBC\x04\x10\x46\xE1\xEB\xC0\x32\xFA\x73\xC4'), #75
		struct.unpack('<6Q', b'\xA0\x2C\x38\x52\xF1\x73\x2D\x8F\xA2\xCB\x4F\x48\x31\x36\xDB\x36\x85\x4B\x53\x58\xAC\x7D\x8E\xAB\xF0\xAC\xDC\x4B\xF6\x83\xDC\xC9\x7A\xEC\x04\xA4\xD5\x86\x8A\xE1\x7E\xAE\xCC\xCB\x02\x80\x4C\xF7'), #76
		struct.unpack('<6Q', b'\x38\x48\xB1\x85\xC9\xE7\xB3\xAB\x0C\x10\x2E\xB0\x91\x69\x52\x1E\xF1\x2E\x30\x5F\x76\x33\x03\xB6\x28\x80\x6A\x09\x11\x72\x02\x82\x8A\xB7\xDF\x32\x3C\xB6\x5F\xDB\x44\xD2\x68\xAA\x36\x0E\xC9\xED'), #77
		struct.unpack('<6Q', b'\x56\xFC\x04\xE8\x4C\x82\xF8\xDB\xA9\x09\x05\xDF\xCD\x3C\x3E\x83\x27\xEB\xB2\x08\x1D\x93\x12\xC6\x39\x57\x0E\x40\x76\xE5\xC1\xE8\xA9\xB9\x41\x0E\x6C\xE1\x46\x40\x2F\x4B\xE4\x31\xE6\x44\x2A\xD5'), #78
		struct.unpack('<6Q', b'\xAA\x41\x3F\x86\xA8\x02\xF7\x7B\xB7\xD9\x08\xEA\x15\x13\x1C\x44\x7B\x28\x95\xC2\x5F\x69\x5B\x17\xE7\x7C\xC5\x1D\x02\xA7\xC4\xFF\xA5\x35\x0D\x5C\x10\x2E\x0F\xAE\xC3\xD0\x21\x21\xC2\xC9\x66\x84'), #79
		struct.unpack('<6Q', b'\x9C\x33\x62\x57\x7A\x2B\xC8\xA6\xF3\xD1\xA6\xE6\xDB\x8B\xFD\xC3\x41\x1B\x05\xCF\x76\x7E\xE3\x98\xA0\x9F\xD3\x54\x32\x44\xA1\xFC\x12\xFB\x89\x20\x3C\x0D\x82\xD1\x2B\x0E\x4D\x11\x40\x79\x29\xA6'), #7A
		struct.unpack('<6Q', b'\x84\x9B\x26\x5B\x8F\xE8\xA8\x91\x15\xF9\x39\xB8\x13\xBD\xE5\x5C\xF9\x93\x07\x59\xDF\xBA\xD3\x71\x5C\x63\x2F\x1F\x9A\x79\xA5\x61\xBE\xFC\x0C\x0E\xE7\x8E\x6E\xCB\xE1\x48\xCC\xDC\x32\x85\x51\x7A'), #7B
		struct.unpack('<6Q', b'\xCE\xAC\xD8\x38\xC4\x59\x61\xED\x64\xF8\xFF\x52\xB3\x47\x2D\xBD\xF3\xB0\xD8\xEB\x4D\xFE\xE3\x00\x39\xAE\x5F\x05\x49\x8D\x48\xC8\x50\x76\x65\x34\x7E\xCA\x2B\x08\xC1\x23\x4F\xFB\x2E\x27\xEF\x2C'), #7C
		struct.unpack('<6Q', b'\x50\x29\xAA\x8A\xEC\xBA\xA9\x02\xDA\x42\x6A\xCF\x9C\x08\xC3\x04\xB1\x17\xAE\xAB\x05\x10\xBB\x56\x67\xDF\x37\x4B\xEB\xB6\x89\xDF\xAB\x0B\x29\xB0\xD7\xE8\x45\xC6\x4F\x50\x9B\x5B\x33\x2D\xFE\xBA'), #7D
		struct.unpack('<6Q', b'\x32\x9C\xAC\xC1\x4E\x72\x64\x98\x04\x65\x4E\xEE\x97\xFE\x37\xD0\xF1\xDC\x19\x86\x57\x6A\xCE\xB0\xA9\x51\xD4\x9E\xF5\x89\xA0\x2F\x7D\x89\x8F\xBA\x46\xD8\xA7\x11\x1D\x9D\x26\x68\xBC\x4B\xDC\xB0'), #7E
		struct.unpack('<6Q', b'\xEC\x7B\x7B\x29\x55\x62\xB3\x47\xA1\x7E\x45\x2B\xCD\x26\xB2\x3D\x01\x39\x95\x1B\xCD\x45\xDA\xBB\x1A\xC9\x98\xDB\x10\xC5\xE5\x24\x7E\x1B\xF3\x9E\x87\x4E\x5E\xFA\xD9\x79\x83\x73\x3B\x2D\x21\xA2'), #7F
		struct.unpack('<6Q', b'\x59\x4D\xC2\xD9\x15\x0F\xFB\x3A\x1E\x69\x3C\x84\x5A\x3E\xE3\x45\x65\xFD\x6F\xBF\xBF\xCF\x58\x59\x35\x81\xDC\x03\x50\x13\xFA\xAB\x7E\x70\x95\xF7\x4D\x38\xBE\x3A\xDF\x82\x57\x21\xF2\xB4\x9B\x8E'), #80
		struct.unpack('<6Q', b'\x47\x4B\x00\xF8\xE8\xCE\x5D\x74\xA4\xCD\xD2\x0A\x87\x66\xBC\x93\x71\x50\x26\x17\xFE\x73\xA0\xF2\x5B\x4A\xF7\xE3\xC3\xEB\x2D\x51\x6A\x42\x4F\xBA\x58\x4F\x1C\xFD\x4B\x84\x0C\xE9\x54\xBC\x02\x22'), #81
		struct.unpack('<6Q', b'\x51\x51\x1C\xEF\x32\xE4\x4C\x48\x00\x59\x50\xA4\x94\xD5\xEA\x18\x6E\x1D\x98\x88\xDD\x57\x8F\xA4\x09\xB8\x16\x45\x2A\x10\x9E\x45\x1B\xA2\x88\x17\xA0\xC9\x7E\xF9\x9E\x7D\x53\x5A\xC2\x56\xA3\xBC'), #82
		struct.unpack('<6Q', b'\x7F\x7D\x15\x7C\x17\x68\xFD\x3E\xA3\x1B\x7C\x1A\x15\x45\x5F\x42\x73\xAC\x69\xEA\x46\x89\x2E\x16\x8B\x1A\x31\x03\xF0\x3D\xB5\xA0\xF2\xA1\xBF\xD0\xF3\xF6\x61\x5C\xEB\xAC\xC1\xD3\xA5\x3D\x92\xED'), #83
		struct.unpack('<6Q', b'\xC6\x9D\xFE\xEC\xDE\xE2\x50\x7C\xD7\x08\xDA\x50\x96\x09\xF8\x17\xD4\x03\x07\x94\xAA\xCA\x7B\xC8\xED\x19\xEC\xCB\x8F\x49\x80\xAB\xD9\x03\xF2\x9F\xDF\x0E\x95\x78\x9E\x03\x48\x1F\xAF\xC8\xBB\xAD'), #84
		struct.unpack('<6Q', b'\x75\xB5\xF2\xE4\x91\x64\x49\x08\x63\xA2\xAC\xA0\xD2\x1E\x6D\x06\x8F\xDC\x94\xC3\xBD\xB5\x61\xBA\x50\x1E\x7F\xE9\x12\x63\x33\xE4\x47\xCA\x17\xD2\x15\xBB\x79\x34\xF7\xCD\x95\xF6\x13\xEB\x12\x90'), #85
		struct.unpack('<6Q', b'\x6B\xFA\x1B\x44\xCB\x43\xBC\xDC\x59\xB9\xE9\x52\x82\x10\xC1\xD9\x89\x10\x2D\x62\x87\x00\x0D\x66\x92\x31\x1A\xC7\x54\x5E\xE3\x26\x22\x57\x50\x1A\xEC\xDE\x95\xAE\x42\x3E\x90\xCF\x82\x51\x1F\xB6'), #86
		struct.unpack('<6Q', b'\xAD\x4E\x20\x64\x57\xF7\x03\xAA\x43\x74\x86\x1E\xC5\x95\x90\x76\x73\x0B\x6D\xEA\x99\x63\xF6\xED\x14\xCC\x31\x54\xDE\x6C\x3B\xC7\x40\x33\x0A\x79\x64\xD8\xCC\x5C\x97\xE3\x64\xCF\x03\xE3\x5E\x16'), #87
		struct.unpack('<6Q', b'\x42\xC0\x77\xC5\x56\xF2\xA3\xE5\xB4\xA9\x27\x38\x0E\x30\x41\x5B\x04\xAD\x5C\xFC\x8E\x8A\x0F\xE8\x29\x94\x5E\xB1\xB9\x66\x78\xFF\xE2\x41\x68\xF2\x40\xEA\xBC\xC1\xDA\xA3\x25\xBC\x4A\xFF\xE4\x5B'), #88
		struct.unpack('<6Q', b'\x04\x4C\x30\x9D\x41\xFB\x32\xC9\x8A\xF8\x91\x76\x49\x56\x75\x62\x62\x5A\x5C\x61\xE8\x21\x9D\x14\x29\xF0\x4F\xE5\xB2\x07\x53\x6A\x39\x01\xF4\xD3\xCB\x87\xBE\x1C\xA0\x50\x18\x11\x31\x83\x7C\x86'), #89
		struct.unpack('<6Q', b'\xA0\x90\xA4\x82\x4D\x92\x70\x94\x83\x72\x6C\x24\x6E\x44\x75\x6F\x01\xF1\x9C\x3A\xF9\x11\x49\xBC\xBC\xA4\x22\x85\xC0\x5C\x5E\xD1\xAF\x30\x4A\x5C\xD9\x6D\x19\x91\xB7\x8F\xE7\x88\x4F\x0D\xBB\xA0'), #8A
		struct.unpack('<6Q', b'\x28\xD1\x82\x6D\xE1\x7C\xDA\xC9\x15\x7D\xFD\x7B\xB9\x96\xCA\x07\x26\xC9\x19\x4B\x4A\xA6\x3F\xD6\xC7\x10\x0B\x87\x37\x18\x28\xD1\x87\xE2\x2F\xEE\x87\x6B\xE3\xB1\xBA\xB3\x4A\xA8\x0D\x2C\x4F\x63'), #8B
		struct.unpack('<6Q', b'\x60\xF4\xCB\xF9\xF3\xAC\x72\xD8\xBD\x2B\x76\x55\xDD\x7F\x75\x14\x6C\xB3\xA4\x60\xF0\x6A\xA5\x2B\x8E\x7B\x81\x12\x9C\x89\x93\xD2\xB3\x60\xC0\x38\xE5\x75\xD2\xE4\x97\xB0\x4D\x3C\xDF\x47\x4D\xD8'), #8C
		struct.unpack('<6Q', b'\x7D\xEC\xA2\x50\xDD\x3D\x4C\x5D\xFB\xAA\xB0\xEF\x10\xA4\xB4\x44\xCA\x29\xBD\xD5\xFF\x49\x06\x3E\x20\x8B\xAD\xDD\xE1\xCC\xDA\xA1\x81\xCE\x0D\xF5\x54\x59\xBE\x09\x4C\x38\x60\xC2\xAF\x39\x2D\x08'), #8D
		struct.unpack('<6Q', b'\x94\x7B\xA2\xA7\xD9\xA6\x7A\x63\xEC\x15\x74\x84\x35\xB9\xBA\x09\x20\x89\xD7\xE0\x0E\x3D\x88\x71\xBB\xA0\xB5\x44\x4D\xCB\x95\x40\xE3\x1D\x3E\xD9\x2F\x83\xE1\xC4\x52\xD2\x7D\xD2\xC9\x07\x1E\x2D'), #8E
		struct.unpack('<6Q', b'\x0C\x72\xCB\xE2\x15\xFE\x8F\x78\xF7\x68\x8A\xE9\x80\x83\xF1\xA2\x5B\xB4\xE1\x39\x2A\xD1\x8E\x88\xD3\xBB\xD2\xAF\x5A\xEB\x47\x21\x99\x39\x5A\xA1\xC7\x66\x23\x8A\x6E\xE0\x6E\xA9\x1A\x87\xDB\x95'), #8F
		struct.unpack('<6Q', b'\x70\x24\x2F\x5F\xC8\x95\xB1\x5F\x56\xE2\x5C\xCB\x1D\x42\x42\xB1\x95\xF5\x4F\x59\x38\x62\x3E\x2D\xBF\x42\x37\xA4\x1F\x79\xE0\x6D\x17\x32\x9D\x38\x4C\x56\x7F\x44\xC8\x47\xE7\x76\xCB\x5F\x1D\x42'), #90
		struct.unpack('<6Q', b'\x4E\xB1\x36\x9C\x54\xDC\x9B\x5A\x48\xBD\x8E\x02\xE5\xE5\x4C\xC8\x4D\x90\x73\x1D\x68\x50\x4D\x88\x5E\xB4\x94\x4C\x92\x2C\xA2\xA7\x26\x94\xA3\x80\x47\xCA\x82\x95\x5E\xCA\x58\xDA\xDF\xFA\x04\x2E'), #91
		struct.unpack('<6Q', b'\x44\xAE\x95\xE5\x99\x54\xEB\xF0\x11\x0B\x0F\xBE\x0C\x79\x98\xAF\xB7\x38\x09\xA5\xEC\x5B\x53\x25\x48\xBB\xBD\x08\x7F\x5A\x97\x91\xDC\x14\x8E\xB6\x59\x48\xB2\xD6\xAA\xAE\xBA\xB9\x14\x32\x7E\x82'), #92
		struct.unpack('<6Q', b'\x7E\x54\x57\x16\x40\xA8\xCB\xF3\x49\xCD\x28\x6C\x5B\x8C\x88\x2D\x79\x6C\xAE\x39\x2A\xF2\xD2\x7F\x33\x6C\x06\x5B\xB4\xCC\x43\xD1\x16\x2A\x95\xBD\xE7\xB0\xA5\x98\x1D\x31\x28\xFE\x5E\x53\xB1\xD6'), #93
		struct.unpack('<6Q', b'\x9B\x42\xF6\x59\xB6\x42\xAF\x82\x5D\x31\xD2\x8F\x76\x6E\x2C\x2D\xB1\xC6\x75\x4D\x6D\x0B\x3C\x1D\xDD\xBE\x0B\x0A\x26\x0E\xBD\x6A\x9A\x81\x0A\x99\x58\x25\xFB\x86\xC4\x49\xC8\xCF\x59\x77\x9D\xBF'), #94
		struct.unpack('<6Q', b'\x5C\xAC\x58\x33\x98\xC0\x16\xF9\xDC\xC3\xB7\xC9\xE5\x90\x27\x49\xD4\x45\xA5\x1D\xEF\xB0\xCA\xD4\x12\xE8\x35\x25\xBC\xF3\xF9\xB2\x78\xA5\xAF\xB2\x33\x5D\x53\xC8\x01\x9F\xC3\x9E\x71\xAC\x6C\xD4'), #95
		struct.unpack('<6Q', b'\x05\xBF\xB7\x90\x69\xFE\xA4\x9A\xE8\xF0\x0F\x90\xF0\x52\xD6\x97\x2A\x69\x2E\xDE\x44\xD9\xD1\xE5\x9B\x37\xA6\xB3\x2F\xAA\x38\x0A\xB6\x58\x6D\x14\x65\x1D\x39\xDB\x12\x6D\x06\x83\x69\x72\xB6\x75'), #96
		struct.unpack('<6Q', b'\x99\x69\x12\x5B\xF5\x84\xE7\xF7\x95\xDA\xA3\x24\x8D\xE5\xB7\xF5\x52\x0F\x6D\xC6\x36\x12\x98\xB6\xA1\x36\xE7\x88\x0C\xB1\x44\x0D\x76\x5A\xB4\xFA\xED\x25\x0A\x06\xE7\x8B\xD8\xF2\x9F\xE8\xCA\x3C'), #97
		struct.unpack('<6Q', b'\xFF\x56\xE3\x83\xEE\xB3\xC4\xCB\xD4\xB7\x3E\xE6\x9D\x36\x83\x37\x6F\x39\xC3\xAE\xDE\x0D\x31\x6A\x4C\xC4\x4E\x29\x9E\x39\x4D\x81\x5F\x85\x6E\x92\x42\x7B\x41\x4B\xF2\x59\xCC\x8F\xFF\xFB\x1B\xA6'), #98
		struct.unpack('<6Q', b'\xC9\x4D\x84\xE6\x90\x7D\x0A\x0E\x83\x67\xFB\x5D\x0D\xE6\x70\xC8\xED\x2D\x26\xBD\x52\x0F\xE9\x63\xD8\xD4\x85\xA6\xE1\x0D\xA2\xD2\xB3\x5E\x42\x28\x66\xAC\x07\xC3\x65\x7D\x20\xF6\x11\xD4\xB2\xA4'), #99
		struct.unpack('<6Q', b'\x04\xF0\x74\xC5\x8D\xA1\x95\xA7\x8A\xC7\x44\x62\x36\x9A\x20\xC4\x2A\xF1\x3A\x27\xC5\xE1\xDC\x8E\x70\x42\xF7\x08\x3E\x59\x84\xDD\x13\x05\x53\x29\xFB\x7B\xAB\xB6\x47\x2A\x55\x06\x92\x84\xD7\x6B'), #9A
		struct.unpack('<6Q', b'\x77\x45\xFD\xE6\x4F\xA2\x9F\xE4\x9E\x5D\x8F\x96\xF8\xDF\x37\xC6\x0C\x13\x1A\xC5\x14\x7E\x67\x13\x53\x88\xFA\x96\x9D\xAB\x7D\xD1\xDE\x42\x5D\x18\xE1\x08\x06\x0C\xCC\x93\xCA\x04\x83\x32\xFB\x33'), #9B
		struct.unpack('<6Q', b'\x77\x50\x67\x29\x1B\x96\xA0\x8D\xB6\x50\xB3\x60\xC2\xE2\x47\x7E\x26\xE9\x63\xD1\x61\xF4\x3F\xEB\x3A\xFF\x8C\x5A\x67\x48\xF7\xD6\xE6\x65\x9F\x91\x61\xD1\x87\x18\x4F\xD8\x5C\xF3\x32\x31\x63\x81'), #9C
		struct.unpack('<6Q', b'\xC4\xF8\x29\x6F\x56\x88\x99\xAA\x80\x1E\xCE\xE5\x13\x5C\xC5\x8C\x22\x38\x4D\x72\x04\x6F\x3A\x60\x6C\xC2\x81\x8E\x2B\xFF\x70\x74\xBE\xFC\xC2\x9B\x0C\xAC\x31\x88\x27\x02\x73\x0E\x97\xFB\x65\xB0'), #9D
		struct.unpack('<6Q', b'\x46\xC0\x59\xE0\x37\x77\x7A\xD2\xA6\xE6\xDC\x9B\x0C\xB1\xE5\x27\x7F\xC7\x36\x5D\x56\x82\x5F\x73\x0D\x0B\x5C\x54\x4F\x4A\x6B\x8E\x84\xA5\xA1\xF8\xF5\xAC\xD2\x0B\x33\x4A\xFB\xDA\x99\xA4\xB7\xE4'), #9E
		struct.unpack('<6Q', b'\xA2\xA3\x67\x4E\x65\xC2\x13\x20\x18\x3C\x94\xF6\x90\x8A\x77\x3A\x89\x21\x28\xA0\xFA\xA3\xFE\x03\xD3\x1D\x5E\x64\x50\xA1\x08\x18\x8A\xD4\xA9\x05\x50\xF1\x9D\xED\x25\x2A\x56\xBA\xF7\x65\x5A\xA0'), #9F
		struct.unpack('<6Q', b'\x69\xB9\x74\xF5\x90\x8B\xC7\x16\xB0\x75\x83\xB1\x54\x44\xA5\xAA\x4D\x64\x83\x22\x43\x6B\x2F\xB3\xAD\xDD\x3A\x0F\xCB\x06\xE5\x72\xED\xC0\x30\xFC\xAC\xCF\x25\x3F\x52\x1B\x64\x77\x5D\x22\x4F\x59'), #A0
		struct.unpack('<6Q', b'\x3C\xA7\xE0\x34\x8E\x33\x8A\x3D\x29\x55\xA9\xE7\xB5\x91\x29\xFB\x3E\x81\xAA\xC0\x7E\x4F\xBC\x93\xD8\xD2\x0D\xCB\xC5\x73\x72\xFC\xDC\x71\xE8\xB3\xFE\xAA\xD8\x6F\x96\x8D\x2B\xE2\xB7\x4F\x88\x15'), #A1
		struct.unpack('<6Q', b'\x0B\xA3\x1F\xBB\x36\x4F\xCD\xEB\x88\xFA\x9C\xC7\xFB\x12\x65\xDA\x8A\x8B\x2F\x9C\x3C\xB1\x0D\x29\x81\x11\xE9\x71\x0D\xDE\x97\x43\xE8\xA0\x0E\x0F\xC6\x2A\xC7\x92\x25\xB3\x21\xFB\x93\x3E\xD5\x6B'), #A2
		struct.unpack('<6Q', b'\x44\xFC\x27\xEA\x69\xE2\xA7\x01\x14\xCE\xE2\x27\x6D\x9A\xEE\xB9\xEC\xC8\x4D\xF4\xB0\x39\xE5\x9C\x48\x20\x91\xA3\x66\x0B\x76\x93\xC7\x22\x14\x65\xDE\xA4\x40\x89\x43\xBC\x6A\x37\xE2\x25\x93\x47'), #A3
		struct.unpack('<6Q', b'\xAD\x70\xE8\x3E\xAF\xD1\x78\xB9\xB4\xBF\xC2\xBC\x39\x20\x10\x1A\xA7\xB5\x2F\xF8\x67\xE4\x26\xFE\x18\x15\x37\x8A\xE7\xAA\xA2\x7A\x0C\x03\xFE\x0E\x98\x49\xC0\xA6\x55\xCF\xB9\x31\x5A\x16\x5B\xCE'), #A4
		struct.unpack('<6Q', b'\xF3\x7B\x5E\xFA\x0E\xA7\x79\xE0\x32\x70\xCC\x01\xF8\x15\x12\x59\x2E\x6A\x41\x31\x52\x48\xEB\xC6\xB4\xF7\x3D\x31\x50\x55\x01\x79\x71\x91\xC6\x70\x7A\xE8\xBE\x6D\x30\x98\xAD\x8F\x3B\x22\xA9\xB3'), #A5
		struct.unpack('<6Q', b'\x0A\x2C\xF3\x6B\x24\xC5\x65\x1E\x88\xAC\x12\x71\x88\x71\x6F\x74\x3A\x80\x4C\x1E\x0D\xEE\x68\x37\x2A\x47\xD9\x17\x92\x5A\x28\x13\xDA\x2F\xB9\xC5\x3E\x65\xD3\x50\x16\x1B\x1C\x12\x25\x81\xD0\xD3'), #A6
		struct.unpack('<6Q', b'\xE0\xC6\xD7\xA7\xB8\x58\x66\xB2\x26\x58\xA7\xF6\xB8\xBA\xBA\x98\x2F\x1E\x57\x7C\x5B\xDC\x5C\x0C\x15\x3C\x7C\xE2\x35\x7C\x75\xA0\x23\x53\x9F\x98\xEC\xA5\x4C\x5A\x81\x31\x35\xD1\x17\x09\x04\x12'), #A7
		struct.unpack('<6Q', b'\xCE\x20\x0F\x1D\x5D\x3F\x3C\x28\xFA\x80\x6E\xA7\xF7\xFE\xEF\x35\x7B\x60\x0A\xB6\x6F\xC2\x84\xC2\xAB\xD4\x51\x31\x97\x41\x5C\xF4\xA4\xE6\x81\xC6\xDF\x49\x04\xDF\x40\x29\xBE\x44\x31\x92\x4E\x9F'), #A8
		struct.unpack('<6Q', b'\x09\x6B\xC5\x48\x83\xBA\x7E\xA3\xBB\xA1\x50\x21\xAF\xA5\xA4\x1B\x02\x5E\x86\xA1\xEC\xE2\x38\x85\x4F\xB5\xE6\x94\xF0\xC5\x70\xEC\xCC\x1C\x2F\x3E\x0E\x2F\x53\x28\x67\x24\x52\x8C\xAB\x81\xA4\x6A'), #A9
		struct.unpack('<6Q', b'\xEC\x74\xF4\x64\xCC\xCB\x6B\xCB\x8E\xC9\x99\xA2\x2D\xC9\x6A\xB0\x18\x40\xEF\xD1\xD7\xAA\x03\x67\xE2\xBF\x43\x79\xC0\xC9\x74\xF6\x7B\x9A\xD6\x62\xB9\xF9\x9A\xF3\x31\x46\xFA\xD9\x88\x3C\x56\x2C'), #AA
		struct.unpack('<6Q', b'\x89\xCF\x6A\xEB\x03\xEF\x9F\x91\xE3\xDB\x09\x86\xDA\x22\x7E\x6C\x9E\xAD\xDF\xB8\xE6\xE2\xA1\x1F\xCE\x6C\xE7\x84\x35\xCC\x95\xAB\x15\xFE\xB3\xC2\xA8\xDA\x04\x1D\x9A\x88\x9F\x1F\x7A\x00\x43\x1D'), #AB
		struct.unpack('<6Q', b'\x74\x17\x08\x89\x57\x8E\x81\xE7\x0E\x9E\x65\xAF\x29\x44\x70\xC0\x6D\x96\xD9\xB9\xEC\xCA\x73\xA3\x93\xF4\x4E\x60\xFB\x5B\xB1\x09\xEB\xD8\x30\x8D\x1C\x6D\x41\x37\xC1\xB0\x5B\xC9\x0C\x9C\xE7\x14'), #AC
		struct.unpack('<6Q', b'\x0B\x02\x2A\x14\x5F\x47\x4F\x52\x9A\xAF\x4B\x2E\xBA\x51\x64\x1C\x6E\xAF\x1A\x4A\x99\x9D\x53\x9B\x13\xD1\x97\x2F\xE4\x5F\x27\xF4\xC3\x09\xE7\x61\x7D\xC0\x27\xC4\xB2\x15\xF9\x2F\x22\xDF\xCC\x24'), #AD
		struct.unpack('<6Q', b'\xB8\x69\x16\x8D\x7F\x74\xDE\x14\x43\x2D\xD6\x85\x1B\x53\x0E\x4E\x8D\xE1\xA2\x1B\xD7\xA7\xE2\x78\xBF\x3F\x43\xB1\x89\x6B\x94\x1D\x78\x93\xF8\x73\xED\x87\xE3\x00\xC6\x32\xA8\x83\x82\x2A\x99\x52'), #AE
		struct.unpack('<6Q', b'\x9D\x00\x21\xCB\xFB\x69\xDB\xCB\xEA\x69\x82\x4E\xEC\xFD\xC6\x0E\x3F\xD1\xFE\x89\xAA\x08\xD9\x79\xF0\x3E\x10\xA1\x53\xE5\x48\x75\x44\xC9\x5E\x11\x8A\x04\xE4\xBB\xD1\xDF\xBD\x7F\xBF\x0E\xC6\xE1'), #AF
		struct.unpack('<6Q', b'\xB7\x9F\x21\xEC\xA6\x2A\x52\x24\xDA\xDD\xDE\xA5\xE8\x52\xC9\x78\xF0\x9D\x92\x97\xEF\xF9\x22\x28\x82\xB6\x58\x82\x1D\x9D\x0C\xA7\x93\x1E\x34\x6D\x64\xAC\x37\x8B\x0E\x5E\xBD\x4B\xFF\x34\x7C\xD6'), #B0
		struct.unpack('<6Q', b'\x5A\x42\xBC\xFA\x22\xCD\x70\xE7\x5C\x07\xC5\xAC\xD3\x34\x9C\x7E\x23\x66\xC7\x7B\x8C\xA7\xFF\x0A\xF2\x03\x2C\x85\x51\x98\xED\x4E\xFE\x26\x04\xE6\x6E\xEC\x03\x72\x4E\xDC\x22\x50\xB8\xCF\x17\x65'), #B1
		struct.unpack('<6Q', b'\xA1\x97\xF9\x50\x96\xF8\x81\x15\x9E\xD3\x07\xF4\x79\xE9\x9C\xDB\x3F\x7F\x34\x39\x0E\x80\xD4\x0A\x2F\x39\xD1\x9F\x8D\x5B\xC3\xC0\x43\xAE\x67\x61\x4A\x89\x4D\xE7\xC1\x9A\xCA\xE3\xF4\xE2\x62\xD4'), #B2
		struct.unpack('<6Q', b'\x87\xC7\x41\x5B\x5C\x54\xEA\x05\x90\xC6\x04\x61\xC5\x2D\xCD\xCE\xCB\x63\xC9\x6C\x74\xC0\x37\x73\xB1\xB8\x36\xF1\x81\x8B\xF0\xFD\x90\x29\xC7\x59\x21\x86\x7A\x7F\x2E\xED\xB0\x6C\x1B\x11\xD9\x0A'), #B3
		struct.unpack('<6Q', b'\x67\x03\x92\x2A\x03\x78\xE5\x6F\x64\x21\x83\x5C\xE5\xAE\xEF\x49\x57\xEF\x5A\x36\x97\x79\x15\xB3\x7B\xC6\x03\xC9\x17\x49\xF8\x98\xF3\x7B\xA6\x2A\xDA\x75\xCF\x75\xA4\xD8\x83\x9E\xB6\x8A\xCD\xD2'), #B4
		struct.unpack('<6Q', b'\xA0\x35\xE3\xC4\xB5\x9E\xB9\x65\x80\x52\x25\x27\x92\x75\x6A\x48\xC0\xEA\x86\x8F\x55\x3B\x7A\x4F\xF0\xDB\x86\x26\xCC\x89\xDF\x03\x11\xA3\xEA\x4E\x16\x93\x56\xA9\x09\xDB\x5B\xEF\x07\x93\x14\xD3'), #B5
		struct.unpack('<6Q', b'\x41\x70\x52\x29\xB5\x61\x32\x65\x41\x92\x78\x9A\xCA\x9E\x55\x9A\x7A\x67\xE8\x09\x0E\xFF\x31\x59\xEC\x58\x14\xF4\xFD\x3A\xD5\xF6\x1E\xF8\x6B\xE9\x42\x44\xB9\x02\x7D\x5C\xA9\x1F\x46\x1B\x4A\x5C'), #B6
		struct.unpack('<6Q', b'\x56\x6A\x09\xF3\xB3\x3D\xFA\xB6\xA0\x41\xF1\x69\x83\x67\x1E\xAD\x88\x51\x5E\xED\xA9\x8D\x70\x81\xED\x1D\xA8\x37\xD3\x29\xEE\xE2\xC0\x37\x9B\xAC\xF0\x74\xEC\xD7\x37\x60\xC7\xA3\x6F\x5A\x78\x1B'), #B7
		struct.unpack('<6Q', b'\x33\xED\x02\x82\x61\x3D\x15\xAE\xDD\x46\x5E\x18\x15\x11\xE9\xE8\xBC\x72\x43\x36\x34\xBA\xD9\x05\x3D\xE1\xDB\x9B\x62\xE1\x2E\x67\xCB\x61\x4A\x62\xA8\x5E\x2F\xFC\x35\x50\x92\xE3\x8B\x19\x43\xD8'), #B8
		struct.unpack('<6Q', b'\x82\x05\x13\xFC\xC1\xF7\xE8\xBC\x72\xB4\xC5\xC5\xBD\xAE\x71\xA5\x9C\x1E\x91\x83\x05\xC7\x08\x7F\x58\xD9\xA9\x95\xEA\x16\xBC\xF4\x72\x2C\xEA\xDD\x6B\x3D\xEE\xA9\x83\xF6\x4D\x99\x74\x1B\x65\x1B'), #B9
		struct.unpack('<6Q', b'\x1D\x46\xEA\x16\xDA\xD9\xA7\x13\xB1\x32\xDC\x4E\xFC\x04\xCF\x7F\xF8\x82\x85\x13\x39\x7C\x60\x66\x91\xA0\x73\xAD\x6E\x54\x31\x88\xFD\xF5\xD1\x49\xF3\xC4\xA4\xD8\xDB\x38\x16\x70\xD5\xC9\x75\x49'), #BA
		struct.unpack('<6Q', b'\x8F\xC7\x0E\x1F\xA4\xA6\x0A\xBF\xF3\x26\x02\x49\x4F\xF6\x2B\xDC\x38\x35\xD0\xF2\xAA\x13\xA3\xCD\x63\x5A\x59\x60\x44\xDA\xB8\x04\xBA\x63\xB7\x9B\x72\x1B\x3F\xA2\x72\x72\xF7\x73\x6A\x73\x93\xF7'), #BB
		struct.unpack('<6Q', b'\x3A\x6E\xDF\xBF\xAE\x34\xA3\x3B\xB2\xE5\xA1\xF6\xEA\x37\x20\x21\xE0\x41\xBF\x20\x19\x3C\x4B\x00\x6A\x70\x62\xA8\x5B\x1B\xA1\xD8\xAB\xF8\x15\x73\x02\xE5\x79\x65\x3D\xE1\x54\x6D\xBB\x9C\x1F\x3C'), #BC
		struct.unpack('<6Q', b'\xD7\xA2\xA2\xBD\x02\xDE\x24\xBE\xDE\xE9\x27\xE0\x48\xA6\x69\x4D\xC5\x90\xBD\x73\x53\x2A\xBC\xFE\xDB\x14\xA4\x55\x0E\xAC\xC7\x42\xDC\x9E\x65\x21\xB7\x62\x59\xD6\x23\x0D\xFB\x73\x2E\x1B\x83\x53'), #BD
		struct.unpack('<6Q', b'\x94\x0E\x08\x94\x3D\x8C\xE6\x3D\xDC\x87\xE0\x68\xC8\x0B\x9E\x1D\x5C\xB3\x70\x42\xF1\x1E\xAD\xE7\xE5\x67\x0F\x3E\x37\xCF\x14\x9A\x00\xE6\xB7\xE6\xAB\x98\xEF\x00\xAA\x4E\x23\x18\x1F\x3F\x35\x82'), #BE
		struct.unpack('<6Q', b'\x1C\xF5\x2C\x93\x9E\x54\x34\x60\x07\xD4\xB4\x3E\x6E\xE3\xE2\x3C\x3F\x1E\xB3\x22\xCA\x80\xBA\x9F\xF8\x93\xAD\x8C\xAC\xAE\x2E\x14\xF2\x9C\x90\xE5\xA6\x5C\x3D\x5D\xB5\x25\xB6\x82\xB4\x70\x2C\x1B'), #BF
		struct.unpack('<6Q', b'\x38\x19\x45\xFE\x77\x22\x21\xBC\xDA\xF4\x3E\x3A\xEF\x37\x16\xE2\xD5\x7A\xCE\x55\xD1\x70\x9C\xC3\xF8\x3A\x1F\x49\x78\xDC\x3E\x5C\xDF\x16\x7D\x84\x9B\x5C\x40\x1A\x75\x18\x29\x6E\x05\x6B\x44\x0F'), #C0
		struct.unpack('<6Q', b'\xDC\xE1\xB8\x58\xC7\x91\xD7\x69\xCA\xB0\xB8\x12\x45\xA8\xD8\x49\xBC\x1D\x06\x5E\xB0\xF7\x74\x24\x6D\x3F\x49\xB1\xFF\x97\x9C\xA7\x72\x97\x4A\x76\x7B\xE7\xB9\x6F\xFE\xD8\xB6\x4C\x61\x93\x15\x31'), #C1
		struct.unpack('<6Q', b'\x27\x5E\xA4\xD5\x77\x8D\x7F\xF6\x72\x65\xD2\x58\x7F\x28\xA3\x17\x80\x20\xE5\x95\xE5\x92\x27\x18\x90\x3A\x37\x2C\x05\xB7\x15\x7D\x33\x07\x91\x6B\x97\xF2\x6C\x53\x04\xD5\xCB\x3A\xB4\x83\xE4\xF0'), #C2
		struct.unpack('<6Q', b'\xEE\x27\x8E\xD0\x51\xE3\x1E\x9F\x50\x1C\x25\x01\x01\x8F\x15\x76\xFA\x67\xDE\x2F\xD2\x9F\x26\xC1\x5E\x3F\x6F\x6E\xA2\x30\x19\x6C\x71\xB0\x96\xA3\xAA\xA0\x2A\xA3\xD9\x8B\x93\x16\x94\x59\x7A\x49'), #C3
		struct.unpack('<6Q', b'\x67\x17\x7D\x8E\xF0\x16\x99\x56\x25\xC7\x65\xF1\x46\xF0\x20\xAB\xDA\x32\xDE\xC3\xAF\x1C\x4C\xBD\xDF\x4C\x27\xCA\x1C\xF6\x19\xCF\x79\xF7\x46\x04\xC9\xBD\xEB\x4D\x0E\x9B\x42\x07\x2E\x52\x07\x0D'), #C4
		struct.unpack('<6Q', b'\x1F\x92\xF3\x72\xFB\x95\xF2\xB0\xEE\x80\xAD\x9F\xA7\xDF\x92\xC6\xA6\x8D\x23\xE0\x29\x16\x01\x69\x2C\xEE\x0A\x91\x22\xAF\x46\x00\xBD\x9D\x4A\xC9\x94\xBB\xDE\x77\xA8\x03\x8A\xEE\x10\x11\x40\x4B'), #C5
		struct.unpack('<6Q', b'\x8B\x9F\xD6\x08\xCA\x2F\x6D\xFF\xCF\x5A\x2A\x71\xF1\x2A\x59\x97\xA6\x0C\xBB\x03\xF1\x0A\x54\x9D\x10\x8F\x32\x72\xE5\x33\x59\xD5\xA7\xF7\xB0\x2F\xCA\x18\x72\xEC\xAD\x52\x31\x58\xCE\x4E\x15\x8D'), #C6
		struct.unpack('<6Q', b'\xFA\x8E\xB1\xBB\xA3\xB6\xEF\x6C\x5D\xBB\x48\xAD\x37\x04\x7A\xE4\xB4\x73\xEB\x8C\xD3\x95\x96\xAC\x06\x69\xB0\xCE\xB6\x68\x54\x4E\x1F\x59\xF1\x43\x7B\x00\x57\x3A\x51\x26\x1C\xD2\x0E\xF1\xED\x16'), #C7
		struct.unpack('<6Q', b'\x6B\x2A\xE9\xAE\x4C\xA1\x12\x30\x6E\x7B\x39\x16\x0A\x2D\xCE\xC7\xB6\x36\xDD\x6C\x67\x1D\xCE\xB2\x45\x3E\x8C\x3F\x68\xF9\x63\xF5\x43\x94\x73\xEE\x43\x23\x11\x3C\x7B\x45\x31\x0C\x38\x36\xFD\xD2'), #C8
		struct.unpack('<6Q', b'\xE8\x89\x9C\xE7\xED\xD7\x82\x10\x97\xC5\x07\xD8\x88\xC0\xD1\x57\xAF\xF2\x72\xC7\xCC\xE9\x5C\x9E\xD9\xB8\x1B\xF6\xCA\xD1\x0A\xD3\xA4\x9C\x93\xA6\x24\xD0\x40\x1E\x04\xF1\x8D\x35\xEB\xDB\x85\x49'), #C9
		struct.unpack('<6Q', b'\xE4\xC5\x9B\xC0\x3F\xBF\x6E\x61\xED\x67\xB3\x50\xDA\xC7\x49\xA9\xB9\xCD\x79\xAF\xF8\x9E\x28\x0C\x69\x64\x39\x5F\x3A\x30\x23\x5C\xB8\x9D\x6A\x55\x7F\x72\x16\x33\xFC\xC3\x29\x32\x94\xEC\xCC\xB7'), #CA
		struct.unpack('<6Q', b'\xAE\x5E\x46\x4D\xC0\xB8\xB4\xB9\x98\x77\xAA\x54\x6D\x83\x44\xED\x8E\x91\xF5\x0F\x5F\xDC\xF7\xA6\x23\x4E\x8E\xF6\xC0\xD1\x0D\x70\xAC\x22\x62\x51\xA0\xFA\xF2\x4B\x42\x86\x34\x02\x60\xE2\x2A\x3C'), #CB
		struct.unpack('<6Q', b'\x2A\x53\x79\x8B\x87\xCE\x7E\x8A\x7D\x49\x6B\x11\x36\x38\x61\x23\xC7\x0A\xC1\x04\x98\x5E\x66\xFE\xDB\x50\x1D\x60\x49\xB9\xE6\x45\x92\xAE\xF1\x87\x48\x9E\x2A\x5B\x79\x20\xFF\xA4\x4B\xF9\x7F\x44'), #CC
		struct.unpack('<6Q', b'\x60\x80\x58\x1F\x1B\xED\x1C\xE3\xE7\x1B\xD5\x70\xCE\x18\x0A\xE6\xF3\xE1\xA2\x01\x1E\x8A\xC0\x62\x7C\xE7\xEA\x87\xD6\xB3\x8E\x00\x24\x42\x13\x54\xF2\xA7\x01\x1D\xA0\xF3\x56\x6E\xE7\x7B\x00\x7E'), #CD
		struct.unpack('<6Q', b'\x4B\x8C\x1D\x9E\x29\x95\x19\xE3\x97\xCD\x3B\x61\x50\xE6\xFB\xD1\x6D\xC0\x06\xA8\x36\x3F\x3C\x1D\xC6\xA9\xF4\xD9\xC3\x78\xF2\x49\x19\x0B\xEA\x5F\x8E\x51\xAC\x0E\xDA\x6E\xA0\xC9\x90\xA3\x37\x7D'), #CE
		struct.unpack('<6Q', b'\x3E\x81\x27\xD8\x3E\x2A\x8F\xE3\x7E\x54\x75\xD2\x09\x4C\xCD\xC5\xD6\xBE\x2A\x3F\x31\xE8\xF0\x1C\x5E\x6C\xCF\xEC\x47\xE8\xCF\x9E\x3D\x25\x74\xC9\x84\x81\xE6\x60\xF8\x66\xDD\x6C\x63\xB7\x6A\xED'), #CF
		struct.unpack('<6Q', b'\x83\xEC\x4D\x95\x4E\xDC\xC9\xB6\x7B\x13\x1E\x3C\xD4\x86\x7A\x33\xF2\xD4\x4F\xFD\xEC\x5C\x1A\x61\x0A\x49\x9F\x25\xC6\xE0\xAB\x4F\xF9\xFE\x10\x77\x11\x93\x43\x27\x91\xB5\x29\xDE\xFF\x64\x0A\xF8'), #D0
		struct.unpack('<6Q', b'\x36\x81\x83\x47\xB4\x95\xB9\xCA\xC9\x98\x69\xD6\x20\xEE\xBA\xF8\x07\x2B\x4D\x36\x91\x86\x7F\x03\x10\xF8\x49\x49\x82\xFE\x11\xF7\x44\xE9\xB4\x73\x27\x4E\x1B\x0D\x29\x5E\x4F\xC5\xFE\x24\xBA\xFA'), #D1
		struct.unpack('<6Q', b'\xCE\x8F\x96\x74\x29\x0B\x8C\x30\xFA\xE7\xFD\x64\x2B\xE8\x5C\x30\x07\x9C\xC7\x7A\xFA\x57\xB9\xCB\xB1\x5F\x7C\x9D\x69\xB1\x8B\x93\x59\xD8\xB4\x52\xD6\x5F\xFC\x4D\x54\xB3\x9A\x1D\xC2\x7F\x6D\xC7'), #D2
		struct.unpack('<6Q', b'\xB9\xB2\xB6\xDE\x2A\x9D\x3F\x21\xE6\xA7\x8B\x80\xDE\x1A\x50\x0A\x1D\x12\x58\xAD\xF9\xA9\xF1\xE1\x07\x03\xD3\xE3\xE9\x4C\xDA\xE1\x2E\xC2\x51\xBD\xEE\xF0\x8F\x4F\xF0\x02\xC9\xC0\x79\x5D\xAF\x5E'), #D3
		struct.unpack('<6Q', b'\xFD\x29\xFA\x86\x94\x6B\xD8\x66\x1B\x88\xC6\x57\xC3\xFA\x59\xAC\xA6\x36\x6E\xA3\x95\x55\xC5\x6B\xEC\x03\x51\x31\x6D\xF0\x2B\x16\xF7\x26\x1F\x5E\xAA\xF6\x6C\x12\x2A\x23\xCA\x9C\xC4\x80\xF0\x9D'), #D4
		struct.unpack('<6Q', b'\xFE\x42\x8D\x3C\xCA\xDB\xA2\x26\x29\x6B\x74\xFB\x40\xCB\x9D\xA0\xB7\xB9\xE9\xB3\x06\x77\x00\xCB\x37\x78\xC8\x27\x85\x24\xCE\xAB\x9E\x61\x4B\xD7\xF5\x3D\x5E\x82\x30\x8C\xB5\x9C\x9D\xFC\xD7\xF2'), #D5
		struct.unpack('<6Q', b'\xE3\x1B\x14\x21\xBA\x46\xA5\x5C\x98\xAA\x34\xD7\x6B\x71\x88\x45\xA5\x57\x4D\xAF\x9A\xE6\xEE\x4A\x34\x47\x7F\xD8\xCB\xD8\x1E\xE0\xB6\x4F\x1A\x60\x9D\xA5\x3A\x24\xF5\xE0\xE7\xBB\x4D\xC7\xAC\x3E'), #D6
		struct.unpack('<6Q', b'\x12\x7E\xD3\x18\x7D\xCC\x3D\x12\x03\x9E\x84\x59\x9B\xBD\xD7\xA9\x97\xC8\x13\xA6\x03\xA2\x44\x89\x52\xC4\x2E\xEE\x71\x22\x61\x17\x19\x22\x5F\x3E\x23\x1C\xC1\xCF\xF0\xFD\xF8\xB0\xE1\x7D\xA4\xF0'), #D7
		struct.unpack('<6Q', b'\xAC\x28\x58\xE8\x3B\x26\x84\x4D\xCF\x93\x73\x0F\xF3\x68\x57\xC4\x70\x0A\xF8\xF7\xC0\xC4\x4C\x0D\xFE\x65\x69\x84\xDD\x80\x78\x01\xCC\x8D\xE6\xBB\x2D\x9F\xBC\xC3\x36\x35\xAA\x47\xE2\xC7\x36\xC2'), #D8
		struct.unpack('<6Q', b'\xC3\x54\xDA\xB7\xB8\xE4\xB4\x8E\x09\x94\x38\xD1\xA6\x27\x14\xFA\xB9\xF0\x19\x8B\xAB\x49\x44\x74\x94\xDB\x0B\x2D\x2B\xE3\xFB\x6B\xF5\x4F\x6C\xE0\xB6\x3E\x31\x91\x8D\x57\x78\xE8\x4F\xA6\x70\x01'), #D9
		struct.unpack('<6Q', b'\xF5\x94\x36\x7E\xC1\x05\x9E\x17\x8B\x2F\xE7\xA9\xAE\x9D\x59\x31\xF3\x27\x20\x35\xB1\x04\x87\x02\x9A\xF0\x6E\x5B\x97\xF4\xB6\xDA\xE7\x8F\x6E\xBC\x0A\x73\x3E\x5E\xD1\x63\x47\x41\x47\x36\x75\x41'), #DA
		struct.unpack('<6Q', b'\x8A\xB9\x34\x04\x2D\x4F\xBA\xDD\xAC\xEA\xDA\xF3\x22\xC5\xBA\x0F\x5E\x71\x31\xBB\xD4\xAE\x15\xD3\x25\x54\x9E\xEF\x78\x3D\x1F\x5C\xEB\x37\xF6\x8B\x2D\x54\xDD\xB8\x1C\xCE\x95\xE0\xEE\xA2\xA8\x5F'), #DB
		struct.unpack('<6Q', b'\x1A\xDF\x7E\x11\x3B\x9C\x51\x87\x46\x6E\x30\x4B\x92\x69\xC4\xF7\xD6\x3F\xE9\x6E\xD5\x59\xC8\xD2\xFB\x44\x28\xB4\x5D\x97\x43\x7E\x4E\x39\x83\x04\xFF\xE5\xEC\xB6\x98\x33\x1E\xE9\x95\x25\xD6\x02'), #DC
		struct.unpack('<6Q', b'\xBC\xC7\xD4\x1F\x11\x12\x68\x32\x7D\x1A\xD8\xC2\xA2\x44\x95\x09\x49\xCF\xEE\xAC\x12\xBB\x29\xB4\xC8\x5F\xD1\xAA\x0C\x08\x16\xF9\xC0\xE0\x73\x04\x19\xBE\x14\xE8\x10\xF4\xA6\x5F\x6C\xA6\x45\x82'), #DD
		struct.unpack('<6Q', b'\x06\x1D\x18\x33\x1D\xF5\x28\x7C\xFD\x25\xF2\xD6\xAC\xBF\xE1\xC2\xCA\x58\xB0\x3F\x37\x77\x74\x6D\xB4\x0F\xBA\xAC\xA4\x20\x31\x27\x2F\x3F\xBC\x90\x21\xE2\x40\xE7\x30\x88\x75\x65\x07\xF5\xF7\xEB'), #DE
		struct.unpack('<6Q', b'\xB1\x1E\x15\x5C\x66\xBB\xD4\xA9\x57\x1B\x7F\x57\x32\xEA\xA1\xC6\x7A\xC8\xD1\xBB\x91\x00\x2A\xF1\xA5\x06\x0D\x6D\x26\xEB\xED\x52\x06\x83\xD3\x22\x79\x45\x82\x66\xE1\xFF\x81\xAC\x03\x80\x20\x9E'), #DF
		struct.unpack('<6Q', b'\xEE\x14\xCC\x8F\xFF\x6D\x45\x88\x38\x63\xD1\xF9\x13\x68\xDC\xB7\x40\xCD\x19\x67\x9B\x77\x4C\x77\xCE\xB6\x7C\x56\x66\xAD\x1E\x42\x61\x73\xB1\x4F\x6F\xE6\x04\xEF\x67\xEE\xB8\x95\xF9\x07\x8C\x70'), #E0
		struct.unpack('<6Q', b'\x0A\x66\x15\x52\x06\x6C\x51\xFD\x6D\x42\x7C\x0F\x5E\xFC\x97\xBC\x89\xE3\x5C\xB5\xDA\xE2\x72\x3D\xFD\x89\x72\x4B\x92\x97\xBB\x48\xA5\xFD\x78\x65\x47\xC3\xDB\x01\x56\x6D\x14\x25\x49\xB4\x64\x54'), #E1
		struct.unpack('<6Q', b'\x36\x40\x73\xFE\x28\x5F\xB4\x12\x3C\x4C\x4B\x17\x2D\x85\xAB\x2B\xFA\x2A\xD1\xA0\xF0\xF5\xD3\xF0\xB1\x8F\xF8\xC0\x8F\xCE\x09\x3E\x27\x02\x69\xCF\x05\xE4\x2B\xBA\xAC\x9B\x5E\xAD\xAD\xCD\x00\x31'), #E2
		struct.unpack('<6Q', b'\xF3\xF0\xCB\x16\x2C\xDB\xC5\x43\x83\x16\xAF\x00\x7B\xF7\x58\x41\x4A\x4B\xAE\xD0\x88\x48\x8F\x04\x5F\x1E\x35\x15\xF8\x98\xAD\x36\x02\x7A\x1F\xAF\x9C\x30\xD6\xC7\xC4\xF6\xB4\x45\x4D\x52\xFC\x4E'), #E3
		struct.unpack('<6Q', b'\xC3\xD2\x29\xA9\xE0\x60\x4F\x25\x7F\xD8\x71\xC2\xB4\xE6\xAA\xAE\xAA\x02\xA2\x3A\x71\x0C\xDC\xD0\x7E\x02\x03\xB8\xDA\x04\x50\xD7\x21\xF4\xBD\xF2\x90\xFE\x2F\x82\x72\xBE\x99\x95\x9D\xFF\x98\x99'), #E4
		struct.unpack('<6Q', b'\x18\x40\xAA\x3D\x3D\x6F\xF2\xF4\x16\xEB\xCF\x33\x47\xCC\xA3\x63\xDD\x0B\xAE\x60\xEB\x09\x62\x59\xE7\x63\xD8\xF0\xF9\xB7\xBC\x43\x58\x7F\x74\x1F\x95\xD0\x05\x65\x47\x2D\x10\x7A\x1F\xD7\x34\x9A'), #E5
		struct.unpack('<6Q', b'\x78\xBA\x9A\x54\x0A\x66\x06\x6A\x90\xB1\x53\x13\x08\xF1\x3E\x26\x0D\x7B\xA5\x4B\x23\x35\x67\x09\xC0\x8C\xCE\x93\x02\x12\x4E\x1D\x44\x7C\x70\x2A\xD8\x3C\x4A\x4E\x34\x29\x37\x00\xEE\x27\x90\x69'), #E6
		struct.unpack('<6Q', b'\xDA\xDB\x70\xE6\xFC\x83\x0D\x0E\xDE\x79\xDF\x37\x1D\xD7\x04\x9F\x07\x66\x44\x35\x50\xD3\x43\x94\x57\x84\x88\xCE\x7D\xB8\x1F\x10\x18\x7C\x05\x24\x92\xF7\x7B\x62\xCB\xCC\x3F\x95\xFD\x00\x40\xDC'), #E7
		struct.unpack('<6Q', b'\xA5\xE0\xFF\xC9\xD6\x01\x69\xFF\xEF\x6D\xDF\x05\xDF\xAD\xDA\x2C\xDE\xBB\xEB\xD3\x76\xF4\xDA\x98\xB2\xC5\x42\xCE\x35\x71\xD3\x92\x5E\x90\x51\xCB\x72\x7D\x18\x45\x42\x5E\x44\xAC\x28\xC3\x34\x9B'), #E8
		struct.unpack('<6Q', b'\x32\xA5\x71\x18\x22\x4F\x3B\x59\xD9\x07\x4A\x3E\xC7\xEE\x58\x5B\xA6\x3B\x59\x51\x9B\x57\xA3\x14\xFB\x36\x86\xB9\x6A\x1D\x7C\x2D\x71\x41\x2D\x58\xC6\xF3\x51\x9B\x93\xBF\xFB\xC7\x3E\xE2\x2B\x48'), #E9
		struct.unpack('<6Q', b'\x32\xC9\x0F\x3E\x1D\xAA\x88\xCF\x64\x02\x9B\x98\x37\xF0\x18\x21\x9C\x9E\xAD\xC4\x83\xAC\x7D\xB3\xAA\x2B\xAA\x47\xCC\x94\xDE\xAC\xCD\x83\xCD\x6B\xC2\x39\x88\x11\xDB\xC9\x93\x31\x3A\x84\xB5\x32'), #EA
		struct.unpack('<6Q', b'\x26\x4C\x9C\xF5\xEB\x94\x36\x27\xC2\x48\x0D\xE5\xDE\x6E\xE8\x43\xDB\x3C\x64\xED\x93\x70\xF9\x67\xCE\xAF\x87\xD7\x01\x52\xAF\xA4\xFA\x55\xAC\x43\x32\x9F\xE8\x88\x2F\x5E\x97\x84\xCA\x4C\xCA\x48'), #EB
		struct.unpack('<6Q', b'\x34\x96\xD9\x90\x37\x3D\xF2\x07\x9E\x3E\x97\x09\x0C\x17\x52\x53\x1F\x18\x08\xD2\xEF\xEB\xF7\x4F\xB8\x7D\x17\x8E\xD4\x7E\x09\x5D\xD5\xED\x25\x42\xD5\x33\x32\xC8\xCD\x76\x0C\x36\x1A\x98\xB2\xBD'), #EC
		struct.unpack('<6Q', b'\xAD\x63\x1F\xC2\x35\xB7\x7B\xBF\x71\xC9\x9A\x1B\xB8\xEC\x2A\xD4\x2B\x9D\x61\x56\x95\x20\x84\x41\x37\xB7\x4A\xB2\x8D\x08\xFB\x56\xCD\xD7\x4E\xCC\x17\x90\x45\xEF\xF3\xD4\x02\x01\x75\x4E\x89\x68'), #ED
		struct.unpack('<6Q', b'\x4C\x86\x46\x16\xE1\x09\x30\xEC\xB3\x96\x89\x21\xBE\x6E\xB7\xAB\x84\x36\x43\x8B\xE4\x66\x43\xA4\x5B\x20\x8F\x5F\x9B\xCD\x50\x02\x6A\x6A\x86\x77\x21\x02\xF4\x37\xDF\x99\xBD\x46\xB4\x8E\xDC\x21'), #EE
		struct.unpack('<6Q', b'\xDB\x13\x03\xA5\x2D\x15\xA6\x0E\xA2\x2A\x7F\x7B\xBD\xFC\x18\x4B\x91\x40\xD3\xA4\x2E\xBD\x20\xA5\x41\x34\x32\x36\x08\xC2\x93\x92\x1F\x8B\xFC\x44\x3F\xAA\x2F\xF7\x87\x7A\x87\x0C\x22\xA9\xE7\x1A'), #EF
		struct.unpack('<6Q', b'\x87\xBE\xA8\x8B\xEC\xE7\xF2\xA2\xF6\x2D\xC4\xEA\x38\x4F\x78\x99\xDE\xA5\x54\x1F\xFB\xF5\x18\xD0\xEA\xA8\x89\x8F\x12\x8A\xDB\x8E\x25\x0A\x5F\x00\x01\xFD\xF4\xE9\x84\xA3\x43\xF4\x90\x50\x86\xF6'), #F0
		struct.unpack('<6Q', b'\xC8\xB6\x44\x8E\x02\xF1\x8D\x64\x4E\xA5\x8E\x58\x12\xAE\x9A\x23\xB2\x57\xEC\x8A\x09\x45\x32\xC7\x0E\xB6\xD4\xF8\x38\x99\xB6\x39\x99\x7F\xCC\x18\xA1\xCD\x8C\xE9\xAD\x99\xC8\x25\x9C\x34\x64\x65'), #F1
		struct.unpack('<6Q', b'\xD7\x8C\xAE\x10\x19\x07\xBA\xEE\xEB\x8C\x71\x44\xE4\xD4\x13\xAE\xC0\x6E\x11\x27\xEA\xCC\x38\x5A\x03\x95\x44\xB9\x50\xF9\x26\x66\x2D\x2A\xD9\x08\x0C\xBE\x20\x4D\xDB\x9D\x84\x9A\xE9\xD6\x99\xC0'), #F2
		struct.unpack('<6Q', b'\xEB\x1D\x2B\x57\x09\xF1\x6D\xB9\x27\x36\x50\xEF\x1E\xFA\x4E\xCD\xAE\x66\x87\x13\x2B\xC3\x92\x20\xCB\x1C\xFC\xFA\x09\x84\x73\x5F\xD0\xDB\x12\x87\x9F\xEF\xCE\x48\x79\xC6\xF8\x6A\x76\xE8\x20\x4A'), #F3
		struct.unpack('<6Q', b'\x31\x65\x47\x77\x86\x0D\xFA\x13\xFF\x8A\x21\xFA\x11\x03\x56\xF6\x19\xCE\x80\x27\xF5\x9D\x82\xDD\xF5\xE4\x99\x3D\xDF\x13\x2A\xA8\xC8\x44\x78\x7B\x89\xCF\x70\x55\xCA\xCC\x6C\xCD\x76\xBA\xE0\xB4'), #F4
		struct.unpack('<6Q', b'\x88\xCA\x6D\x59\xBD\x58\x63\x5E\xC4\x1D\xC1\x97\x49\xA8\x80\xC4\x93\x27\x16\xC4\xA1\xFD\xBD\x44\xB6\x81\x90\x35\x20\x59\x57\xEA\xB1\xA8\x42\xDE\xBD\x4D\x40\xCF\xFC\xE7\x5F\xF7\x06\x2A\xB7\x6D'), #F5
		struct.unpack('<6Q', b'\x69\x8F\x6B\x07\x56\x41\xFE\x2F\x1B\xDB\x85\x39\x67\xBD\x08\xBE\x77\x24\x71\x38\x23\xB1\xFB\x2E\x23\xC2\xCB\x86\xCA\xC6\xDA\xC3\x21\x19\x5C\xC6\x1D\x14\xF0\x00\x08\x16\xB8\x16\x68\x55\x4B\x69'), #F6
		struct.unpack('<6Q', b'\x3C\x8B\xF2\x57\x24\xEF\x0C\x59\xB3\x28\xEC\x15\x61\xC5\xB6\x09\xB0\xA6\xB9\xF0\x1E\x68\x38\x3D\x6E\xE8\x69\xF2\x87\xE6\x6E\x5D\x96\xF4\xD7\x04\xE9\x1F\xE0\x36\xE7\x56\x7A\x55\x59\x4C\x38\x7A'), #F7
		struct.unpack('<6Q', b'\xF4\xA7\x87\x23\x0F\xEC\x06\x1C\x4D\x95\xDF\x1D\x58\x41\xC4\xA1\x47\xCC\xE1\xD4\x25\x14\xFA\x61\xBD\x98\x11\x80\xA7\x9C\x7C\xBA\xB8\x46\x80\xF4\x89\x73\x9C\x30\x56\x3A\x1F\x58\xF3\x11\xC5\x6C'), #F8
		struct.unpack('<6Q', b'\x56\xC8\x40\xD4\x96\x92\xF4\x50\x35\xB8\x19\x73\xF2\xDA\x2C\x97\x8B\xB7\xA6\x2A\x86\x5D\x35\x2D\x3A\x34\xB3\xDA\xAF\xC5\x1B\xAF\x14\x0F\xAE\xE8\x12\x1C\x42\x4B\x60\xC6\x1A\x7F\x95\x2D\x59\x37'), #F9
		struct.unpack('<6Q', b'\xD3\x0F\x5E\x26\xA6\x5C\xA4\xA0\x75\xCB\xE8\x78\x6A\xCF\x23\x46\xB7\x57\x5E\x56\x4C\x86\x55\xD0\x9E\xBF\x03\x58\xDC\x87\x14\x9B\x24\xCB\x55\x1B\x56\x5F\xB8\xD9\x7F\xE6\x64\x4A\x01\xE1\x39\x41'), #FA
		struct.unpack('<6Q', b'\xC3\x40\x77\x63\xBA\xAA\x34\x26\xCA\xFE\xEB\xDF\x3A\x0F\x49\xEB\x43\xAB\xF0\xDD\x2F\x93\x68\xB9\xB2\xE9\xFD\xFC\x40\x1F\x48\x4A\x25\xA7\x9C\x2A\xF7\xD3\x26\x4E\x9A\x38\xA3\x85\x4C\x0F\xB9\x59'), #FB
		struct.unpack('<6Q', b'\x1F\xF0\xC4\xE1\xB3\x9D\x85\x2D\x49\x05\xF9\x25\x68\xCB\x8D\xF7\xBC\x3C\x84\x79\x6F\x9D\x63\x5E\x64\xE5\xB0\x32\xDA\x66\x38\x06\x23\x69\xF4\x32\x70\x7D\xC1\x1D\x1C\xA6\x25\xF8\xD8\x9F\xCF\x64'), #FC
		struct.unpack('<6Q', b'\x48\x86\xDC\x02\xBA\x7D\xDE\xD2\x78\x3C\x42\xFE\x4E\x0B\x86\x70\xEE\x3B\x80\x3C\x41\x46\x08\x13\xD7\x0C\x37\xD7\x1B\xE5\xEF\x97\x8B\xF4\x76\x37\xC6\x50\x72\x67\x8D\x15\x6A\xB7\x68\xBB\xD1\xBB'), #FD
		struct.unpack('<6Q', b'\x3B\x9B\xA1\x77\x69\xE1\x9E\x3A\x0E\x12\xB2\x11\x1E\xBE\x29\xF2\xB1\x1D\xC3\x43\xE4\xE6\xB4\xB5\xBE\xFD\x95\x65\x9C\x7F\xD0\x3D\x34\x7C\x54\x67\xAB\x8D\x49\xB4\x14\x63\x16\xB9\xD1\xF4\x43\xAB'), #FE
		struct.unpack('<6Q', b'\x33\xE2\xDD\xE2\x9D\x17\xFC\x08\xBA\x14\xC6\x80\x76\x82\xA1\xBC\x9C\x45\xAF\x0D\x28\x29\x52\x2C\xA7\x27\x05\xEC\xC4\x2B\x52\x1B\xCD\xDB\x23\xBF\x25\x5C\x12\x84\xFB\x83\x3F\x11\x44\xD0\x50\x78'), #FF
		struct.unpack('<6Q', b'\xBD\xC8\xE8\x41\x3B\x8E\x96\x1D\x67\xD7\xEE\x7B\xC7\xB3\xF0\x52\x0A\x86\x9F\x47\x3D\x23\x16\xB9\x56\xBE\xE4\x20\xDA\x3F\xEB\xCE\x6B\x8C\xF2\x5A\xA2\x74\x8F\xEB\x49\x4E\x9E\xB5\xA4\x3F\xC1\xBF'), #ZZ
	)

	__table_mix = (
		struct.unpack('<6B', b'\x05\x03\x04\x01\x02\x00'), #00
		struct.unpack('<6B', b'\x02\x05\x00\x04\x03\x01'), #01
		struct.unpack('<6B', b'\x04\x03\x01\x02\x05\x00'), #02
		struct.unpack('<6B', b'\x03\x05\x04\x00\x02\x01'), #03
		struct.unpack('<6B', b'\x02\x03\x05\x01\x00\x04'), #04
		struct.unpack('<6B', b'\x04\x00\x03\x05\x02\x01'), #05
		struct.unpack('<6B', b'\x03\x02\x05\x01\x00\x04'), #06
		struct.unpack('<6B', b'\x04\x05\x03\x02\x01\x00'), #07
		struct.unpack('<6B', b'\x03\x04\x01\x05\x00\x02'), #08
		struct.unpack('<6B', b'\x01\x02\x04\x00\x05\x03'), #09
		struct.unpack('<6B', b'\x05\x04\x03\x01\x00\x02'), #0A
		struct.unpack('<6B', b'\x04\x02\x01\x05\x03\x00'), #0B
		struct.unpack('<6B', b'\x05\x03\x04\x01\x00\x02'), #0C
		struct.unpack('<6B', b'\x04\x00\x01\x05\x02\x03'), #0D
		struct.unpack('<6B', b'\x02\x05\x03\x04\x00\x01'), #0E
		struct.unpack('<6B', b'\x05\x02\x01\x00\x03\x04'), #0F
		struct.unpack('<6B', b'\x04\x05\x03\x01\x00\x02'), #10
		struct.unpack('<6B', b'\x01\x00\x04\x02\x05\x03'), #11
		struct.unpack('<6B', b'\x02\x05\x01\x04\x03\x00'), #12
		struct.unpack('<6B', b'\x04\x02\x00\x01\x05\x03'), #13
		struct.unpack('<6B', b'\x01\x03\x05\x02\x00\x04'), #14
		struct.unpack('<6B', b'\x04\x00\x01\x05\x03\x02'), #15
		struct.unpack('<6B', b'\x01\x05\x04\x02\x00\x03'), #16
		struct.unpack('<6B', b'\x03\x04\x05\x00\x01\x02'), #17
		struct.unpack('<6B', b'\x01\x02\x00\x05\x03\x04'), #18
		struct.unpack('<6B', b'\x04\x05\x03\x02\x00\x01'), #19
		struct.unpack('<6B', b'\x05\x03\x00\x04\x01\x02'), #1A
		struct.unpack('<6B', b'\x02\x05\x04\x01\x00\x03'), #1B
		struct.unpack('<6B', b'\x05\x00\x03\x04\x02\x01'), #1C
		struct.unpack('<6B', b'\x04\x03\x00\x01\x05\x02'), #1D
		struct.unpack('<6B', b'\x01\x02\x04\x05\x03\x00'), #1E
		struct.unpack('<6B', b'\x04\x00\x03\x02\x05\x01'), #1F
		struct.unpack('<6B', b'\x02\x03\x00\x05\x01\x04'), #20
		struct.unpack('<6B', b'\x03\x04\x01\x00\x05\x02'), #21
		struct.unpack('<6B', b'\x05\x00\x03\x01\x02\x04'), #22
		struct.unpack('<6B', b'\x04\x03\x00\x02\x05\x01'), #23
		struct.unpack('<6B', b'\x05\x02\x03\x01\x00\x04'), #24
		struct.unpack('<6B', b'\x02\x04\x05\x00\x01\x03'), #25
		struct.unpack('<6B', b'\x04\x03\x00\x05\x02\x01'), #26
		struct.unpack('<6B', b'\x05\x02\x01\x04\x00\x03'), #27
		struct.unpack('<6B', b'\x03\x05\x04\x01\x02\x00'), #28
		struct.unpack('<6B', b'\x02\x03\x05\x00\x01\x04'), #29
		struct.unpack('<6B', b'\x04\x05\x00\x01\x02\x03'), #2A
		struct.unpack('<6B', b'\x05\x03\x01\x02\x00\x04'), #2B
		struct.unpack('<6B', b'\x02\x05\x00\x04\x01\x03'), #2C
		struct.unpack('<6B', b'\x05\x02\x04\x01\x03\x00'), #2D
		struct.unpack('<6B', b'\x04\x05\x03\x00\x01\x02'), #2E
		struct.unpack('<6B', b'\x05\x00\x01\x02\x03\x04'), #2F
		struct.unpack('<6B', b'\x03\x05\x04\x01\x00\x02'), #30
		struct.unpack('<6B', b'\x01\x00\x05\x02\x03\x04'), #31
		struct.unpack('<6B', b'\x05\x04\x03\x00\x01\x02'), #32
		struct.unpack('<6B', b'\x03\x00\x04\x05\x02\x01'), #33
		struct.unpack('<6B', b'\x05\x02\x00\x04\x01\x03'), #34
		struct.unpack('<6B', b'\x01\x04\x05\x00\x03\x02'), #35
		struct.unpack('<6B', b'\x03\x00\x01\x05\x02\x04'), #36
		struct.unpack('<6B', b'\x01\x05\x04\x00\x03\x02'), #37
		struct.unpack('<6B', b'\x03\x02\x01\x05\x00\x04'), #38
		struct.unpack('<6B', b'\x02\x00\x05\x04\x01\x03'), #39
		struct.unpack('<6B', b'\x04\x02\x03\x01\x05\x00'), #3A
		struct.unpack('<6B', b'\x03\x05\x01\x04\x00\x02'), #3B
		struct.unpack('<6B', b'\x05\x00\x04\x02\x01\x03'), #3C
		struct.unpack('<6B', b'\x04\x03\x05\x00\x02\x01'), #3D
		struct.unpack('<6B', b'\x05\x02\x01\x04\x03\x00'), #3E
		struct.unpack('<6B', b'\x01\x05\x03\x02\x00\x04'), #3F
		struct.unpack('<6B', b'\x05\x00\x01\x04\x03\x02'), #40
		struct.unpack('<6B', b'\x02\x05\x03\x01\x00\x04'), #41
		struct.unpack('<6B', b'\x03\x04\x05\x02\x01\x00'), #42
		struct.unpack('<6B', b'\x01\x05\x04\x00\x02\x03'), #43
		struct.unpack('<6B', b'\x02\x04\x01\x05\x03\x00'), #44
		struct.unpack('<6B', b'\x04\x00\x05\x01\x02\x03'), #45
		struct.unpack('<6B', b'\x03\x05\x00\x02\x01\x04'), #46
		struct.unpack('<6B', b'\x05\x00\x04\x01\x02\x03'), #47
		struct.unpack('<6B', b'\x03\x02\x00\x05\x01\x04'), #48
		struct.unpack('<6B', b'\x02\x03\x01\x04\x05\x00'), #49
		struct.unpack('<6B', b'\x01\x04\x03\x05\x00\x02'), #4A
		struct.unpack('<6B', b'\x03\x05\x00\x01\x02\x04'), #4B
		struct.unpack('<6B', b'\x02\x00\x03\x04\x05\x01'), #4C
		struct.unpack('<6B', b'\x05\x04\x01\x02\x00\x03'), #4D
		struct.unpack('<6B', b'\x03\x00\x05\x01\x02\x04'), #4E
		struct.unpack('<6B', b'\x01\x03\x04\x02\x05\x00'), #4F
		struct.unpack('<6B', b'\x03\x04\x05\x00\x02\x01'), #50
		struct.unpack('<6B', b'\x04\x02\x00\x05\x01\x03'), #51
		struct.unpack('<6B', b'\x01\x05\x03\x04\x00\x02'), #52
		struct.unpack('<6B', b'\x03\x02\x00\x01\x05\x04'), #53
		struct.unpack('<6B', b'\x04\x03\x05\x00\x01\x02'), #54
		struct.unpack('<6B', b'\x05\x02\x03\x04\x00\x01'), #55
		struct.unpack('<6B', b'\x03\x00\x05\x02\x01\x04'), #56
		struct.unpack('<6B', b'\x01\x03\x04\x00\x05\x02'), #57
		struct.unpack('<6B', b'\x02\x04\x03\x05\x01\x00'), #58
		struct.unpack('<6B', b'\x05\x03\x00\x04\x02\x01'), #59
		struct.unpack('<6B', b'\x04\x02\x03\x05\x01\x00'), #5A
		struct.unpack('<6B', b'\x05\x04\x00\x01\x03\x02'), #5B
		struct.unpack('<6B', b'\x03\x05\x01\x00\x02\x04'), #5C
		struct.unpack('<6B', b'\x04\x03\x05\x02\x00\x01'), #5D
		struct.unpack('<6B', b'\x03\x00\x01\x04\x05\x02'), #5E
		struct.unpack('<6B', b'\x05\x03\x04\x02\x01\x00'), #5F
		struct.unpack('<6B', b'\x04\x05\x01\x00\x02\x03'), #60
		struct.unpack('<6B', b'\x01\x03\x00\x02\x05\x04'), #61
		struct.unpack('<6B', b'\x02\x05\x04\x01\x03\x00'), #62
		struct.unpack('<6B', b'\x05\x04\x03\x02\x00\x01'), #63
		struct.unpack('<6B', b'\x01\x00\x04\x05\x03\x02'), #64
		struct.unpack('<6B', b'\x02\x04\x05\x01\x00\x03'), #65
		struct.unpack('<6B', b'\x03\x02\x01\x04\x05\x00'), #66
		struct.unpack('<6B', b'\x02\x05\x00\x01\x03\x04'), #67
		struct.unpack('<6B', b'\x03\x02\x05\x04\x00\x01'), #68
		struct.unpack('<6B', b'\x01\x05\x03\x00\x02\x04'), #69
		struct.unpack('<6B', b'\x02\x00\x04\x01\x05\x03'), #6A
		struct.unpack('<6B', b'\x01\x02\x03\x05\x00\x04'), #6B
		struct.unpack('<6B', b'\x04\x05\x00\x02\x01\x03'), #6C
		struct.unpack('<6B', b'\x02\x04\x05\x01\x03\x00'), #6D
		struct.unpack('<6B', b'\x04\x03\x01\x05\x00\x02'), #6E
		struct.unpack('<6B', b'\x05\x02\x00\x04\x03\x01'), #6F
		struct.unpack('<6B', b'\x02\x03\x04\x05\x01\x00'), #70
		struct.unpack('<6B', b'\x04\x00\x03\x01\x05\x02'), #71
		struct.unpack('<6B', b'\x01\x03\x05\x00\x02\x04'), #72
		struct.unpack('<6B', b'\x02\x00\x04\x05\x03\x01'), #73
		struct.unpack('<6B', b'\x04\x03\x05\x02\x01\x00'), #74
		struct.unpack('<6B', b'\x01\x04\x03\x00\x05\x02'), #75
		struct.unpack('<6B', b'\x05\x00\x01\x04\x02\x03'), #76
		struct.unpack('<6B', b'\x02\x03\x04\x05\x00\x01'), #77
		struct.unpack('<6B', b'\x05\x04\x03\x02\x01\x00'), #78
		struct.unpack('<6B', b'\x04\x03\x05\x01\x00\x02'), #79
		struct.unpack('<6B', b'\x05\x04\x00\x02\x03\x01'), #7A
		struct.unpack('<6B', b'\x01\x00\x04\x05\x02\x03'), #7B
		struct.unpack('<6B', b'\x02\x05\x03\x00\x01\x04'), #7C
		struct.unpack('<6B', b'\x01\x04\x00\x02\x05\x03'), #7D
		struct.unpack('<6B', b'\x05\x02\x03\x04\x01\x00'), #7E
		struct.unpack('<6B', b'\x04\x05\x00\x01\x03\x02'), #7F
		struct.unpack('<6B', b'\x01\x02\x03\x04\x05\x00'), #80
		struct.unpack('<6B', b'\x05\x00\x04\x02\x03\x01'), #81
		struct.unpack('<6B', b'\x01\x02\x03\x00\x05\x04'), #82
		struct.unpack('<6B', b'\x04\x00\x05\x01\x03\x02'), #83
		struct.unpack('<6B', b'\x02\x04\x03\x05\x00\x01'), #84
		struct.unpack('<6B', b'\x05\x02\x00\x01\x03\x04'), #85
		struct.unpack('<6B', b'\x02\x03\x05\x04\x01\x00'), #86
		struct.unpack('<6B', b'\x04\x05\x00\x02\x03\x01'), #87
		struct.unpack('<6B', b'\x02\x04\x03\x01\x05\x00'), #88
		struct.unpack('<6B', b'\x03\x02\x05\x00\x01\x04'), #89
		struct.unpack('<6B', b'\x01\x05\x00\x04\x03\x02'), #8A
		struct.unpack('<6B', b'\x04\x03\x01\x05\x02\x00'), #8B
		struct.unpack('<6B', b'\x03\x05\x04\x02\x00\x01'), #8C
		struct.unpack('<6B', b'\x02\x04\x00\x01\x05\x03'), #8D
		struct.unpack('<6B', b'\x03\x05\x04\x00\x01\x02'), #8E
		struct.unpack('<6B', b'\x01\x04\x05\x02\x00\x03'), #8F
		struct.unpack('<6B', b'\x04\x05\x01\x00\x03\x02'), #90
		struct.unpack('<6B', b'\x02\x00\x04\x05\x01\x03'), #91
		struct.unpack('<6B', b'\x04\x03\x05\x01\x02\x00'), #92
		struct.unpack('<6B', b'\x02\x00\x01\x04\x05\x03'), #93
		struct.unpack('<6B', b'\x03\x04\x05\x02\x00\x01'), #94
		struct.unpack('<6B', b'\x02\x05\x03\x04\x01\x00'), #95
		struct.unpack('<6B', b'\x05\x03\x04\x02\x00\x01'), #96
		struct.unpack('<6B', b'\x02\x05\x01\x00\x03\x04'), #97
		struct.unpack('<6B', b'\x04\x03\x00\x05\x01\x02'), #98
		struct.unpack('<6B', b'\x01\x00\x05\x04\x02\x03'), #99
		struct.unpack('<6B', b'\x03\x04\x01\x02\x05\x00'), #9A
		struct.unpack('<6B', b'\x05\x00\x03\x04\x01\x02'), #9B
		struct.unpack('<6B', b'\x03\x05\x01\x02\x00\x04'), #9C
		struct.unpack('<6B', b'\x04\x02\x00\x05\x03\x01'), #9D
		struct.unpack('<6B', b'\x03\x00\x05\x04\x01\x02'), #9E
		struct.unpack('<6B', b'\x04\x05\x01\x02\x03\x00'), #9F
		struct.unpack('<6B', b'\x02\x00\x03\x05\x01\x04'), #A0
		struct.unpack('<6B', b'\x01\x03\x00\x04\x05\x02'), #A1
		struct.unpack('<6B', b'\x02\x00\x05\x01\x03\x04'), #A2
		struct.unpack('<6B', b'\x04\x02\x01\x00\x05\x03'), #A3
		struct.unpack('<6B', b'\x05\x03\x00\x02\x01\x04'), #A4
		struct.unpack('<6B', b'\x02\x04\x05\x00\x03\x01'), #A5
		struct.unpack('<6B', b'\x01\x00\x03\x04\x05\x02'), #A6
		struct.unpack('<6B', b'\x05\x03\x01\x00\x02\x04'), #A7
		struct.unpack('<6B', b'\x01\x05\x04\x02\x03\x00'), #A8
		struct.unpack('<6B', b'\x02\x03\x00\x01\x05\x04'), #A9
		struct.unpack('<6B', b'\x04\x02\x01\x05\x00\x03'), #AA
		struct.unpack('<6B', b'\x03\x00\x05\x04\x02\x01'), #AB
		struct.unpack('<6B', b'\x05\x04\x01\x00\x03\x02'), #AC
		struct.unpack('<6B', b'\x02\x03\x05\x04\x00\x01'), #AD
		struct.unpack('<6B', b'\x03\x02\x04\x05\x01\x00'), #AE
		struct.unpack('<6B', b'\x02\x00\x05\x04\x03\x01'), #AF
		struct.unpack('<6B', b'\x05\x02\x03\x00\x01\x04'), #B0
		struct.unpack('<6B', b'\x03\x04\x05\x01\x00\x02'), #B1
		struct.unpack('<6B', b'\x02\x00\x01\x05\x03\x04'), #B2
		struct.unpack('<6B', b'\x03\x05\x04\x02\x01\x00'), #B3
		struct.unpack('<6B', b'\x01\x00\x03\x05\x02\x04'), #B4
		struct.unpack('<6B', b'\x02\x05\x04\x00\x01\x03'), #B5
		struct.unpack('<6B', b'\x05\x04\x01\x02\x03\x00'), #B6
		struct.unpack('<6B', b'\x04\x02\x05\x00\x01\x03'), #B7
		struct.unpack('<6B', b'\x03\x05\x01\x04\x02\x00'), #B8
		struct.unpack('<6B', b'\x05\x00\x04\x01\x03\x02'), #B9
		struct.unpack('<6B', b'\x04\x02\x03\x05\x00\x01'), #BA
		struct.unpack('<6B', b'\x02\x03\x01\x00\x05\x04'), #BB
		struct.unpack('<6B', b'\x01\x05\x03\x04\x02\x00'), #BC
		struct.unpack('<6B', b'\x02\x04\x01\x05\x00\x03'), #BD
		struct.unpack('<6B', b'\x03\x02\x04\x01\x05\x00'), #BE
		struct.unpack('<6B', b'\x05\x04\x01\x00\x02\x03'), #BF
		struct.unpack('<6B', b'\x02\x00\x03\x01\x05\x04'), #C0
		struct.unpack('<6B', b'\x05\x04\x00\x02\x01\x03'), #C1
		struct.unpack('<6B', b'\x04\x02\x05\x00\x03\x01'), #C2
		struct.unpack('<6B', b'\x05\x03\x00\x01\x02\x04'), #C3
		struct.unpack('<6B', b'\x02\x05\x01\x04\x00\x03'), #C4
		struct.unpack('<6B', b'\x04\x00\x05\x02\x03\x01'), #C5
		struct.unpack('<6B', b'\x03\x04\x01\x05\x02\x00'), #C6
		struct.unpack('<6B', b'\x04\x00\x05\x02\x01\x03'), #C7
		struct.unpack('<6B', b'\x03\x02\x04\x00\x05\x01'), #C8
		struct.unpack('<6B', b'\x01\x04\x05\x02\x03\x00'), #C9
		struct.unpack('<6B', b'\x03\x00\x04\x05\x01\x02'), #CA
		struct.unpack('<6B', b'\x05\x03\x01\x04\x02\x00'), #CB
		struct.unpack('<6B', b'\x03\x04\x00\x01\x05\x02'), #CC
		struct.unpack('<6B', b'\x01\x03\x04\x05\x02\x00'), #CD
		struct.unpack('<6B', b'\x05\x00\x03\x02\x01\x04'), #CE
		struct.unpack('<6B', b'\x01\x02\x04\x05\x00\x03'), #CF
		struct.unpack('<6B', b'\x02\x03\x00\x04\x05\x01'), #D0
		struct.unpack('<6B', b'\x05\x02\x04\x00\x01\x03'), #D1
		struct.unpack('<6B', b'\x02\x03\x01\x05\x00\x04'), #D2
		struct.unpack('<6B', b'\x01\x05\x00\x04\x02\x03'), #D3
		struct.unpack('<6B', b'\x02\x04\x03\x00\x05\x01'), #D4
		struct.unpack('<6B', b'\x01\x03\x00\x05\x02\x04'), #D5
		struct.unpack('<6B', b'\x03\x00\x04\x02\x05\x01'), #D6
		struct.unpack('<6B', b'\x01\x03\x05\x04\x00\x02'), #D7
		struct.unpack('<6B', b'\x04\x05\x03\x01\x02\x00'), #D8
		struct.unpack('<6B', b'\x02\x03\x04\x00\x05\x01'), #D9
		struct.unpack('<6B', b'\x03\x02\x05\x04\x01\x00'), #DA
		struct.unpack('<6B', b'\x04\x03\x01\x00\x05\x02'), #DB
		struct.unpack('<6B', b'\x01\x02\x05\x04\x03\x00'), #DC
		struct.unpack('<6B', b'\x03\x04\x00\x02\x05\x01'), #DD
		struct.unpack('<6B', b'\x01\x03\x05\x04\x02\x00'), #DE
		struct.unpack('<6B', b'\x02\x04\x01\x00\x05\x03'), #DF
		struct.unpack('<6B', b'\x03\x05\x00\x04\x01\x02'), #E0
		struct.unpack('<6B', b'\x05\x03\x04\x00\x02\x01'), #E1
		struct.unpack('<6B', b'\x03\x04\x00\x05\x01\x02'), #E2
		struct.unpack('<6B', b'\x04\x05\x01\x02\x00\x03'), #E3
		struct.unpack('<6B', b'\x03\x04\x00\x05\x02\x01'), #E4
		struct.unpack('<6B', b'\x01\x00\x05\x04\x03\x02'), #E5
		struct.unpack('<6B', b'\x03\x02\x04\x05\x00\x01'), #E6
		struct.unpack('<6B', b'\x01\x04\x05\x00\x02\x03'), #E7
		struct.unpack('<6B', b'\x04\x00\x03\x05\x01\x02'), #E8
		struct.unpack('<6B', b'\x03\x02\x00\x04\x05\x01'), #E9
		struct.unpack('<6B', b'\x05\x03\x04\x00\x01\x02'), #EA
		struct.unpack('<6B', b'\x04\x02\x05\x01\x00\x03'), #EB
		struct.unpack('<6B', b'\x01\x05\x00\x02\x03\x04'), #EC
		struct.unpack('<6B', b'\x04\x02\x03\x00\x05\x01'), #ED
		struct.unpack('<6B', b'\x01\x04\x00\x05\x02\x03'), #EE
		struct.unpack('<6B', b'\x05\x03\x01\x04\x00\x02'), #EF
		struct.unpack('<6B', b'\x01\x04\x03\x05\x02\x00'), #F0
		struct.unpack('<6B', b'\x05\x02\x04\x01\x00\x03'), #F1
		struct.unpack('<6B', b'\x03\x00\x01\x02\x05\x04'), #F2
		struct.unpack('<6B', b'\x05\x02\x04\x00\x03\x01'), #F3
		struct.unpack('<6B', b'\x02\x04\x00\x05\x01\x03'), #F4
		struct.unpack('<6B', b'\x01\x02\x05\x00\x03\x04'), #F5
		struct.unpack('<6B', b'\x05\x04\x03\x01\x02\x00'), #F6
		struct.unpack('<6B', b'\x01\x02\x05\x04\x00\x03'), #F7
		struct.unpack('<6B', b'\x02\x04\x00\x05\x03\x01'), #F8
		struct.unpack('<6B', b'\x04\x00\x01\x02\x05\x03'), #F9
		struct.unpack('<6B', b'\x03\x04\x05\x01\x02\x00'), #FA
		struct.unpack('<6B', b'\x01\x00\x03\x02\x05\x04'), #FB
		struct.unpack('<6B', b'\x03\x05\x00\x04\x02\x01'), #FC
		struct.unpack('<6B', b'\x02\x03\x04\x01\x05\x00'), #FD
		struct.unpack('<6B', b'\x04\x05\x03\x00\x02\x01'), #FE
		struct.unpack('<6B', b'\x01\x04\x00\x05\x03\x02'), #FF
	)

	__vector_fin = \
		struct.unpack('<48B', b'\x13\x19\x15\x1B\x00\x11\x08\x1A\x2C\x1E\x12\x0D\x1C\x0E\x17\x26\x14\x2A\x1D\x03\x20\x2D\x01\x1F\x05\x24\x06\x10\x2E\x27\x09\x07\x16\x28\x0B\x04\x22\x0F\x29\x18\x21\x0A\x2F\x0C\x2B\x25\x02\x23')

	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# INTERNAL METHODS
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	@classmethod
	def __iterate(cls, hash, row_xor, row_mix, row_add):
		temp = [0] * cls.__hash_words
		kmul, mask = 0x9DDFEA08EB382D69, 0xFFFFFFFFFFFFFFFF
		for j in range(len(temp)):
			u, v = hash[j] + row_add[j], hash[row_mix[j]]
			v = ((v ^ u) * kmul) & mask
			v ^= (v >> 47)
			v = ((v ^ u) * kmul) & mask
			v ^= (v >> 47)
			temp[j] = ((v * kmul) & mask) ^ row_xor[j]
		hash[:] = temp

	@staticmethod
	def __get_byte(hash, idx):
		return (hash[idx // 8] >> ((idx % 8) * 8)) & 0xFF

	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# INITIALIZATION
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	def __init__(self):
		self.reset()

	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# PUBLIC METHODS
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	def update(self, input):
		if not isinstance(input, (bytes, bytearray, str)):
			raise TypeError('input data must be a sequence of bytes or a string!')
		if isinstance(input, str):
			input = input.encode(encoding='utf-8')
		if self.__finished:
			raise RuntimeError('hash computation is finished -> must call reset() first!')
		cls, hash, ctr = __class__, self.__hash, self.__ctr
		iterate, table_xor, table_mix, table_add = cls.__iterate, cls.__table_xor, cls.__table_mix, cls.__table_add
		for val in input:
			iterate(hash, table_xor[val], table_mix[ctr], table_add[val])
			ctr = (ctr + 1) & 0xFF
		self.__ctr = ctr

	def digest(self):
		if self.__finished:
			raise RuntimeError('hash computation is finished -> must call reset() first!')
		cls, hash, ctr = __class__, self.__hash, self.__ctr
		iterate, get_byte, table_xor, table_mix, table_add, vector_fin = \
			cls.__iterate, cls.__get_byte, cls.__table_xor, cls.__table_mix, cls.__table_add, cls.__vector_fin
		result, val = [0] * cls.__hash_size, 256
		for i in range(cls.__hash_size):
			iterate(hash, table_xor[val], table_mix[ctr], table_add[val])
			ctr = (ctr + 1) & 0xFF
			val = result[i] = get_byte(hash, vector_fin[i])
		self.__finished = True
		self.__ctr = ctr
		return struct.pack('<48B', *result)

	def reset(self):
		self.__hash = list(__class__.__vector_ini)
		self.__ctr = 0
		self.__finished = False

	@classmethod
	def compute(cls, input):
		instance = cls()
		instance.update(input)
		return instance.digest()

	@classmethod
	def version(cls):
		return cls.__version_major, cls.__version_minor, cls.__version_patch


# ===========================================================================
# MAIN
# ===========================================================================

if __name__ == '__main__':
	import mhash384_selftest
	mhash384_selftest.main()
