//******************************************************************************
//
// Simple MIDI Library / SMAppleDLSDevCtrl
//
// Apple DLS (Downloadable Sounds) デバイス制御クラス
//
// Copyright (C) 2010 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "SMAppleDLSDevCtrl.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
SMAppleDLSDevCtrl::SMAppleDLSDevCtrl()
{
	m_AUGraph = NULL;
	m_UnitOut = NULL;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
SMAppleDLSDevCtrl::~SMAppleDLSDevCtrl()
{
	if (m_AUGraph != NULL) {
		DisposeAUGraph(m_AUGraph);
	}
	m_AUGraph = NULL;
	m_UnitOut = NULL;
}

//******************************************************************************
// 初期化
//******************************************************************************
int SMAppleDLSDevCtrl::Initialize()
{
	int result = 0;
	
	if (m_AUGraph != NULL) {
		DisposeAUGraph(m_AUGraph);
	}
	m_AUGraph = NULL;
	m_UnitOut = NULL;
	
	//オーディオ処理グラフ生成
	result = _CreateAUGraph(&m_AUGraph, &m_UnitOut);
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// デバイスオープン
//******************************************************************************
int SMAppleDLSDevCtrl::Open()
{
	int result = 0;
	OSStatus status = 0;
	
	//グラフ開始
	status = AUGraphStart(m_AUGraph);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	
EXIT:;
	return result;
}
//******************************************************************************
// デバイスクローズ
//******************************************************************************
int SMAppleDLSDevCtrl::Close()
{
	int result = 0;
	OSStatus status = 0;
	
	//演奏停止
	status = AUGraphStop(m_AUGraph);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// MIDIメッセージ送信
//******************************************************************************
int SMAppleDLSDevCtrl::SendShortMsg(
		unsigned char* pMsg,
		unsigned long size
	)
{
	int result = 0;
	OSStatus status = 0;
	unsigned char data[3] = {0, 0, 0};
	
	if (size > 3) {
		result = YN_SET_ERR(@"Program error.", size, 0);
		goto EXIT;
	}
	
	if (size >= 1) {
		data[0] = pMsg[0];
	}
	if (size >= 2) {
		data[1] = pMsg[1];
	}
	if (size >= 3) {
		data[2] = pMsg[2];
	}
	
	//MIDIイベント送信
	status = MusicDeviceMIDIEvent(
					m_UnitOut,	//出力ユニット
					data[0],	//ステータス
					data[1],	//データ1
					data[2],	//データ2
					0			//サンプルオフセット
				);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// システムエクスクルーシブ送信
//******************************************************************************
int SMAppleDLSDevCtrl::SendLongMsg(
		unsigned char* pMsg,
		unsigned long size
	)
{
	int result = 0;
	OSStatus status = 0;
	
	//システムエクスクルーシブ送信
	status = MusicDeviceSysEx(
					m_UnitOut,	//出力ユニット
					pMsg,		//データ位置
					size		//データサイズ
				);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, size);
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// 全ノートオフ
//******************************************************************************
int SMAppleDLSDevCtrl::NoteOffAll()
{
	int result = 0;
	unsigned char i = 0;
	unsigned char msg[3];
	
	//全トラックノートオフ
	for (i = 0; i < 16; i++) {
		msg[0] = 0xB0 | i;
		msg[1] = 0x7B;
		msg[2] = 0x00;
		result = SendShortMsg(msg, 3);
		if (result != 0) goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// オーディオ処理グラフ生成
//******************************************************************************
int SMAppleDLSDevCtrl::_CreateAUGraph(
		AUGraph* pOutGraph,
		AudioUnit* pOutUnit
	)
{
	int result = 0;
	OSStatus status = 0;
	AUGraph auGraph;
	//ベースSDK10.5ではAudioComponentDescription未定義でエラーになる
	//AudioComponentDescription cd;
	//AudioComponentDescription cdout;
	ComponentDescription cd;
	ComponentDescription cdout;
	AUNode nodeSynth;
	AUNode nodeLimiter;
	AUNode nodeOut;
	AudioUnit unitOut;
	
	//------------------------------------------------
	// グラフ生成
	//------------------------------------------------
	//オーディオ処理グラフ生成
	status = NewAUGraph(&auGraph);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	
	//------------------------------------------------
	// ノード生成
	//------------------------------------------------
	//ノード生成：ミュージックデバイス／DLSシンセ
	cd.componentType         = kAudioUnitType_MusicDevice;
	cd.componentSubType      = kAudioUnitSubType_DLSSynth;
	cd.componentManufacturer = kAudioUnitManufacturer_Apple;
	cd.componentFlags = 0;
	cd.componentFlagsMask = 0;
	status = AUGraphAddNode(auGraph, &cd, &nodeSynth);
	if (status != noErr) {
		goto EXIT;
	}
	//ノード生成：エフェクト／ピークリミッタ
	cd.componentType         = kAudioUnitType_Effect;
	cd.componentSubType      = kAudioUnitSubType_PeakLimiter;
	cd.componentManufacturer = kAudioUnitManufacturer_Apple;
	cd.componentFlags = 0;
	cd.componentFlagsMask = 0;
	status = AUGraphAddNode(auGraph, &cd, &nodeLimiter);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	//ノード生成：出力／標準出力
	cd.componentType         = kAudioUnitType_Output;
	cd.componentSubType      = kAudioUnitSubType_DefaultOutput;
	cd.componentManufacturer = kAudioUnitManufacturer_Apple;
	cd.componentFlags = 0;
	cd.componentFlagsMask = 0;
	status = AUGraphAddNode(auGraph, &cd, &nodeOut);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	
	//------------------------------------------------
	// ノード接続
	//------------------------------------------------
	//ノード接続：DLSシンセ -> ピークリミッタ
	status = AUGraphConnectNodeInput(
						auGraph,	//入力元グラフ
						nodeSynth,	//入力元ノード
						0,			//入力元出力番号
						nodeLimiter,//出力先ノード
						0			//出力先入力番号
					);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	//ノード接続：ピークリミッタ -> 標準出力 
	status = AUGraphConnectNodeInput(
						auGraph,	//入力元グラフ
						nodeLimiter,//入力元ノード
						0,			//入力元出力番号
						nodeOut,	//出力先ノード
						0			//出力先入力番号
					);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	
	//------------------------------------------------
	// グラフ開始準備
	//------------------------------------------------
	//グラフを開く
	status = AUGraphOpen(auGraph);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	//出力ユニットを取得
	status = AUGraphNodeInfo(
					auGraph,	//グラフ
					nodeSynth,	//入力ノード
					&cdout,		//コンポーネント定義
					&unitOut	//出力ユニット
				);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	//グラフ初期化
	status = AUGraphInitialize(auGraph);
	if (status != noErr) {
		result = YN_SET_ERR(@"AUGraph API error.", status, 0);
		goto EXIT;
	}
	
	*pOutGraph = auGraph;
	*pOutUnit = unitOut;

EXIT:;
	return result;
}


