//******************************************************************************
//
// MIDITrail / MTSecurityScopedBookmark
//
// Security-Scoped Bookmark クラス
//
// Copyright (C) 2017 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTSecurityScopedBookmark.h"


//******************************************************************************
// 読み取り専用ブックマーク登録
//******************************************************************************
int MTSecurityScopedBookmark::SetBookmarkOfReadOnly(
		NSURL* pURL,
		NSString* pKey
	)
{
	int result = 0;
	NSData* pBookmark = nil;
	NSError* pError = nil;
	NSUserDefaults *pDefaults = nil;
	
	//Security-Scoped bookmark 作成
	pBookmark = [pURL bookmarkDataWithOptions:NSURLBookmarkCreationWithSecurityScope|NSURLBookmarkCreationSecurityScopeAllowOnlyReadAccess
			   includingResourceValuesForKeys:nil
								relativeToURL:nil
										error:&pError];
	if (pError != nil) {
		result = YN_SET_ERR(@"Security-Scoped bookmark error.", 0, 0);
		goto EXIT;
	}
	
	//User DefaultsにSecurity-Scoped Bookmarkを保存
	pDefaults = [NSUserDefaults standardUserDefaults];
	[pDefaults setObject:pBookmark forKey:pKey];
	[pDefaults synchronize];
	
EXIT:;
	return result;
}

//******************************************************************************
// ブックマーク取得
//******************************************************************************
int MTSecurityScopedBookmark::GetBookmark(
		NSURL** pPtrURL,
		NSString* pKey,
		BOOL* pIsStale
	)
{
	int result = 0;
	NSUserDefaults *pDefaults = nil;
	NSData* pBookmark = nil;
	NSError* pError = nil;
	NSURL *pURL;
	
	*pPtrURL = nil;
	
	//User Defaultsに保存されているSecurity-Scoped Bookmarkを取得
	pDefaults = [NSUserDefaults standardUserDefaults];
	[pDefaults synchronize];
	pBookmark = [pDefaults objectForKey:pKey];
	if (pBookmark == nil) {
		//Bookmark未登録の場合（エラーとしない）
		goto EXIT;
	}
	if ([pBookmark length] == 0) {
		//データなしの場合（エラーとしない）
		goto EXIT;
	}
	
	//Security-Scoped BookmarkをNSURLに戻す
	pURL = [NSURL URLByResolvingBookmarkData:pBookmark
									 options:NSURLBookmarkResolutionWithSecurityScope
							   relativeToURL:nil
						 bookmarkDataIsStale:pIsStale
									   error:&pError];
	if (pError != nil) {
		result = YN_SET_ERR(@"Security-Scoped bookmark error.", 0, 0);
		goto EXIT;
	}
	
	//ブックマークが古いと判断されている場合
	// ブックマーク対象のファイルが移動された
	// ブックマーク対象のファイルが削除されて同名のファイルに差し替えられた
	if (*pIsStale) {
		NSLog(@"The bookmark is stale.");
	}
	
	//[pURL retain];
	*pPtrURL = pURL;
	
EXIT:;
	return result;
}

//******************************************************************************
// リソースアクセス開始
//******************************************************************************
int MTSecurityScopedBookmark::StartAccessingResource(NSURL* pURL)
{
	int result = 0;
	BOOL apiResult = NO;
	
	//Security-Scoped Resource アクセス開始
	apiResult = [pURL startAccessingSecurityScopedResource];
	if (apiResult == NO) {
		result = YN_SET_ERR(@"Resource access error.", 0, 0);
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// リソースアクセス終了
//******************************************************************************
void MTSecurityScopedBookmark::StopAccessingResource(NSURL* pURL)
{
	//Security-Scoped Resource アクセス終了
	[pURL stopAccessingSecurityScopedResource];
	
	return;
}


