//******************************************************************************
//
// MIDITrail / MTFileEditViewCtrl
//
// ファイル編集ビュー制御クラス
//
// Copyright (C) 2018 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTFileEditViewCtrl.h"


//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTFileEditViewCtrl ()

//テーブルセル作成：ファイル
- (UITableViewCell*)makeFileEditCell;

//通知送信処理
- (int)postNotificationWithName:(NSString*)pName;

@end


@implementation MTFileEditViewCtrl

//******************************************************************************
// ファイル名設定
//******************************************************************************
- (void)setFileName:(NSString*)pFileName
{
	[m_pFileName release];
	[m_pExtension release];
	
	//ファイル名と拡張子を分割して保持する
	m_pFileName = [[pFileName stringByDeletingPathExtension] retain];
	m_pExtension = [[pFileName pathExtension] retain];
	
	return;
}

//******************************************************************************
// 生成
//******************************************************************************
- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
	self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) {
		//ビュー設定
		self.title = @"File Name";
    }
	
    return self;
}

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
	UIBarButtonItem* pSaveButton = nil;
	
    [super viewDidLoad];
	
	// Do any additional setup after loading the view.
	
	//ナビゲーションバー右側にSaveボタンを配置
	pSaveButton = [[UIBarButtonItem alloc]
				   initWithBarButtonSystemItem:UIBarButtonSystemItemSave
				   target:self
				   action:@selector(onSaveButton:)];
	self.navigationItem.rightBarButtonItem = pSaveButton;
	
	return;
}

//******************************************************************************
// インターフェース自動回転確認
//******************************************************************************
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
- (BOOL)shouldAutorotate
{
	//回転を許可する
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
- (NSUInteger)supportedInterfaceOrientations
{
	//全方向に対応する
	return UIInterfaceOrientationMaskAll;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{
	[m_pTableView reloadData];
	return;
}

//******************************************************************************
// ビュー非表示
//******************************************************************************
- (void)viewWillDisappear:(BOOL)animated
{
	return;
}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
	//セクション
	// 0. ファイル名
    return 1;
} 

//******************************************************************************
// セクションヘッダ
//******************************************************************************
- (NSString*)tableView:(UITableView*)tableView titleForHeaderInSection:(NSInteger)section
{
	NSString* pSectionHeader = nil;
	
	switch(section) {
		case 0:
			//ファイル名
			pSectionHeader = @"";
			break;
		default:
			break;
    }
	
    return pSectionHeader;
} 

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	
	//NSLog(@"section %d", section);
	
	switch (section) {
		case 0:
			//ファイル名
			numOfRows = 1;
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//ファイル名
			pCell = [self makeFileEditCell];
			break;
		default:
			break;
	}
	
    return pCell;
}

//******************************************************************************
// テーブルセル作成：ファイル
//******************************************************************************
- (UITableViewCell*)makeFileEditCell
{
    static NSString* pCellIdentifier = @"MTFileEditViewCtrl";
	UITableViewCell* pCell = nil;
	UITextField* pTextField = nil;
	CGRect rectTextField;
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle
									   reuseIdentifier:pCellIdentifier];
		//テキストフィールド追加
		rectTextField = CGRectMake(10, 8, m_pTableView.frame.size.width - (10 * 2), m_pTableView.rowHeight - (8 * 2));
		pTextField = [[UITextField alloc] initWithFrame:rectTextField];
		pTextField.borderStyle = UITextBorderStyleNone;
		pTextField.clearButtonMode = UITextFieldViewModeAlways;
		pTextField.delegate = self;
		pTextField.tag = 1;
		pTextField.borderStyle = UITextBorderStyleNone;
		//pTextField.borderStyle = UITextBorderStyleRoundedRect;
		[pCell.contentView addSubview:pTextField];
	}
	
	//テキストフィールドにファイル名を設定
	pTextField = (UITextField*)[pCell.contentView viewWithTag:1];
	pTextField.placeholder = m_pFileName;
	pTextField.text = m_pFileName;
	
	//ラベルは空文字
	pCell.textLabel.text = @"";
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：列の高さ
//******************************************************************************
- (CGFloat)tableView:(UITableView*)tableView heightForRowAtIndexPath:(NSIndexPath*)indexPath
{
	CGFloat height = 0.0f;
	
	//デフォルトの高さを返す
	height = m_pTableView.rowHeight;
	
	return height;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//ファイル名
			//何もしない
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// Saveボタン選択イベント
//******************************************************************************
- (void)onSaveButton:(id)sender
{
	NSIndexPath *pIndexPath = nil;
	UITableViewCell* pCell = nil;
	UITextField* pTextField = nil;
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	//編集されたファイル名を取得
	pIndexPath = [NSIndexPath indexPathForRow:0 inSection:0];
	pCell = [m_pTableView cellForRowAtIndexPath:pIndexPath];
	pTextField = (UITextField*)[pCell.contentView viewWithTag:1];
	m_pNewFileName = pTextField.text;
	[m_pNewFileName retain];
	
	//フィル名不正チェック
	if (([m_pNewFileName length] == 0) || ([m_pNewFileName containsString:@"/"])) {
		//ファイル名不正アラート表示
		pAlertController = [UIAlertController alertControllerWithTitle:@"Confirmation"
															   message:@"Please enter a valid file name."
														preferredStyle:UIAlertControllerStyleAlert];
		pAlertAction = [UIAlertAction actionWithTitle:@"OK"
												style:UIAlertActionStyleDefault
											  handler:nil];
		[pAlertController addAction:pAlertAction];
		[self presentViewController:pAlertController animated:YES completion:nil];
		goto EXIT;
	}
	//保存確認アラート表示
	pAlertController = [UIAlertController alertControllerWithTitle:@"Confirmation"
														   message:@"Do you want to save changes?"
													preferredStyle:UIAlertControllerStyleAlert];
	pAlertAction = [UIAlertAction actionWithTitle:@"Yes"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSaveYes];}];
	[pAlertController addAction:pAlertAction];
	pAlertAction = [UIAlertAction actionWithTitle:@"Cancel"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSaveCancel];}];
	[pAlertController addAction:pAlertAction];
	[self presentViewController:pAlertController animated:YES completion:nil];
	
EXIT:;
	return;
}

//******************************************************************************
// 保存確認Yesボタン押下
//******************************************************************************
- (void)onAlertSaveYes
{
	//ファイル名更新通知
	[self postNotificationWithName:@"onUpdateFileName"];
	
	//1階層戻る
	[self.navigationController popViewControllerAnimated:YES];
	
	return;
}

//******************************************************************************
// 保存確認Cancelボタン押下
//******************************************************************************
- (void)onAlertSaveCancel
{
	//何もしない
	return;
}

//******************************************************************************
// 通知送信処理
//******************************************************************************
- (int)postNotificationWithName:(NSString*)pName
{
	int result = 0;
	NSNotification* pNotification = nil;
	NSNotificationCenter* pCenter = nil;
	
	//通知オブジェクトを作成
	pNotification = [NSNotification notificationWithName:pName
												  object:self
												userInfo:nil];
	//通知する
	pCenter = [NSNotificationCenter defaultCenter];
	
	//通知に対応する処理を演奏スレッドで処理させる場合
	//[pCenter postNotification:pNotification];
	
	//通知に対応する処理をメインスレッドに処理させる場合
	[pCenter performSelectorOnMainThread:@selector(postNotification:)
							  withObject:pNotification
						   waitUntilDone:NO];
	
	return result;
}

//******************************************************************************
// 新ファイル名取得
//******************************************************************************
- (NSString*)getNewFileName
{
	NSString* pNewFileNameWithExtension = nil;
	
	//編集後のファイル名に拡張子を付与
	pNewFileNameWithExtension = [m_pNewFileName stringByAppendingPathExtension:m_pExtension];
	
	return pNewFileNameWithExtension;
}


@end

