//******************************************************************************
//
// MIDITrail / MTFileList
//
// ファイルリストクラス
//
// Copyright (C) 2021 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTFileList.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTFileList::MTFileList(void)
{
	m_pTargetDirPath = nil;
	m_pCurFilePath = nil;
	m_pFileNameArray = nil;
	m_SelectedFileIndex = 0;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTFileList::~MTFileList(void)
{
	Clear();
}

//******************************************************************************
// クリア
//******************************************************************************
void MTFileList::Clear()
{
	[m_pTargetDirPath release];
	m_pTargetDirPath = nil;
	[m_pCurFilePath release];
	m_pCurFilePath = nil;
	[m_pFileNameArray release];
	m_pFileNameArray = nil;
	m_SelectedFileIndex = 0;
}

//******************************************************************************
// ディレクトリ配下ファイルリスト作成
//******************************************************************************
int MTFileList::MakeFileListWithDirectory(
		NSString* pTargetDirPath
	)
{
	int result = 0;
	int i = 0;
	NSString* pPath = nil;
	NSString* pFile = nil;
	NSArray* pFileArrayTmp = nil;
	NSMutableArray* pFileArray = nil;
	BOOL isDir = NO;
	
	if (pTargetDirPath == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	Clear();
	pFileArray = [[NSMutableArray alloc] init];
	
	//ディレクトリパスを保持する
	m_pTargetDirPath = pTargetDirPath;
	[m_pTargetDirPath retain];
	
	//対象ディレクトリ配下のファイル一覧を取得
	if ([[NSFileManager defaultManager] fileExistsAtPath:m_pTargetDirPath]) {
		pFileArrayTmp = [[NSFileManager defaultManager] contentsOfDirectoryAtPath:m_pTargetDirPath error:nil];
	}
	else {
		//ディレクトリが存在しない場合はファイルなしとして処理を続行
		pFileArrayTmp = [NSArray array];
	}
	
	//ファイル一覧を作成
	for (i = 0; i < [pFileArrayTmp count]; i++) {
		pFile = [pFileArrayTmp objectAtIndex:i];
		pPath = [m_pTargetDirPath stringByAppendingPathComponent:pFile];
		[[NSFileManager defaultManager] fileExistsAtPath:pPath isDirectory:&isDir];
		if (!isDir) {
			//ファイルの場合
			if ([[pPath pathExtension] caseInsensitiveCompare:@"mid"] == NSOrderedSame) {
				[pFileArray addObject:pFile];
			}
		}
	}
	
	//ファイル名ソート：大文字小文字を区別しない
	[m_pFileNameArray release];
	m_pFileNameArray = [pFileArray mutableCopy];
	[m_pFileNameArray sortUsingComparator:^(id str1, id str2) {
		return [((NSString*)str1) compare:((NSString*)str2) options:NSCaseInsensitiveSearch]; }];
	[m_pFileNameArray retain];
	
	//テスト：ファイルが存在しない場合
	//m_pFileArray = [[NSArray alloc] init];
	
EXIT:;
	[pFileArray release];
	return result;
}

//******************************************************************************
// ファイル数取得
//******************************************************************************
NSUInteger MTFileList::GetFileCount()
{
	NSUInteger count = 0;
	
	if (m_pFileNameArray != nil) {
		count = [m_pFileNameArray count];
	}
	
	return count;
}

//******************************************************************************
// ファイルパス取得
//******************************************************************************
NSString* MTFileList::GetFilePath(unsigned int index)
{
	NSString* pFileName = nil;
	
	[m_pCurFilePath release];
	m_pCurFilePath = nil;
	
	pFileName = GetFileName(index);
	if (pFileName != nil) {
		m_pCurFilePath = [m_pTargetDirPath stringByAppendingPathComponent:pFileName];
		[m_pCurFilePath retain];
	}
	
	return m_pCurFilePath;
}

//******************************************************************************
// ファイル名取得
//******************************************************************************
NSString* MTFileList::GetFileName(unsigned int index)
{
	NSString* pFileName = nil;
	
	if (GetFileCount() > index) {
		pFileName = [m_pFileNameArray objectAtIndex:index];
	}
	
	return pFileName;
}

//******************************************************************************
// 選択ファイル登録
//******************************************************************************
int MTFileList::SetSelectedFileName(NSString* pFileName)
{
	int result = 0;
	int i = 0;
	
	if (pFileName == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	m_SelectedFileIndex = 0;
	
	//ファイル名リストから検索（大文字小文字を区別しない）
	for (i = 0; i < [m_pFileNameArray count]; i++) {
		if ([[m_pFileNameArray objectAtIndex:i] caseInsensitiveCompare:pFileName] == NSOrderedSame) {
			m_SelectedFileIndex = i;
			break;
		}
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// 先頭ファイル選択
//******************************************************************************
void MTFileList::SelectFirstFile()
{
	m_SelectedFileIndex = 0;
}

//******************************************************************************
// 前ファイル選択
//******************************************************************************
void MTFileList::SelectPreviousFile(bool* pIsExist)
{
	bool isExist = false;

	//ファイルが存在しない場合
	if (GetFileCount() == 0) {
		//前ファイルなしで終了
	}
	//ファイルリスト先頭を選択中の場合
	else if (m_SelectedFileIndex == 0) {
		//前ファイルなしで終了
	}
	else {
		//前ファイルを選択
		m_SelectedFileIndex -= 1;
		isExist = true;
	}

	if (pIsExist != NULL) {
		*pIsExist = isExist;
	}

	return;
}

//******************************************************************************
// 次ファイル選択
//******************************************************************************
void MTFileList::SelectNextFile(bool* pIsExist)
{
	bool isExist = false;

	//ファイルが存在しない場合
	if (GetFileCount() == 0) {
		//次ファイルなしで終了
	}
	//ファイルリスト末尾を選択中の場合
	else if (m_SelectedFileIndex >= (GetFileCount() - 1)) {
		//次ファイルなしで終了
	}
	else {
		//次ファイルを選択
		m_SelectedFileIndex += 1;
		isExist = true;
	}
	
	if (pIsExist != NULL) {
		*pIsExist = isExist;
	}

	return;
}

//******************************************************************************
// 先頭ファイル選択
//******************************************************************************
unsigned int MTFileList::GetSelectedFileIndex()
{
	return m_SelectedFileIndex;
}


