//******************************************************************************
//
// MIDITrail / MTDeviceScreenInfo
//
// デバイススクリーン情報クラス
//
// Copyright (C) 2018-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "MTDeviceScreenInfo.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTDeviceScreenInfo::MTDeviceScreenInfo(void)
{
	//セーフエリア情報
	m_isEnableSafeArea = false;
	
	//マージン幅
	m_Margin_LandscapeTop = 0.0f;
	m_Margin_LandscapeLeft = 0.0f;
	m_Margin_LandscapeRight = 0.0f;
	m_Margin_LandscapeBottom = 0.0f;
	m_Margin_RoundedCorner = 0.0f;
	
	return;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTDeviceScreenInfo::~MTDeviceScreenInfo(void)
{
}

//******************************************************************************
// 初期化
//******************************************************************************
void MTDeviceScreenInfo::Initialize()
{
	//セーフエリア情報初期化
	_InitializeSafeArea();
	
	//PPI情報初期化
	_InitializePPI();
	
	return;
}

//******************************************************************************
// セーフエリア情報初期化
//******************************************************************************
void MTDeviceScreenInfo::_InitializeSafeArea()
{
	CGFloat height = 0;
	CGFloat width = 0;
	CGFloat longSide = 0;
	
	height = [[UIScreen mainScreen] bounds].size.height;
	width = [[UIScreen mainScreen] bounds].size.width;
	
	longSide = height;
	if (height < width) {
		longSide = width;
	}
	
	//iPhone
	if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
		//iPhone X,XS,XS Max,XR,11,11 Pro,11 Pro Max,12 mini,13 mini はセーフエリアあり
		if ((longSide == 812) || (longSide == 896)) {
			m_isEnableSafeArea = true;
			m_Margin_LandscapeTop = 0.0f;
			m_Margin_LandscapeLeft = 44.0f;
			m_Margin_LandscapeRight = 44.0f;
			m_Margin_LandscapeBottom = 21.0f;
			m_Margin_RoundedCorner = 0.0f;
		}
		//iPhone 12,12 Pro,12 Pro Max,13,13 Pro,13 Pro Max はセーフエリアあり
		//iPhone 12 Pro Maxより大きい未知のデバイスはセーフエリアありとみなす
		else if ((longSide == 844) || (longSide == 926) || (longSide > 926)) {
			m_isEnableSafeArea = true;
			m_Margin_LandscapeTop = 0.0f;
			m_Margin_LandscapeLeft = 47.0f;
			m_Margin_LandscapeRight = 47.0f;
			m_Margin_LandscapeBottom = 21.0f;
			m_Margin_RoundedCorner = 0.0f;
		}
	}
	//iPad
	else {
		//2018年11月発売の iPad Pro 11 inch, iPad Pro 12.9 inch からホームボタン廃止に伴いセーフエリアあり
		//それ以前から存在するホームボタンありのiPad Pro 12.9 inch モデルはセーフエリアなしであるが
		//識別が困難なためセーフエリアありとして処理する
		//iPad Air(4th,5th), iPad mini 6th はセーフエリアあり
		//12.9 inch モデルより大きい未知のデバイスはセーフエリアありとみなす
		if ((longSide == 1133) || (longSide == 1180) || (longSide == 1194) || (longSide == 1366) || (longSide > 1366)) {
			m_isEnableSafeArea = true;
			m_Margin_LandscapeTop = 0.0f;
			m_Margin_LandscapeLeft = 0.0f;
			m_Margin_LandscapeRight = 0.0f;
			m_Margin_LandscapeBottom = 20.0f;
			m_Margin_RoundedCorner = 10.0f;
		}
	}
	
	return;
}

//******************************************************************************
// PPI情報初期化
//******************************************************************************
void MTDeviceScreenInfo::_InitializePPI()
{
	CGFloat height = 0;
	CGFloat width = 0;
	CGFloat longSide = 0;
	CGFloat scale = 0;
	CGFloat downSampling = 1;
	
	height = [[UIScreen mainScreen] bounds].size.height;
	width = [[UIScreen mainScreen] bounds].size.width;
	
	longSide = height;
	if (height < width) {
		longSide = width;
	}
	
	scale = [UIScreen mainScreen].scale;
	
	//iPhone
	if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
		if (longSide == 480) {
			//iPhone 2G,3G,3GS
			if (scale == 1) {
				m_PixcelsPerInch = 163;
			}
			//iPhone 4,4s
			else if (scale == 2) {
				m_PixcelsPerInch = 326;
			}
			//未知のデバイス
			else {
				m_PixcelsPerInch = 326;
			}
		}
		else if (longSide == 568) {
			//iPhone 5,5s,5c,SE
			m_PixcelsPerInch = 326;
		}
		else if (longSide == 667) {
			//iPhone 6,6s,7,8,SE(2nd,3rd)
			m_PixcelsPerInch = 326;
		}
		else if (longSide == 736) {
			//iPhone 6 Plus,6s Plus,7 Plus,8 Plus
			m_PixcelsPerInch = 401;
			downSampling = 1.15;
		}
		else if (longSide == 812) {
			//iPhone X,XS,11 Pro
			m_PixcelsPerInch = 458;
			//iPhone 12 mini,13 mini は 476ppi であるが
			//scaleがX,XS,11 Proと同じ3であるため識別できない
			//m_PixcelsPerInch = 476;
		}
		else if (longSide == 844) {
			//iPhone 12,12 Pro,13,13 Pro
			m_PixcelsPerInch = 460;
		}
		else if (longSide == 896) {
			//iPhone XR,11
			if (scale == 2) {
				m_PixcelsPerInch = 326;
			}
			//iPhone XS Max,11Pro Max
			else if (scale == 3) {
				m_PixcelsPerInch = 458;
			}
			//未知のデバイス
			else {
				m_PixcelsPerInch = 458;
			}
		}
		else if (longSide == 926) {
			//iPhone 12 Pro Max,13 Pro Max
			m_PixcelsPerInch = 458;
		}
		else {
			//未知のデバイス
			m_PixcelsPerInch = 458;
		}
		
		//1ポイントあたりの長さ(mm) memo:1インチは25.4mm
		m_MmPerPoint = scale * 25.4 / m_PixcelsPerInch / downSampling;
	}
	//iPad
	else {
		//iPadは未対応
		m_PixcelsPerInch = 0;
		m_MmPerPoint = 0;
	}
	
	return;
}

//******************************************************************************
// セーフエリア有無
//******************************************************************************
bool MTDeviceScreenInfo::IsEnableSafeArea()
{
	return m_isEnableSafeArea;
}

//******************************************************************************
// マージン幅取得：横向き/上
//******************************************************************************
float MTDeviceScreenInfo::GetMargin_LandscapeTop()
{
	return m_Margin_LandscapeTop;
}

//******************************************************************************
// マージン幅取得：横向き/左
//******************************************************************************
float MTDeviceScreenInfo::GetMargin_LandscapeLeft()
{
	return m_Margin_LandscapeLeft;
}

//******************************************************************************
// マージン幅取得：横向き/右
//******************************************************************************
float MTDeviceScreenInfo::GetMargin_LandscapeRight()
{
	return m_Margin_LandscapeRight;
}

//******************************************************************************
// マージン幅取得：横向き/下
//******************************************************************************
float MTDeviceScreenInfo::GetMargin_LandscapeBottom()
{
	return m_Margin_LandscapeBottom;
}

//******************************************************************************
// マージン幅取得：角丸
//******************************************************************************
float MTDeviceScreenInfo::GetMargin_RoundedCorner()
{
	return m_Margin_RoundedCorner;
}

//******************************************************************************
// Pixcels per inch 取得（1インチあたりのピクセル数）：iPhoneのみ対応
//******************************************************************************
float MTDeviceScreenInfo::GetPixcelsPerInch()
{
	return m_PixcelsPerInch;
}

//******************************************************************************
// mm per point 取得（1ポイントあたりの長さ(mm)）：iPhoneのみ対応
//******************************************************************************
float MTDeviceScreenInfo::GetMmPerPoint()
{
	return m_MmPerPoint;
}

