//******************************************************************************
//
// MIDITrail / MTColorConf
//
// カラー設定クラス
//
// Copyright (C) 2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTParam.h"
#import "MTConfFile.h"
#import "MTColorConf.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTColorConf::MTColorConf(void)
{
	m_pUserConf = nil;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTColorConf::~MTColorConf(void)
{
	int i = 0;
	
	for (i = 0; i < MT_COLOR_PALETTE_NUM_MAX; i++) {
		delete m_pColorPalette[i];
		m_pColorPalette[i] = NULL;
	}
	
	[m_pUserConf release];
	[m_pCategory release];
	
	return;
}

//******************************************************************************
// 初期化
//******************************************************************************
int MTColorConf::Initialize(NSString* pCategory)
{
	int result = 0;
	int i = 0;
	
	//ユーザ設定初期化
	m_pUserConf = [[YNUserConf alloc] init];
	if (m_pUserConf == nil) {
		YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//色パレット生成と初期化
	for (i = 0; i < MT_COLOR_PALETTE_NUM_MAX; i++) {
		try {
			m_pColorPalette[i] = new MTColorPalette();
		}
		catch (std::bad_alloc) {
			result = YN_SET_ERR(@"Could not allocate memory.", i, 0);
			goto EXIT;
		}
		result = m_pColorPalette[i]->Initialize();
		if (result != 0) goto EXIT;
	}
	
	m_pCategory = pCategory;
	[m_pCategory retain];
	
	//ユーザ設定読み込み
	result = _LoadColorConf();
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// 選択カラーパレット番号取得：0 デフォルト、1-6 パレット番号
//******************************************************************************
unsigned int MTColorConf::GetSelectedColorPaletteNo()
{
	return m_SelectedColorPaletteNo;
}

//******************************************************************************
// 選択カラーパレット番号登録：0 デフォルト、1-6 パレット番号
//******************************************************************************
int MTColorConf::SetSelectedColorPaletteNo(unsigned int paletteNo)
{
	int result = 0;
	
	if (paletteNo >= MT_COLOR_PALETTE_NUM_MAX) {
		result = YN_SET_ERR(@"Program error.", paletteNo, 0);
		goto EXIT;
	}
	
	m_SelectedColorPaletteNo = paletteNo;
	
EXIT:;
	return result;
}

//******************************************************************************
// カラーパレット取得：0 デフォルト、1-6 パレット番号
//******************************************************************************
int MTColorConf::GetColorPalette(unsigned int paletteNo, MTColorPalette* pColorPalette)
{
	int result = 0;
	
	if (paletteNo >= MT_COLOR_PALETTE_NUM_MAX) {
		result = YN_SET_ERR(@"Program error.", paletteNo, 0);
		goto EXIT;
	}
	
	pColorPalette->CopyFrom(m_pColorPalette[paletteNo]);
	
EXIT:;
	return result;
}

//******************************************************************************
// 選択カラーパレット取得
//******************************************************************************
void MTColorConf::GetSelectedColorPalette(MTColorPalette* pColorPalette)
{
	pColorPalette->CopyFrom(m_pColorPalette[m_SelectedColorPaletteNo]);
}

//******************************************************************************
// カラーパレット登録：1-6 パレット番号、0 デフォルトは登録不可
//******************************************************************************
int MTColorConf::SetColorPalette(unsigned int paletteNo, MTColorPalette* pColorPalette)
{
	int result = 0;
	
	//デフォルト0のパレットは書き換え不可
	if ((paletteNo == 0) || (paletteNo >= MT_COLOR_PALETTE_NUM_MAX)) {
		result = YN_SET_ERR(@"Program error.", paletteNo, 0);
		goto EXIT;
	}
	
	m_pColorPalette[paletteNo]->CopyFrom(pColorPalette);
	
EXIT:;
	return result;
}

//******************************************************************************
// ユーザ設定読み込み
//******************************************************************************
int MTColorConf::_LoadColorConf()
{
	int result = 0;
	unsigned int paletteNo = 0;
	
	//カテゴリ／セクション設定
	[m_pUserConf setCategory:MT_CONF_CATEGORY_COLOR];
	[m_pUserConf setSection:MT_CONF_SECTION_COLOR_SELECT];
	
	//ユーザ設定値取得：選択カラーパレット番号
	m_SelectedColorPaletteNo = [m_pUserConf intValueForKey:@"SelectedColorPaletteNo" defaultValue:0];
	if (m_SelectedColorPaletteNo >= MT_COLOR_PALETTE_NUM_MAX) {
		m_SelectedColorPaletteNo = 0;
	}
	
	//デフォルトカラーパレット読み込み
	result = _LoadColorPaletteDefault(m_pColorPalette[0]);
	if (result != 0) goto EXIT;

	//カラーパレット設定読み込み
	for (paletteNo = 1; paletteNo < MT_COLOR_PALETTE_NUM_MAX; paletteNo++) {
		result = _LoadColorPalettes(paletteNo, m_pColorPalette[paletteNo]);
		if (result != 0) goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// デフォルトカラーパレット読み込み
//******************************************************************************
int MTColorConf::_LoadColorPaletteDefault(MTColorPalette* pColorPalette)
{
	int result = 0;
	unsigned int chNo = 0;
	NSString* pKey = nil;
	NSString* pHexColor = nil;
	MTConfFile confFile;
	
	//設定ファイル読み込み
	result = confFile.Initialize(m_pCategory);
	if (result != 0) goto EXIT;
	
	//セクション指定
	result = confFile.SetCurSection(@"Color");
	if (result != 0) goto EXIT;
	
	//チャンネル色取得
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		pKey = [NSString stringWithFormat:@"Ch-%02u-NoteRGBA", chNo+1];
		result = confFile.GetStr(pKey, &pHexColor, @"FFFFFFFF");
		if (result != 0) goto EXIT;
		pColorPalette->SetChColor(chNo, OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
	}
	//背景色取得
	result = confFile.GetStr(@"BackGroundRGBA", &pHexColor, @"FFFFFFFF");
	if (result != 0) goto EXIT;
	pColorPalette->SetBackgroundColor(OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
	
	//グリッドライン色取得
	result = confFile.GetStr(@"GridLineRGBA", &pHexColor, @"444444FF");
	if (result != 0) goto EXIT;
	pColorPalette->SetGridLineColor(OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
	
	//カウンター色取得
	result = confFile.GetStr(@"CaptionRGBA", &pHexColor, @"FFFFFFFF");
	if (result != 0) goto EXIT;
	pColorPalette->SetCounterColor(OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
	
EXIT:;
	return result;
}

//******************************************************************************
// カラーパレット読み込み
//******************************************************************************
int MTColorConf::_LoadColorPalettes(unsigned int paletteNo, MTColorPalette* pColorPalette)
{
	int result = 0;
	unsigned int chNo = 0;
	NSString* pSection = nil;
	NSString* pKey = nil;
	NSString* pHexColor = nil;
	
	//カテゴリ／セクション設定
	[m_pUserConf setCategory:MT_CONF_CATEGORY_COLOR];
	pSection = [NSString stringWithFormat:MT_CONF_SECTION_COLOR_PALETTE, paletteNo];
	[m_pUserConf setSection:pSection];
	
	//チャンネル色取得
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		pKey = [NSString stringWithFormat:@"Ch-%02u-NoteRGBA", chNo+1];
		pHexColor = [m_pUserConf strValueForKey:pKey defaultValue:@"FFFFFFFF"];
		pColorPalette->SetChColor(chNo, OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
		//NSLog(@"key: %@, value: %@", pKey, pHexColor);
	}
	
	//背景色取得
	pHexColor = [m_pUserConf strValueForKey:@"BackGroundRGBA" defaultValue:@"000000FF"];
	pColorPalette->SetBackgroundColor(OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
	//NSLog(@"key: BackGroundRGBA, value: %@", pHexColor);
	
	//グリッドライン色取得
	pHexColor = [m_pUserConf strValueForKey:@"GridLineRGBA" defaultValue:@"444444FF"];
	pColorPalette->SetGridLineColor(OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
	//NSLog(@"key: GridLineRGBA, value: %@", pHexColor);
	
	//カウンター色取得
	pHexColor = [m_pUserConf strValueForKey:@"CaptionRGBA" defaultValue:@"FFFFFFFF"];
	pColorPalette->SetCounterColor(OGLColorUtil::MakeColorFromHexRGBA(pHexColor));
	//NSLog(@"key: CaptionRGBA, value: %@", pHexColor);
	
EXIT:;
	return result;
}

//******************************************************************************
// 設定保存
//******************************************************************************
int MTColorConf::Save()
{
	int result = 0;
	unsigned int paletteNo = 0;
	
	//選択カラーパレット番号保存
	[m_pUserConf setCategory:MT_CONF_CATEGORY_COLOR];
	[m_pUserConf setSection:MT_CONF_SECTION_COLOR_SELECT];
	[m_pUserConf setInt:m_SelectedColorPaletteNo forKey:@"SelectedColorPaletteNo"];
	
	//カラーパレット 1-6 保存
	for (paletteNo = 1; paletteNo < MT_COLOR_PALETTE_NUM_MAX; paletteNo++) {
		result = _SaveColorPalette(paletteNo, m_pColorPalette[paletteNo]);
		if (result != 0) goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// カラーパレット保存
//******************************************************************************
int MTColorConf::_SaveColorPalette(unsigned int paletteNo, MTColorPalette* pColorPalette)
{
	int result = 0;
	unsigned int chNo = 0;
	NSString* pSection = nil;
	NSString* pKey = nil;
	NSString* pHexColor = nil;
	OGLCOLOR color;
	
	//カテゴリ／セクション設定
	[m_pUserConf setCategory:MT_CONF_CATEGORY_COLOR];
	pSection = [NSString stringWithFormat:MT_CONF_SECTION_COLOR_PALETTE, paletteNo];
	[m_pUserConf setSection:pSection];
	
	//チャンネル色登録
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		pKey = [NSString stringWithFormat:@"Ch-%02u-NoteRGBA", chNo+1];
		result = pColorPalette->GetChColor(chNo, &color);
		if (result != 0) goto EXIT;
		pHexColor = OGLColorUtil::MakeHexRGBAFromColor(color);
		[m_pUserConf setStr:pHexColor forKey:pKey];
		//NSLog(@"key: %@, value: %@", pKey, pHexColor);
	}
	
	//背景色登録
	pColorPalette->GetBackgroundColor(&color);
	pHexColor = OGLColorUtil::MakeHexRGBAFromColor(color);
	[m_pUserConf setStr:pHexColor forKey:@"BackGroundRGBA"];
	//NSLog(@"key: BackGroundRGBA, value: %@", pHexColor);
	
	//グリッドライン色登録
	pColorPalette->GetGridLineColor(&color);
	pHexColor = OGLColorUtil::MakeHexRGBAFromColor(color);
	[m_pUserConf setStr:pHexColor forKey:@"GridLineRGBA"];
	//NSLog(@"key: GridLineRGBA, value: %@", pHexColor);
	
	//カウンター色登録
	pColorPalette->GetCounterColor(&color);
	pHexColor = OGLColorUtil::MakeHexRGBAFromColor(color);
	[m_pUserConf setStr:pHexColor forKey:@"CaptionRGBA"];
	//NSLog(@"key: CaptionRGBA, value: %@", pHexColor);
	
EXIT:;
	return result;
}
