//******************************************************************************
//
// OGL Utility / OGLTransMatrix
//
// 変換行列クラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "OGLTypes.h"
#import "OGLDevice.h"
#include <simd/simd.h>


//******************************************************************************
// パラメータ定義
//******************************************************************************
//最大乗算数
//  一般的な使用例を考慮して最大6とする
//  (1) 拡大縮小
//  (2) 回転（X,Y,Z軸）
//  (3) 移動
//  (4) 拡大縮小（左手系→右手系変換）
#define OGLTRANSMATRIX_MULTIPLY_MAX  (6)


//******************************************************************************
// 変換行列クラス
//******************************************************************************
class OGLTransMatrix
{
public:
	
	//コンストラクタ／デストラクタ
	OGLTransMatrix(void);
	virtual ~OGLTransMatrix(void);
	
	//クリア
	void Clear();
	
	//拡大縮小
	void RegistScale(OGLfloat x, OGLfloat y, OGLfloat z);
	
	//回転
	void RegistRotationX(OGLfloat angle);
	void RegistRotationY(OGLfloat angle);
	void RegistRotationZ(OGLfloat angle);
	void RegistRotationXYZ(OGLfloat angle, OGLVECTOR3 axisVector);
	
	//移動
	void RegistTranslation(OGLfloat x, OGLfloat y, OGLfloat z);
	
	//変換行列適用／解除
	void push(OGLDevice* pDevice);
	void pop(OGLDevice* pDevice);
	
	//コピー
	void CopyFrom(OGLTransMatrix* pTransMatrix);
	
	//モデル行列取得
	matrix_float4x4 GetModelMatrix();
	
private:
	
	//変換種別
	enum OGLTransType {
		OGLTransNone,			//なし
		OGLTransScale,			//拡大縮小
		OGLTransRotation,		//回転
		OGLTransToranslation	//移動
	};
	
	//変換情報構造隊
	typedef struct {
		OGLTransType type;
		OGLVECTOR3 vector;
		OGLfloat angle;
	} OGLTransInfo;
	
	//変換情報配列
	OGLTransInfo m_TransInfo[OGLTRANSMATRIX_MULTIPLY_MAX];
	
	//変換情報登録数
	unsigned int m_TransNum;
	
	//glRotatef 代替関数
	matrix_float4x4 _glRotatef(
						OGLfloat angle,
						OGLfloat x,
						OGLfloat y,
						OGLfloat z
					);

	//glTranslatef 代替関数
	matrix_float4x4 _glTranslatef(
						OGLfloat x,
						OGLfloat y,
						OGLfloat z
					);

	//glScalef 代替関数
	matrix_float4x4 _glScalef(
						OGLfloat x,
						OGLfloat y,
						OGLfloat z
					);
	
};


