//******************************************************************************
//
// MIDITrail / MTDashboardLive
//
// ライブモニタ用ダッシュボード描画クラス
//
// Copyright (C) 2012-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTParam.h"
#import "MTConfFile.h"
#import "MTColorConf.h"
#import "MTDashboardLive.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTDashboardLive::MTDashboardLive(void)
{
	m_ViewBounds = CGRectMake(0, 0, 0, 0);
	m_PosTitleX = 0.0f;
	m_PosTitleY = 0.0f;
	m_PosCounterX = 0.0f;
	m_PosCounterY = 0.0f;
	m_PosFPSX = 0.0f;
	m_PosFPSY = 0.0f;
	m_CounterMag = MTDASHBOARDLIVE_DEFAULT_MAGRATE;
	m_isMonitoring = false;
	m_NoteCount = 0;
	m_FPS = 0.0f;
	m_CaptionColor = OGLCOLOR(1.0f, 1.0f, 1.0f, 1.0f);
	m_isEnable = true;
	m_isEnableFPS = true;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTDashboardLive::~MTDashboardLive(void)
{
	Release();
}

//******************************************************************************
// ダッシュボード生成
//******************************************************************************
int MTDashboardLive::Create(
		OGLDevice* pOGLDevice,
		NSString* pSceneName,
		CGRect viewBounds
   )
{
	int result = 0;
	char counter[MTDASHBOARDLIVE_COUNTER_SIZE];
	char fps[MTDASHBOARDLIVE_FPS_SIZE];

	Release();
	
	m_ViewBounds = viewBounds;
	
	//デバイススクリーン情報初期化
	m_DeviceScreenInfo.Initialize();
	
	//設定読み込み
	result = _LoadConfFile(pSceneName);
	if (result != 0) goto EXIT;
	
	//タイトルキャプション
	result = SetMIDIINDeviceName(pOGLDevice, @"");
	if (result != 0) goto EXIT;
	
	//カウンタキャプション
	result = m_Counter.Create(
					pOGLDevice,
					MTDASHBOARDLIVE_FONTNAME,		//フォント名称
					MTDASHBOARDLIVE_FONTSIZE,		//フォントサイズ
					MTDASHBOARDLIVE_COUNTER_CHARS,	//表示文字
					MTDASHBOARDLIVE_COUNTER_SIZE	//キャプションサイズ
				);
	if (result != 0) goto EXIT;
	m_Counter.SetColor(m_CaptionColor);
	
	//FPS
	result = m_CounterFPS.Create(
					pOGLDevice,
					MTDASHBOARDLIVE_FONTNAME,		//フォント名称
					MTDASHBOARDLIVE_FONTSIZE,		//フォントサイズ
					MTDASHBOARDLIVE_FPS_CHARS,		//表示文字
					MTDASHBOARDLIVE_FPS_SIZE		//キャプションサイズ
				);
	if (result != 0) goto EXIT;
	m_CounterFPS.SetColor(m_CaptionColor);
	
	//カウンタ表示文字列生成
	result = _GetCounterStr(counter, MTDASHBOARDLIVE_COUNTER_SIZE);
	if (result != 0) goto EXIT;
	
	result = m_Counter.SetString(counter);
	if (result != 0) goto EXIT;
	
	//カウンタ表示位置を算出
	result = _GetCounterPos(&m_PosCounterX, &m_PosCounterY);
	if (result != 0) goto EXIT;
	
	//タイトル表示位置
	m_PosTitleX = m_PosCounterX;
	m_PosTitleY = MTDASHBOARDLIVE_FRAMESIZE + m_DeviceScreenInfo.GetMargin_LandscapeTop();
	
	//FPS表示文字列生成
	result = _GetFPSStr(fps, MTDASHBOARDLIVE_FPS_SIZE);
	if (result != 0) goto EXIT;
	
	result = m_CounterFPS.SetString(fps);
	if (result != 0) goto EXIT;
	
	//FPS表示位置
	result = _GetCounterFPSPos(&m_PosFPSX, &m_PosFPSY);
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// 移動
//******************************************************************************
int MTDashboardLive::Transform(
		OGLDevice* pOGLDevice,
		OGLVECTOR3 camVector
	)
{
	int result = 0;
	return result;
}

//******************************************************************************
// 描画
//******************************************************************************
int MTDashboardLive::Draw(
		OGLDevice* pOGLDevice
	)
{
	int result = 0;
	char counter[MTDASHBOARDLIVE_COUNTER_SIZE];
	char fps[MTDASHBOARDLIVE_FPS_SIZE];
	
	if (pOGLDevice == NULL) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//タイトルとカウンタ文字列描画
	if (m_isEnable) {
		//タイトル描画：カウンタと同じ拡大率で表示する
		result = m_Title.Draw(pOGLDevice, m_PosTitleX, m_PosTitleY, m_CounterMag);
		if (result != 0) goto EXIT;
		
		result = _GetCounterStr(counter, MTDASHBOARDLIVE_COUNTER_SIZE);
		if (result != 0) goto EXIT;
		
		result = m_Counter.SetString(counter);
		if (result != 0) goto EXIT;
		
		result = m_Counter.Draw(pOGLDevice, m_PosCounterX, m_PosCounterY, m_CounterMag);
		if (result != 0) goto EXIT;
	}
	
	//FPS文字列描画
	if (m_isEnableFPS) {
		result = _GetFPSStr(fps, MTDASHBOARDLIVE_FPS_SIZE);
		if (result != 0) goto EXIT;
		
		result = m_CounterFPS.SetString(fps);
		if (result != 0) goto EXIT;
		
		result = m_CounterFPS.Draw(pOGLDevice, m_PosFPSX, m_PosFPSY, m_CounterMag);
		if (result != 0) goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// 解放
//******************************************************************************
void MTDashboardLive::Release()
{
	m_Title.Release();
	m_Counter.Release();
}

//******************************************************************************
// カウンタ表示位置取得
//******************************************************************************
int MTDashboardLive::_GetCounterPos(
		float* pX,
		float* pY
	)
{
	int result = 0;
	CGRect rect;
	unsigned int cw = 0;
	unsigned int ch = 0;
	unsigned int tw = 0;
	unsigned int th = 0;
	unsigned int charHeight = 0;
	unsigned int charWidth = 0;
	unsigned int captionWidth = 0;
	//float newMag = 0.0f;
	
	//クライアント領域のサイズを取得
	rect = m_ViewBounds;
	cw = rect.size.width;
	ch = rect.size.height;
	
	//テクスチャサイズ取得
	m_Counter.GetTextureSize(&th, &tw);
	
	//文字サイズ
	charHeight = th;
	charWidth = tw / strlen(MTDASHBOARDLIVE_COUNTER_CHARS);
	
	//拡大率1.0のキャプションサイズ
	captionWidth = charWidth * MTDASHBOARDLIVE_COUNTER_SIZE;
	
	//カウンタ文字列が画面からはみ出す場合は画面に収まるように拡大率を更新する
	//  タイトルがはみ出すのは気にしないことにする
	//if (((cw - (MTDASHBOARDLIVE_FRAMESIZE*2)) < captionWidth) && (tw > 0)) {
	//	newMag = (float)(cw - (MTDASHBOARDLIVE_FRAMESIZE*2)) / (float)captionWidth;
	//	if (m_CounterMag > newMag) {
	//		m_CounterMag = newMag;
	//	}
	//}
	
	//テクスチャの表示倍率を考慮して表示位置を算出
	*pX = MTDASHBOARDLIVE_FRAMESIZE;
	*pY = (float)ch - ((float)th * m_CounterMag) - MTDASHBOARDLIVE_FRAMESIZE;
	
	//セーフエリア内に表示位置を移動
	if (m_DeviceScreenInfo.IsEnableSafeArea()) {
		*pX += m_DeviceScreenInfo.GetMargin_LandscapeLeft() + m_DeviceScreenInfo.GetMargin_RoundedCorner();
		*pY -= m_DeviceScreenInfo.GetMargin_LandscapeBottom();
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// FPS表示位置取得
//******************************************************************************
int MTDashboardLive::_GetCounterFPSPos(
		float* pX,
		float* pY
	)
{
	int result = 0;
	CGRect rect;
	unsigned int cw = 0;
	unsigned int ch = 0;
	unsigned int tw = 0;
	unsigned int th = 0;
	unsigned int charHeight, charWidth = 0;
	
	//クライアント領域のサイズを取得
	rect = m_ViewBounds;
	cw = rect.size.width;
	ch = rect.size.height;
	
	//テクスチャサイズ取得
	m_CounterFPS.GetTextureSize(&th, &tw);
	
	//文字サイズ
	charHeight = th;
	charWidth = tw / strlen(MTDASHBOARDLIVE_FPS_CHARS);
	
	//テクスチャの表示倍率を考慮して表示位置を算出：画面右下 "000.0" 5文字
	*pX = (float)cw - MTDASHBOARDLIVE_FRAMESIZE - ((float)charWidth * m_CounterMag * 5);
	*pY = (float)ch - ((float)th * m_CounterMag) - MTDASHBOARDLIVE_FRAMESIZE;
	
	//セーフエリア内に表示位置を移動
	if (m_DeviceScreenInfo.IsEnableSafeArea()) {
		*pX -= m_DeviceScreenInfo.GetMargin_LandscapeRight() + m_DeviceScreenInfo.GetMargin_RoundedCorner();
		*pY -= m_DeviceScreenInfo.GetMargin_LandscapeBottom();
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// モニタ状態登録
//******************************************************************************
void MTDashboardLive::SetMonitoringStatus(
		bool isMonitoring
	)
{
	m_isMonitoring = isMonitoring;
}

//******************************************************************************
// ノートON登録
//******************************************************************************
void MTDashboardLive::SetNoteOn()
{
	m_NoteCount++;
}

//******************************************************************************
// FPS登録
//******************************************************************************
void MTDashboardLive::SetFPS(
		float fps
	)
{
	m_FPS = fps;
}

//******************************************************************************
// カウンタ文字列取得
//******************************************************************************
int MTDashboardLive::_GetCounterStr(
		char* pStr,
		unsigned int bufSize
	)
{
	int result = 0;
	int eresult = 0;
	const char* pMonitorStatus = "";
		
	if (m_isMonitoring) {
		pMonitorStatus = "";
	}
	else {
		pMonitorStatus = "[MONITERING OFF]";
	}
	
	eresult = snprintf(
				pStr,
				bufSize,
				"NOTES:%08u %s",
				m_NoteCount,
				pMonitorStatus
			);
	if (eresult < 0) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// FPS文字列取得
//******************************************************************************
int MTDashboardLive::_GetFPSStr(
		char* pStr,
		unsigned int bufSize
	)
{
	int result = 0;
	int eresult = 0;
	const char* pFPSFormat = NULL;
	
	//FPSフォーマット
	pFPSFormat = "%.1f";

	eresult = snprintf(
				pStr,
				bufSize,
				pFPSFormat,
				m_FPS
			);
	if (eresult < 0) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// リセット
//******************************************************************************
void MTDashboardLive::Reset()
{
	m_isMonitoring = false;
	m_NoteCount = 0;
}

//******************************************************************************
// 設定ファイル読み込み
//******************************************************************************
int MTDashboardLive::_LoadConfFile(
		NSString* pSceneName
	)
{
	int result = 0;
	MTColorConf colorConf;
	MTColorPalette colorPalette;
	OGLCOLOR color;
	
	//カラー設定初期化
	result = colorConf.Initialize(pSceneName);
	if (result != 0) goto EXIT;
	
	//選択カラーパレットからカウンター色取得
	colorConf.GetSelectedColorPalette(&colorPalette);
	colorPalette.GetCounterColor(&color);
	m_CaptionColor = color;
	
EXIT:;
	return result;
}

//******************************************************************************
// カウンタ表示設定
//******************************************************************************
void MTDashboardLive::SetEnable(
		bool isEnable
	)
{
	m_isEnable = isEnable;
}

//******************************************************************************
// FPS表示設定
//******************************************************************************
void MTDashboardLive::SetEnableFPS(
		bool isEnable
	)
{
	m_isEnableFPS = isEnable;
}

//******************************************************************************
//MIDI IN デバイス名登録
//******************************************************************************
int MTDashboardLive::SetMIDIINDeviceName(
		 OGLDevice* pOGLDevice,
		 NSString* pName
	)
{
	int result = 0;
	NSString* pTitle = nil;
	NSString* pDisplayName = nil;
	OGLDevice dummyDevice;
	
	m_Title.Release();
	
	if (pName == nil) {
		pDisplayName = @"(none)";
	}
	else if ([pName length] == 0) {
		pDisplayName = @"(none)";
	}
	else {
		pDisplayName = pName;
	}
	
	//タイトルキャプション
	pTitle = [NSString stringWithFormat:@"MIDI IN: %@", pDisplayName];
	result = m_Title.Create(
					pOGLDevice,					//デバイス
					MTDASHBOARDLIVE_FONTNAME,	//フォント名称
					MTDASHBOARDLIVE_FONTSIZE,	//フォントサイズ
					pTitle						//キャプション
				);
	if (result != 0) goto EXIT;
	m_Title.SetColor(m_CaptionColor);

EXIT:;
	return result;
}


