//******************************************************************************
//
// MIDITrail / MTGamePadCtrl
//
// ゲームパッド制御クラス
//
// Copyright (C) 2019-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTGamePadCtrl.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTGamePadCtrl::MTGamePadCtrl(void)
{
	m_pGameController = nil;
	m_isConnected = false;
	_ClearControllerState();
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTGamePadCtrl::~MTGamePadCtrl(void)
{
	[m_pGameController release];
	m_pGameController = nil;
}

//******************************************************************************
// 初期化
//******************************************************************************
int MTGamePadCtrl::Initialize(int userIndex)
{
	int result = 0;
	NSArray* pControllers = nil;
	GCController* pController = nil;
	
	if (userIndex >= MT_USER_MAX_COUNT) {
		result = YN_SET_ERR(@"Program error.", userIndex, 0);
		goto EXIT;
	}
	
	//ゲームコントローラー破棄
	if (m_pGameController != nil) {
		[m_pGameController release];
		m_pGameController = nil;
	}
	
	//ユーザインデックス：現状は未サポート
	m_UserIndex = userIndex;
	
	//ゲームコントローラー検索
	m_isConnected = false;
	pControllers = [GCController controllers];
	for (pController in pControllers) {
		//ゲームパッドをサポートしているコントローラを探す
		if (pController.extendedGamepad != nil) {
			//最初に見つかったものを選択する
			[pController retain];
			m_pGameController = pController;
			m_isConnected = true;
			NSLog(@"Game Controller with gamepad found");
			break;
		}
	}
	
	if (m_pGameController != nil) {
		m_pGameController.playerIndex = GCControllerPlayerIndex1;
	}
	
	//コントローラー状態初期化
	_ClearControllerState();
	
	//コントローラ状態取得
	_GetGamepadState(&m_CurGamePadState);
	
EXIT:;
	return result;
}

//******************************************************************************
// コントローラー取得
//******************************************************************************
GCController* MTGamePadCtrl::GetController()
{
	return m_pGameController;
}

//******************************************************************************
// ゲームパッド状態更新
//******************************************************************************
int MTGamePadCtrl::UpdateState()
{
	int result = 0;
	
	//前回のゲームパッド状態を退避
	m_PrevGamePadState = m_CurGamePadState;
	
	//現在のゲームパッド状態を取得
	_GetGamepadState(&m_CurGamePadState);
	
EXIT:;
	return result;
}

//******************************************************************************
// コントローラー状態初期化
//******************************************************************************
void MTGamePadCtrl::_ClearControllerState()
{
	memset(&m_PrevGamePadState, 0, sizeof(MT_GAMEPAD_STATE));
	memset(&m_CurGamePadState, 0, sizeof(MT_GAMEPAD_STATE));
}

//******************************************************************************
// コントローラー状態取得
//******************************************************************************
void MTGamePadCtrl::_GetGamepadState(MT_GAMEPAD_STATE* pGamepadState)
{
	//コントローラー状態初期化
	memset(pGamepadState, 0, sizeof(MT_GAMEPAD_STATE));
	
	//コントローラーが未接続なら終了
	if (m_pGameController == nil) goto EXIT;
	
	//拡張ゲームパッドでない場合は何もせず終了
	if (m_pGameController.extendedGamepad == nil) goto EXIT;
	
	//方向キー
	pGamepadState->dpadUp    = m_pGameController.extendedGamepad.dpad.up.value;
	pGamepadState->dpadDown  = m_pGameController.extendedGamepad.dpad.down.value;
	pGamepadState->dpadLeft  = m_pGameController.extendedGamepad.dpad.left.value;
	pGamepadState->dpadRight = m_pGameController.extendedGamepad.dpad.right.value;
	//A,B,X,Yボタン
	pGamepadState->buttonA = m_pGameController.extendedGamepad.buttonA.value;
	pGamepadState->buttonB = m_pGameController.extendedGamepad.buttonB.value;
	pGamepadState->buttonX = m_pGameController.extendedGamepad.buttonX.value;
	pGamepadState->buttonY = m_pGameController.extendedGamepad.buttonY.value;
	//L1/R1 ショルダーボタン
	pGamepadState->buttonLShoulder = m_pGameController.extendedGamepad.leftShoulder.value;
	pGamepadState->buttonRShoulder = m_pGameController.extendedGamepad.rightShoulder.value;
	//L2/R2 トリガーボタン
	pGamepadState->buttonLTrigger = m_pGameController.extendedGamepad.leftTrigger.value;
	pGamepadState->buttonRTrigger = m_pGameController.extendedGamepad.rightTrigger.value;
	//左右スティック
	pGamepadState->thumbLX = m_pGameController.extendedGamepad.leftThumbstick.xAxis.value;
	pGamepadState->thumbLY = m_pGameController.extendedGamepad.leftThumbstick.yAxis.value;
	pGamepadState->thumbRX = m_pGameController.extendedGamepad.rightThumbstick.xAxis.value;
	pGamepadState->thumbRY = m_pGameController.extendedGamepad.rightThumbstick.yAxis.value;
	
EXIT:;
	return;
}

//******************************************************************************
// ボタン状態取得：上
//******************************************************************************
bool MTGamePadCtrl::GetState_DPadUp()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.dpadUp >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：下
//******************************************************************************
bool MTGamePadCtrl::GetState_DPadDown()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.dpadDown >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：左
//******************************************************************************
bool MTGamePadCtrl::GetState_DPadLeft()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.dpadLeft >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：右
//******************************************************************************
bool MTGamePadCtrl::GetState_DPadRight()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.dpadRight >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：A
//******************************************************************************
bool MTGamePadCtrl::GetState_A()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonA >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：B
//******************************************************************************
bool MTGamePadCtrl::GetState_B()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonB >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：X
//******************************************************************************
bool MTGamePadCtrl::GetState_X()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonX >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：Y
//******************************************************************************
bool MTGamePadCtrl::GetState_Y()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonY >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：左ショルダー
//******************************************************************************
bool MTGamePadCtrl::GetState_LShoulder()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonLShoulder >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：右ショルダー
//******************************************************************************
bool MTGamePadCtrl::GetState_RShoulder()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonRShoulder >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：左トリガー
//******************************************************************************
bool MTGamePadCtrl::GetState_LTrigger()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonLTrigger >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// ボタン状態取得：右トリガー
//******************************************************************************
bool MTGamePadCtrl::GetState_RTrigger()
{
	bool state = false;
	
	if ((m_isConnected) && (m_CurGamePadState.buttonRTrigger >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
		state = true;
	}
	
	return state;
}

//******************************************************************************
// スティック状態取得：左スティックX
//******************************************************************************
float MTGamePadCtrl::GetState_ThumbLX()
{
	float state = 0.0f;
	
	if (m_isConnected) {
		state = m_CurGamePadState.thumbLX;
	}
	
	return state;
}

//******************************************************************************
// スティック状態取得：左スティックY
//******************************************************************************
float MTGamePadCtrl::GetState_ThumbLY()
{
	float state = 0.0f;
	
	if (m_isConnected) {
		state = m_CurGamePadState.thumbLY;
	}
	
	return state;
}

//******************************************************************************
// スティック状態取得：右スティックX
//******************************************************************************
float MTGamePadCtrl::GetState_ThumbRX()
{
	float state = 0.0f;
	
	if (m_isConnected) {
		state = m_CurGamePadState.thumbRX;
	}
	
	return state;
}

//******************************************************************************
// スティック状態取得：右スティックY
//******************************************************************************
float MTGamePadCtrl::GetState_ThumbRY()
{
	float state = 0.0f;
	
	if (m_isConnected) {
		state = m_CurGamePadState.thumbRY;
	}
	
	return state;
}

//******************************************************************************
// ボタン押下確認：A
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_A()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonA < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonA >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
			isNow = true;
		}
	}
	
	return isNow;
}

//******************************************************************************
// ボタン押下確認：B
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_B()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonB < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonB >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
			isNow = true;
		}
	}
	
	return isNow;
}

//******************************************************************************
// ボタン押下確認：X
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_X()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonX < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonX >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {

			isNow = true;
		}
	}
	
	return isNow;
}

//******************************************************************************
// ボタン押下確認：Y
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_Y()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonY < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonY >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
			isNow = true;
		}
	}
	
	return isNow;
}

//******************************************************************************
// ボタン押下確認：左ショルダー
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_LShoulder()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonLShoulder < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonLShoulder >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
			isNow = true;
		}
	}
	
	return isNow;
}

//******************************************************************************
// ボタン押下確認：右ショルダー
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_RShoulder()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonRShoulder < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonRShoulder >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
			isNow = true;
		}
	}
	
	return isNow;
}

//******************************************************************************
// ボタン押下確認：左トリガー
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_LTrigger()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonLTrigger < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonLTrigger >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
			isNow = true;
		}
	}
	
	return isNow;
}

//******************************************************************************
// ボタン押下確認：右トリガー
//******************************************************************************
bool MTGamePadCtrl::DidPressNow_RTrigger()
{
	bool isNow = false;
	
	if (m_isConnected) {
		if ((m_PrevGamePadState.buttonRTrigger < MT_GAME_PAD_BUTTON_ON_THRESHOLD)
		 && (m_CurGamePadState.buttonRTrigger >= MT_GAME_PAD_BUTTON_ON_THRESHOLD)) {
			isNow = true;
		}
	}
	
	return isNow;
}

