//******************************************************************************
//
// MIDITrail / MTGridRing
//
// グリッドリング描画クラス
//
// Copyright (C) 2019-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTGridRing.h"
#import "OGLH.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTGridRing::MTGridRing(void)
{
	m_isVisible = true;
	m_isEnable = true;
	m_CurTickTime = 0;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTGridRing::~MTGridRing(void)
{
	Release();
}

//******************************************************************************
// グリッド生成
//******************************************************************************
int MTGridRing::Create(
		OGLDevice* pOGLDevice,
		NSString* pSceneName,
		SMSeqData* pSeqData
   )
{
	int result = 0;
	SMBarList barList;
	unsigned int barNum = 0;
	OGLsizei vertexNum = 0;
	OGLsizei indexNum = 0;
	MTGRIDBOX_VERTEX* pVertex = NULL;
	unsigned int* pIndex = NULL;
	unsigned int totalTickTime = 0;
	OGLMATERIAL material;
	OGLCOLOR lineColor;

	Release();

	if ((pOGLDevice == NULL) || (pSeqData == NULL)) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}

	//ノートデザインオブジェクト初期化
	result = m_NoteDesign.Initialize(pSceneName, pSeqData);
	if (result != 0) goto EXIT;

	//シーケンスデータ：時間情報取得
	totalTickTime = pSeqData->GetTotalTickTime();

	//シーケンスデータ：小節リスト取得
	result = pSeqData->GetBarList(&barList);
	if (result != 0) goto EXIT;

	//シーケンスデータ：ポートリスト取得
	result = pSeqData->GetPortList(&m_PortList);
	if (result != 0) goto EXIT;

	//小節数
	barNum = (OGLsizei)barList.GetSize();

	//プリミティブ初期化
	result = m_Primitive.Initialize(
					pOGLDevice,					//描画デバイス
					sizeof(MTGRIDBOX_VERTEX),	//頂点サイズ
					_GetFVFFormat(),			//頂点FVFフォーマット
					OGL_LINES					//プリミティブ種別
				);
	if (result != 0) goto EXIT;

	//頂点バッファ生成：1リング128頂点 * (小節数 + 終端)
	vertexNum = 128 * (barNum + 1);
	result = m_Primitive.CreateVertexBuffer(pOGLDevice, vertexNum);
	if (result != 0) goto EXIT;

	//インデックスバッファ生成：1リング128辺 * 2(始点/終点) * (小節数　+ 終端)
	indexNum = 128 * 2 * (barNum + 1);
	result = m_Primitive.CreateIndexBuffer(pOGLDevice, indexNum);
	if (result != 0) goto EXIT;

	//バッファのロック
	result = m_Primitive.LockVertex((void**)&pVertex);
	if (result != 0) goto EXIT;
	result = m_Primitive.LockIndex(&pIndex);
	if (result != 0) goto EXIT;

	//グリッドボックスの頂点とインデックスを生成
	result = _CreateVertexOfGrid(
					pVertex,		//頂点バッファ書き込み位置
					pIndex,			//インデックスバッファ書き込み位置
					totalTickTime,	//トータルチックタイム
					 &barList		//小節リスト
				 );
	if (result != 0) goto EXIT;

	//バッファのロック解除
	result = m_Primitive.UnlockVertex();
	if (result != 0) goto EXIT;
	result = m_Primitive.UnlockIndex();
	if (result != 0) goto EXIT;

	//マテリアル作成
	_MakeMaterial(&material);
	m_Primitive.SetMaterial(material);

	//グリッドの色を確認
	lineColor = m_NoteDesign.GetGridLineColor();
	if (((OGLuint)lineColor & 0xFF000000) == 0) {
		//透明なら描画しない
		m_isVisible = false;
	}

EXIT:;
	return result;
}

//******************************************************************************
// 移動
//******************************************************************************
int MTGridRing::Transform(
		OGLDevice* pOGLDevice,
		float rollAngle
	)
{
	int result = 0;
	OGLVECTOR3 moveVector;
	OGLTransMatrix transMatrix;

	//回転
	transMatrix.RegistRotationX(rollAngle);

	//移動
	moveVector = m_NoteDesign.GetWorldMoveVector2(m_CurTickTime);
	transMatrix.RegistTranslation(moveVector.x, moveVector.y, moveVector.z);
	
	//変換行列設定
	m_Primitive.Transform(&transMatrix);

	return result;
}

//******************************************************************************
// 描画
//******************************************************************************
int MTGridRing::Draw(
		OGLDevice* pOGLDevice
	)
{
	int result = 0;

	if (m_isEnable && m_isVisible) {
		result = m_Primitive.Draw(pOGLDevice);
		if (result != 0) goto EXIT;
	}

EXIT:;
	return result;
}

//******************************************************************************
// 解放
//******************************************************************************
void MTGridRing::Release()
{
	m_Primitive.Release();
}

//******************************************************************************
// グリッド頂点生成
//******************************************************************************
int MTGridRing::_CreateVertexOfGrid(
		MTGRIDBOX_VERTEX* pVertex,
		unsigned int* pIndex,
		unsigned int totalTickTime,
		SMBarList* pBarList
	)
{
	int result = 0;
	unsigned int i = 0;
	unsigned int vertexIndex = 0;
	unsigned int tickTime = 0;
	OGLVECTOR3 basePos;
	
	//小節単位のリング
	for (i = 0; i < pBarList->GetSize(); i++) {
		result = pBarList->GetBar(i, &tickTime);
		if (result != 0) goto EXIT;
		
		m_NoteDesign.GetGridRingBasePos(tickTime, &basePos);
		result = _CreateVertexOfRing(pVertex, &vertexIndex, pIndex, basePos);
		if (result != 0) goto EXIT;
	}
	
	//終端リング
	m_NoteDesign.GetGridRingBasePos(totalTickTime, &basePos);
	result = _CreateVertexOfRing(pVertex, &vertexIndex, pIndex, basePos);
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// リング頂点生成
//******************************************************************************
int MTGridRing::_CreateVertexOfRing(
		MTGRIDBOX_VERTEX* pVertex,
		unsigned int* pVirtexIndex,
		unsigned int* pIndex,
		OGLVECTOR3 basePos
	)
{
	int result = 0;
	unsigned int virtexIndexStart = 0;
	unsigned int virtexIndex = 0;
	unsigned int i = 0;
	OGLVECTOR3 rotatedPos;
	float angle = 0.0f;
	
	//頂点作成
	virtexIndexStart = *pVirtexIndex;
	virtexIndex = *pVirtexIndex;
	pVertex[virtexIndex].p = basePos;
	pVertex[virtexIndex].n = OGLVECTOR3(-1.0f, 0.0f, 0.0f);
	pVertex[virtexIndex].c = m_NoteDesign.GetGridLineColor();
	for (i = 1; i < 128; i++) {
		virtexIndex++;
		
		//回転後の頂点
		angle = (360.0f / 128.0f) * (float)i;
		rotatedPos = OGLH::RotateYZ(0.0f, 0.0f, basePos, angle);
		pVertex[virtexIndex].p = rotatedPos;
		pVertex[virtexIndex].n = OGLVECTOR3(-1.0f, 0.0f, 0.0f);
		pVertex[virtexIndex].c = m_NoteDesign.GetGridLineColor();
		
		//インデックスバッファ（前回の頂点から今回の頂点）
		pIndex[(virtexIndex - 1) * 2]     = virtexIndex - 1;
		pIndex[(virtexIndex - 1) * 2 + 1] = virtexIndex;
	}
	//終点と始点をつなぐ線
	pIndex[virtexIndex * 2]     = virtexIndex;
	pIndex[virtexIndex * 2 + 1] = virtexIndexStart;
	
	virtexIndex++;
	*pVirtexIndex = virtexIndex;
	
	return result;
}

//******************************************************************************
// マテリアル作成
//******************************************************************************
void MTGridRing::_MakeMaterial(
		OGLMATERIAL* pMaterial
	)
{
	memset(pMaterial, 0, sizeof(OGLMATERIAL));

	//拡散光
	pMaterial->Diffuse.r = 1.0f;
	pMaterial->Diffuse.g = 1.0f;
	pMaterial->Diffuse.b = 1.0f;
	pMaterial->Diffuse.a = 1.0f;
	//環境光：影の色
	pMaterial->Ambient.r = 0.5f;
	pMaterial->Ambient.g = 0.5f;
	pMaterial->Ambient.b = 0.5f;
	pMaterial->Ambient.a = 1.0f;
	//鏡面反射光
	pMaterial->Specular.r = 0.2f;
	pMaterial->Specular.g = 0.2f;
	pMaterial->Specular.b = 0.2f;
	pMaterial->Specular.a = 1.0f;
	//鏡面反射光の鮮明度
	pMaterial->Power = 10.0f;
	//発光色
	pMaterial->Emissive.r = 0.0f;
	pMaterial->Emissive.g = 0.0f;
	pMaterial->Emissive.b = 0.0f;
	pMaterial->Emissive.a = 0.0f;
}

//******************************************************************************
// 表示設定
//******************************************************************************
void MTGridRing::SetEnable(
		bool isEnable
	)
{
	m_isEnable = isEnable;
}

//******************************************************************************
// カレントチックタイム設定
//******************************************************************************
void MTGridRing::SetCurTickTime(
		unsigned int curTickTime
	)
{
	m_CurTickTime = curTickTime;
}

//******************************************************************************
// リセット
//******************************************************************************
void MTGridRing::Reset()
{
	m_CurTickTime = 0;
}

