//******************************************************************************
//
// MIDITrail / MTMainViewCtrl
//
// メインビュー制御クラス
//
// Copyright (C) 2012-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTMainViewCtrl.h"
#import "MTDeviceScreenInfo.h"
#import "OGLRendererInfo.h"


//******************************************************************************
// パラメータ定義
//******************************************************************************
//瞳孔間距離(Interpupillary distance) 単位mm
#define MT_VR_IPD_MM		(63.0)

//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTMainViewCtrl ()

//Metalビュー初期化
- (int)initializeMetalView;

//Metalビュー破棄
- (void)terminateMetalView;

//ボタン初期化
- (void)initButtonsWithDeviceType:(NSString*)pDeviceType;

//シーン描画
- (void)drawScene;

//シーケンサメッセージ処理
- (int)sequencerMsgProc;

//描画処理
- (int)drawProc;

//演奏状態変更通知受信処理
- (int)postPlayStatus:(NSString*)pNotificationName;

//FPS更新
- (void)updateFPS:(uint64_t)startTime;

//タップ
- (void)onTapped;

//通知送信処理
- (int)postNotificationWithName:(NSString*)pName;

//全アイテム非表示
- (void)hideAllItems;

//ボタン表示状態更新
- (void)updateButtonStatus;

//タッチガイド画像表示状態更新
-(void)updateGuideImageStatus;

//デバイス方向更新
- (int)updateDeviceDirection;

//VR用パラメータ初期化
- (int)initializeVRParameters;

//リモートコマンドハンドラ設定
- (void)setRemoteCommandHandlerWithMonitor:(BOOL)isMonitor;

//ロック画面向け演奏情報更新：演奏時間
- (int)setNowPlayingInfoPlaybackTime:(unsigned int)playTimeSec;

//Viewアクション通知
- (void)onViewAction:(NSNotification*)pNotification;

@end


@implementation MTMainViewCtrl

//******************************************************************************
// 生成
//******************************************************************************
- (id)initWithNibName:(NSString *)nibNameOrNil
			   bundle:(NSBundle *)nibBundleOrNil
		rendererParam:(OGLRedererParam)rendererParam
{
	int result = 0;
	NSString* pDeviceType = nil;
	NSString* pNibNameHowToView = nil;
	NSNotificationCenter* pCenter = nil;
	
	//メンバ初期化
	m_ViewBounds =  CGRectMake(0, 0, 0, 0);
	m_pScene = NULL;
	m_pMsgQueue = NULL;
	m_isStopScene = false;
	m_LastTouchTime = 0;
	m_LastTouchTimeOfScreenGamePad = 0;
	m_PrevDrawTime = 0;
	m_FlameCount = 0.0f;
	m_PrevTouchPoint.x = 0;
	m_PrevTouchPoint.y = 0;
	m_TounchesTotalDistance = 0;
	m_isDisplayButtons = YES;
	m_isDisplayButtonsOfScreenGamePad = YES;
	m_isMonitor = false;
	m_ScreenType = MTScene::StandardView;
    m_isScreenLocked = false;
	m_isBackground = false;
	m_SkipBackTimeSpanInMsec = 10000;
	m_SkipForwardTimeSpanInMsec = 10000;
	m_isExistViewActionView = false;
	
	//レンダリングパラメータ
	m_RendererParam = rendererParam;
	
	//バンドル初期化
    self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
	if (self == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//viewプロパティにアクセスすることにより遅延ロードを解消
	[self view];
	
	//Metalビュー初期化
	result = [self initializeMetalView];
	if (result != 0) goto EXIT;
	
	//デバイス種別
    if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
		//iPhone / iPod touch
		pDeviceType = @"iPhone";
    } else {
		//iPad
		pDeviceType = @"iPad";
    }
	
	//Nibファイル名称
	pNibNameHowToView = [NSString stringWithFormat:@"MTHowToView_%@", pDeviceType];
	
	//ビュー制御生成
	m_pHowToViewCtrl = [[MTHowToViewCtrl alloc] initWithNibName:pNibNameHowToView bundle:nil];
	
	//ボタン初期化
	[self initButtonsWithDeviceType:pDeviceType];
	
	//デバイスモーション管理
	m_pMotionManager = [[CMMotionManager alloc] init];
	
	//ゲームパッド制御初期化
	result = m_GamePadCtrl.Initialize(0);
	if (result != 0) goto EXIT;
	
	//ゲームパッド用視点番号
	m_GamePadViewPointNo = 0;
	
	//ゲームコントローラー接続/切断通知先登録
	pCenter = [NSNotificationCenter defaultCenter];
	[pCenter addObserver:self
				selector:@selector(onGameControllerDidConnect)
					name:GCControllerDidConnectNotification
				  object:nil];
	[pCenter addObserver:self
				selector:@selector(onGameControllerDidDisconnect)
					name:GCControllerDidDisconnectNotification
				  object:nil];
	
	//瞳孔間距離に対応するポイント数
	m_IPDPoints = 0;
	
	//Viewアクションビュー制御生成
	m_pViewActionViewCtrl = [[MTViewActionViewCtrl alloc] init];
	if (m_pViewActionViewCtrl == nil) {
		YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//通知先登録：Viewアクション
	pCenter = [NSNotificationCenter defaultCenter];
	[pCenter addObserver:self
				selector:@selector(onViewAction:)
					name:MT_NOTIFICATION_ON_VIEW_ACTION
				  object:nil];
	
	//ユーザ設定初期化
	m_pUserConf = [[YNUserConf alloc] init];
	if (m_pUserConf == nil) {
		YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return self;
}

//******************************************************************************
// 破棄
//******************************************************************************
- (void)dealloc
{
	//ボタン画像破棄：上段
	[m_pImgButtonCloseN release];
	[m_pImgButtonCloseH release];
	[m_pImgButtonFolderPlaybackN release];
	[m_pImgButtonFolderPlaybackH release];
	[m_pImgButtonFolderPlaybackS release];
	[m_pImgButtonRepeatN release];
	[m_pImgButtonRepeatH release];
	[m_pImgButtonRepeatS release];
	[m_pImgButtonViewN release];
	[m_pImgButtonViewH release];
	[m_pImgButtonHelpN release];
	[m_pImgButtonHelpH release];
	//ボタン画像破棄：下段
	[m_pImgButtonPlayN release];
	[m_pImgButtonPlayH release];
	[m_pImgButtonPauseN release];
	[m_pImgButtonPauseH release];
	[m_pImgButtonStopN release];
	[m_pImgButtonStopH release];
	[m_pImgButtonSkipBackwardN release];
	[m_pImgButtonSkipBackwardH release];
	[m_pImgButtonSkipForwardN release];
	[m_pImgButtonSkipForwardH release];
	[m_pImgButtonSpeedDownN release];
	[m_pImgButtonSpeedDownH release];
	[m_pImgButtonSpeedUpN release];
	[m_pImgButtonSpeedUpH release];
	
	//ビュー破棄
	[m_pHowToViewCtrl release];
	
	//Metalビュー破棄
	[self terminateMetalView];
	
	[super dealloc];
	
	return;
}

//******************************************************************************
// Metalビュー初期化
//******************************************************************************
- (int)initializeMetalView
{
	int result = 0;
	CGRect frame;
	CGFloat scale = 0.0f;
	CGSize drawableSize;
	
	//Metalデバイス生成
	m_MTLDevice = MTLCreateSystemDefaultDevice();
	if (m_MTLDevice == nil) {
		result = YN_SET_ERR(@"The system does not support Metal.", 0, 0);
		goto EXIT;
	}
	
	//Metalビューの生成と初期化
	//TODO:xibで定義したビューサイズとデバイスの画面サイズが一致していることを前提にしているため見直す必要あり
	frame = [self.view bounds];
	m_pMetalView = [[MTKView alloc] initWithFrame:frame device:m_MTLDevice];
	if (m_pMetalView == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//Metalビューの登録と最背面への移動
	[self.view addSubview:m_pMetalView];
	[self.view sendSubviewToBack:m_pMetalView];
	
	//Metaビューのデリゲート登録
	m_pMetalView.delegate = self;
	
	//ドローアブル自動リサイズを無効化
	//  デフォルトのYESのままにすると、レンダリング時点でドローアブルのサイズが意図しないサイズに勝手に変わる場合がある。
	//  例えば iPhone 7 Plus では、ドローアブルに(2)のサイズを設定しても、後で(3)のサイズに変わってしまう。
	//  画面の下端にカウンタを描画するときに、座標が画面外に飛び出すことになる。
	//  (1) Points           414 x  736
	//  (2) Render Pixcels  1242 x  736 (x3)
	//  (3) Physical Pixels 1080 x 1920 (/1.15)
	m_pMetalView.autoResizeDrawable = NO;
	
	//描画サイズ設定
	scale = [UIScreen mainScreen].scale;
	drawableSize.width  = frame.size.width  * scale;
	drawableSize.height = frame.size.height * scale;
	m_pMetalView.drawableSize = drawableSize;
	
	//アンチエイリアシング：サンプルモード
	if ((m_RendererParam.isEnableAntialiasing) && (m_RendererParam.sampleMode == OGLMultisampleBit)) {
		m_pMetalView.sampleCount = m_RendererParam.sampleNum;
	}
	else {
		m_pMetalView.sampleCount = 1;
	}
	
	//現在のビューのサイズを確認する
	m_ViewBounds = [m_pMetalView bounds];
	
EXIT:;
	return result;
}

//******************************************************************************
//Metalビュー破棄
//******************************************************************************
- (void)terminateMetalView
{
	[m_pMetalView release];
	[m_MTLDevice release];
}

//******************************************************************************
// ボタン初期化
//******************************************************************************
- (void)initButtonsWithDeviceType:(NSString*)pDeviceType;
{
	//ボタン：上段
	NSString* pPathImgButtonCloseN = nil;
	NSString* pPathImgButtonCloseH = nil;
	NSString* pPathImgButtonFolderPlaybackN = nil;
	NSString* pPathImgButtonFolderPlaybackH = nil;
	NSString* pPathImgButtonFolderPlaybackS = nil;
	NSString* pPathImgButtonRepeatN = nil;
	NSString* pPathImgButtonRepeatH = nil;
	NSString* pPathImgButtonRepeatS = nil;
	NSString* pPathImgButtonViewN = nil;
	NSString* pPathImgButtonViewH = nil;
	NSString* pPathImgButtonHelpN = nil;
	NSString* pPathImgButtonHelpH = nil;
	//ボタン：下段
	NSString* pPathImgButtonPlayN = nil;
	NSString* pPathImgButtonPlayH = nil;
	NSString* pPathImgButtonPauseN = nil;
	NSString* pPathImgButtonPauseH = nil;
	NSString* pPathImgButtonStopN = nil;
	NSString* pPathImgButtonStopH = nil;
	NSString* pPathImgButtonSkipBackwardN = nil;
	NSString* pPathImgButtonSkipBackwardH = nil;
	NSString* pPathImgButtonSkipForwardN = nil;
	NSString* pPathImgButtonSkipForwardH = nil;
	NSString* pPathImgButtonSpeedDownN = nil;
	NSString* pPathImgButtonSpeedDownH = nil;
	NSString* pPathImgButtonSpeedUpN = nil;
	NSString* pPathImgButtonSpeedUpH = nil;
	
	//ボタン画像ファイルパス：上段
	pPathImgButtonCloseN = [NSString stringWithFormat:@"img/%@/Button-Close-N", pDeviceType];
	pPathImgButtonCloseH = [NSString stringWithFormat:@"img/%@/Button-Close-H", pDeviceType];
	pPathImgButtonFolderPlaybackN = [NSString stringWithFormat:@"img/%@/Button-FolderPlayback-N", pDeviceType];
	pPathImgButtonFolderPlaybackH = [NSString stringWithFormat:@"img/%@/Button-FolderPlayback-H", pDeviceType];
	pPathImgButtonFolderPlaybackS = [NSString stringWithFormat:@"img/%@/Button-FolderPlayback-S", pDeviceType];
	pPathImgButtonRepeatN = [NSString stringWithFormat:@"img/%@/Button-Repeat-N", pDeviceType];
	pPathImgButtonRepeatH = [NSString stringWithFormat:@"img/%@/Button-Repeat-H", pDeviceType];
	pPathImgButtonRepeatS = [NSString stringWithFormat:@"img/%@/Button-Repeat-S", pDeviceType];
	pPathImgButtonViewN = [NSString stringWithFormat:@"img/%@/Button-View-N", pDeviceType];
	pPathImgButtonViewH = [NSString stringWithFormat:@"img/%@/Button-View-H", pDeviceType];
	pPathImgButtonHelpN = [NSString stringWithFormat:@"img/%@/Button-Help-N", pDeviceType];
	pPathImgButtonHelpH = [NSString stringWithFormat:@"img/%@/Button-Help-H", pDeviceType];
	//ボタン画像ファイルパス：下段
	pPathImgButtonPlayN  = [NSString stringWithFormat:@"img/%@/Button-Play-N", pDeviceType];
	pPathImgButtonPlayH  = [NSString stringWithFormat:@"img/%@/Button-Play-H", pDeviceType];
	pPathImgButtonPauseN = [NSString stringWithFormat:@"img/%@/Button-Pause-N", pDeviceType];
	pPathImgButtonPauseH = [NSString stringWithFormat:@"img/%@/Button-Pause-H", pDeviceType];
	pPathImgButtonStopN = [NSString stringWithFormat:@"img/%@/Button-Stop-N", pDeviceType];
	pPathImgButtonStopH = [NSString stringWithFormat:@"img/%@/Button-Stop-H", pDeviceType];
	pPathImgButtonSkipBackwardN = [NSString stringWithFormat:@"img/%@/Button-Skip-backward-N", pDeviceType];
	pPathImgButtonSkipBackwardH = [NSString stringWithFormat:@"img/%@/Button-Skip-backward-H", pDeviceType];
	pPathImgButtonSkipForwardN = [NSString stringWithFormat:@"img/%@/Button-Skip-forward-N", pDeviceType];
	pPathImgButtonSkipForwardH = [NSString stringWithFormat:@"img/%@/Button-Skip-forward-H", pDeviceType];
	pPathImgButtonSpeedDownN = [NSString stringWithFormat:@"img/%@/Button-Play-speed-down-N", pDeviceType];
	pPathImgButtonSpeedDownH = [NSString stringWithFormat:@"img/%@/Button-Play-speed-down-H", pDeviceType];
	pPathImgButtonSpeedUpN = [NSString stringWithFormat:@"img/%@/Button-Play-speed-up-N", pDeviceType];
	pPathImgButtonSpeedUpH = [NSString stringWithFormat:@"img/%@/Button-Play-speed-up-H", pDeviceType];
	
	//ボタン画像読み込み：上段
	m_pImgButtonCloseN = [[UIImage imageNamed:pPathImgButtonCloseN] retain];
	m_pImgButtonCloseH = [[UIImage imageNamed:pPathImgButtonCloseH] retain];
	m_pImgButtonFolderPlaybackN = [[UIImage imageNamed:pPathImgButtonFolderPlaybackN] retain];
	m_pImgButtonFolderPlaybackH = [[UIImage imageNamed:pPathImgButtonFolderPlaybackH] retain];
	m_pImgButtonFolderPlaybackS = [[UIImage imageNamed:pPathImgButtonFolderPlaybackS] retain];
	m_pImgButtonRepeatN = [[UIImage imageNamed:pPathImgButtonRepeatN] retain];
	m_pImgButtonRepeatH = [[UIImage imageNamed:pPathImgButtonRepeatH] retain];
	m_pImgButtonRepeatS = [[UIImage imageNamed:pPathImgButtonRepeatS] retain];
	m_pImgButtonViewN = [[UIImage imageNamed:pPathImgButtonViewN] retain];
	m_pImgButtonViewH = [[UIImage imageNamed:pPathImgButtonViewH] retain];
	m_pImgButtonHelpN = [[UIImage imageNamed:pPathImgButtonHelpN] retain];
	m_pImgButtonHelpH = [[UIImage imageNamed:pPathImgButtonHelpH] retain];
	//ボタン画像読み込み：下段
	m_pImgButtonPlayN  = [[UIImage imageNamed:pPathImgButtonPlayN] retain];
	m_pImgButtonPlayH  = [[UIImage imageNamed:pPathImgButtonPlayH] retain];
	m_pImgButtonPauseN = [[UIImage imageNamed:pPathImgButtonPauseN] retain];
	m_pImgButtonPauseH = [[UIImage imageNamed:pPathImgButtonPauseH] retain];
	m_pImgButtonStopN = [[UIImage imageNamed:pPathImgButtonStopN] retain];
	m_pImgButtonStopH = [[UIImage imageNamed:pPathImgButtonStopH] retain];
	m_pImgButtonSkipBackwardN = [[UIImage imageNamed:pPathImgButtonSkipBackwardN] retain];
	m_pImgButtonSkipBackwardH = [[UIImage imageNamed:pPathImgButtonSkipBackwardH] retain];
	m_pImgButtonSkipForwardN = [[UIImage imageNamed:pPathImgButtonSkipForwardN] retain];
	m_pImgButtonSkipForwardH = [[UIImage imageNamed:pPathImgButtonSkipForwardH] retain];
	m_pImgButtonSpeedDownN = [[UIImage imageNamed:pPathImgButtonSpeedDownN] retain];
	m_pImgButtonSpeedDownH = [[UIImage imageNamed:pPathImgButtonSpeedDownH] retain];
	m_pImgButtonSpeedUpN = [[UIImage imageNamed:pPathImgButtonSpeedUpN] retain];
	m_pImgButtonSpeedUpH = [[UIImage imageNamed:pPathImgButtonSpeedUpH] retain];
	
	//ボタンに画像を登録：上段
	[m_pCloseButton setBackgroundImage:m_pImgButtonCloseN forState:UIControlStateNormal];
	[m_pCloseButton setBackgroundImage:m_pImgButtonCloseH forState:UIControlStateHighlighted];
	[m_pFolderPlaybackButton setBackgroundImage:m_pImgButtonFolderPlaybackN forState:UIControlStateNormal];
	[m_pFolderPlaybackButton setBackgroundImage:m_pImgButtonFolderPlaybackH forState:UIControlStateHighlighted];
	[m_pFolderPlaybackButton setBackgroundImage:m_pImgButtonFolderPlaybackS forState:UIControlStateSelected];
	[m_pRepeatButton setBackgroundImage:m_pImgButtonRepeatN forState:UIControlStateNormal];
	[m_pRepeatButton setBackgroundImage:m_pImgButtonRepeatH forState:UIControlStateHighlighted];
	[m_pRepeatButton setBackgroundImage:m_pImgButtonRepeatS forState:UIControlStateSelected];
	[m_pViewButton setBackgroundImage:m_pImgButtonViewN forState:UIControlStateNormal];
	[m_pViewButton setBackgroundImage:m_pImgButtonViewH forState:UIControlStateHighlighted];
	[m_pHelpButton setBackgroundImage:m_pImgButtonHelpN forState:UIControlStateNormal];
	[m_pHelpButton setBackgroundImage:m_pImgButtonHelpH forState:UIControlStateHighlighted];
	//ボタンに画像を登録：下段
	[m_pPlayButton setBackgroundImage:m_pImgButtonPlayN forState:UIControlStateNormal];
	[m_pPlayButton setBackgroundImage:m_pImgButtonPlayH forState:UIControlStateHighlighted];
	[m_pStopButton setBackgroundImage:m_pImgButtonStopN forState:UIControlStateNormal];
	[m_pStopButton setBackgroundImage:m_pImgButtonStopH forState:UIControlStateHighlighted];
	[m_pSkipBackwardButton setBackgroundImage:m_pImgButtonSkipBackwardN forState:UIControlStateNormal];
	[m_pSkipBackwardButton setBackgroundImage:m_pImgButtonSkipBackwardH forState:UIControlStateHighlighted];
	[m_pSkipForwardButton setBackgroundImage:m_pImgButtonSkipForwardN forState:UIControlStateNormal];
	[m_pSkipForwardButton setBackgroundImage:m_pImgButtonSkipForwardH forState:UIControlStateHighlighted];
	[m_pPlaySpeedDownButton setBackgroundImage:m_pImgButtonSpeedDownN forState:UIControlStateNormal];
	[m_pPlaySpeedDownButton setBackgroundImage:m_pImgButtonSpeedDownH forState:UIControlStateHighlighted];
	[m_pPlaySpeedUpButton setBackgroundImage:m_pImgButtonSpeedUpN forState:UIControlStateNormal];
	[m_pPlaySpeedUpButton setBackgroundImage:m_pImgButtonSpeedUpH forState:UIControlStateHighlighted];
	
	//ボタンのタイトルを設定：上段
	[m_pCloseButton setTitle:@"" forState:UIControlStateNormal];
	[m_pCloseButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pFolderPlaybackButton setTitle:@"" forState:UIControlStateNormal];
	[m_pFolderPlaybackButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pFolderPlaybackButton setTitle:@"" forState:UIControlStateSelected];
	[m_pRepeatButton setTitle:@"" forState:UIControlStateNormal];
	[m_pRepeatButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pRepeatButton setTitle:@"" forState:UIControlStateSelected];
	[m_pViewButton setTitle:@"" forState:UIControlStateNormal];
	[m_pViewButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pHelpButton setTitle:@"" forState:UIControlStateNormal];
	[m_pHelpButton setTitle:@"" forState:UIControlStateHighlighted];
	//ボタンのタイトルを設定：下段
	[m_pPlayButton setTitle:@"" forState:UIControlStateNormal];
	[m_pPlayButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pStopButton setTitle:@"" forState:UIControlStateNormal];
	[m_pStopButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pSkipBackwardButton setTitle:@"" forState:UIControlStateNormal];
	[m_pSkipBackwardButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pSkipForwardButton setTitle:@"" forState:UIControlStateNormal];
	[m_pSkipForwardButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pPlaySpeedDownButton setTitle:@"" forState:UIControlStateNormal];
	[m_pPlaySpeedDownButton setTitle:@"" forState:UIControlStateHighlighted];
	[m_pPlaySpeedUpButton setTitle:@"" forState:UIControlStateNormal];
	[m_pPlaySpeedUpButton setTitle:@"" forState:UIControlStateHighlighted];

	//スクリーンゲームパッド用ボタン設定
	m_pScreenGamePad_DPadUp.layer.cornerRadius = 10;
	m_pScreenGamePad_DPadUp.layer.masksToBounds = YES;
	m_pScreenGamePad_DPadDown.layer.cornerRadius = 10;
	m_pScreenGamePad_DPadDown.layer.masksToBounds = YES;
	m_pScreenGamePad_DPadLeft.layer.cornerRadius = 10;
	m_pScreenGamePad_DPadLeft.layer.masksToBounds = YES;
	m_pScreenGamePad_DPadRight.layer.cornerRadius = 10;
	m_pScreenGamePad_DPadRight.layer.masksToBounds = YES;
	m_pScreenGamePad_X.layer.cornerRadius = 10;
	m_pScreenGamePad_X.layer.masksToBounds = YES;
	m_pScreenGamePad_Y.layer.cornerRadius = 10;
	m_pScreenGamePad_Y.layer.masksToBounds = YES;
	
	return;
}

//******************************************************************************
// 初期化処理
//******************************************************************************
- (int)initialize:(SMMsgQueue*)pMsgQueue
{
	int result = 0;
	
	//メッセージキュー
	m_pMsgQueue = pMsgQueue;
	
	//レンダラ初期化
	result = m_Renderer.Initialize(m_pMetalView, m_RendererParam);
	if (result != 0) goto EXIT;
	
	//時間初期化
	result = m_MachTime.Initialize();
	if (result != 0) goto EXIT;
	
	//モニタ状態
	m_isMonitor = NO;
	
	//全アイテム非表示
	[self hideAllItems];
	
	//VR用パラメータ初期化
	result = [self initializeVRParameters];
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// 終了処理
//******************************************************************************
- (void)terminate
{	
	//レンダラ終了
	m_Renderer.Terminate();
	
	return;
}

//******************************************************************************
// ビュー境界取得
//******************************************************************************
- (CGRect)getViewBounds
{
	return m_ViewBounds;
}

//******************************************************************************
// デバイス取得
//******************************************************************************
- (OGLDevice*)getDevice
{
	return m_Renderer.GetDevice();
}

//******************************************************************************
// ビュー取得
//******************************************************************************
- (UIView*)getView
{
	return m_pMetalView;
}

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{	
    [super viewDidLoad];
	// Do any additional setup after loading the view.
	
	return;
}

//******************************************************************************
// インターフェース自動回転確認
//******************************************************************************
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
	BOOL isRotate = NO;
	
	//横長のみに対応する
	if ((interfaceOrientation == UIInterfaceOrientationLandscapeLeft)
	 || (interfaceOrientation == UIInterfaceOrientationLandscapeRight)) {
		isRotate = YES;
	}
	
	return isRotate;
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
- (BOOL)shouldAutorotate
{
	//回転を許可する
	//NSLog(@"shouldAutorotate");
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
- (NSUInteger)supportedInterfaceOrientations
{
	//横長のみに対応する
	//NSLog(@"supportedInterfaceOrientations");
	return UIInterfaceOrientationMaskLandscape;
}

//******************************************************************************
// ビュー表示開始
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{
	//最終タッチ時間
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	m_LastTouchTimeOfScreenGamePad = m_MachTime.GetCurTimeInNanosec();
	
	//前回描画時間
	m_PrevDrawTime = m_MachTime.GetCurTimeInNanosec();
	m_PrevFPSDrawTime = m_MachTime.GetCurTimeInNanosec();
	
	//タイマー開始
	m_pTimer = [NSTimer timerWithTimeInterval:(1.0 / 60.0)
									   target:self
									 selector:@selector(onTimer:)
									 userInfo:nil
									  repeats:YES];
	[[NSRunLoop currentRunLoop] addTimer:m_pTimer
								 forMode:NSDefaultRunLoopMode];
	
	return;
}

//******************************************************************************
// ビュー表示終了
//******************************************************************************
- (void)viewDidDisappear:(BOOL)animated
{
	//タイマー破棄
	[m_pTimer invalidate];
	m_pTimer = nil;
	
	return;
}

//******************************************************************************
// シーン生成
//******************************************************************************
- (int)createScene:(MTScene*)pScene
		   seqData:(SMSeqData*)pSeqData
{
	int result = 0;
	
	//シーン初期化
	m_Renderer.InitScene();
	
	//シーン生成
	result = pScene->Create(m_pMetalView, m_ViewBounds, m_Renderer.GetDevice(), pSeqData);
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// シーン開始
//******************************************************************************
- (int)startScene:(MTScene*)pScene
		isMonitor:(BOOL)isMonitor
{
	int result = 0;
	
	//シーンオブジェクト
	m_pScene = pScene;
	
	//ダミーのクリックイベントを送ることでタッチ操作を有効化する
	m_pScene->OnWindowClicked(WM_LBUTTONDOWN, 0, 0);
	
	//スクリーン種別設定
	m_pScene->SetScreenType(m_ScreenType);
	
    //ロック状態設定
    m_pScene->LockScreen(m_isScreenLocked);
    
	//スクリーンゲームパッド初期化
	result = m_ScreenGamePadCtrl.Initialize();
	if (result != 0) goto EXIT;
	
	//スクリーンゲームパッド登録
	m_pScene->SetScreenGamePadCtrl(&m_ScreenGamePadCtrl);
	
	//モニタフラグ
	m_isMonitor = isMonitor;
	
	//シーンメッセージキューをクリア
	m_SceneMsgQueue.Clear();
	
	//最終タッチ時間
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	m_LastTouchTimeOfScreenGamePad = m_MachTime.GetCurTimeInNanosec();
	
	//VRビューの場合はモーションセンサ利用開始
	if (m_pMotionManager.deviceMotionAvailable) {
		if ((m_ScreenType == MTScene::VRViewSingleLens)
		 || (m_ScreenType == MTScene::VRViewDualLens)) {
			[m_pMotionManager startDeviceMotionUpdates];
		}
	}
	
	//ゲームコントローラー初期化
	[self initializeGameController];
	
	//リモートコマンドハンドラ設定
	[self setRemoteCommandHandlerWithMonitor:isMonitor];
	
	//前回演奏経過時間
	m_PrevPlayTimeSec = 0;
	
EXIT:;
	return result;
}

//******************************************************************************
// シーン停止
//******************************************************************************
- (int)stopScene
{
	int result = 0;
	
	//シーンオブジェクト
	m_pScene = NULL;
	
	//モーションセンサ利用終了
	if (m_pMotionManager.deviceMotionAvailable) {
		[m_pMotionManager stopDeviceMotionUpdates];
	}
	
	return result;
}

//******************************************************************************
// シーン操作：演奏開始
//******************************************************************************
- (int)scene_PlayStart
{
	int result = 0;
	
	result = m_pScene->OnPlayStart();
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// シーン操作：演奏終了
//******************************************************************************
- (int)scene_PlayEnd
{
	int result = 0;
	
	result = m_pScene->OnPlayEnd();
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// シーン操作：巻き戻し
//******************************************************************************
- (int)scene_Rewind
{
	int result = 0;
	
	result = m_pScene->Rewind();
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// シーン操作：視点リセット
//******************************************************************************
- (int)scene_ResetViewpoint
{
	int result = 0;
	
	m_pScene->ResetViewpoint();
	
	return result;
}

//******************************************************************************
// シーン操作：静的視点移動
//******************************************************************************
- (int)scene_MoveToStaticViewpoint:(unsigned int)viewpointNo
{
	int result = 0;
	
	m_pScene->MoveToStaticViewpoint(viewpointNo);
	
	return result;
}

//******************************************************************************
// シーン操作：視点登録
//******************************************************************************
- (int)scene_SetViewpoint:(MTScene::MTViewParamMap*)pParamMap
{
	int result = 0;
	
	m_pScene->SetViewParam(pParamMap);
	
	return result;
}

//******************************************************************************
// シーン操作：視点取得
//******************************************************************************
- (int)scene_GetViewpoint:(MTScene::MTViewParamMap*)pParamMap
{
	int result = 0;
	
	m_pScene->GetViewParam(pParamMap);
	
	return result;
}

//******************************************************************************
// シーン操作：私の視点移動
//******************************************************************************
- (int) scene_MoveToMyViewpoint:(unsigned long)viewpointNo
{
	int result = 0;
	MTScene::MTViewParamMap defParamMap;
	MTScene::MTViewParamMap viewParamMap;
	MTScene::MTViewParamMap::iterator itr;
	NSString* pSection = nil;
	NSString* pKey = nil;
	float param = 0.0f;
	
	//シーンからデフォルト視点を取得
	m_pScene->GetDefaultViewParam(&defParamMap);
	
	//カテゴリ／セクション設定
	[m_pUserConf setCategory:MT_CONF_CATEGORY_VIEW];
	pSection = [NSString stringWithFormat:MT_CONF_SECTION_MYVIEWPOINT, viewpointNo, m_pScene->GetName()];
	[m_pUserConf setSection:pSection];
	
	//パラメータを設定ファイルから取得（未設定の場合はデフォルト視点を適用）
	for (itr = defParamMap.begin(); itr != defParamMap.end(); itr++) {
		pKey = [NSString stringWithCString:(itr->first).c_str() encoding:NSASCIIStringEncoding];
		if ([pKey isEqualToString:@"Z"] || [pKey isEqualToString:@"Phi"]) {
			//MIDITrail Ver.1.x.x と互換を保つため、Z,Phiのプラスマイナスを反転
			param = [m_pUserConf floatValueForKey:pKey defaultValue:-(itr->second)];
			viewParamMap.insert(MTScene::MTViewParamMapPair((itr->first).c_str(), -param));
		}
		else {
			param = [m_pUserConf floatValueForKey:pKey defaultValue:(itr->second)];
			viewParamMap.insert(MTScene::MTViewParamMapPair((itr->first).c_str(), param));
		}
	}
	
	//視点が切り替えられたことをシーンに伝達
	m_pScene->SetViewParam(&viewParamMap);
	
	return result;
}

//******************************************************************************
// シーン操作：私の視点保存
//******************************************************************************
- (int) scene_SaveMyViewpoint:(unsigned long)viewpointNo
{
	int result = 0;
	MTScene::MTViewParamMap viewParamMap;
	MTScene::MTViewParamMap::iterator itr;
	NSString* pSection = nil;
	NSString* pKey = nil;
	
	//シーンから現在の視点を取得
	m_pScene->GetViewParam(&viewParamMap);
	
	//カテゴリ／セクション設定
	[m_pUserConf setCategory:MT_CONF_CATEGORY_VIEW];
	pSection = [NSString stringWithFormat:MT_CONF_SECTION_MYVIEWPOINT, viewpointNo, m_pScene->GetName()];
	[m_pUserConf setSection:pSection];

	//パラメータを設定ファイルに登録
	for (itr = viewParamMap.begin(); itr != viewParamMap.end(); itr++) {
		pKey = [NSString stringWithCString:(itr->first).c_str() encoding:NSASCIIStringEncoding];
		if ([pKey isEqualToString:@"Z"] || [pKey isEqualToString:@"Phi"]) {
			//MIDITrail Ver.1.x.x と互換を保つため、Z,Phiのプラスマイナスを反転して保存
			[m_pUserConf setFloat:-(itr->second) forKey:pKey];
		}
		else {
			[m_pUserConf setFloat:(itr->second) forKey:pKey];
		}
	}

EXIT:;
	return result;
}

//******************************************************************************
// シーン操作：エフェクト設定
//******************************************************************************
- (int)scene_SetEffect:(MTScene::EffectType)type isEnable:(bool)isEnable
{
	int result = 0;
	
	m_pScene->SetEffect(type, isEnable);
	
	return result;
}

//******************************************************************************
// シーン操作：スクリーン種別設定
//******************************************************************************
- (int)scene_SetScreenType:(MTScene::ScreenType)type;
{
	int result = 0;
	
	m_ScreenType = type;
	m_pScene->SetScreenType(m_ScreenType);
	
	//VRビューの場合はモーションセンサ利用開始
	//通常ビューに戻る場合はモーションセンサ利用終了
	if (m_pMotionManager.deviceMotionAvailable) {
		if ((m_ScreenType == MTScene::VRViewSingleLens)
		 || (m_ScreenType == MTScene::VRViewDualLens)) {
			[m_pMotionManager startDeviceMotionUpdates];
		}
		else {
			[m_pMotionManager stopDeviceMotionUpdates];
		}
	}
	
	return result;
}

//******************************************************************************
// シーン操作：スクリーンロック
//******************************************************************************
- (int)scene_LockScreen:(bool)isLocked;
{
    int result = 0;
    
    m_pScene->LockScreen(isLocked);
    
    return result;
}

//******************************************************************************
// シーン操作：ゲームコントローラー更新イベント
//******************************************************************************
- (int)scene_OnGameControllerChanged
{
	int result = 0;
	
	if (m_pScene != NULL) {
		result = m_pScene->OnGameControllerChanged();
		if (result != 0) goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// 描画
//******************************************************************************
- (void)drawInMTKView:(nonnull MTKView *)view
{
	//NSLog(@"drawInMTKView");
	
	//シーン描画（メインスレッドによる描画処理）
	@autoreleasepool {
		[self drawScene];
	}
	
	return;
}

//******************************************************************************
// 描画サイズ変更
//******************************************************************************
- (void)mtkView:(nonnull MTKView *)view drawableSizeWillChange:(CGSize)size
{
	//何もしない
	return;
}

//******************************************************************************
// タイマー
//******************************************************************************
- (void)onTimer:(NSTimer*)timer
{
	int result = 0;
	
	if (m_pScene == NULL) goto EXIT;
	
	//ロック画面表示中やアプリがバックグラウンドに遷移している場合、画面描画処理(drawInMTKView)は呼び出されないため、
	//タイマーを利用してシーケンサメッセージ処理で演奏状態変更通知の送信を継続する。
	//このタイマーは画面描画処理と同様にメインスレッドで実行するため、シーケンサメッセージ処理が並行して実行されることはない。
	
	//シーケンサメッセージ処理
	result = [self sequencerMsgProc];
	if (result != 0) goto EXIT;
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// シーン描画
//******************************************************************************
- (void)drawScene
{
	int result = 0;
	
	if (m_pScene == NULL) goto EXIT;
	
	m_isStopScene = NO;
	
	//シーケンサメッセージ処理
	result = [self sequencerMsgProc];
	if (result != 0) goto EXIT;
	
	//バックグラウンドの場合は描画処理を行わずに終了
	if (m_isBackground) goto EXIT;
	
	//デバイス方向更新
	if ((m_ScreenType == MTScene::VRViewSingleLens)
	 || (m_ScreenType == MTScene::VRViewDualLens)) {
		result = [self updateDeviceDirection];
		if (result != 0) goto EXIT;
	}
	
	//描画
	result = [self drawProc];
	if (result != 0) goto EXIT;
	
	//FPS更新
	[self updateFPS:m_PrevDrawTime];

	//描画処理開始時刻
	m_PrevDrawTime = m_MachTime.GetCurTimeInNanosec();
	
	//ボタン表示状態を更新
	[self updateButtonStatus];
	
	//タッチガイド画像表示状態を更新
	[self updateGuideImageStatus];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
}

//******************************************************************************
// シーケンサメッセージ処理
//******************************************************************************
- (int)sequencerMsgProc
{
	int result = 0;
	bool isExist = false;
	unsigned int wParam = 0;
	unsigned int lParam = 0;
	SMMsgParser parser;
	
	while (YES) {
		//メッセージ取り出し
		result = m_pMsgQueue->GetMessage(&isExist, &wParam, &lParam);
		if (result != 0) goto EXIT;
		
		//メッセージがなければ終了
		if (!isExist) break;
		
		//メッセージ通知
		result = m_pScene->OnRecvSequencerMsg(wParam, lParam);
		if (result != 0) goto EXIT;	
		
		//演奏状態変更通知への対応
		parser.Parse(wParam, lParam);
		if (parser.GetMsg() == SMMsgParser::MsgPlayStatus) {
			//一時停止
			if (parser.GetPlayStatus() == SMMsgParser::StatusPause) {
				[self postPlayStatus:@"onChangePlayStatusPause"];
			}
			//停止（演奏終了）
			else if (parser.GetPlayStatus() == SMMsgParser::StatusStop) {
				[self postPlayStatus:@"onChangePlayStatusStop"];
			}
		}
		
		//演奏時間
		if (parser.GetMsg() == SMMsgParser::MsgPlayTime) {
			if (m_PrevPlayTimeSec != parser.GetPlayTimeSec()) {
				[self setNowPlayingInfoPlaybackTime:parser.GetPlayTimeSec()];
				m_PrevPlayTimeSec = parser.GetPlayTimeSec();
			}
		}
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// 描画処理
//******************************************************************************
- (int)drawProc
{
	int result = 0;
	bool isVRViewDualLens = false;
	
	if (m_ScreenType == MTScene::VRViewDualLens) {
		isVRViewDualLens = true;
	}
	
	//レンダリング
	result = m_Renderer.RenderScene(m_pMetalView, (OGLScene*)m_pScene, isVRViewDualLens, (CGFloat)m_IPDPoints);
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// 演奏状態変更通知受信処理
//******************************************************************************
- (int)postPlayStatus:(NSString*)pNotificationName
{
	int result = 0;
	NSNotification* pNotification = nil;
	NSNotificationCenter* pCenter = nil;
	
	//通知オブジェクトを作成
    pNotification = [NSNotification notificationWithName:pNotificationName
												  object:self
												userInfo:nil];
	//通知する
	pCenter = [NSNotificationCenter defaultCenter];
	
	//通知に対応する処理を演奏スレッドで処理させる場合
	//[pCenter postNotification:pNotification];
	
	//通知に対応する処理をメインスレッドに処理させる場合
	[pCenter performSelectorOnMainThread:@selector(postNotification:)
							  withObject:pNotification
						   waitUntilDone:NO];
	
	return result;
}

//******************************************************************************
// FPS更新
//******************************************************************************
- (void)updateFPS:(uint64_t)startTime
{
	uint64_t curTime = 0;
	uint64_t deltaTime = 0;
	float fps = 0.0f;
	
	//表示フレーム数カウント
	m_FlameCount += 1.0f;
	
	//現在時刻
	curTime = m_MachTime.GetCurTimeInNanosec();
	
	//前回FPS表示更新からの経過時間（ナノ秒）
	deltaTime = curTime - m_PrevFPSDrawTime;
	
	//1秒おきにFPS表示更新
	if (deltaTime >= (1000 * 1000000)) {
		fps = (double)(1000 * 1000000) / ((double)deltaTime / m_FlameCount);
		m_pScene->SetFPS(fps);
		m_PrevFPSDrawTime = curTime;
		m_FlameCount = 0.0f;
		//NSLog(@"FPS:%.1f", fps);
	}
}

//******************************************************************************
// クローズボタン押下
//******************************************************************************
- (IBAction)onCloseButton
{
	//クローズ通知
	[self postNotificationWithName:@"onCloseButton"];
	
	return;
}

//******************************************************************************
// 再生ボタン押下
//******************************************************************************
- (IBAction)onPlayButton
{	
	//再生通知
	[self postNotificationWithName:@"onPlayButton"];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// 停止ボタン押下
//******************************************************************************
- (IBAction)onStopButton
{
	//停止通知
	[self postNotificationWithName:@"onStopButton"];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// 後方スキップボタン押下
//******************************************************************************
- (IBAction)onSkipBackwardButton
{
	//後方スキップ通知
	[self postNotificationWithName:@"onSkipBackwardButton"];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// 前方スキップボタン押下
//******************************************************************************
- (IBAction)onSkipForwardButton
{
	//前方スキップ通知
	[self postNotificationWithName:@"onSkipForwardButton"];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// 再生スピードダウンボタン押下
//******************************************************************************
- (IBAction)onPlaySpeedDownButton
{
	//再生スピードダウン通知
	[self postNotificationWithName:@"onPlaySpeedDownButton"];	
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// 再生スピードアップボタン押下
//******************************************************************************
- (IBAction)onPlaySpeedUpButton
{
	//再生スピードアップ通知
	[self postNotificationWithName:@"onPlaySpeedUpButton"];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// 自動演奏ボタン押下
//******************************************************************************
- (IBAction)onFolderPlaybackButton
{
	//通知
	[self postNotificationWithName:@"onFolderPlaybackButton"];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// リピートボタン押下
//******************************************************************************
- (IBAction)onRepeatButton
{
	//通知
	[self postNotificationWithName:@"onRepeatButton"];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// ビューボタン押下
//******************************************************************************
- (IBAction)onViewButton
{
	CGFloat width = 0;
	CGFloat height = 0;
	
	//ポップオーバービューのサイズ
	if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
		//iPhone / iPod touch
		width = 300.0;
		height = 200.0;
	}
	else {
		//iPad
		width = 300.0;
		height = 400.0;
	}
	
	//ポップオーバービュー設定
	m_pViewActionViewCtrl.modalPresentationStyle = UIModalPresentationPopover;
	m_pViewActionViewCtrl.preferredContentSize = CGSizeMake(width, height);
	m_pViewActionViewCtrl.popoverPresentationController.delegate = self;
	m_pViewActionViewCtrl.popoverPresentationController.permittedArrowDirections = UIPopoverArrowDirectionUp;
	m_pViewActionViewCtrl.popoverPresentationController.sourceView = m_pViewButton;
	m_pViewActionViewCtrl.popoverPresentationController.sourceRect = m_pViewButton.bounds;
	
	//ポップオーバービュー表示
	[m_pViewActionViewCtrl setScreenLockStatus:m_isScreenLocked];
	[self presentViewController:m_pViewActionViewCtrl animated:YES completion:NULL];
	m_isExistViewActionView = true;
	
	return;
}

//******************************************************************************
// プレゼンテーション制御に対するプレゼンテーションスタイル適用
//******************************************************************************
- (UIModalPresentationStyle)adaptivePresentationStyleForPresentationController:(UIPresentationController *)controller
{
	//Viewアクションビューのポップオーバ表示（吹き出し表示）を実現するため実装している
	//この実装がないとiPhoneでViewアクションビューが全画面表示になってしまう
	return UIModalPresentationNone;
}

//******************************************************************************
// プレゼンテーション制御に対するプレゼンテーションスタイル適用
//******************************************************************************
- (UIModalPresentationStyle)adaptivePresentationStyleForPresentationController:(UIPresentationController *)controller
															   traitCollection:(UITraitCollection *)traitCollection
{
	//Viewアクションビューのポップオーバ表示（吹き出し表示）を実現するため実装している
	//この実装がないとiPhone 6 plus, 7 plus でViewアクションビューが全画面表示になってしまう
	return UIModalPresentationNone;
}

//******************************************************************************
// Viewアクション通知
//******************************************************************************
- (void)onViewAction:(NSNotification*)pNotification
{
	int result = 0;
	NSDictionary* pDictionary = nil;
	NSString* pActionName = nil;
	
	pDictionary = [pNotification userInfo];
	if (pDictionary == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	pActionName = [pDictionary objectForKey:@"ActionName"];
	//NSLog(@"ActionName: %@", pActionName);
	
	//Viewpoint 1
	if ([pActionName isEqualToString:@"Viewpoint1"]) {
		[self scene_MoveToStaticViewpoint:1];
	}
	//Viewpoint 2
	else if ([pActionName isEqualToString:@"Viewpoint2"]) {
		[self scene_MoveToStaticViewpoint:2];
	}
	//Viewpoint 3
	else if ([pActionName isEqualToString:@"Viewpoint3"]) {
		[self scene_MoveToStaticViewpoint:3];
	}
	//My Viewpoint 1
	else if ([pActionName isEqualToString:@"MyViewpoint1"]) {
		[self scene_MoveToMyViewpoint:1];
	}
	//My Viewpoint 2
	else if ([pActionName isEqualToString:@"MyViewpoint2"]) {
		[self scene_MoveToMyViewpoint:2];
	}
	//My Viewpoint 3
	else if ([pActionName isEqualToString:@"MyViewpoint3"]) {
		[self scene_MoveToMyViewpoint:3];
	}
	//Save My Viewpoint 1
	else if ([pActionName isEqualToString:@"SaveMyViewpoint1"]) {
		[self scene_SaveMyViewpoint:1];
	}
	//Save My Viewpoint 2
	else if ([pActionName isEqualToString:@"SaveMyViewpoint2"]) {
		[self scene_SaveMyViewpoint:2];
	}
	//Save My Viewpoint 3
	else if ([pActionName isEqualToString:@"SaveMyViewpoint3"]) {
		[self scene_SaveMyViewpoint:3];
	}
	//Rotation: Switch Rotation Mode
	else if ([pActionName isEqualToString:@"SwitchRotationMode"]) {
		m_pScene->OnWindowClicked(WM_MBUTTONDOWN, 0, 0);
	}
	//View Style: Standard View
	else if ([pActionName isEqualToString:@"ViewStyleStandardView"]) {
		[self scene_SetScreenType:(MTScene::StandardView)];
	}
	//View Style: VR View (Single Lens)
	else if ([pActionName isEqualToString:@"ViewStyleVRViewSingleLens"]) {
		[self scene_SetScreenType:(MTScene::VRViewSingleLens)];
	}
	//View Style: VR View (Dual Lens)
	else if ([pActionName isEqualToString:@"ViewStyleVRViewDualLens"]) {
		[self scene_SetScreenType:(MTScene::VRViewDualLens)];
	}
	//Screen: Lock
	else if ([pActionName isEqualToString:@"ScreenLock"]) {
		m_isScreenLocked = true;
		[self scene_LockScreen:m_isScreenLocked];
	}
	//Screen: Unlock
	else if ([pActionName isEqualToString:@"ScreenUnlock"]) {
		m_isScreenLocked = false;
		[self scene_LockScreen:m_isScreenLocked];
	}
	//Action View Did Disappear
	else if ([pActionName isEqualToString:@"DidDisappear"]) {
		m_isExistViewActionView = false;
	}
	else {
		result = YN_SET_ERR(@"Program error.", 1, 0);
		goto EXIT;
	}
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ヘルプボタン押下
//******************************************************************************
- (IBAction)onHelpButton
{
	//HowToViewヘルプ表示
	m_pHowToViewCtrl.modalTransitionStyle = UIModalTransitionStyleCoverVertical;
    [self presentViewController:m_pHowToViewCtrl animated:YES completion:nil];
	
	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// タッチイベント：開始
//******************************************************************************
- (void)touchesBegan:(NSSet*)touches withEvent:(UIEvent *)event
{
	CGPoint point;
	
	//シーンに通知
	if (m_pScene != NULL) {
		//タッチイベント通知
		m_pScene->OnTouchesBegan(touches);
		
		//タップ判定開始
		point = [[touches anyObject] locationInView:m_pMetalView];
		m_PrevTouchPoint.x = point.x;
		m_PrevTouchPoint.y = point.y;
		m_TounchesTotalDistance = 0;
		//NSLog(@"touchesBegan %f, %f", point.x, point.y);
	}
	
	return;
}

//******************************************************************************
// タッチイベント：移動
//******************************************************************************
- (void)touchesMoved:(NSSet*)touches withEvent:(UIEvent *)event
{
	CGPoint point;
	
	//メモ：本イベントは指を移動させなくても定期的に発生する（同じ座標で通知される）
	
	//シーンに通知
	if (m_pScene != NULL) {
		//タッチイベント通知
		m_pScene->OnTouchesMoved(touches);
		
		//移動距離加算
		point = [[touches anyObject] locationInView:m_pMetalView];
		m_TounchesTotalDistance += sqrtf(powf(point.x - m_PrevTouchPoint.x, 2)
										 + powf(point.y - m_PrevTouchPoint.y, 2));
		m_PrevTouchPoint = point;
		//NSLog(@"touchesMoved %f, %f", point.x, point.y);
	}
	
	return;
}

//******************************************************************************
// タッチイベント：終了
//******************************************************************************
- (void)touchesEnded:(NSSet*)touches withEvent:(UIEvent *)event
{
	//シーンに通知
	if (m_pScene != NULL) {
		//タッチイベント通知
		m_pScene->OnTouchesEnd(touches);
		
		//移動距離がゼロの場合はタップとみなす
		//NSLog(@"touchesEnded %f", m_TounchesTotalDistance);
		if (m_TounchesTotalDistance == 0.0f) {
			[self onTapped];
			m_PrevTouchPoint.x = 0;
			m_PrevTouchPoint.y = 0;
			m_TounchesTotalDistance = 0;
		}
	}
	
	return;
}

//******************************************************************************
// タッチイベント：キャンセル
//******************************************************************************
- (void)touchesCancelled:(NSSet*)touches withEvent:(UIEvent *)event
{
	//シーンに通知
	if (m_pScene != NULL) {
		//タッチイベント通知
		m_pScene->OnTouchesCanceled(touches);
		
		//キャンセルが発生した場合はタップ判定解除
		m_PrevTouchPoint.x = 0;
		m_PrevTouchPoint.y = 0;
		m_TounchesTotalDistance = 0;
	}
	
	return;
}

//******************************************************************************
// タップ
//******************************************************************************
- (void)onTapped
{
	//ボタン表示状態を反転
	m_isDisplayButtons = !m_isDisplayButtons;
	m_isDisplayButtonsOfScreenGamePad = m_isDisplayButtons;

	//最終タッチ時間更新
	m_LastTouchTime = m_MachTime.GetCurTimeInNanosec();
	m_LastTouchTimeOfScreenGamePad = m_MachTime.GetCurTimeInNanosec();
	
	return;
}

//******************************************************************************
// 通知送信処理
//******************************************************************************
- (int)postNotificationWithName:(NSString*)pName
{
	int result = 0;
	NSNotification* pNotification = nil;
	NSNotificationCenter* pCenter = nil;
	
	//通知オブジェクトを作成
    pNotification = [NSNotification notificationWithName:pName
												  object:self
												userInfo:nil];
	//通知する
	pCenter = [NSNotificationCenter defaultCenter];
	
	//通知に対応する処理を演奏スレッドで処理させる場合
	//[pCenter postNotification:pNotification];
	
	//通知に対応する処理をメインスレッドに処理させる場合
	[pCenter performSelectorOnMainThread:@selector(postNotification:)
							  withObject:pNotification
						   waitUntilDone:NO];
	
	return result;
}

//******************************************************************************
// 全アイテム非表示
//******************************************************************************
- (void)hideAllItems
{
	//ボタン：上段
	m_pCloseButton.hidden = YES;
	m_pFolderPlaybackButton.hidden = YES;
	m_pRepeatButton.hidden = YES;
	m_pViewButton.hidden = YES;
	m_pHelpButton.hidden = YES;
	
	//ボタン：下段
	m_pPlayButton.hidden = YES;
	m_pStopButton.hidden = YES;
	m_pSkipBackwardButton.hidden = YES;
	m_pSkipForwardButton.hidden = YES;
	m_pPlaySpeedDownButton.hidden = YES;
	m_pPlaySpeedUpButton.hidden = YES;
	
	//ガイド
	m_pGuideImageUD.hidden = YES;
	m_pGuideImageFB.hidden = YES;
	m_pGuideImageLR.hidden = YES;
	m_pGuideImageRT.hidden = YES;
	
	//インジケータ
	m_pActivityIndicator.hidden = YES;
	m_pLabel.hidden = YES;
	
	//スクリーンゲームパッド
	m_pScreenGamePad_DPadUp.hidden = YES;
	m_pScreenGamePad_DPadDown.hidden = YES;
	m_pScreenGamePad_DPadLeft.hidden = YES;
	m_pScreenGamePad_DPadRight.hidden = YES;
	m_pScreenGamePad_X.hidden = YES;
	m_pScreenGamePad_Y.hidden = YES;
	
	return;
}

//******************************************************************************
// ボタン表示状態更新
//******************************************************************************
- (void)updateButtonStatus
{
	uint64_t elapsedTime = 0;
	
	//--------------------------------------
	// ボタン表示更新
	//--------------------------------------
	//最終タッチ時刻から5秒経過したらボタンを隠す
	//ただしViewアクションビュー表示中は5秒経過しても隠さない
	if (m_isDisplayButtons) {
		elapsedTime = m_MachTime.GetCurTimeInNanosec() - m_LastTouchTime;
		if ((elapsedTime > (5000ULL * 1000000ULL)) && !m_isExistViewActionView) {
			m_isDisplayButtons = NO;
		}
	}
	
	//ボタンの表示更新
	m_pCloseButton.hidden = !m_isDisplayButtons;
	m_pFolderPlaybackButton.hidden = !m_isDisplayButtons;
	m_pRepeatButton.hidden = !m_isDisplayButtons;
	m_pViewButton.hidden = !m_isDisplayButtons;
	m_pHelpButton.hidden = !m_isDisplayButtons;
	m_pPlayButton.hidden = !m_isDisplayButtons;
	m_pStopButton.hidden = !m_isDisplayButtons;
	m_pSkipBackwardButton.hidden = !m_isDisplayButtons;
	m_pSkipForwardButton.hidden = !m_isDisplayButtons;
	m_pPlaySpeedDownButton.hidden = !m_isDisplayButtons;
	m_pPlaySpeedUpButton.hidden = !m_isDisplayButtons;

	//MIDI IN モニタでは一部のボタンを表示しない
	if (m_isMonitor) {
		m_pFolderPlaybackButton.hidden = YES;
		m_pRepeatButton.hidden = YES;
		m_pPlayButton.hidden = YES;
		m_pStopButton.hidden = YES;
		m_pSkipBackwardButton.hidden = YES;
		m_pSkipForwardButton.hidden = YES;
		m_pPlaySpeedDownButton.hidden = YES;
		m_pPlaySpeedUpButton.hidden = YES;
	}
	
	//--------------------------------------
	// スクリーンゲームパッド表示更新
	//--------------------------------------
	//スクリーンゲームパッドの最終タッチ時刻を更新
	if (m_ScreenGamePadCtrl.GetState_DPadUp()
		|| m_ScreenGamePadCtrl.GetState_DPadDown()
		|| m_ScreenGamePadCtrl.GetState_DPadLeft()
		|| m_ScreenGamePadCtrl.GetState_DPadRight()
		|| m_ScreenGamePadCtrl.GetState_X()
		|| m_ScreenGamePadCtrl.GetState_Y()) {
		m_LastTouchTimeOfScreenGamePad = m_MachTime.GetCurTimeInNanosec();
	}
	
	//最終タッチ時刻から2.5秒経過したらボタンを隠す（スクリーンゲームパッド）
	//ただしスクリーンロック中はボタンを表示しない
	if (m_isDisplayButtonsOfScreenGamePad) {
		elapsedTime = m_MachTime.GetCurTimeInNanosec() - m_LastTouchTimeOfScreenGamePad;
		if ((elapsedTime > (2500ULL * 1000000ULL)) || m_isScreenLocked) {
			m_isDisplayButtonsOfScreenGamePad = NO;
		}
	}
	
	//スクリーンゲームパッドの表示更新
	m_pScreenGamePad_DPadUp.hidden = !m_isDisplayButtonsOfScreenGamePad;
	m_pScreenGamePad_DPadDown.hidden = !m_isDisplayButtonsOfScreenGamePad;
	m_pScreenGamePad_DPadLeft.hidden = !m_isDisplayButtonsOfScreenGamePad;
	m_pScreenGamePad_DPadRight.hidden = !m_isDisplayButtonsOfScreenGamePad;
	m_pScreenGamePad_X.hidden = !m_isDisplayButtonsOfScreenGamePad;
	m_pScreenGamePad_Y.hidden = !m_isDisplayButtonsOfScreenGamePad;
	
	//--------------------------------------
	// タッチガイド表示中の表示制御
	//--------------------------------------
	//タッチガイド表示中はすべてのボタンを表示しない
	if ((m_pScene->GetCurOperationType() == MTTouchCtrl::OperationMoveUD)
	 || (m_pScene->GetCurOperationType() == MTTouchCtrl::OperationMoveFB)
	 || (m_pScene->GetCurOperationType() == MTTouchCtrl::OperationMoveLR)
	 || (m_pScene->GetCurOperationType() == MTTouchCtrl::OperationRotate)) {
		//ボタン
		m_pCloseButton.hidden = YES;
		m_pFolderPlaybackButton.hidden = YES;
		m_pRepeatButton.hidden = YES;
		m_pViewButton.hidden = YES;
		m_pHelpButton.hidden = YES;
		m_pPlayButton.hidden = YES;
		m_pStopButton.hidden = YES;
		m_pSkipBackwardButton.hidden = YES;
		m_pSkipForwardButton.hidden = YES;
		m_pPlaySpeedDownButton.hidden = YES;
		m_pPlaySpeedUpButton.hidden = YES;
		//スクリーンゲームパッド
		m_pScreenGamePad_DPadUp.hidden = YES;
		m_pScreenGamePad_DPadDown.hidden = YES;
		m_pScreenGamePad_DPadLeft.hidden = YES;
		m_pScreenGamePad_DPadRight.hidden = YES;
		m_pScreenGamePad_X.hidden = YES;
		m_pScreenGamePad_Y.hidden = YES;
	}
	
	return;
}

//******************************************************************************
// タッチガイド画像表示状態更新
//******************************************************************************
-(void)updateGuideImageStatus
{
	m_pGuideImageUD.hidden = YES;
	m_pGuideImageFB.hidden = YES;
	m_pGuideImageLR.hidden = YES;
	m_pGuideImageRT.hidden = YES;
	
	if (m_pScene == NULL) goto EXIT;
	
	//操作中のガイドのみを表示
	switch (m_pScene->GetCurOperationType()) {
		case (MTTouchCtrl::OperationNone):
			break;
		case (MTTouchCtrl::OperationEyeDir):
			break;
		case (MTTouchCtrl::OperationMoveUD):
			m_pGuideImageUD.hidden = NO;
			break;
		case (MTTouchCtrl::OperationMoveFB):
			m_pGuideImageFB.hidden = NO;
			break;
		case (MTTouchCtrl::OperationMoveLR):
			m_pGuideImageLR.hidden = NO;
			break;
		case (MTTouchCtrl::OperationRotate):
			m_pGuideImageRT.hidden = NO;
			break;
		default:
			break;
	}
	
EXIT:;
	return;
}

//******************************************************************************
// フォルダ演奏状態設定
//******************************************************************************
- (void)setFolderPlaybackStatus:(BOOL)isFolderPlayback
{
	if (isFolderPlayback) {
		[m_pFolderPlaybackButton setSelected:YES];
	}
	else {
		[m_pFolderPlaybackButton setSelected:NO];
	}
	
	return;
}

//******************************************************************************
// リピート状態設定
//******************************************************************************
- (void)setRepeatStatus:(BOOL)isRepeat
{
	if (isRepeat) {
		[m_pRepeatButton setSelected:YES];
	}
	else {
		[m_pRepeatButton setSelected:NO];
	}
	
	return;
}

//******************************************************************************
// インジケータアニメーション開始
//******************************************************************************
- (void)startActivityIndicator
{
	[self hideAllItems];
	
	m_pActivityIndicator.hidden = NO;
	m_pLabel.hidden = NO;
	
	[m_pActivityIndicator startAnimating];
	
	return;
}

//******************************************************************************
// インジケータアニメーション停止
//******************************************************************************
- (void)stopActivityIndicator
{
	m_pActivityIndicator.hidden = YES;
	m_pLabel.hidden = YES;
	
	[m_pActivityIndicator stopAnimating];
	
	return;
}

//******************************************************************************
// 演奏状態設定
//******************************************************************************
- (void)setPlayStatus:(PlayStatus)status
{
	//再生ボタンの画像を初期化
	[m_pPlayButton setBackgroundImage:m_pImgButtonPlayN forState:UIControlStateNormal];
	[m_pPlayButton setBackgroundImage:m_pImgButtonPlayH forState:UIControlStateHighlighted];
	
	switch(status) {
		//データなし
		case NoData:
			break;
		//停止状態
		case Stop:
			break;
		//再生中
		case Play:
			//再生ボタンの画像を一時停止アイコンに置き換える
			[m_pPlayButton setBackgroundImage:m_pImgButtonPauseN forState:UIControlStateNormal];
			[m_pPlayButton setBackgroundImage:m_pImgButtonPauseH forState:UIControlStateHighlighted];
			break;
		//一時停止
		case Pause:
			break;	
		//モニタ停止
		case MonitorOFF:
			break;
		//モニタ中
		case MonitorON:
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// ステータスバー非表示
//******************************************************************************
- (BOOL)prefersStatusBarHidden
{
    return YES;
}

//******************************************************************************
// デバイス方向更新
//******************************************************************************
- (int)updateDeviceDirection
{
	int result = 0;
	CMQuaternion quaternion;
	OGLQuaternion q;
	
	//モーションセンサが有効でなければ何もしない
	if (!m_pMotionManager.deviceMotionAvailable) goto EXIT;
	
	//デバイス方向（クォータニオン）取得
	quaternion = m_pMotionManager.deviceMotion.attitude.quaternion;
	q.x = quaternion.x;
	q.y = quaternion.y;
	q.z = quaternion.z;
	q.w = quaternion.w;
	
	//デバイス方向設定
	if (m_pScene != NULL) {
		m_pScene->SetDeviceDirection(q);
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// VR用パラメータ初期化
//******************************************************************************
- (int)initializeVRParameters
{
	int result = 0;
	MTDeviceScreenInfo deviceScreenInfo;
	
	//デバイススクリーン情報初期化
	deviceScreenInfo.Initialize();
	
	//瞳孔間距離(Interpupillary distance)ポイント数を算出
	if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
		//iPhone / iPod touch
		m_IPDPoints = MT_VR_IPD_MM / deviceScreenInfo.GetMmPerPoint();
	}
	else {
		//iPadの場合は無効値を設定
		m_IPDPoints = 0;
	}
	
	return result;
}

//******************************************************************************
// ゲームコントローラー接続通知
//******************************************************************************
- (void)onGameControllerDidConnect
{
	int result = 0;
	
	NSLog(@"Game Controller connected");
	
	//ゲームコントローラー初期化
	result = [self initializeGameController];
	if (result != 0) goto EXIT;
	
	//ゲームコントローラー更新をシーンに通知
	result = [self scene_OnGameControllerChanged];
	if (result != 0) goto EXIT;
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ゲームコントローラー切断通知
//******************************************************************************
- (void)onGameControllerDidDisconnect
{
	int result = 0;
	
	NSLog(@"Game Controller disconnected");
	
	//ゲームコントローラー初期化
	result = [self initializeGameController];
	if (result != 0) goto EXIT;
	
	//ゲームコントローラー更新をシーンに通知
	result = [self scene_OnGameControllerChanged];
	if (result != 0) goto EXIT;
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ゲームコントローラー初期化
//******************************************************************************
- (int)initializeGameController
{
	int result = 0;
	GCController* pController = nil;
	
	//ゲームパッド制御初期化
	result = m_GamePadCtrl.Initialize(0);
	if (result != 0) goto EXIT;
	
	//ゲームコントローラーが存在しなければ何もせず終了
	pController = m_GamePadCtrl.GetController();
	if (pController == nil) goto EXIT;
	
	//拡張ゲームパッドでない場合は何もせず終了
	if (pController.extendedGamepad == nil) goto EXIT;
	
	//ポーズボタン：演奏開始／一時停止
	pController.controllerPausedHandler
	= ^(GCController* controller) {
		[self onPlayButton];
	};
	//Aボタン：演奏開始／一時停止
	pController.extendedGamepad.buttonA.valueChangedHandler
	= ^(GCControllerButtonInput* pButton, float value, BOOL isPressed) {
		m_GamePadCtrl.UpdateState();
		if (m_GamePadCtrl.DidPressNow_A()){
			[self onPlayButton];
		}
	};
	//Bボタン：演奏停止
	pController.extendedGamepad.buttonB.valueChangedHandler
	= ^(GCControllerButtonInput* pButton, float value, BOOL isPressed) {
		m_GamePadCtrl.UpdateState();
		if (m_GamePadCtrl.DidPressNow_B()){
			[self onStopButton];
		}
	};
	//L1 ショルダーボタン：視点切り替え
	pController.extendedGamepad.leftShoulder.valueChangedHandler
	= ^(GCControllerButtonInput* pButtonInput, float value, BOOL isPressed) {
		m_GamePadCtrl.UpdateState();
		if (m_GamePadCtrl.DidPressNow_LShoulder()){
			[self changeViewPoint:-1];
		}
	};
	//R1 ショルダーボタン：視点切り替え
	pController.extendedGamepad.rightShoulder.valueChangedHandler
	= ^(GCControllerButtonInput* pButtonInput, float value, BOOL isPressed) {
		m_GamePadCtrl.UpdateState();
		if (m_GamePadCtrl.DidPressNow_RShoulder()){
			[self changeViewPoint:+1];
		}
	};
	//L2 トリガーボタン：再生リワインド
	pController.extendedGamepad.leftTrigger.valueChangedHandler
	= ^(GCControllerButtonInput* pButtonInput, float value, BOOL isPressed) {
		m_GamePadCtrl.UpdateState();
		if (m_GamePadCtrl.DidPressNow_LTrigger()){
			[self onSkipBackwardButton];
		}
	};
	//R2 トリガーボタン：再生スキップ
	pController.extendedGamepad.rightTrigger.valueChangedHandler
	= ^(GCControllerButtonInput* pButtonInput, float value, BOOL isPressed) {
		m_GamePadCtrl.UpdateState();
		if (m_GamePadCtrl.DidPressNow_RTrigger()){
			[self onSkipForwardButton];
		}
	};
	
EXIT:;
	return result;
}

//******************************************************************************
// ゲームコントローラーボタンイベントに関するメモ
//******************************************************************************
// ボタンを1回押すだけでもvalueの異なる複数の通知が連続発生する
// ボタンを半押しで押し続けるとvalueが異なる通知が発生し続ける
// ボタンを全押しするとvalueが1.0となり通知が止まる
// ボタンを離すとvalueが0.0の通知が発生し、次にボタンを押すまで通知が発生しない

//******************************************************************************
// 視点切り替え
//******************************************************************************
- (void)changeViewPoint:(int)step
{
	//ゲームパッド用視点番号更新
	m_GamePadViewPointNo += step;
	
	if (m_GamePadViewPointNo < 0) {
		m_GamePadViewPointNo = 2;
	}
	else if (m_GamePadViewPointNo > 2) {
		m_GamePadViewPointNo = 0;
	}
	
	//視点切り替え
	switch (m_GamePadViewPointNo) {
		case 0:
			[self scene_ResetViewpoint];
			break;
		case 1:
			[self scene_MoveToStaticViewpoint:2];
			break;
		case 2:
			[self scene_MoveToStaticViewpoint:3];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// バックグラウンド状態設定
//******************************************************************************
- (void)setBackground:(bool)isBackground
{
	m_isBackground = isBackground;
}

//******************************************************************************
// リワインド／スキップ設定
//******************************************************************************
- (void)setSkipTimeSpanWithBack:(int)backTimeSpanInMsec
						 foward:(int)forwardTimeSpanInMSec
{
	m_SkipBackTimeSpanInMsec = backTimeSpanInMsec;
	m_SkipForwardTimeSpanInMsec = forwardTimeSpanInMSec;
}

//******************************************************************************
// リモートコマンドハンドラ設定
//******************************************************************************
- (void)setRemoteCommandHandlerWithMonitor:(BOOL)isMonitor
{
	MPRemoteCommandCenter* pCommandCenter = nil;
	bool isEnablePlayPause = true;
	
	pCommandCenter = [MPRemoteCommandCenter sharedCommandCenter];
	
	//MIDI IN モニタの場合は再生と一時停止は操作不可
	if (isMonitor) {
		isEnablePlayPause = false;
	}
	
	//対応コマンド
	pCommandCenter.playCommand.enabled = isEnablePlayPause;				//再生
	pCommandCenter.pauseCommand.enabled = isEnablePlayPause;			//一時停止
	pCommandCenter.togglePlayPauseCommand.enabled = isEnablePlayPause;	//再生/一時停止：ヘッドホンセンターボタン
	pCommandCenter.stopCommand.enabled = isEnablePlayPause;				//停止
	pCommandCenter.skipBackwardCommand.enabled = isEnablePlayPause;		//後方スキップ
	pCommandCenter.skipForwardCommand.enabled = isEnablePlayPause;		//前方スキップ
	
	//スキップ時間間隔
	[pCommandCenter.skipBackwardCommand setPreferredIntervals:@[@(m_SkipBackTimeSpanInMsec/1000)]];
	[pCommandCenter.skipForwardCommand setPreferredIntervals:@[@(m_SkipForwardTimeSpanInMsec/1000)]];
	
	//非対応コマンド
	pCommandCenter.seekBackwardCommand.enabled = false;					//後方シーク
	pCommandCenter.seekForwardCommand.enabled = false;					//前方シーク
	pCommandCenter.changePlaybackPositionCommand.enabled = false;		//再生位置
	pCommandCenter.changePlaybackRateCommand.enabled = false;			//再生速度
	pCommandCenter.previousTrackCommand.enabled = false;				//前トラック
	pCommandCenter.nextTrackCommand.enabled = false;					//次トラック
	
	//コマンドハンドラ登録
    [pCommandCenter.playCommand addTarget:self action:@selector(onPlayCommand:)];
    [pCommandCenter.pauseCommand addTarget:self action:@selector(onPauseCommand:)];
	[pCommandCenter.togglePlayPauseCommand addTarget:self action:@selector(onTogglePlayPauseCommand:)];
	[pCommandCenter.stopCommand addTarget:self action:@selector(onStopCommand:)];
	[pCommandCenter.skipBackwardCommand addTarget:self action:@selector(onSkipBackwardCommand:)];
	[pCommandCenter.skipForwardCommand addTarget:self action:@selector(onSkipForwardCommand:)];
	
	//非対応
	//[pCommandCenter.seekBackwardCommand addTarget:self action:@selector(onSeekBackwardCommand:)];
	//[pCommandCenter.seekForwardCommand addTarget:self action:@selector(onSeekForwardCommand:)];
	//[pCommandCenter.changePlaybackPositionCommand addTarget:self action:@selector(onChangePlaybackPositionCommand:)];
	//[pCommandCenter.changePlaybackRateCommand addTarget:self action:@selector(onChangePlaybackRateCommand:)];
	//[pCommandCenter.changePlaybackRateCommand addTarget:self action:@selector(onChangePlaybackRateCommand:)];
	//[pCommandCenter.previousTrackCommand addTarget:self action:@selector(onPreviousTrackCommand:)];
	//[pCommandCenter.nextTrackCommand addTarget:self action:@selector(onNextTrackCommand:)];
	
	return;
}

//******************************************************************************
// リモートコマンドハンドラ：再生
//******************************************************************************
- (MPRemoteCommandHandlerStatus)onPlayCommand:(MPRemoteCommandEvent*)pCommandEvent
{
	NSLog(@"onPlayCommand");
	
	[self onPlayButton];
	
	return MPRemoteCommandHandlerStatusSuccess;
}

//******************************************************************************
// リモートコマンドハンドラ：一時停止
//******************************************************************************
- (MPRemoteCommandHandlerStatus)onPauseCommand:(MPRemoteCommandEvent*)pCommandEvent
{
	NSLog(@"onPauseCommand");
	
	[self onPlayButton];
	
	return MPRemoteCommandHandlerStatusSuccess;
}

//******************************************************************************
// リモートコマンドハンドラ：再生/一時停止トグル
//******************************************************************************
- (MPRemoteCommandHandlerStatus)onTogglePlayPauseCommand:(MPRemoteCommandEvent*)pCommandEvent
{
	NSLog(@"onTogglePlayPauseCommand");
	
	[self onPlayButton];
	
	return MPRemoteCommandHandlerStatusSuccess;
}

//******************************************************************************
// リモートコマンドハンドラ：停止
//******************************************************************************
- (MPRemoteCommandHandlerStatus)onStopCommand:(MPRemoteCommandEvent*)pCommandEvent
{
	NSLog(@"onStopCommand");
	
	[self onStopButton];
	
	return MPRemoteCommandHandlerStatusSuccess;
}

//******************************************************************************
// リモートコマンドハンドラ：後方スキップ
//******************************************************************************
- (MPRemoteCommandHandlerStatus)onSkipBackwardCommand:(MPRemoteCommandEvent*)pCommandEvent
{
	NSLog(@"onSkipBackwardCommand");
	
	[self onSkipBackwardButton];
	
	return MPRemoteCommandHandlerStatusSuccess;
}

//******************************************************************************
// リモートコマンドハンドラ：前方スキップ
//******************************************************************************
- (MPRemoteCommandHandlerStatus)onSkipForwardCommand:(MPRemoteCommandEvent*)pCommandEvent
{
	NSLog(@"onSkipForwardCommand");
	
	[self onSkipForwardButton];
	
	return MPRemoteCommandHandlerStatusSuccess;
}

//******************************************************************************
// 演奏情報更新：演奏時間
//******************************************************************************
- (int)setNowPlayingInfoPlaybackTime:(unsigned int)playTimeSec
{
	int result = 0;
	NSMutableDictionary* pNowPlayingInfo = nil;
	MPNowPlayingInfoCenter* pNowPlayingInfoCenter = nil;
		
	//現在の演奏情報を取得
	pNowPlayingInfoCenter = [MPNowPlayingInfoCenter defaultCenter];
	pNowPlayingInfo = [[NSMutableDictionary alloc] initWithDictionary:pNowPlayingInfoCenter.nowPlayingInfo];
	
	//経過時間
	[pNowPlayingInfo setObject:[NSNumber numberWithInt:playTimeSec] forKey:MPNowPlayingInfoPropertyElapsedPlaybackTime];
	
	//演奏情報を更新
	[pNowPlayingInfoCenter setNowPlayingInfo:pNowPlayingInfo];
	
	[pNowPlayingInfo release];
	return result;
}

//******************************************************************************
// サブビュー表示状態取得
//******************************************************************************
- (bool)isSubViewDisplayed
{
	bool isDisplayed = false;
	
	//HowToViewヘルプ
	if (m_pHowToViewCtrl.isViewLoaded && (m_pHowToViewCtrl.view.window != nil)) {
		isDisplayed = true;
	}
	
	//Viewアクションビュー
	if (m_isExistViewActionView) {
		isDisplayed = true;
	}
	
	return isDisplayed;
}

//******************************************************************************
// スクリーンゲームパッドイベント：ボタン上
//******************************************************************************
- (IBAction)onScreenGamePad_DPadUp_TouchDown:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadUp(1.0f);
}
- (IBAction)onScreenGamePad_DPadUp_TouchCancel:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadUp(0.0f);
}
- (IBAction)onScreenGamePad_DPadUp_TouchUpInside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadUp(0.0f);
}
- (IBAction)onScreenGamePad_DPadUp_TouchUpOutside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadUp(0.0f);
}

//******************************************************************************
// スクリーンゲームパッドイベント：ボタン下
//******************************************************************************
- (IBAction)onScreenGamePad_DPadDown_TouchDown:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadDown(1.0f);
}
- (IBAction)onScreenGamePad_DPadDown_TouchCancel:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadDown(0.0f);
}
- (IBAction)onScreenGamePad_DPadDown_TouchUpInside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadDown(0.0f);
}
- (IBAction)onScreenGamePad_DPadDown_TouchUpOutside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadDown(0.0f);
}

//******************************************************************************
// スクリーンゲームパッドイベント：ボタン左
//******************************************************************************
- (IBAction)onScreenGamePad_DPadLeft_TouchDown:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadLeft(1.0f);
}
- (IBAction)onScreenGamePad_DPadLeft_TouchCancel:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadLeft(0.0f);
}
- (IBAction)onScreenGamePad_DPadLeft_TouchUpInside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadLeft(0.0f);
}
- (IBAction)onScreenGamePad_DPadLeft_TouchUpOutside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadLeft(0.0f);
}

//******************************************************************************
// スクリーンゲームパッドイベント：ボタン右
//******************************************************************************
- (IBAction)onScreenGamePad_DPadRight_TouchDown:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadRight(1.0f);
}
- (IBAction)onScreenGamePad_DPadRight_TouchCancel:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadRight(0.0f);
}
- (IBAction)onScreenGamePad_DPadRight_TouchUpInside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadRight(0.0f);
}
- (IBAction)onScreenGamePad_DPadRight_TouchUpOutside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_DPadRight(0.0f);
}

//******************************************************************************
// スクリーンゲームパッドイベント：ボタンX
//******************************************************************************
- (IBAction)onScreenGamePad_X_TouchDown:(id)sender
{
	m_ScreenGamePadCtrl.SetState_X(1.0f);
}
- (IBAction)onScreenGamePad_X_TouchCancel:(id)sender
{
	m_ScreenGamePadCtrl.SetState_X(0.0f);
}
- (IBAction)onScreenGamePad_X_TouchUpInside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_X(0.0f);
}
- (IBAction)onScreenGamePad_X_TouchUpOutside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_X(0.0f);
}

//******************************************************************************
// スクリーンゲームパッドイベント：ボタンY
//******************************************************************************
- (IBAction)onScreenGamePad_Y_TouchDown:(id)sender
{
	m_ScreenGamePadCtrl.SetState_Y(1.0f);
}
- (IBAction)onScreenGamePad_Y_TouchCancel:(id)sender
{
	m_ScreenGamePadCtrl.SetState_Y(0.0f);
}
- (IBAction)onScreenGamePad_Y_TouchUpInside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_Y(0.0f);
}
- (IBAction)onScreenGamePad_Y_TouchUpOutside:(id)sender
{
	m_ScreenGamePadCtrl.SetState_Y(0.0f);
}


@end

