//******************************************************************************
//
// MIDITrail / MTNoteDesign
//
// ノートデザインクラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

// MEMO:
// ノートボックスの正面を0-1-2-3の四角とする。
// この四角の中心座標が基準点であり、ポート／チャンネル／ノートの番号
// によって決定される。
//
//           +--+
//          /  /|
//         /  / +
//        /  / /      +x
//       /  / /      /
//     0+--+1/ +y   /
//      |  |/   |  /
//     2+--+3   | /
//              |/
//   +z---------+0
//

#import "OGLUtil.h"
#import "SMIDILib.h"


//******************************************************************************
// ノートデザインクラス
//******************************************************************************
class MTNoteDesign
{
public:
	
	//コンストラクタ／デストラクタ
	MTNoteDesign(void);
	virtual ~MTNoteDesign(void);
	
	//初期化
	int Initialize(NSString* pSceneName, SMSeqData* pSeqData);
	
	//演奏位置取得
	float GetPlayPosX(unsigned int curTickTime);
	
	//ライブモニタ用ノート位置取得
	float GetLivePosX(unsigned int elapsedTime);
	
	//ノートボックス中心座標取得
	virtual OGLVECTOR3 GetNoteBoxCenterPosX(
				unsigned int curTickTime,
				unsigned char portNo,
				unsigned char chNo,
				unsigned char noteNo,
				short pitchBendValue = 0,				//省略可：ピッチベンド
				unsigned char pitchBendSensitivity = 0	//省略可：ピッチベンド感度
			);
	
	//ノートボックス縦横サイズ取得
	float GetNoteBoxHeight();
	float GetNoteBoxWidth();
	
	//ノート間隔取得
	float GetNoteStep();
	
	//チャンネル間隔取得
	float GetChStep();
	
	//ライブモニタ表示期限
	unsigned int GetLiveMonitorDisplayDuration();
	
	//ノートボックス頂点座標取得
	virtual void GetNoteBoxVirtexPos(
				unsigned int curTickTime,
				unsigned char portNo,
				unsigned char chNo,
				unsigned char noteNo,
				OGLVECTOR3* pVector0,	//YZ平面+X軸方向を見て左上
				OGLVECTOR3* pVector1,	//YZ平面+X軸方向を見て右上
				OGLVECTOR3* pVector2,	//YZ平面+X軸方向を見て左下
				OGLVECTOR3* pVector3,	//YZ平面+X軸方向を見て右下
				short pitchBendValue = 0,				//省略可：ピッチベンド
				unsigned char pitchBendSensitivity = 0	//省略可：ピッチベンド感度
			);
	
	//発音中ノートボックス頂点座標取得
	virtual void GetActiveNoteBoxVirtexPos(
				unsigned int curTickTime,
				unsigned char portNo,
				unsigned char chNo,
				unsigned char noteNo,
				OGLVECTOR3* pVector0,	//YZ平面+X軸方向を見て左上
				OGLVECTOR3* pVector1,	//YZ平面+X軸方向を見て右上
				OGLVECTOR3* pVector2,	//YZ平面+X軸方向を見て左下
				OGLVECTOR3* pVector3,	//YZ平面+X軸方向を見て右下
				short pitchBendValue = 0,				//省略可：ピッチベンド
				unsigned char pitchBendSensitivity = 0,	//省略可：ピッチベンド感度
				unsigned int elapsedTime = 0            //省略可：経過時間（ミリ秒）
		);
	
	//ライブモニタ用ノートボックス頂点座標取得
	virtual void GetNoteBoxVirtexPosLive(
				unsigned int elapsedTime,	//経過時間（ミリ秒）
				unsigned char portNo,
				unsigned char chNo,
				unsigned char noteNo,
				OGLVECTOR3* pVector0,	//YZ平面+X軸方向を見て左上
				OGLVECTOR3* pVector1,	//YZ平面+X軸方向を見て右上
				OGLVECTOR3* pVector2,	//YZ平面+X軸方向を見て左下
				OGLVECTOR3* pVector3,	//YZ平面+X軸方向を見て右下
				short pitchBendValue = 0,				//省略可：ピッチベンド
				unsigned char pitchBendSensitivity = 0	//省略可：ピッチベンド感度
			);
	
	//グリッドボックス頂点座標取得
	void GetGridBoxVirtexPos(
				unsigned int curTickTime,
				unsigned char portNo,
				OGLVECTOR3* pVector0,	//YZ平面+X軸方向を見て左上
				OGLVECTOR3* pVector1,	//YZ平面+X軸方向を見て右上
				OGLVECTOR3* pVector2,	//YZ平面+X軸方向を見て左下
				OGLVECTOR3* pVector3 	//YZ平面+X軸方向を見て右下
			);
	
	//ライブモニタ用グリッドボックス頂点座標取得
	void GetGridBoxVirtexPosLive(
				unsigned int elapsedTime,	//経過時間（ミリ秒）
				unsigned char portNo,	//ポート番号
				OGLVECTOR3* pVector0,	//YZ平面+X軸方向を見て左上
				OGLVECTOR3* pVector1,	//YZ平面+X軸方向を見て右上
				OGLVECTOR3* pVector2,	//YZ平面+X軸方向を見て左下
				OGLVECTOR3* pVector3 	//YZ平面+X軸方向を見て右下
			);
	
	//再生面頂点座標取得
	void GetPlaybackSectionVirtexPos(
				unsigned int curTickTime,
				OGLVECTOR3* pVector0,	//YZ平面+X軸方向を見て左上
				OGLVECTOR3* pVector1,	//YZ平面+X軸方向を見て右上
				OGLVECTOR3* pVector2,	//YZ平面+X軸方向を見て左下
				OGLVECTOR3* pVector3 	//YZ平面+X軸方向を見て右下
			);
	
	//波紋サイズ取得：経過時間（ミリ秒）は省略可
	float GetRippleHeight(unsigned int elapsedTime = 0);
	float GetRippleWidth(unsigned int elapsedTime = 0);
	float GetRippleAlpha(unsigned int elapsedTime = 0);
	
	//ピクチャボード相対位置取得
	float GetPictBoardRelativePos();
	
	//ポート原点座標取得
	virtual float GetPortOriginY(unsigned char portNo);
	virtual float GetPortOriginZ(unsigned char portNo);
	
	//世界座標配置移動ベクトル取得
	virtual OGLVECTOR3 GetWorldMoveVector();
	virtual OGLVECTOR3 GetWorldMoveVector2(unsigned int curTickTime);
	
	//ノートボックスカラー取得
	OGLCOLOR GetNoteBoxColor(
				unsigned char portNo,
				unsigned char chNo,
				unsigned char noteNo
			);
	
	//発音中ノートボックスカラー取得
	OGLCOLOR GetActiveNoteBoxColor(
				unsigned char portNo,
				unsigned char chNo,
				unsigned char noteNo,
				unsigned int elapsedTime
			);
	
	//発音中ノートボックスエミッシブ取得（マテリアル用）
	OGLCOLOR GetActiveNoteEmissive();
	
	//グリッドラインカラー取得
	OGLCOLOR GetGridLineColor();
	
	//再生面カラー取得
	OGLCOLOR GetPlaybackSectionColor();
	
protected:
	
	unsigned int m_TimeDivision;
	float m_QuarterNoteLength;
	float m_NoteBoxHeight;
	float m_NoteBoxWidth;
	float m_NoteStep;
	float m_ChStep;
	float m_RippleHeight;
	float m_RippleWidth;
	float m_PictBoardRelativePos;
	SMPortList m_PortList;
	unsigned char m_PortIndex[256];
	
	OGLCOLOR m_NoteColor[16];
	OGLCOLOR m_ActiveNoteEmissive;
	OGLCOLOR m_GridLineColor;
	OGLCOLOR m_PlaybackSectionColor;
	
	int m_ActiveNoteDuration;
	float m_ActiveNoteWhiteRate;
	float m_ActiveNoteBoxSizeRatio;
	
	int m_RippleDuration;
	
	int m_LiveMonitorDisplayDuration;
	float m_LiveNoteLengthPerSecond;
	
	void _Clear();
	virtual int _LoadConfFile(NSString* pSceneName);
	int _LoadUserConf();
	
};


