//******************************************************************************
//
// MIDITrail / MTNoteRainLive
//
// ライブモニタ用ノートレイン描画クラス
//
// Copyright (C) 2012-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

// MEMO:
// Windows版のソースを移植しているため、座標は左手系(DirectX)で処理している。
// 左手系(DirectX,Metal)=>右手系(OpenGL)への変換は LH2RH マクロで実現する。

#import "YNBaseLib.h"
#import "MTNoteRainLive.h"


//******************************************************************************
// パラメータ定義
//******************************************************************************
//1ノートあたりの頂点数 = 1長方形4頂点 * 1面
#define NOTE_VERTEX_NUM  (4 * 1)

//1ノートあたりのインデックス数 = 1三角形3頂点 * 2個 * 1面
#define NOTE_INDEX_NUM   (3 * 2 * 1)

//******************************************************************************
// コンストラクタ
//******************************************************************************
MTNoteRainLive::MTNoteRainLive(void)
{
	m_NoteNum = 0;
	m_pNoteStatus = NULL;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTNoteRainLive::~MTNoteRainLive(void)
{
	Release();
}

//******************************************************************************
// 生成処理
//******************************************************************************
int MTNoteRainLive::Create(
		OGLDevice* pOGLDevice,
		NSString* pSceneName,
		MTNotePitchBend* pNotePitchBend
	)
{
	int result = 0;
	
	Release();
	
	//ノートデザインオブジェクト初期化
	result = m_NoteDesign.Initialize(pSceneName, NULL);
	if (result != 0) goto EXIT;
	
	//キーボードデザインオブジェクト初期化
	result = m_KeyboardDesign.Initialize(pSceneName, NULL);
	if (result != 0) goto EXIT;
	
	//ライブモニタ表示期限
	m_LiveMonitorDisplayDuration = m_NoteDesign.GetLiveMonitorDisplayDuration();
	
	//ノート情報配列生成
	result = _CreateNoteStatus();
	if (result != 0) goto EXIT;
	
	//ノートボックス生成（バッファ確保）
	result = _CreateNoteRain(pOGLDevice);
	if (result != 0) goto EXIT;
	
	//ピッチベンド情報
	m_pNotePitchBend = pNotePitchBend;
	
	//Mach時間
	m_MachTime.Initialize();
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// ノート情報配列生成
//******************************************************************************
int MTNoteRainLive::_CreateNoteStatus()
{
	int result = 0;
	unsigned int i = 0;
	
	//ノート情報配列生成
	try {
		m_pNoteStatus = new NoteStatus[MTNOTERAIN_MAX_LIVENOTE_NUM];
	}
	catch (std::bad_alloc) {
		result = YN_SET_ERR(@"Could not allocate memory.", 0, 0);
		goto EXIT;
	}
	
	//ノート状態リスト初期化
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		m_pNoteStatus[i].isActive = false;
		m_pNoteStatus[i].portNo = 0;
		m_pNoteStatus[i].chNo = 0;
		m_pNoteStatus[i].noteNo = 0;
		m_pNoteStatus[i].startTime = 0;
		m_pNoteStatus[i].endTime = 0;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// 発音中ノートレイン生成（バッファ確保）
//******************************************************************************
int MTNoteRainLive::_CreateNoteRain(
		OGLDevice* pOGLDevice
   )
{
	int result = 0;
	OGLsizei vertexNum = 0;
	OGLsizei indexNum = 0;
	MTNOTERAIN_VERTEX* pVertex = NULL;
	unsigned int* pIndex = NULL;
	unsigned int i = 0;
	OGLMATERIAL material;
	NoteStatus note;
	
	memset(&note, 0, sizeof(NoteStatus));
	
	//プリミティブ初期化
	result = m_PrimitiveNotes.Initialize(
					pOGLDevice,					//描画デバイス
					sizeof(MTNOTERAIN_VERTEX),	//頂点サイズ
					_GetFVFFormat(),			//頂点FVFフォーマット
					OGL_TRIANGLES				//プリミティブ種別
				);
	if (result != 0) goto EXIT;
	
	//頂点バッファ生成
	vertexNum = NOTE_VERTEX_NUM * MTNOTERAIN_MAX_LIVENOTE_NUM;
	result = m_PrimitiveNotes.CreateVertexBuffer(pOGLDevice, vertexNum);
	if (result != 0) goto EXIT;
	
	//インデックスバッファ生成
	indexNum = NOTE_INDEX_NUM * MTNOTERAIN_MAX_LIVENOTE_NUM;
	result = m_PrimitiveNotes.CreateIndexBuffer(pOGLDevice, indexNum);
	if (result != 0) goto EXIT;
	
	//バッファのロック
	result = m_PrimitiveNotes.LockVertex((void**)&pVertex);
	if (result != 0) goto EXIT;
	result = m_PrimitiveNotes.LockIndex(&pIndex);
	if (result != 0) goto EXIT;
	
	//バッファに頂点とインデックスを書き込む
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		result = _CreateVertexOfNote(
						m_pNoteStatus[i],				//ノート状態
						&(pVertex[NOTE_VERTEX_NUM * i]),//頂点バッファ書き込み位置
						NOTE_VERTEX_NUM * i,			//頂点バッファインデックスオフセット
						&(pIndex[NOTE_INDEX_NUM * i]),	//インデックスバッファ書き込み位置
						0								//現在時刻
					);
		if (result != 0) goto EXIT;
	}
	
	//バッファのロック解除
	result = m_PrimitiveNotes.UnlockVertex();
	if (result != 0) goto EXIT;
	result = m_PrimitiveNotes.UnlockIndex();
	if (result != 0) goto EXIT;
	
	//マテリアル作成
	_MakeMaterial(&material);
	m_PrimitiveNotes.SetMaterial(material);
	
EXIT:;
	return result;
}

//******************************************************************************
// 移動
//******************************************************************************
int MTNoteRainLive::Transform(
		OGLDevice* pOGLDevice,
		float rollAngle
	)
{
	int result = 0;
	OGLVECTOR3 moveVector;
	OGLTransMatrix transMatrix;
	
	//ノートの頂点生成
	result = _TransformNotes(pOGLDevice);
	if (result != 0) goto EXIT;
		
	//移動行列
	//  ノートを移動させる場合
	moveVector = OGLVECTOR3(0.0f, 0, 0.0f);
	//  ノートを移動させずにカメラとキーボードを移動させる場合
	//moveVector = OGLVECTOR3(0.0f, 0.0f, 0.0f);
	transMatrix.RegistTranslation(moveVector.x, moveVector.y, moveVector.z);
	
	//回転行列
	transMatrix.RegistRotationY(rollAngle);
	
	//変換行列設定
	m_PrimitiveNotes.Transform(&transMatrix);
	
EXIT:;
	return result;
}

//******************************************************************************
// ノートの頂点処理
//******************************************************************************
int MTNoteRainLive::_TransformNotes(
		OGLDevice* pOGLDevice
	)
{
	int result = 0;
	
	//ノートの状態更新
	result = _UpdateStatusOfNotes(pOGLDevice);
	if (result != 0) goto EXIT;
	
	//ノートの頂点更新
	result = _UpdateVertexOfNotes(pOGLDevice);
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// ノートの状態更新
//******************************************************************************
int MTNoteRainLive::_UpdateStatusOfNotes(
		OGLDevice* pOGLDevice
	)
{
	int result = 0;
	unsigned int i = 0;
	uint64_t curTime = 0;
	
	curTime = m_MachTime.GetCurTimeInMsec();
	
	//古いノート情報を破棄する
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		if (m_pNoteStatus[i].isActive) {
			if ((m_pNoteStatus[i].endTime != 0)
				&& ((curTime - m_pNoteStatus[i].endTime) > m_LiveMonitorDisplayDuration)) {
				m_pNoteStatus[i].isActive = false;
				m_pNoteStatus[i].portNo = 0;
				m_pNoteStatus[i].chNo = 0;
				m_pNoteStatus[i].noteNo = 0;
				m_pNoteStatus[i].startTime = 0;
				m_pNoteStatus[i].endTime = 0;
			}
		}
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// ノートの頂点更新
//******************************************************************************
int MTNoteRainLive::_UpdateVertexOfNotes(
		OGLDevice* pOGLDevice
	)
{
	int result = 0;
	unsigned int i = 0;
	OGLsizei noteNum = 0;
	uint64_t curTime = 0;
	MTNOTERAIN_VERTEX* pVertex = NULL;
	unsigned int* pIndex = NULL;
	
	curTime = m_MachTime.GetCurTimeInMsec();
	
	//バッファのロック
	result = m_PrimitiveNotes.LockVertex((void**)&pVertex);
	if (result != 0) goto EXIT;
	result = m_PrimitiveNotes.LockIndex(&pIndex);
	if (result != 0) goto EXIT;
	
	//発音中ノートについて頂点を更新
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		if (m_pNoteStatus[i].isActive) {
			//頂点更新
			result = _CreateVertexOfNote(
							m_pNoteStatus[i],						//ノート状態
							&(pVertex[NOTE_VERTEX_NUM * noteNum]),	//頂点バッファ書き込み位置
							NOTE_VERTEX_NUM * noteNum,				//頂点バッファインデックスオフセット
							&(pIndex[NOTE_INDEX_NUM * noteNum]),	//インデックスバッファ書き込み位置
							(unsigned int)curTime,					//現在の時間
							true									//ピッチベンド適用
						);
			if (result != 0) goto EXIT;
			
			noteNum++;
		}
	}
	m_NoteNum = noteNum;
	
	//バッファのロック解除
	result = m_PrimitiveNotes.UnlockVertex();
	if (result != 0) goto EXIT;
	result = m_PrimitiveNotes.UnlockIndex();
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// 描画
//******************************************************************************
int MTNoteRainLive::Draw(
		OGLDevice* pOGLDevice
   )
{
	int result = 0;
	
	//発音中ノートの描画
	if (m_NoteNum > 0) {
		result = m_PrimitiveNotes.Draw(
						pOGLDevice,						//デバイス
						NULL,							//テクスチャ：なし
						//(NOTE_INDEX_NUM/3)*m_NoteNum	//プリミティブ数
						NOTE_INDEX_NUM * m_NoteNum		//インデックス数
					);
		if (result != 0) goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// 解放
//******************************************************************************
void MTNoteRainLive::Release()
{
	m_PrimitiveNotes.Release();
	
	delete [] m_pNoteStatus;
	m_pNoteStatus = NULL;
}

//******************************************************************************
// ノートレインの頂点生成
//******************************************************************************
int MTNoteRainLive::_CreateVertexOfNote(
		NoteStatus note,
		MTNOTERAIN_VERTEX* pVertex,
		unsigned int vertexOffset,
		unsigned int* pIndex,
		unsigned int curTime,
		bool isEnablePitchBend
	)
{
	int result = 0;
	OGLVECTOR3 startVector;
	OGLVECTOR3 endVector;
	OGLVECTOR3 moveVector;
	OGLCOLOR color;
	float rainWidth = m_KeyboardDesign.GetBlackKeyWidth();
	float pitchBendShift = 0.0f;
	short pitchBendValue = 0;
	unsigned char pitchBendSensitivity = SM_DEFAULT_PITCHBEND_SENSITIVITY;
	unsigned int elapsedTime = 0;
	
	if ((isEnablePitchBend) && (note.endTime == 0)) {
		pitchBendValue =       m_pNotePitchBend->GetValue(note.portNo, note.chNo);
		pitchBendSensitivity = m_pNotePitchBend->GetSensitivity(note.portNo, note.chNo);
		pitchBendShift = m_KeyboardDesign.GetPitchBendShift(pitchBendValue, pitchBendSensitivity);
	}
	
	//ノートON座標
	elapsedTime = (unsigned int)(curTime - note.startTime);
	if (elapsedTime > m_LiveMonitorDisplayDuration) {
		elapsedTime = m_LiveMonitorDisplayDuration;
	}
	startVector.x = pitchBendShift;
	startVector.y = m_NoteDesign.GetLivePosX(elapsedTime);
	startVector.z = 0.0f;
	
	//ノートOFF座標
	elapsedTime = 0;
	if (note.endTime != 0) {
		elapsedTime = (unsigned int)(curTime - note.endTime);
	}
	endVector.x = pitchBendShift;
	endVector.y = m_NoteDesign.GetLivePosX(elapsedTime);
	endVector.z = 0.0f;
	
	//移動ベクトル
	moveVector    = m_KeyboardDesign.GetKeyboardBasePos(note.portNo, note.chNo);
	moveVector.x += m_KeyboardDesign.GetKeyCenterPosX(note.noteNo);
	moveVector.y += m_KeyboardDesign.GetWhiteKeyHeight() / 2.0f;
	moveVector.z += m_KeyboardDesign.GetNoteDropPosZ(note.noteNo);
	
	//座標更新
	startVector = startVector + moveVector;
	endVector   = endVector   + moveVector;
	
	//頂点
	pVertex[0].p = OGLVECTOR3(startVector.x - rainWidth/2.0f, startVector.y, startVector.z);
	pVertex[1].p = OGLVECTOR3(startVector.x + rainWidth/2.0f, startVector.y, startVector.z);
	pVertex[2].p = OGLVECTOR3(endVector.x   + rainWidth/2.0f, endVector.y,   endVector.z);
	pVertex[3].p = OGLVECTOR3(endVector.x   - rainWidth/2.0f, endVector.y,   endVector.z);
	
	//法線
	//実際の面の方向に合わせて(0,0,-1)とするとライトを適用したときに暗くなる
	//鍵盤の上面に合わせることで明るくする
	pVertex[0].n = OGLVECTOR3(0.0f, 1.0f, 0.0f);
	pVertex[1].n = OGLVECTOR3(0.0f, 1.0f, 0.0f);
	pVertex[2].n = OGLVECTOR3(0.0f, 1.0f, 0.0f);
	pVertex[3].n = OGLVECTOR3(0.0f, 1.0f, 0.0f);
	
	//色
	color = m_NoteDesign.GetNoteBoxColor(note.portNo, note.chNo, note.noteNo);
	pVertex[0].c = OGLCOLOR(color.r, color.g, color.b, 1.0f);
	pVertex[1].c = OGLCOLOR(color.r, color.g, color.b, 1.0f);
	pVertex[2].c = OGLCOLOR(color.r, color.g, color.b, 0.5f); //ノートOFFに近づくほど半透明にする
	pVertex[3].c = OGLCOLOR(color.r, color.g, color.b, 0.5f); //ノートOFFに近づくほど半透明にする
	
	//インデックス
	pIndex[0] = vertexOffset + 0;
	pIndex[1] = vertexOffset + 2;
	pIndex[2] = vertexOffset + 1;
	pIndex[3] = vertexOffset + 0;
	pIndex[4] = vertexOffset + 3;
	pIndex[5] = vertexOffset + 2;
	
	return result;
}

//******************************************************************************
// マテリアル作成
//******************************************************************************
void MTNoteRainLive::_MakeMaterial(
		OGLMATERIAL* pMaterial
	)
{
	memset(pMaterial, 0, sizeof(OGLMATERIAL));
	
	//拡散光
	pMaterial->Diffuse.r = 1.0f;
	pMaterial->Diffuse.g = 1.0f;
	pMaterial->Diffuse.b = 1.0f;
	pMaterial->Diffuse.a = 1.0f;
	//環境光：影の色
	pMaterial->Ambient.r = 0.5f;
	pMaterial->Ambient.g = 0.5f;
	pMaterial->Ambient.b = 0.5f;
	pMaterial->Ambient.a = 1.0f;
	//鏡面反射光
	pMaterial->Specular.r = 0.2f;
	pMaterial->Specular.g = 0.2f;
	pMaterial->Specular.b = 0.2f;
	pMaterial->Specular.a = 1.0f;
	//鏡面反射光の鮮明度
	pMaterial->Power = 40.0f;
	//発光色
	pMaterial->Emissive.r = 0.0f;
	pMaterial->Emissive.g = 0.0f;
	pMaterial->Emissive.b = 0.0f;
	pMaterial->Emissive.a = 0.0f;
}

//******************************************************************************
// リセット
//******************************************************************************
void MTNoteRainLive::Reset()
{
	unsigned int i = 0;
	
	m_NoteNum = 0;
	
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		m_pNoteStatus[i].isActive = false;
		m_pNoteStatus[i].portNo = 0;
		m_pNoteStatus[i].chNo = 0;
		m_pNoteStatus[i].noteNo = 0;
		m_pNoteStatus[i].startTime = 0;
		m_pNoteStatus[i].endTime = 0;
	}

	return;
}

//******************************************************************************
// ノートON登録
//******************************************************************************
void MTNoteRainLive::SetNoteOn(
		unsigned char portNo,
		unsigned char chNo,
		unsigned char noteNo,
		unsigned char velocity
	)
{
	unsigned int i = 0;
	unsigned int cleardIndex = 0;
	bool isFind = false;
	uint64_t curTime = 0;
	
	curTime = m_MachTime.GetCurTimeInMsec();
	
	//空きスペースにノート情報を登録
	//空きが見つからなければ登録をあきらめる
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		if (m_pNoteStatus[i].isActive) {
			if ((m_pNoteStatus[i].endTime != 0)
				&& ((curTime - m_pNoteStatus[i].endTime) > m_LiveMonitorDisplayDuration)) {
				m_pNoteStatus[i].isActive = false;
				cleardIndex = i;
				isFind = true;
				break;
			}
		}
		else {
			m_pNoteStatus[i].isActive = false;
			cleardIndex = i;
			isFind = true;
			break;
		}
	}
	
	//空きスペースが見つからない場合は最も古いノート情報をクリアする
	if (!isFind) {
		_ClearOldestNoteStatus(&cleardIndex);
	}
	
	//ノート情報登録
	m_pNoteStatus[cleardIndex].isActive = true;
	m_pNoteStatus[cleardIndex].portNo = portNo;
	m_pNoteStatus[cleardIndex].chNo = chNo;
	m_pNoteStatus[cleardIndex].noteNo = noteNo;
	m_pNoteStatus[cleardIndex].startTime = m_MachTime.GetCurTimeInMsec();
	m_pNoteStatus[cleardIndex].endTime = 0;
	
	return;
}

//******************************************************************************
// ノートOFF登録
//******************************************************************************
void MTNoteRainLive::SetNoteOff(
		unsigned char portNo,
		unsigned char chNo,
		unsigned char noteNo
	)
{
	unsigned int i = 0;
	
	//該当のノート情報に終了時刻を設定
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		if ((m_pNoteStatus[i].isActive)
			&& (m_pNoteStatus[i].portNo == portNo)
			&& (m_pNoteStatus[i].chNo == chNo)
			&& (m_pNoteStatus[i].noteNo == noteNo)
			&& (m_pNoteStatus[i].endTime == 0)) {
			m_pNoteStatus[i].endTime = m_MachTime.GetCurTimeInMsec();
			break;
		}
	}
	
	return;
}

//******************************************************************************
// 全ノートOFF
//******************************************************************************
void MTNoteRainLive::AllNoteOff()
{
	unsigned int i = 0;
	
	//ノートOFFが設定されていないノート情報に終了時刻を設定
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		if ((m_pNoteStatus[i].isActive) && (m_pNoteStatus[i].endTime == 0)) {
			m_pNoteStatus[i].endTime = m_MachTime.GetCurTimeInMsec();
		}
	}
	
	return;
}

//******************************************************************************
// 全ノートOFF（チャンネル指定）
//******************************************************************************
void MTNoteRainLive::AllNoteOffOnCh(
		unsigned char portNo,
		unsigned char chNo
	)
{
	unsigned int i = 0;
	
	//指定チャンネルでノートOFFが設定されていないノート情報に終了時刻を設定
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		if ((m_pNoteStatus[i].isActive) && (m_pNoteStatus[i].endTime == 0)
			&& (m_pNoteStatus[i].portNo == portNo) && (m_pNoteStatus[i].chNo == chNo)) {
			m_pNoteStatus[i].endTime = m_MachTime.GetCurTimeInMsec();
		}
	}
	
	return;
}

//******************************************************************************
// 最も古いノート情報のクリア
//******************************************************************************
void MTNoteRainLive::_ClearOldestNoteStatus(
		unsigned int* pCleardIndex
	)
{
	unsigned int i = 0;
	unsigned int oldestIndex = 0;
	bool isFind = false;
	
	//ノートON時刻が最も古いノート情報をクリアする
	for (i = 0; i < MTNOTERAIN_MAX_LIVENOTE_NUM; i++) {
		if (m_pNoteStatus[i].isActive) {
			if (!isFind) {
				//有効なノートが現れた場合：初回
				oldestIndex = i;
				isFind = true;
			}
			else {
				//有効なノートが現れた場合：2回目以降
				if (m_pNoteStatus[i].startTime < m_pNoteStatus[oldestIndex].startTime) {
					oldestIndex = i;
				}
			}
		}
	}
	
	//ノート情報クリア
	//  ノートが一つも登録されていない場合は配列の先頭がクリア対象となる	
	m_pNoteStatus[oldestIndex].isActive = false;
	*pCleardIndex = oldestIndex;
	
	return;
}

