//******************************************************************************
//
// MIDITrail / MTSettingColorPaletteViewCtrl
//
// カラーパレット設定ビュークラス
//
// Copyright (C) 2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTSettingColorPaletteViewCtrl.h"


//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTSettingColorPaletteViewCtrl ()

//テーブルセル作成：カラーウェル
- (UITableViewCell*)makeColorWellCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル作成：グラデーションツール
- (UITableViewCell*)makeGradationToolCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル作成：パラメータ設定ツール
- (UITableViewCell*)makeParameterSetupToolCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：カラーパレット
- (void)onSelectColorWellCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：グラデーションツール
- (void)onSelectGradationToolCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：パラメータ設定ツール
- (void)onSelectParameterSetupToolCellForIndexPath:(NSIndexPath*)indexPath;

//カラーウェル変更イベント
- (void)onColorWellValueChanged:(id)sender;

//グラデーションツール カラーウェル変更イベント
- (void)onGradationColorWellValueChanged:(id)sender;

//Set Gradation Colors 選択イベント
- (void)onSetGradationColors;

//Set Gradation Colors 確認Yesボタン押下
- (void)onAlertSetGradationColorsYes;

//Set Gradation Colors 確認Cancelボタン押下
- (void)onAlertSetGradationColorsCancel;

//グラデーションカラー設定
- (void)setGradationColorWithChNoStart:(unsigned int)chNoStart
							   chNoEnd:(unsigned int)chNoEnd
							colorStart:(OGLCOLOR)colorStart
							  colorEnd:(OGLCOLOR)colorEnd;

//Set Default Colors 選択イベント
- (void)onSetDefaultColors;

//Set Default Colors 確認Yesボタン押下
- (void)onAlertSetDefaultColorsYes;

//Set Default Colors 確認Cancelボタン押下
- (void)onAlertSetDefaultColorsCancel;

//Saveボタン選択イベント
- (void)onSaveButton:(id)sender;

//保存確認Yesボタン押下
- (void)onAlertSaveYes;

//保存確認Cancelボタン押下
- (void)onAlertSaveCancel;

//パラメータインポート実行通知イベント
- (void)onImportParam:(NSNotification*)pNotification;

//通知送信処理
- (int)postNotificationWithName:(NSString*)pName;

//出力用パラメータ文字列生成
- (NSString*)makeColorParamForExport;

//パラメータ入力処理
- (int)importColorParam:(NSString*)pString;

@end


@implementation MTSettingColorPaletteViewCtrl

//******************************************************************************
// 初期化
//******************************************************************************
- (id)init
{
	int result = 0;
	NSString* pNibName = nil;
	unsigned int i = 0;
	
	//ビュー生成
	if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
		//iPhone / iPod touch
		pNibName = @"MTSettingColorPaletteView_iPhone";
	}
	else {
		//iPad
		pNibName = @"MTSettingColorPaletteView_iPad";
	}
	
	self = [super initWithNibName:pNibName bundle:nil];
	if (self == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//パレット初期化
	m_DefaultColorPalette.Initialize();
	m_ColorPalette.Initialize();
	m_ColorPaletteNo = 0;
	
	//グラデーションツール
	m_GradationChNoStart = 0;
	m_GradationChNoEnd = SM_MAX_CH_NUM - 1;
	m_GradationColorStart = OGLCOLOR(1.0f, 1.0f, 1.0f, 1.0f);
	m_GradationColorEnd = OGLCOLOR(1.0f, 1.0f, 1.0f, 1.0f);
	
	//カラーウェルタイトルリスト初期化
	for (i = 0; i < (SM_MAX_CH_NUM + 3); i++) {
		m_pColorWellTitleList[i] = nil;
	}
	
	//チャンネル選択ビュー生成
	m_pGradationChNoStartViewCtrl = [[MTSettingChannelViewCtrl alloc] init];
	m_pGradationChNoEndViewCtrl = [[MTSettingChannelViewCtrl alloc] init];
	[m_pGradationChNoStartViewCtrl setSelectedChNo:m_GradationChNoStart];
	[m_pGradationChNoEndViewCtrl setSelectedChNo:m_GradationChNoEnd];
	
	//カラーパラメータ出力ビュー生成
	m_pSettingColorParamExportViewCtrl = [[MTSettingColorParamExportViewCtrl alloc] init];
	
	//カラーパラメータ入力ビュー生成
	m_pSettingColorParamImportViewCtrl = [[MTSettingColorParamImportViewCtrl alloc] init];
	
	//カラーパレット変更通知名称
	m_pNotificationNameForChangedColorPalette = nil;
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return self;
}

//******************************************************************************
// 破棄
//******************************************************************************
- (void)dealloc
{
	unsigned int i = 0;
	
	for (i = 0; i < (SM_MAX_CH_NUM + 3); i++) {
		[m_pColorWellTitleList[i] release];
	}
	[m_pGradationChNoStartViewCtrl release];
	[m_pGradationChNoEndViewCtrl release];
	[m_pSettingColorParamExportViewCtrl release];
	[m_pSettingColorParamImportViewCtrl release];
	[m_pNotificationNameForChangedColorPalette release];
	
	[super dealloc];
}

//******************************************************************************
//　カラーパレット設定
//******************************************************************************
- (void)setColorPalette:(MTColorPalette*)pColorPalette
	defaultColorPalette:(MTColorPalette*)pDefaultColorPalette
		 colorPaletteNo:(unsigned int)colorPaletteNo
{
	m_ColorPalette.CopyFrom(pColorPalette);
	m_DefaultColorPalette.CopyFrom(pDefaultColorPalette);
	m_ColorPaletteNo = colorPaletteNo;
}

//******************************************************************************
// カラーパレット取得
//******************************************************************************
- (void)getColorPalette:(MTColorPalette*)pColorPalette
{
	pColorPalette->CopyFrom(&m_ColorPalette);
}

//******************************************************************************
//カラーパレット番号取得
//******************************************************************************
- (unsigned int)getColorPaletteNo
{
	return m_ColorPaletteNo;
}

//******************************************************************************
// カラーパレット変更通知名称設定
//******************************************************************************
- (void)setNotificationNameForChangedColorPalette:(NSString*)pNotificationName
{
	[m_pNotificationNameForChangedColorPalette release];
	m_pNotificationNameForChangedColorPalette = pNotificationName;
	[m_pNotificationNameForChangedColorPalette retain];
}

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
	UIBarButtonItem* pSaveButton = nil;
	NSNotificationCenter* pCenter = nil;
	
    [super viewDidLoad];
	
	//ウィンドウタイトル
	self.title = [NSString stringWithFormat:@"Color Palette %u", m_ColorPaletteNo];
	
	//ナビゲーションバー右側にSaveボタンを配置
	pSaveButton = [[UIBarButtonItem alloc]
				   initWithBarButtonSystemItem:UIBarButtonSystemItemSave
				   target:self
				   action:@selector(onSaveButton:)];
	self.navigationItem.rightBarButtonItem = pSaveButton;
	
	//通知先登録：パラメータインポート処理実行通知
	pCenter = [NSNotificationCenter defaultCenter];
	[pCenter addObserver:self
				selector:@selector(onImportParam:)
					name:@"MTSettingColorPaletteViewCtrl_onImportParam"
				  object:nil];
	[m_pSettingColorParamImportViewCtrl setNotificationNameForImportParam:@"MTSettingColorPaletteViewCtrl_onImportParam"];
	
	return;
}

//******************************************************************************
// インターフェース自動回転確認
//******************************************************************************
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    return (interfaceOrientation == UIInterfaceOrientationPortrait);
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
- (BOOL)shouldAutorotate
{
	//回転を許可する
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
- (NSUInteger)supportedInterfaceOrientations
{
	//全方向に対応する
	return UIInterfaceOrientationMaskAll;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{
	//テーブル再読み込み
	[m_pTableView reloadData];
	
	return;
}

//******************************************************************************
// ビュー非表示
//******************************************************************************
- (void)viewWillDisappear:(BOOL)animated
{
	return;
}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
    return 3;
} 

//******************************************************************************
// セクションヘッダ
//******************************************************************************
- (NSString*)tableView:(UITableView*)tableView titleForHeaderInSection:(NSInteger)section
{
	NSString* pSectionHeader = nil;
	
	switch(section) {
		case 0:
			pSectionHeader = @"Color Palette";
			break;
		case 1:
			pSectionHeader = @"Gradation Tool";
			break;
		case 2:
			pSectionHeader = @"Parameter Setup Tool";
			break;
		default:
			break;
	}
	
	return pSectionHeader;
} 

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	
	switch (section) {
		case 0:
			//Color Palette
			numOfRows = SM_MAX_CH_NUM + 3;
			break;
		case 1:
			//Gradation Tool
			numOfRows = 5;
			break;
		case 2:
			//Parameter Setup Tool
			numOfRows = 3;
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//Color Palette
			pCell = [self makeColorWellCellForIndexPath:indexPath];
			break;
		case 1:
			//Gradation Tool
			pCell = [self makeGradationToolCellForIndexPath:indexPath];
			break;
		case 2:
			//Parameter Setup Tool
			pCell = [self makeParameterSetupToolCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：カラーウェル
//******************************************************************************
- (UITableViewCell*)makeColorWellCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	unsigned int index = 0;
	NSString* pTitle = nil;
	MTColorPalette colorPalette;
	OGLCOLOR color;
	int result = 0;
	UIColorWell* pColorWell = nil;
	
	//セル識別子
	pCellIdentifier = [NSString stringWithFormat:@"MTSettingColorPaletteViewCtrl_ColorWell_%ld", (long)indexPath.row];
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleValue1
									   reuseIdentifier:pCellIdentifier];
	}
	
	index = (unsigned int)indexPath.row;
	
	//項目名
	if (index < SM_MAX_CH_NUM) {
		pTitle= [NSString stringWithFormat:@"Ch.%u", index + 1];
	}
	if (index == (SM_MAX_CH_NUM + 0)) {
		pTitle = @"Background";
	}
	if (index == (SM_MAX_CH_NUM + 1)) {
		pTitle = @"Grid Line";
	}
	if (index == (SM_MAX_CH_NUM + 2)) {
		pTitle = @"Counter";
	}
	[m_pColorWellTitleList[index] release];
	m_pColorWellTitleList[index] = pTitle;
	[m_pColorWellTitleList[index] retain];
	
	//色
	if (index < SM_MAX_CH_NUM) {
		result = m_ColorPalette.GetChColor(index, &color);
		if (result != 0) goto EXIT;
	}
	if (index == (SM_MAX_CH_NUM + 0)) {
		m_ColorPalette.GetBackgroundColor(&color);
	}
	if (index == (SM_MAX_CH_NUM + 1)) {
		m_ColorPalette.GetGridLineColor(&color);
	}
	if (index == (SM_MAX_CH_NUM + 2)) {
		m_ColorPalette.GetCounterColor(&color);
	}
	
	//カラーウェル作成
	pColorWell = [[UIColorWell alloc] initWithFrame:CGRectMake(0, 0, 20, 20)];
	pColorWell.title = pTitle;
	pColorWell.supportsAlpha = YES;
	pColorWell.selectedColor = [UIColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
	[pColorWell addTarget:self action:@selector(onColorWellValueChanged:) forControlEvents:UIControlEventValueChanged];
	
	//セル設定：テキスト
	pCell.textLabel.text = pTitle;
	
	//セル設定：詳細テキスト
	pCell.detailTextLabel.text = OGLColorUtil::MakeHexRGBAFromColor(color);
	
	//アクセサリビュー設定
	pCell.accessoryView = pColorWell;
	
	//セル設定：アクセサリ設定
	pCell.accessoryType = UITableViewCellAccessoryNone;
	
EXIT:;
	return pCell;
}

//******************************************************************************
// テーブルセル作成：グラデーションツール
//******************************************************************************
- (UITableViewCell*)makeGradationToolCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	unsigned int index = 0;
	MTColorPalette colorPalette;
	OGLCOLOR color;
	UIColorWell* pColorWell = nil;
	
	//セル識別子
	pCellIdentifier = [NSString stringWithFormat:@"MTSettingColorPaletteViewCtrl_GradationTool_%ld", (long)indexPath.row];
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleValue1
									   reuseIdentifier:pCellIdentifier];
	}
			
	//セル設定
	index = (unsigned int)indexPath.row;
	switch (index) {
		case 0:
			m_GradationChNoStart = [m_pGradationChNoStartViewCtrl getSelectedChNo];
			pCell.textLabel.text = @"Start";
			pCell.detailTextLabel.text = [NSString stringWithFormat:@"Ch.%u", m_GradationChNoStart + 1];
			pCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
			break;
		case 1:
			color = m_GradationColorStart;
			pCell.textLabel.text = @"Start";
			pCell.detailTextLabel.text = OGLColorUtil::MakeHexRGBAFromColor(color);
			pCell.accessoryType = UITableViewCellAccessoryNone;
			//カラーウェル
			pColorWell = [[UIColorWell alloc] initWithFrame:CGRectMake(0, 0, 20, 20)];
			pColorWell.title = @"Start";
			pColorWell.supportsAlpha = YES;
			pColorWell.selectedColor = [UIColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
			[pColorWell addTarget:self action:@selector(onGradationColorWellValueChanged:) forControlEvents:UIControlEventValueChanged];
			pCell.accessoryView = pColorWell;
			break;
		case 2:
			m_GradationChNoEnd = [m_pGradationChNoEndViewCtrl getSelectedChNo];
			pCell.textLabel.text = @"End";
			pCell.detailTextLabel.text = [NSString stringWithFormat:@"Ch.%u", m_GradationChNoEnd + 1];
			pCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
			break;
		case 3:
			color = m_GradationColorEnd;
			pCell.textLabel.text = @"End";
			pCell.detailTextLabel.text = OGLColorUtil::MakeHexRGBAFromColor(color);
			pCell.accessoryType = UITableViewCellAccessoryNone;
			//カラーウェル
			pColorWell = [[UIColorWell alloc] initWithFrame:CGRectMake(0, 0, 20, 20)];
			pColorWell.title = @"End";
			pColorWell.supportsAlpha = YES;
			pColorWell.selectedColor = [UIColor colorWithRed:color.r green:color.g blue:color.b alpha:color.a];
			[pColorWell addTarget:self action:@selector(onGradationColorWellValueChanged:) forControlEvents:UIControlEventValueChanged];
			pCell.accessoryView = pColorWell;
			break;
		case 4:
			pCell.textLabel.text = @"Set Gradation Colors";
			pCell.detailTextLabel.text = @"";
			pCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
			break;
		default:
			break;
	}
	
EXIT:;
	return pCell;
}

//******************************************************************************
// テーブルセル作成：パラメータ設定ツール
//******************************************************************************
- (UITableViewCell*)makeParameterSetupToolCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	unsigned int index = 0;
	MTColorPalette colorPalette;
	OGLCOLOR color;
	
	//セル識別子
	pCellIdentifier = [NSString stringWithFormat:@"MTSettingColorPaletteViewCtrl_ParameterSetupTool_%ld", (long)indexPath.row];
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
	}
	
	//セル設定
	index = (unsigned int)indexPath.row;
	switch (index) {
		case 0:
			pCell.textLabel.text = @"Set Default Colors";
			pCell.detailTextLabel.text = @"";
			pCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
			break;
		case 1:
			pCell.textLabel.text = @"Export Color Parameters";
			pCell.detailTextLabel.text = @"";
			pCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
			break;
		case 2:
			pCell.textLabel.text = @"Import Color Parameters";
			pCell.detailTextLabel.text = @"";
			pCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
			break;
		default:
			break;
	}
	
EXIT:;
	return pCell;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
	[tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//Color Palette
			[self onSelectColorWellCellForIndexPath:indexPath];
			break;
		case 1:
			//Gradation Tool
			[self onSelectGradationToolCellForIndexPath:indexPath];
			break;
		case 2:
			//Parameter Setup Tool
			[self onSelectParameterSetupToolCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：カラーパレット
//******************************************************************************
- (void)onSelectColorWellCellForIndexPath:(NSIndexPath*)indexPath
{
	return;
}

//******************************************************************************
// テーブルセル選択イベント：グラデーションツール
//******************************************************************************
- (void)onSelectGradationToolCellForIndexPath:(NSIndexPath*)indexPath
{
	unsigned int index = 0;
	
	//セル設定
	index = (unsigned int)indexPath.row;
	switch (index) {
		case 0:
			//Start Ch.
			//チャンネル選択ビュー表示
			[self.navigationController pushViewController:m_pGradationChNoStartViewCtrl animated:YES];
			break;
		case 1:
			//Start Color
			break;
		case 2:
			//End Ch.
			//チャンネル選択ビュー表示
			[self.navigationController pushViewController:m_pGradationChNoEndViewCtrl animated:YES];
			break;
		case 3:
			//End Color
			break;
		case 4:
			//Set Gradation Colors
			[self onSetGradationColors];
			break;
		default:
			break;
	}

	return;
}

//******************************************************************************
// テーブルセル選択イベント：パラメータ設定ツール
//******************************************************************************
- (void)onSelectParameterSetupToolCellForIndexPath:(NSIndexPath*)indexPath
{
	unsigned int index = 0;
	NSString* pParamString = nil;
	
	//セル設定
	index = (unsigned int)indexPath.row;
	switch (index) {
		case 0:
			//Set Default Colors
			[self onSetDefaultColors];
			break;
		case 1:
			//Export Color Parameters
			pParamString = [self makeColorParamForExport];
			[m_pSettingColorParamExportViewCtrl setParamString:pParamString];
			[self.navigationController pushViewController:m_pSettingColorParamExportViewCtrl animated:YES];
			break;
		case 2:
			//Import Color Parameters
			[self.navigationController pushViewController:m_pSettingColorParamImportViewCtrl animated:YES];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// カラーウェル変更イベント
//******************************************************************************
- (void)onColorWellValueChanged:(id)sender
{
	UIColorWell* pColorWell = nil;
	unsigned int index = 0;
	CGFloat red = 0.0f;
	CGFloat green = 0.0f;
	CGFloat blue = 0.0f;
	CGFloat alpha = 0.0f;
	OGLCOLOR color;
	BOOL isConverted = NO;
	bool isFound = false;
	
	pColorWell = (UIColorWell*)sender;
	
	//NSLog(@"onColorWellValueChanged %@", pColorWell.title);
	
	//カラーウェルから色を取得
	isConverted = [pColorWell.selectedColor getRed:&red green:&green blue:&blue alpha:&alpha];
	if (!isConverted) goto EXIT;
	color = OGLCOLOR(red, green, blue, alpha);
	
	//通知元カラーウェルの確認
	for (index = 0; index < (SM_MAX_CH_NUM + 3); index++) {
		if ([pColorWell.title isEqualToString:m_pColorWellTitleList[index]]) {
			isFound = true;
			break;
		}
	}
	if (!isFound) goto EXIT;
	
	//カラーパレット更新
	if (index < SM_MAX_CH_NUM) {
		m_ColorPalette.SetChColor(index, color);
	}
	else if (index == (SM_MAX_CH_NUM + 0)) {
		m_ColorPalette.SetBackgroundColor(color);
	}
	else if (index == (SM_MAX_CH_NUM + 1)) {
		m_ColorPalette.SetGridLineColor(color);
	}
	else if (index == (SM_MAX_CH_NUM + 2)) {
		m_ColorPalette.SetCounterColor(color);
	}
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	return;
}

//******************************************************************************
// グラデーションツール カラーウェル変更イベント
//******************************************************************************
- (void)onGradationColorWellValueChanged:(id)sender
{
	UIColorWell* pColorWell = nil;
	CGFloat red = 0.0f;
	CGFloat green = 0.0f;
	CGFloat blue = 0.0f;
	CGFloat alpha = 0.0f;
	OGLCOLOR color;
	BOOL isConverted = NO;
	
	pColorWell = (UIColorWell*)sender;
	
	//NSLog(@"onGradationColorWellValueChanged %@", pColorWell.title);
	
	//カラーウェルから色を取得
	isConverted = [pColorWell.selectedColor getRed:&red green:&green blue:&blue alpha:&alpha];
	if (!isConverted) goto EXIT;
	color = OGLCOLOR(red, green, blue, alpha);
	
	//色を記録
	if ([pColorWell.title isEqualToString:@"Start"]) {
		m_GradationColorStart = color;
	}
	else if ([pColorWell.title isEqualToString:@"End"]) {
		m_GradationColorEnd = color;
	}
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	return;
}

//******************************************************************************
// Set Gradation Colors 選択イベント
//******************************************************************************
- (void)onSetGradationColors
{
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	//確認アラート表示
	pAlertController = [UIAlertController alertControllerWithTitle:@"Confirmation"
														   message:@"Do you want to set gradation colors?"
													preferredStyle:UIAlertControllerStyleAlert];
	pAlertAction = [UIAlertAction actionWithTitle:@"Yes"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSetGradationColorsYes];}];
	[pAlertController addAction:pAlertAction];
	pAlertAction = [UIAlertAction actionWithTitle:@"Cancel"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSetGradationColorsCancel];}];
	[pAlertController addAction:pAlertAction];
	[self presentViewController:pAlertController animated:YES completion:nil];
	
	return;
}

//******************************************************************************
// Set Gradation Colors 確認Yesボタン押下
//******************************************************************************
- (void)onAlertSetGradationColorsYes
{
	//グラデーションカラー設定
	[self setGradationColorWithChNoStart:m_GradationChNoStart
								 chNoEnd:m_GradationChNoEnd
							  colorStart:m_GradationColorStart
								colorEnd:m_GradationColorEnd];
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
	return;
}

//******************************************************************************
// Set Gradation Colors 確認Cancelボタン押下
//******************************************************************************
- (void)onAlertSetGradationColorsCancel
{
	return;
}

//******************************************************************************
// グラデーションカラー設定
//******************************************************************************
- (void)setGradationColorWithChNoStart:(unsigned int)chNoStart
							   chNoEnd:(unsigned int)chNoEnd
							colorStart:(OGLCOLOR)colorStart
							  colorEnd:(OGLCOLOR)colorEnd
{
	int result = 0;
	unsigned int chNo = 0;
	OGLCOLOR color;
	float ratio = 0.0f;
	
	if ((chNoStart >= SM_MAX_CH_NUM) || (chNoEnd >= SM_MAX_CH_NUM)) {
		//何もしない
	}
	
	if (chNoStart == chNoEnd) {
		//何もしない
	}
	else if (chNoStart < chNoEnd) {
		for (chNo = chNoStart; chNo <= chNoEnd; chNo++) {
			ratio = (float)(chNo - chNoStart) / (float)(chNoEnd - chNoStart);
			color = OGLCOLOR((colorEnd.r - colorStart.r) * ratio + colorStart.r,
							 (colorEnd.g - colorStart.g) * ratio + colorStart.g,
							 (colorEnd.b - colorStart.b) * ratio + colorStart.b,
							 1.0f);
			//カラーパレット設定
			result = m_ColorPalette.SetChColor(chNo, color);
			if (result != 0) goto EXIT;
		}
	}
	else {
		for (chNo = chNoEnd; chNo <= chNoStart; chNo++) {
			ratio = (float)(chNo - chNoEnd) / (float)(chNoStart - chNoEnd);
			color = OGLCOLOR((colorStart.r - colorEnd.r) * ratio + colorEnd.r,
							 (colorStart.g - colorEnd.g) * ratio + colorEnd.g,
							 (colorStart.b - colorEnd.b) * ratio + colorEnd.b,
							 1.0f);
			//カラーパレット設定
			result = m_ColorPalette.SetChColor(chNo, color);
			if (result != 0) goto EXIT;
		}
	}
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// Set Default Colors 選択イベント
//******************************************************************************
- (void)onSetDefaultColors
{
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	//確認アラート表示
	pAlertController = [UIAlertController alertControllerWithTitle:@"Confirmation"
														   message:@"Do you want to set default colors?"
													preferredStyle:UIAlertControllerStyleAlert];
	pAlertAction = [UIAlertAction actionWithTitle:@"Yes"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSetDefaultColorsYes];}];
	[pAlertController addAction:pAlertAction];
	pAlertAction = [UIAlertAction actionWithTitle:@"Cancel"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSetDefaultColorsCancel];}];
	[pAlertController addAction:pAlertAction];
	[self presentViewController:pAlertController animated:YES completion:nil];
	
	return;
}

//******************************************************************************
// Set Default Colors 確認Yesボタン押下
//******************************************************************************
- (void)onAlertSetDefaultColorsYes
{
	//カラーパレット更新
	m_ColorPalette.CopyFrom(&m_DefaultColorPalette);
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
	return;
}

//******************************************************************************
// Set Default Colors 確認Cancelボタン押下
//******************************************************************************
- (void)onAlertSetDefaultColorsCancel
{
	return;
}

//******************************************************************************
// Saveボタン選択イベント
//******************************************************************************
- (void)onSaveButton:(id)sender
{
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	//確認アラート表示
	pAlertController = [UIAlertController alertControllerWithTitle:@"Confirmation"
														   message:@"Do you want to save changes?"
													preferredStyle:UIAlertControllerStyleAlert];
	pAlertAction = [UIAlertAction actionWithTitle:@"Yes"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSaveYes];}];
	[pAlertController addAction:pAlertAction];
	pAlertAction = [UIAlertAction actionWithTitle:@"Cancel"
											style:UIAlertActionStyleDefault
										  handler:^(UIAlertAction *action) {[self onAlertSaveCancel];}];
	[pAlertController addAction:pAlertAction];
	[self presentViewController:pAlertController animated:YES completion:nil];
	
	return;
}

//******************************************************************************
// 保存確認Yesボタン押下
//******************************************************************************
- (void)onAlertSaveYes
{
	//カラーパレット変更通知
	[self postNotificationWithName:m_pNotificationNameForChangedColorPalette];
	
	//1階層戻る
	[self.navigationController popViewControllerAnimated:YES];
	
	return;
}

//******************************************************************************
// 保存確認Cancelボタン押下
//******************************************************************************
- (void)onAlertSaveCancel
{
	return;
}

//******************************************************************************
// パラメータインポート実行通知イベント
//******************************************************************************
- (void)onImportParam:(NSNotification*)pNotification
{
	int result = 0;
	NSString* pParamString = nil;
	
	//インポートパラメータ文字列取得
	pParamString = [m_pSettingColorParamImportViewCtrl getParamString];
	
	//インポート処理実行
	result = [self importColorParam:pParamString];
	if (result != 0) goto EXIT;
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// 通知送信処理
//******************************************************************************
- (int)postNotificationWithName:(NSString*)pName
{
	int result = 0;
	NSNotification* pNotification = nil;
	NSNotificationCenter* pCenter = nil;
	
	//通知オブジェクトを作成
	pNotification = [NSNotification notificationWithName:pName
												  object:self
												userInfo:nil];
	//通知する
	pCenter = [NSNotificationCenter defaultCenter];
	
	//通知に対応する処理を演奏スレッドで処理させる場合
	//[pCenter postNotification:pNotification];
	
	//通知に対応する処理をメインスレッドに処理させる場合
	[pCenter performSelectorOnMainThread:@selector(postNotification:)
							  withObject:pNotification
						   waitUntilDone:NO];
	
	return result;
}

//******************************************************************************
// 出力用パラメータ文字列生成
//******************************************************************************
- (NSString*)makeColorParamForExport
{
	int result = 0;
	unsigned int chNo = 0;
	OGLCOLOR color;
	NSMutableString* pLines = nil;
	NSString* pLine = nil;
	NSString* pExportString = nil;
	
	pLines = [[NSMutableString alloc] initWithString:@""];
	
	//パラメータ文字列生成：Ch.1-16
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		result = m_ColorPalette.GetChColor(chNo, &color);
		if (result !=0) goto EXIT;
		pLine = [NSString stringWithFormat:@"Ch-%02u-NoteRGBA=%@\n", chNo+1, OGLColorUtil::MakeHexRGBAFromColor(color)];
		[pLines appendString:pLine];
	}
	
	//パラメータ文字列生成：背景
	m_ColorPalette.GetBackgroundColor(&color);
	pLine = [NSString stringWithFormat:@"BackGroundRGBA=%@\n", OGLColorUtil::MakeHexRGBAFromColor(color)];
	[pLines appendString:pLine];
	
	//パラメータ文字列生成：グリッドライン
	m_ColorPalette.GetGridLineColor(&color);
	pLine = [NSString stringWithFormat:@"GridLineRGBA=%@\n", OGLColorUtil::MakeHexRGBAFromColor(color)];
	[pLines appendString:pLine];
	
	//パラメータ文字列生成：カウンター
	m_ColorPalette.GetCounterColor(&color);
	pLine = [NSString stringWithFormat:@"CounterRGBA=%@\n", OGLColorUtil::MakeHexRGBAFromColor(color)];
	[pLines appendString:pLine];
	
	pExportString = [NSString stringWithString:pLines];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	[pLines release];
	return pExportString;
}

//******************************************************************************
// パラメータ入力処理
//******************************************************************************
- (int)importColorParam:(NSString*)pString
{
	int result = 0;
	YNConfFile* pConfFile = nil;
	unsigned int chNo = 0;
	NSString* initext = nil;
	NSString* pKey = nil;
	NSString* pHexColor = nil;
	OGLCOLOR color;
	
	//設定ファイル初期化
	pConfFile = [[YNConfFile alloc] init];
	if (pConfFile == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//テキスト読み込み
	initext = [NSString stringWithFormat:@"[Color]\n%@", pString];
	result = [pConfFile loadText:initext];
	if (result != 0) goto EXIT;
	
	//セクション設定
	[pConfFile setCurSection:@"Color"];
	
	//カラーパレット反映：Ch.1-16（パラメータが取得できた場合のみ）
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		pKey = [NSString stringWithFormat:@"Ch-%02u-NoteRGBA", chNo+1];
		pHexColor = [pConfFile strValueForKey:pKey defaultValue:@"NOT_FOUND"];
		if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
			color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
			result = m_ColorPalette.SetChColor(chNo, color);
			if (result != 0) goto EXIT;
		}
	}
	
	//カラーパレット反映：背景（パラメータが取得できた場合のみ）
	pHexColor = [pConfFile strValueForKey:@"BackGroundRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_ColorPalette.SetBackgroundColor(color);
	}
	
	//カラーパレット反映：グリッドライン（パラメータが取得できた場合のみ）
	pHexColor = [pConfFile strValueForKey:@"GridLineRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_ColorPalette.SetGridLineColor(color);
	}
	
	//カラーパレット反映：カウンター（パラメータが取得できた場合のみ）
	pHexColor = [pConfFile strValueForKey:@"CounterRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_ColorPalette.SetCounterColor(color);
	}
	
	//カラーウェル反映：カウンター（パラメータが取得できた場合のみ）
	// 互換性を保つため ini ファイルに定義されている"CaptionRGBA"をインポート可能とする
	pHexColor = [pConfFile strValueForKey:@"CaptionRGBA" defaultValue:@"NOT_FOUND"];
	if (![pHexColor isEqualToString:@"NOT_FOUND"] && [pHexColor length] == 8) {
		color = OGLColorUtil::MakeColorFromHexRGBA(pHexColor);
		m_ColorPalette.SetCounterColor(color);
	}
	
EXIT:;
	[pConfFile release];
	return result;
}

@end

