//******************************************************************************
//
// MIDITrail / MTSettingColorViewCtrl
//
// カラー設定ビュークラス
//
// Copyright (C) 2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTSettingColorViewCtrl.h"


//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTSettingColorViewCtrl ()

//テーブルセル作成：カラーパレット
- (UITableViewCell*)makeColorPaletteCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：カラーパレット
- (void)onSelectColorPaletteCellForIndexPath:(NSIndexPath*)indexPath tableView:(UITableView*)tableView;

//カラーパレット画像取得
- (UIImage*)getColorPaletteImageWithPalette:(MTColorPalette*)colorPalette;

@end


@implementation MTSettingColorViewCtrl

//******************************************************************************
// 初期化
//******************************************************************************
- (id)init
{
	int result = 0;
	NSString* pNibName = nil;
	
	m_pSettingColorPaletteViewCtrl = nil;
	
	//ビュー生成
	if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
		//iPhone / iPod touch
		pNibName = @"MTSettingColorView_iPhone";
	}
	else {
		//iPad
		pNibName = @"MTSettingColorView_iPad";
	}
	
	self = [super initWithNibName:pNibName bundle:nil];
	if (self == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//ビュー設定
	self.title = @"Color";
	
	//色設定初期化：PianoRoll3D.ini の色設定をデフォルト設定として表示
	result = m_ColorConf.Initialize(@"PianoRoll3D");
	if (result != 0) goto EXIT;
	
	//カラーパレット設定ビュー生成
	m_pSettingColorPaletteViewCtrl = [[MTSettingColorPaletteViewCtrl alloc] init];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return self;
}

//******************************************************************************
// 破棄
//******************************************************************************
- (void)dealloc
{
	[m_pSettingColorPaletteViewCtrl release];
	[super dealloc];
}

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
	NSNotificationCenter* pCenter = nil;
	
    [super viewDidLoad];
	
	//ナビゲーションバー右側に編集ボタンを登録
	self.navigationItem.rightBarButtonItems = @[[self editButtonItem]];
	
	//通知先登録：カラーパレット変更通知
	pCenter = [NSNotificationCenter defaultCenter];
	[pCenter addObserver:self
				selector:@selector(onChangedColorPalette:)
					name:@"MTSettingColorViewCtrl_onChangedColorPalette"
				  object:nil];
	[m_pSettingColorPaletteViewCtrl setNotificationNameForChangedColorPalette:@"MTSettingColorViewCtrl_onChangedColorPalette"];
	
	return;
}

//******************************************************************************
// インターフェース自動回転確認
//******************************************************************************
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    return (interfaceOrientation == UIInterfaceOrientationPortrait);
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
- (BOOL)shouldAutorotate
{
	//回転を許可する
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
- (NSUInteger)supportedInterfaceOrientations
{
	//全方向に対応する
	return UIInterfaceOrientationMaskAll;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{	
	return;
}

//******************************************************************************
// ビュー非表示
//******************************************************************************
- (void)viewWillDisappear:(BOOL)animated
{
	return;
}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
    return 1;
} 

//******************************************************************************
// セクションヘッダ
//******************************************************************************
- (NSString*)tableView:(UITableView*)tableView titleForHeaderInSection:(NSInteger)section
{
    return nil;
} 

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	
	switch (section) {
		case 0:
			numOfRows = 7;
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			pCell = [self makeColorPaletteCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：カラーパレット
//******************************************************************************
- (UITableViewCell*)makeColorPaletteCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	unsigned int colorPaletteNo = 0;
	MTColorPalette colorPalette;
	UIImage* pColorPaletteImage = nil;
	int result = 0;
	
	//セル識別子
	pCellIdentifier = [NSString stringWithFormat:@"MTSettingColorViewCtrl_ColorPalette_%ld", (long)indexPath.row];
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
	}
	
	//カラーパレット番号
	colorPaletteNo = (unsigned int)indexPath.row;
	
	//カラーパレット取得
	result = m_ColorConf.GetColorPalette(colorPaletteNo, &colorPalette);
	if (result != 0) goto EXIT;
	
	//カラーパレット画像生成
	pColorPaletteImage = [self getColorPaletteImageWithPalette:&colorPalette];
		
	//セル設定：テキスト
	if (colorPaletteNo == 0) {
		pCell.textLabel.text = @"Default";
	}
	else {
		pCell.textLabel.text = [NSString stringWithFormat:@"Palette %u", colorPaletteNo];
	}
	pCell.detailTextLabel.text = @"";
	
	//セル設定：画像
	pCell.imageView.image = pColorPaletteImage;
	
	//セル設定：チェックマーク設定
	if (m_ColorConf.GetSelectedColorPaletteNo() == colorPaletteNo) {
		pCell.accessoryType = UITableViewCellAccessoryCheckmark;
	}
	else {
		pCell.accessoryType = UITableViewCellAccessoryNone;
	}
	
	//セル設定：編集用アクセサリ設定（Default意外は編集可能）
	if (colorPaletteNo == 0) {
		pCell.editingAccessoryType = UITableViewCellAccessoryNone;
	}
	else {
		pCell.editingAccessoryType = UITableViewCellAccessoryDisclosureIndicator;
	}
	
EXIT:;
//	[pColorPaletteImage release];
	return pCell;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
	[tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			[self onSelectColorPaletteCellForIndexPath:indexPath tableView:tableView];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：カラーパレット
//******************************************************************************
- (void)onSelectColorPaletteCellForIndexPath:(NSIndexPath*)indexPath tableView:(UITableView *)tableView
{
	int result = 0;
	unsigned int colorPaletteNo = 0;
	MTColorPalette colorPalette;
	MTColorPalette defaultColorPalette;

	//選択されたカラーパレット番号
	colorPaletteNo = (unsigned int)indexPath.row;
	
	//カラーパレット取得
	result = m_ColorConf.GetColorPalette(colorPaletteNo, &colorPalette);
	if (result != 0) goto EXIT;
	result = m_ColorConf.GetColorPalette(0, &defaultColorPalette);
	if (result != 0) goto EXIT;
	
	//カラーパレット選択処理
	if (super.editing) {
		//編集モードであればカラーパレット設定ビューを表示
		//ただし先頭行(Default)の場合はなにもしない
		if (indexPath.row != 0) {
			//カラーパレット設定
			[m_pSettingColorPaletteViewCtrl setColorPalette:&colorPalette
										defaultColorPalette:&defaultColorPalette
											 colorPaletteNo:colorPaletteNo];
			
			//カラーパレット設定ビュー表示
			[self.navigationController pushViewController:m_pSettingColorPaletteViewCtrl animated:YES];
		}
	}
	else {
		//編集中でなければ選択されたカラーパレットを登録
		result = m_ColorConf.SetSelectedColorPaletteNo(colorPaletteNo);
		if (result != 0) goto EXIT;
		
		//設定保存
		result = m_ColorConf.Save();
		if (result != 0) goto EXIT;
		
		//再表示
		[tableView performSelector:@selector(reloadData) withObject:nil afterDelay:0.1f];
	}
	
EXIT:;
	return;
}

//******************************************************************************
// 編集開始終了通知
//******************************************************************************
- (void)setEditing:(BOOL)editing animated:(BOOL)animated
{
	//編集開始終了処理
	[super setEditing:editing animated:animated];
	[m_pTableView setEditing:editing animated:animated];
	
	return;
}

//******************************************************************************
// 行単位 編集可否判定
//******************************************************************************
- (BOOL)tableView:(UITableView *)tableView canEditRowAtIndexPath:(NSIndexPath *)indexPath
{
	return YES;
}

//******************************************************************************
// 行単位 編集スタイル
//******************************************************************************
- (UITableViewCellEditingStyle)tableView:(UITableView *)tableView editingStyleForRowAtIndexPath:(NSIndexPath *)indexPath
{
	return UITableViewCellEditingStyleNone;
}

//******************************************************************************
// 行単位 編集時インデント
//******************************************************************************
- (BOOL)tableView:(UITableView *)tableview shouldIndentWhileEditingRowAtIndexPath:(NSIndexPath *)indexPath
{
	return NO;
}

//******************************************************************************
// カラーパレット画像取得
//******************************************************************************
- (UIImage*)getColorPaletteImageWithPalette:(MTColorPalette*)pColorPalette
{
	int result = 0;
	UIImage* pColorPaletteImage = nil;
	CGFloat width = 0;
	CGFloat height = 0;
	CGContextRef context = nil;
	CGRect rectangle;
	unsigned int chNo = 0;
	OGLCOLOR color;
	
	//カラーバレット画像（サイズ:139x20, 余白:2, 各色矩形:5x16）
	//  +----------------------+-+----------------+
	//  | +--+ +--+  ...  +--+ | | +--+ +--+ +--+ |
	//  | |Ch| |Ch|  ...  |Ch| | | |BG| |GL| |CT| |
	//  | | 0| | 1|  ...  |15| | | |  | |  | |  | |
	//  | +--+ +--+  ...  +--+ | | +--+ +--+ +--+ |
	//  +----------------------+-+----------------+
	
	//パレット画像サイズ（ポイント）
	width = 139;
	height = 20;
	
	//ビットマップ画像コンテキスト生成
	UIGraphicsBeginImageContextWithOptions(
				CGSizeMake(width, height),	//画像サイズ（ポイント）
				NO,			//不透明フラグ
				0.0			//スケール：メインスクリーンのスケールファクタ
			);
	
	//コンテキスト取得
	context = UIGraphicsGetCurrentContext();
	if (context == nil) {
		result = YN_SET_ERR(@"UIKit API error.", 0, 0);
	}
	
	//コンテキスト有効化
	UIGraphicsPushContext(context);
	
	//Ch1-16用 背景を描画（灰色）
	rectangle = CGRectMake(0, 0, (2 + 5) * SM_MAX_CH_NUM + 2, 20); //x, y, width, height
	CGContextSetRGBFillColor(context, 0.8, 0.8, 0.8, 1.0);
	CGContextFillRect(context, rectangle);
	
	//セパレータ 背景を描画（透明）
	rectangle = CGRectMake((2 + 5) * SM_MAX_CH_NUM + 2, 0, 2, 20);
	CGContextSetRGBFillColor(context, 0.0, 0.0, 0.0, 0.0);
	CGContextFillRect(context, rectangle);
	
	//BG/GL/CT用 背景を描画（灰色）
	rectangle = CGRectMake((2 + 5) * SM_MAX_CH_NUM + 4, 0, (2 + 5) * 3 + 2, 20); //x, y, width, height
	CGContextSetRGBFillColor(context, 0.8, 0.8, 0.8, 1.0);
	CGContextFillRect(context, rectangle);
	
	//チャンネルカラー(Ch.1-16)の矩形を描画
	for (chNo = 0; chNo < SM_MAX_CH_NUM; chNo++) {
		pColorPalette->GetChColor(chNo, &color);
		rectangle = CGRectMake((2 + 5) * chNo  + 2, 2, 5, 16);
		CGContextSetRGBFillColor(context, color.r, color.g, color.b, 1.0);
		CGContextFillRect(context, rectangle);
	}
	
	//背景色の矩形を描画
	pColorPalette->GetBackgroundColor(&color);
	rectangle = CGRectMake((2 + 5) * SM_MAX_CH_NUM + 6 + (2 + 5) * 0, 2, 5, 16);
	CGContextSetRGBFillColor(context, color.r, color.g, color.b, 1.0);
	CGContextFillRect(context, rectangle);
	
	//グリッドライン色の矩形を描画
	pColorPalette->GetGridLineColor(&color);
	rectangle = CGRectMake((2 + 5) * SM_MAX_CH_NUM + 6 + (2 + 5) * 1, 2, 5, 16);
	CGContextSetRGBFillColor(context, color.r, color.g, color.b, 1.0);
	CGContextFillRect(context, rectangle);
	
	//カウンター色の矩形を描画
	pColorPalette->GetCounterColor(&color);
	rectangle = CGRectMake((2 + 5) * SM_MAX_CH_NUM + 6 + (2 + 5) * 2, 2, 5, 16);
	CGContextSetRGBFillColor(context, color.r, color.g, color.b, 1.0);
	CGContextFillRect(context, rectangle);

	//コンテキスト無効化
	UIGraphicsPopContext();
	
	//画像取得
	pColorPaletteImage = UIGraphicsGetImageFromCurrentImageContext();
	
	//ビットマップ画像コンテキスト破棄
	UIGraphicsEndImageContext();
	
EXIT:;
	return pColorPaletteImage;
}

//******************************************************************************
// 選択カラーパレット画像取得
//******************************************************************************
- (UIImage*)getSelectedColorPaletteImage
{
	MTColorPalette colorPalette;
	UIImage* pColorPaletteImage = nil;
	
	//選択カラーパレットの画像を取得
	m_ColorConf.GetSelectedColorPalette(&colorPalette);
	pColorPaletteImage = [self getColorPaletteImageWithPalette:&colorPalette];
	
	return pColorPaletteImage;
}

//******************************************************************************
// 選択カラーパレットテキスト取得
//******************************************************************************
- (NSString*)getSelectedColorPaletteText
{
	NSString* pText = nil;
	unsigned int colorPaletteNo = 0;
	
	colorPaletteNo = m_ColorConf.GetSelectedColorPaletteNo();
	
	if (colorPaletteNo == 0) {
		pText = @"Default";
	}
	else {
		pText = [NSString stringWithFormat:@"Palette %u", colorPaletteNo];
	}
	
	return pText;
}

//******************************************************************************
// カラーパレット変更イベント
//******************************************************************************
- (void)onChangedColorPalette:(NSNotification*)pNotification
{
	int result = 0;
	MTColorPalette colorPalette;
	unsigned int colorPaletteNo = 0;
	
	//変更されたカラーパレットを取得
	[m_pSettingColorPaletteViewCtrl getColorPalette:&colorPalette];
	
	//カラー情報更新
	colorPaletteNo = [m_pSettingColorPaletteViewCtrl getColorPaletteNo];
	result = m_ColorConf.SetColorPalette(colorPaletteNo, &colorPalette);
	if (result != 0) goto EXIT;
	
	//カラー設定保存
	result = m_ColorConf.Save();
	if (result != 0) goto EXIT;
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

@end

