//******************************************************************************
//
// MIDITrail / MTTimeIndicatorRing
//
// タイムインジケータリング描画クラス
//
// Copyright (C) 2019-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTTimeIndicatorRing.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTTimeIndicatorRing::MTTimeIndicatorRing(void)
{
	m_CurPos = 0.0f;
	m_CurTickTime = 0;
	m_isEnable = true;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTTimeIndicatorRing::~MTTimeIndicatorRing(void)
{
	Release();
}

//******************************************************************************
// タイムインジケータ生成
//******************************************************************************
int MTTimeIndicatorRing::Create(
		OGLDevice* pOGLDevice,
		NSString* pSceneName,
		SMSeqData* pSeqData
	)
{
	int result = 0;
	SMBarList barList;

	Release();

	//ノートデザインオブジェクト初期化
	result = m_NoteDesign.Initialize(pSceneName, pSeqData);
	if (result != 0) goto EXIT;

	//プリミティブ生成：タイムライン
	result = _CreatePrimitiveLine(pOGLDevice);
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// プリミティブ生成
//******************************************************************************
int MTTimeIndicatorRing::_CreatePrimitiveLine(
		OGLDevice* pOGLDevice
	)
{
	int result = 0;
	OGLsizei vertexNum = 0;
	OGLsizei indexNum = 0;
	MTTIMEINDICATOR_VERTEX* pVertex = NULL;
	unsigned int* pIndex = NULL;

	//プリミティブ初期化
	result = m_PrimitiveLine.Initialize(
					pOGLDevice,						//描画デバイス
					sizeof(MTTIMEINDICATOR_VERTEX),	//頂点サイズ
					_GetFVFFormat(),				//頂点FVFフォーマット
					OGL_LINES						//プリミティブ種別
				);
	if (result != 0) goto EXIT;

	//頂点バッファ生成：1サークル128頂点
	vertexNum = 128;
	result = m_PrimitiveLine.CreateVertexBuffer(pOGLDevice, vertexNum);
	if (result != 0) goto EXIT;

	//インデックスバッファ生成：1サークル128辺 * 2(始点/終点)
	indexNum = 128 * 2;
	result = m_PrimitiveLine.CreateIndexBuffer(pOGLDevice, indexNum);
	if (result != 0) goto EXIT;

	//バッファのロック
	result = m_PrimitiveLine.LockVertex((void**)&pVertex);
	if (result != 0) goto EXIT;
	result = m_PrimitiveLine.LockIndex(&pIndex);
	if (result != 0) goto EXIT;

	//バッファに頂点とインデックスを書き込む
	result = _CreateVertexOfIndicatorLine(pVertex, pIndex);
	if (result != 0) goto EXIT;

	//バッファのロック解除
	result = m_PrimitiveLine.UnlockVertex();
	if (result != 0) goto EXIT;
	result = m_PrimitiveLine.UnlockIndex();
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// 移動
//******************************************************************************
int MTTimeIndicatorRing::Transform(
		OGLDevice* pOGLDevice,
		OGLVECTOR3 camVector,
		float rollAngle
	)
{
	int result = 0;
	OGLVECTOR3 moveVector;
	OGLTransMatrix transMatrix;

	//回転
	transMatrix.RegistRotationX(rollAngle);

	//演奏位置
	m_CurPos = m_NoteDesign.GetPlayPosX(m_CurTickTime);

	//移動行列
	moveVector = m_NoteDesign.GetWorldMoveVector();
	transMatrix.RegistTranslation(moveVector.x + m_CurPos, moveVector.y, moveVector.z);
	
	//変換行列設定
	m_PrimitiveLine.Transform(&transMatrix);

	return result;
}

//******************************************************************************
// 描画
//******************************************************************************
int MTTimeIndicatorRing::Draw(
		OGLDevice* pOGLDevice
	)
{
	int result = 0;
	
	if (!m_isEnable) goto EXIT;
	
	result = m_PrimitiveLine.Draw(pOGLDevice);
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// 解放
//******************************************************************************
void MTTimeIndicatorRing::Release()
{
	m_PrimitiveLine.Release();
}

//******************************************************************************
// タイムインジケータライン頂点生成
//******************************************************************************
int MTTimeIndicatorRing::_CreateVertexOfIndicatorLine(
		MTTIMEINDICATOR_VERTEX* pVertex,
		unsigned int* pIndex
	)
{
	int result = 0;
	unsigned int i = 0;
	unsigned int virtexIndex = 0;
	unsigned int virtexIndexStart = 0;
	OGLVECTOR3 basePos;
	OGLVECTOR3 rotatedPos;
	float angle = 0.0f;
	
	//基準座標
	m_NoteDesign.GetGridRingBasePos(0, &basePos);
	
	//頂点作成
	virtexIndexStart = virtexIndex;
	pVertex[virtexIndex].p = basePos;
	pVertex[virtexIndex].n = OGLVECTOR3(-1.0f, 0.0f, 0.0f);
	pVertex[virtexIndex].c = m_NoteDesign.GetGridLineColor();
	for (i = 1; i < 128; i++) {
		virtexIndex++;
		
		//回転後の頂点
		angle = (360.0f / 128.0f) * (float)i;
		rotatedPos = OGLH::RotateYZ(0.0f, 0.0f, basePos, angle);
		pVertex[virtexIndex].p = rotatedPos;
		pVertex[virtexIndex].n = OGLVECTOR3(-1.0f, 0.0f, 0.0f);
		pVertex[virtexIndex].c = m_NoteDesign.GetGridLineColor();
		
		//インデックスバッファ（前回の頂点から今回の頂点）
		pIndex[(virtexIndex - 1) * 2]     = virtexIndex - 1;
		pIndex[(virtexIndex - 1) * 2 + 1] = virtexIndex;
	}
	//終点と始点をつなぐ線
	pIndex[virtexIndex * 2]     = virtexIndex;
	pIndex[virtexIndex * 2 + 1] = virtexIndexStart;

	return result;
}

//******************************************************************************
// チックタイム設定
//******************************************************************************
void MTTimeIndicatorRing::SetCurTickTime(
		unsigned int curTickTime
	)
{
	m_CurTickTime = curTickTime;
	m_CurPos = m_NoteDesign.GetPlayPosX(m_CurTickTime);
}

//******************************************************************************
// リセット
//******************************************************************************
void MTTimeIndicatorRing::Reset()
{
	m_CurTickTime = 0;
	m_CurPos = 0.0f;
}

//******************************************************************************
// 現在位置取得
//******************************************************************************
float MTTimeIndicatorRing::GetPos()
{
	return m_CurPos;
}

//******************************************************************************
// 移動ベクトル取得
//******************************************************************************
OGLVECTOR3 MTTimeIndicatorRing::GetMoveVector()
{
	return OGLVECTOR3(m_CurPos, 0.0f, 0.0f);
}

//******************************************************************************
// 表示設定
//******************************************************************************
void MTTimeIndicatorRing::SetEnable(
		bool isEnable
	)
{
	m_isEnable = isEnable;
}

