//******************************************************************************
//
// MIDITrail / MTViewActionViewCtrl
//
// Viewアクションビュー制御クラス
//
// Copyright (C) 2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "MTViewActionViewCtrl.h"


//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTViewActionViewCtrl ()

//テーブルセル作成：標準視点
- (UITableViewCell*)makeStarndardViewpointCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル作成：私の視点
- (UITableViewCell*)makeMyViewpointCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル作成：私の視点保存
- (UITableViewCell*)makeSaveMyViewpointCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル作成：回転
- (UITableViewCell*)makeRotationCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル作成：ビュースタイル
- (UITableViewCell*)makeViewStyleCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル作成：スクリーン
- (UITableViewCell*)makeScreenCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：固定視点
- (void)onSelectStandardViewpointCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：私の視点
- (void)onSelectMyViewpointCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：私の視点保存
- (void)onSelectSaveMyViewpointCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：回転
- (void)onSelectRotationCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：ビュースタイル
- (void)onSelectViewStyleCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：スクリーン
- (void)onSelectScreenCellForIndexPath:(NSIndexPath*)indexPath;

//通知送信処理
- (int)postNotificationWithActionName:(NSString*)pName;

@end


@implementation MTViewActionViewCtrl

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
    [super viewDidLoad];
		
	//テーブルビュー生成
	m_pTableView =  [[UITableView alloc] initWithFrame:CGRectMake(0, 0, 100, 100) style:UITableViewStyleGrouped];
	m_pTableView.delegate = self;
	m_pTableView.dataSource = self;
	
	//テーブルビュー登録
    [self.view addSubview:m_pTableView];
	
	return;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{
	//ポップオーバービューのフレームサイズに合わせる（吹き出しの分だけサイズが大きくなる）
	m_pTableView.frame = self.view.frame;
	
	//テーブルビュー再表示
	[m_pTableView reloadData];
	
	return;
}

//******************************************************************************
// ビュー非表示
//******************************************************************************
- (void)viewDidDisappear:(BOOL)animated
{
	[self postNotificationWithActionName:@"DidDisappear"];
	return;
}

//******************************************************************************
// スクリーンロック状態設定
//******************************************************************************
- (void)setScreenLockStatus:(BOOL)isLocked
{
	m_isScreenLocked = isLocked;
	return;
}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
	//セクション
	// 0. Standard Viewpoint
	// 1. My Viewpoint
	// 2. Save My Viewpoint
	// 3. Rotation
	// 4. View Style
	// 5. Screen
	
	return 6;
}

//******************************************************************************
// セクションヘッダ
//******************************************************************************
- (NSString*)tableView:(UITableView*)tableView titleForHeaderInSection:(NSInteger)section
{
	NSString* pSectionHeader = nil;
	
	switch(section) {
		case 0:
			pSectionHeader = @"Standard Viewpoint";
			break;
		case 1:
			pSectionHeader = @"My Viewpoint";
			break;
		case 2:
			pSectionHeader = @"Save My Viewpoint";
			break;
		case 3:
			pSectionHeader = @"Rotation";
			break;
		case 4:
			pSectionHeader = @"View Style";
			break;
		case 5:
			pSectionHeader = @"Screen";
			break;
		default:
			break;
	}
	
	return pSectionHeader;
}

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	
	//NSLog(@"section %d", section);
	
	switch (section) {
		case 0:
			//Standard Viewpoint
			numOfRows = 3;
			break;
		case 1:
			//My Viewpoint
			numOfRows = 3;
			break;
		case 2:
			//Save My Viewpoint
			numOfRows = 3;
			break;
		case 3:
			//Rotation
			numOfRows = 1;
			break;
		case 4:
			//View Style
			numOfRows = 3;
			break;
		case 5:
			//Screen
			numOfRows = 1;
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		 cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//Standard Viewpoint
			pCell = [self makeStarndardViewpointCellForIndexPath:indexPath];
			break;
		case 1:
			//My Viewpoint
			pCell = [self makeMyViewpointCellForIndexPath:indexPath];
			break;
		case 2:
			//Save My Viewpoint
			pCell = [self makeSaveMyViewpointCellForIndexPath:indexPath];
			break;
		case 3:
			//Rotation
			pCell = [self makeRotationCellForIndexPath:indexPath];
			break;
		case 4:
			//View Style
			pCell = [self makeViewStyleCellForIndexPath:indexPath];
			break;
		case 5:
			//Screen
			pCell = [self makeScreenCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：標準視点
//******************************************************************************
- (UITableViewCell*)makeStarndardViewpointCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	
	//再利用可能セル生成
	pCellIdentifier = [NSString stringWithFormat:@"MTViewActionViewCtrl_StandardViewpoint_%ld", (long)indexPath.row];
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	switch (indexPath.row) {
		case 0:
			pCell.textLabel.text = @"Viewpoint 1";
			break;
		case 1:
			pCell.textLabel.text = @"Viewpoint 2";
			break;
		case 2:
			pCell.textLabel.text = @"Viewpoint 3";
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：私の視点
//******************************************************************************
- (UITableViewCell*)makeMyViewpointCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	
	//再利用可能セル生成
	pCellIdentifier = [NSString stringWithFormat:@"MTViewActionViewCtrl_MyViewpoint_%ld", (long)indexPath.row];
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	switch (indexPath.row) {
		case 0:
			pCell.textLabel.text = @"My Viewpoint 1";
			break;
		case 1:
			pCell.textLabel.text = @"My Viewpoint 2";
			break;
		case 2:
			pCell.textLabel.text = @"My Viewpoint 3";
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：私の視点保存
//******************************************************************************
- (UITableViewCell*)makeSaveMyViewpointCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	
	//再利用可能セル生成
	pCellIdentifier = [NSString stringWithFormat:@"MTViewActionViewCtrl_SaveMyViewpoint_%ld", (long)indexPath.row];
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	switch (indexPath.row) {
		case 0:
			pCell.textLabel.text = @"Save My Viewpoint 1";
			break;
		case 1:
			pCell.textLabel.text = @"Save My Viewpoint 2";
			break;
		case 2:
			pCell.textLabel.text = @"Save My Viewpoint 3";
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：回転
//******************************************************************************
- (UITableViewCell*)makeRotationCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	
	//再利用可能セル生成
	pCellIdentifier = [NSString stringWithFormat:@"MTViewActionViewCtrl_Rotation_%ld", (long)indexPath.row];
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	switch (indexPath.row) {
		case 0:
			pCell.textLabel.text = @"Switch Rotation Mode";
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：ビュースタイル
//******************************************************************************
- (UITableViewCell*)makeViewStyleCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	
	//再利用可能セル生成
	pCellIdentifier = [NSString stringWithFormat:@"MTViewActionViewCtrl_ViewStyle_%ld", (long)indexPath.row];
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	switch (indexPath.row) {
		case 0:
			pCell.textLabel.text = @"Standard View";
			break;
		case 1:
			pCell.textLabel.text = @"VR View (Single Lens)";
			break;
		case 2:
			pCell.textLabel.text = @"VR View (Dual Lens)";
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：スクリーン
//******************************************************************************
- (UITableViewCell*)makeScreenCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = nil;
	UITableViewCell* pCell = nil;
	UISwitch* pSwitch = nil;
	
	//再利用可能セル生成
	pCellIdentifier = [NSString stringWithFormat:@"MTViewActionViewCtrl_Screen_%ld", (long)indexPath.row];
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
									   reuseIdentifier:pCellIdentifier];
		pSwitch = [[UISwitch alloc] initWithFrame:CGRectZero];
		pSwitch.userInteractionEnabled = NO; //ユーザ操作を透過
		pCell.accessoryView = pSwitch;
	}
	
	//ラベル設定
	switch (indexPath.row) {
		case 0:
			pCell.textLabel.text = @"Lock";
			break;
		default:
			break;
	}
	
	//スイッチ設定
	pSwitch = (UISwitch*)pCell.accessoryView;
	[pSwitch setOn:m_isScreenLocked animated:YES];
	
	return pCell;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
	[tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//Standard Viewpoint
			[self onSelectStandardViewpointCellForIndexPath:indexPath];
			break;
		case 1:
			//My Viewpoint
			[self onSelectMyViewpointCellForIndexPath:indexPath];
			break;
		case 2:
			//Save My Viewpoint
			[self onSelectSaveMyViewpointCellForIndexPath:indexPath];
			break;
		case 3:
			//Rotation
			[self onSelectRotationCellForIndexPath:indexPath];
			break;
		case 4:
			//View Style
			[self onSelectViewStyleCellForIndexPath:indexPath];
			break;
		case 5:
			//Screen
			[self onSelectScreenCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	[tableView performSelector:@selector(reloadData) withObject:nil afterDelay:0.1f];
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：固定視点
//******************************************************************************
- (void)onSelectStandardViewpointCellForIndexPath:(NSIndexPath*)indexPath
{
	//NSLog(@"selected %d", indexPath.row);
	
	switch (indexPath.row) {
		case 0:
			//Viewpoint 1
			[self postNotificationWithActionName:@"Viewpoint1"];
			break;
		case 1:
			//Viewpoint 2
			[self postNotificationWithActionName:@"Viewpoint2"];
			break;
		case 2:
			//Viewpoint 3
			[self postNotificationWithActionName:@"Viewpoint3"];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：私の視点
//******************************************************************************
- (void)onSelectMyViewpointCellForIndexPath:(NSIndexPath*)indexPath
{
	//NSLog(@"selected %d", indexPath.row);
	
	switch (indexPath.row) {
		case 0:
			//My Viewpoint 1
			[self postNotificationWithActionName:@"MyViewpoint1"];
			break;
		case 1:
			//My Viewpoint 2
			[self postNotificationWithActionName:@"MyViewpoint2"];
			break;
		case 2:
			//My Viewpoint 3
			[self postNotificationWithActionName:@"MyViewpoint3"];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：私の視点保存
//******************************************************************************
- (void)onSelectSaveMyViewpointCellForIndexPath:(NSIndexPath*)indexPath
{
	//NSLog(@"selected %d", indexPath.row);
	
	switch (indexPath.row) {
		case 0:
			//Save My Viewpoint 1
			[self postNotificationWithActionName:@"SaveMyViewpoint1"];
			break;
		case 1:
			//Save My Viewpoint 2
			[self postNotificationWithActionName:@"SaveMyViewpoint2"];
			break;
		case 2:
			//Save My Viewpoint 3
			[self postNotificationWithActionName:@"SaveMyViewpoint3"];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：回転
//******************************************************************************
- (void)onSelectRotationCellForIndexPath:(NSIndexPath*)indexPath
{
	//NSLog(@"selected %d", indexPath.row);
	
	switch (indexPath.row) {
		case 0:
			//Switch Rotation Mode
			[self postNotificationWithActionName:@"SwitchRotationMode"];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：ビュースタイル
//******************************************************************************
- (void)onSelectViewStyleCellForIndexPath:(NSIndexPath*)indexPath
{
	//NSLog(@"selected %d", indexPath.row);
	
	switch (indexPath.row) {
		case 0:
			//Standard View
			[self postNotificationWithActionName:@"ViewStyleStandardView"];
			break;
		case 1:
			//VR View (Single Lens)
			[self postNotificationWithActionName:@"ViewStyleVRViewSingleLens"];
			break;
		case 2:
			//VR View (Dual Lens)
			[self postNotificationWithActionName:@"ViewStyleVRViewDualLens"];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：スクリーン
//******************************************************************************
- (void)onSelectScreenCellForIndexPath:(NSIndexPath*)indexPath
{
	//NSLog(@"selected %d", indexPath.row);
	
	switch (indexPath.row) {
		case 0:
			//Lock
			if (m_isScreenLocked) {
				[self postNotificationWithActionName:@"ScreenUnlock"];
				m_isScreenLocked = NO;
			}
			else {
				[self postNotificationWithActionName:@"ScreenLock"];
				m_isScreenLocked = YES;
			}
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// 通知送信処理
//******************************************************************************
- (int)postNotificationWithActionName:(NSString*)pName
{
	int result = 0;
	NSNotification* pNotification = nil;
	NSNotificationCenter* pCenter = nil;
	NSDictionary* pDictionary = nil;
	
	//ユーザ情報作成
	pDictionary = [NSDictionary dictionaryWithObjectsAndKeys:pName, @"ActionName", nil];
	
	//通知オブジェクト作成
	pNotification = [NSNotification notificationWithName:MT_NOTIFICATION_ON_VIEW_ACTION
												  object:self
												userInfo:pDictionary];
	//通知する
	pCenter = [NSNotificationCenter defaultCenter];
	
	//通知に対応する処理を演奏スレッドで処理させる場合
	//[pCenter postNotification:pNotification];
	
	//通知に対応する処理をメインスレッドに処理させる場合
	[pCenter performSelectorOnMainThread:@selector(postNotification:)
							  withObject:pNotification
						   waitUntilDone:NO];
	
	return result;
}

//******************************************************************************
// タッチイベント：開始
//******************************************************************************
- (void)touchesBegan:(NSSet*)touches withEvent:(UIEvent *)event
{
	//メインビューにタッチイベントを伝搬させないようにオーバライドする
	return;
}

//******************************************************************************
// タッチイベント：移動
//******************************************************************************
- (void)touchesMoved:(NSSet*)touches withEvent:(UIEvent *)event
{
	//メインビューにタッチイベントを伝搬させないようにオーバライドする
	return;
}

//******************************************************************************
// タッチイベント：終了
//******************************************************************************
- (void)touchesEnded:(NSSet*)touches withEvent:(UIEvent *)event
{
	//メインビューにタッチイベントを伝搬させないようにオーバライドする
	return;
}

//******************************************************************************
// タッチイベント：キャンセル
//******************************************************************************
- (void)touchesCancelled:(NSSet*)touches withEvent:(UIEvent *)event
{
	//メインビューにタッチイベントを伝搬させないようにオーバライドする
	return;
}

@end

