//******************************************************************************
//
// OGL Utility / OGLDevice
//
// デバイスクラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "OGLDevice.h"
#import <string.h>  //for memset


//******************************************************************************
// コンストラクタ
//******************************************************************************
OGLDevice::OGLDevice(void)
{
	//Metalデバイス
	m_MetalDevice = nil;
	m_Encoder = nil;
	
	//ビューパラメータ
	m_SampleCount = 1;
	memset(&m_ViewPort, 0, sizeof(OGLVIEWPORT));
	
	//制御情報初期化
	InitUniforms();
	
	return;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
OGLDevice::~OGLDevice(void)
{
}

//******************************************************************************
// 初期化
//******************************************************************************
int OGLDevice::Initialize(id <MTLDevice> metalDevice)
{
	m_MetalDevice = metalDevice;
	
	return 0;
}

//******************************************************************************
// 破棄
//******************************************************************************
int OGLDevice::Release()
{
	return 0;
}

//******************************************************************************
// Metalデバイス取得
//******************************************************************************
id <MTLDevice> OGLDevice::GetMetalDevice()
{
	return m_MetalDevice;
}

//******************************************************************************
// サンプル数設定
//******************************************************************************
void OGLDevice::SetSampleCount(NSUInteger count)
{
	m_SampleCount = count;
}

//******************************************************************************
// サンプル数取得
//******************************************************************************
NSUInteger OGLDevice::GetSampleCount()
{
	return m_SampleCount;
}

//******************************************************************************
// ビューポート設定
//******************************************************************************
void OGLDevice::SetViewPort(OGLVIEWPORT viewPort)
{
	m_ViewPort = viewPort;
}

//******************************************************************************
// ビューポート取得
//******************************************************************************
void OGLDevice::GetViewPort(OGLVIEWPORT* pViewPort)
{
	*pViewPort = m_ViewPort;
}

//******************************************************************************
// モデル行列設定
//******************************************************************************
void OGLDevice::SetModelMatrix(simd::float4x4 matrix)
{
	m_ModelMatrix = matrix;
}

//******************************************************************************
// ビュー行列設定
//******************************************************************************
void OGLDevice::SetViewMatrix(simd::float4x4 matrix)
{
	m_ViewMatrix = matrix;
}

//******************************************************************************
// ビュー行列取得
//******************************************************************************
simd::float4x4 OGLDevice::GetViewMatrix()
{
	return m_ViewMatrix;
}

//******************************************************************************
// プロジェクション行列設定
//******************************************************************************
void OGLDevice::SetProjectionMatrix(simd::float4x4 matrix)
{
	m_ProjectionMatrix = matrix;
}

//******************************************************************************
// マテリアル設定
//******************************************************************************
void OGLDevice::SetMaterial(OGLMaterial material)
{
	m_Material = material;
}

//******************************************************************************
//ポイントサイズ設定
//******************************************************************************
void OGLDevice::SetPointSize(float pointSize)
{
	m_PointSize = pointSize;
}

//******************************************************************************
// ライト有無設定
//******************************************************************************
void OGLDevice::SetEnableLights(bool isEnable)
{
	m_EnableLights = isEnable ? 1 : 0;
}

//******************************************************************************
// ディレクショナルライト0設定
//******************************************************************************
void OGLDevice::SetLight0(OGLLight light)
{
	m_Light0 = light;
}

//******************************************************************************
// ディレクショナルライト1設定
//******************************************************************************
void OGLDevice::SetLight1(OGLLight light)
{
	m_Light1 = light;;
}

//******************************************************************************
// 制御情報初期化
//******************************************************************************
void OGLDevice::InitUniforms()
{
	m_ModelMatrix = matrix_identity_float4x4;
	m_ViewMatrix = matrix_identity_float4x4;
	m_ProjectionMatrix = matrix_identity_float4x4;
	memset(&m_Material, 0, sizeof(OGLMaterial));
	m_PointSize = 1.0f;
	m_EnableLights = 0;
	memset(&m_Light0, 0, sizeof(OGLLight));
	memset(&m_Light1, 0, sizeof(OGLLight));
	
	return;
}

//******************************************************************************
// 制御情報取得
//******************************************************************************
OGLUniforms OGLDevice::GetUniforms()
{
	OGLUniforms uniforms;
	
	uniforms.modelViewProjectionMatrix = matrix_multiply(m_ProjectionMatrix, matrix_multiply(m_ViewMatrix, m_ModelMatrix));
	uniforms.modelViewMatrix = matrix_multiply(m_ViewMatrix, m_ModelMatrix);
	uniforms.normalMatrix = matrix_transpose(matrix_invert(uniforms.modelViewMatrix));
	uniforms.material = m_Material;
	uniforms.enableLights = m_EnableLights;
	uniforms.light0 = m_Light0;
	uniforms.light1 = m_Light1;
	uniforms.pointSize = m_PointSize;
	
	//ライト方向をビュー座標系に変換
	float4 lightDir0 = { m_Light0.direction.x, m_Light0.direction.y, m_Light0.direction.z, 0.0f };
	float4 lightDir1 = { m_Light1.direction.x, m_Light1.direction.y, m_Light1.direction.z, 0.0f };
	uniforms.light0.direction  = (m_ViewMatrix * lightDir0).xyz;
	uniforms.light1.direction  = (m_ViewMatrix * lightDir1).xyz;
	
	return uniforms;
}

//******************************************************************************
// コマンドエンコーダ設定
//******************************************************************************
void OGLDevice::SetCurrentCommandEncoder(id <MTLRenderCommandEncoder> encoder)
{
	m_Encoder = encoder;
}

//******************************************************************************
// コマンドエンコーダ取得
//******************************************************************************
id <MTLRenderCommandEncoder> OGLDevice::GetCurrentCommandEncoder()
{
	return m_Encoder;
}

