//******************************************************************************
//
// OGL Utility / OGLRenderer
//
// レンダラクラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#include "YNBaseLib.h"
#include "OGLRenderer.h"
#include <new>


//******************************************************************************
// コンストラクタ
//******************************************************************************
OGLRenderer::OGLRenderer()
{
	m_pOGLDevice = NULL;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
OGLRenderer::~OGLRenderer()
{
	Terminate();
}

//******************************************************************************
// 初期化
//******************************************************************************
int OGLRenderer::Initialize(
		MTKView* pView,
		OGLRedererParam rendererParam
	)
{
	int result = 0;
	OGLVIEWPORT viewPort;
	
	//ディスプレイデバイス作成
	try {
		m_pOGLDevice = new OGLDevice();
	}
	catch (std::bad_alloc) {
		result = YN_SET_ERR(@"Could not allocate memory.", 0, 0);
		goto EXIT;
	}
	
	//ディスプレイデバイス初期化
	result = m_pOGLDevice->Initialize(pView.device);
	if (result != 0) goto EXIT;
	
	//セマフォ生成：複数同時実行を無効とする
	m_DisplaySemaphore = dispatch_semaphore_create(1);
	
	//ビューポート情報作成
	viewPort.originx = 0.0f;
	viewPort.originy = 0.0f;
	viewPort.width   = pView.drawableSize.width;
	viewPort.height  = pView.drawableSize.height;
	m_pOGLDevice->SetViewPort(viewPort);
	
	//ビュープロパティ設定
	pView.colorPixelFormat = MTLPixelFormatBGRA8Unorm;
	pView.depthStencilPixelFormat = MTLPixelFormatDepth32Float_Stencil8;
	pView.clearColor = MTLClearColorMake(0, 0, 0, 1);
	pView.clearDepth = 1.0;
	
	//サンプル数設定確認
	m_pOGLDevice->SetSampleCount(pView.sampleCount);
	
	//コマンドキュー生成
	m_CommandQueue = [pView.device newCommandQueue];
	
EXIT:;
	return result;
}

//******************************************************************************
// デバイス取得
//******************************************************************************
OGLDevice* OGLRenderer::GetDevice()
{
	return m_pOGLDevice;
}

//*****************************************************************************
// シーン初期化
//******************************************************************************
int OGLRenderer::InitScene()
{
	int result = 0;
	
	//制御情報初期化
	m_pOGLDevice->InitUniforms();
	
EXIT:;
	return result;
}

//*****************************************************************************
// シーン描画
//******************************************************************************
int OGLRenderer::RenderScene(
		 MTKView* pView,
		 OGLScene* pScene,
		 bool isVRViewDualLens,	//VR View(Dual Lens)描画フラグ
		 CGFloat ipdPoints		//瞳孔間距離ポイント数
	)
{
	int result = 0;
	id <MTLCommandBuffer> commandBuffer = nil;
	id <MTLRenderCommandEncoder> commandEncoder = nil;
	MTLRenderPassDescriptor* renderPassDescriptor = nil;
	OGLCOLOR bgcolor;
	__block dispatch_semaphore_t block_semaphore;
	
	//NSLog(@"RenderScene start");
	
	if (pScene == NULL) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	
	//バッファ初期化色の指定：RGBA
	bgcolor = pScene->GetBGColor();
	pView.clearColor = MTLClearColorMake(bgcolor.r, bgcolor.g, bgcolor.b, bgcolor.a);
	
	//セマフォ割り当て待機
	dispatch_semaphore_wait(m_DisplaySemaphore, DISPATCH_TIME_FOREVER);
	
	//コマンドバッファ生成
	commandBuffer = [m_CommandQueue commandBuffer];
	commandBuffer.label = @"OGLRenderer_CommandBuffer";
	
	//コマンドバッファに完了時のイベントハンドラを登録
	block_semaphore = m_DisplaySemaphore;
	[commandBuffer addCompletedHandler:^(id <MTLCommandBuffer> buffer) {
		dispatch_semaphore_signal(block_semaphore);
	}];
	
	//ドロウアブル取得
	renderPassDescriptor = pView.currentRenderPassDescriptor;
	if ((renderPassDescriptor != nil)) {
		//コマンドバッファからレンダーエンコーダを取得
		commandEncoder = [commandBuffer renderCommandEncoderWithDescriptor:renderPassDescriptor];
		commandEncoder.label = @"OGLRenderer_CommandEncoder";
		[commandEncoder pushDebugGroup:@"OGLRenderer_RenderScene"];
		m_pOGLDevice->SetCurrentCommandEncoder(commandEncoder);
		
		//描画
		result = _DrawScene(pView, pScene, isVRViewDualLens, ipdPoints);
		if (result != 0) {
			//失敗しても描画定型処理を続行する
			YN_SHOW_ERR();
		}
		
		//エンコード終了
		m_pOGLDevice->SetCurrentCommandEncoder(nil);
		[commandEncoder popDebugGroup];
		[commandEncoder endEncoding];
		
		//フレームへの描画を依頼
		[commandBuffer presentDrawable:pView.currentDrawable];
	}
	
	//コマンドバッファコミット
	[commandBuffer commit];
	
EXIT:;
	//NSLog(@"RenderScene end");
	return result;
}

//*****************************************************************************
// シーン描画
//******************************************************************************
int OGLRenderer::_DrawScene(
		MTKView* pView,
		OGLScene* pScene,
		bool isVRViewDualLens,	//VR View(Dual Lens)描画フラグ
		CGFloat ipdPoints		//瞳孔間距離ポイント数
	)
{
	int result = 0;
	CGRect baseRect;
	OGLVIEWPORT viewPort;
	CGFloat ipdSize = 0.0f;
	
	baseRect.size = pView.drawableSize;
	baseRect.origin = CGPointZero;
	ipdSize = ipdPoints * [UIScreen mainScreen].scale;
	
	//描画
	if (pScene != NULL) {
		//VRビュー
		if (isVRViewDualLens) {
			//瞳孔間距離が画面横幅の1/2より大きい場合または瞳孔間距離が無効な場合
			if ((ipdSize > (baseRect.size.width / 2)) || (ipdSize <= 0)) {
				ipdSize = baseRect.size.width / 2;
			}
			
			//左目用の描画
			viewPort.originx = baseRect.origin.x + (baseRect.size.width / 2) - ipdSize;
			viewPort.originy = baseRect.origin.y;
			viewPort.width   = ipdSize;
			viewPort.height  = baseRect.size.height;
			m_pOGLDevice->SetViewPort(viewPort);
			result = pScene->Draw(m_pOGLDevice, OGLScene::CameraLeftEye);
			if (result != 0) goto EXIT;
			
			//右目用の描画
			viewPort.originx = baseRect.origin.x + (baseRect.size.width / 2);
			viewPort.originy = baseRect.origin.y;
			viewPort.width   = ipdSize;
			viewPort.height  = baseRect.size.height;
			m_pOGLDevice->SetViewPort(viewPort);
			result = pScene->Draw(m_pOGLDevice, OGLScene::CameraRightEye);
			if (result != 0) goto EXIT;
		}
		//通常ビュー
		else {
			viewPort.originx = baseRect.origin.x;
			viewPort.originy = baseRect.origin.y;
			viewPort.width   = baseRect.size.width;
			viewPort.height  = baseRect.size.height;
			m_pOGLDevice->SetViewPort(viewPort);
			result = pScene->Draw(m_pOGLDevice, OGLScene::CameraNormal);
			if (result != 0) goto EXIT;
		}
	}
	
EXIT:;
	return result;
}

//*****************************************************************************
// 終了処理
//******************************************************************************
void OGLRenderer::Terminate()
{
	if (m_pOGLDevice != NULL) {
		m_pOGLDevice->Release();
		m_pOGLDevice = NULL;
	}
}

