/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2008  Kouhei Sutou <kou@cozmixng.org>
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <errno.h>

#include <gcutter.h>
#include <glib/gstdio.h>

#define shutdown inet_shutdown
#include <milter-test-utils.h>
#include <milter/core/milter-reader.h>
#undef shutdown
#include <unistd.h>

void test_reader_io_channel (void);
void test_reader_io_channel_binary (void);
void test_reader_huge_data (void);
void test_io_error (void);
void test_finished_signal (void);
void test_shutdown (void);

static MilterReader *reader;

static GIOChannel *channel;

static gsize actual_read_size;
static GString *actual_read_string;

static guint signal_id;
static guint finished_signal_id;

static GError *actual_error;
static GError *expected_error;

static gboolean finished;

void
setup (void)
{
    signal_id = 0;
    finished_signal_id = 0;

    channel = gcut_string_io_channel_new(NULL);
    g_io_channel_set_encoding(channel, NULL, NULL);

    reader = milter_reader_io_channel_new(channel);
    milter_reader_start(reader);

    actual_read_string = g_string_new(NULL);
    actual_read_size = 0;

    actual_error = NULL;
    expected_error = NULL;

    finished = FALSE;
}

void
teardown (void)
{
    if (reader) {
        if (signal_id > 0)
            g_signal_handler_disconnect(reader, signal_id);
        if (finished_signal_id > 0)
            g_signal_handler_disconnect(reader, finished_signal_id);
        g_object_unref(reader);
    }

    if (channel)
        g_io_channel_unref(channel);

    if (actual_read_string)
        g_string_free(actual_read_string, TRUE);

    if (actual_error)
        g_error_free(actual_error);
    if (expected_error)
        g_error_free(expected_error);
}

static void
cb_flow (MilterReader *reader, const gchar *data, gsize data_size)
{
    g_string_append_len(actual_read_string, data, data_size);
    actual_read_size += data_size;
}

static void
cb_error (MilterErrorEmittable *emittable, GError *error, gsize data_size)
{
    actual_error = g_error_copy(error);
}

static void
cb_finished (MilterFinishedEmittable *emittable)
{
    finished = TRUE;
}

static void
write_data (GIOChannel *channel, const gchar *data, gsize size)
{
    gsize bytes_written;
    GError *error = NULL;

    g_io_channel_write_chars(channel, data, size, &bytes_written, &error);
    gcut_assert_error(error);

    g_io_channel_seek_position(channel, -bytes_written, G_SEEK_CUR, &error);
    gcut_assert_error(error);

    g_main_context_iteration(NULL, FALSE);
}

void
test_reader_io_channel (void)
{
    const gchar data[] = "first\nsecond\nthird\n";

    signal_id = g_signal_connect(reader, "flow", G_CALLBACK(cb_flow), NULL);

    write_data(channel, data, strlen(data));
    cut_assert_equal_memory(data, strlen(data),
                            actual_read_string->str, actual_read_size);
}

void
test_reader_io_channel_binary (void)
{
    const gchar binary_data[4] = {0x00, 0x01, 0x02, 0x03};

    signal_id = g_signal_connect(reader, "flow", G_CALLBACK(cb_flow), NULL);

    write_data(channel, binary_data, 4);
    cut_assert_equal_memory(binary_data, 4,
                            actual_read_string->str, actual_read_size);
}

void
test_reader_huge_data (void)
{
    gchar *binary_data;
    guint data_size = 192 * 8192;
    guint i = 0;
    guint reader_buffer_size = 4096;

    signal_id = g_signal_connect(reader, "flow", G_CALLBACK(cb_flow), NULL);

    binary_data = g_new(gchar, data_size);
    cut_take_memory(binary_data);
    memset(binary_data, '\0', data_size);
    write_data(channel, binary_data, data_size);

    for (i = 0; i < data_size; i += reader_buffer_size)
        g_main_context_iteration(NULL, FALSE);
    cut_assert_equal_memory(binary_data, data_size,
                            actual_read_string->str, actual_read_size);
}

void
test_io_error (void)
{
    gcut_string_io_channel_set_read_fail(channel, TRUE);

    signal_id = g_signal_connect(reader, "error", G_CALLBACK(cb_error), NULL);

    write_data(channel, "first", strlen("first"));

    expected_error = g_error_new(MILTER_READER_ERROR,
                                 MILTER_READER_ERROR_IO_ERROR,
                                 "I/O error: %s:%d: %s",
                                 g_quark_to_string(G_IO_CHANNEL_ERROR),
                                 g_io_channel_error_from_errno(EIO),
                                 g_strerror(EIO));
    gcut_assert_equal_error(expected_error, actual_error);
}

void
test_finished_signal (void)
{
    finished_signal_id = g_signal_connect(reader, "finished",
                                          G_CALLBACK(cb_finished), NULL);
    signal_id = g_signal_connect(reader, "flow", G_CALLBACK(cb_flow), NULL);

    write_data(channel, "first", strlen("first"));
    cut_assert_equal_memory("first", strlen("first"),
                            actual_read_string->str, actual_read_size);
    cut_assert_true(finished);
}

void
test_shutdown (void)
{
    cut_assert_true(milter_reader_is_watching(reader));
    milter_reader_shutdown(reader);
    cut_assert_true(milter_reader_is_watching(reader));
    g_main_context_iteration(NULL, FALSE);
    cut_assert_false(milter_reader_is_watching(reader));
}

/*
vi:ts=4:nowrap:ai:expandtab:sw=4
*/
