/*
    charset
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "misc_charset.h"
#include <gdk/gdkkeysyms.h>


static void misc_charset_class_init (MiscCharsetClass *klass);
static void misc_charset_init       (MiscCharset      *charset);


#if GTK_CHECK_VERSION(2,4,0)
static GtkComboBoxEntryClass *parent_class = NULL;
#else /* not GTK_CHECK_VERSION(2,4,0) */
static GtkComboClass *parent_class = NULL;
#endif /* not GTK_CHECK_VERSION(2,4,0) */


enum
{
  ACTIVATE_SIGNAL,
  EDITED_SIGNAL,
  LAST_SIGNAL
};


static gint misc_charset_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
misc_charset_get_type (void)
{
  static GType misc_charset_type = 0;

  if (!misc_charset_type)
    {
      static const GTypeInfo charset_info =
      {
        sizeof (MiscCharsetClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)misc_charset_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (MiscCharset),
        0,              /* n_preallocs */
        (GInstanceInitFunc)misc_charset_init,
      };

#if GTK_CHECK_VERSION(2,4,0)
      misc_charset_type = g_type_register_static (GTK_TYPE_COMBO_BOX_ENTRY,
                                            "MiscCharset", &charset_info, 0);
#else /* not GTK_CHECK_VERSION(2,4,0) */
      misc_charset_type = g_type_register_static (GTK_TYPE_COMBO,
                                            "MiscCharset", &charset_info, 0);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
    }

  return misc_charset_type;
}


static void
misc_charset_class_init (MiscCharsetClass *klass)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = (GObjectClass*) klass;
  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;

  parent_class = g_type_class_peek_parent (klass);

  misc_charset_signals[ACTIVATE_SIGNAL]
                = g_signal_new ("activate",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                                G_STRUCT_OFFSET (MiscCharsetClass, activate),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);
  misc_charset_signals[EDITED_SIGNAL]
                = g_signal_new ("edited",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                                G_STRUCT_OFFSET (MiscCharsetClass, edited),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);
}


static void
misc_charset_activate (GtkWidget   *widget,
                       MiscCharset *charset)
{
  g_signal_emit_by_name (G_OBJECT (charset), "activate");
}


static void
misc_charset_changed (GtkWidget   *widget,
                      MiscCharset *charset)
{
  g_signal_emit_by_name (G_OBJECT (charset), "edited");
}


#if ! GTK_CHECK_VERSION(2,4,0)
static void
misc_charset_grab_focus (GtkWidget *widget,
                         gpointer   user_data)
{
  gtk_widget_grab_focus (GTK_COMBO (widget)->entry);
}


static gboolean
misc_charset_mnemonic (GtkWidget *widget,
                       gboolean   arg1,
                       gpointer   user_data)
{
  gtk_widget_grab_focus (GTK_COMBO (widget)->entry);
  return TRUE;
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


static void
misc_charset_init (MiscCharset *charset)
{
#if GTK_CHECK_VERSION(2,4,0)
  charset->store = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (charset),
                                            GTK_TREE_MODEL (charset->store));
  gtk_combo_box_entry_set_text_column (GTK_COMBO_BOX_ENTRY (charset), 0);
  g_signal_connect (G_OBJECT (GTK_BIN (charset)->child), "activate",
                                G_CALLBACK (misc_charset_activate), charset);
  g_signal_connect (G_OBJECT (GTK_BIN (charset)->child), "changed",
                                G_CALLBACK (misc_charset_changed), charset);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_combo_set_value_in_list (GTK_COMBO (charset), FALSE, FALSE);
  gtk_combo_set_case_sensitive (GTK_COMBO (charset), TRUE);
  g_signal_connect (G_OBJECT (GTK_COMBO (charset)->entry), "activate",
                                G_CALLBACK (misc_charset_activate), charset);
  g_signal_connect (G_OBJECT (GTK_COMBO (charset)->entry), "changed",
                                G_CALLBACK (misc_charset_changed), charset);
  g_signal_connect (G_OBJECT (charset), "focus",
                                G_CALLBACK (misc_charset_grab_focus), NULL);
  g_signal_connect (G_OBJECT (charset), "mnemonic-activate",
                                G_CALLBACK (misc_charset_mnemonic), NULL);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
}


/******************************************************************************
*                                                                             *
* ja:文字符号化方式関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
misc_charset_new (void)
{
  return GTK_WIDGET (g_object_new (MISC_TYPE_CHARSET, NULL));
}


/*  ja:文字符号化方式を取得する
    charset,ウィジェット
        RET,文字符号化方式                                                  */
gchar *
misc_charset_get_charset (MiscCharset *charset)
{
#if GTK_CHECK_VERSION(2,4,0)
  return charset ? g_strdup (gtk_entry_get_text
                            (GTK_ENTRY (GTK_BIN (charset)->child))) : NULL;
#else /* not GTK_CHECK_VERSION(2,4,0) */
  return charset ? g_strdup (gtk_entry_get_text
                            (GTK_ENTRY (GTK_COMBO (charset)->entry))) : NULL;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
}


/*  ja:文字符号化方式を設定する
    charset,ウィジェット
        str,文字符号化方式
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
misc_charset_set_charset (MiscCharset *charset,
                          const gchar *str)
{
  if (!charset)
    return FALSE;
#if GTK_CHECK_VERSION(2,4,0)
  gtk_entry_set_text (GTK_ENTRY (GTK_BIN (charset)->child), str ? str : "");
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (charset)->entry), str ? str : "");
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  return TRUE;

}


/*  ja:文字符号化方式の候補を取得する
    charset,ウィジェット
        RET,登録されている文字符号化方式の候補                              */
gchar *
misc_charset_get_candidate (MiscCharset *charset)
{
  gchar *candidate = NULL;
  gint i;
#if ! GTK_CHECK_VERSION(2,4,0)
  GList *glist;
#endif /* not GTK_CHECK_VERSION(2,4,0) */

  if (!charset)
    return NULL;
#if GTK_CHECK_VERSION(2,4,0)
  for (i = gtk_tree_model_iter_n_children
                    (GTK_TREE_MODEL (charset->store), NULL) - 1; i >= 0; i--)
    {
      gchar *str, *tmp;
      GtkTreeIter iter;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charset->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (charset->store), &iter, 0, &tmp, -1);
      str = g_strjoin (",", tmp, candidate, NULL);
      g_free (tmp);
      g_free (candidate);
      candidate = str;
    }
#else /* not GTK_CHECK_VERSION(2,4,0) */
  glist = gtk_container_children (GTK_CONTAINER (GTK_COMBO (charset)->list));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      gchar *str;

      str = g_strjoin (",", gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                    (GTK_BIN (g_list_nth_data (glist, i))))), candidate, NULL);
      g_free (candidate);
      candidate = str;
    }
   g_list_free (glist);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
   return candidate;
}


/*  ja:文字符号化方式の候補を追加する
      charset,ウィジェット
    candidate,文字符号化方式の候補
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
misc_charset_add_candidate (MiscCharset *charset,
                            const gchar *candidate)
{
  gboolean result = TRUE;
  gchar **str;
  gint i;

  if (!charset || !candidate)
    return FALSE;
  str = g_strsplit (candidate, ",", G_MAXINT);
  for (i = 0; str[i]; i++)
    if (charuty_is_valid (str[i]))
      {
        gint j;

#if GTK_CHECK_VERSION(2,4,0)
        GtkTreeIter iter;

        for (j = gtk_tree_model_iter_n_children
                    (GTK_TREE_MODEL (charset->store), NULL) - 1; j >= 0; j--)
          {
            gchar *tmp;
            gint cmp;

            gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charset->store),
                                                            &iter, NULL, j);
            gtk_tree_model_get (GTK_TREE_MODEL (charset->store), &iter,
                                                                0, &tmp, -1);
            cmp = g_strcasecmp (tmp, str[i]);
            g_free (tmp);
            if (cmp == 0)
              {
                gtk_list_store_remove (charset->store, &iter);
                j--;
              }
            if (cmp <= 0)
              break;
          }
        gtk_list_store_insert (charset->store, &iter, j + 1);
        gtk_list_store_set (charset->store, &iter, 0, str[i], -1);
#else /* not GTK_CHECK_VERSION(2,4,0) */
        GList *glist;
        GtkWidget *item;

        glist = gtk_container_children
                                (GTK_CONTAINER (GTK_COMBO (charset)->list));
        for (j = g_list_length (glist) - 1; j >= 0; j--)
          {
            gint cmp;

            cmp = g_strcasecmp (gtk_label_get_text (GTK_LABEL
                (gtk_bin_get_child (GTK_BIN (g_list_nth_data (glist, j))))),
                                                                    str[i]);
            if (cmp == 0)
              {
                gtk_list_clear_items (GTK_LIST (GTK_COMBO (charset)->list),
                                                                    j, j + 1);
                j--;
              }
            if (cmp <= 0)
              break;
          }
        g_list_free (glist);
        item = gtk_list_item_new_with_label (str[i]);
        gtk_widget_show (item);
        gtk_list_insert_items (GTK_LIST (GTK_COMBO (charset)->list),
                                            g_list_append (NULL, item), j + 1);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      }
    else
      {
        result = FALSE;
      }
  g_strfreev (str);
  return result;
}


/*  ja:文字符号化方式の候補を削除する
      charset,ウィジェット
    candidate,文字符号化方式の候補,NULL:すべて削除
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
misc_charset_remove_candidate (MiscCharset *charset,
                               const gchar *candidate)
{
  gboolean result = TRUE;
  gchar **str;
  gint i;

  if (!charset)
    return FALSE;
  if (candidate)
    {
      str = g_strsplit (candidate, ",", G_MAXINT);
      for (i = 0; str[i]; i++)
        {
          gint j;

#if GTK_CHECK_VERSION(2,4,0)
          for (j = gtk_tree_model_iter_n_children
                    (GTK_TREE_MODEL (charset->store), NULL) - 1; j >= 0; j--)
            {
              gchar *tmp;
              gint cmp;
              GtkTreeIter iter;

              gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (charset->store),
                                                            &iter, NULL, j);
              gtk_tree_model_get (GTK_TREE_MODEL (charset->store), &iter,
                                                                0, &tmp, -1);
              cmp = g_strcasecmp (tmp, str[i]);
              g_free (tmp);
              if (cmp == 0)
                {
                  gtk_list_store_remove (charset->store, &iter);
                  break;
                }
            }
#else /* not GTK_CHECK_VERSION(2,4,0) */
          GList *glist;

          glist = gtk_container_children
                                (GTK_CONTAINER (GTK_COMBO (charset)->list));
          for (j = g_list_length (glist) - 1; j >= 0; j--)
            if (g_strcasecmp (gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                        (GTK_BIN (g_list_nth_data (glist, j))))), str[i]) == 0)
              {
                gtk_list_clear_items (GTK_LIST (GTK_COMBO (charset)->list),
                                                                    j, j + 1);
                break;
              }
          g_list_free (glist);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
          if (j < 0)
            result = FALSE;
        }
      g_strfreev (str);
    }
  else
    {
#if GTK_CHECK_VERSION(2,4,0)
      gtk_list_store_clear (charset->store);
#else /* not GTK_CHECK_VERSION(2,4,0) */
      gtk_list_clear_items (GTK_LIST (GTK_COMBO (charset)->list), 0, G_MAXINT);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
    }
  return result;
}


/*  ja:文字符号化方式を検査する
    charset,ウィジェット
        RET,TRUE:正しい文字符号化方式,FALSE:不正な文字符号化方式            */
gboolean
misc_charset_is_valid (MiscCharset *charset)
{
#if GTK_CHECK_VERSION(2,4,0)
  return charset ? charuty_is_valid (gtk_entry_get_text
                            (GTK_ENTRY (GTK_BIN (charset)->child))) : FALSE;
#else /* not GTK_CHECK_VERSION(2,4,0) */
  return charset ? charuty_is_valid (gtk_entry_get_text
                            (GTK_ENTRY (GTK_COMBO (charset)->entry))) : FALSE;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
}
