/*
    Melody maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "abort.h"
#include "add.h"
#include "command.h"
#include "edit.h"
#include "export.h"
#include "file.h"
#include "general.h"
#include "mm_dialog.h"
#include "mm_list.h"
#include "reload.h"
#include "root.h"
#include "misc/fileio.h"
#include "misc/misc.h"
#include "misc/misc_charlist.h"
#include "misc/misc_charset.h"
#include "misc/misc_conf.h"
#include "misc/misc_history.h"
#include "misc/misc_mdi.h"
#ifdef G_OS_WIN32
# include <gdk/gdkwin32.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:メニュー関数群                                                           *
*                                                                             *
******************************************************************************/
void
command_new (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  file_open_edit (NULL);
}


#if ! GTK_CHECK_VERSION(2,4,0)
static void
command_open_ok (GtkWidget *widget,
                 GtkWidget *dialog)
{
  gchar **file;

  file = g_object_get_data (G_OBJECT (dialog), "user_data");
  *file
    = g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog)));
  gtk_widget_destroy (dialog);
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


void
command_open (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *file = NULL, *path;
  GtkWidget *dialog;

#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_file_chooser_dialog_new (_("Open"),
                        GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_OPEN,
                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        NULL);
  if (open_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), open_path);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  gtk_widget_destroy (dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog = gtk_file_selection_new (_("Open"));
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), dialog);
  g_signal_connect_swapped
                        (G_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", &file);
  if (open_path)
    {
      path = g_strconcat (open_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  if (file)
    {
      g_free (open_path);
      path = fileio_get_full_path (file);
      open_path = g_path_get_dirname (path);
      g_free (path);
      file_open_edit (file);
      g_free (file);
    }
}


void
command_close (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  gint page;

  if ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0
    && prompt_close ((MmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), page)))
    gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page);
}


void
command_save (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *root = NULL, *tmp;
  const gchar *file;
  gint i = G_MAXINT, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  file = misc_mdi_get_file (MISC_MDI (mdi), page_num);
  tmp = g_path_get_dirname (file);
  root = g_strconcat (tmp, G_DIR_SEPARATOR_S, NULL);
  g_free (tmp);
  if (!misc_mdi_get_created (MISC_MDI (mdi), page_num))
    {
      gchar *path;
      gint leng;

      path = g_filename_to_utf8 (root, -1, NULL, NULL, NULL);
      leng = g_strlen (path);
      for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0; i--)
        {
          const ID3Tag *id3tag;

          id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
          if (g_strfilencmp (id3tag->file, path, leng))
            break;
        }
      g_free (path);
    }
  if (i >= 0)
    {
      command_saveas (callback_data, callback_action, widget);
    }
  else
    {
      gchar *text, *utf8str;

      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("%s\nThis file is existed.\n\nOver write?"),
                                                                    utf8str);
      g_free (utf8str);
      if ((!fileio_isfile (file)
            || misc_message_box ("Melody maid", text, 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
                        &&  (mmaid->filekind == FILE_KIND_DB
                        ? file_save_db (file, mmaid, root, mmaid->charset)
                        : file_save_list (file, mmaid, root, mmaid->charset,
                                                            mmaid->playlist))
                && (mmaid->filekind == FILE_KIND_DB || mmaid->playlist != 2))
        misc_mdi_set_edited (MISC_MDI (mdi), page_num, FALSE);
    }
  g_free (root);
}


typedef struct _CommandSaveOk CommandSaveOk;
struct _CommandSaveOk
{
  gchar *file, *root, *charset;
  guint filekind;
  gint playlist;
  GtkWidget *dialog, *hbox, *combo0, *combo1;
  GtkWidget *radio0, *radio1, *radio2, *radio3, *radio4;
};


/* ja:ラジオボタンが変更された */
static void
command_save_toggled_radio0 (GtkWidget     *widget,
                             CommandSaveOk *cmd_sv)
{
  gtk_widget_set_sensitive (cmd_sv->hbox, FALSE);
}


/* ja:ラジオボタンが変更された */
static void
command_save_toggled_radio1 (GtkWidget     *widget,
                             CommandSaveOk *cmd_sv)
{
  gtk_widget_set_sensitive (cmd_sv->hbox, TRUE);
}


#if ! GTK_CHECK_VERSION(2,4,0)
static void
command_save_ok (GtkWidget     *widget,
                 CommandSaveOk *cmd_sv)
{
  cmd_sv->file    = g_strdup (gtk_file_selection_get_filename
                                        (GTK_FILE_SELECTION (cmd_sv->dialog)));
  cmd_sv->root    = g_strdup (gtk_entry_get_text
                            (GTK_ENTRY (GTK_COMBO (cmd_sv->combo)->entry)));
  cmd_sv->charset = misc_charset_get_charset (MISC_CHARSET (cmd_sv->combo1));
  cmd_sv->filekind = gtk_toggle_button_get_active
        (GTK_TOGGLE_BUTTON (cmd_sv->radio0)) ? FILE_KIND_DB : FILE_KIND_LIST;
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cmd_sv->radio3)))
    cmd_sv->playlist = 1;
  else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cmd_sv->radio4)))
    cmd_sv->playlist = 2;
  else
    cmd_sv->playlist = 0;
  gtk_widget_destroy (cmd_sv->dialog);
}


/* ja:コンボボックスが変更された */
static void
command_save_changed (GtkWidget     *widget,
                      CommandSaveOk *cmd_sv)
{
  gchar *file, *name;

  name = g_path_get_basename (gtk_file_selection_get_filename
                                        (GTK_FILE_SELECTION (cmd_sv->dialog)));
  file = g_strconcat (gtk_entry_get_text (GTK_ENTRY (widget)), name, NULL);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (cmd_sv->dialog), file);
  g_free (file);
  g_free (name);
}


/* ja:コンボボックスが変更された */
static void
command_save_edited (GtkWidget     *widget,
                     CommandSaveOk *cmd_sv)
{
  gtk_widget_set_sensitive (GTK_FILE_SELECTION (cmd_sv->dialog)->ok_button,
                                misc_charset_is_valid (MISC_CHARSET (widget)));
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


void
command_saveas (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *name, *path, *root = NULL, *tmp, **ary;
  gint i, page_num;
  GList *glist = NULL;
  GtkWidget *label0, *label1, *hbox, *vbox;
  CommandSaveOk cmd_sv;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  g_memset (&cmd_sv, 0 ,sizeof (CommandSaveOk));
  /* ja:共通のルートを求める */
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0; i--)
    {
      const ID3Tag *id3tag;

      id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
      if (root)
        {
          gint j;

          for (j = 0; root[j] != '\0'; j++)
            if (root[j] != id3tag->file[j])
              break;
          root[j] = '\0';
        }
      else
        {
          root = g_strdup (id3tag->file);
        }
    }
  path = g_strrchr (root, G_DIR_SEPARATOR);
  if (!path)
    {
      misc_message_box ("Melody maid", _("Root Directory Error"), 0,
                                                            _("OK"), NULL);
      return;
    }
  *path = '\0';
#ifdef G_OS_WIN32
  if (g_strlen (root) > 2
                && root[0] == G_DIR_SEPARATOR && root[1] == G_DIR_SEPARATOR)
    {
      path = g_strdup (G_DIR_SEPARATOR_S G_DIR_SEPARATOR_S);
      ary = g_strsplit (root + 2, G_DIR_SEPARATOR_S, G_MAXINT);
    }
  else
    {
      ary = g_strsplit (root, G_DIR_SEPARATOR_S, G_MAXINT);
    }
#else /* not G_OS_WIN32 */
  ary = g_strsplit (root, G_DIR_SEPARATOR_S, G_MAXINT);
  path = NULL;
#endif /* not G_OS_WIN32 */
  g_free (root);
  root = NULL;
  for (i = 0; ary[i]; i++)
    {
      path = path ? g_strconcat (path, ary[i], G_DIR_SEPARATOR_S, NULL)
                  : g_strconcat (      ary[i], G_DIR_SEPARATOR_S, NULL);
      glist = g_list_append (glist, path);
    }
  g_strfreev (ary);
  if (save_path)
    {
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
      if (!g_strfilencmp (utf8str, path, g_strlen (utf8str)))
        root = g_strdup (save_path);
      g_free (utf8str);
    }
  if (!root)
    root = g_filename_from_utf8 (path, -1, NULL, NULL, NULL);
  /* ja:ラジオボタン */
  cmd_sv.radio0 = gtk_radio_button_new_with_mnemonic
                    (NULL, _("_iDB File"));
  cmd_sv.radio1 = gtk_radio_button_new_with_mnemonic_from_widget
                    (GTK_RADIO_BUTTON (cmd_sv.radio0), _("_Play List"));
  cmd_sv.radio2 = gtk_radio_button_new_with_mnemonic
                    (NULL, _("Rela_tive Path"));
  cmd_sv.radio3 = gtk_radio_button_new_with_mnemonic_from_widget
                    (GTK_RADIO_BUTTON (cmd_sv.radio2), _("Ab_solute Path"));
  cmd_sv.radio4 = gtk_radio_button_new_with_mnemonic_from_widget
                    (GTK_RADIO_BUTTON (cmd_sv.radio3), _("File Name _Only"));
  cmd_sv.hbox = gtk_hbox_new (FALSE, 0);
  if (mmaid->filekind == FILE_KIND_DB)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cmd_sv.radio0), TRUE);
      gtk_widget_set_sensitive (cmd_sv.hbox, FALSE);
    }
  else
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cmd_sv.radio1), TRUE);
    }
  switch (mmaid->playlist)
    {
      case  1: gtk_toggle_button_set_active
                            (GTK_TOGGLE_BUTTON (cmd_sv.radio3), TRUE); break;
      case  2: gtk_toggle_button_set_active
                            (GTK_TOGGLE_BUTTON (cmd_sv.radio4), TRUE); break;
      default: gtk_toggle_button_set_active
                            (GTK_TOGGLE_BUTTON (cmd_sv.radio2), TRUE);
    }
  g_signal_connect (G_OBJECT (cmd_sv.radio0), "toggled",
                            G_CALLBACK (command_save_toggled_radio0), &cmd_sv);
  g_signal_connect (G_OBJECT (cmd_sv.radio1), "toggled",
                            G_CALLBACK (command_save_toggled_radio1), &cmd_sv);
  /* ja:コンボボックス */
#if GTK_CHECK_VERSION(2,4,0)
  cmd_sv.combo0 = gtk_combo_box_new_text ();
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      gtk_combo_box_prepend_text (GTK_COMBO_BOX (cmd_sv.combo0),
                                                g_list_nth_data (glist, i));
      if (g_strfilecmp (g_list_nth_data (glist, i), root) == 0)
        gtk_combo_box_set_active (GTK_COMBO_BOX (cmd_sv.combo0), 0);
    }
#else /* not GTK_CHECK_VERSION(2,4,0) */
  cmd_sv.combo0 = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (cmd_sv.combo0), glist);
  gtk_combo_set_value_in_list (GTK_COMBO (cmd_sv.combo0), FALSE, FALSE);
  gtk_combo_set_case_sensitive (GTK_COMBO (cmd_sv.combo0), TRUE);
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (cmd_sv.combo0)->entry),
                                                                        FALSE);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (cmd_sv.combo0)->entry), root);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    g_free (g_list_nth_data (glist, i));
  g_list_free (glist);
  /* ja:文字符号化方式 */
  cmd_sv.combo1 = misc_charset_new ();
  misc_charset_add_candidate (MISC_CHARSET (cmd_sv.combo1), charset_list);
  misc_charset_set_charset (MISC_CHARSET (cmd_sv.combo1), mmaid->charset);
  /* ja:ラベル */
  label0 = gtk_label_new_with_mnemonic (_("_Root"));
  label1 = gtk_label_new_with_mnemonic (_("Character _Encoding"));
#if GTK_CHECK_VERSION(2,4,0)
  gtk_label_set_mnemonic_widget (GTK_LABEL(label0), cmd_sv.combo0);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL(label0),
                                            GTK_COMBO (cmd_sv.combo0)->entry);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL(label1), cmd_sv.combo1);
  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), cmd_sv.radio0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), cmd_sv.radio1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (cmd_sv.hbox),
                                        gtk_label_new ("("), FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (cmd_sv.hbox), cmd_sv.radio2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (cmd_sv.hbox), cmd_sv.radio3, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (cmd_sv.hbox), cmd_sv.radio4, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (cmd_sv.hbox),
                                        gtk_label_new (")"), FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), cmd_sv.hbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), label0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), cmd_sv.combo0, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), label1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), cmd_sv.combo1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show_all (vbox);
  /* ja:ダイアログ */
#if GTK_CHECK_VERSION(2,4,0)
  cmd_sv.dialog = gtk_file_chooser_dialog_new (_("Save As"),
                        GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_SAVE,
                        GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        NULL);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (cmd_sv.dialog), vbox);
  if (root)
    gtk_file_chooser_set_current_folder
                                    (GTK_FILE_CHOOSER (cmd_sv.dialog), root);
  name = g_path_get_basename (misc_mdi_get_file (MISC_MDI (mdi), page_num));
  tmp = g_filename_to_utf8 (name, -1, NULL, NULL, NULL);
  gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (cmd_sv.dialog), tmp);
  g_free (tmp);
  g_free (name);
  if (gtk_dialog_run (GTK_DIALOG (cmd_sv.dialog)) == GTK_RESPONSE_ACCEPT)
    {
# if ! GTK_CHECK_VERSION(2,6,0)
      GtkTreeIter iter;
# endif /* not GTK_CHECK_VERSION(2,6,0) */

      cmd_sv.file = gtk_file_chooser_get_filename
                                            (GTK_FILE_CHOOSER (cmd_sv.dialog));
# if GTK_CHECK_VERSION(2,6,0)
      cmd_sv.root = gtk_combo_box_get_active_text
                                            (GTK_COMBO_BOX (cmd_sv.combo0));
# else /* not GTK_CHECK_VERSION(2,6,0) */
      if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (cmd_sv.combo0), &iter))
        gtk_tree_model_get
                    (gtk_combo_box_get_model (GTK_COMBO_BOX (cmd_sv.combo0)),
                                                &iter, 0, &cmd_sv.root, -1);
# endif /* not GTK_CHECK_VERSION(2,6,0) */
      cmd_sv.charset = misc_charset_get_charset (MISC_CHARSET (cmd_sv.combo1));
      cmd_sv.filekind = gtk_toggle_button_get_active
        (GTK_TOGGLE_BUTTON (cmd_sv.radio0)) ? FILE_KIND_DB : FILE_KIND_LIST;
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cmd_sv.radio3)))
        cmd_sv.playlist = 1;
      else if (gtk_toggle_button_get_active
                                        (GTK_TOGGLE_BUTTON (cmd_sv.radio4)))
        cmd_sv.playlist = 2;
      else
        cmd_sv.playlist = 0;
    }
  gtk_widget_destroy (cmd_sv.dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  cmd_sv.dialog = gtk_file_selection_new (_("Save As"));
  g_signal_connect (G_OBJECT (cmd_sv.dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->ok_button),
                            "clicked", G_CALLBACK (command_save_ok), &cmd_sv);
  g_signal_connect_swapped
                (G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->cancel_button),
                                    "clicked", G_CALLBACK (gtk_widget_destroy),
                                                GTK_OBJECT (cmd_sv.dialog));
  g_signal_connect_swapped (G_OBJECT (GTK_COMBO (cmd_sv.combo0)->entry),
                    "activate", G_CALLBACK (gtk_button_clicked),
                    G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->ok_button));
  g_signal_connect_swapped (G_OBJECT (cmd_sv.combo0),
                    "activate", G_CALLBACK (gtk_button_clicked),
                    G_OBJECT (GTK_FILE_SELECTION (cmd_sv.dialog)->ok_button));
  g_signal_connect (G_OBJECT (GTK_COMBO (cmd_sv.combo0)->entry), "changed",
                                G_CALLBACK (command_save_changed), &cmd_sv);
  g_signal_connect (G_OBJECT (cmd_sv.combo1), "edited",
                                G_CALLBACK (command_save_edited), &cmd_sv);
  gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (cmd_sv.dialog)->main_vbox),
                                                        vbox, FALSE, FALSE, 0);
  name = g_path_get_basename (misc_mdi_get_file (MISC_MDI (mdi), page_num));
  tmp = root ? g_strconcat (root, name, NULL)
             : g_strconcat (save_path, G_DIR_SEPARATOR_S, name, NULL);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (cmd_sv.dialog), tmp);
  g_free (tmp);
  g_free (name);
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (cmd_sv.dialog));
  gtk_widget_show (cmd_sv.dialog);
  gtk_grab_add (cmd_sv.dialog);
  gtk_main ();
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  if (cmd_sv.file)
    {
      gchar *text, *utf8str;

      utf8str = g_filename_to_utf8 (cmd_sv.file, -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("%s\nThis file is existed.\n\nOver write?"),
                                                                    utf8str);
      g_free (utf8str);
      if ((!fileio_isfile (cmd_sv.file)
                                || misc_message_box ("Melody maid", text, 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
            && (cmd_sv.filekind == FILE_KIND_DB
            ? file_save_db (cmd_sv.file, mmaid, cmd_sv.root, cmd_sv.charset)
            : file_save_list (cmd_sv.file, mmaid, cmd_sv.root, cmd_sv.charset,
                                                            cmd_sv.playlist)))
        {
          gchar *path;

          g_free (save_path);
          path = fileio_get_full_path (cmd_sv.file);
          save_path = g_path_get_dirname (path);
          g_free (path);
          /* ja:キャラクターセット */
          g_free (mmaid->charset);
          mmaid->charset = g_strdup (cmd_sv.charset);
          if (cmd_sv.filekind == FILE_KIND_DB || cmd_sv.playlist != 2)
            {
              /* ja:諸設定 */
              misc_mdi_set_created (MISC_MDI (mdi), page_num, FALSE);
              misc_mdi_set_edited (MISC_MDI (mdi), page_num, FALSE);
              misc_mdi_set_file (MISC_MDI (mdi), page_num, cmd_sv.file);
            }
          mmaid->filekind = cmd_sv.filekind;
          if (cmd_sv.filekind == FILE_KIND_LIST)
            playlist = mmaid->playlist = cmd_sv.playlist;
          misc_history_add_file (MISC_HISTORY (history), cmd_sv.file);
        }
      g_free (text);
    }
  g_free (cmd_sv.file);
  g_free (cmd_sv.root);
  g_free (cmd_sv.charset);
  g_free (root);
}


void
command_reload (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *text;
  const gchar *file;
  gint page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  file = misc_mdi_get_file (MISC_MDI (mdi), page_num);
  text = misc_mdi_get_edited (MISC_MDI (mdi), page_num) ? g_strdup_printf
            (_("File %s was edited.\nRestore to existed file?"), file) : NULL;
  if (reload_dialog (mmaid, text))
    {
      /* ja:削除 */
      mm_list_delete_all (MM_LIST (mmaid->list));
      /* ja:再読み込み */
      file_open_db (file, mmaid);
      misc_mdi_set_edited (MISC_MDI (mdi), page_num, FALSE);
      charlist_renewal_all (mmaid->charset);
      set_menu_bar (mmaid);
    }
}


static void
command_dialog_abort (GtkWidget *widget,
                      GtkWidget *dialog)
{
  userbreak = FALSE;
}


void
command_export (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gint i, count;
  GtkWidget *dialog;
  ExportInfo *expinfo;
  MmaidWindow *mmaid;

  mmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  expinfo = export_dialog (mmaid->charset);
  if (!expinfo)
    return;
  def_hold = expinfo->hold;
  def_gnum = expinfo->gnum;
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Export"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 0; i < count && userbreak; i++)
    {
      while (gtk_events_pending ())
        gtk_main_iteration ();
      if (mm_list_is_selected (MM_LIST (mmaid->list), i))
        {
          gchar *file, *str;
          gint j, k, n, gnum;
          ID3Tag *id3tag;

          id3tag = copy_id3tag (mm_list_get_nth (MM_LIST (mmaid->list), i));
          file = g_filename_from_utf8 (id3tag->file, -1, NULL, NULL, NULL);
          switch (file_is_type (file))
            {
              case FILE_KIND_ERROR:
                /* ja:ファイルが存在しないとき */
                str = _("File Open Error");
                break;
              case FILE_KIND_TAG:
                if (expinfo->hold)
                  {
                    /* ja:既存のタグを保持し、かつ既にタグがあるとき */
                    str = _("Have Tag");
                    break;
                  }
              default:
                gnum = -1;
                if (id3tag->genre)
                  for (j = 0; j < genres; j++)
                    for (k = 0; k < 256; k++)
                      if (genre[j][k]
                                    && !g_strcmp (genre[j][k], id3tag->genre))
                        {
                          gnum = k;
                          goto loop;
                        }
                loop:
                if (gnum == -1)
                  {
                    gnum = expinfo->gnum;
                    g_free (id3tag->genre);
                    id3tag->genre = NULL;
                    for (j = 0; j < genres; j++)
                      if (genre[j][gnum])
                        {
                          id3tag->genre = g_strdup (genre[j][gnum]);
                          break;
                        }
                  }
                if (!id3tag->charset)
                  id3tag->charset = g_strdup (expinfo->charset);
                switch (file_save_id3tag (file, id3tag, gnum))
                  {
                    case FILE_SAVE_SUCCESS:
                      str = _("Success");
                      break;
                    case FILE_SAVE_CHARSET_TITLE:
                      str = _("Invalid Character Encoding of Title");
                      break;
                    case FILE_SAVE_CHARSET_ARTIST:
                      str = _("Invalid Character Encoding of Artist");
                      break;
                    case FILE_SAVE_CHARSET_ALBUM:
                      str = _("Invalid Character Encoding of Album");
                      break;
                    case FILE_SAVE_CHARSET_YEAR:
                      str = _("Invalid Character Encoding of Year");
                      break;
                    case FILE_SAVE_CHARSET_COMMENT:
                      str = _("Invalid Character Encoding of Comment");
                      break;
                    case FILE_SAVE_LENGTH_TITLE:
                      str = _("Invalid Length of Title");
                      break;
                    case FILE_SAVE_LENGTH_ARTIST:
                      str = _("Invalid Length of Artist");
                      break;
                    case FILE_SAVE_LENGTH_ALBUM:
                      str = _("Invalid Length of Album");
                      break;
                    case FILE_SAVE_LENGTH_YEAR:
                      str = _("Invalid Length of Year");
                      break;
                    case FILE_SAVE_LENGTH_COMMENT:
                      str = _("Invalid Length of Comment");
                      break;
                    default:
                      str = _("Error");
                  }
            }
          g_free (file);
          n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list), id3tag);
          mm_list_set_status (MM_LIST (MM_DIALOG (dialog)->list), n, str);
        }
    }
  g_free (expinfo->charset);
  g_free (expinfo);
  gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (MM_DIALOG (dialog)->ok_button);
  gtk_main ();
}


void
command_exit (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GdkEvent event;

  event.any.type = GDK_DELETE;
  event.any.window = window->window;
  event.any.send_event = FALSE;
  gdk_event_put (&event);
}


void
command_up (gpointer   callback_data,
            guint      callback_action,
            GtkWidget *widget)
{
  gint i, count, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 1; i < count; i++)
    if (!mm_list_is_selected (MM_LIST (mmaid->list), i - 1)
                            && mm_list_is_selected (MM_LIST (mmaid->list), i))
      mm_list_swap (MM_LIST (mmaid->list), i - 1, i);
  misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
}


void
command_down (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gint i, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 2; i >= 0 ; i--)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i)
                        && !mm_list_is_selected (MM_LIST (mmaid->list), i + 1))
      mm_list_swap (MM_LIST (mmaid->list), i, i + 1);
  misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
}


void
command_root (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  root_dialog ((MmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1));
}


static void
command_dialog_ok (GtkWidget *widget,
                    GtkWidget *dialog)
{
  gboolean edited = FALSE;
  gint i, count;
  MmaidWindow *mmaid;

  mmaid = g_object_get_data (G_OBJECT (dialog), "user_data");
  count = mm_list_length (MM_LIST (MM_DIALOG (dialog)->list));
  for (i = 0; i < count; i++)
    {
      gint n;
      const ID3Tag *id3tag;

      id3tag = mm_list_get_nth (MM_LIST (MM_DIALOG (dialog)->list), i);
      n = mm_list_search (MM_LIST (mmaid->list),
                                            MM_LIST_FILE, id3tag->file, -1);
      if (n >= 0)
        {
          if (mm_list_get_active_refresh
                                    (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_change (MM_LIST (mmaid->list), n,
                                    MM_LIST_ID3TAG, copy_id3tag (id3tag), -1);
              edited = TRUE;
            }
          else if (mm_list_get_active_delete
                                    (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_delete (MM_LIST (mmaid->list), n);
              edited = TRUE;
            }
        }
      else
        {
          if (mm_list_get_active_add (MM_LIST (MM_DIALOG (dialog)->list), i))
            {
              mm_list_append (MM_LIST (mmaid->list), copy_id3tag (id3tag));
              edited = TRUE;
            }
        }
    }
  if (edited)
    misc_mdi_set_edited (MISC_MDI (mdi),
                    misc_mdi_get_page_from_data (MISC_MDI (mdi), mmaid), TRUE);
  gtk_widget_destroy (dialog);
}


void
command_refresh (gpointer   callback_data,
                 guint      callback_action,
                 GtkWidget *widget)
{
  gboolean selected = FALSE;
  gint i, count;
  GtkWidget *dialog;
  MmaidWindow *mmaid;

  mmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  /* ja:選択の数を数える */
  count = mm_list_length (MM_LIST (mmaid->list));
  for (i = 0; i < count; i++)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i))
      {
        selected = TRUE;
        break;
      }
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Refresh"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                        "clicked", G_CALLBACK (command_dialog_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->cancel_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  for (i = 0; i < count && userbreak; i++)
    {
      while (gtk_events_pending ())
        gtk_main_iteration ();
      if (!selected || mm_list_is_selected (MM_LIST (mmaid->list), i))
        {
          gchar *file;
          gint n;
          const ID3Tag *id3tag0;
          ID3Tag *id3tag1;

          id3tag0 = mm_list_get_nth (MM_LIST (mmaid->list), i);
          file = g_filename_from_utf8 (id3tag0->file, -1, NULL, NULL, NULL);
          switch (file_is_type (file))
            {
              case FILE_KIND_UNKNOWN:
                id3tag1 = file_open_id3tag (file, id3tag0->charset
                                            ? id3tag0->charset : charset_list);
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                if (!cmp_id3tag (id3tag0, id3tag1))
                  mm_list_show_refresh (MM_LIST (MM_DIALOG (dialog)->list), n);
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                break;
              case FILE_KIND_TAG:
                id3tag1 = file_open_id3tag (file, id3tag0->charset
                                            ? id3tag0->charset : charset_list);
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                if (!cmp_id3tag (id3tag0, id3tag1))
                  {
                    mm_list_set_sensitive_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n, FALSE);
                    mm_list_show_refresh (MM_LIST (MM_DIALOG (dialog)->list),
                                                                            n);
                    mm_list_set_active_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n, TRUE);
                  }
                break;
              default:
                n = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                        copy_id3tag (id3tag0));
                mm_list_show_delete (MM_LIST (MM_DIALOG (dialog)->list), n);
                mm_list_set_active_delete (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    n, TRUE);
                break;
            }
          g_free (file);
        }
    }
  if (userbreak)
    {
      gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
      gtk_widget_show (MM_DIALOG (dialog)->ok_button);
      gtk_widget_show (MM_DIALOG (dialog)->cancel_button);
      g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
      gtk_main ();
    }
  else
    {
      gtk_widget_destroy (dialog);
    }
}


void
command_add (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  add_dialog ((MmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1));
}


void
command_edit (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  edit_dialog ((MmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1));
}


static void
command_search_recursive (const gchar *path,
                          GtkWidget   *dialog)
{
  GDir *gdir;

  gdir = g_dir_open (path, 0, NULL);
  if (gdir)
    {
      const gchar *name;

      while (userbreak && (name = g_dir_read_name (gdir)))
        {
          gchar *file;

          while (gtk_events_pending ())
            gtk_main_iteration ();
#ifndef G_OS_WIN32
          if (name[0] == '.')
            continue;
#endif /* not G_OS_WIN32 */
          file = g_strconcat (path, G_DIR_SEPARATOR_S, name, NULL);
          if (g_file_test (file, G_FILE_TEST_IS_DIR))
            {
              command_search_recursive (file, dialog);
            }
          else
            {
              gchar *utf8file;
              gint n0, n1;
              const ID3Tag *id3tag0;
              ID3Tag *id3tag1;
              MmaidWindow *mmaid;

              mmaid = g_object_get_data (G_OBJECT (dialog), "user_data");
              utf8file = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
              n0 = mm_list_search (MM_LIST (mmaid->list),
                                                MM_LIST_FILE, utf8file, -1);
              g_free (utf8file);
              id3tag0 = n0 >= 0 ? mm_list_get_nth (MM_LIST (mmaid->list), n0)
                                : NULL;
              switch (file_is_type (file))
                {
                  case FILE_KIND_UNKNOWN:
                    id3tag1 = file_open_id3tag (file, charset_list);
                    n1 = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                    if (n0 >= 0)
                      {
                        if (!cmp_id3tag (id3tag0, id3tag1))
                          mm_list_show_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                      }
                    else
                      {
                        mm_list_show_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        if (g_strlen (file) > 4 && !g_strcasecmp (".mp3",
                                                file + g_strlen (file) - 4))
                          mm_list_set_active_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                    break;
                  case FILE_KIND_TAG:
                    id3tag1 = file_open_id3tag (file, charset_list);
                    n1 = mm_list_append (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    id3tag1);
                    if (n0 >= 0)
                      {
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        if (!cmp_id3tag (id3tag0, id3tag1))
                          {
                            mm_list_set_sensitive_delete
                                (MM_LIST (MM_DIALOG (dialog)->list),
                                                                    n1, FALSE);
                            mm_list_show_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                            mm_list_set_active_refresh
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                          }
                      }
                    else
                      {
                        mm_list_show_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_set_active_add
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                    break;
                  default:
                    if (n0 >= 0)
                      {
                        n1 = mm_list_append
                                (MM_LIST (MM_DIALOG (dialog)->list),
                                                        copy_id3tag (id3tag0));
                        mm_list_show_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1);
                        mm_list_set_active_delete
                                (MM_LIST (MM_DIALOG (dialog)->list), n1, TRUE);
                      }
                }
            }
          g_free (file);
        }
      g_dir_close (gdir);
    }
}


void
command_search (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *path, *tmp;
  const gchar *file;
  gint page_num;
  GtkWidget *dialog;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  file = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  path = g_path_is_absolute (file) ? g_path_get_dirname (file)
       : open_path ? g_strdup (open_path)
       : save_path ? g_strdup (save_path)
       : NULL;
#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_file_chooser_dialog_new (_("Browse"),
                        GTK_WINDOW (window),
                        GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                        GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        NULL);
  if (path)
    {
      gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), path);
      g_free (path);
      path = NULL;
    }
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  gtk_widget_destroy (dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog = gtk_file_selection_new (_("Browse"));
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), dialog);
  g_signal_connect_swapped
                        (G_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", &path);
  if (path)
    {
      if (path[0] != '\0')
        {
          gchar *tmp;

          if (path[g_strlen (path) - 1] != G_DIR_SEPARATOR)
            {
              tmp = g_strconcat (path, G_DIR_SEPARATOR_S, NULL);
              g_free (path);
              path = tmp;
            }
          gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), path);
        }
      g_free (path);
      path = NULL;
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_hide
            (gtk_widget_get_parent (GTK_FILE_SELECTION (dialog)->file_list));
  gtk_widget_hide (GTK_FILE_SELECTION (dialog)->selection_entry);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
#endif /* not GTK_CHECK_VERSION(2,4,0) */

  if (!path)
    return;
  tmp = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
  g_free (path);
  if (!tmp || tmp[0] =='\0')
    {
      g_free (tmp);
      return;
    }
  if (tmp[g_strlen (tmp) - 1] == G_DIR_SEPARATOR)
    tmp[g_strlen (tmp) - 1] = '\0';
  path = g_filename_from_utf8 (tmp, -1, NULL, NULL, NULL);
  g_free (tmp);
  if (!path)
    return;
  userbreak = TRUE;
  dialog = mm_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dialog), _("Search"));
  mm_list_set_editable (MM_LIST (MM_DIALOG (dialog)->list), TRUE);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->abort_button),
                        "clicked", G_CALLBACK (command_dialog_abort), dialog);
  g_signal_connect (G_OBJECT (MM_DIALOG (dialog)->ok_button),
                        "clicked", G_CALLBACK (command_dialog_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MM_DIALOG (dialog)->cancel_button),
                "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", mmaid);
  gtk_widget_show (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  command_search_recursive (path, dialog);
  g_free (path);
  gtk_widget_hide (MM_DIALOG (dialog)->abort_button);
  gtk_widget_show (MM_DIALOG (dialog)->ok_button);
  gtk_widget_show (MM_DIALOG (dialog)->cancel_button);
  gtk_main ();
}


void
command_cut (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  command_copy   (callback_data, callback_action, widget);
  command_delete (callback_data, callback_action, widget);
}


void
command_copy (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
#ifndef G_OS_WIN32
  if (gtk_selection_owner_set (window, GDK_SELECTION_CLIPBOARD,
                                                            GDK_CURRENT_TIME))
    {
#endif /* not G_OS_WIN32 */
      gint i, count;
      MmaidWindow *mmaid;

      mmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
      g_free (clipboard_id3tag);
      clipboard_id3tag = NULL;
      count = mm_list_length (MM_LIST (mmaid->list));
      for (i = 0; i < count; i++)
        if (mm_list_is_selected (MM_LIST (mmaid->list), i))
          {
            gchar *str;
            const ID3Tag *id3tag;

            id3tag = mm_list_get_nth (MM_LIST (mmaid->list), i);
            str = g_strdup_printf ("%s\t%s\t%s\t%s\t%s\t%s\t%d\t%s\t%s",
                                    id3tag->file    ? id3tag->file     : "",
                                    id3tag->title   ? id3tag->title    : "",
                                    id3tag->artist  ? id3tag->artist   : "",
                                    id3tag->album   ? id3tag->album    : "",
                                    id3tag->year    ? id3tag->year     : "",
                                    id3tag->comment ? id3tag->comment  : "",
                                    id3tag->track,
                                    id3tag->genre   ? id3tag->genre    : "",
                                    id3tag->charset ? id3tag->charset  : "");
            if (clipboard_id3tag)
              {
                gchar *tmp;

                tmp = g_strconcat (clipboard_id3tag, "\n", str, NULL);
                g_free (clipboard_id3tag);
                g_free (str);
                clipboard_id3tag = tmp;
              }
            else
              {
                clipboard_id3tag = str;
              }
          }
#ifdef G_OS_WIN32
      if (clipboard_id3tag)
        {
          HGLOBAL hGlobal;
          LPSTR lpID3Tag;

          hGlobal = GlobalAlloc (GMEM_MOVEABLE,
                        (g_strlen (clipboard_id3tag) + 1) * sizeof (gchar));
          if (hGlobal)
            {
              lpID3Tag = GlobalLock (hGlobal);
              if (lpID3Tag)
                {
                  g_strcpy (lpID3Tag, clipboard_id3tag);
                  GlobalUnlock (hGlobal);
                  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
                    {
                      if (!EmptyClipboard ()
                                    || !SetClipboardData (uFormat, hGlobal))
                        GlobalFree (hGlobal);
                      CloseClipboard ();
                    }
                  else
                    {
                      GlobalFree (hGlobal);
                    }
                }
              else
                {
                  GlobalFree (hGlobal);
                }
            }
        }
#else /* not G_OS_WIN32 */
    }
#endif /* not G_OS_WIN32 */
}


void
command_paste (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  /* ja:Windowsのセレクションの実装があやしいので暫定的処理 */
#ifdef G_OS_WIN32
  gchar *id3tag = NULL;
  HGLOBAL hGlobal;
  LPSTR lpID3Tag;

  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
    {
      if (hGlobal = GetClipboardData (uFormat))
        {
          if (lpID3Tag = GlobalLock (hGlobal))
            id3tag = g_strdup (lpID3Tag);
          GlobalUnlock (hGlobal);
        }
      CloseClipboard ();
      if (id3tag)
        {
          GtkSelectionData data;

          data.selection = GDK_SELECTION_CLIPBOARD;
          data.target = atom_id3tag;
          data.type = atom_id3tag;
          data.format = 8;
          data.data = id3tag;
          data.length = g_strlen (id3tag);
          signal_selection_received (window, &data, GDK_CURRENT_TIME, NULL);
          g_free (id3tag);
        }
    }
#else /* not G_OS_WIN32 */
  gtk_selection_convert (window, GDK_SELECTION_CLIPBOARD, atom_targets,
                                                            GDK_CURRENT_TIME);
#endif /* not G_OS_WIN32 */
}


void
command_delete (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gint i, page_num;
  MmaidWindow *mmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  mmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0 ; i--)
    if (mm_list_is_selected (MM_LIST (mmaid->list), i))
      mm_list_delete (MM_LIST (mmaid->list), i);
  misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
}


void
command_all (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  gint i;
  MmaidWindow *mmaid;

  mmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  for (i = mm_list_length (MM_LIST (mmaid->list)) - 1; i >= 0 ; i--)
    if (!mm_list_is_selected (MM_LIST (mmaid->list), i))
      break;
  if (i >= 0)
    mm_list_select_all (MM_LIST (mmaid->list));
  else
    mm_list_unselect_all (MM_LIST (mmaid->list));
}


static void
command_conf_ok (GtkWidget *widget,
                 GtkWidget *dialog)
{
  misc_history_set_num (MISC_HISTORY (history),
                                misc_conf_get_history (MISC_CONF (dialog)));
  newfile = misc_conf_get_newfile (MISC_CONF (dialog));
  second = misc_conf_get_second (MISC_CONF (dialog));
  misc_conf_get_size (MISC_CONF (dialog), &def_width, &def_height);
  n_pos = misc_conf_get_tab (MISC_CONF (dialog));
  gtk_widget_destroy (dialog);
  /* ja:ノートの位置 */
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (mdi), n_pos);
}


void
command_conf (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GtkWidget *dialog;

  dialog = misc_conf_new ();
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MISC_CONF (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_conf_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MISC_CONF (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  misc_conf_set_history (MISC_CONF(dialog),
                                misc_history_get_num (MISC_HISTORY (history)));
  misc_conf_set_newfile (MISC_CONF(dialog), newfile);
  misc_conf_set_second (MISC_CONF(dialog), second);
  misc_conf_set_size (MISC_CONF(dialog), def_width, def_height);
  misc_conf_set_tab (MISC_CONF(dialog), n_pos);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
}


static void
command_charset_ok (GtkWidget *widget,
                    GtkWidget *dialog)
{
  g_free (charset_list);
  charset_list = misc_charlist_get_charlist (MISC_CHARLIST (dialog));
  charlist_renewal_all (charset_list);
  gtk_widget_destroy (dialog);
}


void
command_charset (gpointer   callback_data,
                 guint      callback_action,
                 GtkWidget *widget)
{
  GtkWidget *dialog;

  dialog = misc_charlist_new ();
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MISC_CHARLIST (dialog)->ok_button),
                        "clicked", G_CALLBACK (command_charset_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MISC_CHARLIST (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  misc_charlist_set_charlist (MISC_CHARLIST (dialog), charset_list);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
}


void
command_play (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *cmd;

  cmd = g_filename_from_utf8 (cmd_play, -1, NULL, NULL, NULL);
  if (cmd)
    {
      gint n;
      MmaidWindow *mmaid;

      mmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
      mm_list_get_cursor (MM_LIST (mmaid->list), &n, NULL);
      if (n >= 0)
        {
          gchar *file;
          const ID3Tag *id3tag;

          id3tag = mm_list_get_nth (MM_LIST (mmaid->list), n);
          file = g_filename_from_utf8 (id3tag->file, -1, NULL, NULL, NULL);
          if (file)
            {
              gchar *quoted, *str;

              quoted = g_shell_quote (file);
              str = g_strconcat (cmd, " ", quoted, NULL);
              if (quoted && str)
                g_spawn_command_line_async (str, NULL);
              g_free (str);
              g_free (quoted);
              g_free (file);
            }
        }
      g_free (cmd);
    }
}


void
command_stop (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *cmd;

  cmd = g_filename_from_utf8 (cmd_stop, -1, NULL, NULL, NULL);
  if (cmd)
    {
      g_spawn_command_line_async (cmd, NULL);
      g_free (cmd);
    }
}
