/*
    Profile
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "fileio.h"
#include "misc.h"
#include "profile.h"


typedef struct _ProfileList
{
  gchar *data, *section, *key, *value;
  guint type;
  struct _ProfileList *prev, *next;
} ProfileList;
struct _Profile
{
  gboolean edit;
  gchar *file;
  ProfileList *list, *sublist;
};


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:ファイルをリスト化する
    file,ファイル名
     RET,リスト                                                             */
static ProfileList *
profile_list (const gchar *file)
{
  gchar buf[256], *data = NULL, *tmp, *section = NULL;
  gint n;
  FileIO *fio;
  ProfileList *p = NULL, *q, *r;

  fio = fileio_open (file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (!fio)
      return NULL;
  while (fileio_gets (buf, 256, fio))
    {
      if (!data)
        {
          data = g_strdup (buf);
        }
      else
        {
          tmp = data;
          data = g_strconcat (tmp, buf, NULL);
          g_free (tmp);
        }
      n = data ? g_strlen (data) : 0;
      if (n > 0 && data[n - 1] == '\n')
        {
          data[n - 1] = '\0';
          q = g_malloc0 (sizeof (ProfileList));
          q->data = data;
          q->prev = p;
          if (p)
            p->next = q;
          p = q;
          data = NULL;
        }
    }
  if (!fileio_close (fio) || !p)
    {
      while (p)
        {
          q = p->prev;
          g_free (p->data);
          g_free (p);
          p = q;
        }
      return NULL;
    }
  while (p->prev)
    p = p->prev;
  r = p;
  /* ja:リストを整理する */
    while (p)
      {
        data = p->data ? g_strstrip (g_strdup (p->data)) : NULL;
        n = data ? g_strlen (data) : 0;
        if (n <= 0)
          {
            p->type = PROFILE_DATA_TYPE_SPACE;
          }
        else if (data[0] == '#' || data[0] == ';')
          {
            p->type = PROFILE_DATA_TYPE_COMMENT;
          }
        else if (data[0] == '[' && data[n - 1] == ']')
          {
            p->type = PROFILE_DATA_TYPE_SECTION;
            g_free (section);
            section = g_strdup (data + 1);
            section[n - 2] = '\0';
            for (q = p->prev; q && q->type == PROFILE_DATA_TYPE_SPACE;
                                                                q = q->prev)
              {
                g_free (q->section);
                q->section = NULL;
              }
          }
        else if (g_strchr (data, '='))
          {
            p->type = PROFILE_DATA_TYPE_KEY;
            p->key = g_strdup (data);
            *g_strchr (p->key, '=') = '\0';
            p->value = g_strchr (p->data, '=') + 1;
          }
        else
          {
            p->type = PROFILE_DATA_TYPE_UNKNOW;
          }
        p->section = g_strdup (section);
        g_free (data);
        p = p->next;
      }
    g_free (section);
    return r;
}


/*  ja:初期化ファイルを開く
    file,ファイル名
     RET,プロファイル                                                       */
Profile *
profile_open (const gchar *file)
{
  gchar *base = NULL, *name;
  gint n;
  Profile *profile;

  profile = g_malloc0 (sizeof (Profile));
  name = g_get_prgname ();
  if (file)
    {
      base = g_path_get_basename (file);
      if (g_strfilecmp (file, base) != 0)
        {
          g_free (base);
          base = NULL;
        }
    }
  else
    {
      base = g_path_get_basename (name);
      n = g_strlen (base);
      if (n >= 5 && g_strfilecmp (base + n - 4, ".exe") == 0)
        base[n -= 4] = '\0';
    }
  if (base)
    {
      gchar *tmp;

      tmp = g_strdup (base);
      n = g_strlen (tmp);
      while (tmp[0] == '.')
        g_memmove (tmp, tmp + 1, n--);
      if (n > 0)
        {
          gchar *sys;

#ifdef SYSCONFDIR
          sys = g_strconcat (SYSCONFDIR, G_DIR_SEPARATOR_S,
                                                        tmp, ".conf", NULL);
#else /* not SYSCONFDIR */
          gchar *dir;

          dir = g_path_get_dirname (name);
          sys = g_strconcat (dir, G_DIR_SEPARATOR_S, tmp, ".conf", NULL);
          g_free (dir);
#endif /* not SYSCONFDIR */
          profile->sublist = profile_list (sys);
          g_free (sys);
        }
      g_free (tmp);
    }
  profile->file = !base ? g_strdup (file)
#ifdef G_OS_WIN32
      : g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, base, ".ini", NULL);
#else /* not G_OS_WIN32 */
      : g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".", base, NULL);
#endif /* not G_OS_WIN32 */
  g_free (base);
  profile->list = profile_list (profile->file);
  return profile;
}


/*  ja:初期化ファイルを閉じる
    profile,プロファイル
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_close (Profile *profile)
{
  FileIO *fio;
  ProfileList *p, *q;

  if (!profile)
    return TRUE;
  if (profile->edit && (fio = fileio_open (profile->file,
                                        FILEIO_ACCESS_WRITE, FILEIO_SHARE_READ,
                                        FILEIO_MODE_CREATE_ALWAYS)))
    {
      /* ja:変更があったとき */
      for (p = profile->list; p; p = p->next)
        {
          if (p->data)
            fileio_puts (p->data, fio);
          fileio_putc ('\n', fio);
        }
      fileio_close (fio);
    }
  g_free (profile->file);
  for (p = profile->list; p; p = q)
    {
      q = p->next;
      g_free (p->data);
      g_free (p->section);
      g_free (p->key);
      g_free (p);
    }
  for (p = profile->sublist; p; p = q)
    {
      q = p->next;
      g_free (p->data);
      g_free (p->section);
      g_free (p->key);
      g_free (p);
    }
  g_free (profile);
  return TRUE;
}


/*  ja:初期化ファイルから文字列を取得する
    profile,プロファイル
    section,セクション
        key,キー
        RET,文字列,NULL:エラー                                              */
gchar *
profile_get_string (Profile     *profile,
                    const gchar *section,
                    const gchar *key)
{
  ProfileList *p;

  if (!profile || !section || !key)
    return FALSE;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      return g_strdup (p->value);
  for (p = profile->sublist; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      return g_strdup (p->value);
  return NULL;
}


/*  ja:初期化ファイルから値のサイズ取得する
    profile,プロファイル
    section,セクション
        key,キー
       type,タイプ
        RET,バイト数,0:エラー                                               */
gint
profile_get_size (Profile     *profile,
                  const gchar *section,
                  const gchar *key,
                  const guint  type)
{
  guint8 *array;
  gint n;
  ProfileList *p;

  if (!profile || !section || !key)
    return 0;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    for (p = profile->sublist; p; p = p->next)
      if (p->type == PROFILE_DATA_TYPE_KEY
        && g_strcmp (p->section, section) == 0 && g_strcmp (p->key, key) == 0)
        break;
  if (!p)
    return 0;
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        return g_strcasecmp (p->value, "true") == 0
                || g_strcasecmp (p->value, "false") == 0
                || g_strcasecmp (p->value, "yes") == 0
                || g_strcasecmp (p->value, "no") == 0
                || g_strcasecmp (p->value, "on") == 0
                || g_strcasecmp (p->value, "off") == 0
                || g_strcasecmp (p->value, "ok") == 0
                || g_strcasecmp (p->value, "cancel") == 0
                ? sizeof (gboolean) : 0;
      case PROFILE_VALUE_TYPE_INT:
        return sizeof (gint);
      case PROFILE_VALUE_TYPE_STRING:
        return g_strlen (p->value) + 1;
      case PROFILE_VALUE_TYPE_ARRAY:
        array = misc_str_to_array (&n, 8, p->value, 10, FALSE);
        if (!array)
          return 0;
        g_free (array);
        return n;
    }
  return 0;
}


/*  ja:初期化ファイルから値を取得する
    profile,プロファイル
    section,セクション
        key,キー
      value,値を入れるバッファ
       size,値を入れるバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_get_value (Profile     *profile,
                   const gchar *section,
                   const gchar *key,
                   gpointer     value,
                   const gint   size,
                   const guint  type)
{
  guint8 *array;
  gint n;
  ProfileList *p;

  if (!profile || !section || !key || !value)
    return FALSE;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    for (p = profile->sublist; p; p = p->next)
      if (p->type == PROFILE_DATA_TYPE_KEY
        && g_strcmp (p->section, section) == 0 && g_strcmp (p->key, key) == 0)
        break;
  if (!p)
    return FALSE;
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        if (size < sizeof (gboolean))
          return FALSE;
        if (g_strcasecmp (p->value, "true") == 0
            || g_strcasecmp (p->value, "yes") == 0
            || g_strcasecmp (p->value, "on") == 0
            || g_strcasecmp (p->value, "ok") == 0)
          *((gboolean *)value) = TRUE;
        else if (g_strcmp (p->value, "false") == 0
            || g_strcasecmp (p->value, "no") == 0
            || g_strcasecmp (p->value, "off") == 0
            || g_strcasecmp (p->value, "cancel") == 0)
          *((gboolean *)value) = FALSE;
        else
          return FALSE;
        break;
      case PROFILE_VALUE_TYPE_INT:
        if (size < sizeof (gint))
          return FALSE;
        misc_str_to_val ((gint *)value, p->value, 10, TRUE);
          break;
      case PROFILE_VALUE_TYPE_STRING:
        if (size < g_strlen (p->value) + 1)
          return FALSE;
        g_strcpy ((gchar *)value, p->value);
        break;
      case PROFILE_VALUE_TYPE_ARRAY:
        array = misc_str_to_array (&n, 8, p->value, 10, FALSE);
        if (!array)
          return FALSE;
        if (size <= n)
          g_memmove (value, array, size);
        g_free (array);
        if (n < size)
          return FALSE;
        break;
      default:
        return FALSE;
    }
  return TRUE;
}


/*  ja:初期化ファイルに値を設定する
    profile,プロファイル
    section,セクション
        key,キー
      value,値が入っているバッファ
       size,値が入っているバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_set_value (Profile       *profile,
                   const gchar   *section,
                   const gchar   *key,
                   gconstpointer  value,
                   const gint     size,
                   const guint    type)
{
  gchar *data;
  gint i;
  ProfileList *p ,*q = NULL;

  if (!profile || !section || !key || !value)
    return FALSE;
  for (p = profile->list; p; q = p, p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    {
      for (p = q; p; p = p->prev)
        if (p->section && g_strcmp (p->section, section) == 0)
          break;
      if (!p)
        {
          if (q)
            {
              /* ja:前のデータとの間にスペースを入れる */
              p = g_malloc (sizeof (ProfileList));
              p->type = PROFILE_DATA_TYPE_SPACE;
              p->data = p->section = p->key = p->value = NULL;
              p->prev = q;
              p->next = q->next;
              q->next = p;
              q = p;
            }
            /* ja:セクションもキーもないときにはセクションを作る */
          p = g_malloc (sizeof (ProfileList));
          p->type = PROFILE_DATA_TYPE_SECTION;
          p->data = g_strdup_printf ("[%s]", section);
          p->section = g_strdup (section);
          p->key = p->value = NULL;
          p->prev = q;
          if (q)
            {
              p->next = q->next;
              q->next = p;
            }
          else
            {
              p->next = NULL;
              profile->list = p;
            }
        }
      q = p;
      while (q->type == PROFILE_DATA_TYPE_SPACE && q->section
                            && g_strcmp (p->section, section) == 0 && q->prev)
        q = q->prev;
      /* ja:セクションの最後にキーを作る */
      p = g_malloc (sizeof (ProfileList));
      p->type = PROFILE_DATA_TYPE_KEY;
      p->data = g_strdup_printf ("%s=", key);
      p->section = g_strdup (section);
      p->key = g_strdup (key);
      p->value = g_strchr (p->data, '=') + 1;
      p->prev = q;
      p->next = q->next;
      q->next = p;
      if (p->next)
        p->next->prev = p;
    }
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%s",
                            p->key, *((gboolean *)value) ? "true" : "false");
          break;
      case PROFILE_VALUE_TYPE_INT:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%d", p->key, *((gint *)value));
        break;
      case PROFILE_VALUE_TYPE_STRING:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%s", p->key, (gchar *)value);
        break;
      case PROFILE_VALUE_TYPE_ARRAY:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%u", p->key, ((guint8 *)value)[0]);
        for (i = 1; i < size; i++)
          {
            data = g_strdup_printf ("%s %u", p->data, ((guint8 *)value)[i]);
            g_free (p->data);
            p->data = data;
          }
        break;
      default:
        return FALSE;
    }
  p->value = g_strchr (p->data, '=') + 1;
  profile->edit = TRUE;
  return TRUE;
}


/*  ja:初期化ファイルのセクションを削除する
    profile,プロファイル
    section,セクション
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_section (Profile     *profile,
                        const gchar *section)
{
  gboolean result = FALSE;
  ProfileList *p, *q;

  if (!profile || !section)
    return FALSE;
  for (p = profile->list; p; p = q)
    {
      q = p->next;
      if (p->section && g_strcmp (p->section, section) == 0)
        {
          if (p->prev)
            p->prev->next = p->next;
          if (p->next)
            p->next->prev = p->prev;
          g_free (p->data);
          g_free (p->section);
          g_free (p->key);
          g_free (p);
          profile->edit = TRUE;
          result = TRUE;
        }
    }
  return result;
}


/*  ja:初期化ファイルのキーを削除する
    profile,プロファイル
    section,セクション
        key,キー
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_key (Profile     *profile,
                    const gchar *section,
                    const gchar *key)
{
  gboolean result = FALSE;
  ProfileList *p, *q;

  if (!profile || !section || !key)
    return FALSE;
  for (p = profile->list; p; p = q)
    {
      q = p->next;
      if (p->section && p->key && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
        {
          if (p->prev)
            p->prev->next = p->next;
          if (p->next)
            p->next->prev = p->prev;
          g_free (p->data);
          g_free (p->section);
          g_free (p->key);
          g_free (p);
          profile->edit = TRUE;
          result = TRUE;
        }
    }
  return result;
}


/*  ja:初期化ファイルのセクションを列挙する
    profile,プロファイル
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_section (Profile *profile)
{
  GList *glist = NULL;
  ProfileList *p;

  if (!profile)
    return NULL;
  for (p = profile->list; p; p = p->next)
    if (p->section && (!glist || !g_list_find_custom (glist, p->section,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->section, (GCompareFunc)strcmp);
  for (p = profile->sublist; p; p = p->next)
    if (p->section && (!glist || !g_list_find_custom (glist, p->section,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->section, (GCompareFunc)strcmp);
  return glist;
}


/*  ja:初期化ファイルのキーを列挙する
    profile,プロファイル
    section,セクション
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_key (Profile     *profile,
                  const gchar *section)
{
  GList *glist = NULL;
  ProfileList *p;

  if (!profile)
    return NULL;
  for (p = profile->list; p; p = p->next)
    if (p->section && p->key && g_strcmp (p->section, section) == 0
                            && (!glist || !g_list_find_custom (glist, p->key,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->key, (GCompareFunc)strcmp);
  for (p = profile->sublist; p; p = p->next)
    if (p->section && p->key && g_strcmp (p->section, section) == 0
                            && (!glist || !g_list_find_custom (glist, p->key,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->key, (GCompareFunc)strcmp);
  return glist;
}
