/*
    charuty
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "charuty.h"


/******************************************************************************
*                                                                             *
* ja:文字符号化方式関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:文字符号化方式が正しいか判定する
    charset,文字符号化方式
        RET,TRUE:正しい,FALSE:不正                                          */
gboolean
charuty_is_valid (const gchar *charset)
{
  gboolean parenthesis = TRUE;
  gint i;
  GIConv converter;

  if (!charset || !g_ascii_isalnum (charset[0]))
    return FALSE;
  for (i = 1; charset[i] != '\0'; i++)
    if (charset[i] == '(' && parenthesis)
      parenthesis = FALSE;
    else if (charset[i] == ')' && !parenthesis)
      parenthesis = TRUE;
    else if (!g_ascii_isalnum (charset[i]) && charset[i] != '-'
                && charset[i] != '.' && charset[i] != ':' && charset[i] != '_')
      return FALSE;
  i--;
  if (!g_ascii_isalnum (charset[i]) && charset[i] != ')')
    return FALSE;
  converter = g_iconv_open (charset, "UTF-8");
  if (converter == (GIConv)-1 || g_iconv_close (converter) != 0)
    return FALSE;
  converter = g_iconv_open ("UTF-8", charset);
  return converter != (GIConv)-1 && g_iconv_close (converter) == 0;
}


/*  ja:文字符号化方式のリストを整形する
    charlist,文字符号化方式のリスト
         RET,整形された文字符号化方式のリスト                               */
gchar *
charuty_strip (const gchar *charlist)
{
  gchar *result = NULL;

  if (charlist)
    {
      gchar **charset;
      gint i;

      charset = g_strsplit (charlist, ",", 0);
      for (i = 0; charset[i]; i++)
        {
          g_strstrip (charset[i]);
          if (charuty_is_valid (charset[i]))
            {
              gint j;

              for (j = 0; j < i; j++)
                if (!g_strcasecmp (charset[i], charset[j]))
                  break;
              if (i <= j)
                {
                  gchar *tmp;

                  tmp = result ? g_strconcat (result, ",", charset[i], NULL)
                               : g_strdup (charset[i]);
                  g_free (result);
                  result = tmp;
                }
            }
        }
      g_strfreev (charset);
    }
  return result;
}


/*  ja:文字符号化方式を置換する
    charlist0,文字符号化方式のリスト
    charlist1,置換する文字符号化方式のリスト
         RET,置換された文字符号化方式のリスト                               */
gchar *
charuty_rename (const gchar *charlist0,
                const gchar *charlist1)
{
  gchar **charset0, **charset1, *result = NULL;
  gint i;

  if (!charlist0 || !charlist1)
    return charlist0 ? g_strdup (charlist0) : NULL;
  charset0 = g_strsplit (charlist0, ",", 0);
  charset1 = g_strsplit (charlist1, ",", 0);
  for (i = 0; charset0[i]; i++)
    {
      gchar *tmp;
      gint j;

      for (j = 0; charset1[j]; j++)
        if (!g_strcasecmp (charset0[i], charset1[j]))
          break;
      if (charset1[j])
        tmp = result ? g_strconcat (result, ",", charset1[j], NULL)
                     : g_strdup (charset1[j]);
      else
        tmp = result ? g_strconcat (result, ",", charset0[i], NULL)
                     : g_strdup (charset0[i]);
      g_free (result);
      result = tmp;
    }
  return result;
}


/*  ja:文字符号化方式のリストを追加する
    charlist0,文字符号化方式のリスト
    charlist1,追加する文字符号化方式のリスト
          RET,追加された文字符号化方式のリスト                              */
gchar *
charuty_append (const gchar *charlist0,
                const gchar *charlist1)
{
  gchar **charset, *tmp, *result;
  gint i;

  if (!charlist0 || !charlist1)
    return charlist0 ? g_strdup (charlist0)
                     : charlist1 ? g_strdup (charlist1) : NULL;
  tmp = charuty_strip (charlist1);
  result = charuty_rename (charlist0, tmp);
  g_free (tmp);
  charset = g_strsplit (charlist1, ",", 0);
  for (i = 0; charset[i]; i++)
    if (!g_strstr (result, charset[i]))
      {
        tmp = g_strconcat (result, ",", charset[i], NULL);
        g_free (result);
        result = tmp;
      }
  g_strfreev (charset);
  return result;
}


/*  ja:文字符号化方式を削除する
    charlist,文字符号化方式のリスト
     charset,削除する文字符号化方式
         RET,削除された文字符号化方式のリスト                               */
gchar *
charuty_remove (const gchar *charlist,
                const gchar *charset)
{
  gchar **str_array, *result = NULL;
  gint i;

  if (!charlist || !charset)
    return charlist ? g_strdup (charlist) : NULL;
  str_array = g_strsplit (charlist, ",", 0);
  for (i = 0; str_array[i]; i++)
    if (g_strcasecmp (str_array[i], charset))
      {
        gchar *tmp;

        tmp = result ? g_strconcat (result, ",", str_array[i], NULL)
                     : g_strdup (str_array[i]);
        g_free (result);
        result = tmp;
      }
  g_strfreev (str_array);
  return result;
}
