/*
    fileio
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef __FILEIO_H__
#define __FILEIO_H__


#include "common.h"
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif /* HAVE_FCNTL_H */
#ifdef HAVE_SYS_FILE_H
# include <sys/file.h>
#endif /* HAVE_SYS_FILE_H */
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif /* HAVE_SYS_STAT_H */
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif /* HAVE_UNISTD_H */
#ifdef G_OS_WIN32
# include <tchar.h>
# include <windows.h>
#endif /* G_OS_WIN32 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#ifdef G_OS_WIN32
# define FILEIO_ACCESS_READ         GENERIC_READ
# define FILEIO_ACCESS_WRITE        GENERIC_WRITE
# define FILEIO_SHARE_READ          FILE_SHARE_READ
# define FILEIO_SHARE_WRITE         FILE_SHARE_WRITE
# define FILEIO_MODE_CREATE_NEW     CREATE_NEW
# define FILEIO_MODE_CREATE_ALWAYS  CREATE_ALWAYS
# define FILEIO_MODE_OPEN_EXISTING  OPEN_EXISTING
# define FILEIO_MODE_OPEN_ALWAYS    OPEN_ALWAYS
# define FILEIO_SEEK_SET            FILE_BEGIN
# define FILEIO_SEEK_CUR            FILE_CURRENT
# define FILEIO_SEEK_END            FILE_END
#else /* not G_OS_WIN32 */
# define FILEIO_ACCESS_READ         1
# define FILEIO_ACCESS_WRITE        2
# define FILEIO_SHARE_READ          1
# define FILEIO_SHARE_WRITE         2
# define FILEIO_MODE_CREATE_NEW     0
# define FILEIO_MODE_CREATE_ALWAYS  1
# define FILEIO_MODE_OPEN_EXISTING  2
# define FILEIO_MODE_OPEN_ALWAYS    3
# define FILEIO_SEEK_SET            SEEK_SET
# define FILEIO_SEEK_CUR            SEEK_CUR
# define FILEIO_SEEK_END            SEEK_END
#endif /* not G_OS_WIN32 */
#define FILEIO_SHARE_DENY           0
#define FILEIO_ACCESS_ALL           (FILEIO_ACCESS_READ | FILEIO_ACCESS_WRITE)
#define FILEIO_SHARE_ALL            (FILEIO_SHARE_READ | FILEIO_SHARE_WRITE)
#define FILEIO_TEST_IS_REGULAR      (1 << 0)
#define FILEIO_TEST_IS_SYMLINK      (1 << 1)
#define FILEIO_TEST_IS_DIR          (1 << 2)
#define FILEIO_TEST_IS_EXECUTABLE   (1 << 3)
#define FILEIO_TEST_EXISTS          (1 << 4)
#define FILEIO_ICON_SIZE_SMALL      0
#define FILEIO_ICON_SIZE_LARGE      1


#define fileio_isfile(file) g_file_test(file,G_FILE_TEST_EXISTS)


typedef struct _FileIO FileIO;
#ifdef G_OS_WIN32
typedef FILETIME FileIOTime;
#else /* not G_OS_WIN32 */
typedef time_t FileIOTime;
#endif /* not G_OS_WIN32 */
typedef struct _FileIOSysTime
{
  gint year;    /* ja:年(1900...) */
  gint mon;     /* ja:月(1...12) */
  gint mday;    /* ja:日(1..31) */
  gint hour;    /* ja:時間(0..23) */
  gint min;     /* ja:分(0..59) */
  gint sec;     /* ja:秒(0..61) */
  gint wday;    /* ja:曜日(0:日曜日,1:月曜日,...) */
} FileIOSysTime;



/******************************************************************************
*                                                                             *
* ja:ファイル名関数群                                                         *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの存在確認
    file,ファイル名
    test,条件
     RET,TRUE:合致する,FALSE:合致しない                                     */
gboolean
fileio_test (const gchar *file,
             const guint  test);


/*  ja:ファイルのフルパスを取得する
    file,ファイル名
     RET,フルパス                                                           */
#define fileio_get_full_path(file) fileio_get_absolute_path(file,NULL)


/*  ja:ファイルの絶対パスを取得する
    file,ファイル名
    path,パス名
     RET,フルパス                                                           */
gchar *
fileio_get_absolute_path (const gchar *file,
                          const gchar *path);


/*  ja:ファイルの相対パスを取得する
    file,ファイル名(終点)
    path,パス名(起点)
     RET,フルパス                                                           */
gchar *
fileio_get_relative_path (const gchar *file,
                          const gchar *path);


/******************************************************************************
*                                                                             *
* ja:低レベルファイル入出力関数群                                             *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの作成
      file,ファイル名
    access,アクセス
     share,共有
      mode,モード
       RET,ファイルポインタ,NULL:エラー                                     */
FileIO *
fileio_open (const gchar *file,
             const guint  access,
             const guint  share,
             const guint  mode);


/*  ja:ファイルポインタの参照数を増やす
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_ref (FileIO *fio);


/*  ja:ファイルポインタの参照数を減らす
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_unref (FileIO *fio);


/*  ja:ファイルポインタを閉じる
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
#define fileio_close(fio) fileio_unref(fio)


/*  ja:ファイルから読み込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_read (FileIO       *fio,
             gpointer      data,
             const gssize  length);


/*  ja:ファイルに書き込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_write (FileIO        *fio,
              gconstpointer  data,
              const gssize   length);


/*  ja:ファイルをシークする
       fio,ファイルポインタ
    offset,位置
    whence,モード
       RET,バイト数,-1:エラー                                               */
goffset
fileio_seek (FileIO        *fio,
             const goffset  offset,
             const guint    whence);


/*  ja:ファイルサイズを求める
    fio,ファイルポインタ
    RET,ファイルサイズ,-1:エラー                                            */
goffset
fileio_size (FileIO *fio);


/******************************************************************************
*                                                                             *
* ja:高レベルファイル入出力関数群                                             *
*                                                                             *
******************************************************************************/
/*  ja:ファイルをメモリに読み込む
      file,ファイル名
    length,ファイルサイズ
       RET,メモリ,NULL:エラー                                               */
gpointer
fileio_load (const gchar *file,
             gssize      *length);


/*  ja:ファイルにメモリから書き込む
      file,ファイル名
      data,メモリ
    length,サイズ
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
fileio_save (const gchar   *file,
             gconstpointer  data,
             const gssize   length);


/*  ja:1文字読み込む
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_getc (FileIO *fio);


/*  ja:1文字書き込む
      c,文字
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_putc (gint    c,
             FileIO *fio);


/*  ja:1行読み込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,data:正常終了,NULL:エラー                                        */
gchar *
fileio_gets (gchar       *data,
             const gsize  length,
             FileIO      *fio);


/*  ja:1行書き込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,正数:正常終了,EOF:エラー                                         */
gint
fileio_puts (const gchar *data,
             FileIO      *fio);


/******************************************************************************
*                                                                             *
* ja:ディレクトリ関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ディレクトリを作る
    dir,ディレクトリ名
    RET,TRUE:正常終了,FALSE,エラー                                          */
gboolean
fileio_mkdir (const gchar *dir);


/******************************************************************************
*                                                                             *
* ja:ファイル時刻関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの時刻を取得する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_gettime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime);


/*  ja:ファイルの時刻を設定する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_settime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime);


/*  ja:ファイルの時刻を変換する
    ftime,ファイルの時刻
    stime,システム時刻                                                      */
void
fileio_getsystime (FileIOTime    *ftime,
                   FileIOSysTime *stime);


/******************************************************************************
*                                                                             *
* ja:応用ファイル関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ファイルのアイコンを取得する
         file,ファイル名
    icon_size,FILEIO_ICON_SIZE_SMALL/FILEIO_ICON_SIZE_LARGE
          RET,pixbuf,NULL:エラー                                            */
GdkPixbuf *
fileio_extract_icon (const gchar *file,
                     const guint  icon_size);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __FILEIO_H__ */
